/*--------------------------------------------------------------------------+
$Id: CollectionUtilsTest.java 27324 2010-03-31 08:53:26Z hummelb $
|                                                                          |
| Copyright 2005-2010 Technische Universitaet Muenchen                     |
|                                                                          |
| Licensed under the Apache License, Version 2.0 (the "License");          |
| you may not use this file except in compliance with the License.         |
| You may obtain a copy of the License at                                  |
|                                                                          |
|    http://www.apache.org/licenses/LICENSE-2.0                            |
|                                                                          |
| Unless required by applicable law or agreed to in writing, software      |
| distributed under the License is distributed on an "AS IS" BASIS,        |
| WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. |
| See the License for the specific language governing permissions and      |
| limitations under the License.                                           |
+--------------------------------------------------------------------------*/
package edu.tum.cs.commons.collections;

import java.text.Collator;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Date;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

import junit.framework.TestCase;
import edu.tum.cs.commons.string.StringUtils;

/**
 * Test for {@link CollectionUtils}.
 * 
 * @author Florian Deissenboeck
 * @author $Author: hummelb $
 * @version $Rev: 27324 $
 * @levd.rating GREEN Hash: FA1B9F8C2A6003A581249B9C6B46DE27
 */
public class CollectionUtilsTest extends TestCase {

	/** Test for {@link CollectionUtils#sort(java.util.Collection)}. */
	public void testSort() {
		HashSet<String> set = new HashSet<String>();
		int stringCount = 100;

		for (int i = 0; i < stringCount; i++) {
			set.add(StringUtils.randomString(10));
		}

		// make sure that the original set is unchanged
		Set<String> unmodifiable = CollectionUtils.asUnmodifiable(set);
		List<String> sortedList = CollectionUtils.sort(unmodifiable);
		assertSorted(sortedList);
	}

	/** Test for {@link CollectionUtils#reverse(Collection)}. */
	public void testReverse() {
		List<Integer> list = new ArrayList<Integer>();
		for (int i = 100; i >= 0; i--) {
			list.add(i);
		}

		// make sure that the original list is unchanged
		List<Integer> unmodifiable = CollectionUtils.asUnmodifiable(list);
		List<Integer> reversed = CollectionUtils.reverse(unmodifiable);
		assertSorted(reversed);
	}

	/** Asserts that list is sorted */
	private <T extends Comparable<? super T>> void assertSorted(List<T> reversed) {
		for (int i = 1; i < reversed.size(); i++) {
			assertTrue(reversed.get(i).compareTo(reversed.get(i - 1)) >= 0);
		}
	}

	/**
	 * Test for
	 * {@link CollectionUtils#sort(java.util.Collection, java.util.Comparator)}.
	 */
	public void testSortCollectionWithComparator() {
		HashSet<String> set = new HashSet<String>();

		int stringCount = 100;

		for (int i = 0; i < stringCount; i++) {
			set.add(StringUtils.randomString(10));
		}

		Collator collator = Collator.getInstance();
		List<String> sortedList = CollectionUtils.sort(set,
				new InvertingComparator<String>(collator));

		for (int i = 1; i < stringCount; i++) {
			assertTrue(collator.compare(sortedList.get(i), sortedList
					.get(i - 1)) <= 0);
		}
	}

	/**
	 * Test for {@link CollectionUtils#asHashSet(Object[])}.
	 */
	public void testAsHashSet() {
		HashSet<String> set = CollectionUtils.asHashSet("x", "y", "z");
		assertEquals(3, set.size());
		assertTrue(set.containsAll(Arrays.asList("x", "y", "z")));

		// to equal elements
		set = CollectionUtils.asHashSet("x", "x", "z");
		assertEquals(2, set.size());
		assertTrue(set.containsAll(Arrays.asList("x", "z")));

		// empty set
		set = CollectionUtils.asHashSet();
		assertTrue(set.isEmpty());
	}

	/** Check if empty list works and does not create a class cast exceptionn. */
	public void testEmptyList() {
		UnmodifiableList<String> emptyList = CollectionUtils.emptyList();
		assertTrue(emptyList.isEmpty());
	}

	/** Check if empty set works and does not create a class cast exceptionn. */
	public void testEmptySet() {
		UnmodifiableSet<String> emptySet = CollectionUtils.emptySet();
		assertTrue(emptySet.isEmpty());
	}

	/** Check if empty map works and does not create a class cast exceptionn. */
	public void testEmptyMap() {
		UnmodifiableMap<String, Date> emptyMap = CollectionUtils.emptyMap();
		assertTrue(emptyMap.isEmpty());
	}

	/** Test {@link CollectionUtils#toArray(Collection, Class)}. */
	public void testToArray() {
		String[] array = new String[] { "test1", "test2", "test3" };
		List<String> set = Arrays.asList(array);
		assertTrue(Arrays.equals(array, CollectionUtils.toArray(set,
				String.class)));

		assertTrue(CollectionUtils.toArray(new ArrayList<String>(),
				String.class).length == 0);

		Object[] objectArray = CollectionUtils.toArray(set, Object.class);
		assertTrue(objectArray.length == 3);

	}

	/** Test {@link CollectionUtils#copyArray(Object[])} */
	public void testCopyArray() {
		String[] test = { "test1", "test2", "test3" };
		String[] copy = CollectionUtils.copyArray(test);
		assertTrue(Arrays.equals(test, copy));
		assertNotSame(test, copy);

		test = new String[0];
		copy = CollectionUtils.copyArray(test);
		assertEquals(0, copy.length);
	}

	/** Test {@link CollectionUtils#computeUnorderedPairs(Collection)} */
	public void testComputeUnorderedPairs() {
		List<Integer> elements = Arrays.asList(new Integer[] { 1, 2, 3, 4 });

		List<ImmutablePair<Integer, Integer>> unorderedPairs = CollectionUtils
				.computeUnorderedPairs(elements);
		assertEquals("Unexpected number of pairs", 6, unorderedPairs.size());

		assertPair(unorderedPairs.get(0), 1, 2);
		assertPair(unorderedPairs.get(1), 1, 3);
		assertPair(unorderedPairs.get(2), 1, 4);
		assertPair(unorderedPairs.get(3), 2, 3);
		assertPair(unorderedPairs.get(4), 2, 4);
		assertPair(unorderedPairs.get(5), 3, 4);
	}

	/** Assert expected values of pair */
	private void assertPair(ImmutablePair<Integer, Integer> pair,
			int expectedFirst, int expectedSecond) {
		assertEquals(expectedFirst, pair.first.intValue());
		assertEquals(expectedSecond, pair.second.intValue());
	}

	/** Test {@link CollectionUtils#getLast(List)} */
	public void testGetLast() {
		List<String> list = new ArrayList<String>();
		assertEquals(null, CollectionUtils.getLast(list));

		String s1 = "A";
		list.add(s1);
		assertEquals(s1, CollectionUtils.getLast(list));

		String s2 = "B";
		list.add(s2);
		assertEquals(s2, CollectionUtils.getLast(list));

		list.remove(s2);
		assertEquals(s1, CollectionUtils.getLast(list));
	}
}