/*--------------------------------------------------------------------------+
$Id: RegexReplacementProcessorTest.java 26283 2010-02-18 11:18:57Z juergens $
|                                                                          |
| Copyright 2005-2010 Technische Universitaet Muenchen                     |
|                                                                          |
| Licensed under the Apache License, Version 2.0 (the "License");          |
| you may not use this file except in compliance with the License.         |
| You may obtain a copy of the License at                                  |
|                                                                          |
|    http://www.apache.org/licenses/LICENSE-2.0                            |
|                                                                          |
| Unless required by applicable law or agreed to in writing, software      |
| distributed under the License is distributed on an "AS IS" BASIS,        |
| WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. |
| See the License for the specific language governing permissions and      |
| limitations under the License.                                           |
+--------------------------------------------------------------------------*/
package edu.tum.cs.commons.string;

import java.util.ArrayList;
import java.util.List;
import java.util.regex.PatternSyntaxException;

import junit.framework.TestCase;

/**
 * Test class for the <code>RegexReplacementProcessor</code>.
 * 
 * 
 * @author Florian Deissenboeck
 * @author $Author: juergens $
 * @version $Rev: 26283 $
 * @levd.rating GREEN Hash: 83915E36AC6C3A843B619151068F6A18
 */
public class RegexReplacementProcessorTest extends TestCase {

    /** The processor under test. */
    private RegexReplacementProcessor processor;

    /** Test with emtpy list of replacements. */
    public void testEmptyRegexList() {
        List<IRegexReplacement> regexes = new ArrayList<IRegexReplacement>();
        String test = "test";
        processor = new RegexReplacementProcessor(regexes);
        assertEquals("test", processor.process(test));
    }

    /** Test a simple delete replacement. */
    public void testSimpleDelete() {
        List<IRegexReplacement> regexes = new ArrayList<IRegexReplacement>();
        regexes.add(new RegexReplacement("e"));
        String test = "test";
        processor = new RegexReplacementProcessor(regexes);
        assertEquals("tst", processor.process(test));
    }

    /** Test multiple delete replacements. */
    public void testMultiDelete() {
        List<IRegexReplacement> regexes = new ArrayList<IRegexReplacement>();
        regexes.add(new RegexReplacement("e"));
        regexes.add(new RegexReplacement("s"));
        String test = "test";
        processor = new RegexReplacementProcessor(regexes);
        assertEquals("tt", processor.process(test));
    }

    /** Test a simple replacement. */
    public void testSimpleReplace() {
        List<IRegexReplacement> regexes = new ArrayList<IRegexReplacement>();
        regexes.add(new RegexReplacement("e", "x"));
        String test = "test";
        processor = new RegexReplacementProcessor(regexes);
        assertEquals("txst", processor.process(test));
    }

    /** Test a simple replacement using a capturing group. */
    public void testSimpleReplaceWithCapturingGroup() {
        List<IRegexReplacement> regexes = new ArrayList<IRegexReplacement>();
        regexes.add(new RegexReplacement("//(.*)", "xx$1"));
        String test = "test3 // test4";
        processor = new RegexReplacementProcessor(regexes);
        assertEquals("test3 xx test4", processor.process(test));
    }

    /** Test a simple replacement using a illegal capturing group. */
    public void testSimpleReplaceWithIllegalCapturingGroup() {
        List<IRegexReplacement> regexes = new ArrayList<IRegexReplacement>();

        // there is not capturing group $1
        regexes.add(new RegexReplacement("//.*", "xx$1"));
        String test = "test3 // test4";
        processor = new RegexReplacementProcessor(regexes);
        try {            
            processor.process(test);
            fail();
        } catch (PatternSyntaxException ex) {
            // this is expected
        }

    }

    /** Test a multiple replacements. */
    public void testMultiReplace() {
        List<IRegexReplacement> regexes = new ArrayList<IRegexReplacement>();
        regexes.add(new RegexReplacement("e", "x"));
        regexes.add(new RegexReplacement("s", "u"));
        String test = "test";
        processor = new RegexReplacementProcessor(regexes);
        assertEquals("txut", processor.process(test));
    }

    /** Test a multiple replacements where order is relevant. */
    public void testMultiReplaceWithOrder() {
        List<IRegexReplacement> regexes = new ArrayList<IRegexReplacement>();
        regexes.add(new RegexReplacement("e", "x"));
        regexes.add(new RegexReplacement("x", "u"));
        String test = "test";
        processor = new RegexReplacementProcessor(regexes);
        assertEquals("tust", processor.process(test));
    }

    /**
     * Test if processor is case sensitive an if embedded flag expression for
     * case sensitiveness works.
     */
    public void testCaseSensitiveness() {
        List<IRegexReplacement> regexes = new ArrayList<IRegexReplacement>();
        regexes.add(new RegexReplacement("E"));
        String test = "test";
        processor = new RegexReplacementProcessor(regexes);
        assertEquals("test", processor.process(test));

        regexes.clear();

        // use embedded flag expression for case sensitiveness
        regexes.add(new RegexReplacement("(?i)E"));
        processor = new RegexReplacementProcessor(regexes);
        assertEquals("tst", processor.process(test));
    }

    /** Test if deletion works for pattern spanning multiple lines. */
    public void testMultiLineDelete() {
        String[] testArray = { "test1/** test 3", "test4 */test2" };
        String test = StringUtils.concat(testArray, StringUtils.CR);
        List<IRegexReplacement> regexes = new ArrayList<IRegexReplacement>();
        regexes.add(new RegexReplacement("(?s)\\Q/**\\E.*\\Q*/\\E"));
        processor = new RegexReplacementProcessor(regexes);
        assertEquals("test1test2", processor.process(test));
    }

    /** Test if replacement works for pattern spanning multiple lines. */
    public void testMultiLineReplace() {
        String[] testArray = { "test1/** test 3", "test4 */test2" };
        String test = StringUtils.concat(testArray, StringUtils.CR);
        List<IRegexReplacement> regexes = new ArrayList<IRegexReplacement>();
        regexes.add(new RegexReplacement("(?s)\\Q/**\\E.*\\Q*/\\E", "y"));
        processor = new RegexReplacementProcessor(regexes);
        assertEquals("test1ytest2", processor.process(test));
    }

    /**
     * Test if deletion works for pattern spanning multiple lines (with
     * capturing group).
     */
    public void testMultiLineReplaceWithCapturingGroup() {
        String[] testArray = { "test1/** test 3", "test4 */test2" };
        String test = StringUtils.concat(testArray, StringUtils.CR);
        List<IRegexReplacement> regexes = new ArrayList<IRegexReplacement>();
        regexes.add(new RegexReplacement("(?s)\\Q/**\\E(.*)\\Q*/\\E", "$1"));
        processor = new RegexReplacementProcessor(regexes);
        assertEquals("test1 test 3" + StringUtils.CR + "test4 test2", processor
                .process(test));
    }
}