use v6;
#-------------------------------------------------------------------------------
=begin pod

=head1 Gnome::Gdk3::Window

Onscreen display areas in the target window system

=head1 Description

A B<Gnome::Gdk3::Window> is a (usually) rectangular region on the screen. It’s a low-level object, used to implement high-level objects such as B<Gnome::Gtk3::Widget> and B<Gnome::Gtk3::Window> on the GTK+ level. A B<Gnome::Gtk3::Window> is a toplevel window, the thing a user might think of as a “window” with a titlebar and so on; a B<Gnome::Gtk3::Window> may contain many B<Gnome::Gdk3::Windows>. For example, each B<Gnome::Gtk3::Button> has a B<Gnome::Gdk3::Window> associated with it.


=head2 Composited Windows

Normally, the windowing system takes care of rendering the contents of a child window onto its parent window. This mechanism can be intercepted by calling C<gdk_window_set_composited()> on the child window. For a “composited” window it is the responsibility of the application to render the window contents at the right spot.


=head2 Offscreen Windows

Offscreen windows are more general than composited windows, since they allow not only to modify the rendering of the child window onto its parent, but also to apply coordinate transformations.

To integrate an offscreen window into a window hierarchy, one has to call C<gdk_offscreen_window_set_embedder()> and handle a number of signals. The  I<pick-embedded-child> signal on the embedder window is used to select an offscreen child at given coordinates, and the  I<to-embedder> and  I<from-embedder> signals on the offscreen window are used to translate coordinates between the embedder and the offscreen window.

For rendering an offscreen window onto its embedder, the contents of the offscreen window are available as a surface, via C<gdk_offscreen_window_get_surface()>.


=head2 Testing

Some routines are added to this module for GUI testing. They were in a separate C file but could be included here. These are test utilities to test some of the events generated by mouse and keyboard. This set of routines will not be of much use for the GUI developer. However, for interface testing, this module is may be of use. A package which will be using this, is in development and is called B<Gnome::T>.


=head1 Synopsis
=head2 Declaration

  unit class Gnome::Gdk3::Window;
  also is Gnome::GObject::Object;

=comment head2 Example

=end pod
#-------------------------------------------------------------------------------
use NativeCall;

use Gnome::N::X:api<1>;
use Gnome::N::NativeLib:api<1>;
use Gnome::N::N-GObject:api<1>;
use Gnome::N::GlibToRakuTypes:api<1>;

use Gnome::Glib::Error:api<1>;
use Gnome::Glib::List:api<1>;

use Gnome::GObject::Object:api<1>;

use Gnome::Gdk3::Types:api<1>;
use Gnome::Gdk3::Events:api<1>;
use Gnome::Gdk3::Device:api<1>;

#use Gnome::Cairo;
use Gnome::Cairo::Types:api<1>;
use Gnome::Cairo::Enums:api<1>;

#-------------------------------------------------------------------------------
# /usr/include/gtk-3.0/gdk/gdkwindow.h
# https://developer.gnome.org/gdk3/stable/gdk3-Windows.html
unit class Gnome::Gdk3::Window:auth<github:MARTIMM>:api<1>;
also is Gnome::GObject::Object;

#-------------------------------------------------------------------------------
=begin pod
=head1 Types
=end pod
#-------------------------------------------------------------------------------
=begin pod
=head2 enum GdkAnchorHints

Positioning hints for aligning a window relative to a rectangle.

These hints determine how the window should be positioned in the case that
the window would fall off-screen if placed in its ideal position.

For example, C<GDK_ANCHOR_FLIP_X> will replace C<GDK_GRAVITY_NORTH_WEST> with
C<GDK_GRAVITY_NORTH_EAST> and vice versa if the window extends beyond the left
or right edges of the monitor.

If C<GDK_ANCHOR_SLIDE_X> is set, the window can be shifted horizontally to fit
on-screen. If C<GDK_ANCHOR_RESIZE_X> is set, the window can be shrunken
horizontally to fit.

In general, when multiple flags are set, flipping should take precedence over
sliding, which should take precedence over resizing.

Stability: Unstable


=item GDK_ANCHOR_FLIP_X: allow flipping anchors horizontally
=item GDK_ANCHOR_FLIP_Y: allow flipping anchors vertically
=item GDK_ANCHOR_SLIDE_X: allow sliding window horizontally
=item GDK_ANCHOR_SLIDE_Y: allow sliding window vertically
=item GDK_ANCHOR_RESIZE_X: allow resizing window horizontally
=item GDK_ANCHOR_RESIZE_Y: allow resizing window vertically
=item GDK_ANCHOR_FLIP: allow flipping anchors on both axes
=item GDK_ANCHOR_SLIDE: allow sliding window on both axes
=item GDK_ANCHOR_RESIZE: allow resizing window on both axes


=end pod

#TE:0:GdkAnchorHints:
enum GdkAnchorHints is export (
  'GDK_ANCHOR_FLIP_X'   => 1 +< 0,
  'GDK_ANCHOR_FLIP_Y'   => 1 +< 1,
  'GDK_ANCHOR_SLIDE_X'  => 1 +< 2,
  'GDK_ANCHOR_SLIDE_Y'  => 1 +< 3,
  'GDK_ANCHOR_RESIZE_X' => 1 +< 4,
  'GDK_ANCHOR_RESIZE_Y' => 1 +< 5,

#  'GDK_ANCHOR_FLIP'     => GDK_ANCHOR_FLIP_X +| GDK_ANCHOR_FLIP_Y,
#  'GDK_ANCHOR_SLIDE'    => GDK_ANCHOR_SLIDE_X +| GDK_ANCHOR_SLIDE_Y,
#  'GDK_ANCHOR_RESIZE'   => GDK_ANCHOR_RESIZE_X +| GDK_ANCHOR_RESIZE_Y
  'GDK_ANCHOR_FLIP'     => 1 +< 0 +| 1 +< 1,
  'GDK_ANCHOR_SLIDE'    => 1 +< 2 +| 1 +< 3,
  'GDK_ANCHOR_RESIZE'   => 1 +< 4 +| 1 +< 5
);

#-------------------------------------------------------------------------------
=begin pod
=head2 enum GdkFullscreenMode

Indicates which monitor (in a multi-head setup) a window should span over
when in fullscreen mode.



=item GDK_FULLSCREEN_ON_CURRENT_MONITOR: Fullscreen on current monitor only.
=item GDK_FULLSCREEN_ON_ALL_MONITORS: Span across all monitors when fullscreen.


=end pod

#TE:0:GdkFullscreenMode:
enum GdkFullscreenMode is export (
  'GDK_FULLSCREEN_ON_CURRENT_MONITOR',
  'GDK_FULLSCREEN_ON_ALL_MONITORS'
);

#-------------------------------------------------------------------------------
=begin pod
=head2 enum GdkGravity

Defines the reference point of a window and the meaning of coordinates
passed to C<gtk_window_move()>. See C<gtk_window_move()> and the "implementation
notes" section of the
[Extended Window Manager Hints](http://www.freedesktop.org/Standards/wm-spec)
specification for more details.


=item GDK_GRAVITY_NORTH_WEST: the reference point is at the top left corner.
=item GDK_GRAVITY_NORTH: the reference point is in the middle of the top edge.
=item GDK_GRAVITY_NORTH_EAST: the reference point is at the top right corner.
=item GDK_GRAVITY_WEST: the reference point is at the middle of the left edge.
=item GDK_GRAVITY_CENTER: the reference point is at the center of the window.
=item GDK_GRAVITY_EAST: the reference point is at the middle of the right edge.
=item GDK_GRAVITY_SOUTH_WEST: the reference point is at the lower left corner.
=item GDK_GRAVITY_SOUTH: the reference point is at the middle of the lower edge.
=item GDK_GRAVITY_SOUTH_EAST: the reference point is at the lower right corner.
=item GDK_GRAVITY_STATIC: the reference point is at the top left corner of the window itself, ignoring window manager decorations.


=end pod

#TE:0:GdkGravity:
enum GdkGravity is export (
  'GDK_GRAVITY_NORTH_WEST' => 1,
  'GDK_GRAVITY_NORTH',
  'GDK_GRAVITY_NORTH_EAST',
  'GDK_GRAVITY_WEST',
  'GDK_GRAVITY_CENTER',
  'GDK_GRAVITY_EAST',
  'GDK_GRAVITY_SOUTH_WEST',
  'GDK_GRAVITY_SOUTH',
  'GDK_GRAVITY_SOUTH_EAST',
  'GDK_GRAVITY_STATIC'
);

#-------------------------------------------------------------------------------
=begin pod
=head2 enum GdkWMDecoration

These are hints originally defined by the Motif toolkit.
The window manager can use them when determining how to decorate
the window. The hint must be set before mapping the window.


=item GDK_DECOR_ALL: all decorations should be applied.
=item GDK_DECOR_BORDER: a frame should be drawn around the window.
=item GDK_DECOR_RESIZEH: the frame should have resize handles.
=item GDK_DECOR_TITLE: a titlebar should be placed above the window.
=item GDK_DECOR_MENU: a button for opening a menu should be included.
=item GDK_DECOR_MINIMIZE: a minimize button should be included.
=item GDK_DECOR_MAXIMIZE: a maximize button should be included.


=end pod

#TE:0:GdkWMDecoration:
enum GdkWMDecoration is export (
  'GDK_DECOR_ALL'		=> 1 +< 0,
  'GDK_DECOR_BORDER'	=> 1 +< 1,
  'GDK_DECOR_RESIZEH'	=> 1 +< 2,
  'GDK_DECOR_TITLE'	=> 1 +< 3,
  'GDK_DECOR_MENU'	=> 1 +< 4,
  'GDK_DECOR_MINIMIZE'	=> 1 +< 5,
  'GDK_DECOR_MAXIMIZE'	=> 1 +< 6
);

#-------------------------------------------------------------------------------
=begin pod
=head2 enum GdkWMFunction

These are hints originally defined by the Motif toolkit. The window manager
can use them when determining the functions to offer for the window. The
hint must be set before mapping the window.


=item GDK_FUNC_ALL: all functions should be offered.
=item GDK_FUNC_RESIZE: the window should be resizable.
=item GDK_FUNC_MOVE: the window should be movable.
=item GDK_FUNC_MINIMIZE: the window should be minimizable.
=item GDK_FUNC_MAXIMIZE: the window should be maximizable.
=item GDK_FUNC_CLOSE: the window should be closable.


=end pod

#TE:0:GdkWMFunction:
enum GdkWMFunction is export (
  'GDK_FUNC_ALL'		=> 1 +< 0,
  'GDK_FUNC_RESIZE'	=> 1 +< 1,
  'GDK_FUNC_MOVE'		=> 1 +< 2,
  'GDK_FUNC_MINIMIZE'	=> 1 +< 3,
  'GDK_FUNC_MAXIMIZE'	=> 1 +< 4,
  'GDK_FUNC_CLOSE'	=> 1 +< 5
);

#-------------------------------------------------------------------------------
=begin pod
=head2 enum GdkWindowAttributesType

Used to indicate which fields in the B<Gnome::Gdk3::WindowAttr> struct should be honored.
For example, if you filled in the “cursor” and “x” fields of B<Gnome::Gdk3::WindowAttr>,
pass “I<GDK_WA_X> | I<GDK_WA_CURSOR>” to C<new()>. Fields in
B<Gnome::Gdk3::WindowAttr> not covered by a bit in this enum are required; for example,
the I<width>/I<height>, I<wclass>, and I<window_type> fields are required, they have
no corresponding flag in B<Gnome::Gdk3::WindowAttributesType>.


=item GDK_WA_TITLE: Honor the title field
=item GDK_WA_X: Honor the X coordinate field
=item GDK_WA_Y: Honor the Y coordinate field
=item GDK_WA_CURSOR: Honor the cursor field
=item GDK_WA_VISUAL: Honor the visual field
=item GDK_WA_WMCLASS: Honor the wmclass_class and wmclass_name fields
=item GDK_WA_NOREDIR: Honor the override_redirect field
=item GDK_WA_TYPE_HINT: Honor the type_hint field


=end pod

#TE:0:GdkWindowAttributesType:
enum GdkWindowAttributesType is export (
  'GDK_WA_TITLE'	   => 1 +< 1,
  'GDK_WA_X'	   => 1 +< 2,
  'GDK_WA_Y'	   => 1 +< 3,
  'GDK_WA_CURSOR'	   => 1 +< 4,
  'GDK_WA_VISUAL'	   => 1 +< 5,
  'GDK_WA_WMCLASS'   => 1 +< 6,
  'GDK_WA_NOREDIR'   => 1 +< 7,
  'GDK_WA_TYPE_HINT' => 1 +< 8
);

#-------------------------------------------------------------------------------
=begin pod
=head2 enum GdkWindowEdge

Determines a window edge or corner.


=item GDK_WINDOW_EDGE_NORTH_WEST: the top left corner.
=item GDK_WINDOW_EDGE_NORTH: the top edge.
=item GDK_WINDOW_EDGE_NORTH_EAST: the top right corner.
=item GDK_WINDOW_EDGE_WEST: the left edge.
=item GDK_WINDOW_EDGE_EAST: the right edge.
=item GDK_WINDOW_EDGE_SOUTH_WEST: the lower left corner.
=item GDK_WINDOW_EDGE_SOUTH: the lower edge.
=item GDK_WINDOW_EDGE_SOUTH_EAST: the lower right corner.


=end pod

#TE:0:GdkWindowEdge:
enum GdkWindowEdge is export (
  'GDK_WINDOW_EDGE_NORTH_WEST',
  'GDK_WINDOW_EDGE_NORTH',
  'GDK_WINDOW_EDGE_NORTH_EAST',
  'GDK_WINDOW_EDGE_WEST',
  'GDK_WINDOW_EDGE_EAST',
  'GDK_WINDOW_EDGE_SOUTH_WEST',
  'GDK_WINDOW_EDGE_SOUTH',
  'GDK_WINDOW_EDGE_SOUTH_EAST'
);

#-------------------------------------------------------------------------------
=begin pod
=head2 enum GdkWindowHints

Used to indicate which fields of a B<Gnome::Gdk3::Geometry> struct should be paid
attention to. Also, the presence/absence of I<GDK_HINT_POS>,
I<GDK_HINT_USER_POS>, and I<GDK_HINT_USER_SIZE> is significant, though they don't
directly refer to B<Gnome::Gdk3::Geometry> fields. I<GDK_HINT_USER_POS> will be set
automatically by B<Gnome::Gdk3::Window> if you call C<gtk_window_move()>.
I<GDK_HINT_USER_POS> and I<GDK_HINT_USER_SIZE> should be set if the user
specified a size/position using a --geometry command-line argument;
C<gtk_window_parse_geometry()> automatically sets these flags.


=item GDK_HINT_POS: indicates that the program has positioned the window
=item GDK_HINT_MIN_SIZE: min size fields are set
=item GDK_HINT_MAX_SIZE: max size fields are set
=item GDK_HINT_BASE_SIZE: base size fields are set
=item GDK_HINT_ASPECT: aspect ratio fields are set
=item GDK_HINT_RESIZE_INC: resize increment fields are set
=item GDK_HINT_WIN_GRAVITY: window gravity field is set
=item GDK_HINT_USER_POS: indicates that the window’s position was explicitly set by the user
=item GDK_HINT_USER_SIZE: indicates that the window’s size was explicitly set by the user


=end pod

#TE:1:GdkWindowHints:
enum GdkWindowHints is export (
  'GDK_HINT_POS'	       => 1 +< 0,
  'GDK_HINT_MIN_SIZE'    => 1 +< 1,
  'GDK_HINT_MAX_SIZE'    => 1 +< 2,
  'GDK_HINT_BASE_SIZE'   => 1 +< 3,
  'GDK_HINT_ASPECT'      => 1 +< 4,
  'GDK_HINT_RESIZE_INC'  => 1 +< 5,
  'GDK_HINT_WIN_GRAVITY' => 1 +< 6,
  'GDK_HINT_USER_POS'    => 1 +< 7,
  'GDK_HINT_USER_SIZE'   => 1 +< 8
);

#-------------------------------------------------------------------------------
=begin pod
=head2 enum GdkWindowType

Describes the kind of window.


=item GDK_WINDOW_ROOT: root window; this window has no parent, covers the entire screen, and is created by the window system
=item GDK_WINDOW_TOPLEVEL: toplevel window (used to implement B<Gnome::Gdk3::Window>)
=item GDK_WINDOW_CHILD: child window (used to implement e.g. B<Gnome::Gdk3::Entry>)
=item GDK_WINDOW_TEMP: override redirect temporary window (used to implement B<Gnome::Gdk3::Menu>)
=item GDK_WINDOW_FOREIGN: foreign window (see C<foreign_new()>)
=item GDK_WINDOW_OFFSCREEN: offscreen window (see [Offscreen Windows][OFFSCREEN-WINDOWS]). Since 2.18
=item GDK_WINDOW_SUBSURFACE: subsurface-based window; This window is visually tied to a toplevel, and is moved/stacked with it. Currently this window type is only implemented in Wayland. Since 3.14


=end pod

#TE:0:GdkWindowType:
enum GdkWindowType is export (
  'GDK_WINDOW_ROOT',
  'GDK_WINDOW_TOPLEVEL',
  'GDK_WINDOW_CHILD',
  'GDK_WINDOW_TEMP',
  'GDK_WINDOW_FOREIGN',
  'GDK_WINDOW_OFFSCREEN',
  'GDK_WINDOW_SUBSURFACE'
);

#-------------------------------------------------------------------------------
=begin pod
=head2 enum GdkWindowWindowClass

I<GDK_INPUT_OUTPUT> windows are the standard kind of window you might expect.
Such windows receive events and are also displayed on screen.
I<GDK_INPUT_ONLY> windows are invisible; they are usually placed above other
windows in order to trap or filter the events. You can’t draw on
I<GDK_INPUT_ONLY> windows.

=item GDK_INPUT_OUTPUT: window for graphics and events
=item GDK_INPUT_ONLY: window for events only

=end pod

#TE:0:GdkWindowWindowClass:
enum GdkWindowWindowClass is export (
  'GDK_INPUT_OUTPUT', 'GDK_INPUT_ONLY'
);

#-------------------------------------------------------------------------------
=begin pod
=head2 class N-GdkGeometry

The B<Gnome::Gdk3::Geometry> struct gives the window manager information about
a window’s geometry constraints. Normally you would set these on
the GTK+ level using C<gtk_window_set_geometry_hints()>. B<Gnome::Gdk3::Window>
then sets the hints on the B<Gnome::Gdk3::Window> it creates.

C<set_geometry_hints()> expects the hints to be fully valid already
and simply passes them to the window manager; in contrast,
C<gtk_window_set_geometry_hints()> performs some interpretation. For example,
B<Gnome::Gdk3::Window> will apply the hints to the geometry widget instead of the
toplevel window, if you set a geometry widget. Also, the
I<min_width>/I<min_height>/I<max_width>/I<max_height> fields may be set to -1, and
B<Gnome::Gdk3::Window> will substitute the size request of the window or geometry widget.
If the minimum size hint is not provided, B<Gnome::Gdk3::Window> will use its requisition
as the minimum size. If the minimum size is provided and a geometry widget is
set, B<Gnome::Gdk3::Window> will take the minimum size as the minimum size of the
geometry widget rather than the entire window. The base size is treated
similarly.

The canonical use-case for C<gtk_window_set_geometry_hints()> is to get a
terminal widget to resize properly. Here, the terminal text area should be
the geometry widget; B<Gnome::Gdk3::Window> will then automatically set the base size to
the size of other widgets in the terminal window, such as the menubar and
scrollbar. Then, the I<width_inc> and I<height_inc> fields should be set to the
size of one character in the terminal. Finally, the base size should be set
to the size of one character. The net effect is that the minimum size of the
terminal will have a 1x1 character terminal area, and only terminal sizes on
the “character grid” will be allowed.

Here’s an example of how the terminal example would be implemented, assuming
a terminal area widget called “terminal” and a toplevel window “toplevel”:

=begin comment
|[<!-- language="C" -->
N-GdkGeometry hints;

hints.base_width = terminal->char_width;
hints.base_height = terminal->char_height;
hints.min_width = terminal->char_width;
hints.min_height = terminal->char_height;
hints.width_inc = terminal->char_width;
hints.height_inc = terminal->char_height;

gtk_window_set_geometry_hints (GTK_WINDOW (toplevel),
GTK_WIDGET (terminal),
&hints,
GDK_HINT_RESIZE_INC |
GDK_HINT_MIN_SIZE |
GDK_HINT_BASE_SIZE);
]|
=end comment

The other useful fields are the I<min_aspect> and I<max_aspect> fields; these
contain a width/height ratio as a floating point number. If a geometry widget
is set, the aspect applies to the geometry widget rather than the entire
window. The most common use of these hints is probably to set I<min_aspect> and
I<max_aspect> to the same value, thus forcing the window to keep a constant
aspect ratio.


=item C<Int()> $.min_width: minimum width of window (or -1 to use requisition, with B<Gnome::Gdk3::Window> only)
=item C<Int()> $.min_height: minimum height of window (or -1 to use requisition, with B<Gnome::Gdk3::Window> only)
=item C<Int()> $.max_width: maximum width of window (or -1 to use requisition, with B<Gnome::Gdk3::Window> only)
=item C<Int()> $.max_height: maximum height of window (or -1 to use requisition, with B<Gnome::Gdk3::Window> only)
=item C<Int()> $.base_width: allowed window widths are I<base_width> + I<width_inc> * N where N is any integer (-1 allowed with B<Gnome::Gdk3::Window>)
=item C<Int()> $.base_height: allowed window widths are I<base_height> + I<height_inc> * N where N is any integer (-1 allowed with B<Gnome::Gdk3::Window>)
=item C<Int()> $.width_inc: width resize increment
=item C<Int()> $.height_inc: height resize increment
=item C<Num()> $.min_aspect: minimum width/height ratio
=item C<Num()> $.max_aspect: maximum width/height ratio
=item GdkGravity $.win_gravity: window gravity, see C<gtk_window_set_gravity()>


=end pod

#TT:1:N-GdkGeometry:
class N-GdkGeometry is export is repr('CStruct') {
  has gint $.min_width;
  has gint $.min_height;
  has gint $.max_width;
  has gint $.max_height;
  has gint $.base_width;
  has gint $.base_height;
  has gint $.width_inc;
  has gint $.height_inc;
  has gdouble $.min_aspect;
  has gdouble $.max_aspect;
  has GEnum $.win_gravity;
}

#-------------------------------------------------------------------------------
=begin pod
=head2 class N-GdkWindowAttr

Attributes to use for a newly-created window.


=item Str $.title: title of the window (for toplevel windows)
=item C<Int()> $.event_mask: event mask (see C<set_events()>)
=item C<Int()> $.x: X coordinate relative to parent window (see C<move()>)
=item C<Int()> $.y: Y coordinate relative to parent window (see C<move()>)
=item C<Int()> $.width: width of window
=item C<Int()> $.height: height of window
=item GdkWindowWindowClass $.wclass: B<Gnome::Gdk3::DK_INPUT_OUTPUT> (normal window) or B<Gnome::Gdk3::DK_INPUT_ONLY> (invisible window that receives events)
=item N-GObject $.visual: B<Gnome::Gdk3::Visual> for window
=item GdkWindowType $.window_type: type of window
=item N-GObject $.cursor: cursor for the window (see C<set_cursor()>)
=item Str $.wmclass_name: don’t use (see C<gtk_window_set_wmclass()>)
=item Str $.wmclass_class: don’t use (see C<gtk_window_set_wmclass()>)
=item Bool $.override_redirect: C<True> to bypass the window manager
=item GdkWindowTypeHint $.type_hint: a hint of the function of the window


=end pod

#TT:0:N-GdkWindowAttr:
class N-GdkWindowAttr is export is repr('CStruct') {
  has gchar-ptr $.title;
  has gint $.event_mask;
  has gint $.x;
  has gint $.y;
  has gint $.width;
  has gint $.height;
  has GEnum $.wclass;
  has N-GObject $.visual;
  has GEnum $.window_type;
  has N-GObject $.cursor;
  has gchar-ptr $.wmclass_name;
  has gchar-ptr $.wmclass_class;
  has gboolean $.override_redirect;
  has GEnum $.type_hint;
}

#-------------------------------------------------------------------------------
my Bool $signals-added = False;
#-------------------------------------------------------------------------------
=begin pod
=head1 Methods
=head2 new

=begin comment
=head3 new( :event_mask, :wclass, :window_type, :override_redirect)

Creates a new B<Gnome::Gdk3::Window> using the attributes from I<attributes>. See B<Gnome::Gdk3::WindowAttr> and B<Gnome::Gdk3::WindowAttributesType> for more details.

Note: to use this on displays other than the default display, I<parent> must be specified.

  multi method new ( N-GdkWindowAttr :$attributes, Int $attributes_mask --> N-GObject  )

=item N-GdkWindowAttr $attributes; attributes of the new window
=item Int $attributes_mask; (type B<Gnome::Gdk3::WindowAttributesType>): mask indicating which fields in I<attributes> are valid
=end comment
=end pod

#TM:1:new():
#TM:0:new(:):

submethod BUILD ( *%options ) {

  # add signal info in the form of group<signal-name>.
  # groups are e.g. signal, event, nativeobject etc
  $signals-added = self.add-signal-types( $?CLASS.^name,
    :w2<pick-embedded-child create-surface>,
    :w4<to-embedder from-embedder moved-to-rect>,
  ) unless $signals-added;

  # prevent creating wrong widgets
  if self.^name eq 'Gnome::Gdk3::Window' #`{{ or %options<GdkWindow> }} {

    if self.is-valid { }

    # process all named arguments
    elsif %options<native-object>:exists { }

    else {
      my $no;

      if %options.keys.elems {
        die X::Gnome.new(
          :message('Unsupported options for ' ~ self.^name ~
                   ': ' ~ %options.keys.join(', ')
                  )
        );
      }

      else {

        # GDK_WINDOW_ROOT cannot be used because it covers the entire
        # screen, and is created by the window system. (there can only be one!).
        my N-GdkWindowAttr $attrs .= new(
          :event_mask(0), :wclass(GDK_INPUT_OUTPUT),
          :window_type(GDK_WINDOW_TOPLEVEL), :override_redirect(0)
        );

        # No parent, no extra attributes, toplevel
        $no = _gdk_window_new( Any, $attrs, 0);
      }

      self._set-native-object($no);
    }

    # only after creating the native-object, the gtype is known
    self._set-class-info('GdkWindow');
  }
}

#-------------------------------------------------------------------------------
# no pod. user does not have to know about it.
method _fallback ( $native-sub is copy --> Callable ) {

  my Callable $s;
  try { $s = &::("gdk_window_$native-sub"); };
  try { $s = &::("gdk_$native-sub"); } unless ?$s;
  try { $s = &::($native-sub); } if !$s and $native-sub ~~ m/^ 'gdk_' /;

  self._set-class-name-of-sub('GdkWindow');
  $s = callsame unless ?$s;

  $s;
}

#`{{
#-------------------------------------------------------------------------------
#TM:0:add-filter:
=begin pod
=head2 add-filter

Adds an event filter to I<window>, allowing you to intercept events before they reach GDK. This is a low-level operation and makes it easy to break GDK and/or GTK+, so you have to know what you're doing. Pass C<undefined> for I<window> to get all events for all windows, instead of events for a specific window.

If you are interested in X GenericEvents, bear in mind that C<XGetEventData()> has been already called on the event, and C<XFreeEventData()> must not be called within I<function>.

  method add-filter ( GdkFilterFunc $function, Pointer $data )

=item $function; filter callback
=item $data; data to pass to filter callback
=end pod

method add-filter ( GdkFilterFunc $function, Pointer $data ) {

  gdk_window_add_filter(
    self._get-native-object-no-reffing, $function, $data
  );
}

sub gdk_window_add_filter (
  N-GObject $window, GdkFilterFunc $function, gpointer $data
) is native(&gdk-lib)
  { * }
}}

#-------------------------------------------------------------------------------
#TM:1:beep:
=begin pod
=head2 beep

Emits a short beep associated to I<window> in the appropriate display, if supported. Otherwise, emits a short beep on the display just as C<Gnome::Gdk3::Display.beep()>.

  method beep ( )

=end pod

method beep ( ) {
  gdk_window_beep(self._get-native-object-no-reffing);
}

sub gdk_window_beep (
  N-GObject $window
) is native(&gdk-lib)
  { * }

#`{{
#-------------------------------------------------------------------------------
#TM:0:begin-draw-frame:
=begin pod
=head2 begin-draw-frame

Indicates that you are beginning the process of redrawing I<region> on I<window>, and provides you with a B<Gnome::Gdk3::DrawingContext>.

If I<window> is a top level B<Gnome::Gdk3::Window>, backed by a native window implementation, a backing store (offscreen buffer) large enough to contain I<region> will be created. The backing store will be initialized with the background color or background surface for I<window>. Then, all drawing operations performed on I<window> will be diverted to the backing store. When you call C<end_frame()>, the contents of the backing store will be copied to I<window>, making it visible on screen. Only the part of I<window> contained in I<region> will be modified; that is, drawing operations are clipped to I<region>.

The net result of all this is to remove flicker, because the user sees the finished product appear all at once when you call C<end_draw_frame()>. If you draw to I<window> directly without calling C<begin_draw_frame()>, the user may see flicker as individual drawing operations are performed in sequence.

When using GTK+, the widget system automatically places calls to C<begin_draw_frame()> and C<end_draw_frame()> around emissions of the `GtkWidget::draw` signal. That is, if you’re drawing the contents of the widget yourself, you can assume that the widget has a cleared background, is already set as the clip region, and already has a backing store. Therefore in most cases, application code in GTK does not need to call C<begin_draw_frame()> explicitly.

Returns: a B<Gnome::Gdk3::DrawingContext> context that should be used to draw the contents of the window; the returned context is owned by GDK.

  method begin-draw-frame ( cairo_region_t $region --> N-GObject )

=item $region; a Cairo region
=end pod

method begin-draw-frame ( cairo_region_t $region --> N-GObject ) {
  gdk_window_begin_draw_frame(
    self._get-native-object-no-reffing, $region
  )
}

sub gdk_window_begin_draw_frame (
  N-GObject $window, cairo_region_t $region --> N-GObject
) is native(&gdk-lib)
  { * }
}}

#-------------------------------------------------------------------------------
#TM:0:begin-move-drag:
=begin pod
=head2 begin-move-drag

Begins a window move operation (for a toplevel window).

This function assumes that the drag is controlled by the client pointer device, use C<begin_move_drag_for_device()> to begin a drag with a different device.

  method begin-move-drag (
    Int() $button, Int() $root_x, Int() $root_y, UInt $timestamp
  )

=item $button; the button being used to drag, or 0 for a keyboard-initiated drag
=item $root_x; root window X coordinate of mouse click that began the drag
=item $root_y; root window Y coordinate of mouse click that began the drag
=item $timestamp; timestamp of mouse click that began the drag
=end pod

method begin-move-drag (
  Int() $button, Int() $root_x, Int() $root_y, UInt $timestamp
) {
  gdk_window_begin_move_drag(
    self._get-native-object-no-reffing, $button, $root_x, $root_y, $timestamp
  );
}

sub gdk_window_begin_move_drag (
  N-GObject $window, gint $button, gint $root_x, gint $root_y, guint32 $timestamp
) is native(&gdk-lib)
  { * }

#-------------------------------------------------------------------------------
#TM:0:begin-move-drag-for-device:
=begin pod
=head2 begin-move-drag-for-device

Begins a window move operation (for a toplevel window). You might use this function to implement a “window move grip,” for example. The function works best with window managers that support the [Extended Window Manager Hints](http://www.freedesktop.org/Standards/wm-spec) but has a fallback implementation for other window managers.

  method begin-move-drag-for-device (
    N-GObject() $device, Int() $button,
    Int() $root_x, Int() $root_y, UInt $timestamp
  )

=item $device; the device used for the operation
=item $button; the button being used to drag, or 0 for a keyboard-initiated drag
=item $root_x; root window X coordinate of mouse click that began the drag
=item $root_y; root window Y coordinate of mouse click that began the drag
=item $timestamp; timestamp of mouse click that began the drag
=end pod

method begin-move-drag-for-device (
  N-GObject() $device, Int() $button,
  Int() $root_x, Int() $root_y, UInt $timestamp
) {
  gdk_window_begin_move_drag_for_device(
    self._get-native-object-no-reffing, $device, $button,
    $root_x, $root_y, $timestamp
  );
}

sub gdk_window_begin_move_drag_for_device (
  N-GObject $window, N-GObject $device, gint $button, gint $root_x, gint $root_y, guint32 $timestamp
) is native(&gdk-lib)
  { * }

#-------------------------------------------------------------------------------
#TM:0:begin-resize-drag:
=begin pod
=head2 begin-resize-drag

Begins a window resize operation (for a toplevel window).

This function assumes that the drag is controlled by the client pointer device, use C<begin_resize_drag_for_device()> to begin a drag with a different device.

  method begin-resize-drag (
    GdkWindowEdge $edge, Int() $button,
    Int() $root_x, Int() $root_y, UInt $timestamp
  )

=item $edge; the edge or corner from which the drag is started
=item $button; the button being used to drag, or 0 for a keyboard-initiated drag
=item $root_x; root window X coordinate of mouse click that began the drag
=item $root_y; root window Y coordinate of mouse click that began the drag
=item $timestamp; timestamp of mouse click that began the drag (use C<gdk_event_get_time()>)
=end pod

method begin-resize-drag (
  GdkWindowEdge $edge, Int() $button,
  Int() $root_x, Int() $root_y, UInt $timestamp
) {
  gdk_window_begin_resize_drag(
    self._get-native-object-no-reffing, $edge, $button, $root_x, $root_y, $timestamp
  );
}

sub gdk_window_begin_resize_drag (
  N-GObject $window, GEnum $edge, gint $button, gint $root_x, gint $root_y, guint32 $timestamp
) is native(&gdk-lib)
  { * }

#-------------------------------------------------------------------------------
#TM:0:begin-resize-drag-for-device:
=begin pod
=head2 begin-resize-drag-for-device

Begins a window resize operation (for a toplevel window). You might use this function to implement a “window resize grip,” for example; in fact B<Gnome::Gdk3::Statusbar> uses it. The function works best with window managers that support the [Extended Window Manager Hints](http://www.freedesktop.org/Standards/wm-spec) but has a fallback implementation for other window managers.

  method begin-resize-drag-for-device (
    GdkWindowEdge $edge, N-GObject() $device, Int() $button,
    Int() $root_x, Int() $root_y, UInt $timestamp
  )

=item $edge; the edge or corner from which the drag is started
=item $device; the device used for the operation
=item $button; the button being used to drag, or 0 for a keyboard-initiated drag
=item $root_x; root window X coordinate of mouse click that began the drag
=item $root_y; root window Y coordinate of mouse click that began the drag
=item $timestamp; timestamp of mouse click that began the drag (use C<gdk_event_get_time()>)
=end pod

method begin-resize-drag-for-device (
  GdkWindowEdge $edge, N-GObject() $device, Int() $button,
  Int() $root_x, Int() $root_y, UInt $timestamp
) {
  gdk_window_begin_resize_drag_for_device(
    self._get-native-object-no-reffing, $edge, $device, $button, $root_x, $root_y, $timestamp
  );
}

sub gdk_window_begin_resize_drag_for_device (
  N-GObject $window, GEnum $edge, N-GObject $device, gint $button, gint $root_x, gint $root_y, guint32 $timestamp
) is native(&gdk-lib)
  { * }

#-------------------------------------------------------------------------------
#TM:0:constrain-size:
=begin pod
=head2 constrain-size

Constrains a desired width and height according to a set of geometry hints (such as minimum and maximum size).

  method constrain-size (
    N-GdkGeometry $geometry, GdkWindowHints $flags,
    Int() $width, Int() $height
  )

=item $geometry; a B<Gnome::Gdk3::Geometry> structure
=item $flags; a mask indicating what portions of I<geometry> are set
=item $width; desired width of window
=item $height; desired height of the window
=item $new_width; location to store resulting width
=item $new_height; location to store resulting height
=end pod

method constrain-size (
  N-GdkGeometry $geometry, GdkWindowHints $flags, Int() $width, Int() $height
) {
  gdk_window_constrain_size(
    self._get-native-object-no-reffing, $geometry, $flags,
    $width, $height, my gint $new_width, my gint $new_height
  );
}

sub gdk_window_constrain_size (
  N-GdkGeometry $geometry, GEnum $flags, gint $width, gint $height, gint $new_width is rw, gint $new_height is rw
) is native(&gdk-lib)
  { * }

#-------------------------------------------------------------------------------
#TM:0:coords-from-parent:
=begin pod
=head2 coords-from-parent

Transforms window coordinates from a parent window to a child window, where the parent window is the normal parent as returned by C<get_parent()> for normal windows, and the window's embedder as returned by C<gdk_offscreen_window_get_embedder()> for offscreen windows.

For normal windows, calling this function is equivalent to subtracting the return values of C<get_position()> from the parent coordinates. For offscreen windows however (which can be arbitrarily transformed), this function calls the GdkWindow::from-embedder: signal to translate the coordinates.

You should always use this function when writing generic code that walks down a window hierarchy.

See also: C<coords_to_parent()>

  method coords-from-parent (
    Num() $parent_x, Num() $parent_y, Num() $x, Num() $y
  )

=item $parent_x; X coordinate in parent’s coordinate system
=item $parent_y; Y coordinate in parent’s coordinate system
=item $x; return location for X coordinate in child’s coordinate system
=item $y; return location for Y coordinate in child’s coordinate system
=end pod

method coords-from-parent (
  Num() $parent_x, Num() $parent_y, Num() $x, Num() $y
) {
  gdk_window_coords_from_parent(
    self._get-native-object-no-reffing, $parent_x, $parent_y, $x, $y
  );
}

sub gdk_window_coords_from_parent (
  N-GObject $window, gdouble $parent_x, gdouble $parent_y, gdouble $x, gdouble $y
) is native(&gdk-lib)
  { * }

#-------------------------------------------------------------------------------
#TM:0:coords-to-parent:
=begin pod
=head2 coords-to-parent

Transforms window coordinates from a child window to its parent window, where the parent window is the normal parent as returned by C<get_parent()> for normal windows, and the window's embedder as returned by C<gdk_offscreen_window_get_embedder()> for offscreen windows.

For normal windows, calling this function is equivalent to adding the return values of C<get_position()> to the child coordinates. For offscreen windows however (which can be arbitrarily transformed), this function calls the GdkWindow::to-embedder: signal to translate the coordinates.

You should always use this function when writing generic code that walks up a window hierarchy.

See also: C<coords_from_parent()>

  method coords-to-parent (
    Num() $x, Num() $y, Num() $parent_x, Num() $parent_y
  )

=item $x; X coordinate in child’s coordinate system
=item $y; Y coordinate in child’s coordinate system
=item $parent_x; return location for X coordinate in parent’s coordinate system, or C<undefined>
=item $parent_y; return location for Y coordinate in parent’s coordinate system, or C<undefined>
=end pod

method coords-to-parent (
  Num() $x, Num() $y, Num() $parent_x, Num() $parent_y
) {
  gdk_window_coords_to_parent(
    self._get-native-object-no-reffing, $x, $y, $parent_x, $parent_y
  );
}

sub gdk_window_coords_to_parent (
  N-GObject $window, gdouble $x, gdouble $y, gdouble $parent_x, gdouble $parent_y
) is native(&gdk-lib)
  { * }

#`{{
#-------------------------------------------------------------------------------
#TM:0:create-gl-context:
=begin pod
=head2 create-gl-context

Creates a new B<Gnome::Gdk3::GLContext> matching the framebuffer format to the visual of the B<Gnome::Gdk3::Window>. The context is disconnected from any particular window or surface.

If the creation of the B<Gnome::Gdk3::GLContext> failed, I<error> will be set.

Before using the returned B<Gnome::Gdk3::GLContext>, you will need to call C<gdk_gl_context_make_current()> or C<gdk_gl_context_realize()>.

  method create-gl-context ( --> List )

The returned list holds
=item an error object. This object is valid when there is an error.
=item a native B<Gnome::Gdk3::GLContext>. The object is invalid on error.

=end pod

method create-gl-context ( $error is copy --> N-GObject ) {
  my CArray[N-GError] $error .= new;
  my $no = gdk_window_create_gl_context(
    self._get-native-object-no-reffing, $error
  );

  ( N-GError.new($no.defined ?? $error[0] !! N-GError),
    self._wrap-native-type( 'Gnome::Gdk3::GLContext', $no)
  )
}

sub gdk_window_create_gl_context (
  N-GObject $window, N-GError $error --> N-GObject
) is native(&gdk-lib)
  { * }
}}

#-------------------------------------------------------------------------------
#TM:1:create-similar-image-surface:
=begin pod
=head2 create-similar-image-surface

Create a new image surface that is efficient to draw on the given I<window>.

Initially the surface contents are all 0 (transparent if contents have transparency, black otherwise.)

The I<width> and I<height> of the new surface are not affected by the scaling factor of the I<window>, or by the I<scale> argument; they are the size of the surface in device pixels. If you wish to create an image surface capable of holding the contents of I<window> you can use:

=begin comment
|[<!-- language="C" --> int scale = get_scale_factor (window); int width = get_width (window) * scale; int height = get_height (window) * scale;

// format is set elsewhere cairo_surface_t *surface = create_similar_image_surface (window, format, width, height, scale); ]|
=end comment

Note that unlike C<cairo_surface_create_similar_image()>, the new surface's device scale is set to I<scale>, or to the scale factor of I<window> if I<scale> is 0.

Returns: a pointer to the newly allocated surface. The caller owns the surface and should call C<cairo_surface_destroy()> when done with it.

This function always returns a valid pointer, but it will return a pointer to a “nil” surface if I<other> is already in an error state or any other error occurs.

  method create-similar-image-surface (
    cairo_format_t $format, Int() $width, Int() $height, Int() $scale
    --> cairo_surface_t
  )

=item $format; (type int): the format for the new surface
=item $width; width of the new surface
=item $height; height of the new surface
=item $scale; the scale of the new surface, or 0 to use same as I<window>
=end pod

method create-similar-image-surface (
  cairo_format_t $format, Int() $width, Int() $height, Int() $scale
  --> cairo_surface_t
) {
  gdk_window_create_similar_image_surface(
    self._get-native-object-no-reffing, $format, $width, $height, $scale
  )
}

sub gdk_window_create_similar_image_surface (
  N-GObject $window, GEnum $format, gint $width, gint $height, gint $scale --> cairo_surface_t
) is native(&gdk-lib)
  { * }

#-------------------------------------------------------------------------------
#TM:1:create-similar-surface:
=begin pod
=head2 create-similar-surface

Create a new surface that is as compatible as possible with the given I<window>. For example the new surface will have the same fallback resolution and font options as I<window>. Generally, the new surface will also use the same backend as I<window>, unless that is not possible for some reason. The type of the returned surface may be examined with C<cairo_surface_get_type()>.

Initially the surface contents are all 0 (transparent if contents have transparency, black otherwise.)

Returns: a pointer to the newly allocated surface. The caller owns the surface and should call C<cairo_surface_destroy()> when done with it.

This function always returns a valid pointer, but it will return a pointer to a “nil” surface if I<other> is already in an error state or any other error occurs.

  method create-similar-surface (
    cairo_content_t $content, Int() $width, Int() $height
    --> cairo_surface_t
  )

=item $content; the content for the new surface
=item $width; width of the new surface
=item $height; height of the new surface
=end pod

method create-similar-surface ( cairo_content_t $content, Int() $width, Int() $height --> cairo_surface_t ) {

  gdk_window_create_similar_surface(
    self._get-native-object-no-reffing, $content, $width, $height
  )
}

sub gdk_window_create_similar_surface (
  N-GObject $window, GEnum $content, gint $width, gint $height --> cairo_surface_t
) is native(&gdk-lib)
  { * }

#-------------------------------------------------------------------------------
#TM:1:deiconify:
=begin pod
=head2 deiconify

Attempt to deiconify (unminimize) I<window>. On X11 the window manager may choose to ignore the request to deiconify. When using GTK+, use C<gtk_window_deiconify()> instead of the B<Gnome::Gdk3::Window> variant. Or better yet, you probably want to use C<gtk_window_present_with_time()>, which raises the window, focuses it, unminimizes it, and puts it on the current desktop.

  method deiconify ( )

=end pod

method deiconify ( ) {
  gdk_window_deiconify(self._get-native-object-no-reffing);
}

sub gdk_window_deiconify (
  N-GObject $window
) is native(&gdk-lib)
  { * }

#-------------------------------------------------------------------------------
#TM:1:destroy:
=begin pod
=head2 destroy

Destroys the window system resources associated with I<window> and decrements I<window>'s reference count. The window system resources for all children of I<window> are also destroyed, but the children’s reference counts are not decremented.

Note that a window will not be destroyed automatically when its reference count reaches zero. You must call this function yourself before that happens.

After this call the object becomes invalid and its native object lost.

  method destroy ( )

=end pod

method destroy ( ) {
  gdk_window_destroy(self._get-native-object-no-reffing);
  self._set-native-object(N-GObject);
}

sub gdk_window_destroy (
  N-GObject $window
) is native(&gdk-lib)
  { * }

#`{{
#-------------------------------------------------------------------------------
#TM:0:end-draw-frame:
=begin pod
=head2 end-draw-frame

Indicates that the drawing of the contents of I<window> started with C<begin-draw-frame()> has been completed.

This function will take care of destroying the B<Gnome::Gdk3::DrawingContext>.

It is an error to call this function without a matching C<begin-draw-frame()> first.

  method end-draw-frame ( N-GObject() $context )

=item $context; the B<Gnome::Gdk3::DrawingContext> created by C<begin_draw_frame()>
=end pod

method end-draw-frame ( N-GObject() $context ) {
  gdk_window_end_draw_frame( self._get-native-object-no-reffing, $context);
}

sub gdk_window_end_draw_frame (
  N-GObject $window, N-GObject $context
) is native(&gdk-lib)
  { * }
}}

#-------------------------------------------------------------------------------
#TM:1:ensure-native:
=begin pod
=head2 ensure-native

Tries to ensure that there is a window-system native window for this GdkWindow. This may fail in some situations, returning C<False>.

Offscreen window and children of them can never have native windows.

Some backends may not support native child windows.

Returns: C<True> if the window has a native window, C<False> otherwise

  method ensure-native ( --> Bool )

=end pod

method ensure-native ( --> Bool ) {
  gdk_window_ensure_native(self._get-native-object-no-reffing).Bool
}

sub gdk_window_ensure_native (
  N-GObject $window --> gboolean
) is native(&gdk-lib)
  { * }

#-------------------------------------------------------------------------------
#TM:1:focus:
=begin pod
=head2 focus

Sets keyboard focus to I<window>. In most cases, C<Gnome::Gtk3::Window.present-with-time()> should be used on a B<Gnome::Gtk3::Window>, rather than calling this function.

  method focus ( UInt $timestamp )

=item $timestamp; timestamp of the event triggering the window focus
=end pod

method focus ( UInt $timestamp ) {
  gdk_window_focus( self._get-native-object-no-reffing, $timestamp);
}

sub gdk_window_focus (
  N-GObject $window, guint32 $timestamp
) is native(&gdk-lib)
  { * }

#-------------------------------------------------------------------------------
#TM:0:freeze-updates:
=begin pod
=head2 freeze-updates

Temporarily freezes a window such that it won’t receive expose events. The window will begin receiving expose events again when C<thaw_updates()> is called. If C<freeze_updates()> has been called more than once, C<thaw_updates()> must be called an equal number of times to begin processing exposes.

  method freeze-updates ( )

=end pod

method freeze-updates ( ) {
  gdk_window_freeze_updates(self._get-native-object-no-reffing);
}

sub gdk_window_freeze_updates (
  N-GObject $window
) is native(&gdk-lib)
  { * }

#-------------------------------------------------------------------------------
#TM:1:fullscreen:
=begin pod
=head2 fullscreen

Moves the window into fullscreen mode. This means the window covers the entire screen and is above any panels or task bars.

If the window was already fullscreen, then this function does nothing.

On X11, asks the window manager to put I<window> in a fullscreen state, if the window manager supports this operation. Not all window managers support this, and some deliberately ignore it or don’t have a concept of “fullscreen”; so you can’t rely on the fullscreenification actually happening. But it will happen with most standard window managers, and GDK makes a best effort to get it to happen.

  method fullscreen ( )

=end pod

method fullscreen ( ) {
  gdk_window_fullscreen(self._get-native-object-no-reffing);
}

sub gdk_window_fullscreen (
  N-GObject $window
) is native(&gdk-lib)
  { * }

#-------------------------------------------------------------------------------
#TM:1:fullscreen-on-monitor:
=begin pod
=head2 fullscreen-on-monitor

Moves the window into fullscreen mode on the given monitor. This means the window covers the entire screen and is above any panels or task bars.

If the window was already fullscreen, then this function does nothing.

  method fullscreen-on-monitor ( Int() $monitor )

=item $monitor; Which monitor to display fullscreen on.
=end pod

method fullscreen-on-monitor ( Int() $monitor ) {
  gdk_window_fullscreen_on_monitor(
    self._get-native-object-no-reffing, $monitor
  );
}

sub gdk_window_fullscreen_on_monitor (
  N-GObject $window, gint $monitor
) is native(&gdk-lib)
  { * }

#-------------------------------------------------------------------------------
#TM:0:geometry-changed:
=begin pod
=head2 geometry-changed

This function informs GDK that the geometry of an embedded offscreen window has changed. This is necessary for GDK to keep track of which offscreen window the pointer is in.

  method geometry-changed ( )

=end pod

method geometry-changed ( ) {
  gdk_window_geometry_changed(self._get-native-object-no-reffing);
}

sub gdk_window_geometry_changed (
  N-GObject $window
) is native(&gdk-lib)
  { * }

#-------------------------------------------------------------------------------
#TM:0:get-accept-focus:
=begin pod
=head2 get-accept-focus

Determines whether or not the desktop environment shuld be hinted that the window does not want to receive input focus.

Returns: whether or not the window should receive input focus.

  method get-accept-focus ( --> Bool )

=end pod

method get-accept-focus ( --> Bool ) {
  gdk_window_get_accept_focus(self._get-native-object-no-reffing).Bool
}

sub gdk_window_get_accept_focus (
  N-GObject $window --> gboolean
) is native(&gdk-lib)
  { * }

#-------------------------------------------------------------------------------
#TM:0:get-children:
=begin pod
=head2 get-children

Gets the list of children of I<window> known to GDK. This function only returns children created via GDK, so for example it’s useless when used with the root window; it only returns windows an application created itself.

The returned list must be freed, but the elements in the list need not be.

Returns: (transfer container) (element-type GdkWindow): list of child windows inside I<window>

  method get-children ( --> N-GList )

=end pod

method get-children ( --> N-GList ) {
  gdk_window_get_children(self._get-native-object-no-reffing)
}

sub gdk_window_get_children (
  N-GObject $window --> N-GList
) is native(&gdk-lib)
  { * }

#-------------------------------------------------------------------------------
#TM:0:get-children-with-user-data:
=begin pod
=head2 get-children-with-user-data

Gets the list of children of I<window> known to GDK with a particular I<user_data> set on it.

The returned list must be freed, but the elements in the list need not be.

The list is returned in (relative) stacking order, i.e. the lowest window is first.

Returns: (transfer container) (element-type GdkWindow): list of child windows inside I<window>

  method get-children-with-user-data ( Pointer $user_data --> N-GList )

=item $user_data; user data to look for
=end pod

method get-children-with-user-data ( Pointer $user_data --> N-GList ) {
  gdk_window_get_children_with_user_data(
    self._get-native-object-no-reffing, $user_data
  )
}

sub gdk_window_get_children_with_user_data (
  N-GObject $window, gpointer $user_data --> N-GList
) is native(&gdk-lib)
  { * }

#`{{
#-------------------------------------------------------------------------------
#TM:0:get-clip-region:
=begin pod
=head2 get-clip-region

Computes the region of a window that potentially can be written to by drawing primitives. This region may not take into account other factors such as if the window is obscured by other windows, but no area outside of this region will be affected by drawing primitives.

Returns: a B<cairo_region_t>. This must be freed with C<cairo_region_destroy()> when you are done.

  method get-clip-region ( --> cairo_region_t )

=end pod

method get-clip-region ( --> cairo_region_t ) {
  gdk_window_get_clip_region(self._get-native-object-no-reffing)
}

sub gdk_window_get_clip_region (
  N-GObject $window --> cairo_region_t
) is native(&gdk-lib)
  { * }
}}

#-------------------------------------------------------------------------------
#TM:0:get-cursor:
=begin pod
=head2 get-cursor

Retrieves a B<Gnome::Gdk3::Cursor> pointer for the cursor currently set on the specified B<Gnome::Gdk3::Window>, or C<undefined>. If the return value is C<undefined> then there is no custom cursor set on the specified window, and it is using the cursor for its parent window.

Returns: a B<Gnome::Gdk3::Cursor>, or C<undefined>. The returned object is owned by the B<Gnome::Gdk3::Window> and should not be unreferenced directly. Use C<set_cursor()> to unset the cursor of the window

  method get-cursor ( --> N-GObject )

=end pod

method get-cursor ( --> N-GObject ) {
  gdk_window_get_cursor(self._get-native-object-no-reffing)
}

sub gdk_window_get_cursor (
  N-GObject $window --> N-GObject
) is native(&gdk-lib)
  { * }

#-------------------------------------------------------------------------------
#TM:0:get-decorations:
=begin pod
=head2 get-decorations

Returns the decorations set on the GdkWindow with C<set-decorations()>. Returns -1 if no decorations were set. The returned value is a mask with bits from C<GdkWMDecoration> enumeration.

  method get-decorations ( --> Int )

=end pod

method get-decorations ( --> Int ) {
  my Bool $r = gdk_window_get_decorations(
    self._get-native-object-no-reffing, my GFlag $decorations
  ).Bool;

  $r ?? $decorations !! -1;
}

sub gdk_window_get_decorations (
  N-GObject $window, GFlag $decorations is rw --> gboolean
) is native(&gdk-lib)
  { * }

#-------------------------------------------------------------------------------
#TM:0:get-default-root-window:
=begin pod
=head2 get-default-root-window

Obtains the root window (parent all other windows are inside) for the default display and screen.

Returns: the default root window

  method get-default-root-window ( --> N-GObject )

=end pod

method get-default-root-window ( --> N-GObject ) {
  gdk_get_default_root_window(self._get-native-object-no-reffing)
}

sub gdk_get_default_root_window (
   --> N-GObject
) is native(&gdk-lib)
  { * }

#-------------------------------------------------------------------------------
#TM:0:get-device-cursor:
=begin pod
=head2 get-device-cursor

Retrieves a B<Gnome::Gdk3::Cursor> pointer for the I<device> currently set on the specified B<Gnome::Gdk3::Window>, or C<undefined>. If the return value is C<undefined> then there is no custom cursor set on the specified window, and it is using the cursor for its parent window.

Returns: a B<Gnome::Gdk3::Cursor>, or C<undefined>. The returned object is owned by the B<Gnome::Gdk3::Window> and should not be unreferenced directly. Use C<set_cursor()> to unset the cursor of the window

  method get-device-cursor ( N-GObject() $device --> N-GObject )

=item $device; a master, pointer B<Gnome::Gdk3::Device>.
=end pod

method get-device-cursor ( N-GObject() $device --> N-GObject ) {
  gdk_window_get_device_cursor( self._get-native-object-no-reffing, $device)
}

sub gdk_window_get_device_cursor (
  N-GObject $window, N-GObject $device --> N-GObject
) is native(&gdk-lib)
  { * }

#-------------------------------------------------------------------------------
#TM:0:get-device-events:
=begin pod
=head2 get-device-events

Returns the event mask for I<window> corresponding to an specific device.

Returns: device event mask for I<window>

  method get-device-events ( N-GObject() $device --> GdkEventMask )

=item $device; a B<Gnome::Gdk3::Device>.
=end pod

method get-device-events ( N-GObject() $device --> GdkEventMask ) {
  GdkEventMask(
    gdk_window_get_device_events( self._get-native-object-no-reffing, $device)
  )
}

sub gdk_window_get_device_events (
  N-GObject $window, N-GObject $device --> GEnum
) is native(&gdk-lib)
  { * }

#-------------------------------------------------------------------------------
#TM:0:get-device-position:
=begin pod
=head2 get-device-position

Obtains the current device position and modifier state. The position is given in coordinates relative to the upper left corner of I<window>.

Use C<get_device_position_double()> if you need subpixel precision.

Returns: The window underneath I<device> (as with C<gdk_device_get_window_at_position()>), or C<undefined> if the window is not known to GDK.

  method get-device-position (
    N-GObject() $device, GdkModifierType $mask --> N-GObject
  )

=item $device; pointer B<Gnome::Gdk3::Device> to query to.
=item $x; return location for the X coordinate of I<device>, or C<undefined>.
=item $y; return location for the Y coordinate of I<device>, or C<undefined>.
=item $mask; return location for the modifier mask, or C<undefined>.
=end pod

method get-device-position (
  N-GObject() $device, GdkModifierType $mask --> N-GObject
) {
  gdk_window_get_device_position(
    self._get-native-object-no-reffing, $device, my gint $x, my gint $y, $mask
  )
}

sub gdk_window_get_device_position (
  N-GObject $window, N-GObject $device, gint $x is rw, gint $y is rw, GEnum $mask --> N-GObject
) is native(&gdk-lib)
  { * }

#-------------------------------------------------------------------------------
#TM:0:get-device-position-double:
=begin pod
=head2 get-device-position-double

Obtains the current device position in doubles and modifier state. The position is given in coordinates relative to the upper left corner of I<window>.

Returns: The window underneath I<device> (as with C<gdk_device_get_window_at_position()>), or C<undefined> if the window is not known to GDK.

  method get-device-position-double (
    N-GObject() $device, Num() $x, Num() $y, GdkModifierType $mask
    --> N-GObject
  )

=item $device; pointer B<Gnome::Gdk3::Device> to query to.
=item $x; return location for the X coordinate of I<device>, or C<undefined>.
=item $y; return location for the Y coordinate of I<device>, or C<undefined>.
=item $mask; return location for the modifier mask, or C<undefined>.
=end pod

method get-device-position-double (
  N-GObject() $device, Num() $x, Num() $y,
  GdkModifierType $mask --> N-GObject
) {
  gdk_window_get_device_position_double(
    self._get-native-object-no-reffing, $device, $x, $y, $mask
  )
}

sub gdk_window_get_device_position_double (
  N-GObject $window, N-GObject $device, gdouble $x, gdouble $y, GEnum $mask --> N-GObject
) is native(&gdk-lib)
  { * }

#-------------------------------------------------------------------------------
#TM:0:get-display:
=begin pod
=head2 get-display

Gets the B<Gnome::Gdk3::Display> associated with a B<Gnome::Gdk3::Window>.

Returns: the B<Gnome::Gdk3::Display> associated with I<window>

  method get-display ( --> N-GObject )

=end pod

method get-display ( --> N-GObject ) {
  gdk_window_get_display(self._get-native-object-no-reffing)
}

sub gdk_window_get_display (
  N-GObject $window --> N-GObject
) is native(&gdk-lib)
  { * }

#`{{
#-------------------------------------------------------------------------------
#TM:0:get-drag-protocol:
=begin pod
=head2 get-drag-protocol

Finds out the DND protocol supported by a window.

Returns: the supported DND protocol.

  method get-drag-protocol ( N-GObject() $target --> GdkDragProtocol )

=item $target; location of the window where the drop should happen. This may be I<window> or a proxy window, or C<undefined> if I<window> does not support Drag and Drop.
=end pod

method get-drag-protocol ( N-GObject() $target --> GdkDragProtocol ) {
  GdkDragProtocol(
    gdk_window_get_drag_protocol( self._get-native-object-no-reffing, $target)
  )
}

sub gdk_window_get_drag_protocol (
  N-GObject $window, N-GObject $target --> GEnum
) is native(&gdk-lib)
  { * }
}}

#-------------------------------------------------------------------------------
#TM:0:get-effective-parent:
=begin pod
=head2 get-effective-parent

Obtains the parent of I<window>, as known to GDK. Works like C<get_parent()> for normal windows, but returns the window’s embedder for offscreen windows.

See also: C<gdk_offscreen_window_get_embedder()>

Returns: effective parent of I<window>

  method get-effective-parent ( --> N-GObject )

=end pod

method get-effective-parent ( --> N-GObject ) {
  gdk_window_get_effective_parent(self._get-native-object-no-reffing)
}

sub gdk_window_get_effective_parent (
  N-GObject $window --> N-GObject
) is native(&gdk-lib)
  { * }

#-------------------------------------------------------------------------------
#TM:0:get-effective-toplevel:
=begin pod
=head2 get-effective-toplevel

Gets the toplevel window that’s an ancestor of I<window>.

Works like C<get_toplevel()>, but treats an offscreen window's embedder as its parent, using C<get_effective_parent()>.

See also: C<gdk_offscreen_window_get_embedder()>

Returns: the effective toplevel window containing I<window>

  method get-effective-toplevel ( --> N-GObject )

=end pod

method get-effective-toplevel ( --> N-GObject ) {
  gdk_window_get_effective_toplevel(self._get-native-object-no-reffing)
}

sub gdk_window_get_effective_toplevel (
  N-GObject $window --> N-GObject
) is native(&gdk-lib)
  { * }

#-------------------------------------------------------------------------------
#TM:0:get-event-compression:
=begin pod
=head2 get-event-compression

Get the current event compression setting for this window.

Returns: C<True> if motion events will be compressed

  method get-event-compression ( --> Bool )

=end pod

method get-event-compression ( --> Bool ) {
  gdk_window_get_event_compression(self._get-native-object-no-reffing).Bool
}

sub gdk_window_get_event_compression (
  N-GObject $window --> gboolean
) is native(&gdk-lib)
  { * }

#-------------------------------------------------------------------------------
#TM:0:get-events:
=begin pod
=head2 get-events

Gets the event mask for I<window> for all master input devices. See C<set_events()>.

Returns: event mask for I<window>

  method get-events ( --> GdkEventMask )

=end pod

method get-events ( --> GdkEventMask ) {
  GdkEventMask(
    gdk_window_get_events(self._get-native-object-no-reffing)
  )
}

sub gdk_window_get_events (
  N-GObject $window --> GEnum
) is native(&gdk-lib)
  { * }

#-------------------------------------------------------------------------------
#TM:0:get-focus-on-map:
=begin pod
=head2 get-focus-on-map

Determines whether or not the desktop environment should be hinted that the window does not want to receive input focus when it is mapped.

Returns: whether or not the window wants to receive input focus when it is mapped.

  method get-focus-on-map ( --> Bool )

=end pod

method get-focus-on-map ( --> Bool ) {
  gdk_window_get_focus_on_map(self._get-native-object-no-reffing).Bool
}

sub gdk_window_get_focus_on_map (
  N-GObject $window --> gboolean
) is native(&gdk-lib)
  { * }

#-------------------------------------------------------------------------------
#TM:0:get-frame-clock:
=begin pod
=head2 get-frame-clock

Gets the frame clock for the window. The frame clock for a window never changes unless the window is reparented to a new toplevel window.

Returns: the frame clock

  method get-frame-clock ( --> N-GObject )

=end pod

method get-frame-clock ( --> N-GObject ) {
  gdk_window_get_frame_clock(self._get-native-object-no-reffing)
}

sub gdk_window_get_frame_clock (
  N-GObject $window --> N-GObject
) is native(&gdk-lib)
  { * }

#-------------------------------------------------------------------------------
#TM:0:get-frame-extents:
=begin pod
=head2 get-frame-extents

Obtains the bounding box of the window, including window manager titlebar/borders if any. The frame position is given in root window coordinates. To get the position of the window itself (rather than the frame) in root window coordinates, use C<get_origin()>.

  method get-frame-extents ( N-GObject() $rect )

=item $rect; rectangle to fill with bounding box of the window frame
=end pod

method get-frame-extents ( N-GObject() $rect ) {
  gdk_window_get_frame_extents( self._get-native-object-no-reffing, $rect);
}

sub gdk_window_get_frame_extents (
  N-GObject $window, N-GObject $rect
) is native(&gdk-lib)
  { * }

#-------------------------------------------------------------------------------
#TM:0:get-fullscreen-mode:
=begin pod
=head2 get-fullscreen-mode

Obtains the B<Gnome::Gdk3::FullscreenMode> of the I<window>.

Returns: The B<Gnome::Gdk3::FullscreenMode> applied to the window when fullscreen.

  method get-fullscreen-mode ( --> GdkFullscreenMode )

=end pod

method get-fullscreen-mode ( --> GdkFullscreenMode ) {
  GdkFullscreenMode(
    gdk_window_get_fullscreen_mode(self._get-native-object-no-reffing)
  )
}

sub gdk_window_get_fullscreen_mode (
  N-GObject $window --> GEnum
) is native(&gdk-lib)
  { * }

#-------------------------------------------------------------------------------
#TM:1:get-geometry:
=begin pod
=head2 get-geometry

Any of the return location arguments to this function may be C<undefined>, if you aren’t interested in getting the value of that field.

The X and Y coordinates returned are relative to the parent window of I<window>, which for toplevels usually means relative to the window decorations (titlebar, etc.) rather than relative to the root window (screen-size background window).

On the X11 platform, the geometry is obtained from the X server, so reflects the latest position of I<window>; this may be out-of-sync with the position of I<window> delivered in the most-recently-processed B<Gnome::Gdk3::EventConfigure>. C<get_position()> in contrast gets the position from the most recent configure event.

Note: If I<window> is not a toplevel, it is much better to call C<get_position()>, C<get_width()> and C<get_height()> instead, because it avoids the roundtrip to the X server and because these functions support the full 32-bit coordinate space, whereas C<get_geometry()> is restricted to the 16-bit coordinates of X11.

  method get-geometry ( --> List )

The List returns
=item Int; X coordinate of window (relative to its parent)
=item Int; Y coordinate of window (relative to its parent)
=item Int; width of window
=item Int; height of window
=end pod

method get-geometry ( ) {
  gdk_window_get_geometry(
    self._get-native-object-no-reffing, my gint $x, my gint $y, my gint $width, my gint $height
  );

  ( $x, $y, $width, $height)
}

sub gdk_window_get_geometry (
  N-GObject $window, gint $x is rw, gint $y is rw, gint $width is rw, gint $height is rw
) is native(&gdk-lib)
  { * }

#-------------------------------------------------------------------------------
#TM:0:get-group:
=begin pod
=head2 get-group

Returns the group leader window for I<window>. See C<set_group()>.

Returns: the group leader window for I<window>

  method get-group ( --> N-GObject )

=end pod

method get-group ( --> N-GObject ) {

  gdk_window_get_group(
    self._get-native-object-no-reffing,
  )
}

sub gdk_window_get_group (
  N-GObject $window --> N-GObject
) is native(&gdk-lib)
  { * }

#-------------------------------------------------------------------------------
#TM:0:get-height:
=begin pod
=head2 get-height

Returns the height of the given I<window>.

On the X11 platform the returned size is the size reported in the most-recently-processed configure event, rather than the current size on the X server.

Returns: The height of I<window>

  method get-height ( --> Int )

=end pod

method get-height ( --> Int ) {

  gdk_window_get_height(
    self._get-native-object-no-reffing,
  )
}

sub gdk_window_get_height (
  N-GObject $window --> gint
) is native(&gdk-lib)
  { * }

#-------------------------------------------------------------------------------
#TM:0:get-modal-hint:
=begin pod
=head2 get-modal-hint

Determines whether or not the window manager is hinted that I<window> has modal behaviour.

Returns: whether or not the window has the modal hint set.

  method get-modal-hint ( --> Bool )

=end pod

method get-modal-hint ( --> Bool ) {

  gdk_window_get_modal_hint(
    self._get-native-object-no-reffing,
  ).Bool
}

sub gdk_window_get_modal_hint (
  N-GObject $window --> gboolean
) is native(&gdk-lib)
  { * }

#-------------------------------------------------------------------------------
#TM:0:get-origin:
=begin pod
=head2 get-origin

Obtains the position of a window in root window coordinates. (Compare with C<get_position()> and C<get_geometry()> which return the position of a window relative to its parent window.)

Returns: not meaningful, ignore

  method get-origin ( --> Int )

=item $x; return location for X coordinate
=item $y; return location for Y coordinate
=end pod

method get-origin ( --> Int ) {

  gdk_window_get_origin(
    self._get-native-object-no-reffing, my gint $x, my gint $y
  )
}

sub gdk_window_get_origin (
  N-GObject $window, gint $x is rw, gint $y is rw --> gint
) is native(&gdk-lib)
  { * }

#-------------------------------------------------------------------------------
#TM:0:get-parent:
=begin pod
=head2 get-parent

Obtains the parent of I<window>, as known to GDK. Does not query the X server; thus this returns the parent as passed to C<new()>, not the actual parent. This should never matter unless you’re using Xlib calls mixed with GDK calls on the X11 platform. It may also matter for toplevel windows, because the window manager may choose to reparent them.

Note that you should use C<get_effective_parent()> when writing generic code that walks up a window hierarchy, because C<get_parent()> will most likely not do what you expect if there are offscreen windows in the hierarchy.

Returns: parent of I<window>

  method get-parent ( --> N-GObject )

=end pod

method get-parent ( --> N-GObject ) {

  gdk_window_get_parent(
    self._get-native-object-no-reffing,
  )
}

sub gdk_window_get_parent (
  N-GObject $window --> N-GObject
) is native(&gdk-lib)
  { * }

#-------------------------------------------------------------------------------
#TM:0:get-pass-through:
=begin pod
=head2 get-pass-through

Returns whether input to the window is passed through to the window below.

See C<set_pass_through()> for details

  method get-pass-through ( --> Bool )

=end pod

method get-pass-through ( --> Bool ) {

  gdk_window_get_pass_through(
    self._get-native-object-no-reffing,
  ).Bool
}

sub gdk_window_get_pass_through (
  N-GObject $window --> gboolean
) is native(&gdk-lib)
  { * }

#-------------------------------------------------------------------------------
#TM:0:get-position:
=begin pod
=head2 get-position

Obtains the position of the window as reported in the most-recently-processed B<Gnome::Gdk3::EventConfigure>. Contrast with C<get_geometry()> which queries the X server for the current window position, regardless of which events have been received or processed.

The position coordinates are relative to the window’s parent window.

  method get-position ( )

=item $x; X coordinate of window
=item $y; Y coordinate of window
=end pod

method get-position ( ) {

  gdk_window_get_position(
    self._get-native-object-no-reffing, my gint $x, my gint $y
  );
}

sub gdk_window_get_position (
  N-GObject $window, gint $x is rw, gint $y is rw
) is native(&gdk-lib)
  { * }

#-------------------------------------------------------------------------------
#TM:0:get-root-coords:
=begin pod
=head2 get-root-coords

Obtains the position of a window position in root window coordinates. This is similar to C<get_origin()> but allows you to pass in any position in the window, not just the origin.

  method get-root-coords ( Int() $x, Int() $y )

=item $x; X coordinate in window
=item $y; Y coordinate in window
=item $root_x; return location for X coordinate
=item $root_y; return location for Y coordinate
=end pod

method get-root-coords ( Int() $x, Int() $y ) {

  gdk_window_get_root_coords(
    self._get-native-object-no-reffing, $x, $y, my gint $root_x, my gint $root_y
  );
}

sub gdk_window_get_root_coords (
  N-GObject $window, gint $x, gint $y, gint $root_x is rw, gint $root_y is rw
) is native(&gdk-lib)
  { * }

#-------------------------------------------------------------------------------
#TM:0:get-root-origin:
=begin pod
=head2 get-root-origin

Obtains the top-left corner of the window manager frame in root window coordinates.

  method get-root-origin ( )

=item $x; return location for X position of window frame
=item $y; return location for Y position of window frame
=end pod

method get-root-origin ( ) {

  gdk_window_get_root_origin(
    self._get-native-object-no-reffing, my gint $x, my gint $y
  );
}

sub gdk_window_get_root_origin (
  N-GObject $window, gint $x is rw, gint $y is rw
) is native(&gdk-lib)
  { * }

#-------------------------------------------------------------------------------
#TM:0:get-scale-factor:
=begin pod
=head2 get-scale-factor

Returns the internal scale factor that maps from window coordiantes to the actual device pixels. On traditional systems this is 1, but on very high density outputs this can be a higher value (often 2).

A higher value means that drawing is automatically scaled up to a higher resolution, so any code doing drawing will automatically look nicer. However, if you are supplying pixel-based data the scale value can be used to determine whether to use a pixel resource with higher resolution data.

The scale of a window may change during runtime, if this happens a configure event will be sent to the toplevel window.

Returns: the scale factor

  method get-scale-factor ( --> Int )

=end pod

method get-scale-factor ( --> Int ) {

  gdk_window_get_scale_factor(
    self._get-native-object-no-reffing,
  )
}

sub gdk_window_get_scale_factor (
  N-GObject $window --> gint
) is native(&gdk-lib)
  { * }

#-------------------------------------------------------------------------------
#TM:0:get-screen:
=begin pod
=head2 get-screen

Gets the B<Gnome::Gdk3::Screen> associated with a B<Gnome::Gdk3::Window>.

Returns: the B<Gnome::Gdk3::Screen> associated with I<window>

  method get-screen ( --> N-GObject )

=end pod

method get-screen ( --> N-GObject ) {

  gdk_window_get_screen(
    self._get-native-object-no-reffing,
  )
}

sub gdk_window_get_screen (
  N-GObject $window --> N-GObject
) is native(&gdk-lib)
  { * }

#`{{
#-------------------------------------------------------------------------------
#TM:0:get-source-events:
=begin pod
=head2 get-source-events

Returns the event mask for I<window> corresponding to the device class specified by I<source>.

Returns: source event mask for I<window>

  method get-source-events ( GdkInputSource $source --> GdkEventMask )

=item $source; a B<Gnome::Gdk3::InputSource> to define the source class.
=end pod

method get-source-events ( GdkInputSource $source --> GdkEventMask ) {
  GdkEventMask(
    gdk_window_get_source_events( self._get-native-object-no-reffing, $source)
  )
}

sub gdk_window_get_source_events (
  N-GObject $window, GEnum $source --> GEnum
) is native(&gdk-lib)
  { * }
}}

#-------------------------------------------------------------------------------
#TM:0:get-state:
=begin pod
=head2 get-state

Gets the bitwise OR of the currently active window state flags, from the B<Gnome::Gdk3::WindowState> enumeration.

Returns: window state bitfield

  method get-state ( --> GdkWindowState )

=end pod

method get-state ( --> GdkWindowState ) {
  GdkWindowState(gdk_window_get_state(self._get-native-object-no-reffing))
}

sub gdk_window_get_state (
  N-GObject $window --> GEnum
) is native(&gdk-lib)
  { * }

#-------------------------------------------------------------------------------
#TM:0:get-support-multidevice:
=begin pod
=head2 get-support-multidevice

Returns C<True> if the window is aware of the existence of multiple devices.

Returns: C<True> if the window handles multidevice features.

  method get-support-multidevice ( --> Bool )

=end pod

method get-support-multidevice ( --> Bool ) {

  gdk_window_get_support_multidevice(
    self._get-native-object-no-reffing,
  ).Bool
}

sub gdk_window_get_support_multidevice (
  N-GObject $window --> gboolean
) is native(&gdk-lib)
  { * }

#-------------------------------------------------------------------------------
#TM:0:get-toplevel:
=begin pod
=head2 get-toplevel

Gets the toplevel window that’s an ancestor of I<window>.

Any window type but C<GDK_WINDOW_CHILD> is considered a toplevel window, as is a C<GDK_WINDOW_CHILD> window that has a root window as parent.

Note that you should use C<get_effective_toplevel()> when you want to get to a window’s toplevel as seen on screen, because C<get_toplevel()> will most likely not do what you expect if there are offscreen windows in the hierarchy.

Returns: the toplevel window containing I<window>

  method get-toplevel ( --> N-GObject )

=end pod

method get-toplevel ( --> N-GObject ) {

  gdk_window_get_toplevel(
    self._get-native-object-no-reffing,
  )
}

sub gdk_window_get_toplevel (
  N-GObject $window --> N-GObject
) is native(&gdk-lib)
  { * }

#-------------------------------------------------------------------------------
#TM:0:get-type-hint:
=begin pod
=head2 get-type-hint

This function returns the type hint set for a window.

Returns: The type hint set for I<window>

  method get-type-hint ( --> GdkWindowTypeHint )

=end pod

method get-type-hint ( --> GdkWindowTypeHint ) {
  GdkWindowTypeHint(
    gdk_window_get_type_hint(self._get-native-object-no-reffing)
  )
}

sub gdk_window_get_type_hint (
  N-GObject $window --> GEnum
) is native(&gdk-lib)
  { * }

#`{{
#-------------------------------------------------------------------------------
#TM:0:get-update-area:
=begin pod
=head2 get-update-area

Transfers ownership of the update area from I<window> to the caller of the function. That is, after calling this function, I<window> will no longer have an invalid/dirty region; the update area is removed from I<window> and handed to you. If a window has no update area, C<get_update_area()> returns C<undefined>. You are responsible for calling C<cairo_region_destroy()> on the returned region if it’s non-C<undefined>.

Returns: the update area for I<window>

  method get-update-area ( --> cairo_region_t )

=end pod

method get-update-area ( --> cairo_region_t ) {
  gdk_window_get_update_area(
    self._get-native-object-no-reffing,
  )
}

sub gdk_window_get_update_area (
  N-GObject $window --> cairo_region_t
) is native(&gdk-lib)
  { * }
}}

#-------------------------------------------------------------------------------
#TM:0:get-user-data:
=begin pod
=head2 get-user-data

Retrieves the user data for I<window>, which is normally the widget that I<window> belongs to. See C<set_user_data()>.

  method get-user-data ( Pointer $data )

=item $data; return location for user data
=end pod

method get-user-data ( Pointer $data ) {
  gdk_window_get_user_data( self._get-native-object-no-reffing, $data);
}

sub gdk_window_get_user_data (
  N-GObject $window, gpointer $data
) is native(&gdk-lib)
  { * }

#`{{
#-------------------------------------------------------------------------------
#TM:0:get-visible-region:
=begin pod
=head2 get-visible-region

Computes the region of the I<window> that is potentially visible. This does not necessarily take into account if the window is obscured by other windows, but no area outside of this region is visible.

Returns: a B<cairo_region_t>. This must be freed with C<cairo_region_destroy()> when you are done.

  method get-visible-region ( --> cairo_region_t )

=end pod

method get-visible-region ( --> cairo_region_t ) {
  gdk_window_get_visible_region(self._get-native-object-no-reffing)
}

sub gdk_window_get_visible_region (
  N-GObject $window --> cairo_region_t
) is native(&gdk-lib)
  { * }
}}

#-------------------------------------------------------------------------------
#TM:0:get-visual:
=begin pod
=head2 get-visual

Gets the B<Gnome::Gdk3::Visual> describing the pixel format of I<window>.

Returns: a native B<Gnome::Gdk3::Visual>

  method get-visual ( --> N-GObject )

=end pod

method get-visual ( --> N-GObject ) {
  gdk_window_get_visual(self._get-native-object-no-reffing)
}

sub gdk_window_get_visual (
  N-GObject $window --> N-GObject
) is native(&gdk-lib)
  { * }

#-------------------------------------------------------------------------------
#TM:0:get-width:
=begin pod
=head2 get-width

Returns the width of the given I<window>.

On the X11 platform the returned size is the size reported in the most-recently-processed configure event, rather than the current size on the X server.

Returns: The width of I<window>

  method get-width ( --> Int )

=end pod

method get-width ( --> Int ) {
  gdk_window_get_width(self._get-native-object-no-reffing)
}

sub gdk_window_get_width (
  N-GObject $window --> gint
) is native(&gdk-lib)
  { * }

#-------------------------------------------------------------------------------
#TM:1:get-window-type:
=begin pod
=head2 get-window-type

Gets the type of the window. See B<Gnome::Gdk3::WindowType>.

Returns: type of window

  method get-window-type ( --> GdkWindowType )

=end pod

method get-window-type ( --> GdkWindowType ) {
  GdkWindowType(
    gdk_window_get_window_type(self._get-native-object-no-reffing)
  )
}

sub gdk_window_get_window_type (
  N-GObject $window --> GEnum
) is native(&gdk-lib)
  { * }

#-------------------------------------------------------------------------------
#TM:0:has-native:
=begin pod
=head2 has-native

Checks whether the window has a native window or not. Note that you can use C<ensure_native()> if a native window is needed.

Returns: C<True> if the I<window> has a native window, C<False> otherwise.

  method has-native ( --> Bool )

=end pod

method has-native ( --> Bool ) {
  gdk_window_has_native(self._get-native-object-no-reffing).Bool
}

sub gdk_window_has_native (
  N-GObject $window --> gboolean
) is native(&gdk-lib)
  { * }

#-------------------------------------------------------------------------------
#TM:0:hide:
=begin pod
=head2 hide

For toplevel windows, withdraws them, so they will no longer be known to the window manager; for all windows, unmaps them, so they won’t be displayed. Normally done automatically as part of C<gtk_widget_hide()>.

  method hide ( )

=end pod

method hide ( ) {
  gdk_window_hide(self._get-native-object-no-reffing);
}

sub gdk_window_hide (
  N-GObject $window
) is native(&gdk-lib)
  { * }

#-------------------------------------------------------------------------------
#TM:0:iconify:
=begin pod
=head2 iconify

Asks to iconify (minimize) I<window>. The window manager may choose to ignore the request, but normally will honor it. Using C<gtk_window_iconify()> is preferred, if you have a B<Gnome::Gdk3::Window> widget.

This function only makes sense when I<window> is a toplevel window.

  method iconify ( )

=end pod

method iconify ( ) {
  gdk_window_iconify(self._get-native-object-no-reffing);
}

sub gdk_window_iconify (
  N-GObject $window
) is native(&gdk-lib)
  { * }

#`{{
#-------------------------------------------------------------------------------
#TM:0:input-shape-combine-region:
=begin pod
=head2 input-shape-combine-region

Like C<shape_combine_region()>, but the shape applies only to event handling. Mouse events which happen while the pointer position corresponds to an unset bit in the mask will be passed on the window below I<window>.

An input shape is typically used with RGBA windows. The alpha channel of the window defines which pixels are invisible and allows for nicely antialiased borders, and the input shape controls where the window is “clickable”.

On the X11 platform, this requires version 1.1 of the shape extension.

On the Win32 platform, this functionality is not present and the function does nothing.

  method input-shape-combine-region ( cairo_region_t $shape_region, Int() $offset_x, Int() $offset_y )

=item $shape_region; region of window to be non-transparent
=item $offset_x; X position of I<shape_region> in I<window> coordinates
=item $offset_y; Y position of I<shape_region> in I<window> coordinates
=end pod

method input-shape-combine-region (
  cairo_region_t $shape_region, Int() $offset_x, Int() $offset_y
) {
  gdk_window_input_shape_combine_region(
    self._get-native-object-no-reffing, $shape_region, $offset_x, $offset_y
  );
}

sub gdk_window_input_shape_combine_region (
  N-GObject $window, cairo_region_t $shape_region, gint $offset_x, gint $offset_y
) is native(&gdk-lib)
  { * }
}}

#`{{
#-------------------------------------------------------------------------------
#TM:0:invalidate-maybe-recurse:
=begin pod
=head2 invalidate-maybe-recurse

Adds I<region> to the update area for I<window>. The update area is the region that needs to be redrawn, or “dirty region.” The call C<process_updates()> sends one or more expose events to the window, which together cover the entire update area. An application would normally redraw the contents of I<window> in response to those expose events.

GDK will call C<process_all_updates()> on your behalf whenever your program returns to the main loop and becomes idle, so normally there’s no need to do that manually, you just need to invalidate regions that you know should be redrawn.

The I<child_func> parameter controls whether the region of each child window that intersects I<region> will also be invalidated. Only children for which I<child_func> returns B<TRUE> will have the area invalidated.

  method invalidate-maybe-recurse ( cairo_region_t $region, GdkWindowChildFunc $child_func, Pointer $user_data )

=item $region; a B<cairo_region_t>
=item $child_func; (scope call) : function to use to decide if to recurse to a child, C<undefined> means never recurse.
=item $user_data; data passed to I<child_func>
=end pod

method invalidate-maybe-recurse ( cairo_region_t $region, GdkWindowChildFunc $child_func, Pointer $user_data ) {

  gdk_window_invalidate_maybe_recurse(
    self._get-native-object-no-reffing, $region, $child_func, $user_data
  );
}

sub gdk_window_invalidate_maybe_recurse (
  N-GObject $window, cairo_region_t $region, GdkWindowChildFunc $child_func, gpointer $user_data
) is native(&gdk-lib)
  { * }
}}

#-------------------------------------------------------------------------------
#TM:0:invalidate-rect:
=begin pod
=head2 invalidate-rect

A convenience wrapper around C<invalidate_region()> which invalidates a rectangular region. See C<invalidate_region()> for details.

  method invalidate-rect ( N-GObject() $rect, Bool $invalidate_children )

=item $rect; rectangle to invalidate or C<undefined> to invalidate the whole window
=item $invalidate_children; whether to also invalidate child windows
=end pod

method invalidate-rect ( N-GObject() $rect, Bool $invalidate_children ) {
  gdk_window_invalidate_rect(
    self._get-native-object-no-reffing, $rect, $invalidate_children
  );
}

sub gdk_window_invalidate_rect (
  N-GObject $window, N-GObject $rect, gboolean $invalidate_children
) is native(&gdk-lib)
  { * }

#`{{
#-------------------------------------------------------------------------------
#TM:0:invalidate-region:
=begin pod
=head2 invalidate-region

Adds I<region> to the update area for I<window>. The update area is the region that needs to be redrawn, or “dirty region.” The call C<process_updates()> sends one or more expose events to the window, which together cover the entire update area. An application would normally redraw the contents of I<window> in response to those expose events.

GDK will call C<process_all_updates()> on your behalf whenever your program returns to the main loop and becomes idle, so normally there’s no need to do that manually, you just need to invalidate regions that you know should be redrawn.

The I<invalidate_children> parameter controls whether the region of each child window that intersects I<region> will also be invalidated. If C<False>, then the update area for child windows will remain unaffected. See invalidate_maybe_recurse if you need fine grained control over which children are invalidated.

  method invalidate-region ( cairo_region_t $region, Bool $invalidate_children )

=item $region; a B<cairo_region_t>
=item $invalidate_children; C<True> to also invalidate child windows
=end pod

method invalidate-region ( cairo_region_t $region, Bool $invalidate_children ) {
  gdk_window_invalidate_region(
    self._get-native-object-no-reffing, $region, $invalidate_children
  );
}

sub gdk_window_invalidate_region (
  N-GObject $window, cairo_region_t $region, gboolean $invalidate_children
) is native(&gdk-lib)
  { * }
}}

#-------------------------------------------------------------------------------
#TM:0:is-destroyed:
=begin pod
=head2 is-destroyed

Check to see if a window is destroyed..

Returns: C<True> if the window is destroyed

  method is-destroyed ( --> Bool )

=end pod

method is-destroyed ( --> Bool ) {
  gdk_window_is_destroyed(self._get-native-object-no-reffing).Bool
}

sub gdk_window_is_destroyed (
  N-GObject $window --> gboolean
) is native(&gdk-lib)
  { * }

#-------------------------------------------------------------------------------
#TM:0:is-input-only:
=begin pod
=head2 is-input-only

Determines whether or not the window is an input only window.

Returns: C<True> if I<window> is input only

  method is-input-only ( --> Bool )

=end pod

method is-input-only ( --> Bool ) {
  gdk_window_is_input_only(self._get-native-object-no-reffing).Bool
}

sub gdk_window_is_input_only (
  N-GObject $window --> gboolean
) is native(&gdk-lib)
  { * }

#-------------------------------------------------------------------------------
#TM:0:is-shaped:
=begin pod
=head2 is-shaped

Determines whether or not the window is shaped.

Returns: C<True> if I<window> is shaped

  method is-shaped ( --> Bool )

=end pod

method is-shaped ( --> Bool ) {
  gdk_window_is_shaped(self._get-native-object-no-reffing).Bool
}

sub gdk_window_is_shaped (
  N-GObject $window --> gboolean
) is native(&gdk-lib)
  { * }

#-------------------------------------------------------------------------------
#TM:0:is-viewable:
=begin pod
=head2 is-viewable

Check if the window and all ancestors of the window are mapped. (This is not necessarily "viewable" in the X sense, since we only check as far as we have GDK window parents, not to the root window.)

Returns: C<True> if the window is viewable

  method is-viewable ( --> Bool )

=end pod

method is-viewable ( --> Bool ) {
  gdk_window_is_viewable(self._get-native-object-no-reffing).Bool
}

sub gdk_window_is_viewable (
  N-GObject $window --> gboolean
) is native(&gdk-lib)
  { * }

#-------------------------------------------------------------------------------
#TM:0:is-visible:
=begin pod
=head2 is-visible

Checks whether the window has been mapped (with C<show()> or C<show_unraised()>).

Returns: C<True> if the window is mapped

  method is-visible ( --> Bool )

=end pod

method is-visible ( --> Bool ) {
  gdk_window_is_visible(self._get-native-object-no-reffing).Bool
}

sub gdk_window_is_visible (
  N-GObject $window --> gboolean
) is native(&gdk-lib)
  { * }

#-------------------------------------------------------------------------------
#TM:0:lower:
=begin pod
=head2 lower

Lowers I<window> to the bottom of the Z-order (stacking order), so that other windows with the same parent window appear above I<window>. This is true whether or not the other windows are visible.

If I<window> is a toplevel, the window manager may choose to deny the request to move the window in the Z-order, C<lower()> only requests the restack, does not guarantee it.

Note that C<show()> raises the window again, so don’t call this function before C<show()>. (Try C<show_unraised()>.)

  method lower ( )

=end pod

method lower ( ) {
  gdk_window_lower(self._get-native-object-no-reffing);
}

sub gdk_window_lower (
  N-GObject $window
) is native(&gdk-lib)
  { * }

#-------------------------------------------------------------------------------
#TM:0:mark-paint-from-clip:
=begin pod
=head2 mark-paint-from-clip

If you call this during a paint (e.g. between C<begin_paint_region()> and C<end_paint()> then GDK will mark the current clip region of the window as being drawn. This is required when mixing GL rendering via C<gdk_cairo_draw_from_gl()> and cairo rendering, as otherwise GDK has no way of knowing when something paints over the GL-drawn regions.

This is typically called automatically by GTK+ and you don't need to care about this.

  method mark-paint-from-clip ( cairo_t $cr )

=item $cr; a B<cairo_t>
=end pod

method mark-paint-from-clip ( cairo_t $cr ) {
  gdk_window_mark_paint_from_clip( self._get-native-object-no-reffing, $cr);
}

sub gdk_window_mark_paint_from_clip (
  N-GObject $window, cairo_t $cr
) is native(&gdk-lib)
  { * }

#-------------------------------------------------------------------------------
#TM:0:maximize:
=begin pod
=head2 maximize

Maximizes the window. If the window was already maximized, then this function does nothing.

On X11, asks the window manager to maximize I<window>, if the window manager supports this operation. Not all window managers support this, and some deliberately ignore it or don’t have a concept of “maximized”; so you can’t rely on the maximization actually happening. But it will happen with most standard window managers, and GDK makes a best effort to get it to happen.

On Windows, reliably maximizes the window.

  method maximize ( )

=end pod

method maximize ( ) {
  gdk_window_maximize(self._get-native-object-no-reffing);
}

sub gdk_window_maximize (
  N-GObject $window
) is native(&gdk-lib)
  { * }

#-------------------------------------------------------------------------------
#TM:0:merge-child-input-shapes:
=begin pod
=head2 merge-child-input-shapes

Merges the input shape masks for any child windows into the input shape mask for I<window>. i.e. the union of all input masks for I<window> and its children will become the new input mask for I<window>. See C<input_shape_combine_region()>.

This function is distinct from C<set_child_input_shapes()> because it includes I<window>’s input shape mask in the set of shapes to be merged.

  method merge-child-input-shapes ( )

=end pod

method merge-child-input-shapes ( ) {
  gdk_window_merge_child_input_shapes(self._get-native-object-no-reffing);
}

sub gdk_window_merge_child_input_shapes (
  N-GObject $window
) is native(&gdk-lib)
  { * }

#-------------------------------------------------------------------------------
#TM:0:merge-child-shapes:
=begin pod
=head2 merge-child-shapes

Merges the shape masks for any child windows into the shape mask for I<window>. i.e. the union of all masks for I<window> and its children will become the new mask for I<window>. See C<shape_combine_region()>.

This function is distinct from C<set_child_shapes()> because it includes I<window>’s shape mask in the set of shapes to be merged.

  method merge-child-shapes ( )

=end pod

method merge-child-shapes ( ) {
  gdk_window_merge_child_shapes(self._get-native-object-no-reffing);
}

sub gdk_window_merge_child_shapes (
  N-GObject $window
) is native(&gdk-lib)
  { * }

#-------------------------------------------------------------------------------
#TM:0:move:
=begin pod
=head2 move

Repositions a window relative to its parent window. For toplevel windows, window managers may ignore or modify the move; you should probably use C<gtk_window_move()> on a B<Gnome::Gdk3::Window> widget anyway, instead of using GDK functions. For child windows, the move will reliably succeed.

If you’re also planning to resize the window, use C<move_resize()> to both move and resize simultaneously, for a nicer visual effect.

  method move ( Int() $x, Int() $y )

=item $x; X coordinate relative to window’s parent
=item $y; Y coordinate relative to window’s parent
=end pod

method move ( Int() $x, Int() $y ) {
  gdk_window_move( self._get-native-object-no-reffing, $x, $y);
}

sub gdk_window_move (
  N-GObject $window, gint $x, gint $y
) is native(&gdk-lib)
  { * }

#`{{
#-------------------------------------------------------------------------------
#TM:0:move-region:
=begin pod
=head2 move-region

Move the part of I<window> indicated by I<region> by I<dy> pixels in the Y direction and I<dx> pixels in the X direction. The portions of I<region> that not covered by the new position of I<region> are invalidated.

Child windows are not moved.

  method move-region ( cairo_region_t $region, Int() $dx, Int() $dy )

=item $region; The B<cairo_region_t> to move
=item $dx; Amount to move in the X direction
=item $dy; Amount to move in the Y direction
=end pod

method move-region ( cairo_region_t $region, Int() $dx, Int() $dy ) {
  gdk_window_move_region(
    self._get-native-object-no-reffing, $region, $dx, $dy
  );
}

sub gdk_window_move_region (
  N-GObject $window, cairo_region_t $region, gint $dx, gint $dy
) is native(&gdk-lib)
  { * }
}}

#-------------------------------------------------------------------------------
#TM:0:move-resize:
=begin pod
=head2 move-resize

Equivalent to calling C<move()> and C<resize()>, except that both operations are performed at once, avoiding strange visual effects. (i.e. the user may be able to see the window first move, then resize, if you don’t use C<move_resize()>.)

  method move-resize ( Int() $x, Int() $y, Int() $width, Int() $height )

=item $x; new X position relative to window’s parent
=item $y; new Y position relative to window’s parent
=item $width; new width
=item $height; new height
=end pod

method move-resize ( Int() $x, Int() $y, Int() $width, Int() $height ) {
  gdk_window_move_resize(
    self._get-native-object-no-reffing, $x, $y, $width, $height
  );
}

sub gdk_window_move_resize (
  N-GObject $window, gint $x, gint $y, gint $width, gint $height
) is native(&gdk-lib)
  { * }

#-------------------------------------------------------------------------------
#TM:0:move-to-rect:
=begin pod
=head2 move-to-rect

Moves I<window> to I<rect>, aligning their anchor points.

I<rect> is relative to the top-left corner of the window that I<window> is transient for. I<rect_anchor> and I<window_anchor> determine anchor points on I<rect> and I<window> to pin together. I<rect>'s anchor point can optionally be offset by I<rect_anchor_dx> and I<rect_anchor_dy>, which is equivalent to offsetting the position of I<window>.

I<anchor_hints> determines how I<window> will be moved if the anchor points cause it to move off-screen. For example, C<GDK_ANCHOR_FLIP_X> will replace C<GDK_GRAVITY_NORTH_WEST> with C<GDK_GRAVITY_NORTH_EAST> and vice versa if I<window> extends beyond the left or right edges of the monitor.

Connect to the  I<moved-to-rect> signal to find out how it was actually positioned.

  method move-to-rect (
    N-GdkRectangle $rect, GdkGravity $rect_anchor,
    GdkGravity $window_anchor, GdkAnchorHints $anchor_hints,
    Int() $rect_anchor_dx, Int() $rect_anchor_dy
  )

=item $rect; the destination rectangle to align I<window> with
=item $rect_anchor; the point on I<rect> to align with I<window>'s anchor point
=item $window_anchor; the point on I<window> to align with I<rect>'s anchor point
=item $anchor_hints; positioning hints to use when limited on space
=item $rect_anchor_dx; horizontal offset to shift I<window>, i.e. I<rect>'s anchor point
=item $rect_anchor_dy; vertical offset to shift I<window>, i.e. I<rect>'s anchor point
=end pod

method move-to-rect (
  N-GdkRectangle $rect, GdkGravity $rect_anchor, GdkGravity $window_anchor,
  GdkAnchorHints $anchor_hints, Int() $rect_anchor_dx, Int() $rect_anchor_dy
) {
  gdk_window_move_to_rect(
    self._get-native-object-no-reffing, $rect, $rect_anchor, $window_anchor,
    $anchor_hints, $rect_anchor_dx, $rect_anchor_dy
  );
}

sub gdk_window_move_to_rect (
  N-GObject $window, N-GdkRectangle $rect, GEnum $rect_anchor,
  GEnum $window_anchor, GEnum $anchor_hints, gint $rect_anchor_dx,
  gint $rect_anchor_dy
) is native(&gdk-lib)
  { * }

#-------------------------------------------------------------------------------
#TM:0:offscreen-window-get-embedder:
=begin pod
=head2 offscreen-window-get-embedder

Gets the window that window is embedded in.

  method gdk-offscreen-window-get-embedder ( --> N-GObject )

=end pod

method offscreen-window-get-embedder ( --> N-GObject ) {
  gdk_offscreen_window_get_embedder(self._get-native-object-no-reffing)
}

sub gdk_offscreen_window_get_embedder (
  N-GObject $window --> N-GObject
) is native(&gdk-lib)
  { * }

#-------------------------------------------------------------------------------
#TM:0:offscreen-window-get-surface:
=begin pod
=head2 offscreen-window-get-surface

Gets the offscreen surface that an offscreen window renders into. If you need to keep this around over window resizes, you need to add a reference to it.

  method offscreen-window-get-surface ( --> cairo_surface_t )

=end pod

method offscreen-window-get-surface ( --> cairo_surface_t ) {
  gdk_offscreen_window_get_surface(self._get-native-object-no-reffing)
}

sub gdk_offscreen_window_get_surface (
  N-GObject $window --> cairo_surface_t
) is native(&gdk-lib)
  { * }

#-------------------------------------------------------------------------------
#TM:0:offscreen-window-set-embedder:
=begin pod
=head2 offscreen-window-set-embedder

Sets window to be embedded in embedder.

To fully embed an offscreen window, in addition to calling this function, it is also necessary to handle the “pick-embedded-child” signal on the embedder and the “to-embedder” and “from-embedder” signals on window .

  method offscreen-window-set-embedder ( N-GObject() $embedder )

=item $embedder;
=end pod

method offscreen-window-set-embedder ( N-GObject() $embedder ) {
  gdk_offscreen_window_set_embedder(
    self._get-native-object-no-reffing, $embedder
  );
}

sub gdk_offscreen_window_set_embedder (
  N-GObject $window, N-GObject $embedder
) is native(&gdk-lib)
  { * }

#-------------------------------------------------------------------------------
#TM:0:peek-children:
=begin pod
=head2 peek-children

Like C<get_children()>, but does not copy the list of children, so the list does not need to be freed.

Returns:  (element-type GdkWindow): a reference to the list of child windows in I<window>

  method peek-children ( --> N-GList )

=end pod

method peek-children ( --> N-GList ) {
  gdk_window_peek_children(self._get-native-object-no-reffing)
}

sub gdk_window_peek_children (
  N-GObject $window --> N-GList
) is native(&gdk-lib)
  { * }

#-------------------------------------------------------------------------------
#TM:0:raise:
=begin pod
=head2 raise

Raises I<window> to the top of the Z-order (stacking order), so that other windows with the same parent window appear below I<window>. This is true whether or not the windows are visible.

If I<window> is a toplevel, the window manager may choose to deny the request to move the window in the Z-order, C<raise()> only requests the restack, does not guarantee it.

  method raise ( )

=end pod

method raise ( ) {
  gdk_window_raise(self._get-native-object-no-reffing);
}

sub gdk_window_raise (
  N-GObject $window
) is native(&gdk-lib)
  { * }

#-------------------------------------------------------------------------------
#TM:0:register-dnd:
=begin pod
=head2 register-dnd

Registers a window as a potential drop destination.

  method register-dnd ( )

=end pod

method register-dnd ( ) {
  gdk_window_register_dnd(self._get-native-object-no-reffing);
}

sub gdk_window_register_dnd (
  N-GObject $window
) is native(&gdk-lib)
  { * }

#`{{
#-------------------------------------------------------------------------------
#TM:0:remove-filter:
=begin pod
=head2 remove-filter

Remove a filter previously added with C<add_filter()>.

  method remove-filter ( GdkFilterFunc $function, Pointer $data )

=item $function; previously-added filter function
=item $data; user data for previously-added filter function
=end pod

method remove-filter ( GdkFilterFunc $function, Pointer $data ) {
  gdk_window_remove_filter(
    self._get-native-object-no-reffing, $function, $data
  );
}

sub gdk_window_remove_filter (
  N-GObject $window, GdkFilterFunc $function, gpointer $data
) is native(&gdk-lib)
  { * }
}}

#-------------------------------------------------------------------------------
#TM:0:reparent:
=begin pod
=head2 reparent

Reparents I<window> into the given I<new_parent>. The window being reparented will be unmapped as a side effect.

  method reparent ( N-GObject() $new_parent, Int() $x, Int() $y )

=item $new_parent; new parent to move I<window> into
=item $x; X location inside the new parent
=item $y; Y location inside the new parent
=end pod

method reparent ( N-GObject() $new_parent, Int() $x, Int() $y ) {
  gdk_window_reparent( self._get-native-object-no-reffing, $new_parent, $x, $y);
}

sub gdk_window_reparent (
  N-GObject $window, N-GObject $new_parent, gint $x, gint $y
) is native(&gdk-lib)
  { * }

#-------------------------------------------------------------------------------
#TM:0:resize:
=begin pod
=head2 resize

Resizes I<window>; for toplevel windows, asks the window manager to resize the window. The window manager may not allow the resize. When using GTK+, use C<gtk_window_resize()> instead of this low-level GDK function.

Windows may not be resized below 1x1.

If you’re also planning to move the window, use C<move_resize()> to both move and resize simultaneously, for a nicer visual effect.

  method resize ( Int() $width, Int() $height )

=item $width; new width of the window
=item $height; new height of the window
=end pod

method resize ( Int() $width, Int() $height ) {
  gdk_window_resize(
    self._get-native-object-no-reffing, $width, $height
  );
}

sub gdk_window_resize (
  N-GObject $window, gint $width, gint $height
) is native(&gdk-lib)
  { * }

#-------------------------------------------------------------------------------
#TM:0:restack:
=begin pod
=head2 restack

Changes the position of I<window> in the Z-order (stacking order), so that it is above I<sibling> (if I<above> is C<True>) or below I<sibling> (if I<above> is C<False>).

If I<sibling> is C<undefined>, then this either raises (if I<above> is C<True>) or lowers the window.

If I<window> is a toplevel, the window manager may choose to deny the request to move the window in the Z-order, C<restack()> only requests the restack, does not guarantee it.

  method restack ( N-GObject() $sibling, Bool $above )

=item $sibling; a B<Gnome::Gdk3::Window> that is a sibling of I<window>, or C<undefined>
=item $above; a boolean
=end pod

method restack ( N-GObject() $sibling, Bool $above ) {
  gdk_window_restack( self._get-native-object-no-reffing, $sibling, $above);
}

sub gdk_window_restack (
  N-GObject $window, N-GObject $sibling, gboolean $above
) is native(&gdk-lib)
  { * }

#-------------------------------------------------------------------------------
#TM:0:scroll:
=begin pod
=head2 scroll

Scroll the contents of I<window>, both pixels and children, by the given amount. I<window> itself does not move. Portions of the window that the scroll operation brings in from offscreen areas are invalidated. The invalidated region may be bigger than what would strictly be necessary.

For X11, a minimum area will be invalidated if the window has no subwindows, or if the edges of the window’s parent do not extend beyond the edges of the window. In other cases, a multi-step process is used to scroll the window which may produce temporary visual artifacts and unnecessary invalidations.

  method scroll ( Int() $dx, Int() $dy )

=item $dx; Amount to scroll in the X direction
=item $dy; Amount to scroll in the Y direction
=end pod

method scroll ( Int() $dx, Int() $dy ) {
  gdk_window_scroll( self._get-native-object-no-reffing, $dx, $dy);
}

sub gdk_window_scroll (
  N-GObject $window, gint $dx, gint $dy
) is native(&gdk-lib)
  { * }

#-------------------------------------------------------------------------------
#TM:0:set-accept-focus:
=begin pod
=head2 set-accept-focus

Setting I<accept_focus> to C<False> hints the desktop environment that the window doesn’t want to receive input focus.

On X, it is the responsibility of the window manager to interpret this hint. ICCCM-compliant window manager usually respect it.

  method set-accept-focus ( Bool $accept_focus )

=item $accept_focus; C<True> if the window should receive input focus
=end pod

method set-accept-focus ( Bool $accept_focus ) {
  gdk_window_set_accept_focus(
    self._get-native-object-no-reffing, $accept_focus
  );
}

sub gdk_window_set_accept_focus (
  N-GObject $window, gboolean $accept_focus
) is native(&gdk-lib)
  { * }

#-------------------------------------------------------------------------------
#TM:0:set-child-input-shapes:
=begin pod
=head2 set-child-input-shapes

Sets the input shape mask of I<window> to the union of input shape masks for all children of I<window>, ignoring the input shape mask of I<window> itself. Contrast with C<merge_child_input_shapes()> which includes the input shape mask of I<window> in the masks to be merged.

  method set-child-input-shapes ( )

=end pod

method set-child-input-shapes ( ) {
  gdk_window_set_child_input_shapes(self._get-native-object-no-reffing);
}

sub gdk_window_set_child_input_shapes (
  N-GObject $window
) is native(&gdk-lib)
  { * }

#-------------------------------------------------------------------------------
#TM:0:set-child-shapes:
=begin pod
=head2 set-child-shapes

Sets the shape mask of I<window> to the union of shape masks for all children of I<window>, ignoring the shape mask of I<window> itself. Contrast with C<merge_child_shapes()> which includes the shape mask of I<window> in the masks to be merged.

  method set-child-shapes ( )

=end pod

method set-child-shapes ( ) {
  gdk_window_set_child_shapes(self._get-native-object-no-reffing);
}

sub gdk_window_set_child_shapes (
  N-GObject $window
) is native(&gdk-lib)
  { * }

#`{{
#-------------------------------------------------------------------------------
#TM:0:set-cursor:
=begin pod
=head2 set-cursor

Sets the default mouse pointer for a B<Gnome::Gdk3::Window>.

Note that I<cursor> must be for the same display as I<window>.

Use C<gdk_cursor_new_for_display()> or C<gdk_cursor_new_from_pixbuf()> to create the cursor. To make the cursor invisible, use C<GDK_BLANK_CURSOR>. Passing C<undefined> for the I<cursor> argument to C<set_cursor()> means that I<window> will use the cursor of its parent window. Most windows should use this default.

  method set-cursor ( N-GObject() $cursor )

=item $cursor; a B<Gnome;:Gdk3::Cursor>
=end pod

method set-cursor ( N-GObject() $cursor ) {
  gdk_window_set_cursor( self._get-native-object-no-reffing, $cursor);
}

sub gdk_window_set_cursor (
  N-GObject $window, N-GObject $cursor
) is native(&gdk-lib)
  { * }
}}

#-------------------------------------------------------------------------------
#TM:4:set-decorations:
=begin pod
=head2 set-decorations

“Decorations” are the features the window manager adds to a toplevel B<Gnome::Gdk3::Window>. This function sets the traditional Motif window manager hints that tell the window manager which decorations you would like your window to have. Usually you should use C<Gnome::Gtk3::Window.set_decorated()> on a B<Gnome::Gdk3::Window> instead of using the GDK function directly.

The I<$decorations> argument is the logical OR of the fields in the C<GdkWMDecoration> enumeration. If C<GDK_DECOR_ALL> is included in the mask, the other bits indicate which decorations should be turned off. If C<GDK_DECOR_ALL> is not included, then the other bits indicate which decorations should be turned on.

Most window managers honor a decorations hint of 0 to disable all decorations, but very few honor all possible combinations of bits.

  method set-decorations ( Int $decorations )

=item $decorations; decoration hint mask
=end pod

method set-decorations ( Int $decorations ) {
  gdk_window_set_decorations( self._get-native-object-no-reffing, $decorations);
}

sub gdk_window_set_decorations (
  N-GObject $window, GFlag $decorations
) is native(&gdk-lib)
  { * }

#`{{
#-------------------------------------------------------------------------------
#TM:0:set-device-cursor:
=begin pod
=head2 set-device-cursor

Sets a specific B<Gnome::Gdk3::Cursor> for a given device when it gets inside I<window>. Use C<gdk_cursor_new_for_display()> or C<gdk_cursor_new_from_pixbuf()> to create the cursor. To make the cursor invisible, use C<GDK_BLANK_CURSOR>. Passing C<undefined> for the I<cursor> argument to C<set_cursor()> means that I<window> will use the cursor of its parent window. Most windows should use this default.

  method set-device-cursor ( N-GObject() $device, N-GObject() $cursor )

=item $device; a master, pointer B<Gnome::Gdk3::Device>
=item $cursor; a B<Gnome::Gdk3::Cursor>
=end pod

method set-device-cursor ( N-GObject() $device, N-GObject() $cursor ) {
  gdk_window_set_device_cursor(
    self._get-native-object-no-reffing, $device, $cursor
  );
}

sub gdk_window_set_device_cursor (
  N-GObject $window, N-GObject $device, N-GObject $cursor
) is native(&gdk-lib)
  { * }
}}

#-------------------------------------------------------------------------------
#TM:0:set-device-events:
=begin pod
=head2 set-device-events

Sets the event mask for a given device (Normally a floating device, not attached to any visible pointer) to I<window>. For example, an event mask including B<Gnome::Gdk3::DK_BUTTON_PRESS_MASK> means the window should report button press events. The event mask is the bitwise OR of values from the B<Gnome::Gdk3::EventMask> enumeration.

See the [input handling overview][event-masks] for details.

  method set-device-events ( N-GObject() $device, GdkEventMask $event_mask )

=item $device; B<Gnome::Gdk3::Device> to enable events for.
=item $event_mask; event mask for I<window>
=end pod

method set-device-events ( N-GObject() $device, GdkEventMask $event_mask ) {
  gdk_window_set_device_events(
    self._get-native-object-no-reffing, $device, $event_mask
  );
}

sub gdk_window_set_device_events (
  N-GObject $window, N-GObject $device, GEnum $event_mask
) is native(&gdk-lib)
  { * }

#-------------------------------------------------------------------------------
#TM:0:set-event-compression:
=begin pod
=head2 set-event-compression

Determines whether or not extra unprocessed motion events in the event queue can be discarded. If C<True> only the most recent event will be delivered.

Some types of applications, e.g. paint programs, need to see all motion events and will benefit from turning off event compression.

By default, event compression is enabled.

  method set-event-compression ( Bool $event_compression )

=item $event_compression; C<True> if motion events should be compressed
=end pod

method set-event-compression ( Bool $event_compression ) {
  gdk_window_set_event_compression(
    self._get-native-object-no-reffing, $event_compression
  );
}

sub gdk_window_set_event_compression (
  N-GObject $window, gboolean $event_compression
) is native(&gdk-lib)
  { * }

#-------------------------------------------------------------------------------
#TM:0:set-events:
=begin pod
=head2 set-events

The event mask for a window determines which events will be reported for that window from all master input devices. For example, an event mask including B<Gnome::Gdk3::DK_BUTTON_PRESS_MASK> means the window should report button press events. The event mask is the bitwise OR of values from the B<Gnome::Gdk3::EventMask> enumeration.

See the [input handling overview][event-masks] for details.

  method set-events ( GdkEventMask $event_mask )

=item $event_mask; event mask for I<window>
=end pod

method set-events ( GdkEventMask $event_mask ) {
  gdk_window_set_events( self._get-native-object-no-reffing, $event_mask);
}

sub gdk_window_set_events (
  N-GObject $window, GEnum $event_mask
) is native(&gdk-lib)
  { * }

#-------------------------------------------------------------------------------
#TM:0:set-focus-on-map:
=begin pod
=head2 set-focus-on-map

Setting I<focus_on_map> to C<False> hints the desktop environment that the window doesn’t want to receive input focus when it is mapped. focus_on_map should be turned off for windows that aren’t triggered interactively (such as popups from network activity).

On X, it is the responsibility of the window manager to interpret this hint. Window managers following the freedesktop.org window manager extension specification should respect it.

  method set-focus-on-map ( Bool $focus_on_map )

=item $focus_on_map; C<True> if the window should receive input focus when mapped
=end pod

method set-focus-on-map ( Bool $focus_on_map ) {
  gdk_window_set_focus_on_map(
    self._get-native-object-no-reffing, $focus_on_map
  );
}

sub gdk_window_set_focus_on_map (
  N-GObject $window, gboolean $focus_on_map
) is native(&gdk-lib)
  { * }

#-------------------------------------------------------------------------------
#TM:0:set-fullscreen-mode:
=begin pod
=head2 set-fullscreen-mode

Specifies whether the I<window> should span over all monitors (in a multi-head setup) or only the current monitor when in fullscreen mode.

The I<mode> argument is from the B<Gnome::Gdk3::FullscreenMode> enumeration. If B<Gnome::Gdk3::DK_FULLSCREEN_ON_ALL_MONITORS> is specified, the fullscreen I<window> will span over all monitors from the B<Gnome::Gdk3::Screen>.

On X11, searches through the list of monitors from the B<Gnome::Gdk3::Screen> the ones which delimit the 4 edges of the entire B<Gnome::Gdk3::Screen> and will ask the window manager to span the I<window> over these monitors.

If the XINERAMA extension is not available or not usable, this function has no effect.

Not all window managers support this, so you can’t rely on the fullscreen window to span over the multiple monitors when B<Gnome::Gdk3::DK_FULLSCREEN_ON_ALL_MONITORS> is specified.

  method set-fullscreen-mode ( GdkFullscreenMode $mode )

=item $mode; fullscreen mode
=end pod

method set-fullscreen-mode ( GdkFullscreenMode $mode ) {
  gdk_window_set_fullscreen_mode( self._get-native-object-no-reffing, $mode);
}

sub gdk_window_set_fullscreen_mode (
  N-GObject $window, GEnum $mode
) is native(&gdk-lib)
  { * }

#`{{
#-------------------------------------------------------------------------------
#TM:0:set-functions:
=begin pod
=head2 set-functions

Sets hints about the window management functions to make available via buttons on the window frame.

On the X backend, this function sets the traditional Motif window manager hint for this purpose. However, few window managers do anything reliable or interesting with this hint. Many ignore it entirely.

The I<functions> argument is the logical OR of values from the B<Gnome::Gdk3::WMFunction> enumeration. If the bitmask includes B<Gnome::Gdk3::DK_FUNC_ALL>, then the other bits indicate which functions to disable; if it doesn’t include B<Gnome::Gdk3::DK_FUNC_ALL>, it indicates which functions to enable.

  method set-functions ( GdkWMFunction $functions )

=item $functions; bitmask of operations to allow on I<window>
=end pod

method set-functions ( GdkWMFunction $functions ) {
  gdk_window_set_functions( self._get-native-object-no-reffing, $functions);
}

sub gdk_window_set_functions (
  N-GObject $window, GEnum $functions
) is native(&gdk-lib)
  { * }
}}

#-------------------------------------------------------------------------------
#TM:1:set-geometry-hints:
=begin pod
=head2 set-geometry-hints

Sets the geometry hints for I<window>. Hints flagged in I<geom_mask> are set, hints not flagged in I<geom_mask> are unset. To unset all hints, use a I<geom_mask> of 0 and a I<geometry> of C<undefined>.

This function provides hints to the windowing system about acceptable sizes for a toplevel window. The purpose of this is to constrain user resizing, but the windowing system will typically (but is not required to) also constrain the current size of the window to the provided values and constrain programatic resizing via C<resize()> or C<move_resize()>.

Note that on X11, this effect has no effect on windows of type C<GDK_WINDOW_TEMP> or windows where override redirect has been turned on via C<set_override_redirect()> since these windows are not resizable by the user.

Since you can’t count on the windowing system doing the constraints for programmatic resizes, you should generally call C<constrain_size()> yourself to determine appropriate sizes.

  method set-geometry-hints (
    N-GdkGeometry $geometry, GdkWindowHints $geom_mask
  )

=item $geometry; geometry hints
=item $geom_mask; bitmask indicating fields of I<geometry> to pay attention to
=end pod

method set-geometry-hints (
  N-GdkGeometry $geometry, GdkWindowHints $geom_mask
) {
  gdk_window_set_geometry_hints(
    self._get-native-object-no-reffing, $geometry, $geom_mask
  );
}

sub gdk_window_set_geometry_hints (
  N-GObject $window, N-GdkGeometry $geometry, GEnum $geom_mask
) is native(&gdk-lib)
  { * }

#-------------------------------------------------------------------------------
#TM:0:set-group:
=begin pod
=head2 set-group

Sets the group leader window for I<window>. By default, GDK sets the group leader for all toplevel windows to a global window implicitly created by GDK. With this function you can override this default.

The group leader window allows the window manager to distinguish all windows that belong to a single application. It may for example allow users to minimize/unminimize all windows belonging to an application at once. You should only set a non-default group window if your application pretends to be multiple applications.

  method set-group ( N-GObject() $leader )

=item $leader; group leader window, or C<undefined> to restore the default group leader window
=end pod

method set-group ( N-GObject() $leader ) {
  gdk_window_set_group( self._get-native-object-no-reffing, $leader);
}

sub gdk_window_set_group (
  N-GObject $window, N-GObject $leader
) is native(&gdk-lib)
  { * }

#-------------------------------------------------------------------------------
#TM:0:set-icon-list:
=begin pod
=head2 set-icon-list

Sets a list of icons for the window. One of these will be used to represent the window when it has been iconified. The icon is usually shown in an icon box or some sort of task bar. Which icon size is shown depends on the window manager. The window manager can scale the icon but setting several size icons can give better image quality since the window manager may only need to scale the icon by a small amount or not at all.

Note that some platforms don't support window icons.

  method set-icon-list ( N-GList $pixbufs )

=item $pixbufs;  (element-type GdkPixbuf): A list of pixbufs, of different sizes.
=end pod

method set-icon-list ( $pixbufs is copy ) {
  $pixbufs .= _get-native-object-no-reffing unless $pixbufs ~~ N-GList;
  gdk_window_set_icon_list( self._get-native-object-no-reffing, $pixbufs);
}

sub gdk_window_set_icon_list (
  N-GObject $window, N-GList $pixbufs
) is native(&gdk-lib)
  { * }

#-------------------------------------------------------------------------------
#TM:0:set-icon-name:
=begin pod
=head2 set-icon-name

Windows may have a name used while minimized, distinct from the name they display in their titlebar. Most of the time this is a bad idea from a user interface standpoint. But you can set such a name with this function, if you like.

After calling this with a non-C<undefined> I<name>, calls to C<set_title()> will not update the icon title.

Using C<undefined> for I<name> unsets the icon title; further calls to C<set_title()> will again update the icon title as well.

Note that some platforms don't support window icons.

  method set-icon-name ( Str $name )

=item $name; name of window while iconified (minimized)
=end pod

method set-icon-name ( Str $name ) {
  gdk_window_set_icon_name( self._get-native-object-no-reffing, $name);
}

sub gdk_window_set_icon_name (
  N-GObject $window, gchar-ptr $name
) is native(&gdk-lib)
  { * }

#`{{
#-------------------------------------------------------------------------------
#TM:0:set-invalidate-handler:
=begin pod
=head2 set-invalidate-handler

Registers an invalidate handler for a specific window. This will get called whenever a region in the window or its children is invalidated.

This can be used to record the invalidated region, which is useful if you are keeping an offscreen copy of some region and want to keep it up to date. You can also modify the invalidated region in case you’re doing some effect where e.g. a child widget appears in multiple places.

  method set-invalidate-handler ( GdkWindowInvalidateHandlerFunc $handler )

=item $handler; a B<Gnome::Gdk3::WindowInvalidateHandlerFunc> callback function
=end pod

method set-invalidate-handler ( GdkWindowInvalidateHandlerFunc $handler ) {
  gdk_window_set_invalidate_handler(
    self._get-native-object-no-reffing, $handler
  );
}

sub gdk_window_set_invalidate_handler (
  N-GObject $window, GdkWindowInvalidateHandlerFunc $handler
) is native(&gdk-lib)
  { * }
}}

#-------------------------------------------------------------------------------
#TM:0:set-keep-above:
=begin pod
=head2 set-keep-above

Set if I<window> must be kept above other windows. If the window was already above, then this function does nothing.

On X11, asks the window manager to keep I<window> above, if the window manager supports this operation. Not all window managers support this, and some deliberately ignore it or don’t have a concept of “keep above”; so you can’t rely on the window being kept above. But it will happen with most standard window managers, and GDK makes a best effort to get it to happen.

  method set-keep-above ( Bool $setting )

=item $setting; whether to keep I<window> above other windows
=end pod

method set-keep-above ( Bool $setting ) {
  gdk_window_set_keep_above( self._get-native-object-no-reffing, $setting);
}

sub gdk_window_set_keep_above (
  N-GObject $window, gboolean $setting
) is native(&gdk-lib)
  { * }

#-------------------------------------------------------------------------------
#TM:0:set-keep-below:
=begin pod
=head2 set-keep-below

Set if I<window> must be kept below other windows. If the window was already below, then this function does nothing.

On X11, asks the window manager to keep I<window> below, if the window manager supports this operation. Not all window managers support this, and some deliberately ignore it or don’t have a concept of “keep below”; so you can’t rely on the window being kept below. But it will happen with most standard window managers, and GDK makes a best effort to get it to happen.

  method set-keep-below ( Bool $setting )

=item $setting; whether to keep I<window> below other windows
=end pod

method set-keep-below ( Bool $setting ) {
  gdk_window_set_keep_below( self._get-native-object-no-reffing, $setting);
}

sub gdk_window_set_keep_below (
  N-GObject $window, gboolean $setting
) is native(&gdk-lib)
  { * }

#-------------------------------------------------------------------------------
#TM:0:set-modal-hint:
=begin pod
=head2 set-modal-hint

The application can use this hint to tell the window manager that a certain window has modal behaviour. The window manager can use this information to handle modal windows in a special way.

You should only use this on windows for which you have previously called C<set_transient_for()>

  method set-modal-hint ( Bool $modal )

=item $modal; C<True> if the window is modal, C<False> otherwise.
=end pod

method set-modal-hint ( Bool $modal ) {
  gdk_window_set_modal_hint( self._get-native-object-no-reffing, $modal);
}

sub gdk_window_set_modal_hint (
  N-GObject $window, gboolean $modal
) is native(&gdk-lib)
  { * }

#-------------------------------------------------------------------------------
#TM:0:set-opacity:
=begin pod
=head2 set-opacity

Set I<window> to render as partially transparent, with opacity 0 being fully transparent and 1 fully opaque. (Values of the opacity parameter are clamped to the [0,1] range.)

For toplevel windows this depends on support from the windowing system that may not always be there. For instance, On X11, this works only on X screens with a compositing manager running. On Wayland, there is no per-window opacity value that the compositor would apply. Instead, use `set_opaque_region (window, NULL)` to tell the compositor that the entire window is (potentially) non-opaque, and draw your content with alpha, or use C<gtk_widget_set_opacity()> to set an overall opacity for your widgets.

For child windows this function only works for non-native windows.

For setting up per-pixel alpha topelevels, see C<gdk_screen_get_rgba_visual()>, and for non-toplevels, see C<set_composited()>.

Support for non-toplevel windows was added in 3.8.

  method set-opacity ( Num() $opacity )

=item $opacity; opacity
=end pod

method set-opacity ( Num() $opacity ) {
  gdk_window_set_opacity( self._get-native-object-no-reffing, $opacity);
}

sub gdk_window_set_opacity (
  N-GObject $window, gdouble $opacity
) is native(&gdk-lib)
  { * }

#`{{
#-------------------------------------------------------------------------------
#TM:0:set-opaque-region:
=begin pod
=head2 set-opaque-region

For optimisation purposes, compositing window managers may like to not draw obscured regions of windows, or turn off blending during for these regions. With RGB windows with no transparency, this is just the shape of the window, but with ARGB32 windows, the compositor does not know what regions of the window are transparent or not.

This function only works for toplevel windows.

GTK+ will update this property automatically if the I<window> background is opaque, as we know where the opaque regions are. If your window background is not opaque, please update this property in your  I<style-updated> handler.

  method set-opaque-region ( cairo_region_t $region )

=item $region; a region, or C<undefined>
=end pod

method set-opaque-region ( cairo_region_t $region ) {

  gdk_window_set_opaque_region(
    self._get-native-object-no-reffing, $region
  );
}

sub gdk_window_set_opaque_region (
  N-GObject $window, cairo_region_t $region
) is native(&gdk-lib)
  { * }
}}

#-------------------------------------------------------------------------------
#TM:0:set-override-redirect:
=begin pod
=head2 set-override-redirect

An override redirect window is not under the control of the window manager. This means it won’t have a titlebar, won’t be minimizable, etc. - it will be entirely under the control of the application. The window manager can’t see the override redirect window at all.

Override redirect should only be used for short-lived temporary windows, such as popup menus. B<Gnome::Gdk3::Menu> uses an override redirect window in its implementation, for example.

  method set-override-redirect ( Bool $override_redirect )

=item $override_redirect; C<True> if window should be override redirect
=end pod

method set-override-redirect ( Bool $override_redirect ) {
  gdk_window_set_override_redirect(
    self._get-native-object-no-reffing, $override_redirect
  );
}

sub gdk_window_set_override_redirect (
  N-GObject $window, gboolean $override_redirect
) is native(&gdk-lib)
  { * }

#-------------------------------------------------------------------------------
#TM:0:set-pass-through:
=begin pod
=head2 set-pass-through

Sets whether input to the window is passed through to the window below.

The default value of this is C<False>, which means that pointer events that happen inside the window are send first to the window, but if the event is not selected by the event mask then the event is sent to the parent window, and so on up the hierarchy.

If I<pass_through> is C<True> then such pointer events happen as if the window wasn't there at all, and thus will be sent first to any windows below I<window>. This is useful if the window is used in a transparent fashion. In the terminology of the web this would be called "pointer-events: none".

Note that a window with I<pass_through> C<True> can still have a subwindow without pass through, so you can get events on a subset of a window. And in that cases you would get the in-between related events such as the pointer enter/leave events on its way to the destination window.

  method set-pass-through ( Bool $pass_through )

=item $pass_through; a boolean
=end pod

method set-pass-through ( Bool $pass_through ) {
  gdk_window_set_pass_through(
    self._get-native-object-no-reffing, $pass_through
  );
}

sub gdk_window_set_pass_through (
  N-GObject $window, gboolean $pass_through
) is native(&gdk-lib)
  { * }

#-------------------------------------------------------------------------------
#TM:0:set-role:
=begin pod
=head2 set-role

When using GTK+, typically you should use C<gtk_window_set_role()> instead of this low-level function.

The window manager and session manager use a window’s role to distinguish it from other kinds of window in the same application. When an application is restarted after being saved in a previous session, all windows with the same title and role are treated as interchangeable. So if you have two windows with the same title that should be distinguished for session management purposes, you should set the role on those windows. It doesn’t matter what string you use for the role, as long as you have a different role for each non-interchangeable kind of window.

  method set-role ( Str $role )

=item $role; a string indicating its role
=end pod

method set-role ( Str $role ) {
  gdk_window_set_role( self._get-native-object-no-reffing, $role);
}

sub gdk_window_set_role (
  N-GObject $window, gchar-ptr $role
) is native(&gdk-lib)
  { * }

#-------------------------------------------------------------------------------
#TM:0:set-shadow-width:
=begin pod
=head2 set-shadow-width

Newer GTK+ windows using client-side decorations use extra geometry around their frames for effects like shadows and invisible borders. Window managers that want to maximize windows or snap to edges need to know where the extents of the actual frame lie, so that users don’t feel like windows are snapping against random invisible edges.

Note that this property is automatically updated by GTK+, so this function should only be used by applications which do not use GTK+ to create toplevel windows.

  method set-shadow-width (
    Int() $left, Int() $right, Int() $top, Int() $bottom
  )

=item $left; The left extent
=item $right; The right extent
=item $top; The top extent
=item $bottom; The bottom extent
=end pod

method set-shadow-width (
  Int() $left, Int() $right, Int() $top, Int() $bottom
) {
  gdk_window_set_shadow_width(
    self._get-native-object-no-reffing, $left, $right, $top, $bottom
  );
}

sub gdk_window_set_shadow_width (
  N-GObject $window, gint $left, gint $right, gint $top, gint $bottom
) is native(&gdk-lib)
  { * }

#-------------------------------------------------------------------------------
#TM:0:set-skip-pager-hint:
=begin pod
=head2 set-skip-pager-hint

Toggles whether a window should appear in a pager (workspace switcher, or other desktop utility program that displays a small thumbnail representation of the windows on the desktop). If a window’s semantic type as specified with C<set_type_hint()> already fully describes the window, this function should not be called in addition, instead you should allow the window to be treated according to standard policy for its semantic type.

  method set-skip-pager-hint ( Bool $skips_pager )

=item $skips_pager; C<True> to skip the pager
=end pod

method set-skip-pager-hint ( Bool $skips_pager ) {
  gdk_window_set_skip_pager_hint(
    self._get-native-object-no-reffing, $skips_pager
  );
}

sub gdk_window_set_skip_pager_hint (
  N-GObject $window, gboolean $skips_pager
) is native(&gdk-lib)
  { * }

#-------------------------------------------------------------------------------
#TM:0:set-skip-taskbar-hint:
=begin pod
=head2 set-skip-taskbar-hint

Toggles whether a window should appear in a task list or window list. If a window’s semantic type as specified with C<set_type_hint()> already fully describes the window, this function should not be called in addition, instead you should allow the window to be treated according to standard policy for its semantic type.

  method set-skip-taskbar-hint ( Bool $skips_taskbar )

=item $skips_taskbar; C<True> to skip the taskbar
=end pod

method set-skip-taskbar-hint ( Bool $skips_taskbar ) {
  gdk_window_set_skip_taskbar_hint(
    self._get-native-object-no-reffing, $skips_taskbar
  );
}

sub gdk_window_set_skip_taskbar_hint (
  N-GObject $window, gboolean $skips_taskbar
) is native(&gdk-lib)
  { * }

#-------------------------------------------------------------------------------
#TM:0:set-source-events:
=begin pod
=head2 set-source-events

Sets the event mask for any floating device (i.e. not attached to any visible pointer) that has the source defined as I<source>. This event mask will be applied both to currently existing, newly added devices after this call, and devices being attached/detached.

  method set-source-events ( GdkInputSource $source, GdkEventMask $event_mask )

=item $source; a B<Gnome::Gdk3::InputSource> to define the source class.
=item $event_mask; event mask for I<window>
=end pod

method set-source-events ( GdkInputSource $source, GdkEventMask $event_mask ) {
  gdk_window_set_source_events(
    self._get-native-object-no-reffing, $source, $event_mask
  );
}

sub gdk_window_set_source_events (
  N-GObject $window, GEnum $source, GEnum $event_mask
) is native(&gdk-lib)
  { * }

#-------------------------------------------------------------------------------
#TM:0:set-startup-id:
=begin pod
=head2 set-startup-id

When using GTK+, typically you should use C<gtk_window_set_startup_id()> instead of this low-level function.

  method set-startup-id ( Str $startup_id )

=item $startup_id; a string with startup-notification identifier
=end pod

method set-startup-id ( Str $startup_id ) {
  gdk_window_set_startup_id(
    self._get-native-object-no-reffing, $startup_id
  );
}

sub gdk_window_set_startup_id (
  N-GObject $window, gchar-ptr $startup_id
) is native(&gdk-lib)
  { * }

#-------------------------------------------------------------------------------
#TM:0:set-support-multidevice:
=begin pod
=head2 set-support-multidevice

This function will enable multidevice features in I<window>.

Multidevice aware windows will need to handle properly multiple, per device enter/leave events, device grabs and grab ownerships.

  method set-support-multidevice ( Bool $support_multidevice )

=item $support_multidevice; C<True> to enable multidevice support in I<window>.
=end pod

method set-support-multidevice ( Bool $support_multidevice ) {
  gdk_window_set_support_multidevice(
    self._get-native-object-no-reffing, $support_multidevice
  );
}

sub gdk_window_set_support_multidevice (
  N-GObject $window, gboolean $support_multidevice
) is native(&gdk-lib)
  { * }

#-------------------------------------------------------------------------------
#TM:0:set-title:
=begin pod
=head2 set-title

Sets the title of a toplevel window, to be displayed in the titlebar. If you haven’t explicitly set the icon name for the window (using C<set_icon_name()>), the icon name will be set to I<title> as well. I<title> must be in UTF-8 encoding (as with all user-readable strings in GDK/GTK+). I<title> may not be C<undefined>.

  method set-title ( Str $title )

=item $title; title of I<window>
=end pod

method set-title ( Str $title ) {
  gdk_window_set_title( self._get-native-object-no-reffing, $title);
}

sub gdk_window_set_title (
  N-GObject $window, gchar-ptr $title
) is native(&gdk-lib)
  { * }

#-------------------------------------------------------------------------------
#TM:0:set-transient-for:
=begin pod
=head2 set-transient-for

Indicates to the window manager that I<window> is a transient dialog associated with the application window I<parent>. This allows the window manager to do things like center I<window> on I<parent> and keep I<window> above I<parent>.

See C<gtk_window_set_transient_for()> if you’re using B<Gnome::Gdk3::Window> or B<Gnome::Gdk3::Dialog>.

  method set-transient-for ( N-GObject() $parent )

=item $parent; another toplevel B<Gnome::Gdk3::Window>
=end pod

method set-transient-for ( N-GObject() $parent ) {
  gdk_window_set_transient_for( self._get-native-object-no-reffing, $parent);
}

sub gdk_window_set_transient_for (
  N-GObject $window, N-GObject $parent
) is native(&gdk-lib)
  { * }

#-------------------------------------------------------------------------------
#TM:0:set-type-hint:
=begin pod
=head2 set-type-hint

The application can use this call to provide a hint to the window manager about the functionality of a window. The window manager can use this information when determining the decoration and behaviour of the window.

The hint must be set before the window is mapped.

  method set-type-hint ( GdkWindowTypeHint $hint )

=item $hint; A hint of the function this window will have
=end pod

method set-type-hint ( GdkWindowTypeHint $hint ) {
  gdk_window_set_type_hint( self._get-native-object-no-reffing, $hint);
}

sub gdk_window_set_type_hint (
  N-GObject $window, GEnum $hint
) is native(&gdk-lib)
  { * }

#-------------------------------------------------------------------------------
#TM:0:set-urgency-hint:
=begin pod
=head2 set-urgency-hint

Toggles whether a window needs the user's urgent attention.

  method set-urgency-hint ( Bool $urgent )

=item $urgent; C<True> if the window is urgent
=end pod

method set-urgency-hint ( Bool $urgent ) {
  gdk_window_set_urgency_hint( self._get-native-object-no-reffing, $urgent);
}

sub gdk_window_set_urgency_hint (
  N-GObject $window, gboolean $urgent
) is native(&gdk-lib)
  { * }

#-------------------------------------------------------------------------------
#TM:0:set-user-data:
=begin pod
=head2 set-user-data

For most purposes this function is deprecated in favor of C<g_object_set_data()>. However, for historical reasons GTK+ stores the B<Gnome::Gdk3::Widget> that owns a B<Gnome::Gdk3::Window> as user data on the B<Gnome::Gdk3::Window>. So, custom widget implementations should use this function for that. If GTK+ receives an event for a B<Gnome::Gdk3::Window>, and the user data for the window is non-C<undefined>, GTK+ will assume the user data is a B<Gnome::Gdk3::Widget>, and forward the event to that widget.

  method set-user-data ( Pointer $user_data )

=item $user_data;  (type GObject.Object): user data
=end pod

method set-user-data ( Pointer $user_data ) {
  gdk_window_set_user_data( self._get-native-object-no-reffing, $user_data);
}

sub gdk_window_set_user_data (
  N-GObject $window, gpointer $user_data
) is native(&gdk-lib)
  { * }

#`{{
#-------------------------------------------------------------------------------
#TM:0:shape-combine-region:
=begin pod
=head2 shape-combine-region

Makes pixels in I<window> outside I<shape_region> be transparent, so that the window may be nonrectangular.

If I<shape_region> is C<undefined>, the shape will be unset, so the whole window will be opaque again. I<offset_x> and I<offset_y> are ignored if I<shape_region> is C<undefined>.

On the X11 platform, this uses an X server extension which is widely available on most common platforms, but not available on very old X servers, and occasionally the implementation will be buggy. On servers without the shape extension, this function will do nothing.

This function works on both toplevel and child windows.

  method shape-combine-region ( cairo_region_t $shape_region, Int() $offset_x, Int() $offset_y )

=item $shape_region; region of window to be non-transparent
=item $offset_x; X position of I<shape_region> in I<window> coordinates
=item $offset_y; Y position of I<shape_region> in I<window> coordinates
=end pod

method shape-combine-region ( cairo_region_t $shape_region, Int() $offset_x, Int() $offset_y ) {

  gdk_window_shape_combine_region(
    self._get-native-object-no-reffing, $shape_region, $offset_x, $offset_y
  );
}

sub gdk_window_shape_combine_region (
  N-GObject $window, cairo_region_t $shape_region, gint $offset_x, gint $offset_y
) is native(&gdk-lib)
  { * }
}}

#-------------------------------------------------------------------------------
#TM:0:show:
=begin pod
=head2 show

Like C<show_unraised()>, but also raises the window to the top of the window stack (moves the window to the front of the Z-order).

This function maps a window so it’s visible onscreen. Its opposite is C<hide()>.

When implementing a B<Gnome::Gdk3::Widget>, you should call this function on the widget's B<Gnome::Gdk3::Window> as part of the “map” method.

  method show ( )

=end pod

method show ( ) {
  gdk_window_show( self._get-native-object-no-reffing);
}

sub gdk_window_show (
  N-GObject $window
) is native(&gdk-lib)
  { * }

#-------------------------------------------------------------------------------
#TM:0:show-unraised:
=begin pod
=head2 show-unraised

Shows a B<Gnome::Gdk3::Window> onscreen, but does not modify its stacking order. In contrast, C<show()> will raise the window to the top of the window stack.

On the X11 platform, in Xlib terms, this function calls C<XMapWindow()> (it also updates some internal GDK state, which means that you can’t really use C<XMapWindow()> directly on a GDK window).

  method show-unraised ( )

=end pod

method show-unraised ( ) {
  gdk_window_show_unraised(self._get-native-object-no-reffing);
}

sub gdk_window_show_unraised (
  N-GObject $window
) is native(&gdk-lib)
  { * }

#-------------------------------------------------------------------------------
#TM:0:show-window-menu:
=begin pod
=head2 show-window-menu

Asks the windowing system to show the window menu. The window menu is the menu shown when right-clicking the titlebar on traditional windows managed by the window manager. This is useful for windows using client-side decorations, activating it with a right-click on the window decorations.

Returns: C<True> if the window menu was shown and C<False> otherwise.

  method show-window-menu ( GdkEvent $event --> Bool )

=item $event; a B<Gnome::Gdk3::Event> to show the menu for
=end pod

method show-window-menu ( GdkEvent $event --> Bool ) {
  gdk_window_show_window_menu( self._get-native-object-no-reffing, $event).Bool
}

sub gdk_window_show_window_menu (
  N-GObject $window, GdkEvent $event --> gboolean
) is native(&gdk-lib)
  { * }

#-------------------------------------------------------------------------------
#TM:0:stick:
=begin pod
=head2 stick

“Pins” a window such that it’s on all workspaces and does not scroll with viewports, for window managers that have scrollable viewports. (When using B<Gnome::Gdk3::Window>, C<gtk_window_stick()> may be more useful.)

On the X11 platform, this function depends on window manager support, so may have no effect with many window managers. However, GDK will do the best it can to convince the window manager to stick the window. For window managers that don’t support this operation, there’s nothing you can do to force it to happen.

  method stick ( )

=end pod

method stick ( ) {
  gdk_window_stick(self._get-native-object-no-reffing);
}

sub gdk_window_stick (
  N-GObject $window
) is native(&gdk-lib)
  { * }

#-------------------------------------------------------------------------------
#TM:0:test-render-sync:
=begin pod
=head2 test-render-sync

Retrieves a pixel from window to force the windowing system to carry out any pending rendering commands.

This function is intended to be used to synchronize with rendering pipelines, to benchmark windowing system rendering operations.

  method test-render-sync ( )

=end pod

method test-render-sync ( ) {
  gdk_test_render_sync(self._get-native-object-no-reffing);
}

sub gdk_test_render_sync (
  N-GObject $window
) is native(&gdk-lib)
  { * }

#-------------------------------------------------------------------------------
#TM:0:test-simulate-button:
=begin pod
=head2 test-simulate-button

This function is intended to be used in GTK+ test programs. It will warp the mouse pointer to the given (x ,y ) coordinates within window and simulate a button press or release event. Because the mouse pointer needs to be warped to the target location, use of this function outside of test programs that run in their own virtual windowing system (e.g. Xvfb) is not recommended.

Also, gdk_test_simulate_button() is a fairly low level function, for most testing purposes, gtk_test_widget_click() is the right function to call which will generate a button press event followed by its accompanying button release event.

  method test-simulate-button (
    Int() $x, Int() $y, UInt() $button,
    GdkModifierType modifiers, GdkEventType $button_pressrelease
    --> Bool
  )

=item $x; x coordinate within window for the button event
=item $y; y coordinate within window for the button event
=item $button; Number of the pointer button for the event, usually 1, 2 or 3
=item modifiers; Keyboard modifiers the event is setup with
=item $button_pressrelease; either GDK_BUTTON_PRESS or GDK_BUTTON_RELEASE
=end pod

method test-simulate-button (
  Int() $x, Int() $y, UInt $button,
  GdkModifierType $modifiers, GdkEventType $button_pressrelease
  --> Bool
) {
  gdk_test_simulate_button(
    self._get-native-object-no-reffing, $x, $y, $button,
    $modifiers, $button_pressrelease
  ).Bool
}

sub gdk_test_simulate_button (
  N-GObject $window, gint $x, gint $y, guint $button,
  GEnum $modifiers, GEnum $button_pressrelease --> gboolean
) is native(&gdk-lib)
  { * }

#-------------------------------------------------------------------------------
#TM:0:test-simulate-key:
=begin pod
=head2 test-simulate-key

This function is intended to be used in GTK+ test programs. If (x ,y ) are > (-1,-1), it will warp the mouse pointer to the given (x ,y ) coordinates within window and simulate a key press or release event.

When the mouse pointer is warped to the target location, use of this function outside of test programs that run in their own virtual windowing system (e.g. Xvfb) is not recommended. If (x ,y ) are passed as (-1,-1), the mouse pointer will not be warped and window origin will be used as mouse pointer location for the event.

Also, gdk_test_simulate_key() is a fairly low level function, for most testing purposes, gtk_test_widget_send_key() is the right function to call which will generate a key press event followed by its accompanying key release event.

  method test-simulate-key (
    Int() $x, Int() $y, UInt $keyval,
    GdkModifierType $modifiers, GdkEventType $key_pressrelease
    --> Bool
  )

=item $x; x coordinate within window for the key event
=item $y; y coordinate within window for the key event
=item $keyval; A GDK keyboard value from B<Gnome::Gdk3::Keysyms>
=item $modifiers; Keyboard modifiers the event is setup with
=item $key_pressrelease; either GDK_KEY_PRESS or GDK_KEY_RELEASE
=end pod

method test-simulate-key (
  Int() $x, Int() $y, UInt $keyval,
  GdkModifierType $modifiers, GdkEventType $key_pressrelease
  --> Bool
) {
  gdk_test_simulate_key(
    self._get-native-object-no-reffing, $x, $y, $keyval,
    $modifiers, $key_pressrelease
  ).Bool
}

sub gdk_test_simulate_key (
  N-GObject $window, gint $x, gint $y, guint $keyval, GEnum $modifiers, GEnum $key_pressrelease --> gboolean
) is native(&gdk-lib)
  { * }

#-------------------------------------------------------------------------------
#TM:0:thaw-updates:
=begin pod
=head2 thaw-updates

Thaws a window frozen with C<freeze_updates()>.

  method thaw-updates ( )

=end pod

method thaw-updates ( ) {
  gdk_window_thaw_updates( self._get-native-object-no-reffing);
}

sub gdk_window_thaw_updates (
  N-GObject $window
) is native(&gdk-lib)
  { * }

#-------------------------------------------------------------------------------
#TM:1:unfullscreen:
=begin pod
=head2 unfullscreen

Moves the window out of fullscreen mode. If the window was not fullscreen, does nothing.

On X11, asks the window manager to move I<window> out of the fullscreen state, if the window manager supports this operation. Not all window managers support this, and some deliberately ignore it or don’t have a concept of “fullscreen”; so you can’t rely on the unfullscreenification actually happening. But it will happen with most standard window managers, and GDK makes a best effort to get it to happen.

  method unfullscreen ( )

=end pod

method unfullscreen ( ) {
  gdk_window_unfullscreen(self._get-native-object-no-reffing);
}

sub gdk_window_unfullscreen (
  N-GObject $window
) is native(&gdk-lib)
  { * }

#-------------------------------------------------------------------------------
#TM:0:unmaximize:
=begin pod
=head2 unmaximize

Unmaximizes the window. If the window wasn’t maximized, then this function does nothing.

On X11, asks the window manager to unmaximize I<window>, if the window manager supports this operation. Not all window managers support this, and some deliberately ignore it or don’t have a concept of “maximized”; so you can’t rely on the unmaximization actually happening. But it will happen with most standard window managers, and GDK makes a best effort to get it to happen.

On Windows, reliably unmaximizes the window.

  method unmaximize ( )

=end pod

method unmaximize ( ) {
  gdk_window_unmaximize(self._get-native-object-no-reffing);
}

sub gdk_window_unmaximize (
  N-GObject $window
) is native(&gdk-lib)
  { * }

#-------------------------------------------------------------------------------
#TM:0:unstick:
=begin pod
=head2 unstick

Reverse operation for C<stick()>; see C<stick()>, and C<gtk_window_unstick()>.

  method unstick ( )

=end pod

method unstick ( ) {
  gdk_window_unstick(self._get-native-object-no-reffing);
}

sub gdk_window_unstick (
  N-GObject $window
) is native(&gdk-lib)
  { * }

#-------------------------------------------------------------------------------
#TM:0:withdraw:
=begin pod
=head2 withdraw

Withdraws a window (unmaps it and asks the window manager to forget about it). This function is not really useful as C<hide()> automatically withdraws toplevel windows before hiding them.

  method withdraw ( )

=end pod

method withdraw ( ) {
  gdk_window_withdraw(self._get-native-object-no-reffing);
}

sub gdk_window_withdraw (
  N-GObject $window
) is native(&gdk-lib)
  { * }

#-------------------------------------------------------------------------------
#TM:1:_gdk_window_new:
#`{{
=begin pod
=head2 _gdk_window_new

Creates a new B<Gnome::Gdk3::Window> using the attributes from I<attributes>. See B<Gnome::Gdk3::WindowAttr> and B<Gnome::Gdk3::WindowAttributesType> for more details. Note: to use this on displays other than the default display, I<parent> must be specified.

Returns: the new B<Gnome::Gdk3::Window>

  method _gdk_window_new (
    N-GdkWindowAttr $attributes, Int() $attributes_mask --> N-GObject
  )

=item $attributes; attributes of the new window
=item $attributes_mask; (type N-GdkWindowAttributesType): mask indicating which fields in I<attributes> are valid
=end pod
}}

sub _gdk_window_new ( N-GObject $parent, N-GdkWindowAttr $attributes, gint $attributes_mask --> N-GObject )
  is native(&gdk-lib)
  is symbol('gdk_window_new')
  { * }

#-------------------------------------------------------------------------------
=begin pod
=head1 Signals

There are two ways to connect to a signal. The first option you have is to use C<register-signal()> from B<Gnome::GObject::Object>. The second option is to use C<connect-object()> directly from B<Gnome::GObject::Signal>.

=head2 First method

The positional arguments of the signal handler are all obligatory as well as their types. The named attributes C<:$widget> and user data are optional.

  # handler method
  method mouse-event ( GdkEvent $event, :$widget ) { ... }

  # connect a signal on window object
  my Gnome::Gtk3::Window $w .= new( ... );
  $w.register-signal( self, 'mouse-event', 'button-press-event');

=head2 Second method

  my Gnome::Gtk3::Window $w .= new( ... );
  my Callable $handler = sub (
    N-GObject $native, GdkEvent $event, OpaquePointer $data
  ) {
    ...
  }

  $w.connect-object( 'button-press-event', $handler);

Also here, the types of positional arguments in the signal handler are important. This is because both methods C<register-signal()> and C<connect-object()> are using the signatures of the handler routines to setup the native call interface.

=head2 Supported signals


=comment -----------------------------------------------------------------------
=comment #TS:0:create-surface:
=head3 create-surface

The I<create-surface> signal is emitted when an offscreen window needs its surface (re)created, which happens either when the window is first drawn to, or when the window is being resized. The first signal handler that returns a non-C<undefined> surface will stop any further signal emission, and its surface will be used.

Note that it is not possible to access the window's previous surface from within any callback of this signal. Calling C<gdk_offscreen_window_get_surface()> will lead to a crash.

Returns: the newly created B<cairo_surface_t> for the offscreen window

  method handler (
    Int $width,
    Int $height,
    Int :$_handle_id,
    Gnome::GObject::Object :_widget($window),
    *%user-options
    --> Unknown type CAIRO_GOBJECT_TYPE_SURFACE
  );

=item $window; the offscreen window on which the signal is emitted

=item $width; the width of the offscreen surface to create

=item $height; the height of the offscreen surface to create

=item $_handle_id; the registered event handler id

=begin comment
=comment -----------------------------------------------------------------------
=comment #TS:0:from-embedder:
=head3 from-embedder

The I<from-embedder> signal is emitted to translate coordinates in the embedder of an offscreen window to the offscreen window.

See also  I<to-embedder>.


  method handler (
    num64 #`{ use NativeCall } $embedder_x,
    num64 #`{ use NativeCall } $embedder_y,
    Unknown type G_TYPE_POINTER $offscreen_x,
    Unknown type G_TYPE_POINTER $offscreen_y,
    Int :$_handle_id,
    Gnome::GObject::Object :_widget($window),
    *%user-options
  );

=item $window; the offscreen window on which the signal is emitted

=item $embedder_x; x coordinate in the embedder window

=item $embedder_y; y coordinate in the embedder window

=item $offscreen_x; (out) (type double): return location for the x
coordinate in the offscreen window
=item $offscreen_y; (out) (type double): return location for the y
coordinate in the offscreen window
=item $_handle_id; the registered event handler id
=end comment

=begin comment
=comment -----------------------------------------------------------------------
=comment #TS:0:moved-to-rect:
=head3 moved-to-rect

Emitted when the position of I<window> is finalized after being moved to a
destination rectangle.

I<window> might be flipped over the destination rectangle in order to keep
it on-screen, in which case I<flipped_x> and I<flipped_y> will be set to C<True>
accordingly.

I<flipped_rect> is the ideal position of I<window> after any possible
flipping, but before any possible sliding. I<final_rect> is I<flipped_rect>,
but possibly translated in the case that flipping is still ineffective in
keeping I<window> on-screen.

Stability: Private

  method handler (
    Unknown type G_TYPE_POINTER $flipped_rect,
    Unknown type G_TYPE_POINTER $final_rect,
    Int $flipped_x,
    Int $flipped_y,
    Int :$_handle_id,
    Gnome::GObject::Object :_widget($window),
    *%user-options
  );

=item $window; the B<Gnome::Gdk3::Window> that moved

=item $flipped_rect; (nullable): the position of I<window> after any possible
flipping or C<undefined> if the backend can't obtain it
=item $final_rect; (nullable): the final position of I<window> or C<undefined> if the
backend can't obtain it
=item $flipped_x; C<True> if the anchors were flipped horizontally

=item $flipped_y; C<True> if the anchors were flipped vertically

=item $_handle_id; the registered event handler id
=end comment

=begin comment
=comment -----------------------------------------------------------------------
=comment #TS:0:pick-embedded-child:
=head3 pick-embedded-child

The I<pick-embedded-child> signal is emitted to find an embedded
child at the given position.

Returns: (nullable) (transfer none): the B<Gnome::Gdk3::Window> of the
embedded child at I<x>, I<y>, or C<undefined>


  method handler (
    num64 #`{ use NativeCall } $x,
    num64 #`{ use NativeCall } $y,
    Int :$_handle_id,
    Gnome::GObject::Object :_widget($window),
    *%user-options
    --> Unknown type GDK_TYPE_WINDOW
  );

=item $window; the window on which the signal is emitted

=item $x; x coordinate in the window

=item $y; y coordinate in the window

=item $_handle_id; the registered event handler id
=end comment

=begin comment
=comment -----------------------------------------------------------------------
=comment #TS:0:to-embedder:
=head3 to-embedder

The I<to-embedder> signal is emitted to translate coordinates
in an offscreen window to its embedder.

See also  I<from-embedder>.


  method handler (
    num64 #`{ use NativeCall } $offscreen_x,
    num64 #`{ use NativeCall } $offscreen_y,
    Unknown type G_TYPE_POINTER $embedder_x,
    Unknown type G_TYPE_POINTER $embedder_y,
    Int :$_handle_id,
    Gnome::GObject::Object :_widget($window),
    *%user-options
  );

=item $window; the offscreen window on which the signal is emitted

=item $offscreen_x; x coordinate in the offscreen window

=item $offscreen_y; y coordinate in the offscreen window

=item $embedder_x; (out) (type double): return location for the x
coordinate in the embedder window
=item $embedder_y; (out) (type double): return location for the y
coordinate in the embedder window
=item $_handle_id; the registered event handler id

=end comment
=end pod


#-------------------------------------------------------------------------------
=begin pod
=head1 Properties

An example of using a string type property of a B<Gnome::Gtk3::Label> object. This is just showing how to set/read a property, not that it is the best way to do it. This is because a) The class initialization often provides some options to set some of the properties and b) the classes provide many methods to modify just those properties. In the case below one can use B<new(:label('my text label'))> or B<.set-text('my text label')>.

  my Gnome::Gtk3::Label $label .= new;
  my Gnome::GObject::Value $gv .= new(:init(G_TYPE_STRING));
  $label.get-property( 'label', $gv);
  $gv.set-string('my text label');

=head2 Supported properties

=comment -----------------------------------------------------------------------
=comment #TP:0:cursor:
=head3 Cursor: cursor


The mouse pointer for a B<Gnome::Gdk3::Window>. See C<set_cursor()> and
C<get_cursor()> for details.

   Widget type: GDK_TYPE_CURSOR

The B<Gnome::GObject::Value> type of property I<cursor> is C<G_TYPE_OBJECT>.
=end pod
