package org.mandas.docker.client.messages;

import com.fasterxml.jackson.annotation.JsonProperty;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import org.mandas.docker.Nullable;

/**
 * Immutable implementation of {@link ExecCreation}.
 * <p>
 * Use the builder to create immutable instances:
 * {@code ImmutableExecCreation.builder()}.
 */
@SuppressWarnings({"all"})
final class ImmutableExecCreation implements ExecCreation {
  private final String id;
  private final @Nullable List<String> warnings;

  private ImmutableExecCreation(String id, @Nullable List<String> warnings) {
    this.id = id;
    this.warnings = warnings;
  }

  /**
   * @return The value of the {@code id} attribute
   */
  @JsonProperty("Id")
  @Override
  public String id() {
    return id;
  }

  /**
   * @return The value of the {@code warnings} attribute
   */
  @JsonProperty("Warnings")
  @Override
  public @Nullable List<String> warnings() {
    return warnings;
  }

  /**
   * Copy the current immutable object by setting a value for the {@link ExecCreation#id() id} attribute.
   * An equals check used to prevent copying of the same value by returning {@code this}.
   * @param value A new value for id
   * @return A modified copy of the {@code this} object
   */
  public final ImmutableExecCreation withId(String value) {
    String newValue = Objects.requireNonNull(value, "id");
    if (this.id.equals(newValue)) return this;
    return new ImmutableExecCreation(newValue, this.warnings);
  }

  /**
   * Copy the current immutable object with elements that replace the content of {@link ExecCreation#warnings() warnings}.
   * @param elements The elements to set
   * @return A modified copy of {@code this} object
   */
  public final ImmutableExecCreation withWarnings(@Nullable String... elements) {
    if (elements == null) {
      return new ImmutableExecCreation(this.id, null);
    }
    @Nullable List<String> newValue = Arrays.asList(elements) == null ? null : createUnmodifiableList(false, createSafeList(Arrays.asList(elements), true, false));
    return new ImmutableExecCreation(this.id, newValue);
  }

  /**
   * Copy the current immutable object with elements that replace the content of {@link ExecCreation#warnings() warnings}.
   * A shallow reference equality check is used to prevent copying of the same value by returning {@code this}.
   * @param elements An iterable of warnings elements to set
   * @return A modified copy of {@code this} object
   */
  public final ImmutableExecCreation withWarnings(@Nullable Iterable<String> elements) {
    if (this.warnings == elements) return this;
    @Nullable List<String> newValue = elements == null ? null : createUnmodifiableList(false, createSafeList(elements, true, false));
    return new ImmutableExecCreation(this.id, newValue);
  }

  /**
   * This instance is equal to all instances of {@code ImmutableExecCreation} that have equal attribute values.
   * @return {@code true} if {@code this} is equal to {@code another} instance
   */
  @Override
  public boolean equals(Object another) {
    if (this == another) return true;
    return another instanceof ImmutableExecCreation
        && equalTo(0, (ImmutableExecCreation) another);
  }

  private boolean equalTo(int synthetic, ImmutableExecCreation another) {
    return id.equals(another.id)
        && Objects.equals(warnings, another.warnings);
  }

  /**
   * Computes a hash code from attributes: {@code id}, {@code warnings}.
   * @return hashCode value
   */
  @Override
  public int hashCode() {
    int h = 5381;
    h += (h << 5) + id.hashCode();
    h += (h << 5) + Objects.hashCode(warnings);
    return h;
  }

  /**
   * Prints the immutable value {@code ExecCreation} with attribute values.
   * @return A string representation of the value
   */
  @Override
  public String toString() {
    return "ExecCreation{"
        + "id=" + id
        + ", warnings=" + warnings
        + "}";
  }

  /**
   * Creates an immutable copy of a {@link ExecCreation} value.
   * Uses accessors to get values to initialize the new immutable instance.
   * If an instance is already immutable, it is returned as is.
   * @param instance The instance to copy
   * @return A copied immutable ExecCreation instance
   */
  public static ImmutableExecCreation copyOf(ExecCreation instance) {
    if (instance instanceof ImmutableExecCreation) {
      return (ImmutableExecCreation) instance;
    }
    return ImmutableExecCreation.builder()
        .from(instance)
        .build();
  }

  /**
   * Creates a builder for {@link ImmutableExecCreation ImmutableExecCreation}.
   * <pre>
   * ImmutableExecCreation.builder()
   *    .id(String) // required {@link ExecCreation#id() id}
   *    .warnings(List&amp;lt;String&amp;gt; | null) // nullable {@link ExecCreation#warnings() warnings}
   *    .build();
   * </pre>
   * @return A new ImmutableExecCreation builder
   */
  public static ImmutableExecCreation.Builder builder() {
    return new ImmutableExecCreation.Builder();
  }

  /**
   * Builds instances of type {@link ImmutableExecCreation ImmutableExecCreation}.
   * Initialize attributes and then invoke the {@link #build()} method to create an
   * immutable instance.
   * <p><em>{@code Builder} is not thread-safe and generally should not be stored in a field or collection,
   * but instead used immediately to create instances.</em>
   */
  static final class Builder {
    private static final long INIT_BIT_ID = 0x1L;
    private long initBits = 0x1L;

    private String id;
    private List<String> warnings = null;

    private Builder() {
    }

    /**
     * Fill a builder with attribute values from the provided {@code ExecCreation} instance.
     * Regular attribute values will be replaced with those from the given instance.
     * Absent optional values will not replace present values.
     * Collection elements and entries will be added, not replaced.
     * @param instance The instance from which to copy values
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder from(ExecCreation instance) {
      Objects.requireNonNull(instance, "instance");
      this.id(instance.id());
      @Nullable List<String> warningsValue = instance.warnings();
      if (warningsValue != null) {
        addAllWarnings(warningsValue);
      }
      return this;
    }

    /**
     * Initializes the value for the {@link ExecCreation#id() id} attribute.
     * @param id The value for id 
     * @return {@code this} builder for use in a chained invocation
     */
    @JsonProperty("Id")
    public final Builder id(String id) {
      this.id = Objects.requireNonNull(id, "id");
      initBits &= ~INIT_BIT_ID;
      return this;
    }

    /**
     * Adds one element to {@link ExecCreation#warnings() warnings} list.
     * @param element A warnings element
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder warning(String element) {
      if (this.warnings == null) {
        this.warnings = new ArrayList<String>();
      }
      this.warnings.add(Objects.requireNonNull(element, "warnings element"));
      return this;
    }

    /**
     * Adds elements to {@link ExecCreation#warnings() warnings} list.
     * @param elements An array of warnings elements
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder warnings(String... elements) {
      if (this.warnings == null) {
        this.warnings = new ArrayList<String>();
      }
      for (String element : elements) {
        this.warnings.add(Objects.requireNonNull(element, "warnings element"));
      }
      return this;
    }


    /**
     * Sets or replaces all elements for {@link ExecCreation#warnings() warnings} list.
     * @param elements An iterable of warnings elements
     * @return {@code this} builder for use in a chained invocation
     */
    @JsonProperty("Warnings")
    public final Builder warnings(@Nullable Iterable<String> elements) {
      if (elements == null) {
        this.warnings = null;
        return this;
      }
      this.warnings = new ArrayList<String>();
      return addAllWarnings(elements);
    }

    /**
     * Adds elements to {@link ExecCreation#warnings() warnings} list.
     * @param elements An iterable of warnings elements
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder addAllWarnings(Iterable<String> elements) {
      Objects.requireNonNull(elements, "warnings element");
      if (this.warnings == null) {
        this.warnings = new ArrayList<String>();
      }
      for (String element : elements) {
        this.warnings.add(Objects.requireNonNull(element, "warnings element"));
      }
      return this;
    }

    /**
     * Builds a new {@link ImmutableExecCreation ImmutableExecCreation}.
     * @return An immutable instance of ExecCreation
     * @throws java.lang.IllegalStateException if any required attributes are missing
     */
    public ImmutableExecCreation build() {
      if (initBits != 0) {
        throw new IllegalStateException(formatRequiredAttributesMessage());
      }
      return new ImmutableExecCreation(id, warnings == null ? null : createUnmodifiableList(true, warnings));
    }

    private String formatRequiredAttributesMessage() {
      List<String> attributes = new ArrayList<>();
      if ((initBits & INIT_BIT_ID) != 0) attributes.add("id");
      return "Cannot build ExecCreation, some of required attributes are not set " + attributes;
    }
  }

  private static <T> List<T> createSafeList(Iterable<? extends T> iterable, boolean checkNulls, boolean skipNulls) {
    ArrayList<T> list;
    if (iterable instanceof Collection<?>) {
      int size = ((Collection<?>) iterable).size();
      if (size == 0) return Collections.emptyList();
      list = new ArrayList<>(size);
    } else {
      list = new ArrayList<>();
    }
    for (T element : iterable) {
      if (skipNulls && element == null) continue;
      if (checkNulls) Objects.requireNonNull(element, "element");
      list.add(element);
    }
    return list;
  }

  private static <T> List<T> createUnmodifiableList(boolean clone, List<T> list) {
    switch(list.size()) {
    case 0: return Collections.emptyList();
    case 1: return Collections.singletonList(list.get(0));
    default:
      if (clone) {
        return Collections.unmodifiableList(new ArrayList<>(list));
      } else {
        if (list instanceof ArrayList<?>) {
          ((ArrayList<?>) list).trimToSize();
        }
        return Collections.unmodifiableList(list);
      }
    }
  }
}
