/*
 * Copyright 2012-2025 CodeLibs Project and the Others.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.codelibs.nekohtml;

/**
 * Collection of HTML element information.
 * This class provides metadata and definitions for HTML elements used by the parser.
 *
 * @author Andy Clark
 * @author Ahmed Ashour
 * @author Marc Guillemot
 *
 * @version $Id: HTMLElements.java,v 1.12 2005/02/14 07:16:59 andyc Exp $
 */
public class HTMLElements {

    /**
     * Default constructor. This class serves as a static utility for HTML element definitions.
     */
    public HTMLElements() {
        // Default constructor
    }

    //
    // Constants
    //

    // element codes

    // NOTE: The element codes *must* start with 0 and increment in
    //       sequence. The parent and closes references depends on
    //       this assumption. -Ac

    /** Element code for A (anchor) element. */
    public static final short A = 0;
    /** Element code for ABBR (abbreviation) element. */
    public static final short ABBR = A + 1;
    /** Element code for ACRONYM element. */
    public static final short ACRONYM = ABBR + 1;
    /** Element code for ADDRESS element. */
    public static final short ADDRESS = ACRONYM + 1;
    /** Element code for APPLET element. */
    public static final short APPLET = ADDRESS + 1;
    /** Element code for AREA element. */
    public static final short AREA = APPLET + 1;
    /** Element code for ARTICLE element. */
    public static final short ARTICLE = AREA + 1;
    /** Element code for ASIDE element. */
    public static final short ASIDE = ARTICLE + 1;
    /** Element code for AUDIO element. */
    public static final short AUDIO = ASIDE + 1;
    /** Element code for B (bold) element. */
    public static final short B = AUDIO + 1;
    /** Element code for BASE element. */
    public static final short BASE = B + 1;
    /** Element code for BASEFONT element. */
    public static final short BASEFONT = BASE + 1;
    /** Element code for BDI (bidirectional isolate) element. */
    public static final short BDI = BASEFONT + 1;
    /** Element code for BDO (bidirectional override) element. */
    public static final short BDO = BDI + 1;
    /** Element code for BGSOUND element. */
    public static final short BGSOUND = BDO + 1;
    /** Element code for BIG element. */
    public static final short BIG = BGSOUND + 1;
    /** Element code for BLINK element. */
    public static final short BLINK = BIG + 1;
    /** Element code for BLOCKQUOTE element. */
    public static final short BLOCKQUOTE = BLINK + 1;
    /** Element code for BODY element. */
    public static final short BODY = BLOCKQUOTE + 1;
    /** Element code for BR (line break) element. */
    public static final short BR = BODY + 1;
    /** Element code for BUTTON element. */
    public static final short BUTTON = BR + 1;
    /** Element code for CANVAS element. */
    public static final short CANVAS = BUTTON + 1;
    /** Element code for CAPTION element. */
    public static final short CAPTION = CANVAS + 1;
    /** Element code for CENTER element. */
    public static final short CENTER = CAPTION + 1;
    /** Element code for CITE element. */
    public static final short CITE = CENTER + 1;
    /** Element code for CODE element. */
    public static final short CODE = CITE + 1;
    /** Element code for COL (column) element. */
    public static final short COL = CODE + 1;
    /** Element code for COLGROUP (column group) element. */
    public static final short COLGROUP = COL + 1;
    /** Element code for COMMENT element. */
    public static final short COMMENT = COLGROUP + 1;
    /** Element code for DATA element. */
    public static final short DATA = COMMENT + 1;
    /** Element code for DATALIST element. */
    public static final short DATALIST = DATA + 1;
    /** Element code for DEL (deleted text) element. */
    public static final short DEL = DATALIST + 1;
    /** Element code for DETAILS element. */
    public static final short DETAILS = DEL + 1;
    /** Element code for DFN (definition) element. */
    public static final short DFN = DETAILS + 1;
    /** Element code for DIALOG element. */
    public static final short DIALOG = DFN + 1;
    /** Element code for DIR (directory) element. */
    public static final short DIR = DIALOG + 1;
    /** Element code for DIV element. */
    public static final short DIV = DIR + 1;
    /** Element code for DD (description definition) element. */
    public static final short DD = DIV + 1;
    /** Element code for DL (description list) element. */
    public static final short DL = DD + 1;
    /** Element code for DT (description term) element. */
    public static final short DT = DL + 1;
    /** Element code for EM (emphasis) element. */
    public static final short EM = DT + 1;
    /** Element code for EMBED element. */
    public static final short EMBED = EM + 1;
    /** Element code for FIELDSET element. */
    public static final short FIELDSET = EMBED + 1;
    /** Element code for FIGCAPTION element. */
    public static final short FIGCAPTION = FIELDSET + 1;
    /** Element code for FIGURE element. */
    public static final short FIGURE = FIGCAPTION + 1;
    /** Element code for FONT element. */
    public static final short FONT = FIGURE + 1;
    /** Element code for FOOTER element. */
    public static final short FOOTER = FONT + 1;
    /** Element code for FORM element. */
    public static final short FORM = FOOTER + 1;
    /** Element code for FRAME element. */
    public static final short FRAME = FORM + 1;
    /** Element code for FRAMESET element. */
    public static final short FRAMESET = FRAME + 1;
    /** Element code for H1 (heading level 1) element. */
    public static final short H1 = FRAMESET + 1;
    /** Element code for H2 (heading level 2) element. */
    public static final short H2 = H1 + 1;
    /** Element code for H3 (heading level 3) element. */
    public static final short H3 = H2 + 1;
    /** Element code for H4 (heading level 4) element. */
    public static final short H4 = H3 + 1;
    /** Element code for H5 (heading level 5) element. */
    public static final short H5 = H4 + 1;
    /** Element code for H6 (heading level 6) element. */
    public static final short H6 = H5 + 1;
    /** Element code for HGROUP (heading group) element. */
    public static final short HGROUP = H6 + 1;
    /** Element code for HEAD element. */
    public static final short HEAD = HGROUP + 1;
    /** Element code for HEADER element. */
    public static final short HEADER = HEAD + 1;
    /** Element code for HR (horizontal rule) element. */
    public static final short HR = HEADER + 1;
    /** Element code for HTML element. */
    public static final short HTML = HR + 1;
    /** Element code for I (italic) element. */
    public static final short I = HTML + 1;
    /** Element code for IFRAME (inline frame) element. */
    public static final short IFRAME = I + 1;
    /** Element code for ILAYER element. */
    public static final short ILAYER = IFRAME + 1;
    /** Element code for IMG (image) element. */
    public static final short IMG = ILAYER + 1;
    /** Element code for INPUT element. */
    public static final short INPUT = IMG + 1;
    /** Element code for INS (inserted text) element. */
    public static final short INS = INPUT + 1;
    /** Element code for ISINDEX element. */
    public static final short ISINDEX = INS + 1;
    /** Element code for KBD (keyboard input) element. */
    public static final short KBD = ISINDEX + 1;
    /** Element code for KEYGEN element. */
    public static final short KEYGEN = KBD + 1;
    /** Element code for LABEL element. */
    public static final short LABEL = KEYGEN + 1;
    /** Element code for LAYER element. */
    public static final short LAYER = LABEL + 1;
    /** Element code for LEGEND element. */
    public static final short LEGEND = LAYER + 1;
    /** Element code for LI (list item) element. */
    public static final short LI = LEGEND + 1;
    /** Element code for LINK element. */
    public static final short LINK = LI + 1;
    /** Element code for LISTING element. */
    public static final short LISTING = LINK + 1;
    /** Element code for MAIN element. */
    public static final short MAIN = LISTING + 1;
    /** Element code for MAP element. */
    public static final short MAP = MAIN + 1;
    /** Element code for MARK element. */
    public static final short MARK = MAP + 1;
    /** Element code for MARQUEE element. */
    public static final short MARQUEE = MARK + 1;
    /** Element code for MENU element. */
    public static final short MENU = MARQUEE + 1;
    /** Element code for META element. */
    public static final short META = MENU + 1;
    /** Element code for METER element. */
    public static final short METER = META + 1;
    /** Element code for MULTICOL element. */
    public static final short MULTICOL = METER + 1;
    /** Element code for NAV (navigation) element. */
    public static final short NAV = MULTICOL + 1;
    /** Element code for NEXTID element. */
    public static final short NEXTID = NAV + 1;
    /** Element code for NOBR (no break) element. */
    public static final short NOBR = NEXTID + 1;
    /** Element code for NOEMBED element. */
    public static final short NOEMBED = NOBR + 1;
    /** Element code for NOFRAMES element. */
    public static final short NOFRAMES = NOEMBED + 1;
    /** Element code for NOLAYER element. */
    public static final short NOLAYER = NOFRAMES + 1;
    /** Element code for NOSCRIPT element. */
    public static final short NOSCRIPT = NOLAYER + 1;
    /** Element code for OBJECT element. */
    public static final short OBJECT = NOSCRIPT + 1;
    /** Element code for OL (ordered list) element. */
    public static final short OL = OBJECT + 1;
    /** Element code for OPTION element. */
    public static final short OPTION = OL + 1;
    /** Element code for OPTGROUP (option group) element. */
    public static final short OPTGROUP = OPTION + 1;
    /** Element code for OUTPUT element. */
    public static final short OUTPUT = OPTGROUP + 1;
    /** Element code for P (paragraph) element. */
    public static final short P = OUTPUT + 1;
    /** Element code for PARAM element. */
    public static final short PARAM = P + 1;
    /** Element code for PICTURE element. */
    public static final short PICTURE = PARAM + 1;
    /** Element code for PLAINTEXT element. */
    public static final short PLAINTEXT = PICTURE + 1;
    /** Element code for PRE (preformatted text) element. */
    public static final short PRE = PLAINTEXT + 1;
    /** Element code for PROGRESS element. */
    public static final short PROGRESS = PRE + 1;
    /** Element code for Q (quotation) element. */
    public static final short Q = PROGRESS + 1;
    /** Element code for RB (ruby base) element. */
    public static final short RB = Q + 1;
    /** Element code for RBC (ruby base container) element. */
    public static final short RBC = RB + 1;
    /** Element code for RP (ruby parentheses) element. */
    public static final short RP = RBC + 1;
    /** Element code for RT (ruby text) element. */
    public static final short RT = RP + 1;
    /** Element code for RTC (ruby text container) element. */
    public static final short RTC = RT + 1;
    /** Element code for RUBY element. */
    public static final short RUBY = RTC + 1;
    /** Element code for S (strikethrough) element. */
    public static final short S = RUBY + 1;
    /** Element code for SAMP (sample output) element. */
    public static final short SAMP = S + 1;
    /** Element code for SCRIPT element. */
    public static final short SCRIPT = SAMP + 1;
    /** Element code for SECTION element. */
    public static final short SECTION = SCRIPT + 1;
    /** Element code for SEARCH (search section) element. */
    public static final short SEARCH = SECTION + 1;
    /** Element code for SELECT element. */
    public static final short SELECT = SEARCH + 1;
    /** Element code for SLOT (Web Components slot) element. */
    public static final short SLOT = SELECT + 1;
    /** Element code for SMALL element. */
    public static final short SMALL = SLOT + 1;
    /** Element code for SOUND element. */
    public static final short SOUND = SMALL + 1;
    /** Element code for SOURCE element. */
    public static final short SOURCE = SOUND + 1;
    /** Element code for SPACER element. */
    public static final short SPACER = SOURCE + 1;
    /** Element code for SPAN element. */
    public static final short SPAN = SPACER + 1;
    /** Element code for STRIKE element. */
    public static final short STRIKE = SPAN + 1;
    /** Element code for STRONG element. */
    public static final short STRONG = STRIKE + 1;
    /** Element code for STYLE element. */
    public static final short STYLE = STRONG + 1;
    /** Element code for SUB (subscript) element. */
    public static final short SUB = STYLE + 1;
    /** Element code for SUMMARY element. */
    public static final short SUMMARY = SUB + 1;
    /** Element code for SUP (superscript) element. */
    public static final short SUP = SUMMARY + 1;
    /** Element code for TABLE element. */
    public static final short TABLE = SUP + 1;
    /** Element code for TEMPLATE element. */
    public static final short TEMPLATE = TABLE + 1;
    /** Element code for TBODY (table body) element. */
    public static final short TBODY = TEMPLATE + 1;
    /** Element code for TD (table data cell) element. */
    public static final short TD = TBODY + 1;
    /** Element code for TEXTAREA element. */
    public static final short TEXTAREA = TD + 1;
    /** Element code for TFOOT (table footer) element. */
    public static final short TFOOT = TEXTAREA + 1;
    /** Element code for TH (table header cell) element. */
    public static final short TH = TFOOT + 1;
    /** Element code for THEAD (table header) element. */
    public static final short THEAD = TH + 1;
    /** Element code for TIME element. */
    public static final short TIME = THEAD + 1;
    /** Element code for TITLE element. */
    public static final short TITLE = TIME + 1;
    /** Element code for TR (table row) element. */
    public static final short TR = TITLE + 1;
    /** Element code for TRACK element. */
    public static final short TRACK = TR + 1;
    /** Element code for TT (teletype text) element. */
    public static final short TT = TRACK + 1;
    /** Element code for U (underline) element. */
    public static final short U = TT + 1;
    /** Element code for UL (unordered list) element. */
    public static final short UL = U + 1;
    /** Element code for VAR (variable) element. */
    public static final short VAR = UL + 1;
    /** Element code for VIDEO element. */
    public static final short VIDEO = VAR + 1;
    /** Element code for WBR (word break opportunity) element. */
    public static final short WBR = VIDEO + 1;
    /** Element code for XML element. */
    public static final short XML = WBR + 1;
    /** Element code for XMP (example) element. */
    public static final short XMP = XML + 1;
    /** Element code for unknown/unrecognized elements. */
    public static final short UNKNOWN = XMP + 1;

    // information

    /** Element information organized by first letter. */
    protected static final Element[][] ELEMENTS_ARRAY = new Element[26][];

    /** Element information as a contiguous list. */
    protected static final ElementList ELEMENTS = new ElementList();

    /** No such element. */
    public static final Element NO_SUCH_ELEMENT = new Element(UNKNOWN, "", Element.CONTAINER, new short[] { BODY, HEAD }/*HTML*/, null);

    //
    // Static initializer
    //

    /**
     * Initializes the element information.
     * <p>
     * <strong>Note:</strong>
     * The <code>getElement</code> method requires that the HTML elements
     * are added to the list in alphabetical order. If new elements are
     * added, then they <em>must</em> be inserted in alphabetical order.
     */
    static {
        // <!ENTITY % heading "H1|H2|H3|H4|H5|H6">
        // <!ENTITY % fontstyle "TT | I | B | BIG | SMALL">
        // <!ENTITY % phrase "EM | STRONG | DFN | CODE | SAMP | KBD | VAR | CITE | ABBR | ACRONYM" >
        // <!ENTITY % special "A | IMG | OBJECT | BR | SCRIPT | MAP | Q | SUB | SUP | SPAN | BDO">
        // <!ENTITY % formctrl "INPUT | SELECT | TEXTAREA | LABEL | BUTTON">
        // <!ENTITY % inline "#PCDATA | %fontstyle; | %phrase; | %special; | %formctrl;">
        // <!ENTITY % block "P | %heading; | %list; | %preformatted; | DL | DIV | NOSCRIPT | BLOCKQUOTE | FORM | HR | TABLE | FIELDSET | ADDRESS">
        // <!ENTITY % flow "%block; | %inline;">

        // initialize array of element information
        ELEMENTS_ARRAY['A' - 'A'] = new Element[] {
                // A - - (%inline;)* -(A)
                new Element(A, "A", Element.CONTAINER, BODY, new short[] { A }),
                // ABBR - - (%inline;)*
                new Element(ABBR, "ABBR", Element.INLINE, BODY, null),
                // ACRONYM - - (%inline;)*
                new Element(ACRONYM, "ACRONYM", Element.INLINE, BODY, null),
                // ADDRESS - - (%inline;)*
                new Element(ADDRESS, "ADDRESS", Element.BLOCK, BODY, new short[] { P }),
                // APPLET
                new Element(APPLET, "APPLET", Element.CONTAINER, BODY, null),
                // AREA - O EMPTY
                new Element(AREA, "AREA", Element.EMPTY, MAP, null),
                // ARTICLE - flexible parent for real-world HTML patterns
                new Element(ARTICLE, "ARTICLE", Element.CONTAINER, new short[] { BODY, MAIN, SECTION, ARTICLE, ASIDE }, new short[] { P }),
                // ASIDE - flexible parent for real-world HTML patterns; closes P
                new Element(ASIDE, "ASIDE", Element.CONTAINER, new short[] { BODY, MAIN, ARTICLE, SECTION }, new short[] { P }),
                // AUDIO
                new Element(AUDIO, "AUDIO", Element.CONTAINER, BODY, null), };
        ELEMENTS_ARRAY['B' - 'A'] = new Element[] {
                // B - - (%inline;)*
                new Element(B, "B", Element.INLINE, BODY, null),
                // BASE - O EMPTY
                new Element(BASE, "BASE", Element.EMPTY, HEAD, null),
                // BASEFONT
                new Element(BASEFONT, "BASEFONT", Element.EMPTY, HEAD, null),
                // BDI
                new Element(BDI, "BDI", Element.INLINE, BODY, null),
                // BDO - - (%inline;)*
                new Element(BDO, "BDO", Element.INLINE, BODY, null),
                // BGSOUND
                new Element(BGSOUND, "BGSOUND", Element.EMPTY, HEAD, null),
                // BIG - - (%inline;)*
                new Element(BIG, "BIG", Element.INLINE, BODY, null),
                // BLINK
                new Element(BLINK, "BLINK", Element.INLINE, BODY, null),
                // BLOCKQUOTE - - (%block;|SCRIPT)+
                new Element(BLOCKQUOTE, "BLOCKQUOTE", Element.BLOCK, BODY, new short[] { P }),
                // BODY O O (%block;|SCRIPT)+ +(INS|DEL)
                new Element(BODY, "BODY", Element.CONTAINER, HTML, new short[] { HEAD }),
                // BR - O EMPTY
                new Element(BR, "BR", Element.EMPTY, BODY, null),
                // BUTTON - - (%flow;)* -(A|%formctrl;|FORM|FIELDSET)
                new Element(BUTTON, "BUTTON", Element.INLINE | Element.BLOCK, BODY, new short[] { BUTTON }), };
        ELEMENTS_ARRAY['C' - 'A'] = new Element[] {
                // CANVAS
                new Element(CANVAS, "CANVAS", Element.CONTAINER, BODY, null),
                // CAPTION - - (%inline;)*
                new Element(CAPTION, "CAPTION", Element.INLINE, TABLE, null),
                // CENTER,
                new Element(CENTER, "CENTER", Element.CONTAINER, BODY, new short[] { P }),
                // CITE - - (%inline;)*
                new Element(CITE, "CITE", Element.INLINE, BODY, null),
                // CODE - - (%inline;)*
                new Element(CODE, "CODE", Element.INLINE, BODY, null),
                // COL - O EMPTY
                new Element(COL, "COL", Element.EMPTY, TABLE, null),
                // COLGROUP - O (COL)*
                new Element(COLGROUP, "COLGROUP", Element.CONTAINER, TABLE, new short[] { COL, COLGROUP }),
                // COMMENT
                new Element(COMMENT, "COMMENT", Element.SPECIAL, HTML, null), };
        ELEMENTS_ARRAY['D' - 'A'] = new Element[] {
                // DATA
                new Element(DATA, "DATA", Element.INLINE, BODY, null),
                // DATALIST
                new Element(DATALIST, "DATALIST", Element.CONTAINER, BODY, null),
                // DEL - - (%flow;)*
                new Element(DEL, "DEL", Element.INLINE, BODY, null),
                // DETAILS
                new Element(DETAILS, "DETAILS", Element.BLOCK, BODY, new short[] { P }),
                // DFN - - (%inline;)*
                new Element(DFN, "DFN", Element.INLINE, BODY, null),
                // DIALOG
                new Element(DIALOG, "DIALOG", Element.BLOCK, BODY, new short[] { P }),
                // DIR
                new Element(DIR, "DIR", Element.CONTAINER, BODY, new short[] { P }),
                // DIV - - (%flow;)*
                new Element(DIV, "DIV", Element.CONTAINER, BODY, new short[] { P }),
                // DD - O (%flow;)*
                new Element(DD, "DD", Element.BLOCK, BODY, new short[] { DT, DD, P }),
                // DL - - (DT|DD)+
                new Element(DL, "DL", Element.BLOCK, BODY, new short[] { P }),
                // DT - O (%inline;)*
                new Element(DT, "DT", Element.BLOCK, BODY, new short[] { DT, DD, P }), };
        ELEMENTS_ARRAY['E' - 'A'] = new Element[] {
                // EM - - (%inline;)*
                new Element(EM, "EM", Element.INLINE, BODY, null),
                // EMBED
                new Element(EMBED, "EMBED", Element.EMPTY, BODY, null), };
        ELEMENTS_ARRAY['F' - 'A'] =
                new Element[] {
                        // FIELDSET - - (#PCDATA,LEGEND,(%flow;)*)
                        new Element(FIELDSET, "FIELDSET", Element.CONTAINER, BODY, new short[] { P }),
                        // FIGCAPTION
                        new Element(FIGCAPTION, "FIGCAPTION", Element.BLOCK, BODY, new short[] { P }),
                        // FIGURE
                        new Element(FIGURE, "FIGURE", Element.BLOCK, BODY, new short[] { P }),
                        // FONT
                        new Element(FONT, "FONT", Element.CONTAINER, BODY, null),
                        // FOOTER - flexible parent for real-world HTML patterns; closes P, FOOTER
                        new Element(FOOTER, "FOOTER", Element.CONTAINER, new short[] { BODY, MAIN, ARTICLE, SECTION, ASIDE }, new short[] {
                                P, FOOTER }),
                        // FORM - - (%block;|SCRIPT)+ -(FORM)
                        new Element(FORM, "FORM", Element.CONTAINER, new short[] { BODY, TD, DIV }, new short[] { BUTTON, P }),
                        // FRAME - O EMPTY
                        new Element(FRAME, "FRAME", Element.EMPTY, FRAMESET, null),
                        // FRAMESET - - ((FRAMESET|FRAME)+ & NOFRAMES?)
                        new Element(FRAMESET, "FRAMESET", Element.CONTAINER, HTML, null), };
        ELEMENTS_ARRAY['H' - 'A'] =
                new Element[] {
                        // (H1|H2|H3|H4|H5|H6) - - (%inline;)*
                        new Element(H1, "H1", Element.BLOCK, new short[] { BODY, A }, new short[] { H1, H2, H3, H4, H5, H6, P }),
                        new Element(H2, "H2", Element.BLOCK, new short[] { BODY, A }, new short[] { H1, H2, H3, H4, H5, H6, P }),
                        new Element(H3, "H3", Element.BLOCK, new short[] { BODY, A }, new short[] { H1, H2, H3, H4, H5, H6, P }),
                        new Element(H4, "H4", Element.BLOCK, new short[] { BODY, A }, new short[] { H1, H2, H3, H4, H5, H6, P }),
                        new Element(H5, "H5", Element.BLOCK, new short[] { BODY, A }, new short[] { H1, H2, H3, H4, H5, H6, P }),
                        new Element(H6, "H6", Element.BLOCK, new short[] { BODY, A }, new short[] { H1, H2, H3, H4, H5, H6, P }),
                        // HGROUP - - heading group
                        new Element(HGROUP, "HGROUP", Element.BLOCK, new short[] { BODY, SECTION, ARTICLE }, new short[] { P }),
                        // HEAD O O (%head.content;) +(%head.misc;)
                        new Element(HEAD, "HEAD", 0, HTML, null),
                        // HEADER - flexible parent for real-world HTML patterns; closes P, HEADER
                        new Element(HEADER, "HEADER", Element.CONTAINER, new short[] { BODY, MAIN, ARTICLE, SECTION, ASIDE }, new short[] {
                                P, HEADER }),
                        // HR - O EMPTY
                        new Element(HR, "HR", Element.EMPTY, BODY, new short[] { P }),
                        // HTML O O (%html.content;)
                        new Element(HTML, "HTML", 0, null, null), };
        ELEMENTS_ARRAY['I' - 'A'] = new Element[] {
                // I - - (%inline;)*
                new Element(I, "I", Element.INLINE, BODY, null),
                // IFRAME
                new Element(IFRAME, "IFRAME", Element.BLOCK, BODY, null),
                // ILAYER
                new Element(ILAYER, "ILAYER", Element.BLOCK, BODY, null),
                // IMG - O EMPTY
                new Element(IMG, "IMG", Element.EMPTY, BODY, null),
                // INPUT - O EMPTY
                new Element(INPUT, "INPUT", Element.EMPTY, BODY, null),
                // INS - - (%flow;)*
                new Element(INS, "INS", Element.INLINE, BODY, null),
                // ISINDEX
                new Element(ISINDEX, "ISINDEX", Element.EMPTY, HEAD, null), };
        ELEMENTS_ARRAY['K' - 'A'] = new Element[] {
                // KBD - - (%inline;)*
                new Element(KBD, "KBD", Element.INLINE, BODY, null),
                // KEYGEN
                new Element(KEYGEN, "KEYGEN", Element.EMPTY, BODY, null), };
        ELEMENTS_ARRAY['L' - 'A'] = new Element[] {
                // LABEL - - (%inline;)* -(LABEL)
                new Element(LABEL, "LABEL", Element.INLINE, BODY, null),
                // LAYER
                new Element(LAYER, "LAYER", Element.BLOCK, BODY, null),
                // LEGEND - - (%inline;)*
                new Element(LEGEND, "LEGEND", Element.INLINE, FIELDSET, null),
                // LI - O (%flow;)*
                new Element(LI, "LI", Element.CONTAINER, new short[] { BODY, UL, OL }, new short[] { LI, P }),
                // LINK - O EMPTY
                new Element(LINK, "LINK", Element.EMPTY, HEAD, null),
                // LISTING
                new Element(LISTING, "LISTING", Element.BLOCK, BODY, new short[] { P }), };
        ELEMENTS_ARRAY['M' - 'A'] = new Element[] {
                // MAIN - closes P, ASIDE, HEADER, FOOTER, NAV (body-level sections)
                new Element(MAIN, "MAIN", Element.CONTAINER, BODY, new short[] { P, ASIDE, HEADER, FOOTER, NAV }),
                // MAP - - ((%block;) | AREA)+
                new Element(MAP, "MAP", Element.INLINE, BODY, null),
                // MARK
                new Element(MARK, "MARK", Element.INLINE, BODY, null),
                // MARQUEE
                new Element(MARQUEE, "MARQUEE", Element.CONTAINER, BODY, null),
                // MENU
                new Element(MENU, "MENU", Element.CONTAINER, BODY, new short[] { P }),
                // META - O EMPTY
                new Element(META, "META", Element.EMPTY, HEAD, new short[] { STYLE, TITLE }),
                // METER
                new Element(METER, "METER", Element.CONTAINER, BODY, null),
                // MULTICOL
                new Element(MULTICOL, "MULTICOL", Element.CONTAINER, BODY, null), };
        ELEMENTS_ARRAY['N' - 'A'] =
                new Element[] {
                        // NAV - flexible parent for real-world HTML patterns; closes P, NAV
                        new Element(NAV, "NAV", Element.CONTAINER, new short[] { BODY, MAIN, ARTICLE, SECTION, ASIDE, HEADER, FOOTER },
                                new short[] { P, NAV }),
                        // NEXTID
                        new Element(NEXTID, "NEXTID", Element.EMPTY, BODY, null),
                        // NOBR
                        new Element(NOBR, "NOBR", Element.INLINE, BODY, new short[] { NOBR }),
                        // NOEMBED
                        new Element(NOEMBED, "NOEMBED", Element.CONTAINER, BODY, null),
                        // NOFRAMES - - (BODY) -(NOFRAMES)
                        new Element(NOFRAMES, "NOFRAMES", Element.CONTAINER, null, null),
                        // NOLAYER
                        new Element(NOLAYER, "NOLAYER", Element.CONTAINER, BODY, null),
                        // NOSCRIPT - - (%block;)+
                        new Element(NOSCRIPT, "NOSCRIPT", Element.CONTAINER, new short[] { BODY }, null), };
        ELEMENTS_ARRAY['O' - 'A'] = new Element[] {
                // OBJECT - - (PARAM | %flow;)*
                new Element(OBJECT, "OBJECT", Element.CONTAINER, BODY, null),
                // OL - - (LI)+
                new Element(OL, "OL", Element.BLOCK, BODY, new short[] { P }),
                // OPTGROUP - - (OPTION)+
                new Element(OPTGROUP, "OPTGROUP", 0, new short[] { SELECT }, new short[] { OPTION }),
                // OPTION - O (#PCDATA)
                new Element(OPTION, "OPTION", 0, new short[] { SELECT }, new short[] { OPTGROUP, OPTION }),
                // OUTPUT
                new Element(OUTPUT, "OUTPUT", Element.CONTAINER, BODY, null), };
        ELEMENTS_ARRAY['P' - 'A'] = new Element[] {
                // P - O (%inline;)*
                new Element(P, "P", Element.CONTAINER, BODY, new short[] { P }),
                // PARAM - O EMPTY
                new Element(PARAM, "PARAM", Element.EMPTY, new short[] { OBJECT, APPLET }, null),
                // PICTURE
                new Element(PICTURE, "PICTURE", Element.CONTAINER, BODY, null),
                // PLAINTEXT
                new Element(PLAINTEXT, "PLAINTEXT", Element.SPECIAL, BODY, null),
                // PRE - - (%inline;)* -(%pre.exclusion;)
                new Element(PRE, "PRE", Element.BLOCK, BODY, new short[] { P }),
                // PROGRESS
                new Element(PROGRESS, "PROGRESS", Element.CONTAINER, BODY, null), };
        ELEMENTS_ARRAY['Q' - 'A'] = new Element[] {
        // Q - - (%inline;)*
        new Element(Q, "Q", Element.INLINE, BODY, null), };
        ELEMENTS_ARRAY['R' - 'A'] = new Element[] {
                // RB
                new Element(RB, "RB", Element.INLINE, RUBY, new short[] { RB }),
                // RBC
                new Element(RBC, "RBC", 0, RUBY, null),
                // RP
                new Element(RP, "RP", Element.INLINE, RUBY, new short[] { RB }),
                // RT
                new Element(RT, "RT", Element.INLINE, RUBY, new short[] { RB, RP }),
                // RTC
                new Element(RTC, "RTC", 0, RUBY, new short[] { RBC }),
                // RUBY
                new Element(RUBY, "RUBY", Element.CONTAINER, BODY, new short[] { RUBY }), };
        ELEMENTS_ARRAY['S' - 'A'] = new Element[] {
                // S
                new Element(S, "S", Element.INLINE, BODY, null),
                // SAMP - - (%inline;)*
                new Element(SAMP, "SAMP", Element.INLINE, BODY, null),
                // SCRIPT - - %Script;
                new Element(SCRIPT, "SCRIPT", Element.SPECIAL, new short[] { HEAD, BODY }, null),
                // SECTION - can be in BODY, MAIN, ARTICLE, SECTION, ASIDE, NAV (flexible for real-world HTML)
                new Element(SECTION, "SECTION", Element.CONTAINER, new short[] { BODY, MAIN, ARTICLE, ASIDE }, new short[] { P }),
                // SEARCH - search section
                new Element(SEARCH, "SEARCH", Element.CONTAINER, BODY, new short[] { P }),
                // SELECT - - (OPTGROUP|OPTION)+
                new Element(SELECT, "SELECT", Element.CONTAINER, BODY, new short[] { SELECT }),
                // SLOT - Web Components slot
                new Element(SLOT, "SLOT", Element.CONTAINER, BODY, null),
                // SMALL - - (%inline;)*
                new Element(SMALL, "SMALL", Element.INLINE, BODY, null),
                // SOUND
                new Element(SOUND, "SOUND", Element.EMPTY, HEAD, null),
                // SOURCE
                new Element(SOURCE, "SOURCE", Element.EMPTY, BODY, null),
                // SPACER
                new Element(SPACER, "SPACER", Element.EMPTY, BODY, null),
                // SPAN - - (%inline;)*
                new Element(SPAN, "SPAN", Element.CONTAINER, BODY, null),
                // STRIKE
                new Element(STRIKE, "STRIKE", Element.INLINE, BODY, null),
                // STRONG - - (%inline;)*
                new Element(STRONG, "STRONG", Element.INLINE, BODY, null),
                // STYLE - - %StyleSheet;
                new Element(STYLE, "STYLE", Element.SPECIAL, new short[] { HEAD, BODY }, new short[] { STYLE, TITLE, META }),
                // SUB - - (%inline;)*
                new Element(SUB, "SUB", Element.INLINE, BODY, null),
                // SUMMARY
                new Element(SUMMARY, "SUMMARY", Element.INLINE, DETAILS, null),
                // SUP - - (%inline;)*
                new Element(SUP, "SUP", Element.INLINE, BODY, null), };
        ELEMENTS_ARRAY['T' - 'A'] =
                new Element[] {
                        // TABLE - - (CAPTION?, (COL*|COLGROUP*), THEAD?, TFOOT?, TBODY+)
                        new Element(TABLE, "TABLE", Element.BLOCK | Element.CONTAINER, BODY, null),
                        // TEMPLATE
                        new Element(TEMPLATE, "TEMPLATE", Element.SPECIAL, new short[] { HEAD, BODY }, null),
                        // TBODY O O (TR)+
                        new Element(TBODY, "TBODY", 0, TABLE, new short[] { THEAD, TBODY, TFOOT, TD, TH, TR, COLGROUP }),
                        // TD - O (%flow;)*
                        new Element(TD, "TD", Element.CONTAINER, TR, TABLE, new short[] { TD, TH }),
                        // TEXTAREA - - (#PCDATA)
                        new Element(TEXTAREA, "TEXTAREA", Element.SPECIAL, BODY, null),
                        // TFOOT - O (TR)+
                        new Element(TFOOT, "TFOOT", 0, TABLE, new short[] { THEAD, TBODY, TFOOT, TD, TH, TR }),
                        // TH - O (%flow;)*
                        new Element(TH, "TH", Element.CONTAINER, TR, TABLE, new short[] { TD, TH }),
                        // THEAD - O (TR)+
                        new Element(THEAD, "THEAD", 0, TABLE, new short[] { THEAD, TBODY, TFOOT, TD, TH, TR, COLGROUP }),
                        // TIME
                        new Element(TIME, "TIME", Element.INLINE, BODY, null),
                        // TITLE - - (#PCDATA) -(%head.misc;)
                        new Element(TITLE, "TITLE", Element.SPECIAL, new short[] { HEAD, BODY }, null),
                        // TR - O (TH|TD)+
                        new Element(TR, "TR", Element.BLOCK, new short[] { TBODY, THEAD, TFOOT }, TABLE, new short[] { TD, TH, TR,
                                COLGROUP, DIV }),
                        // TRACK
                        new Element(TRACK, "TRACK", Element.EMPTY, BODY, null),
                        // TT - - (%inline;)*
                        new Element(TT, "TT", Element.INLINE, BODY, null), };
        ELEMENTS_ARRAY['U' - 'A'] = new Element[] {
                // U,
                new Element(U, "U", Element.INLINE, BODY, null),
                // UL - - (LI)+
                new Element(UL, "UL", Element.CONTAINER, BODY, new short[] { P }), };
        ELEMENTS_ARRAY['V' - 'A'] = new Element[] {
                // VAR - - (%inline;)*
                new Element(VAR, "VAR", Element.INLINE, BODY, null),
                // VIDEO
                new Element(VIDEO, "VIDEO", Element.CONTAINER, BODY, null), };
        ELEMENTS_ARRAY['W' - 'A'] = new Element[] {
        // WBR
        new Element(WBR, "WBR", Element.EMPTY, BODY, null), };
        ELEMENTS_ARRAY['X' - 'A'] = new Element[] {
                // XML
                new Element(XML, "XML", 0, BODY, null),
                // XMP
                new Element(XMP, "XMP", Element.SPECIAL, BODY, new short[] { P }), };

        // keep contiguous list of elements for lookups by code
        for (final Element[] elements : ELEMENTS_ARRAY) {
            if (elements != null) {
                for (final Element element : elements) {
                    ELEMENTS.addElement(element);
                }
            }
        }
        ELEMENTS.addElement(NO_SUCH_ELEMENT);

        // initialize cross references to parent elements
        for (int i = 0; i < ELEMENTS.size; i++) {
            final Element element = ELEMENTS.data[i];
            if (element.parentCodes != null) {
                element.parent = new Element[element.parentCodes.length];
                for (int j = 0; j < element.parentCodes.length; j++) {
                    element.parent[j] = ELEMENTS.data[element.parentCodes[j]];
                }
                element.parentCodes = null;
            }
        }

    } // <clinit>()

    //
    // Public static methods
    //

    /**
     * Returns the element information for the specified element code.
     *
     * @param code The element code.
     * @return the element information for the specified code
     */
    public static final Element getElement(final short code) {
        return ELEMENTS.data[code];
    } // getElement(short):Element

    /**
     * Returns the element information for the specified element name.
     *
     * @param ename The element name.
     * @return the element information for the specified name
     */
    public static final Element getElement(final String ename) {
        Element element = getElement(ename, NO_SUCH_ELEMENT);
        if (element == NO_SUCH_ELEMENT) {
            element = new Element(UNKNOWN, ename.toUpperCase(), Element.CONTAINER, new short[] { BODY, HEAD }/*HTML*/, null);
            element.parent = NO_SUCH_ELEMENT.parent;
            element.parentCodes = NO_SUCH_ELEMENT.parentCodes;
        }
        return element;
    } // getElement(String):Element

    /**
     * Returns the element information for the specified element name.
     *
     * @param ename The element name.
     * @param element The default element to return if not found.
     * @return the element information for the specified name, or the default element if not found
     */
    public static final Element getElement(final String ename, final Element element) {

        if (!ename.isEmpty()) {
            int c = ename.charAt(0);
            if (c >= 'a' && c <= 'z') {
                c = 'A' + c - 'a';
            }
            if (c >= 'A' && c <= 'Z') {
                final Element[] elements = ELEMENTS_ARRAY[c - 'A'];
                if (elements != null) {
                    for (final Element elem : elements) {
                        if (elem.name.equalsIgnoreCase(ename)) {
                            return elem;
                        }
                    }
                }
            }
        }
        return element;

    } // getElement(String):Element

    /**
     * Checks if the specified element is a formatting element according to HTML Living Standard.
     * Formatting elements are subject to the Adoption Agency Algorithm.
     *
     * @param ename The element name (case-insensitive).
     * @return true if the element is a formatting element, false otherwise
     */
    public static final boolean isFormattingElement(final String ename) {
        if (ename == null || ename.isEmpty()) {
            return false;
        }
        final String upperName = ename.toUpperCase();
        return "A".equals(upperName) || "B".equals(upperName) || "BIG".equals(upperName) || "CODE".equals(upperName)
                || "EM".equals(upperName) || "FONT".equals(upperName) || "I".equals(upperName) || "NOBR".equals(upperName)
                || "S".equals(upperName) || "SMALL".equals(upperName) || "STRIKE".equals(upperName) || "STRONG".equals(upperName)
                || "TT".equals(upperName) || "U".equals(upperName);
    } // isFormattingElement(String):boolean

    /**
     * Checks if the specified element code is a formatting element.
     *
     * @param code The element code.
     * @return true if the element is a formatting element, false otherwise
     */
    public static final boolean isFormattingElement(final short code) {
        return code == A || code == B || code == BIG || code == CODE || code == EM || code == FONT || code == I || code == NOBR
                || code == S || code == SMALL || code == STRIKE || code == STRONG || code == TT || code == U;
    } // isFormattingElement(short):boolean

    //
    // Classes
    //

    /**
     * Element information.
     *
     * @author Andy Clark
     */
    public static class Element {

        //
        // Constants
        //

        /** Inline element. */
        public static final int INLINE = 0x01;

        /** Block element. */
        public static final int BLOCK = 0x02;

        /** Empty element. */
        public static final int EMPTY = 0x04;

        /** Container element. */
        public static final int CONTAINER = 0x08;

        /** Special element. */
        public static final int SPECIAL = 0x10;

        //
        // Data
        //

        /** The element code. */
        public short code;

        /** The element name. */
        public String name;

        /** Informational flags. */
        public int flags;

        /** Parent elements. */
        public short[] parentCodes;

        /** Parent elements. */
        public Element[] parent;

        /** The bounding element code. */
        public short bounds;

        /** List of elements this element can close. */
        public short[] closes;

        //
        // Constructors
        //

        /**
         * Constructs an element object.
         *
         * @param code The element code.
         * @param name The element name.
         * @param flags Informational flags
         * @param parent Natural closing parent name.
         * @param closes List of elements this element can close.
         */
        public Element(final short code, final String name, final int flags, final short parent, final short[] closes) {
            this(code, name, flags, new short[] { parent }, (short) -1, closes);
        } // <init>(short,String,int,short,short[]);

        /**
         * Constructs an element object.
         *
         * @param code The element code.
         * @param name The element name.
         * @param flags Informational flags
         * @param parent Natural closing parent name.
         * @param bounds Element boundary code for special parsing rules.
         * @param closes List of elements this element can close.
         */
        public Element(final short code, final String name, final int flags, final short parent, final short bounds, final short[] closes) {
            this(code, name, flags, new short[] { parent }, bounds, closes);
        } // <init>(short,String,int,short,short,short[])

        /**
         * Constructs an element object.
         *
         * @param code The element code.
         * @param name The element name.
         * @param flags Informational flags
         * @param parents Natural closing parent names.
         * @param closes List of elements this element can close.
         */
        public Element(final short code, final String name, final int flags, final short[] parents, final short[] closes) {
            this(code, name, flags, parents, (short) -1, closes);
        } // <init>(short,String,int,short[],short[])

        /**
         * Constructs an element object.
         *
         * @param code The element code.
         * @param name The element name.
         * @param flags Informational flags
         * @param parents Natural closing parent names.
         * @param bounds Element boundary code for special parsing rules.
         * @param closes List of elements this element can close.
         */
        public Element(final short code, final String name, final int flags, final short[] parents, final short bounds, final short[] closes) {
            this.code = code;
            this.name = name;
            this.flags = flags;
            this.parentCodes = parents;
            this.parent = null;
            this.bounds = bounds;
            this.closes = closes;
        } // <init>(short,String,int,short[],short,short[])

        //
        // Public methods
        //

        /**
         * Returns true if this element is an inline element.
         * @return true if this is an inline element, false otherwise
         */
        public final boolean isInline() {
            return (flags & INLINE) != 0;
        } // isInline():boolean

        /**
         * Returns true if this element is a block element.
         * @return true if this is a block element, false otherwise
         */
        public final boolean isBlock() {
            return (flags & BLOCK) != 0;
        } // isBlock():boolean

        /**
         * Returns true if this element is an empty element.
         * @return true if this is an empty element, false otherwise
         */
        public final boolean isEmpty() {
            return (flags & EMPTY) != 0;
        } // isEmpty():boolean

        /**
         * Returns true if this element is a container element.
         * @return true if this is a container element, false otherwise
         */
        public final boolean isContainer() {
            return (flags & CONTAINER) != 0;
        } // isContainer():boolean

        /**
         * Returns true if this element is special -- if its content
         * should be parsed ignoring markup.
         * @return true if this is a special element, false otherwise
         */
        public final boolean isSpecial() {
            return (flags & SPECIAL) != 0;
        } // isSpecial():boolean

        /**
         * Returns true if this element can close the specified Element.
         *
         * @param tag The element code to check.
         * @return true if this element can close the specified element
         */
        public boolean closes(final short tag) {
            if (closes != null) {
                for (final short close : closes) {
                    if (close == tag) {
                        return true;
                    }
                }
            }
            return false;

        } // closes(short):boolean

        //
        // Object methods
        //

        /** Returns a hash code for this object. */
        @Override
        public int hashCode() {
            return name.hashCode();
        } // hashCode():int

        /** Returns true if the objects are equal. */
        @Override
        public boolean equals(final Object o) {
            return name.equals(o);
        } // equals(Object):boolean

        /**
         * Provides a simple representation to make debugging easier
         */
        @Override
        public String toString() {
            return super.toString() + "(name=" + name + ")";
        }

        /**
         * Indicates if the provided element is an accepted parent of current element
         * @param element the element to test for "paternity"
         * @return <code>true</code> if <code>element</code> belongs to the {@link #parent}
         */
        public boolean isParent(final Element element) {
            if (parent == null) {
                return false;
            }
            for (final Element element2 : parent) {
                if (element.code == element2.code) {
                    return true;
                }
            }
            return false;
        }
    } // class Element

    /**
     * Unsynchronized list of elements.
     * This class provides a simple dynamic array implementation for storing Element objects.
     */
    public static class ElementList {

        /**
         * Default constructor. Initializes an empty element list with default capacity.
         */
        public ElementList() {
            // Default constructor
        }

        //
        // Data
        //

        /** The size of the list. */
        public int size;

        /** The data in the list. */
        public Element[] data = new Element[120];

        //
        // Public methods
        //

        /**
         * Adds an element to list, resizing if necessary.
         * @param element The element to add to the list
         */
        public void addElement(final Element element) {
            if (size == data.length) {
                final Element[] newarray = new Element[size + 20];
                System.arraycopy(data, 0, newarray, 0, size);
                data = newarray;
            }
            data[size++] = element;
        } // addElement(Element)

    } // class ElementList

} // class HTMLElements
