package org.mandas.docker.client.messages.swarm;

import com.fasterxml.jackson.annotation.JsonProperty;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import org.mandas.docker.Nullable;

/**
 * Immutable implementation of {@link EngineConfig}.
 * <p>
 * Use the builder to create immutable instances:
 * {@code ImmutableEngineConfig.builder()}.
 */
@SuppressWarnings({"all"})
final class ImmutableEngineConfig implements EngineConfig {
  private final String engineVersion;
  private final @Nullable Map<String, String> labels;
  private final @Nullable List<EnginePlugin> plugins;

  private ImmutableEngineConfig(
      String engineVersion,
      @Nullable Map<String, String> labels,
      @Nullable List<EnginePlugin> plugins) {
    this.engineVersion = engineVersion;
    this.labels = labels;
    this.plugins = plugins;
  }

  /**
   * @return The value of the {@code engineVersion} attribute
   */
  @JsonProperty("EngineVersion")
  @Override
  public String engineVersion() {
    return engineVersion;
  }

  /**
   * @return The value of the {@code labels} attribute
   */
  @JsonProperty("Labels")
  @Override
  public @Nullable Map<String, String> labels() {
    return labels;
  }

  /**
   * @return The value of the {@code plugins} attribute
   */
  @JsonProperty("Plugins")
  @Override
  public @Nullable List<EnginePlugin> plugins() {
    return plugins;
  }

  /**
   * Copy the current immutable object by setting a value for the {@link EngineConfig#engineVersion() engineVersion} attribute.
   * An equals check used to prevent copying of the same value by returning {@code this}.
   * @param value A new value for engineVersion
   * @return A modified copy or the {@code this} object
   */
  public final ImmutableEngineConfig withEngineVersion(String value) {
    String newValue = Objects.requireNonNull(value, "engineVersion");
    if (this.engineVersion.equals(newValue)) return this;
    return new ImmutableEngineConfig(newValue, this.labels, this.plugins);
  }

  /**
   * Copy the current immutable object by replacing the {@link EngineConfig#labels() labels} map with the specified map.
   * Nulls are not permitted as keys or values.
   * A shallow reference equality check is used to prevent copying of the same value by returning {@code this}.
   * @param entries The entries to replace the labels map
   * @return A modified copy or {@code this} if not changed
   */
  public final ImmutableEngineConfig withLabels(@Nullable Map<String, ? extends String> entries) {
    if (this.labels == entries) return this;
    @Nullable Map<String, String> newValue = entries == null ? null : createUnmodifiableMap(true, false, entries);
    return new ImmutableEngineConfig(this.engineVersion, newValue, this.plugins);
  }

  /**
   * Copy the current immutable object with elements that replace the content of {@link EngineConfig#plugins() plugins}.
   * @param elements The elements to set
   * @return A modified copy of {@code this} object
   */
  public final ImmutableEngineConfig withPlugins(@Nullable EnginePlugin... elements) {
    if (elements == null) {
      if (this.plugins == null) return this;
      return new ImmutableEngineConfig(this.engineVersion, this.labels, null);
    }
    List<EnginePlugin> newValue = createUnmodifiableList(false, createSafeList(Arrays.asList(elements), true, false));
    return new ImmutableEngineConfig(this.engineVersion, this.labels, newValue);
  }

  /**
   * Copy the current immutable object with elements that replace the content of {@link EngineConfig#plugins() plugins}.
   * A shallow reference equality check is used to prevent copying of the same value by returning {@code this}.
   * @param elements An iterable of plugins elements to set
   * @return A modified copy or {@code this} if not changed
   */
  public final ImmutableEngineConfig withPlugins(@Nullable Iterable<? extends EnginePlugin> elements) {
    if (this.plugins == elements) return this;
    @Nullable List<EnginePlugin> newValue = elements == null ? null : createUnmodifiableList(false, createSafeList(elements, true, false));
    return new ImmutableEngineConfig(this.engineVersion, this.labels, newValue);
  }

  /**
   * This instance is equal to all instances of {@code ImmutableEngineConfig} that have equal attribute values.
   * @return {@code true} if {@code this} is equal to {@code another} instance
   */
  @Override
  public boolean equals(Object another) {
    if (this == another) return true;
    return another instanceof ImmutableEngineConfig
        && equalsByValue((ImmutableEngineConfig) another);
  }

  private boolean equalsByValue(ImmutableEngineConfig another) {
    return engineVersion.equals(another.engineVersion)
        && Objects.equals(labels, another.labels)
        && Objects.equals(plugins, another.plugins);
  }

  /**
   * Computes a hash code from attributes: {@code engineVersion}, {@code labels}, {@code plugins}.
   * @return hashCode value
   */
  @Override
  public int hashCode() {
    int h = 5381;
    h += (h << 5) + engineVersion.hashCode();
    h += (h << 5) + Objects.hashCode(labels);
    h += (h << 5) + Objects.hashCode(plugins);
    return h;
  }

  /**
   * Prints the immutable value {@code EngineConfig} with attribute values.
   * @return A string representation of the value
   */
  @Override
  public String toString() {
    return "EngineConfig{"
        + "engineVersion=" + engineVersion
        + ", labels=" + labels
        + ", plugins=" + plugins
        + "}";
  }

  /**
   * Creates an immutable copy of a {@link EngineConfig} value.
   * Uses accessors to get values to initialize the new immutable instance.
   * If an instance is already immutable, it is returned as is.
   * @param instance The instance to copy
   * @return A copied immutable EngineConfig instance
   */
  public static ImmutableEngineConfig copyOf(EngineConfig instance) {
    if (instance instanceof ImmutableEngineConfig) {
      return (ImmutableEngineConfig) instance;
    }
    return ImmutableEngineConfig.builder()
        .from(instance)
        .build();
  }

  /**
   * Creates a builder for {@link ImmutableEngineConfig ImmutableEngineConfig}.
   * <pre>
   * ImmutableEngineConfig.builder()
   *    .engineVersion(String) // required {@link EngineConfig#engineVersion() engineVersion}
   *    .labels(Map&lt;String, String&gt; | null) // nullable {@link EngineConfig#labels() labels}
   *    .plugins(List&lt;org.mandas.docker.client.messages.swarm.EnginePlugin&gt; | null) // nullable {@link EngineConfig#plugins() plugins}
   *    .build();
   * </pre>
   * @return A new ImmutableEngineConfig builder
   */
  public static ImmutableEngineConfig.Builder builder() {
    return new ImmutableEngineConfig.Builder();
  }

  /**
   * Builds instances of type {@link ImmutableEngineConfig ImmutableEngineConfig}.
   * Initialize attributes and then invoke the {@link #build()} method to create an
   * immutable instance.
   * <p><em>{@code Builder} is not thread-safe and generally should not be stored in a field or collection,
   * but instead used immediately to create instances.</em>
   */
  static final class Builder {
    private static final long INIT_BIT_ENGINE_VERSION = 0x1L;
    private long initBits = 0x1L;

    private String engineVersion;
    private Map<String, String> labels = null;
    private List<EnginePlugin> plugins = null;

    private Builder() {
    }

    /**
     * Fill a builder with attribute values from the provided {@code EngineConfig} instance.
     * Regular attribute values will be replaced with those from the given instance.
     * Absent optional values will not replace present values.
     * Collection elements and entries will be added, not replaced.
     * @param instance The instance from which to copy values
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder from(EngineConfig instance) {
      Objects.requireNonNull(instance, "instance");
      this.engineVersion(instance.engineVersion());
      @Nullable Map<String, String> labelsValue = instance.labels();
      if (labelsValue != null) {
        putAllLabels(labelsValue);
      }
      @Nullable List<EnginePlugin> pluginsValue = instance.plugins();
      if (pluginsValue != null) {
        addAllPlugins(pluginsValue);
      }
      return this;
    }

    /**
     * Initializes the value for the {@link EngineConfig#engineVersion() engineVersion} attribute.
     * @param engineVersion The value for engineVersion 
     * @return {@code this} builder for use in a chained invocation
     */
    @JsonProperty("EngineVersion")
    public final Builder engineVersion(String engineVersion) {
      this.engineVersion = Objects.requireNonNull(engineVersion, "engineVersion");
      initBits &= ~INIT_BIT_ENGINE_VERSION;
      return this;
    }

    /**
     * Put one entry to the {@link EngineConfig#labels() labels} map.
     * @param key The key in the labels map
     * @param value The associated value in the labels map
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder addLabel(String key, String value) {
      if (this.labels == null) {
        this.labels = new LinkedHashMap<String, String>();
      }
      this.labels.put(
          Objects.requireNonNull(key, "labels key"),
          Objects.requireNonNull(value, value == null ? "labels value for key: " + key : null));
      return this;
    }

    /**
     * Put one entry to the {@link EngineConfig#labels() labels} map. Nulls are not permitted
     * @param entry The key and value entry
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder addLabel(Map.Entry<String, ? extends String> entry) {
      if (this.labels == null) {
        this.labels = new LinkedHashMap<String, String>();
      }
      String k = entry.getKey();
      String v = entry.getValue();
      this.labels.put(
          Objects.requireNonNull(k, "labels key"),
          Objects.requireNonNull(v, v == null ? "labels value for key: " + k : null));
      return this;
    }

    /**
     * Sets or replaces all mappings from the specified map as entries for the {@link EngineConfig#labels() labels} map. Nulls are not permitted as keys or values, but parameter itself can be null
     * @param entries The entries that will be added to the labels map
     * @return {@code this} builder for use in a chained invocation
     */
    @JsonProperty("Labels")
    public final Builder labels(@Nullable Map<String, ? extends String> entries) {
      if (entries == null) {
        this.labels = null;
        return this;
      }
      this.labels = new LinkedHashMap<String, String>();
      return putAllLabels(entries);
    }

    /**
     * Put all mappings from the specified map as entries to {@link EngineConfig#labels() labels} map. Nulls are not permitted
     * @param entries The entries that will be added to the labels map
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder putAllLabels(Map<String, ? extends String> entries) {
      if (this.labels == null) {
        this.labels = new LinkedHashMap<String, String>();
      }
      for (Map.Entry<String, ? extends String> e : entries.entrySet()) {
        String k = e.getKey();
        String v = e.getValue();
        this.labels.put(
            Objects.requireNonNull(k, "labels key"),
            Objects.requireNonNull(v, v == null ? "labels value for key: " + k : null));
      }
      return this;
    }

    /**
     * Adds one element to {@link EngineConfig#plugins() plugins} list.
     * @param element A plugins element
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder plugin(EnginePlugin element) {
      if (this.plugins == null) {
        this.plugins = new ArrayList<EnginePlugin>();
      }
      this.plugins.add(Objects.requireNonNull(element, "plugins element"));
      return this;
    }

    /**
     * Adds elements to {@link EngineConfig#plugins() plugins} list.
     * @param elements An array of plugins elements
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder plugins(EnginePlugin... elements) {
      if (this.plugins == null) {
        this.plugins = new ArrayList<EnginePlugin>();
      }
      for (EnginePlugin element : elements) {
        this.plugins.add(Objects.requireNonNull(element, "plugins element"));
      }
      return this;
    }


    /**
     * Sets or replaces all elements for {@link EngineConfig#plugins() plugins} list.
     * @param elements An iterable of plugins elements
     * @return {@code this} builder for use in a chained invocation
     */
    @JsonProperty("Plugins")
    public final Builder plugins(@Nullable Iterable<? extends EnginePlugin> elements) {
      if (elements == null) {
        this.plugins = null;
        return this;
      }
      this.plugins = new ArrayList<EnginePlugin>();
      return addAllPlugins(elements);
    }

    /**
     * Adds elements to {@link EngineConfig#plugins() plugins} list.
     * @param elements An iterable of plugins elements
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder addAllPlugins(Iterable<? extends EnginePlugin> elements) {
      Objects.requireNonNull(elements, "plugins element");
      if (this.plugins == null) {
        this.plugins = new ArrayList<EnginePlugin>();
      }
      for (EnginePlugin element : elements) {
        this.plugins.add(Objects.requireNonNull(element, "plugins element"));
      }
      return this;
    }

    /**
     * Builds a new {@link ImmutableEngineConfig ImmutableEngineConfig}.
     * @return An immutable instance of EngineConfig
     * @throws java.lang.IllegalStateException if any required attributes are missing
     */
    public ImmutableEngineConfig build() {
      if (initBits != 0) {
        throw new IllegalStateException(formatRequiredAttributesMessage());
      }
      return new ImmutableEngineConfig(
          engineVersion,
          labels == null ? null : createUnmodifiableMap(false, false, labels),
          plugins == null ? null : createUnmodifiableList(true, plugins));
    }

    private String formatRequiredAttributesMessage() {
      List<String> attributes = new ArrayList<>();
      if ((initBits & INIT_BIT_ENGINE_VERSION) != 0) attributes.add("engineVersion");
      return "Cannot build EngineConfig, some of required attributes are not set " + attributes;
    }
  }

  private static <T> List<T> createSafeList(Iterable<? extends T> iterable, boolean checkNulls, boolean skipNulls) {
    ArrayList<T> list;
    if (iterable instanceof Collection<?>) {
      int size = ((Collection<?>) iterable).size();
      if (size == 0) return Collections.emptyList();
      list = new ArrayList<>(size);
    } else {
      list = new ArrayList<>();
    }
    for (T element : iterable) {
      if (skipNulls && element == null) continue;
      if (checkNulls) Objects.requireNonNull(element, "element");
      list.add(element);
    }
    return list;
  }

  private static <T> List<T> createUnmodifiableList(boolean clone, List<? extends T> list) {
    switch(list.size()) {
    case 0: return Collections.emptyList();
    case 1: return Collections.singletonList(list.get(0));
    default:
      if (clone) {
        return Collections.unmodifiableList(new ArrayList<>(list));
      } else {
        if (list instanceof ArrayList<?>) {
          ((ArrayList<?>) list).trimToSize();
        }
        return Collections.unmodifiableList(list);
      }
    }
  }

  private static <K, V> Map<K, V> createUnmodifiableMap(boolean checkNulls, boolean skipNulls, Map<? extends K, ? extends V> map) {
    switch (map.size()) {
    case 0: return Collections.emptyMap();
    case 1: {
      Map.Entry<? extends K, ? extends V> e = map.entrySet().iterator().next();
      K k = e.getKey();
      V v = e.getValue();
      if (checkNulls) {
        Objects.requireNonNull(k, "key");
        Objects.requireNonNull(v, v == null ? "value for key: " + k : null);
      }
      if (skipNulls && (k == null || v == null)) {
        return Collections.emptyMap();
      }
      return Collections.singletonMap(k, v);
    }
    default: {
      Map<K, V> linkedMap = new LinkedHashMap<>(map.size() * 4 / 3 + 1);
      if (skipNulls || checkNulls) {
        for (Map.Entry<? extends K, ? extends V> e : map.entrySet()) {
          K k = e.getKey();
          V v = e.getValue();
          if (skipNulls) {
            if (k == null || v == null) continue;
          } else if (checkNulls) {
            Objects.requireNonNull(k, "key");
            Objects.requireNonNull(v, v == null ? "value for key: " + k : null);
          }
          linkedMap.put(k, v);
        }
      } else {
        linkedMap.putAll(map);
      }
      return Collections.unmodifiableMap(linkedMap);
    }
    }
  }
}
