"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.getEmbeddedFilesByLevel = exports.forEachEmbeddedDocument = exports.safeCall = exports.languageFeatureWorker = exports.documentFeatureWorker = void 0;
async function documentFeatureWorker(context, uri, valid, worker, transformResult, combineResult) {
    return languageFeatureWorker(context, uri, () => void 0, function* (map) {
        if (valid(map)) {
            yield;
        }
    }, worker, transformResult, combineResult);
}
exports.documentFeatureWorker = documentFeatureWorker;
async function languageFeatureWorker(context, uri, getReadDocParams, eachVirtualDocParams, worker, transformResult, combineResult) {
    const sourceScript = context.language.scripts.get(uri);
    if (!sourceScript) {
        return;
    }
    let results = [];
    if (sourceScript.generated) {
        for (const map of forEachEmbeddedDocument(context, sourceScript.id, sourceScript.generated.root)) {
            for (const mappedArg of eachVirtualDocParams(map)) {
                for (const [serviceId, service] of Object.entries(context.services)) {
                    if (context.disabledServicePlugins.has(service[1])) {
                        continue;
                    }
                    const embeddedResult = await safeCall(() => worker(service, map.embeddedDocument, mappedArg, map), 'service ' + serviceId + ' crashed on ' + map.embeddedDocument.uri);
                    if (!embeddedResult) {
                        continue;
                    }
                    const result = transformResult(embeddedResult, map);
                    if (!result) {
                        continue;
                    }
                    results.push(result);
                    if (!combineResult) {
                        break;
                    }
                }
            }
        }
    }
    else {
        const document = context.documents.get(uri, sourceScript.languageId, sourceScript.snapshot);
        const params = getReadDocParams();
        for (const [serviceId, service] of Object.entries(context.services)) {
            if (context.disabledServicePlugins.has(service[1])) {
                continue;
            }
            const embeddedResult = await safeCall(() => worker(service, document, params, undefined), 'service ' + serviceId + ' crashed on ' + uri);
            if (!embeddedResult) {
                continue;
            }
            const result = transformResult(embeddedResult, undefined);
            if (!result) {
                continue;
            }
            results.push(result);
            if (!combineResult) {
                break;
            }
        }
    }
    if (combineResult && results.length > 0) {
        return combineResult(results);
    }
    else if (results.length > 0) {
        return results[0];
    }
}
exports.languageFeatureWorker = languageFeatureWorker;
async function safeCall(cb, errorMsg) {
    try {
        return await cb();
    }
    catch (err) {
        console.warn(errorMsg, err);
    }
}
exports.safeCall = safeCall;
function* forEachEmbeddedDocument(context, sourceScriptId, current) {
    if (current.embeddedCodes) {
        for (const embeddedCode of current.embeddedCodes) {
            yield* forEachEmbeddedDocument(context, sourceScriptId, embeddedCode);
        }
    }
    for (const map of context.documents.getMaps(current)) {
        if (sourceScriptId === map.sourceDocument.uri
            && !context.disabledEmbeddedDocumentUris.has(context.encodeEmbeddedDocumentUri(sourceScriptId, current.id))) {
            yield map;
        }
    }
}
exports.forEachEmbeddedDocument = forEachEmbeddedDocument;
function getEmbeddedFilesByLevel(context, sourceFileUri, rootFile, level) {
    const embeddedFilesByLevel = [[rootFile]];
    while (true) {
        if (embeddedFilesByLevel.length > level) {
            return embeddedFilesByLevel[level];
        }
        const nextLevel = [];
        for (const file of embeddedFilesByLevel[embeddedFilesByLevel.length - 1]) {
            if (file.embeddedCodes) {
                for (const embedded of file.embeddedCodes) {
                    if (!context.disabledEmbeddedDocumentUris.has(context.encodeEmbeddedDocumentUri(sourceFileUri, embedded.id))) {
                        nextLevel.push(embedded);
                    }
                }
            }
        }
        embeddedFilesByLevel.push(nextLevel);
    }
}
exports.getEmbeddedFilesByLevel = getEmbeddedFilesByLevel;
//# sourceMappingURL=featureWorkers.js.map