/*
 * Copyright (c) 2015 Goldman Sachs.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * and Eclipse Distribution License v. 1.0 which accompany this distribution.
 * The Eclipse Public License is available at http://www.eclipse.org/legal/epl-v10.html
 * and the Eclipse Distribution License is available at
 * http://www.eclipse.org/org/documents/edl-v10.php.
 */

package org.eclipse.collections.impl.bag.immutable.primitive;

import java.io.Externalizable;
import java.io.IOException;
import java.io.ObjectInput;
import java.io.ObjectOutput;
import java.io.Serializable;

import org.eclipse.collections.api.FloatIterable;
import org.eclipse.collections.api.LazyFloatIterable;
import org.eclipse.collections.api.bag.ImmutableBag;
import org.eclipse.collections.api.bag.primitive.FloatBag;
import org.eclipse.collections.api.bag.primitive.ImmutableFloatBag;
import org.eclipse.collections.api.bag.primitive.MutableFloatBag;
import org.eclipse.collections.api.block.function.primitive.FloatToObjectFunction;
import org.eclipse.collections.api.block.function.primitive.ObjectFloatToObjectFunction;
import org.eclipse.collections.api.block.predicate.primitive.FloatPredicate;
import org.eclipse.collections.api.block.procedure.primitive.FloatIntProcedure;
import org.eclipse.collections.api.block.procedure.primitive.FloatProcedure;
import org.eclipse.collections.api.iterator.FloatIterator;
import org.eclipse.collections.api.list.primitive.MutableFloatList;
import org.eclipse.collections.api.set.primitive.MutableFloatSet;
import org.eclipse.collections.impl.bag.mutable.primitive.FloatHashBag;
import org.eclipse.collections.impl.block.procedure.checked.primitive.CheckedFloatIntProcedure;
import org.eclipse.collections.impl.iterator.UnmodifiableFloatIterator;

/**
 * ImmutableFloatHashBag is the non-modifiable equivalent of {@link FloatHashBag}.
 * This file was automatically generated from template file immutablePrimitiveHashBag.stg.
 *
 * @since 4.0.
 */
final class ImmutableFloatHashBag implements ImmutableFloatBag, Serializable
{
    private static final long serialVersionUID = 1L;

    private final MutableFloatBag delegate;

    private ImmutableFloatHashBag(float[] newElements)
    {
        this.checkOptimizedSize(newElements.length);
        this.delegate = FloatHashBag.newBagWith(newElements);
    }

    private void checkOptimizedSize(int length)
    {
        if (length <= 1)
        {
            throw new IllegalArgumentException("Use FloatBags.immutable.with() to instantiate an optimized collection");
        }
    }

    public static ImmutableFloatHashBag newBagWith(float... elements)
    {
        return new ImmutableFloatHashBag(elements);
    }

    public ImmutableFloatBag newWith(float element)
    {
        return FloatHashBag.newBag(this.delegate).with(element).toImmutable();
    }

    public ImmutableFloatBag newWithout(float element)
    {
        FloatHashBag hashBag = FloatHashBag.newBag(this.delegate);
        hashBag.remove(element);
        return hashBag.toImmutable();
    }

    public ImmutableFloatBag newWithAll(FloatIterable elements)
    {
        FloatHashBag bag = FloatHashBag.newBag(this.delegate);
        bag.addAll(elements);
        return bag.toImmutable();
    }

    public ImmutableFloatBag newWithoutAll(FloatIterable elements)
    {
        FloatHashBag bag = FloatHashBag.newBag(this.delegate);
        bag.removeAll(elements);
        return bag.toImmutable();
    }

    public int size()
    {
        return this.delegate.size();
    }

    public boolean isEmpty()
    {
        return this.delegate.isEmpty();
    }

    public boolean notEmpty()
    {
        return this.delegate.notEmpty();
    }

    public boolean contains(float value)
    {
        return this.delegate.contains(value);
    }

    public boolean containsAll(FloatIterable source)
    {
        return this.delegate.containsAll(source);
    }

    public boolean containsAll(float... elements)
    {
        return this.delegate.containsAll(elements);
    }

    public void forEach(FloatProcedure procedure)
    {
        this.each(procedure);
    }

    /**
     * @since 7.0.
     */
    public void each(FloatProcedure procedure)
    {
        this.delegate.forEach(procedure);
    }

    public ImmutableFloatBag select(FloatPredicate predicate)
    {
        return this.delegate.select(predicate).toImmutable();
    }

    public ImmutableFloatBag reject(FloatPredicate predicate)
    {
        return this.delegate.reject(predicate).toImmutable();
    }

    public <V> ImmutableBag<V> collect(FloatToObjectFunction<? extends V> function)
    {
        return this.delegate.collect(function).toImmutable();
    }

    public MutableFloatList toList()
    {
        return this.delegate.toList();
    }

    public int sizeDistinct()
    {
        return this.delegate.sizeDistinct();
    }

    public int occurrencesOf(float item)
    {
        return this.delegate.occurrencesOf(item);
    }

    public void forEachWithOccurrences(FloatIntProcedure floatIntProcedure)
    {
        this.delegate.forEachWithOccurrences(floatIntProcedure);
    }

    public float detectIfNone(FloatPredicate predicate, float ifNone)
    {
        return this.delegate.detectIfNone(predicate, ifNone);
    }

    public int count(FloatPredicate predicate)
    {
        return this.delegate.count(predicate);
    }

    public boolean anySatisfy(FloatPredicate predicate)
    {
        return this.delegate.anySatisfy(predicate);
    }

    public double sum()
    {
        return this.delegate.sum();
    }

    public float min()
    {
        return this.delegate.min();
    }

    public float max()
    {
        return this.delegate.max();
    }

    public float maxIfEmpty(float defaultValue)
    {
        return this.delegate.maxIfEmpty(defaultValue);
    }

    public float minIfEmpty(float defaultValue)
    {
        return this.delegate.minIfEmpty(defaultValue);
    }

    public double average()
    {
        return this.delegate.average();
    }

    public double median()
    {
        return this.delegate.median();
    }

    public float[] toSortedArray()
    {
        return this.delegate.toSortedArray();
    }

    public MutableFloatList toSortedList()
    {
        return this.delegate.toSortedList();
    }

    public boolean noneSatisfy(FloatPredicate predicate)
    {
        return this.delegate.noneSatisfy(predicate);
    }

    public boolean allSatisfy(FloatPredicate predicate)
    {
        return this.delegate.allSatisfy(predicate);
    }

    public <T> T injectInto(T injectedValue, ObjectFloatToObjectFunction<? super T, ? extends T> function)
    {
        return this.delegate.injectInto(injectedValue, function);
    }

    @Override
    public boolean equals(Object obj)
    {
        return this.delegate.equals(obj);
    }

    @Override
    public int hashCode()
    {
        return this.delegate.hashCode();
    }

    public MutableFloatSet toSet()
    {
        return this.delegate.toSet();
    }

    public MutableFloatBag toBag()
    {
        return this.delegate.toBag();
    }

    public ImmutableFloatBag toImmutable()
    {
        return this;
    }

    public LazyFloatIterable asLazy()
    {
        return this.delegate.asLazy();
    }

    public float[] toArray()
    {
        return this.delegate.toArray();
    }

    @Override
    public String toString()
    {
        return this.delegate.toString();
    }

    public String makeString()
    {
        return this.delegate.makeString();
    }

    public String makeString(String separator)
    {
        return this.delegate.makeString(separator);
    }

    public String makeString(String start, String separator, String end)
    {
        return this.delegate.makeString(start, separator, end);
    }

    public void appendString(Appendable appendable)
    {
        this.delegate.appendString(appendable);
    }

    public void appendString(Appendable appendable, String separator)
    {
        this.delegate.appendString(appendable, separator);
    }

    public void appendString(Appendable appendable, String start, String separator, String end)
    {
        this.delegate.appendString(appendable, start, separator, end);
    }

    public FloatIterator floatIterator()
    {
        return new UnmodifiableFloatIterator(this.delegate.floatIterator());
    }

    private Object writeReplace()
    {
        return new ImmutableFloatBagSerializationProxy(this);
    }

    protected static class ImmutableFloatBagSerializationProxy implements Externalizable
    {
        private static final long serialVersionUID = 1L;

        private FloatBag bag;

        @SuppressWarnings("UnusedDeclaration")
        public ImmutableFloatBagSerializationProxy()
        {
            // Empty constructor for Externalizable class
        }

        protected ImmutableFloatBagSerializationProxy(FloatBag bag)
        {
            this.bag = bag;
        }

        public void writeExternal(final ObjectOutput out) throws IOException
        {
            out.writeInt(this.bag.sizeDistinct());
            try
            {
                this.bag.forEachWithOccurrences(new CheckedFloatIntProcedure()
                {
                    @Override
                    public void safeValue(float item, int count) throws IOException
                    {
                        out.writeFloat(item);
                        out.writeInt(count);
                    }
                });
            }
            catch (RuntimeException e)
            {
                if (e.getCause() instanceof IOException)
                {
                    throw (IOException) e.getCause();
                }
                throw e;
            }
        }

        public void readExternal(ObjectInput in) throws IOException, ClassNotFoundException
        {
            int size = in.readInt();
            MutableFloatBag deserializedBag = new FloatHashBag();

            for (int i = 0; i < size; i++)
            {
                deserializedBag.addOccurrences(in.readFloat(), in.readInt());
            }

            this.bag = deserializedBag;
        }

        protected Object readResolve()
        {
            return this.bag.toImmutable();
        }
    }
}
