/*
 * Copyright (c) 2015 Goldman Sachs.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * and Eclipse Distribution License v. 1.0 which accompany this distribution.
 * The Eclipse Public License is available at http://www.eclipse.org/legal/epl-v10.html
 * and the Eclipse Distribution License is available at
 * http://www.eclipse.org/org/documents/edl-v10.php.
 */

package org.eclipse.collections.impl.map.mutable.primitive;

import java.io.Externalizable;
import java.io.IOException;
import java.io.ObjectInput;
import java.io.ObjectOutput;
import java.lang.reflect.Array;
import java.util.Arrays;
import java.util.Collection;
import java.util.Iterator;
import java.util.NoSuchElementException;
import java.util.Set;

import org.eclipse.collections.api.FloatIterable;
import org.eclipse.collections.api.LazyFloatIterable;
import org.eclipse.collections.api.LazyIterable;
import org.eclipse.collections.api.RichIterable;
import org.eclipse.collections.api.bag.primitive.MutableFloatBag;
import org.eclipse.collections.api.block.function.primitive.FloatFunction;
import org.eclipse.collections.api.block.function.primitive.FloatFunction0;
import org.eclipse.collections.api.block.function.primitive.FloatToFloatFunction;
import org.eclipse.collections.api.block.function.primitive.FloatToObjectFunction;
import org.eclipse.collections.api.block.function.primitive.ObjectFloatToObjectFunction;
import org.eclipse.collections.api.block.predicate.primitive.FloatPredicate;
import org.eclipse.collections.api.block.predicate.primitive.ObjectFloatPredicate;
import org.eclipse.collections.api.block.procedure.Procedure;
import org.eclipse.collections.api.block.procedure.Procedure2;
import org.eclipse.collections.api.block.procedure.primitive.FloatProcedure;
import org.eclipse.collections.api.block.procedure.primitive.ObjectFloatProcedure;
import org.eclipse.collections.api.collection.MutableCollection;
import org.eclipse.collections.api.collection.primitive.ImmutableFloatCollection;
import org.eclipse.collections.api.collection.primitive.MutableFloatCollection;
import org.eclipse.collections.api.iterator.FloatIterator;
import org.eclipse.collections.api.iterator.MutableFloatIterator;
import org.eclipse.collections.api.list.MutableList;
import org.eclipse.collections.api.list.primitive.MutableFloatList;
import org.eclipse.collections.api.map.primitive.ImmutableObjectFloatMap;
import org.eclipse.collections.api.map.primitive.MutableObjectFloatMap;
import org.eclipse.collections.api.map.primitive.ObjectFloatMap;
import org.eclipse.collections.api.set.primitive.FloatSet;
import org.eclipse.collections.api.set.primitive.MutableFloatSet;
import org.eclipse.collections.api.tuple.primitive.ObjectFloatPair;
import org.eclipse.collections.impl.bag.mutable.primitive.FloatHashBag;
import org.eclipse.collections.impl.collection.mutable.primitive.SynchronizedFloatCollection;
import org.eclipse.collections.impl.collection.mutable.primitive.UnmodifiableFloatCollection;
import org.eclipse.collections.impl.lazy.AbstractLazyIterable;
import org.eclipse.collections.impl.factory.primitive.FloatLists;
import org.eclipse.collections.impl.factory.primitive.ObjectFloatMaps;
import org.eclipse.collections.impl.lazy.primitive.LazyFloatIterableAdapter;
import org.eclipse.collections.impl.list.mutable.FastList;
import org.eclipse.collections.impl.list.mutable.primitive.FloatArrayList;
import org.eclipse.collections.impl.set.mutable.primitive.FloatHashSet;
import org.eclipse.collections.api.block.procedure.primitive.ObjectIntProcedure;
import org.eclipse.collections.impl.tuple.primitive.PrimitiveTuples;
/**
 * This file was automatically generated from template file objectPrimitiveHashMap.stg.
 *
 * @since 3.0.
 */
public class ObjectFloatHashMap<K> implements MutableObjectFloatMap<K>, Externalizable
{
    public static final float EMPTY_VALUE = 0.0f;

    private static final long serialVersionUID = 1L;
    private static final int OCCUPIED_DATA_RATIO = 2;
    private static final int OCCUPIED_SENTINEL_RATIO = 4;
    private static final int DEFAULT_INITIAL_CAPACITY = 8;

    private static final Object NULL_KEY = new Object()
    {
        @Override
        public boolean equals(Object obj)
        {
            throw new RuntimeException("Possible corruption through unsynchronized concurrent modification.");
        }

        @Override
        public int hashCode()
        {
            throw new RuntimeException("Possible corruption through unsynchronized concurrent modification.");
        }

        @Override
        public String toString()
        {
            return "ObjectFloatHashMap.NULL_KEY";
        }
    };

    private static final Object REMOVED_KEY = new Object()
    {
        @Override
        public boolean equals(Object obj)
        {
            throw new RuntimeException("Possible corruption through unsynchronized concurrent modification.");
        }

        @Override
        public int hashCode()
        {
            throw new RuntimeException("Possible corruption through unsynchronized concurrent modification.");
        }

        @Override
        public String toString()
        {
            return "ObjectFloatHashMap.REMOVED_KEY";
        }
    };

    private Object[] keys;
    private float[] values;

    private int occupiedWithData;
    private int occupiedWithSentinels;

    public ObjectFloatHashMap()
    {
        this.allocateTable(DEFAULT_INITIAL_CAPACITY << 1);
    }

    public ObjectFloatHashMap(int initialCapacity)
    {
        if (initialCapacity < 0)
        {
            throw new IllegalArgumentException("initial capacity cannot be less than 0");
        }
        int capacity = this.smallestPowerOfTwoGreaterThan(this.fastCeil(initialCapacity * OCCUPIED_DATA_RATIO));
        this.allocateTable(capacity);
    }

    public ObjectFloatHashMap(ObjectFloatMap<? extends K> map)
    {
        this(Math.max(map.size(), DEFAULT_INITIAL_CAPACITY));
        this.putAll(map);
    }

    public static <K> ObjectFloatHashMap<K> newMap()
    {
        return new ObjectFloatHashMap<K>();
    }

    public static <K> ObjectFloatHashMap<K> newWithKeysValues(K key1, float value1)
    {
        return new ObjectFloatHashMap<K>(1).withKeyValue(key1, value1);
    }

    public static <K> ObjectFloatHashMap<K> newWithKeysValues(K key1, float value1, K key2, float value2)
    {
        return new ObjectFloatHashMap<K>(2).withKeysValues(key1, value1, key2, value2);
    }

    public static <K> ObjectFloatHashMap<K> newWithKeysValues(K key1, float value1, K key2, float value2, K key3, float value3)
    {
        return new ObjectFloatHashMap<K>(3).withKeysValues(key1, value1, key2, value2, key3, value3);
    }

    public static <K> ObjectFloatHashMap<K> newWithKeysValues(K key1, float value1, K key2, float value2, K key3, float value3, K key4, float value4)
    {
        return new ObjectFloatHashMap<K>(4).withKeysValues(key1, value1, key2, value2, key3, value3, key4, value4);
    }

    private int smallestPowerOfTwoGreaterThan(int n)
    {
        return n > 1 ? Integer.highestOneBit(n - 1) << 1 : 1;
    }

    private int fastCeil(float v)
    {
        int possibleResult = (int) v;
        if (v - possibleResult > 0.0F)
        {
            possibleResult++;
        }
        return possibleResult;
    }

    @Override
    public boolean equals(Object obj)
    {
        if (this == obj)
        {
            return true;
        }

        if (!(obj instanceof ObjectFloatMap))
        {
            return false;
        }

        ObjectFloatMap<K> other = (ObjectFloatMap<K>) obj;

        if (this.size() != other.size())
        {
            return false;
        }

        for (int i = 0; i < this.keys.length; i++)
        {
            if (isNonSentinel(this.keys[i]) && (!other.containsKey(this.toNonSentinel(this.keys[i])) || Float.compare(this.values[i], other.getOrThrow(this.toNonSentinel(this.keys[i]))) != 0))
            {
                return false;
            }
        }
        return true;
    }

    @Override
    public int hashCode()
    {
        int result = 0;

        for (int i = 0; i < this.keys.length; i++)
        {
            if (isNonSentinel(this.keys[i]))
            {
                result += (this.toNonSentinel(this.keys[i]) == null ? 0 : this.keys[i].hashCode()) ^ Float.floatToIntBits(this.values[i]);
            }
        }
        return result;
    }

    @Override
    public String toString()
    {
        StringBuilder appendable = new StringBuilder();

        appendable.append("{");

        boolean first = true;

        for (int i = 0; i < this.keys.length; i++)
        {
            Object key = this.keys[i];
            if (isNonSentinel(key))
            {
                if (!first)
                {
                    appendable.append(", ");
                }
                appendable.append(this.toNonSentinel(key)).append("=").append(this.values[i]);
                first = false;
            }
        }
        appendable.append("}");

        return appendable.toString();
    }

    public int size()
    {
        return this.occupiedWithData;
    }

    public boolean isEmpty()
    {
        return this.size() == 0;
    }

    public boolean notEmpty()
    {
        return this.size() != 0;
    }

    public String makeString()
    {
        return this.makeString(", ");
    }

    public String makeString(String separator)
    {
        return this.makeString("", separator, "");
    }

    public String makeString(String start, String separator, String end)
    {
        Appendable stringBuilder = new StringBuilder();
        this.appendString(stringBuilder, start, separator, end);
        return stringBuilder.toString();
    }

    public void appendString(Appendable appendable)
    {
        this.appendString(appendable, ", ");
    }

    public void appendString(Appendable appendable, String separator)
    {
        this.appendString(appendable, "", separator, "");
    }

    public void appendString(Appendable appendable, String start, String separator, String end)
    {
        try
        {
            appendable.append(start);

            boolean first = true;

            for (int i = 0; i < this.keys.length; i++)
            {
                Object key = this.keys[i];
                if (isNonSentinel(key))
                {
                    if (!first)
                    {
                        appendable.append(separator);
                    }
                    appendable.append(String.valueOf(String.valueOf(this.values[i])));
                    first = false;
                }
            }
            appendable.append(end);
        }
        catch (IOException e)
        {
            throw new RuntimeException(e);
        }
    }

    public MutableFloatIterator floatIterator()
    {
        return new InternalFloatIterator();
    }

    public float[] toArray()
    {
        float[] result = new float[this.size()];
        int index = 0;

        for (int i = 0; i < this.keys.length; i++)
        {
            if (isNonSentinel(this.keys[i]))
            {
                result[index] = this.values[i];
                index++;
            }
        }
        return result;
    }

    public boolean contains(float value)
    {
        return this.containsValue(value);
    }

    public boolean containsAll(float... source)
    {
        for (float item : source)
        {
            if (!this.containsValue(item))
            {
                return false;
            }
        }
        return true;
    }

    public boolean containsAll(FloatIterable source)
    {
        return this.containsAll(source.toArray());
    }

    public void clear()
    {
        this.occupiedWithData = 0;
        this.occupiedWithSentinels = 0;
        Arrays.fill(this.keys, null);
        Arrays.fill(this.values, EMPTY_VALUE);
    }

    public void put(K key, float value)
    {
        int index = this.probe(key);

        if (isNonSentinel(this.keys[index]) && nullSafeEquals(this.toNonSentinel(this.keys[index]), key))
        {
            // key already present in map
            this.values[index] = value;
            return;
        }

        this.addKeyValueAtIndex(key, value, index);
    }

    public void putAll(ObjectFloatMap<? extends K> map)
    {
        map.forEachKeyValue(new ObjectFloatProcedure<K>()
        {
            public void value(K key, float value)
            {
                ObjectFloatHashMap.this.put(key, value);
            }
        });
    }

    public void removeKey(K key)
    {
        int index = this.probe(key);
        this.removeKeyAtIndex(key, index);
    }

    private void removeKeyAtIndex(K key, int index)
    {
        if (isNonSentinel(this.keys[index]) && nullSafeEquals(this.toNonSentinel(this.keys[index]), key))
        {
            this.keys[index] = REMOVED_KEY;
            this.values[index] = EMPTY_VALUE;
            this.occupiedWithData--;
            this.occupiedWithSentinels++;
        }
    }

    public void remove(Object key)
    {
        this.removeKey((K) key);
    }

    public float removeKeyIfAbsent(K key, float value)
    {
        int index = this.probe(key);
        if (isNonSentinel(this.keys[index]) && nullSafeEquals(this.toNonSentinel(this.keys[index]), key))
        {
            this.keys[index] = REMOVED_KEY;
            float oldValue = this.values[index];
            this.values[index] = EMPTY_VALUE;
            this.occupiedWithData--;
            this.occupiedWithSentinels++;

            return oldValue;
        }
        return value;
    }

    public float getIfAbsentPut(K key, float value)
    {
        int index = this.probe(key);
        if (isNonSentinel(this.keys[index]) && nullSafeEquals(this.toNonSentinel(this.keys[index]), key))
        {
            return this.values[index];
        }
        this.addKeyValueAtIndex(key, value, index);
        return value;
    }

    public float getIfAbsentPut(K key, FloatFunction0 function)
    {
        int index = this.probe(key);
        if (isNonSentinel(this.keys[index]) && nullSafeEquals(this.toNonSentinel(this.keys[index]), key))
        {
            return this.values[index];
        }
        float value = function.value();
        this.addKeyValueAtIndex(key, value, index);
        return value;
    }

    public <P> float getIfAbsentPutWith(K key, FloatFunction<? super P> function, P parameter)
    {
        int index = this.probe(key);
        if (isNonSentinel(this.keys[index]) && nullSafeEquals(this.toNonSentinel(this.keys[index]), key))
        {
            return this.values[index];
        }
        float value = function.floatValueOf(parameter);
        this.addKeyValueAtIndex(key, value, index);
        return value;
    }

    public float getIfAbsentPutWithKey(K key, FloatFunction<? super K> function)
    {
        int index = this.probe(key);
        if (isNonSentinel(this.keys[index]) && nullSafeEquals(this.toNonSentinel(this.keys[index]), key))
        {
            return this.values[index];
        }
        float value = function.floatValueOf(key);
        this.addKeyValueAtIndex(key, value, index);
        return value;
    }

    public float updateValue(K key, float initialValueIfAbsent, FloatToFloatFunction function)
    {
        int index = this.probe(key);
        if (isNonSentinel(this.keys[index]) && nullSafeEquals(this.toNonSentinel(this.keys[index]), key))
        {
            this.values[index] = function.valueOf(this.values[index]);
            return this.values[index];
        }
        float value = function.valueOf(initialValueIfAbsent);
        this.addKeyValueAtIndex(key, value, index);
        return value;
    }

    private void addKeyValueAtIndex(K key, float value, int index)
    {
        if (this.keys[index] == REMOVED_KEY)
        {
            --this.occupiedWithSentinels;
        }
        this.keys[index] = toSentinelIfNull(key);
        this.values[index] = value;
        ++this.occupiedWithData;
        if (this.occupiedWithData + this.occupiedWithSentinels > this.maxOccupiedWithData())
        {
            this.rehashAndGrow();
        }
    }

    public float addToValue(K key, float toBeAdded)
    {
        int index = this.probe(key);
        if (isNonSentinel(this.keys[index]) && nullSafeEquals(this.toNonSentinel(this.keys[index]), key))
        {
            this.values[index] += toBeAdded;
            return this.values[index];
        }
        this.addKeyValueAtIndex(key, toBeAdded, index);
        return this.values[index];
    }

    public ObjectFloatHashMap<K> withKeyValue(K key1, float value1)
    {
        this.put(key1, value1);
        return this;
    }

    public ObjectFloatHashMap<K> withKeysValues(K key1, float value1, K key2, float value2)
    {
        this.put(key1, value1);
        this.put(key2, value2);
        return this;
    }

    public ObjectFloatHashMap<K> withKeysValues(K key1, float value1, K key2, float value2, K key3, float value3)
    {
        this.put(key1, value1);
        this.put(key2, value2);
        this.put(key3, value3);
        return this;
    }

    public ObjectFloatHashMap<K> withKeysValues(K key1, float value1, K key2, float value2, K key3, float value3, K key4, float value4)
    {
        this.put(key1, value1);
        this.put(key2, value2);
        this.put(key3, value3);
        this.put(key4, value4);
        return this;
    }

    public ObjectFloatHashMap<K> withoutKey(K key)
    {
        this.removeKey(key);
        return this;
    }

    public ObjectFloatHashMap<K> withoutAllKeys(Iterable<? extends K> keys)
    {
        for (K key : keys)
        {
            this.removeKey(key);
        }
        return this;
    }

    public MutableObjectFloatMap<K> asUnmodifiable()
    {
        return new UnmodifiableObjectFloatMap<K>(this);
    }

    public MutableObjectFloatMap<K> asSynchronized()
    {
        return new SynchronizedObjectFloatMap<K>(this);
    }

    public ImmutableObjectFloatMap<K> toImmutable()
    {
        return ObjectFloatMaps.immutable.withAll(this);
    }

    public float get(Object key)
    {
        return this.getIfAbsent(key, EMPTY_VALUE);
    }

    public float getOrThrow(Object key)
    {
        int index = this.probe(key);
        if (isNonSentinel(this.keys[index]))
        {
            return this.values[index];
        }
        throw new IllegalStateException("Key " + key + " not present.");
    }

    public float getIfAbsent(Object key, float ifAbsent)
    {
        int index = this.probe(key);
        if (isNonSentinel(this.keys[index]) && nullSafeEquals(this.toNonSentinel(this.keys[index]), key))
        {
            return this.values[index];
        }
        return ifAbsent;
    }

    public boolean containsKey(Object key)
    {
        int index = this.probe(key);
        return isNonSentinel(this.keys[index]) && nullSafeEquals(this.toNonSentinel(this.keys[index]), key);
    }

    public boolean containsValue(float value)
    {
        for (int i = 0; i < this.values.length; i++)
        {
            if (isNonSentinel(this.keys[i]) && Float.compare(this.values[i], value) == 0)
            {
                return true;
            }
        }
        return false;
    }

    public void forEach(FloatProcedure procedure)
    {
        this.each(procedure);
    }

    /**
     * @since 7.0.
     */
    public void each(FloatProcedure procedure)
    {
        this.forEachValue(procedure);
    }

    public void forEachValue(FloatProcedure procedure)
    {
        for (int i = 0; i < this.keys.length; i++)
        {
            if (isNonSentinel(this.keys[i]))
            {
                procedure.value(this.values[i]);
            }
        }
    }

    public void forEachKey(Procedure<? super K> procedure)
    {
        for (int i = 0; i < this.keys.length; i++)
        {
            if (isNonSentinel(this.keys[i]))
            {
                procedure.value(this.toNonSentinel(this.keys[i]));
            }
        }
    }

    public void forEachKeyValue(ObjectFloatProcedure<? super K> procedure)
    {
        for (int i = 0; i < this.keys.length; i++)
        {
            if (isNonSentinel(this.keys[i]))
            {
                procedure.value(this.toNonSentinel(this.keys[i]), this.values[i]);
            }
        }
    }

    public ObjectFloatHashMap<K> select(ObjectFloatPredicate<? super K> predicate)
    {
        ObjectFloatHashMap<K> result = ObjectFloatHashMap.newMap();

        for (int i = 0; i < this.keys.length; i++)
        {
            if (isNonSentinel(this.keys[i]) && predicate.accept(this.toNonSentinel(this.keys[i]), this.values[i]))
            {
                result.put(this.toNonSentinel(this.keys[i]), this.values[i]);
            }
        }
        return result;
    }

    public ObjectFloatHashMap<K> reject(ObjectFloatPredicate<? super K> predicate)
    {
        ObjectFloatHashMap<K> result = ObjectFloatHashMap.newMap();

        for (int i = 0; i < this.keys.length; i++)
        {
            if (isNonSentinel(this.keys[i]) && !predicate.accept(this.toNonSentinel(this.keys[i]), this.values[i]))
            {
                result.put(this.toNonSentinel(this.keys[i]), this.values[i]);
            }
        }
        return result;
    }

    public MutableFloatCollection select(FloatPredicate predicate)
    {
        FloatArrayList result = new FloatArrayList();

        for (int i = 0; i < this.keys.length; i++)
        {
            if (isNonSentinel(this.keys[i]) && predicate.accept(this.values[i]))
            {
                result.add(this.values[i]);
            }
        }
        return result;
    }

    public MutableFloatCollection reject(FloatPredicate predicate)
    {
        FloatArrayList result = new FloatArrayList();

        for (int i = 0; i < this.keys.length; i++)
        {
            if (isNonSentinel(this.keys[i]) && !predicate.accept(this.values[i]))
            {
                result.add(this.values[i]);
            }
        }
        return result;
    }

    public float detectIfNone(FloatPredicate predicate, float ifNone)
    {
        for (int i = 0; i < this.keys.length; i++)
        {
            if (isNonSentinel(this.keys[i]) && predicate.accept(this.values[i]))
            {
                return this.values[i];
            }
        }
        return ifNone;
    }

    public <V> MutableCollection<V> collect(FloatToObjectFunction<? extends V> function)
    {
        MutableList<V> result = FastList.newList(this.size());
        for (int i = 0; i < this.keys.length; i++)
        {
            if (isNonSentinel(this.keys[i]))
            {
                result.add(function.valueOf(this.values[i]));
            }
        }
        return result;
    }

    public int count(FloatPredicate predicate)
    {
        int count = 0;

        for (int i = 0; i < this.keys.length; i++)
        {
            if (isNonSentinel(this.keys[i]) && predicate.accept(this.values[i]))
            {
                count++;
            }
        }
        return count;
    }

    public boolean anySatisfy(FloatPredicate predicate)
    {
        for (int i = 0; i < this.keys.length; i++)
        {
            if (isNonSentinel(this.keys[i]) && predicate.accept(this.values[i]))
            {
                return true;
            }
        }
        return false;
    }

    public boolean allSatisfy(FloatPredicate predicate)
    {
        for (int i = 0; i < this.keys.length; i++)
        {
            if (isNonSentinel(this.keys[i]) && !predicate.accept(this.values[i]))
            {
                return false;
            }
        }
        return true;
    }

    public boolean noneSatisfy(FloatPredicate predicate)
    {
        for (int i = 0; i < this.keys.length; i++)
        {
            if (isNonSentinel(this.keys[i]) && predicate.accept(this.values[i]))
            {
                return false;
            }
        }
        return true;
    }

    public <V> V injectInto(V injectedValue, ObjectFloatToObjectFunction<? super V, ? extends V> function)
    {
        V result = injectedValue;

        for (int i = 0; i < this.keys.length; i++)
        {
            if (isNonSentinel(this.keys[i]))
            {
                result = function.valueOf(result, this.values[i]);
            }
        }

        return result;
    }

    public double sum()
    {
        double result = 0.0;
        double compensation = 0.0;

        for (int i = 0; i < this.keys.length; i++)
        {
            if (isNonSentinel(this.keys[i]))
            {
                double adjustedValue = this.values[i] - compensation;
                double nextSum = result + adjustedValue;
                compensation = nextSum - result - adjustedValue;
                result = nextSum;
            }
        }
        return result;
    }

    public float max()
    {
        if (this.isEmpty())
        {
            throw new NoSuchElementException();
        }
        float max = 0.0f;
        boolean isMaxSet = false;

        for (int i = 0; i < this.keys.length; i++)
        {
            if (isNonSentinel(this.keys[i]) && (!isMaxSet || Float.compare(max, this.values[i]) < 0))
            {
                max = this.values[i];
                isMaxSet = true;
            }
        }
        return max;
    }

    public float min()
    {
        if (this.isEmpty())
        {
            throw new NoSuchElementException();
        }
        float min = 0.0f;
        boolean isMinSet = false;

        for (int i = 0; i < this.keys.length; i++)
        {
            if (isNonSentinel(this.keys[i]) && (!isMinSet || Float.compare(this.values[i], min) < 0))
            {
                min = this.values[i];
                isMinSet = true;
            }
        }
        return min;
    }

    public float maxIfEmpty(float defaultValue)
    {
        if (this.isEmpty())
        {
            return defaultValue;
        }
        float max = 0.0f;
        boolean isMaxSet = false;

        for (int i = 0; i < this.keys.length; i++)
        {
            if (isNonSentinel(this.keys[i]) && (!isMaxSet || Float.compare(max, this.values[i]) < 0))
            {
                max = this.values[i];
                isMaxSet = true;
            }
        }
        return max;
    }

    public float minIfEmpty(float defaultValue)
    {
        if (this.isEmpty())
        {
            return defaultValue;
        }
        float min = 0.0f;
        boolean isMinSet = false;

        for (int i = 0; i < this.keys.length; i++)
        {
            if (isNonSentinel(this.keys[i]) && (!isMinSet || Float.compare(this.values[i], min) < 0))
            {
                min = this.values[i];
                isMinSet = true;
            }
        }
        return min;
    }

    public double average()
    {
        if (this.isEmpty())
        {
            throw new ArithmeticException();
        }
        return this.sum() / (double) this.size();
    }

    public double median()
    {
        if (this.isEmpty())
        {
            throw new ArithmeticException();
        }
        float[] sortedArray = this.toSortedArray();
        int middleIndex = sortedArray.length >> 1;
        if (sortedArray.length > 1 && (sortedArray.length & 1) == 0)
        {
            float first = sortedArray[middleIndex];
            float second = sortedArray[middleIndex - 1];
            return ((double) first + (double) second) / 2.0;
        }
        return (double) sortedArray[middleIndex];
    }

    public MutableFloatList toList()
    {
        MutableFloatList result = new FloatArrayList(this.size());

        for (int i = 0; i < this.keys.length; i++)
        {
            if (isNonSentinel(this.keys[i]))
            {
                result.add(this.values[i]);
            }
        }
        return result;
    }

    public MutableFloatSet toSet()
    {
        MutableFloatSet result = new FloatHashSet(this.size());

        for (int i = 0; i < this.keys.length; i++)
        {
            if (isNonSentinel(this.keys[i]))
            {
                result.add(this.values[i]);
            }
        }
        return result;
    }

    public MutableFloatBag toBag()
    {
        MutableFloatBag result = new FloatHashBag(this.size());

        for (int i = 0; i < this.keys.length; i++)
        {
            if (isNonSentinel(this.keys[i]))
            {
                result.add(this.values[i]);
            }
        }
        return result;
    }

    public LazyFloatIterable asLazy()
    {
        return new LazyFloatIterableAdapter(this);
    }

    public float[] toSortedArray()
    {
        float[] array = this.toArray();
        Arrays.sort(array);
        return array;
    }

    public MutableFloatList toSortedList()
    {
        return this.toList().sortThis();
    }

    public void writeExternal(ObjectOutput out) throws IOException
    {
        out.writeInt(this.size());
        for (int i = 0; i < this.keys.length; i++)
        {
            if (isNonSentinel(this.keys[i]))
            {
                out.writeObject(this.toNonSentinel(this.keys[i]));
                out.writeFloat(this.values[i]);
            }
        }
    }

    public void readExternal(ObjectInput in) throws IOException, ClassNotFoundException
    {
        int size = in.readInt();
        int capacity = this.smallestPowerOfTwoGreaterThan(this.fastCeil(size * OCCUPIED_DATA_RATIO));
        this.allocateTable(capacity);

        for (int i = 0; i < size; i++)
        {
            this.put((K) in.readObject(), in.readFloat());
        }
    }

    public LazyIterable<K> keysView()
    {
        return new KeysView();
    }

    public RichIterable<ObjectFloatPair<K>> keyValuesView()
    {
        return new KeyValuesView();
    }

    /**
     * Rehashes every element in the set into a new backing table of the smallest possible size and eliminating removed sentinels.
     */
    public void compact()
    {
        this.rehash(this.smallestPowerOfTwoGreaterThan(this.size()));
    }

    private void rehashAndGrow()
    {
        this.rehash(this.keys.length << 1);
    }

    private void rehash(int newCapacity)
    {
        int oldLength = this.keys.length;
        Object[] old = this.keys;
        float[] oldValues = this.values;
        this.allocateTable(newCapacity);
        this.occupiedWithData = 0;
        this.occupiedWithSentinels = 0;

        for (int i = 0; i < oldLength; i++)
        {
            if (isNonSentinel(old[i]))
            {
                this.put(this.toNonSentinel(old[i]), oldValues[i]);
            }
        }
    }

    // exposed for testing
    int probe(Object element)
    {
        int index = this.spread(element);

        int removedIndex = -1;
        if (isRemovedKey(this.keys[index]))
        {
            removedIndex = index;
        }

        else if (this.keys[index] == null || nullSafeEquals(this.toNonSentinel(this.keys[index]), element))
        {
            return index;
        }

        int nextIndex = index;
        int probe = 17;

        // loop until an empty slot is reached
        while (true)
        {
            // Probe algorithm: 17*n*(n+1)/2 where n = no. of collisions
            nextIndex += probe;
            probe += 17;
            nextIndex &= this.keys.length - 1;

            if (isRemovedKey(this.keys[nextIndex]))
            {
                if (removedIndex == -1)
                {
                    removedIndex = nextIndex;
                }
            }
            else if (nullSafeEquals(this.toNonSentinel(this.keys[nextIndex]), element))
            {
                return nextIndex;
            }
            else if (this.keys[nextIndex] == null)
            {
                return removedIndex == -1 ? nextIndex : removedIndex;
            }
        }
    }

    // exposed for testing
    int spread(Object element)
    {
        // This function ensures that hashCodes that differ only by
        // constant multiples at each bit position have a bounded
        // number of collisions (approximately 8 at default load factor).
        int h = element == null ? 0 : element.hashCode();
        h ^= h >>> 20 ^ h >>> 12;
        h ^= h >>> 7 ^ h >>> 4;
        return h & (this.keys.length - 1);
    }

    private static boolean nullSafeEquals(Object value, Object other)
    {
        if (value == null)
        {
            if (other == null)
            {
                return true;
            }
        }
        else if (other == value || value.equals(other))
        {
            return true;
        }
        return false;
    }

    private void allocateTable(int sizeToAllocate)
    {
        this.keys = new Object[sizeToAllocate];
        this.values = new float[sizeToAllocate];
    }

    private static boolean isRemovedKey(Object key)
    {
        return key == REMOVED_KEY;
    }

    private static <K> boolean isNonSentinel(K key)
    {
        return key != null && !isRemovedKey(key);
    }

    private K toNonSentinel(Object key)
    {
        return key == NULL_KEY ? null : (K) key;
    }

    private static Object toSentinelIfNull(Object key)
    {
        return key == null ? NULL_KEY : key;
    }

    private int maxOccupiedWithData()
    {
        int capacity = this.keys.length;
        // need at least one free slot for open addressing
        return Math.min(capacity - 1, capacity / OCCUPIED_DATA_RATIO);
    }

    private int maxOccupiedWithSentinels()
    {
        return this.keys.length / OCCUPIED_SENTINEL_RATIO;
    }

    private class InternalFloatIterator implements MutableFloatIterator
    {
        private int count;
        private int position;

        public boolean hasNext()
        {
            return this.count != ObjectFloatHashMap.this.size();
        }

        public float next()
        {
            if (!this.hasNext())
            {
                throw new NoSuchElementException();
            }

            Object[] keys = ObjectFloatHashMap.this.keys;
            while (!isNonSentinel(keys[this.position]))
            {
                this.position++;
            }
            float result = ObjectFloatHashMap.this.values[this.position];
            this.count++;
            this.position++;
            return result;
        }

        public void remove()
        {
            if (this.position == 0 || !isNonSentinel(ObjectFloatHashMap.this.keys[this.position - 1]))
            {
                throw new IllegalStateException();
            }
            ObjectFloatHashMap.this.remove(ObjectFloatHashMap.this.keys[this.position - 1]);
            this.count--;
        }
    }

    public Set<K> keySet()
    {
        return new KeySet();
    }

    public MutableFloatCollection values()
    {
        return new ValuesCollection();
    }

    private class KeySet implements Set<K>
    {
        @Override
        public boolean equals(Object obj)
        {
            if (obj instanceof Set)
            {
                Set<?> other = (Set<?>) obj;
                if (other.size() == this.size())
                {
                    return this.containsAll(other);
                }
            }
            return false;
        }

        @Override
        public int hashCode()
        {
            int hashCode = 0;
            Object[] table = ObjectFloatHashMap.this.keys;
            for (int i = 0; i < table.length; i++)
            {
                Object key = table[i];
                if (ObjectFloatHashMap.isNonSentinel(key))
                {
                    K nonSentinelKey = ObjectFloatHashMap.this.toNonSentinel(key);
                    hashCode += nonSentinelKey == null ? 0 : nonSentinelKey.hashCode();
                }
            }
            return hashCode;
        }

        public int size()
        {
            return ObjectFloatHashMap.this.size();
        }

        public boolean isEmpty()
        {
            return ObjectFloatHashMap.this.isEmpty();
        }

        public boolean contains(Object o)
        {
            return ObjectFloatHashMap.this.containsKey(o);
        }

        public Object[] toArray()
        {
            int size = ObjectFloatHashMap.this.size();
            Object[] result = new Object[size];
            this.copyKeys(result);
            return result;
        }

        public <T> T[] toArray(T[] result)
        {
            int size = ObjectFloatHashMap.this.size();
            if (result.length < size)
            {
                result = (T[]) Array.newInstance(result.getClass().getComponentType(), size);
            }
            this.copyKeys(result);
            if (size < result.length)
            {
                result[size] = null;
            }
            return result;
        }

        public boolean add(K key)
        {
            throw new UnsupportedOperationException("Cannot call add() on " + this.getClass().getSimpleName());
        }

        public boolean remove(Object key)
        {
            int oldSize = ObjectFloatHashMap.this.size();
            ObjectFloatHashMap.this.removeKey((K) key);
            return oldSize != ObjectFloatHashMap.this.size();
        }

        public boolean containsAll(Collection<?> collection)
        {
            for (Object aCollection : collection)
            {
                if (!ObjectFloatHashMap.this.containsKey(aCollection))
                {
                    return false;
                }
            }
            return true;
        }

        public boolean addAll(Collection<? extends K> collection)
        {
            throw new UnsupportedOperationException("Cannot call addAll() on " + this.getClass().getSimpleName());
        }

        public boolean retainAll(Collection<?> collection)
        {
            int oldSize = ObjectFloatHashMap.this.size();
            Iterator<K> iterator = this.iterator();
            while (iterator.hasNext())
            {
                K next = iterator.next();
                if (!collection.contains(next))
                {
                    iterator.remove();
                }
            }
            return oldSize != ObjectFloatHashMap.this.size();
        }

        public boolean removeAll(Collection<?> collection)
        {
            int oldSize = ObjectFloatHashMap.this.size();
            for (Object object : collection)
            {
                ObjectFloatHashMap.this.removeKey((K) object);
            }
            return oldSize != ObjectFloatHashMap.this.size();
        }

        public void clear()
        {
            ObjectFloatHashMap.this.clear();
        }

        public Iterator<K> iterator()
        {
            return new KeySetIterator();
        }

        private void copyKeys(Object[] result)
        {
            int count = 0;
            for (int i = 0; i < ObjectFloatHashMap.this.keys.length; i++)
            {
                Object key = ObjectFloatHashMap.this.keys[i];
                if (ObjectFloatHashMap.isNonSentinel(key))
                {
                    result[count++] = ObjectFloatHashMap.this.keys[i];
                }
            }
        }
    }

    private class KeySetIterator implements Iterator<K>
    {
        private int count;
        private int position;
        private K currentKey;
        private boolean isCurrentKeySet;

        public boolean hasNext()
        {
            return this.count < ObjectFloatHashMap.this.size();
        }

        public K next()
        {
            if (!this.hasNext())
            {
                throw new NoSuchElementException();
            }
            this.count++;
            Object[] keys = ObjectFloatHashMap.this.keys;
            while (!isNonSentinel(keys[this.position]))
            {
                this.position++;
            }
            this.currentKey = (K) ObjectFloatHashMap.this.keys[this.position];
            this.isCurrentKeySet = true;
            this.position++;
            return ObjectFloatHashMap.this.toNonSentinel(this.currentKey);
        }

        public void remove()
        {
            if (!this.isCurrentKeySet)
            {
                throw new IllegalStateException();
            }

            this.isCurrentKeySet = false;
            this.count--;

            if (isNonSentinel(this.currentKey))
            {
                int index = this.position - 1;
                ObjectFloatHashMap.this.removeKeyAtIndex(ObjectFloatHashMap.this.toNonSentinel(this.currentKey), index);
            }
            else
            {
                ObjectFloatHashMap.this.removeKey(this.currentKey);
            }
        }
    }

    private class ValuesCollection implements MutableFloatCollection
    {
        public int size()
        {
            return ObjectFloatHashMap.this.size();
        }

        public boolean isEmpty()
        {
            return ObjectFloatHashMap.this.isEmpty();
        }

        public boolean notEmpty()
        {
            return ObjectFloatHashMap.this.notEmpty();
        }

        public String makeString()
        {
            return this.makeString(", ");
        }

        public String makeString(String separator)
        {
            return this.makeString("", separator, "");
        }

        public String makeString(String start, String separator, String end)
        {
            Appendable stringBuilder = new StringBuilder();
            this.appendString(stringBuilder, start, separator, end);
            return stringBuilder.toString();
        }

        public void appendString(Appendable appendable)
        {
            this.appendString(appendable, ", ");
        }

        public void appendString(Appendable appendable, String separator)
        {
            this.appendString(appendable, "", separator, "");
        }

        public void appendString(Appendable appendable, String start, String separator, String end)
        {
            try
            {
                appendable.append(start);

                boolean first = true;

                for (int i = 0; i < ObjectFloatHashMap.this.keys.length; i++)
                {
                    Object key = ObjectFloatHashMap.this.keys[i];
                    if (isNonSentinel(key))
                    {
                        if (!first)
                        {
                            appendable.append(separator);
                        }
                        appendable.append(String.valueOf(ObjectFloatHashMap.this.values[i]));
                        first = false;
                    }
                }
                appendable.append(end);
            }
            catch (IOException e)
            {
                throw new RuntimeException(e);
            }
        }

        public boolean add(float element)
        {
            throw new UnsupportedOperationException("Cannot call add() on " + this.getClass().getSimpleName());
        }

        public boolean addAll(float... source)
        {
            throw new UnsupportedOperationException("Cannot call addAll() on " + this.getClass().getSimpleName());
        }

        public boolean addAll(FloatIterable source)
        {
            throw new UnsupportedOperationException("Cannot call addAll() on " + this.getClass().getSimpleName());
        }

        public boolean remove(float item)
        {
            int oldSize = ObjectFloatHashMap.this.size();

            for (int i = 0; i < ObjectFloatHashMap.this.keys.length; i++)
            {
                if (isNonSentinel(ObjectFloatHashMap.this.keys[i]) && Float.compare(item, ObjectFloatHashMap.this.values[i]) == 0)
                {
                    ObjectFloatHashMap.this.removeKey((K) ObjectFloatHashMap.this.keys[i]);
                }
            }
            return oldSize != ObjectFloatHashMap.this.size();
        }

        public boolean removeAll(FloatIterable source)
        {
            int oldSize = ObjectFloatHashMap.this.size();

            FloatIterator iterator = source.floatIterator();
            while (iterator.hasNext())
            {
                this.remove(iterator.next());
            }
            return oldSize != ObjectFloatHashMap.this.size();
        }

        public boolean removeAll(float... source)
        {
            int oldSize = ObjectFloatHashMap.this.size();

            for (float item : source)
            {
                this.remove(item);
            }
            return oldSize != ObjectFloatHashMap.this.size();
        }

        public boolean retainAll(FloatIterable source)
        {
            int oldSize = ObjectFloatHashMap.this.size();
            final FloatSet sourceSet = source instanceof FloatSet ? (FloatSet) source : source.toSet();
            ObjectFloatHashMap<K> retained = ObjectFloatHashMap.this.select(new ObjectFloatPredicate<K>()
            {
                public boolean accept(K object, float value)
                {
                    return sourceSet.contains(value);
                }
            });
            if (retained.size() != oldSize)
            {
                ObjectFloatHashMap.this.keys = retained.keys;
                ObjectFloatHashMap.this.values = retained.values;
                ObjectFloatHashMap.this.occupiedWithData = retained.occupiedWithData;
                ObjectFloatHashMap.this.occupiedWithSentinels = retained.occupiedWithSentinels;
                return true;
            }
            return false;
        }

        public boolean retainAll(float... source)
        {
            return this.retainAll(FloatHashSet.newSetWith(source));
        }

        public void clear()
        {
            ObjectFloatHashMap.this.clear();
        }

        public MutableFloatCollection with(float element)
        {
            throw new UnsupportedOperationException("Cannot call with() on " + this.getClass().getSimpleName());
        }

        public MutableFloatCollection without(float element)
        {
            throw new UnsupportedOperationException("Cannot call without() on " + this.getClass().getSimpleName());
        }

        public MutableFloatCollection withAll(FloatIterable elements)
        {
            throw new UnsupportedOperationException("Cannot call withAll() on " + this.getClass().getSimpleName());
        }

        public MutableFloatCollection withoutAll(FloatIterable elements)
        {
            throw new UnsupportedOperationException("Cannot call withoutAll() on " + this.getClass().getSimpleName());
        }

        public MutableFloatCollection asUnmodifiable()
        {
            return UnmodifiableFloatCollection.of(this);
        }

        public MutableFloatCollection asSynchronized()
        {
            return SynchronizedFloatCollection.of(this);
        }

        public ImmutableFloatCollection toImmutable()
        {
            return FloatLists.immutable.withAll(this);
        }

        public MutableFloatIterator floatIterator()
        {
            return ObjectFloatHashMap.this.floatIterator();
        }

        public float[] toArray()
        {
            return ObjectFloatHashMap.this.toArray();
        }

        public boolean contains(float value)
        {
            return ObjectFloatHashMap.this.containsValue(value);
        }

        public boolean containsAll(float... source)
        {
            return ObjectFloatHashMap.this.containsAll(source);
        }

        public boolean containsAll(FloatIterable source)
        {
            return ObjectFloatHashMap.this.containsAll(source);
        }

        public void forEach(FloatProcedure procedure)
        {
            this.each(procedure);
        }

        public void each(FloatProcedure procedure)
        {
            ObjectFloatHashMap.this.forEach(procedure);
        }

        public MutableFloatCollection select(FloatPredicate predicate)
        {
            return ObjectFloatHashMap.this.select(predicate);
        }

        public MutableFloatCollection reject(FloatPredicate predicate)
        {
            return ObjectFloatHashMap.this.reject(predicate);
        }

        public <V> MutableCollection<V> collect(FloatToObjectFunction<? extends V> function)
        {
            return ObjectFloatHashMap.this.collect(function);
        }

        public <T> T injectInto(T injectedValue, ObjectFloatToObjectFunction<? super T, ? extends T> function)
        {
            return ObjectFloatHashMap.this.injectInto(injectedValue, function);
        }

        public float detectIfNone(FloatPredicate predicate, float ifNone)
        {
            return ObjectFloatHashMap.this.detectIfNone(predicate, ifNone);
        }

        public int count(FloatPredicate predicate)
        {
            return ObjectFloatHashMap.this.count(predicate);
        }

        public boolean anySatisfy(FloatPredicate predicate)
        {
            return ObjectFloatHashMap.this.anySatisfy(predicate);
        }

        public boolean allSatisfy(FloatPredicate predicate)
        {
            return ObjectFloatHashMap.this.allSatisfy(predicate);
        }

        public boolean noneSatisfy(FloatPredicate predicate)
        {
            return ObjectFloatHashMap.this.noneSatisfy(predicate);
        }

        public MutableFloatList toList()
        {
            return ObjectFloatHashMap.this.toList();
        }

        public MutableFloatSet toSet()
        {
            return ObjectFloatHashMap.this.toSet();
        }

        public MutableFloatBag toBag()
        {
            return ObjectFloatHashMap.this.toBag();
        }

        public LazyFloatIterable asLazy()
        {
            return new LazyFloatIterableAdapter(this);
        }

        public float[] toSortedArray()
        {
            return ObjectFloatHashMap.this.toSortedArray();
        }

        public MutableFloatList toSortedList()
        {
            return ObjectFloatHashMap.this.toSortedList();
        }

        public double sum()
        {
            return ObjectFloatHashMap.this.sum();
        }

        public float max()
        {
            return ObjectFloatHashMap.this.max();
        }

        public float maxIfEmpty(float defaultValue)
        {
            return ObjectFloatHashMap.this.maxIfEmpty(defaultValue);
        }

        public float min()
        {
            return ObjectFloatHashMap.this.min();
        }

        public float minIfEmpty(float defaultValue)
        {
            return ObjectFloatHashMap.this.minIfEmpty(defaultValue);
        }

        public double average()
        {
            return ObjectFloatHashMap.this.average();
        }

        public double median()
        {
            return ObjectFloatHashMap.this.median();
        }
    }

    private class KeysView extends AbstractLazyIterable<K>
    {
        public void each(Procedure<? super K> procedure)
        {
            ObjectFloatHashMap.this.forEachKey(procedure);
        }

        public void forEachWithIndex(ObjectIntProcedure<? super K> objectFloatProcedure)
        {
            int index = 0;
            for (int i = 0; i < ObjectFloatHashMap.this.keys.length; i++)
            {
                if (ObjectFloatHashMap.isNonSentinel(ObjectFloatHashMap.this.keys[i]))
                {
                    objectFloatProcedure.value(ObjectFloatHashMap.this.toNonSentinel(ObjectFloatHashMap.this.keys[i]), index);
                    index++;
                }
            }
        }

        public <P> void forEachWith(Procedure2<? super K, ? super P> procedure, P parameter)
        {
            for (int i = 0; i < ObjectFloatHashMap.this.keys.length; i++)
            {
                if (ObjectFloatHashMap.isNonSentinel(ObjectFloatHashMap.this.keys[i]))
                {
                    procedure.value(ObjectFloatHashMap.this.toNonSentinel(ObjectFloatHashMap.this.keys[i]), parameter);
                }
            }
        }

        public Iterator<K> iterator()
        {
            return new InternalKeysViewIterator();
        }

        public class InternalKeysViewIterator implements Iterator<K>
        {
            private int count;
            private int position;

            public K next()
            {
                if (!this.hasNext())
                {
                    throw new NoSuchElementException();
                }

                Object[] keys = ObjectFloatHashMap.this.keys;
                while (!isNonSentinel(keys[this.position]))
                {
                    this.position++;
                }
                K result = ObjectFloatHashMap.this.toNonSentinel(ObjectFloatHashMap.this.keys[this.position]);
                this.count++;
                this.position++;
                return result;
            }

            public void remove()
            {
                throw new UnsupportedOperationException("Cannot call remove() on " + this.getClass().getSimpleName());
            }

            public boolean hasNext()
            {
                return this.count != ObjectFloatHashMap.this.size();
            }
        }
    }

    private class KeyValuesView extends AbstractLazyIterable<ObjectFloatPair<K>>
    {
        public void each(Procedure<? super ObjectFloatPair<K>> procedure)
        {
            for (int i = 0; i < ObjectFloatHashMap.this.keys.length; i++)
            {
                if (ObjectFloatHashMap.isNonSentinel(ObjectFloatHashMap.this.keys[i]))
                {
                    procedure.value(PrimitiveTuples.pair(ObjectFloatHashMap.this.toNonSentinel(ObjectFloatHashMap.this.keys[i]), ObjectFloatHashMap.this.values[i]));
                }
            }
        }

        public void forEachWithIndex(ObjectIntProcedure<? super ObjectFloatPair<K>> objectIntProcedure)
        {
            int index = 0;
            for (int i = 0; i < ObjectFloatHashMap.this.keys.length; i++)
            {
                if (ObjectFloatHashMap.isNonSentinel(ObjectFloatHashMap.this.keys[i]))
                {
                    objectIntProcedure.value(PrimitiveTuples.pair(ObjectFloatHashMap.this.toNonSentinel(ObjectFloatHashMap.this.keys[i]), ObjectFloatHashMap.this.values[i]), index);
                    index++;
                }
            }
        }

        public <P> void forEachWith(Procedure2<? super ObjectFloatPair<K>, ? super P> procedure, P parameter)
        {
            for (int i = 0; i < ObjectFloatHashMap.this.keys.length; i++)
            {
                if (ObjectFloatHashMap.isNonSentinel(ObjectFloatHashMap.this.keys[i]))
                {
                    procedure.value(PrimitiveTuples.pair(ObjectFloatHashMap.this.toNonSentinel(ObjectFloatHashMap.this.keys[i]), ObjectFloatHashMap.this.values[i]), parameter);
                }
            }
        }

        public Iterator<ObjectFloatPair<K>> iterator()
        {
            return new InternalKeyValuesIterator();
        }

        public class InternalKeyValuesIterator implements Iterator<ObjectFloatPair<K>>
        {
            private int count;
            private int position;

            public ObjectFloatPair<K> next()
            {
                if (!this.hasNext())
                {
                    throw new NoSuchElementException();
                }

                Object[] keys = ObjectFloatHashMap.this.keys;
                while (!isNonSentinel(keys[this.position]))
                {
                    this.position++;
                }
                ObjectFloatPair<K> result = PrimitiveTuples.pair(ObjectFloatHashMap.this.toNonSentinel(ObjectFloatHashMap.this.keys[this.position]), ObjectFloatHashMap.this.values[this.position]);
                this.count++;
                this.position++;
                return result;
            }

            public void remove()
            {
                throw new UnsupportedOperationException("Cannot call remove() on " + this.getClass().getSimpleName());
            }

            public boolean hasNext()
            {
                return this.count != ObjectFloatHashMap.this.size();
            }
        }
    }
}
