/*
    Copyright 2007-2014 Luigi Auriemma

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA

    http://www.gnu.org/licenses/gpl-2.0.txt
*/

#define _WIN32_WINNT    0x0601
#define _WIN32_WINDOWS  0x0601
#define WINVER          0x0601

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <stdint.h>
#include <stdarg.h>
#include <time.h>
#include <ctype.h>

#ifdef WIN32
    #include <direct.h>
    #include <ws2tcpip.h>
    #include <winsock.h>
    #include "winerr.h"
    
    #include <windows.h>
    #include <tchar.h>

    #define close       closesocket
    #define sleep       Sleep
    #define in_addr_t   uint32_t
    #define ONESEC      1000
    #define sleepms     Sleep
#else
    #include <unistd.h>
    #include <sys/socket.h>
    #include <sys/types.h>
    #include <arpa/inet.h>
    #include <netdb.h>
    #include <sys/ioctl.h>
    #include <net/if.h>
    #include <pthread.h>

    #define ONESEC      1
    #define strnicmp    strncasecmp
    #define stricmp     strcasecmp
    #define sleepms(X)  usleep(X * 1000)
#endif

#ifdef WIN32
    #define quick_thread(NAME, ARG) DWORD WINAPI NAME(ARG)
    #define thread_id   DWORD
#else
    #define quick_thread(NAME, ARG) void *NAME(ARG)
    #define thread_id   pthread_t
#endif

thread_id quick_threadx(void *func, void *data) {
    thread_id       tid;
#ifdef WIN32
    if(!CreateThread(NULL, 0, func, data, 0, &tid)) return(0);
#else
    pthread_attr_t  attr;
    pthread_attr_init(&attr);
    pthread_attr_setdetachstate(&attr, PTHREAD_CREATE_DETACHED);
    if(pthread_create(&tid, &attr, func, data)) return(0);
#endif
    return(tid);
}

typedef uint8_t     u8;
typedef uint16_t    u16;
typedef uint32_t    u32;



#define VER             "0.2.3"
#define BUFFSZ          0xffff                      // for speed and for any UDP packet
#define PORT            8123
#define MAXDNS          1024
#define MAXWAIT         120                         // max timeout for any connection
#define HOSTNAME_LENGTH 255

#define KEEPALIVE                                   // HTTP connections with the remote hosts
                                                    // will be kept active till client or
                                                    // server disconnection or timeout
                                                    // this option is very useful with websites
                                                    // full of images

#define SEND(A,B,C)     if(send(A, B, C, 0) < 0) goto quit
#define RECV(A,B,C)     recv(A, B, C, 0)

#ifndef SO_EXCLUSIVEADDRUSE
    #define SO_EXCLUSIVEADDRUSE ((u_int)(~SO_REUSEADDR))
#endif
#ifndef TCP_NODELAY
    #define TCP_NODELAY 0x0001
#endif

static const u8 HTTPOK[]        = "HTTP/1.0 200 OK\r\n\r\n";
static const u8 HTTPERROR[]     = "HTTP/1.0 500 ERROR\r\n\r\n";
static const u8 SOCKS4ERROR[]   = "\x04" "\x5b" "\x00\x00" "\x00\x00\x00\x00";
static const u8 SOCKS5ERROR[]   = "\x05" "\x01" "\x00" "\x01" "\x00\x00\x00\x00" "\x00\x00";



typedef struct {
    int     sock;
} thread_arg_t;



void mysetsockopt(int sd);
int bind_tcp_socket(struct sockaddr_in *peer, in_addr_t iface);
int bind_udp_socket(struct sockaddr_in *peer, in_addr_t iface);
int handle_http(int sock, int sd, u8 *buff, struct sockaddr_in *peer, int *tunnel);
int handle_socks4(int sock, int sd, u8 *buff, struct sockaddr_in *peer);
int handle_socks5(int sock, int sd, u8 *buff, struct sockaddr_in *peer);
int handle_socks5_udp(int sock, int socku, u8 *buff, struct sockaddr_in *peerl);
quick_thread(client, thread_arg_t *thread_arg);
int tcp_string_recv(int sd, u8 *buff, int buffsz);
int tcp_recv(int sd, u8 *buff, int len);
int myconnect(int sock, int sd, u8 *host, in_addr_t ip, u16 port, struct sockaddr_in *peer);
int parse_http(int sock, u8 *buff, int buffsz, u8 **ret_req, u8 **ret_host, u16 *ret_port, u8 **ret_uri);
u8 *find_string(u8 *buff, int buffsz, u8 *str);
int timeout(int sock);
in_addr_t resolv(char *host);
in_addr_t dnsdb(char *host);
void strtolower(u8 *str);
struct in_addr *get_ifaces(void);
in_addr_t get_sock_ip_port(int sd, u16 *port);
void sock_printf(int sd, char *fmt, ...);
void open_log(u8 *fname);
void std_err(void);



static char     verbose     = 0;
struct in_addr  lhost,
                Lhost,
                *lifaces    = NULL;
int             do_dnsdb    = 1;
u16             lport       = PORT;
FILE            *fdlog      = NULL;



#ifdef WIN32

    // http://stackoverflow.com/questions/18557325/how-to-create-windows-service-in-c-c
    SERVICE_STATUS        g_ServiceStatus = {0}; 
    SERVICE_STATUS_HANDLE g_StatusHandle = NULL; 
    HANDLE g_ServiceStopEvent = INVALID_HANDLE_VALUE;

    VOID WINAPI ServiceMain (DWORD argc, LPTSTR *argv); //service entry point
    VOID WINAPI ServiceCtrlHandler (DWORD);
    DWORD WINAPI ServiceWorkerThread (LPVOID lpParam); //service thread

    #define SERVICE_NAME _T("proxymini")
    
    int     g_service_init  = 0;
    int     g_argc      = 0;
    char    **g_argv    = NULL;


const char *inet_ntop(int af, const void *src, char *dst, socklen_t cnt) 
{ 
    if (af == AF_INET) 
    { 
        struct sockaddr_in in; 
        memset(&in, 0, sizeof(in)); 
        in.sin_family = AF_INET; 
        memcpy(&in.sin_addr, src, sizeof(struct in_addr)); 
        getnameinfo((struct sockaddr *)&in, sizeof(struct sockaddr_in), dst, cnt, NULL, 0, NI_NUMERICHOST); 
        return dst; 
    } 
    else if (af == AF_INET6) 
    { 
        struct sockaddr_in6 in; 
        memset(&in, 0, sizeof(in)); 
        in.sin6_family = AF_INET6; 
        memcpy(&in.sin6_addr, src, sizeof(struct in_addr6)); 
        getnameinfo((struct sockaddr *)&in, sizeof(struct sockaddr_in6), dst, cnt, NULL, 0, NI_NUMERICHOST); 
        return dst; 
    } 
    return NULL; 
} 

int inet_pton(int af, const char *src, void *dst) 
{ 
    struct addrinfo hints, *res, *ressave; 

    memset(&hints, 0, sizeof(struct addrinfo)); 
    hints.ai_family = af; 

    if (getaddrinfo(src, NULL, &hints, &res) != 0) 
    { 
        //dolog(LOG_ERR, "Couldn't resolve host %s\n", src); 
        return -1; 
    } 

    ressave = res; 

    while (res) 
    { 
        memcpy(dst, res->ai_addr, res->ai_addrlen); 
        res = res->ai_next; 
    } 

    freeaddrinfo(ressave); 
    return 0; 
}

int main(int argc, char *argv[]);
DWORD WINAPI Service_main (LPVOID lpParam) {
    main(g_argc, g_argv);
    return ERROR_SUCCESS;
}

VOID WINAPI ServiceCtrlHandler (DWORD CtrlCode)
{
    switch (CtrlCode) 
    {
     case SERVICE_CONTROL_SHUTDOWN :
     case SERVICE_CONTROL_STOP :

        if (g_ServiceStatus.dwCurrentState != SERVICE_RUNNING)
           break;

        /* 
         * Perform tasks necessary to stop the service here 
         */

        g_ServiceStatus.dwControlsAccepted = 0;
        g_ServiceStatus.dwCurrentState = SERVICE_STOP_PENDING;
        g_ServiceStatus.dwWin32ExitCode = 0;
        g_ServiceStatus.dwCheckPoint = 4;

        if (SetServiceStatus (g_StatusHandle, &g_ServiceStatus) == FALSE)
        {
            OutputDebugString(_T(
              SERVICE_NAME ": ServiceCtrlHandler: SetServiceStatus returned error"));
        }

        // This will signal the worker thread to start shutting down
        SetEvent (g_ServiceStopEvent);

        break;

     default:
         break;
    }
} 

VOID WINAPI ServiceMain (DWORD argc, LPTSTR *argv)
{
    // Register our service control handler with the SCM
    g_StatusHandle = RegisterServiceCtrlHandler (SERVICE_NAME, ServiceCtrlHandler);

    if (!g_StatusHandle) 
    {
        goto EXIT;
    }

    // Tell the service controller we are starting
    ZeroMemory (&g_ServiceStatus, sizeof (g_ServiceStatus));
    g_ServiceStatus.dwServiceType = SERVICE_WIN32_OWN_PROCESS;
    g_ServiceStatus.dwControlsAccepted = SERVICE_ACCEPT_SHUTDOWN | SERVICE_ACCEPT_STOP;
    g_ServiceStatus.dwCurrentState = SERVICE_START_PENDING;
    g_ServiceStatus.dwWin32ExitCode = 0;
    g_ServiceStatus.dwServiceSpecificExitCode = 0;
    g_ServiceStatus.dwCheckPoint = 0;

    if (SetServiceStatus (g_StatusHandle , &g_ServiceStatus) == FALSE)
    {
        OutputDebugString(_T(
          SERVICE_NAME ": ServiceMain: SetServiceStatus returned error"));
    }

    /*
     * Perform tasks necessary to start the service here
     */

    // Create a service stop event to wait on later
    g_ServiceStopEvent = CreateEvent (NULL, TRUE, FALSE, NULL);
    if (!g_ServiceStopEvent) 
    {   
        // Error creating event
        // Tell service controller we are stopped and exit
        g_ServiceStatus.dwControlsAccepted = 0;
        g_ServiceStatus.dwCurrentState = SERVICE_STOPPED;
        g_ServiceStatus.dwWin32ExitCode = GetLastError();
        g_ServiceStatus.dwCheckPoint = 1;

        if (SetServiceStatus (g_StatusHandle, &g_ServiceStatus) == FALSE)
        {
            OutputDebugString(_T(
              SERVICE_NAME ": ServiceMain: SetServiceStatus returned error"));
        }
        goto EXIT; 
    }    

    
    // Tell the service controller we are started
    g_ServiceStatus.dwControlsAccepted = SERVICE_ACCEPT_STOP;
    g_ServiceStatus.dwCurrentState = SERVICE_RUNNING;
    g_ServiceStatus.dwWin32ExitCode = 0;
    g_ServiceStatus.dwCheckPoint = 0;

    if (SetServiceStatus (g_StatusHandle, &g_ServiceStatus) == FALSE)
    {
        OutputDebugString(_T(
          SERVICE_NAME ": ServiceMain: SetServiceStatus returned error"));
    }
    
    
    
    // my code (this is the woker thread)
    HANDLE hThread_main = CreateThread (NULL, 0, Service_main, NULL, 0, NULL);

    
    
    // Wait until our worker thread exits signaling that the service needs to stop
    WaitForSingleObject (g_ServiceStopEvent, INFINITE);
    
    
    /*
     * Perform any cleanup tasks 
     */
    TerminateThread(hThread_main, 0);
     

    CloseHandle (g_ServiceStopEvent);

    // Tell the service controller we are stopped
    g_ServiceStatus.dwControlsAccepted = 0;
    g_ServiceStatus.dwCurrentState = SERVICE_STOPPED;
    g_ServiceStatus.dwWin32ExitCode = 0;
    g_ServiceStatus.dwCheckPoint = 3;

    if (SetServiceStatus (g_StatusHandle, &g_ServiceStatus) == FALSE)
    {
        OutputDebugString(_T(
          SERVICE_NAME ": ServiceMain: SetServiceStatus returned error"));
    }

EXIT:
    return;
}

#endif



int main(int argc, char *argv[]) {
    struct  sockaddr_in peer        = {0},
                        peer_bind   = {0};
    int     sdl,
            sda,
            i,
            psz;
    char    tmp[64];

#ifdef WIN32
    if(!g_service_init) {
        g_service_init = 1;
        g_argc = argc;
        g_argv = argv;

        SERVICE_TABLE_ENTRY ServiceTable[] = 
        {
            {SERVICE_NAME, (LPSERVICE_MAIN_FUNCTION) ServiceMain},
            {NULL, NULL}
        };
        
        if (StartServiceCtrlDispatcher (ServiceTable) == FALSE)
        {
            g_service_init = -1;    //return GetLastError ();
        }
        else
        {
            // terminate because the service has been stopped (this is the original instance)
            exit(0);
        }
    }
#endif

#ifdef WIN32
    WSADATA    wsadata;
    WSAStartup(MAKEWORD(1,0), &wsadata);
#endif

    setbuf(stdout, NULL);
    setbuf(stderr, NULL);

    fputs("\n"
        "Proxymini "VER"\n"
        "by Luigi Auriemma\n"
        "e-mail: aluigi@autistici.org\n"
        "web:    aluigi.org\n"
        "\n", stdout);

    lhost.s_addr = INADDR_ANY;
    Lhost.s_addr = INADDR_ANY;
    lifaces = get_ifaces();

    printf(
        "Options:\n"
        "-l IP    local interface to bind (default any), list of local IP available:\n"
        "         ");
    for(i = 0; lifaces[i].s_addr != INADDR_NONE; i++) {
        inet_ntop(AF_INET, &lifaces[i], tmp, sizeof(tmp));
        printf("%s ", tmp);
    }
    printf("\n"
        "-L IP    as above but works only for the outgoing socket, this means you can\n"
        "         decide to use a secondary interface for connecting to the hosts (for\n"
        "         example using a Wireless connection instead of your main one)\n"
        "-p PORT  local port to bind (%hu)\n"
        "-v       verbose logging on standard output\n"
        "-o FILE  log everything in the file FILE, it's just like -v but for file\n"
        "-D       disable the internal DNS cache (debug)\n"
        "\n", lport);

    for(i = 1; i < argc; i++) {
        if(((argv[i][0] != '-') && (argv[i][0] != '/')) || (strlen(argv[i]) != 2)) {
            printf("\nError: wrong argument (%s)\n", argv[i]);
            exit(1);
        }
        switch(argv[i][1]) {
            case '-':
            case '?':
            case 'h':
                exit(1);
                break;
            case 'l':
                if(!argv[++i]) exit(1);
                lhost.s_addr = resolv(argv[i]);
                if(lhost.s_addr == INADDR_NONE) std_err();
                break;
            case 'L':
                if(!argv[++i]) exit(1);
                Lhost.s_addr = resolv(argv[i]);
                if(Lhost.s_addr == INADDR_NONE) std_err();
                break;
            case 'p':
                if(!argv[++i]) exit(1);
                lport       = atoi(argv[i]);
                break;
            case 'v':
                verbose     = 1;
                break;
            case 'o':
                if(!argv[++i]) exit(1);
                open_log(argv[i]);
                break;
            case 'D':
                do_dnsdb = 0;
                break;
            default: {
                printf("\nError: wrong argument (%s)\n", argv[i]);
                exit(1);
            }
        }
    }

    peer_bind.sin_port = htons(lport);
    sdl = bind_tcp_socket(&peer_bind, lhost.s_addr);
    if(sdl < 0) std_err();
    printf("- ready for HTTP, HTTP CONNECT, SOCKS4 and SOCKS5 (both TCP and UDP)\n");

    dnsdb(NULL);

    for(;;) {
        psz = sizeof(struct sockaddr_in);
        sda = accept(sdl, (struct sockaddr *)&peer, &psz);
        if(sda < 0) {
            sock_printf(0, "- accept() failed, continue within one second\n");
            close(sdl);
            sleep(ONESEC);
            sdl = bind_tcp_socket(&peer_bind, lhost.s_addr);
            if(sdl < 0) std_err();
            continue;
        }

        sock_printf(sda, "connected\n");

        // struct+malloc necessary to avoid 64bit incompatibility and
        // overwriting of thread_arg with 2 consecutive connections (rare but may happen)
        // do NOT free here, free in quick_thread!
        thread_arg_t    *thread_arg;
        thread_arg = calloc(1, sizeof(thread_arg_t));
        thread_arg->sock = sda;
        if(!quick_threadx(client, thread_arg)) {
            sock_printf(sda, "unable to create thread\n");
            close(sda);
        }
    }

    close(sdl);
    if(fdlog) fclose(fdlog);
    return(0);
}



void mysetsockopt(int sd) {
    static struct linger ling = {1,1};
    int     on      = 1;
    int     size    = BUFFSZ;
    
    setsockopt(sd, SOL_SOCKET, SO_LINGER,    (char *)&ling, sizeof(ling));
    setsockopt(sd, SOL_SOCKET, SO_BROADCAST, (char *)&on,   sizeof(on));
    setsockopt(sd, SOL_SOCKET, SO_SNDBUF,    (char *)&size, sizeof(size));  // useless
    setsockopt(sd, IPPROTO_TCP, TCP_NODELAY, (char *)&on,   sizeof(on));
}



int bind_tcp_socket(struct sockaddr_in *peer, in_addr_t iface) {
    int     sd,
            psz,
            on  = 1;
    char    tmp[64];

    peer->sin_family      = AF_INET;
    peer->sin_addr.s_addr = iface;
    if(peer->sin_port) {
        inet_ntop(peer->sin_family, &peer->sin_addr, tmp, sizeof(tmp));
        printf("- bind %s:%hu\n", tmp, ntohs(peer->sin_port));
    } else {
        peer->sin_port    = 0;
    }

    sd = socket(AF_INET, SOCK_STREAM, IPPROTO_TCP);
    if(sd < 0) goto quit;
    mysetsockopt(sd);

    if(peer->sin_port) {
        if(setsockopt(sd, SOL_SOCKET, SO_REUSEADDR, (char *)&on, sizeof(on))
          < 0) goto quit;
    }

    if(bind(sd, (struct sockaddr *)peer, sizeof(struct sockaddr_in))
      < 0) goto quit;

    psz = sizeof(struct sockaddr_in);
    if(getsockname(sd, (struct sockaddr *)peer, &psz)
      < 0) goto quit;

    if(listen(sd, SOMAXCONN)
      < 0) goto quit;

    return(sd);
quit:
    if(sd > 0) close(sd);
    sock_printf(0, "bind_tcp_socket() failed\n");
    return(-1);
}



int bind_udp_socket(struct sockaddr_in *peer, in_addr_t iface) {
    int     sd,
            psz;

    peer->sin_addr.s_addr = iface;
    peer->sin_port        = 0;
    peer->sin_family      = AF_INET;

    sd = socket(AF_INET, SOCK_DGRAM, IPPROTO_UDP);
    if(sd < 0) goto quit;
    mysetsockopt(sd);

    if(bind(sd, (struct sockaddr *)peer, sizeof(struct sockaddr_in))
      < 0) goto quit;

    psz = sizeof(struct sockaddr_in);
    if(getsockname(sd, (struct sockaddr *)peer, &psz)
      < 0) goto quit;

    return(sd);
quit:
    if(sd > 0) close(sd);
    sock_printf(0, "bind_udp_socket() failed\n");
    return(-1);
}



/*
some info about variable names:
sock = the socket of the client which is connected to us
sd   = any outgoing socket
sdl  = listening socket, it's temporary
*/



int handle_socks4(int sock, int sd, u8 *buff, struct sockaddr_in *peer) {
    struct sockaddr_in peerl    = {0};
    struct in_addr      ip;
    int     sdl         = -1,
            psz;
    u16     port;
    u8      cmd;
    char    tmp[64+1],
            *host       = NULL;

    if(tcp_recv(sock, buff, 7) < 0) goto quit;      // read CMD, PORT, IP

    cmd  = buff[0];
    port = (buff[1] << 8) | buff[2];
    ip.s_addr = *(in_addr_t *)(buff + 3);

    if(tcp_string_recv(sock, buff, 256)             // read USERID
      < 0) goto quit;

    if(!(ntohl(ip.s_addr) >> 8)) {                  // SOCKS 4a
        if(tcp_string_recv(sock, buff, 256)         // read hostname
          < 0) goto quit;
        host = buff;
    }

    if(cmd == 0x01) {                               // tcp connection
        inet_ntop(AF_INET, &ip, tmp, sizeof(tmp));
        sock_printf(sock, "SOCKS4 TCP %s:%hu\n",
            host ? host : tmp,
            port);

        sd = myconnect(sock, sd, host, ip.s_addr, port, peer);
        if(sd < 0) goto quit;
        ip.s_addr = get_sock_ip_port(sd, &port);           // useless
        if(ip.s_addr == INADDR_NONE) goto quit;

    } else if(cmd == 0x02) {                        // tcp bind
        sock_printf(sock, "SOCKS4 TCP binding\n");

        peerl.sin_port = 0;
        sdl = bind_tcp_socket(&peerl, lhost.s_addr);
        if(sdl < 0) goto quit;
        mysetsockopt(sdl);
        ip.s_addr = get_sock_ip_port(sock, NULL);
        if(ip.s_addr == INADDR_NONE) goto quit;
        port = ntohs(peerl.sin_port);

        sock_printf(sock, "SOCKS4 TCP assigned port %hu\n", port);

    } else {
        sock_printf(sock, "SOCKS4 command 0x%02x not supported\n", cmd);
        goto quit;
    }

    buff[0] = 0;                                    // version, must be 0
    buff[1] = 90;                                   // success
    buff[2] = port >> 8;                            // port
    buff[3] = port;
    *(in_addr_t *)(buff + 4) = ip.s_addr;
    SEND(sock, buff, 8);

    if(cmd == 0x02) {
        if(timeout(sdl) < 0) goto quit;
        psz = sizeof(struct sockaddr_in);
        sd = accept(sdl, (struct sockaddr *)&peerl, &psz);
        if(sd < 0) goto quit;

        sock_printf(sd, "SOCKS4 connected to TCP bound port %hu\n", port);
    }

    return(sd);
quit:
    //sock_printf(0, "handle_socks4() failed\n");
    return(-1);
}



int handle_socks5(int sock, int sd, u8 *buff, struct sockaddr_in *peer) {
    struct sockaddr_in  peerl;
    struct in_addr      ip;
    int     sdl         = 0,
            len         = 0,
            psz         = 0;
    u16     port        = 0;
    u8      cmd         = 0,
            type        = 0;
    char    tmp[64+1]    = "",
            *host       = NULL;

    if(tcp_recv(sock, buff, 1) < 0) goto quit;      // methods
    if(tcp_recv(sock, buff, buff[0]) < 0) goto quit;

    buff[0] = 0x05;
    buff[1] = 0x00;                                 // force no auth!
    SEND(sock, buff, 2);

    if(tcp_recv(sock, buff, 4) < 0) goto quit;

    cmd  = buff[1];
    type = buff[3];

    if(type == 0x01) {
        if(tcp_recv(sock, buff, 4) < 0) goto quit;
        ip.s_addr = *(in_addr_t *)buff;

    } else if(type == 0x03) {
        if(tcp_recv(sock, buff, 1) < 0) goto quit;  // host length
        len = buff[0];                              // host
        if(tcp_recv(sock, buff, len) < 0) goto quit;
        buff[len] = 0;
        host = buff;

    } else if(type == 0x04) {
        sock_printf(sock, "IPv6 not supported\n");
        goto quit;

    } else {
        sock_printf(sock, "SOCKS5 address type 0x%02x not supported\n", type);
        goto quit;
    }

    if(tcp_recv(sock, (void *)&port, 2) < 0) goto quit;
    port = htons(port);

    if(cmd == 0x01) {                               // tcp connect
        inet_ntop(AF_INET, &ip, tmp, sizeof(tmp));
        sock_printf(sock, "SOCKS5 TCP %s:%hu\n",
            host ? host : tmp,
            port);

        sd = myconnect(sock, sd, host, ip.s_addr, port, peer);
        if(sd < 0) goto quit;
        ip.s_addr = get_sock_ip_port(sd, &port);
        if(ip.s_addr == INADDR_NONE) goto quit;

    } else if(cmd == 0x02) {                        // tcp bind
        sock_printf(sock, "SOCKS5 TCP binding\n");

        peerl.sin_port = 0;
        sdl = bind_tcp_socket(&peerl, lhost.s_addr);
        if(sdl < 0) goto quit;
        mysetsockopt(sdl);
        ip.s_addr = get_sock_ip_port(sock, NULL);
        if(ip.s_addr == INADDR_NONE) goto quit;
        port = ntohs(peerl.sin_port);

        sock_printf(sock, "SOCKS5 TCP assigned port %hu\n", port);

    } else if(cmd == 0x03) {                        // udp
        sock_printf(sock, "SOCKS5 UDP binding\n");

        peerl.sin_port = 0;
        sdl = bind_udp_socket(&peerl, lhost.s_addr);
        if(sdl < 0) goto quit;
        mysetsockopt(sdl);
        ip.s_addr = get_sock_ip_port(sock, NULL);
        if(ip.s_addr == INADDR_NONE) goto quit;
        port = ntohs(peerl.sin_port);

        sock_printf(sock, "SOCKS5 UDP assigned port %hu\n", port);

    } else {
        sock_printf(sock, "SOCKS5 command 0x%02x not supported\n", cmd);
        goto quit;
    }

    buff[0] = 0x05;                                 // version
    buff[1] = 0x00;                                 // reply, success
    buff[2] = 0x00;                                 // reserved
    buff[3] = 0x01;                                 // IPv4
    *(in_addr_t *)(buff + 4) = ip.s_addr;           // IP
    buff[8] = port >> 8;                            // port
    buff[9] = port;

    SEND(sock, buff, 10);

    if(cmd == 0x02) {
        psz = sizeof(struct sockaddr_in);
        if(timeout(sdl) < 0) goto quit;
        sd = accept(sdl, (struct sockaddr *)&peerl, &psz);
        if(sd < 0) goto quit;

        sock_printf(sd, "SOCKS5 connected to TCP bound port %hu\n", port);

    } else if(cmd == 0x03) {
        handle_socks5_udp(sock, sdl, buff, &peerl); // this is a totally new function
        return(-1);                                 // that's why I quit here... do not modify!
    }

    return(sd);
quit:
    //sock_printf(0, "handle_socks5() failed\n");
    return(-1);
}



int handle_socks5_udp(int sock, int socku, u8 *buff, struct sockaddr_in *peerl) {
    struct sockaddr_in  peer    = {0};
    struct in_addr  ip;
    struct  timeval tout;
    fd_set  rset;
    int     sd,
            sel,
            len,
            datasz,
            psz;
    u16     port;
    u8      *data,
            *host;

    sd = socket(AF_INET, SOCK_DGRAM, IPPROTO_UDP);
    if(sd < 0) goto quit;
    mysetsockopt(sd);
    peerl->sin_addr.s_addr = Lhost.s_addr;
    peerl->sin_port        = 0;
    peerl->sin_family      = AF_INET;
    if(bind(sd, (struct sockaddr *)peerl, sizeof(struct sockaddr_in))
      < 0) goto quit;

    sel = sock + socku + sd + 1;

    for(;;) {
        FD_ZERO(&rset);
        FD_SET(sock,  &rset);
        FD_SET(socku, &rset);
        FD_SET(sd,    &rset);
        tout.tv_sec  = MAXWAIT;
        tout.tv_usec = 0;
        if(select(sel, &rset, NULL, NULL, &tout)
          <= 0) goto quit;

        if(FD_ISSET(sock, &rset)) {                 // used only to take udp active
            len = RECV(sock, buff, BUFFSZ);
            if(len <= 0) goto quit;
        }

        if(FD_ISSET(socku, &rset)) {                // from udp client
            psz = sizeof(struct sockaddr_in);
            len = recvfrom(socku, buff, BUFFSZ, 0, (struct sockaddr *)peerl, &psz);
            if(len < 7) continue;                   // 7 is the minimum!

                                                    // no, I don't check if the host is valid or not!

            if(buff[0] || buff[1]) continue;        // reserved, must be 0

            if(buff[2]) {                           // fragments not supported
                continue;
            }

            if(buff[3] == 0x01) {
                ip.s_addr = *(in_addr_t *)(buff + 4);
                port   = (buff[8] << 8) | buff[9];
                data   = buff + 10;
                datasz = len  - 10;
                if(datasz < 0) continue;

            } else if(buff[3] == 0x03) {
                host   = buff + 5;
                data   = buff + 5 + buff[4] + 2;
                port   = (data[-2] << 8) | data[-1];
                datasz = len  - (data - buff);
                if(datasz < 0) continue;

                host[buff[4]] = 0;
                ip.s_addr = dnsdb(host);

            } else if(buff[3] == 0x04) {
                continue;

            } else {
                continue;
            }

            if(ip.s_addr == INADDR_NONE) continue;
            if(!port) continue;

            peer.sin_addr.s_addr = ip.s_addr;
            peer.sin_port        = htons(port);
            peer.sin_family      = AF_INET;

            sendto(sd, data, datasz, 0, (struct sockaddr *)&peer, sizeof(struct sockaddr_in));
        }

        if(FD_ISSET(sd, &rset)) {                   // from udp server
            psz = sizeof(struct sockaddr_in);
            len = recvfrom(sd, buff + 10, BUFFSZ - 10, 0, (struct sockaddr *)&peer, &psz);
            if(len < 0) continue;

            buff[0] = 0x00;                         // probably useless, all 0x00 is ok too
            buff[1] = 0x00;
            buff[2] = 0x00;
            buff[3] = 0x01;
            *(in_addr_t *)(buff + 4) = peer.sin_addr.s_addr;
            *(u16 *)(buff + 8)  = peer.sin_port;

            sendto(socku, buff, 10 + len, 0, (struct sockaddr *)peerl, sizeof(struct sockaddr_in));
        }
    }

quit:
    if(sd > 0) close(sd);
    close(socku);
    return(0);
}



int handle_http(int sock, int sd, u8 *buff, struct sockaddr_in *peer, int *tunnel) {
    int     t,
            len,
            ulen,
            hlen,
            rlen;
    u16     port;
    u8      *req,
            *host,
            *uri,
            *p;

    len = 1;                                        // get header (one byte already read!)
    do {
        if(timeout(sock) < 0) goto quit;
        t = RECV(sock, buff + len, BUFFSZ - len);
        if(t <= 0) goto quit;
        len += t;

        p = find_string(buff, len, "\r\n\r\n");
    } while(!p);
    hlen = (p + 4) - buff;

    ulen = parse_http(sock, buff, hlen, &req, &host, &port, &uri);
    if(ulen < 0) goto quit;

    sock_printf(sock, "%s %s:%hu/%s\n", req, host, port, uri);

    sd = myconnect(sock, sd, host, INADDR_NONE, port, peer);
    if(sd < 0) goto quit;

    if(!stricmp(req, "CONNECT")) {
        *tunnel = 1;
        SEND(sock, HTTPOK, sizeof(HTTPOK) - 1);
    } else {                                        // send header
        *tunnel = 0;
        SEND(sd, req,           strlen(req));
        SEND(sd, " /",          2);
        SEND(sd, uri,           strlen(uri));
        SEND(sd, " ",           1);
        SEND(sd, buff + ulen,   hlen - ulen);
    }

    rlen = 0;                                       // remaining data
    p = find_string(buff + ulen, hlen - ulen, "\nContent-Length:");
    if(p) rlen = atoi(p + 16);

    len -= hlen;                                    // how much long is the rest we have?
    if(len < 0) goto quit;
    rlen -= len;                                    // how much we need to download yet?
    if(rlen < 0) goto quit;

    SEND(sd, buff + hlen,   len);                   // send the data we still have

    for(len = BUFFSZ; rlen; rlen -= t) {            // recv data (content-length)
        if(len > rlen) len = rlen;
        if(timeout(sock) < 0) goto quit;
        t = RECV(sock, buff, len);
        if(t <= 0) goto quit;
        SEND(sd, buff,          t);                 // send data
    }

    return(sd);
quit:
    return(-1);
}



quick_thread(client, thread_arg_t *thread_arg) {
    struct  sockaddr_in peer    = {0};
    struct  timeval tout;
    fd_set  rset;
    int     sd,
            sel,
            len,
            tunnel,
            sock;
    u8      *buff   = NULL;

    if(!thread_arg) return 0;
    sock = thread_arg->sock;
    free(thread_arg);
    
    sd   = 0;
    buff = NULL;
    memset(&peer, 0, sizeof(struct sockaddr_in));

    buff = calloc(BUFFSZ + 1, 1);
    if(!buff) goto quit;

    for(;;) {
        if(tcp_recv(sock, buff, 1) < 0) goto quit;

        if(buff[0] == 0x04) {
            sd = handle_socks4(sock, sd, buff, &peer);
            if(sd < 0) goto quit;
            tunnel = 1;

        } else if(buff[0] == 0x05) {
            sd = handle_socks5(sock, sd, buff, &peer);
            if(sd < 0) goto quit;
            tunnel = 1;

        } else {
            sd = handle_http(sock, sd, buff, &peer, &tunnel);
            if(sd < 0) goto quit;
        }

        sel = sock + sd + 1;                        // start recv

        for(;;) {
            FD_ZERO(&rset);
            FD_SET(sock, &rset);
            FD_SET(sd,   &rset);
            tout.tv_sec  = MAXWAIT;
            tout.tv_usec = 0;
            if(select(sel, &rset, NULL, NULL, &tout)
              <= 0) goto quit;

            if(FD_ISSET(sock, &rset)) {             // local port
                if(!tunnel) break;                  // we must re-handle the request
                len = RECV(sock, buff, BUFFSZ);
                if(len <= 0) goto quit;
                SEND(sd, buff, len);
            }

            if(FD_ISSET(sd, &rset)) {               // dest port
                len = RECV(sd, buff, BUFFSZ);
                if(len <= 0) {
#ifdef KEEPALIVE
                    if(tunnel) goto quit;           // break if CONNECT
                    close(sd);                      // we want ever keep-alive???
                    sd = 0;
                    break;
#else
                    goto quit;
#endif
                }
                SEND(sock, buff, len);
            }
        }
    }

quit:
    if(buff) free(buff);
    sock_printf(sock, "disconnected\n");
    if(sd > 0) close(sd);
    close(sock);
    return(0);
}



int tcp_string_recv(int sd, u8 *buff, int buffsz) {
    int     i;

    for(i = 0; i < buffsz; i++) {
        if(timeout(sd) < 0) return(-1);
        if(RECV(sd, buff + i, 1)
          <= 0) return(-1);
        if(!buff[i]) return(i);
    }
    return(-1); // no NULL no party
}



int tcp_recv(int sd, u8 *buff, int len) {
    int     t;

    while(len) {
        if(timeout(sd) < 0) return(-1);
        t = RECV(sd, buff, len);
        if(t <= 0) return(-1);
        buff += t;
        len  -= t;
    }
    return(0);
}



int myconnect(int sock, int sd, u8 *host, in_addr_t ip, u16 port, struct sockaddr_in *peer) {
    struct sockaddr_in  peerl;
    int     i;

    if(host) {
        ip = dnsdb(host);
        if(ip == INADDR_NONE) {                     // host is invalid
            sock_printf(sock, "unknown host \"%s\"\n", host);
            return(-1); // do not use goto quit!
        }
    }
    // else use the ip provided

    port = htons(port);
                                                    // IP or port is not the previous one
    if((sd > 0) && ((peer->sin_addr.s_addr != ip) || (peer->sin_port != port))) {
        close(sd);
        sd = 0;
    }

    if(sd <= 0) {
        for(i = 0; lifaces[i].s_addr != INADDR_NONE; i++) {
            if((ip == lifaces[i].s_addr) && (port == htons(lport))) {
                sock_printf(sock, "loopback connections not allowed\n");
                /* while connections to LAN hosts are allowed */
                goto quit;
            }
        }

        peer->sin_addr.s_addr = ip;
        peer->sin_port        = port;
        peer->sin_family      = AF_INET;

        sd = socket(AF_INET, SOCK_STREAM, IPPROTO_TCP);
        if(sd < 0) goto quit;
        mysetsockopt(sd);
        peerl.sin_addr.s_addr = Lhost.s_addr;
        peerl.sin_port        = 0;
        peerl.sin_family      = AF_INET;
        if(bind(sd, (struct sockaddr *)&peerl, sizeof(struct sockaddr_in))
          < 0) goto quit;
        if(connect(sd, (struct sockaddr *)peer, sizeof(struct sockaddr_in))
          < 0) goto quit;
    }
    return(sd);
quit:
    if(sd > 0) close(sd);
    return(-1);
}



int parse_http(int sock, u8 *buff, int buffsz, u8 **ret_req, u8 **ret_host, u16 *ret_port, u8 **ret_uri) {
    u16     port    = 80;
    u8      *req    = NULL,
            *host   = NULL,
            *uri    = NULL,
            *p      = NULL,
            *limit  = NULL;

    req = buff;

    buff[buffsz] = 0;   // buff is allocated as BUFFSZ + 1 so there is space
    
    host = strchr(buff, ' ');
    if(!host) return(-1);
    *host++ = 0;

    limit = strchr(host, ' ');
    if(!limit) return(-1);
    *limit++ = 0;

    p = strstr(host, "://");                        // get host
    if(p) {
        *p = 0;
        if(stricmp(host, "http")) {
            sock_printf(sock, "protocol \"%s\" not supported\n", host);
            return(-1);
        }
        host = p + 3;
    }

    uri = host;

    p = host + strcspn(host, ":/");                 // get port
    if(*p == ':') {
        *p++ = 0;
        port = atoi(p);
        uri = p + strcspn(p, "/");
    } else {
        uri = p;
    }
    if(*uri) *uri++ = 0;

    *ret_req    = req;
    *ret_port   = port;
    *ret_host   = host;
    *ret_uri    = uri;
    return(limit - buff);
}



u8 *find_string(u8 *buff, int buffsz, u8 *str) {
    int     strsz;
    u8      *limit;

    strsz = strlen(str);
    limit = buff + buffsz - strsz;

    for(; buff <= limit; buff++) {
        if(!strnicmp(buff, str, strsz)) return(buff);
    }
    return(NULL);
}



int timeout(int sock) {
    struct  timeval tout;
    fd_set  fd_read;
    int     err;

    tout.tv_sec  = MAXWAIT;
    tout.tv_usec = 0;
    FD_ZERO(&fd_read);
    FD_SET(sock, &fd_read);
    err = select(sock + 1, &fd_read, NULL, NULL, &tout);
    if(err < 0) return(-1); //std_err();
    if(!err) return(-1);
    return(0);
}



in_addr_t resolv(char *host) {
    struct      hostent *hp;
    in_addr_t   host_ip;

    if(!host) return INADDR_NONE;
    host_ip = inet_addr(host);
    if(host_ip == INADDR_NONE) {
        hp = gethostbyname(host);
        if(hp) host_ip = *(in_addr_t *)(hp->h_addr);
    }
    return(host_ip);
}



    /* the following function caches the IP addresses     */
    /* the instructions which replace the oldest entries  */
    /* are not the best in the world... but work  */
    
    /*
    notes:
    - there is no usage of semaphores and events, so this function may give a wrong result when 2 threads use it at the same moment
    - stricmp with 1024 entries is not so bad, so no need to use a hashing function
    - there are peaks of CPU when using the resolv() function (probably OS related, tested Windows)
    - it's possible that the IP of the old hostname to replace will be returned instead of the one of the new hostname
    currently the names resolution is the only weak point of Proxymini, it's not a real issue but it's not good in production.
    using an external library like adns or c-ares would be great but makes the source code too big.
    SOCKS4 uses IP addresses so these issues don't exist.
    */

in_addr_t dnsdb(char *host) {
    typedef struct {
        in_addr_t   ip;
        time_t      time;
        u8          host[HOSTNAME_LENGTH+1];
        int         resolving;
    } db_t;

    static db_t *db     = NULL;
    static int  oldest  = 0;
    in_addr_t   ret     = INADDR_NONE;
    in_addr_t   fastip;
    int         i;

    if(!do_dnsdb) {
        ret = resolv(host);
        goto quit;
    }
    
    if(!host && !db) {
        db = calloc(MAXDNS, sizeof(db_t));  // allocate
        if(!db) std_err();

        for(i = 0; i < MAXDNS; i++) {
            db[i].ip        = INADDR_NONE;
            db[i].host[0]   = 0;
            db[i].time      = time(NULL);
            db[i].resolving = 0;
        }

        oldest = 0;
        goto quit;
    }

    if(!host || !host[0]) {
        goto quit;
    }

    fastip = inet_addr(host);
    if(fastip != INADDR_NONE) {
        ret = fastip;
        goto quit;
    }

    for(i = 0; i < MAXDNS; i++) {
        if(!db[i].host[0]) break;                   // new host to add

        if(!stricmp(db[i].host, host)) {            // host in cache
            while(db[i].resolving) sleepms(1);      // wait that resolv() returns
            db[i].time = time(NULL);                // update time
            ret = db[i].ip;
            goto quit;
        }

        if(db[i].time < db[oldest].time) {          // what's the oldest entry?
            oldest = i;
        }
    }
    
    if(i == MAXDNS) i = oldest;                     // take the oldest one

    strncpy(db[i].host, host, HOSTNAME_LENGTH);
    db[i].host[HOSTNAME_LENGTH] = 0;
    db[i].time = time(NULL);
    db[i].resolving = 1;
    
    db[i].ip = resolv(host);
    if(db[i].ip == INADDR_NONE) {
        db[i].host[0] = 0;
    } else {
        ret = db[i].ip;
    }
    
    db[i].resolving = 0;
    
quit:
    return ret;
}



void strtolower(u8 *str) {
    u8     *p;
    for(p = str; *p; p++) {
        *p = tolower(*p);
    }
}



struct in_addr *get_ifaces(void) {
#ifdef WIN32
    #define ifr_addr        iiAddress.AddressIn
#else
    struct ifconf   ifc;
    #define INTERFACE_INFO  struct ifreq
#endif

    struct sockaddr_in  *sin;
    struct in_addr  *ifaces,
                    lo;
    INTERFACE_INFO  *ifr;
    int         sd,
                i,
                j,
                num;
    u8          buff[sizeof(INTERFACE_INFO) * 16];

    sd = socket(AF_INET, SOCK_STREAM, IPPROTO_TCP);
    if(sd < 0) std_err();
    mysetsockopt(sd);

#ifdef WIN32
    if(WSAIoctl(sd, SIO_GET_INTERFACE_LIST, 0, 0, buff, sizeof(buff), (void *)&num, 0, 0)
      < 0) std_err();
    ifr = (void *)buff;
#else
    ifc.ifc_len = sizeof(buff);
    ifc.ifc_buf = buff;
    if(ioctl(sd, SIOCGIFCONF, (char *)&ifc)
      < 0) std_err();
    num = ifc.ifc_len;
    ifr = ifc.ifc_req;
#endif

    num /= sizeof(INTERFACE_INFO);
    close(sd);

    ifaces = calloc(num + 1 + 1, sizeof(* ifaces)); // num + lo + NULL
    if(!ifaces) std_err();

    lo.s_addr = inet_addr("127.0.0.1");

    for(j = i = 0; i < num; i++) {
        sin = (struct sockaddr_in *)&ifr[i].ifr_addr;

        if(sin->sin_family      != AF_INET)     continue;
        if(sin->sin_addr.s_addr == INADDR_NONE) continue;

        ifaces[j++].s_addr = sin->sin_addr.s_addr;

        if(sin->sin_addr.s_addr == lo.s_addr) lo.s_addr = INADDR_NONE;
    }

    ifaces[j++].s_addr = lo.s_addr;
    ifaces[j].s_addr   = INADDR_NONE;

    return(ifaces);
}



in_addr_t get_sock_ip_port(int sd, u16 *port) {
    struct sockaddr_in  peer;
    int     psz;

    psz = sizeof(struct sockaddr_in);
    if(getsockname(sd, (struct sockaddr *)&peer, &psz)
      < 0) peer.sin_addr.s_addr = INADDR_NONE;

    if(port) *port = ntohs(peer.sin_port);

    return(peer.sin_addr.s_addr);
}



in_addr_t get_peer_ip_port(int sd, u16 *port) {
    struct sockaddr_in  peer    = {0};
    int     psz;

    psz = sizeof(struct sockaddr_in);
    if(getpeername(sd, (struct sockaddr *)&peer, &psz) < 0) {
        peer.sin_addr.s_addr = 0;                   // avoids possible problems
        peer.sin_port        = 0;
    }

    if(port) *port = ntohs(peer.sin_port);

    return(peer.sin_addr.s_addr);
}



void sock_printf(int sd, char *fmt, ...) {
    struct in_addr  ip;
    va_list ap;
    int     t,
            tlen,
            len;
    u16     port;
    u8      buff[1024];                             // unfortunately its needed for a "one line show"... blah
    char    tmp[64];

    if(!verbose && !fdlog) return;

    if(sd <= 0) ip.s_addr = INADDR_NONE;
    else        ip.s_addr = get_peer_ip_port(sd, &port);

    len = sprintf(buff, "  ");
    va_start(ap, fmt);
    if(sd > 0) {
        inet_ntop(AF_INET, &ip, tmp, sizeof(tmp));
        len = sprintf(buff, "%s:%hu ", tmp, port);
    }
    tlen = (sizeof(buff) - 1) - len;
    t = vsnprintf(buff + len, tlen, fmt, ap);
    if((t < 0) || (t > tlen)) {
        t = tlen;
    }
    len += t;
    buff[len] = 0;
    va_end(ap);

    if((len < 0) || (len >= sizeof(buff))) {
        strcpy(buff + sizeof(buff) - 6, "...\n");
    }

    if(verbose) {
        fputs(buff, stdout);
    }

    if(fdlog) {
        fputs(buff, fdlog);
        fflush(fdlog);
    }
}



void open_log(u8 *fname) {
    struct  tm  *tmx;
    time_t  datex;

    if(!fname) return;
    printf("- open/append log file %s\n", fname);
    fdlog = fopen(fname, "ab");
    if(!fdlog) std_err();

    time(&datex);
    tmx = localtime(&datex);
    fprintf(fdlog,
        "----------------------------------\n"
        "Log file appended at:\n"
        "  %02d/%02d/%02d   (dd/mm/yy)\n"
        "  %02d:%02d:%02d\n"
        "----------------------------------\n"
        "\n",
        tmx->tm_mday, tmx->tm_mon + 1, tmx->tm_year % 100,
        tmx->tm_hour, tmx->tm_min, tmx->tm_sec);
    fflush(fdlog);
}



#ifndef WIN32
    void std_err(void) {
        perror("\nError");
        exit(1);
    }
#endif


