/*
 * broadvoice - a library for the BroadVoice 16 and 32 codecs
 *
 * tables.c - Constant data tables for the BroadVoice32 codec.
 *
 * Adapted by Steve Underwood <steveu@coppice.org> from code which is
 * Copyright 2000-2009 Broadcom Corporation
 *
 * All rights reserved.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License version 2.1,
 * as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * $Id: bv32tables.c,v 1.1.1.1 2009/11/19 12:10:48 steveu Exp $
 */

/*! \file */

#if defined(HAVE_CONFIG_H)
#include "config.h"
#endif

#include "typedef.h"
#include "bv32cnst.h"
#include "bv32externs.h"

/* spec */
/* log-gain change limitation */
const Float bv32_lgclimit[] =
{
     0.00000,   0.13477,   2.26563,   2.94336,   4.71875,   0.00000,   0.00000,   0.00000,   0.00000,   0.00000,   0.00000,
     0.00000,   0.64453,   4.90039,   3.38281,   4.58203,   5.69336,   0.00000,   0.00000,   0.00000,   0.00000,   0.00000,
     0.00000,   0.33594,   7.27734,   5.82422,  11.66211,  11.66211,   0.00000,   0.00000,   0.00000,   0.00000,   0.00000,
     6.31250,   5.50977,   4.83984,   6.99023,   8.22852,  11.49805,   1.89844,   0.00000,   0.00000,   0.00000,   0.00000,
     0.00000,   5.04883,   5.09180,   5.91406,   6.92188,   7.38086,   4.13867,   0.00000,   0.00000,   0.00000,   0.00000,
    -0.36523,   6.15625,   8.26953,   5.40430,   5.88477,  11.53906,   5.31836,  -4.97070,   0.00000,   0.00000,   0.00000,
     5.51172,   6.31641,   9.66602,   7.58594,  10.63281,  12.03906,   8.79297,   3.06836,   0.00000,   0.00000,   0.00000,
     3.95703,  10.51172,   8.42969,   7.62891,  11.45703,  11.95898,  10.85352,   2.83008,   1.50000,   0.00000,   0.00000,
     7.37305,   8.93945,   8.57422,   6.85742,   9.67773,  11.54492,  10.98242,  10.43359,   2.53320,   5.05859,   0.00000,
     7.37305,   8.12109,   6.66406,   5.87891,   7.59766,  10.67969,  10.42578,   9.46875,   6.85938,   3.06445,   0.00000,
     4.39844,   5.94336,   5.73047,   5.10742,   5.69531,   8.31641,  10.05273,   8.23047,   7.11328,   3.04102,  -1.27930,
     0.58789,   5.10938,   5.41602,   4.55273,   4.32813,   5.75586,   7.42383,   6.63867,   6.81055,   4.14258,   3.31641,
     0.14453,   5.64844,   5.05859,   4.06836,   3.51758,   4.07617,   4.56055,   4.99219,   5.51953,   4.82227,   5.19141,
     0.00000,   5.54688,   5.15625,   3.37891,   2.90430,   2.74805,   2.82422,   3.37500,   4.02930,   4.49805,   3.42188,
     0.00000,   0.39258,   3.92188,   2.67383,   2.66602,   2.40039,   4.65039,   3.29883,   2.16016,   2.95703,   0.40820,
     0.00000,   0.00000,   1.15039,   2.56641,   3.98438,   3.61133,   4.66797,   0.58398,  -0.26563,   0.09570,   0.00000,
     0.00000,   0.00000,   0.37695,   4.30664,   7.07031,   0.81641,   2.86914,   1.19336,   0.69922,  -1.23242,   0.00000,
     0.00000,   0.00000,   0.07617,   1.46875,   3.49219,   3.16992,  -0.84180,   3.81250,  -0.50781,   0.00000,   0.00000
};

/* spec */
const Float bv32_lspecb1[] =
{
    -0.00384521, -0.00849915, -0.01591492, -0.00360107, -0.00013733,  0.00610352,  0.01640320, -0.00166321,
    -0.00511169, -0.01313782, -0.01698303, -0.00103760, -0.01216125, -0.00427246, -0.00271606,  0.00846863,
    -0.00367737, -0.00166321,  0.00045776, -0.00309753,  0.01814270, -0.00053406,  0.00256348, -0.00833130,
    -0.00312805, -0.00488281,  0.00282288, -0.00173950,  0.00004578, -0.00094604, -0.01976013,  0.00306702,
    -0.00250244, -0.00323486,  0.00154114,  0.00422668, -0.00964355, -0.01895142,  0.01704407,  0.00219727,
    -0.00090027, -0.00347900, -0.00909424, -0.00746155, -0.00656128, -0.02726746, -0.00769043, -0.00224304,
     0.00399780,  0.01086426,  0.00677490,  0.00090027,  0.00244141, -0.00988770,  0.00549316, -0.00628662,
    -0.00151062, -0.00581360, -0.00186157, -0.00430298, -0.01788330, -0.01603699, -0.03099060, -0.00659180,
    -0.00547791, -0.00958252,  0.00094604,  0.01203918,  0.00695801,  0.02105713,  0.00720215,  0.00140381,
    -0.00393677, -0.00848389, -0.01943970, -0.01473999,  0.01364136, -0.00468445, -0.00344849,  0.00566101,
    -0.00331116, -0.00723267,  0.00175476,  0.03128052,  0.00772095, -0.00163269,  0.00566101, -0.00460815,
    -0.00222778, -0.00709534, -0.00581360,  0.01132202, -0.00482178, -0.00050354, -0.01037598, -0.01887512,
    -0.00325012, -0.00445557,  0.00651550,  0.00497437, -0.01744080,  0.01000977,  0.01194763, -0.00160217,
    -0.00054932, -0.00219727, -0.00631714, -0.01139832, -0.01916504, -0.00711060,  0.00106812, -0.01481628,
    -0.00546265,  0.00070190,  0.02934265,  0.01412964,  0.00656128,  0.00003052,  0.01229858,  0.00367737,
    -0.00254822,  0.00099182,  0.02000427, -0.00164795, -0.01643372, -0.00813293, -0.00671387, -0.01013184,
    -0.00204468,  0.00265503, -0.00135803, -0.02322388,  0.00332642,  0.01715088,  0.01350403,  0.00199890,
    -0.00289917, -0.00740051, -0.01710510, -0.02655029, -0.01350403,  0.00151062, -0.00038147,  0.00778198,
    -0.00028992,  0.00064087,  0.00022888, -0.00819397,  0.00061035,  0.02536011, -0.00822449, -0.02096558,
    -0.00028992,  0.00001526, -0.00805664, -0.02310181, -0.00082397, -0.00106812, -0.02081299, -0.01762390,
    -0.00030518,  0.00170898, -0.00651550, -0.01683044,  0.00083923, -0.00955200,  0.02677917,  0.00958252,
     0.00292969,  0.00251770, -0.00447083, -0.01782227, -0.02940369, -0.02981567,  0.00372314, -0.00421143,
     0.01701355,  0.02578735, -0.00593567,  0.00595093,  0.01370239,  0.01223755,  0.00622559, -0.00111389,
     0.00061035, -0.00015259,  0.00686646,  0.00129700, -0.00637817, -0.02079773, -0.05078125, -0.01544189,
    -0.00398254,  0.00350952,  0.01591492, -0.00076294,  0.02429199,  0.02890015,  0.01559448,  0.00701904,
    -0.00177002, -0.00981140, -0.03118896, -0.01042175, -0.00013733,  0.00044250, -0.00659180, -0.01545715,
     0.00256348,  0.01017761,  0.01966858,  0.01533508,  0.01405334,  0.01646423, -0.00257874, -0.01338196,
     0.00088501, -0.00016785, -0.00163269, -0.00199890, -0.00700378, -0.00726318, -0.02569580, -0.03907776,
     0.00035095,  0.00717163,  0.00427246,  0.00279236,  0.02046204,  0.00689697,  0.02848816,  0.01043701,
     0.00041199,  0.00004578, -0.01815796, -0.03132629, -0.00378418, -0.02220154,  0.00140381, -0.00294495,
     0.01571655,  0.02601624,  0.01066589,  0.03164673,  0.03356934,  0.02770996,  0.01812744,  0.00709534,
     0.00881958,  0.02149963,  0.01010132,  0.00360107,  0.00122070, -0.00657654, -0.01893616, -0.02380371,
    -0.00672913, -0.01612854, -0.02481079, -0.00184631,  0.00761414,  0.01754761,  0.00720215,  0.01480103,
    -0.00515747, -0.01365662, -0.01542664, -0.01049805, -0.01742554,  0.02040100, -0.00880432, -0.00152588,
    -0.00303650, -0.00975037, -0.02221680,  0.01498413,  0.02423096,  0.00935364, -0.00544739, -0.00675964,
    -0.00221252, -0.00933838, -0.02006531,  0.00033569,  0.00292969, -0.01268005, -0.02940369, -0.00543213,
    -0.00231934, -0.00257874,  0.00263977, -0.00134277, -0.00151062, -0.00566101,  0.00665283,  0.03112793,
    -0.00123596, -0.00584412, -0.01034546, -0.01982117, -0.02880859, -0.02052307, -0.01663208,  0.00572205,
     0.00738525,  0.02700806,  0.01812744,  0.02203369,  0.00323486, -0.00514221,  0.01075745,  0.00660706,
     0.00349426,  0.00294495, -0.00387573, -0.01075745, -0.02171326, -0.03224182, -0.02403259, -0.02343750,
    -0.00619507, -0.01358032, -0.01676941,  0.01498413,  0.02687073,  0.02645874,  0.01818848,  0.01010132,
    -0.00459290, -0.00839233, -0.02026367, -0.02606201,  0.02151489,  0.02061462, -0.00651550, -0.00538635,
    -0.00405884, -0.00538635,  0.00645447,  0.03422546,  0.03749084,  0.02166748,  0.00497437, -0.00592041,
    -0.00209045, -0.00204468, -0.00219727,  0.00228882,  0.02597046,  0.00415039, -0.02684021, -0.01873779,
    -0.00489807, -0.00955200, -0.00572205,  0.00482178, -0.00778198,  0.01531982,  0.03317261,  0.01727295,
    -0.00341797, -0.00909424, -0.00500488, -0.00860596, -0.04263306, -0.00547791,  0.00357056,  0.00357056,
    -0.00016785,  0.01191711,  0.03486633,  0.03454590,  0.02195740,  0.01472473,  0.03034973,  0.02073669,
    -0.00109863,  0.00473022,  0.01737976,  0.00859070, -0.00253296, -0.03044128, -0.00776672, -0.01104736,
    -0.00527954, -0.00999451, -0.00939941, -0.00805664, -0.00268555,  0.04862976,  0.01870728,  0.00442505,
    -0.00317383, -0.00744629, -0.00877380, -0.02050781, -0.03236389,  0.01905823,  0.01884460,  0.00524902,
     0.00453186,  0.01782227,  0.00762939, -0.00749207,  0.03543091,  0.01852417, -0.00367737, -0.01086426,
     0.00018311, -0.00355530, -0.01539612, -0.02656555, -0.00277710, -0.01931763, -0.03083801,  0.00360107,
    -0.00143433,  0.00292969,  0.01277161,  0.00936890,  0.00128174, -0.00985718,  0.04154968,  0.02775574,
     0.00213623,  0.00561523,  0.00642395, -0.00889587, -0.03330994, -0.05546570,  0.00897217,  0.00265503,
     0.01060486,  0.05717468,  0.03829956,  0.03216553,  0.02561951,  0.02203369,  0.01969910,  0.00923157,
     0.00221252,  0.00817871,  0.01704407, -0.00007629, -0.00616455, -0.04737854, -0.03558350,  0.00561523,
    -0.00749207, -0.00627136,  0.02369690,  0.02711487,  0.03462219,  0.04241943,  0.02859497,  0.01635742,
    -0.02087402, -0.04931641,  0.00619507,  0.00404358,  0.01080322,  0.00926208,  0.00779724,  0.00225830,
    -0.00173950,  0.01293945,  0.04112244,  0.03024292,  0.03976440,  0.03063965,  0.00881958, -0.00358582,
    -0.00424194, -0.00158691,  0.02459717,  0.01078796,  0.00611877,  0.00105286, -0.02471924, -0.02410889,
    -0.00451660, -0.00415039,  0.00253296,  0.01228333,  0.02276611,  0.02371216,  0.05001831,  0.02963257,
    -0.00369263, -0.01776123, -0.03298950, -0.01219177, -0.03230286, -0.02035522, -0.01049805, -0.00700378,
     0.01309204,  0.03527832,  0.04226685,  0.04809570,  0.04991150,  0.04533386,  0.03337097,  0.01974487,
     0.00236511,  0.01925659,  0.04072571,  0.02778625,  0.01647949, -0.01173401, -0.02360535, -0.01696777,
    -0.00433350, -0.01188660, -0.02235413,  0.01066589,  0.01145935, -0.00656128,  0.02409363,  0.01565552,
    -0.00448608, -0.01176453, -0.02374268, -0.01464844, -0.01629639, -0.01852417,  0.01446533,  0.01126099,
    -0.00320435,  0.00030518,  0.00944519,  0.01014709,  0.03031921,  0.00007629, -0.00328064,  0.01599121,
    -0.00141907, -0.00477600, -0.00032043, -0.00436401, -0.00563049, -0.02128601, -0.03314209,  0.02626038,
    -0.00105286, -0.00151062, -0.00180054, -0.00811768, -0.02941895, -0.01837158,  0.03617859,  0.01126099,
    -0.00238037, -0.00828552, -0.00988770,  0.00376892, -0.02708435, -0.03489685, -0.00431824,  0.00047302,
     0.01274109,  0.02935791,  0.00981140, -0.00921631, -0.01629639, -0.00587463,  0.00247192,  0.00064087,
     0.00193787, -0.00151062, -0.00468445, -0.01261902, -0.02470398, -0.03384399, -0.04949951, -0.00338745,
    -0.00361633, -0.00816345,  0.00148010,  0.03401184,  0.01333618,  0.01911926,  0.02272034,  0.01939392,
    -0.00132751, -0.00799561, -0.02526855, -0.03221130,  0.00328064,  0.00810242,  0.00950623,  0.01345825,
    -0.00360107, -0.00260925,  0.03428650,  0.04959106,  0.01815796,  0.00881958, -0.00042725, -0.00680542,
    -0.00175476, -0.00695801,  0.00030518,  0.02726746, -0.00277710, -0.01660156, -0.02694702, -0.01084900,
    -0.00105286,  0.00723267,  0.02352905,  0.00462341, -0.01211548,  0.02276611,  0.02523804,  0.01177979,
     0.00044250, -0.00314331, -0.00833130, -0.02253723, -0.03590393,  0.00534058, -0.01576233, -0.01797485,
     0.00265503,  0.02203369,  0.05549622,  0.02410889,  0.00866699,  0.00965881,  0.00958252, -0.00190735,
     0.00163269,  0.00399780,  0.01559448, -0.00083923, -0.03933716, -0.01277161, -0.02479553, -0.01690674,
    -0.00450134, -0.00598145, -0.01719666, -0.02134705,  0.02500916,  0.02310181,  0.02972412,  0.01644897,
    -0.00181580, -0.00743103, -0.02114868, -0.03652954, -0.03193665, -0.00167847,  0.00451660,  0.00935364,
    -0.00350952, -0.00422668, -0.00115967,  0.00111389,  0.03088379,  0.04490662, -0.00390625, -0.01063538,
     0.00189209, -0.00177002, -0.01432800, -0.02612305, -0.01161194, -0.01190186, -0.04681396, -0.02130127,
    -0.00234985, -0.00527954, -0.01350403, -0.01855469, -0.00726318, -0.00196838,  0.04997253,  0.02980042,
     0.00024414, -0.00541687, -0.01794434, -0.02980042, -0.03829956, -0.04582214,  0.01480103,  0.01237488,
     0.05004883,  0.03166199,  0.02220154,  0.01562500,  0.00930786,  0.00764465,  0.00833130,  0.00251770,
     0.00430298,  0.00444031,  0.00691223, -0.00653076, -0.01719666, -0.04112244, -0.09020996, -0.00898743,
     0.00041199,  0.00929260,  0.00347900,  0.00259399,  0.05375671,  0.03878784,  0.02937317,  0.01449585,
    -0.00491333, -0.02757263, -0.06730652, -0.02465820, -0.00869751, -0.00566101, -0.00590515,  0.00354004,
     0.01916504,  0.03500366,  0.02929688,  0.03329468,  0.02725220,  0.01902771, -0.00694275, -0.01644897,
     0.00370789,  0.00387573,  0.00061035, -0.00419617, -0.01568604, -0.02262878, -0.05206299, -0.04679871,
     0.01432800,  0.03143311,  0.01612854,  0.00932312,  0.01620483,  0.02969360,  0.03417969,  0.02700806,
     0.00350952,  0.00082397, -0.03111267, -0.06707764, -0.02024841, -0.01860046, -0.00958252, -0.00173950,
     0.05522156,  0.04231262,  0.04219055,  0.03793335,  0.03443909,  0.03150940,  0.02209473,  0.01277161,
     0.03974915,  0.03291321,  0.01431274,  0.00024414,  0.00086975, -0.01142883, -0.03588867, -0.01281738,
    -0.00840759, -0.02593994, -0.04820251,  0.00361633,  0.01782227,  0.03044128,  0.02810669,  0.02386475,
    -0.00660706, -0.02162170, -0.03446960, -0.01261902, -0.02426147,  0.01382446,  0.01550293,  0.01689148,
    -0.00529480, -0.00663757, -0.01538086, -0.00068665,  0.05569458,  0.01844788,  0.00303650,  0.00178528,
    -0.00257874, -0.00895691, -0.00971985, -0.00666809,  0.00314331, -0.00125122, -0.05572510,  0.00030518,
    -0.00048828,  0.00041199, -0.00028992, -0.00938416, -0.00831604, -0.03677368,  0.01962280,  0.03395081,
     0.00151062, -0.00248718, -0.01004028, -0.02021790, -0.03338623, -0.05041504, -0.02108765,  0.00358582,
     0.01791382,  0.06657410,  0.02952576,  0.01698303,  0.00154114, -0.00361633,  0.00166321, -0.00538635,
     0.00483704,  0.01110840,  0.01173401, -0.00868225, -0.03361511, -0.06233215, -0.03771973, -0.02009583,
    -0.00576782, -0.01533508, -0.01855469,  0.03782654,  0.04870605,  0.04002380,  0.02944946,  0.01617432,
    -0.00526428, -0.01495361, -0.03617859, -0.05659485,  0.00096130,  0.01994324,  0.01362610,  0.00981140,
    -0.00202942, -0.00065613,  0.02978516,  0.07008362,  0.05216980,  0.03585815,  0.01889038,  0.00332642,
    -0.00016785,  0.00070190,  0.00903320,  0.03486633,  0.03285217,  0.00875854, -0.03100586, -0.03533936,
    -0.00460815, -0.00889587, -0.00535583, -0.00605774, -0.01533508,  0.03904724,  0.05235291,  0.02601624,
     0.00132751, -0.00230408, -0.00685120, -0.03175354, -0.06939697, -0.02249146, -0.01206970, -0.00375366,
    -0.00267029,  0.01330566,  0.07746887,  0.05206299,  0.03462219,  0.02912903,  0.02680969,  0.01100159,
     0.00160217,  0.01959229,  0.05360413,  0.01110840, -0.02023315, -0.03753662, -0.01402283, -0.01716614,
    -0.00639343, -0.01054382, -0.00729370,  0.00303650,  0.02307129,  0.07855225,  0.04028320,  0.01892090,
    -0.00352478, -0.01866150, -0.02894592, -0.03585815, -0.06474304, -0.00245667,  0.02102661,  0.01512146,
    -0.00077820,  0.01048279,  0.01577759,  0.02217102,  0.07875061,  0.04440308,  0.00306702, -0.00975037,
     0.00099182, -0.01753235, -0.02914429, -0.03764343, -0.03930664, -0.04081726, -0.06845093, -0.01873779,
    -0.00256348, -0.00033569,  0.00053406,  0.00070190, -0.00105286,  0.00088501,  0.08264160,  0.03260803,
     0.00869751,  0.00711060, -0.00775146, -0.03376770, -0.06314087, -0.08934021, -0.01795959,  0.00088501,
     0.02838135,  0.09507751,  0.06649780,  0.05419922,  0.04470825,  0.03926086,  0.02592468,  0.01179504,
     0.04316711,  0.05152893,  0.01130676, -0.02204895, -0.04406738, -0.06632996, -0.08439636, -0.05546570,
    -0.00311279,  0.00325012,  0.02406311,  0.04458618,  0.07960510,  0.07987976,  0.04357910,  0.01593018,
    -0.02593994, -0.07890320, -0.02648926, -0.02957153, -0.01586914, -0.01681519, -0.00686646, -0.00469971,
     0.00602722,  0.03062439,  0.07060242,  0.06431580,  0.04623413,  0.02545166, -0.00128174, -0.01522827,
     0.00640869,  0.01506042,  0.02645874,  0.01609802,  0.00148010, -0.01939392, -0.05572510, -0.06500244,
    -0.00239563,  0.00444031,  0.01907349,  0.03089905,  0.03352356,  0.05075073,  0.07539368,  0.03486633,
    -0.00079346, -0.03021240, -0.05854797, -0.07080078, -0.06494141, -0.05015564, -0.02285767, -0.00508118,
     0.00588989,  0.03402710,  0.08795166,  0.09323120,  0.07124329,  0.05776978,  0.03340149,  0.01075745,
     0.02717590,  0.07472229,  0.08680725,  0.03575134,  0.00018311, -0.03523254, -0.05368042, -0.04931641
};

/* spec */
const Float bv32_lspecb21[] =
{
     0.00281525,  0.00292778,  0.00433731,
    -0.00021553, -0.00037766, -0.00252151,
     0.00709152, -0.00558853, -0.00040245,
    -0.00341034, -0.00456047,  0.00535393,
    -0.00196075,  0.00144005,  0.01340103,
    -0.00179482, -0.00482559, -0.00926208,
    -0.00576019,  0.00680923,  0.00318718,
    -0.00498390, -0.01045990, -0.00181580,
     0.00724030,  0.00892258, -0.00010681,
    -0.00100517,  0.00750542, -0.01124763,
     0.01622772,  0.00503349, -0.00928497,
    -0.01317978, -0.00148201, -0.00485039,
     0.00139236,  0.01294518,  0.01284790,
     0.00160599, -0.00276566, -0.02051735,
     0.00048065,  0.02153206, -0.00239372,
     0.00121498, -0.01841927,  0.00706482,
     0.01221657,  0.00114632,  0.01258469,
     0.00564766,  0.00059319, -0.00907707,
     0.02144051, -0.01291847, -0.00042725,
    -0.01160431, -0.01168442,  0.01208878,
    -0.00497437, -0.00429916,  0.02562332,
    -0.00357437, -0.01308441, -0.01529694,
    -0.01611328,  0.01459503,  0.00725365,
    -0.01193810, -0.02121544, -0.00399017,
     0.01710129,  0.01618958,  0.00624657,
     0.00753784,  0.01832008, -0.02398491,
     0.03960609,  0.01548195, -0.00556374,
    -0.03484535,  0.00230217,  0.00053406,
    -0.00045013,  0.01565170,  0.03667641,
    -0.00150681, -0.01651573, -0.03601646,
     0.00778198,  0.04269028,  0.00644302,
    -0.01263237, -0.04002953,  0.00638008
};

/* spec */
const Float bv32_lspecb22[] =
{
     0.00223160, -0.00800133, -0.00899124,  0.00006485,  0.00058365,
     0.00498199,  0.00384903, -0.00713539, -0.00961494, -0.00307274,
    -0.00000954,  0.00230217,  0.00827026,  0.00367355,  0.00186920,
     0.00362587,  0.01415634,  0.00111580,  0.00265884, -0.00458145,
    -0.01116562,  0.00059700, -0.01137161,  0.00316811, -0.00823975,
     0.00366402,  0.00034904, -0.00654984,  0.00271797, -0.01940155,
    -0.00282288, -0.00809288,  0.00408554, -0.00595474, -0.00964355,
     0.01284599,  0.00154495,  0.00731087,  0.00330925, -0.00998116,
    -0.00849152, -0.00714302,  0.00018120,  0.00532913,  0.00732613,
    -0.00639915,  0.00654030, -0.00492859, -0.00344276,  0.01243401,
    -0.00438499,  0.00685120, -0.00248146,  0.01663589,  0.00031281,
     0.01028252,  0.00627327, -0.00315285,  0.00683403,  0.00990868,
    -0.01620674,  0.00895309,  0.00953102,  0.00367737, -0.00362778,
    -0.00172234,  0.00682259,  0.00998497, -0.01184273,  0.00318718,
    -0.00300217, -0.00821686,  0.00954819,  0.01287270, -0.00807762,
     0.01217651, -0.00773621,  0.00847435, -0.00031281,  0.00645638,
    -0.00471497, -0.01052666, -0.02195930, -0.01058769,  0.00412560,
     0.00894547, -0.00356674, -0.00493240, -0.02550888, -0.00962448,
    -0.00122452,  0.00730324,  0.01606369,  0.01205063,  0.01569366,
    -0.00556946,  0.02675247, -0.00582695, -0.00326729,  0.00189209,
    -0.01784134,  0.00078583, -0.00429535, -0.01312637, -0.00244522,
    -0.00508881,  0.00965881,  0.00708389, -0.01148987, -0.02126884,
    -0.00472450, -0.01339912,  0.00592613, -0.01262474,  0.00816154,
     0.02260780,  0.01769447,  0.00827408, -0.00707054, -0.00349998,
    -0.00877571, -0.00870895, -0.01420212,  0.01482201,  0.01783562,
     0.00730515,  0.00027847, -0.00198555, -0.01367950,  0.02097321,
     0.00929070, -0.00706673, -0.00564384,  0.01904678,  0.00018692,
     0.01049805,  0.01000977, -0.02177620,  0.00494194,  0.00013351,
    -0.02701187, -0.01168251,  0.01052856,  0.00321388,  0.00094223,
     0.00286293, -0.00534248,  0.02644157, -0.00658035, -0.00415039,
     0.00362587, -0.02618980,  0.00177765,  0.00383186, -0.00398064,
     0.02854538, -0.00962830, -0.00597000, -0.00085640, -0.00148964
};

/* spec */
const Float bv32_lspmean[] =  	/* mean value of LSP components */
{
    0.0551453,
    0.1181030,
    0.2249756,
    0.3316040,
    0.4575806,
    0.5720825,
    0.7193298,
    0.8278198
};

/* spec */
const Float bv32_lspp[] =  	/* LSP Predictor Coefficients */
{
    0.7401123, 0.6939697, 0.6031494, 0.5333862, 0.4295044, 0.3234253, 0.2177124, 0.1162720,
    0.7939453, 0.7693481, 0.6712036, 0.5919189, 0.4750366, 0.3556519, 0.2369385, 0.1181030,
    0.7534180, 0.7318115, 0.6326294, 0.5588379, 0.4530029, 0.3394775, 0.2307739, 0.1201172,
    0.7188110, 0.6765747, 0.5792847, 0.5169067, 0.4223022, 0.3202515, 0.2235718, 0.1181030,
    0.6431885, 0.6023560, 0.5112305, 0.4573364, 0.3764038, 0.2803345, 0.2060547, 0.1090698,
    0.5687866, 0.5837402, 0.4616089, 0.4351196, 0.3502808, 0.2602539, 0.1951294, 0.0994263,
    0.5292969, 0.4835205, 0.3890381, 0.3581543, 0.2882080, 0.2261353, 0.1708984, 0.0941162,
    0.5134277, 0.4365845, 0.3521729, 0.3118896, 0.2514038, 0.1951294, 0.1443481, 0.0841064
};

/* spec */
const Float bv32_lgpecb[] =  	/* Log-Gain Prediction Error CodeBook */
{
    -4.91895,
    -3.75049,
    -3.09082,
    -2.59961,
    -2.22656,
    -1.46240,
    -0.88037,
    -0.34717,
    -1.93408,
    -1.25635,
    -0.70117,
    -0.16650,
     0.20361,
     0.82568,
     1.59863,
     2.75684,
    -1.68457,
    -1.06299,
    -0.52588,
     0.01563,
     0.39941,
     1.05664,
     1.91602,
     3.34326,
     0.60693,
     1.31201,
     2.29736,
     4.11426,
     5.20996,
     6.70410,
     8.74316,
    10.92188
};

/* spec */
const Float bv32_lgp[] =	 	/* Log-Gain Predictor */
{
    0.5913086,
    0.5251160,
    0.5724792,
    0.5977783,
    0.4800720,
    0.4939270,
    0.4729614,
    0.4158936,
    0.3805847,
    0.3395081,
    0.2780151,
    0.2455139,
    0.1916199,
    0.1470032,
    0.1138611,
    0.0664673
};

/* spec */
const Float bv32_lgmean = 11.82031;

/* spec */
const Float bv32_cccb[] =
{
    -0.537476,  0.974976, -0.631104, -0.617920,
     1.145142,  1.222534, -1.252441,  0.616211,
     1.174194,  1.399414,  0.330933,  0.823120,
     2.946899,  0.798096, -0.274658, -0.027344,
    -1.704102,  0.098755, -0.526001, -0.395508,
    -0.889038, -0.337402,  0.784546,  0.298462,
    -0.756958, -0.061890,  0.558960, -0.907227,
     1.373169, -0.413330,  0.690552, -0.794067,
    -0.573364, -0.463745, -0.606934, -0.623535,
     1.058716, -0.566040, -1.677246,  0.752563,
     0.432617,  0.441895, -0.630493, -1.445801,
     2.542358,  0.207031, -1.611450,  0.313354,
    -2.572266, -2.758423, -0.499390, -0.020142,
     0.432251, -2.303711, -2.016479,  0.228638,
     0.701538, -1.355591, -0.861572, -0.243042,
     0.857056, -1.842285, -0.006348,  1.216919,
    -1.474365,  1.636108, -0.683838,  0.362915,
    -0.361694,  0.711914, -0.136353,  1.619873,
     0.407104,  1.661255,  0.566406, -0.559937,
     1.670288,  1.159668,  1.760254,  0.524780,
    -1.860596,  0.592285,  1.213379,  0.719482,
    -0.845703,  0.081421,  2.197754,  1.654785,
     0.425415,  0.641357,  1.210205, -1.444580,
     0.208374,  0.481567,  1.808472,  0.685913,
    -1.022583,  0.425781, -0.168945, -1.642700,
     0.502075, -0.491455, -0.296631, -0.068359,
     0.270630,  0.005981,  0.257813, -0.466309,
     2.266357, -1.128540, -0.399414,  0.438477,
    -1.876343, -0.895142, -0.012207,  0.886841,
    -0.389771, -1.818604,  1.185791,  0.913452,
    -0.040771, -1.141968,  0.364258, -0.283691,
     0.448242, -0.755127,  1.767578, -0.691406
};

const Float bv32_pp9cb[] =
{
    -2.9005200e-001,  5.9857400e-001,  2.8242800e-001,  8.6808793e-002,
    -8.4527029e-002,  4.0959403e-002, -2.1032541e-002, -8.9572708e-002,
    -1.9941394e-002,
    -2.5773600e-001,  1.3779820e+000,  8.1913200e-001,  1.7757778e-001,
    -5.6437458e-001,  1.0555990e-001, -1.6606961e-002, -4.7470860e-001,
    -1.6774431e-001,
    -6.8988000e-002,  4.0204400e-001,  2.3288400e-001,  1.3868106e-002,
    -4.6814807e-002,  8.0331007e-003, -1.1898360e-003, -4.0409844e-002,
    -1.3558739e-002,
    -1.0210000e-002,  1.0048600e+000,  7.7118200e-001,  5.1298103e-003,
    -3.8746497e-001,  3.9368841e-003, -2.6061025e-005, -2.5243590e-001,
    -1.4868042e-001,
    -1.1636000e-002,  5.6408400e-001,  1.0782800e-001,  3.2818407e-003,
    -3.0412025e-002,  6.2734330e-004, -3.3849124e-005, -7.9547690e-002,
    -2.9067194e-003,
     4.8679200e-001,  8.9901600e-001,  5.7433200e-001, -2.1881690e-001,
    -2.5816683e-001, -1.3979011e-001, -5.9241613e-002, -2.0205744e-001,
    -8.2464312e-002,
    -1.3678200e-001, -4.6110800e-001,  1.5254600e-001, -3.1535637e-002,
     3.5170090e-002,  1.0432773e-002, -4.6773289e-003, -5.3155147e-002,
    -5.8175705e-003,
     4.1686000e-002,  6.4152800e-001,  4.3733400e-001, -1.3371368e-002,
    -1.4028100e-001, -9.1153526e-003, -4.3443065e-004, -1.0288954e-001,
    -4.7815257e-002,
     1.6641000e-001,  6.6628800e-001,  4.2048000e-002, -5.5438493e-002,
    -1.4008039e-002, -3.4986038e-003, -6.9230720e-003, -1.1098492e-001,
    -4.4200858e-004,
     2.3891400e-001,  1.2416060e+000,  4.7916200e-001, -1.4831853e-001,
    -2.9746521e-001, -5.7239255e-002, -1.4269975e-002, -3.8539636e-001,
    -5.7399056e-002,
     1.5663800e-001,  3.7065800e-001,  1.8346800e-001, -2.9029564e-002,
    -3.4001941e-002, -1.4369030e-002, -6.1338658e-003, -3.4346838e-002,
    -8.4151268e-003,
     3.2286400e-001,  8.4265000e-001,  6.6942000e-001, -1.3603067e-001,
    -2.8204338e-001, -1.0806581e-001, -2.6060291e-002, -1.7751476e-001,
    -1.1203078e-001,
     3.4039200e-001,  4.8907800e-001,  9.7528000e-002, -8.3239119e-002,
    -2.3849400e-002, -1.6598875e-002, -2.8966678e-002, -5.9799323e-002,
    -2.3779277e-003,
     6.8932600e-001,  9.7475400e-001,  2.9750200e-001, -3.3596164e-001,
    -1.4499563e-001, -1.0253793e-001, -1.1879258e-001, -2.3753634e-001,
    -2.2126860e-002,
     2.3204200e-001, -4.9914600e-001,  1.4655400e-001,  5.7911418e-002,
     3.6575921e-002, -1.7003342e-002, -1.3460872e-002, -6.2286682e-002,
    -5.3695187e-003,
     5.7895600e-001,  6.7697000e-001,  4.8903800e-001, -1.9596792e-001,
    -1.6553203e-001, -1.4156574e-001, -8.3797512e-002, -1.1457210e-001,
    -5.9789541e-002,
    -2.6667800e-001,  6.9637800e-001, -1.3954600e-001,  9.2854346e-002,
     4.8588382e-002, -1.8606924e-002, -1.7779289e-002, -1.2123558e-001,
    -4.8682715e-003,
    -1.7600000e-002,  1.6969620e+000,  2.8014800e-001,  1.4933266e-002,
    -2.3770026e-001,  2.4653024e-003, -7.7440000e-005, -7.1992001e-001,
    -1.9620725e-002,
    -1.6124400e-001,  4.1076200e-001, -3.2358000e-002,  3.3116454e-002,
     6.6457184e-003, -2.6087667e-003, -6.4999069e-003, -4.2181355e-002,
    -2.6176004e-004,
    -6.6390000e-002,  9.5376600e-001,  2.3754200e-001,  3.1660262e-002,
    -1.1327974e-001,  7.8852067e-003, -1.1019080e-003, -2.2741740e-001,
    -1.4106550e-002,
     3.8460000e-003,  5.3013600e-001, -2.6145400e-001, -1.0194515e-003,
     6.9303089e-002,  5.0277604e-004, -3.6979290e-006, -7.0261045e-002,
    -1.7089549e-002,
     4.3861400e-001,  1.4447380e+000,  6.2326000e-002, -3.1684116e-001,
    -4.5022370e-002, -1.3668528e-002, -4.8095560e-002, -5.2181697e-001,
    -9.7113257e-004,
     1.5442000e-002,  1.4250800e-001,  2.5232000e-002, -1.1003043e-003,
    -1.7978809e-003, -1.9481627e-004, -5.9613841e-005, -5.0771325e-003,
    -1.5916346e-004,
     2.9463000e-001,  5.2297600e-001,  3.5731600e-001, -7.7042209e-002,
    -9.3433846e-002, -5.2638007e-002, -2.1701709e-002, -6.8375974e-002,
    -3.1918681e-002,
     2.6816800e-001,  9.2876800e-001, -2.0899000e-001, -1.2453293e-001,
     9.7051612e-002,  2.8022215e-002, -1.7978519e-002, -2.1565250e-001,
    -1.0919205e-002,
     8.3488800e-001,  1.3611020e+000, -2.3238000e-001, -5.6818386e-001,
     1.5814644e-001,  9.7005637e-002, -1.7425949e-001, -4.6314966e-001,
    -1.3500116e-002,
     1.2479600e-001,  3.8813600e-001, -5.9852000e-002, -2.4218910e-002,
     1.1615358e-002,  3.7346451e-003, -3.8935104e-003, -3.7662389e-002,
    -8.9556548e-004,
     3.3026200e-001,  8.9619600e-001,  2.9964600e-001, -1.4798974e-001,
    -1.3427077e-001, -4.9480844e-002, -2.7268247e-002, -2.0079182e-001,
    -2.2446931e-002,
     3.2806200e-001,  5.4098200e-001, -2.0613400e-001, -8.8737818e-002,
     5.5757392e-002,  3.3812366e-002, -2.6906169e-002, -7.3165381e-002,
    -1.0622806e-002,
     7.7764400e-001,  1.0285520e+000, -4.9966000e-002, -3.9992365e-001,
     2.5696315e-002,  1.9427880e-002, -1.5118255e-001, -2.6447980e-001,
    -6.2415029e-004,
     8.2660000e-003, -4.3842200e-001, -1.6479000e-001,  1.8119981e-003,
    -3.6123781e-002,  6.8107707e-004, -1.7081689e-005, -4.8053463e-002,
    -6.7889360e-003,
     5.4489000e-001,  7.3032400e-001,  1.0464600e-001, -1.9897312e-001,
    -3.8212743e-002, -2.8510279e-002, -7.4226278e-002, -1.3334329e-001,
    -2.7376963e-003
};

/* Pre-Emphasis or De-Emphasis Filter */
const Float bv32_a_pre[] = { 1.0, PEAPFC };
const Float bv32_b_pre[] = { 1.0, PEAZFC };

/* Combined High-Pass & Pre-Emphasis Filter */
const Float bv32_hpfa[] = {1.00000000000000,  PEAPFC-(127./128), PEAPFC*(-127./128)};
const Float bv32_hpfb[] = {255./256,  255./256*(PEAZFC-1.), -255./256*PEAZFC};

/* spec */
/* coarse pitch search */
const Float bv32_adf[] =
{
     1.00000000000000,
    -3.56023061648011,
     4.85584780946374,
    -2.99882978945971,
     0.70692772790790
};

/* spec */
const Float bv32_bdf[] =
{
     0.0322952,
    -0.1028824,
     0.1446838,
    -0.1028824,
     0.0322952
};

const Float bv32_x[]    = {0.125, 0.25, 0.375, 0.5};
const Float bv32_x2[]   = {0.015625, 0.0625, 0.140625, 0.25}; /* square of x[] */
const Float bv32_MPTH[] = {0.7, 0.55, 0.48, 0.37};

/* spec */
const Float bv32_winl[] =
{
    0.0001831,
    0.0006714,
    0.0015259,
    0.0026855,
    0.0042114,
    0.0060425,
    0.0082397,
    0.0107422,
    0.0135803,
    0.0167542,
    0.0202637,
    0.0240784,
    0.0281982,
    0.0326843,
    0.0374451,
    0.0425110,
    0.0479126,
    0.0536194,
    0.0596008,
    0.0659180,
    0.0725098,
    0.0793762,
    0.0865479,
    0.0939636,
    0.1016846,
    0.1096497,
    0.1179199,
    0.1264038,
    0.1351624,
    0.1441650,
    0.1534119,
    0.1628723,
    0.1725769,
    0.1824951,
    0.1926270,
    0.2029724,
    0.2135010,
    0.2242432,
    0.2351685,
    0.2462769,
    0.2575378,
    0.2689514,
    0.2805481,
    0.2922974,
    0.3041687,
    0.3161926,
    0.3283081,
    0.3405762,
    0.3529053,
    0.3653870,
    0.3779297,
    0.3905334,
    0.4032593,
    0.4160156,
    0.4288330,
    0.4417114,
    0.4546204,
    0.4675598,
    0.4805298,
    0.4934998,
    0.5065002,
    0.5194702,
    0.5324402,
    0.5453796,
    0.5582886,
    0.5711670,
    0.5839844,
    0.5967407,
    0.6094666,
    0.6220703,
    0.6346130,
    0.6470947,
    0.6594238,
    0.6716919,
    0.6838074,
    0.6958313,
    0.7077026,
    0.7194519,
    0.7310486,
    0.7424622,
    0.7537231,
    0.7648315,
    0.7757568,
    0.7864990,
    0.7970276,
    0.8073730,
    0.8175049,
    0.8274231,
    0.8371277,
    0.8465881,
    0.8558350,
    0.8648376,
    0.8735962,
    0.8820801,
    0.8903503,
    0.8983154,
    0.9060364,
    0.9134521,
    0.9206238,
    0.9274902,
    0.9340820,
    0.9403992,
    0.9463806,
    0.9520874,
    0.9574890,
    0.9625549,
    0.9673157,
    0.9718018,
    0.9759216,
    0.9797363,
    0.9832458,
    0.9864197,
    0.9892578,
    0.9917603,
    0.9939575,
    0.9957886,
    0.9973145,
    0.9984741,
    0.9993286,
    0.9998169,
    1.0000000,
    0.9992371,
    0.9969177,
    0.9930725,
    0.9877014,
    0.9807739,
    0.9723816,
    0.9624634,
    0.9510498,
    0.9382019,
    0.9238892,
    0.9081421,
    0.8910217,
    0.8724976,
    0.8526306,
    0.8314819,
    0.8090210,
    0.7853088,
    0.7604065,
    0.7343140,
    0.7070923,
    0.6788025,
    0.6494446,
    0.6190796,
    0.5877991,
    0.5555725,
    0.5224915,
    0.4886169,
    0.4539795,
    0.4186707,
    0.3826904,
    0.3461304,
    0.3090210,
    0.2714539,
    0.2334595,
    0.1950989,
    0.1564331,
    0.1175232,
    0.0784607,
    0.0392456
};

/* spec */
const Float bv32_sstwin[] =
{
    1.00003162253432,
    0.99950664035278,
    0.99802802494523,
    0.99556852541705,
    0.99213540445135,
    0.98773878305059,
    0.98239158372414,
    0.97610947553847,
    0.96891079029551
};

const int bv32_idxord[] =
{
     0,  1,  2,  3,  4,  8, 16,  5,  9, 17,  6, 10, 18,  7, 11, 19,
    12, 20, 24, 13, 21, 25, 14, 22, 26, 15, 23, 27, 28, 29, 30, 31
};

/* spec */
const Float bv32_lgpecb_nh[] =  	/* Log-Gain Prediction Error Next-Higher CodeBook */
{
    -3.75049,
    -3.09082,
    -2.59961,
    -2.22656,
    -1.93408,
    -1.25635,
    -0.70117,
    -0.16650,
    -1.68457,
    -1.06299,
    -0.52588,
     0.01563,
     0.39941,
     1.05664,
     1.91602,
     3.34326,
    -1.46240,
    -0.88037,
    -0.34717,
     0.20361,
     0.60693,
     1.31201,
     2.29736,
     4.11426,
     0.82568,
     1.59863,
     2.75684,
     5.20996,
     6.70410,
     8.74316,
    10.92188
};
