/*
 * Decompiled with CFR 0.152.
 */
package com.intellij.openapi.util.io;

import com.intellij.openapi.util.Pair;
import com.intellij.openapi.util.SystemInfo;
import com.intellij.openapi.util.io.FileUtil;
import com.intellij.openapi.util.text.StringUtil;
import com.intellij.openapi.vfs.CharsetToolkit;
import com.intellij.util.ObjectUtils;
import com.intellij.util.PathUtil;
import com.intellij.util.containers.ContainerUtil;
import java.io.BufferedReader;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.OutputStream;
import java.nio.charset.Charset;
import java.nio.file.FileSystems;
import java.nio.file.Files;
import java.nio.file.attribute.FileAttribute;
import java.util.Collection;
import java.util.Locale;
import java.util.Set;
import java.util.function.Predicate;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import java.util.zip.ZipEntry;
import java.util.zip.ZipOutputStream;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;
import org.junit.Assert;

public class IoTestUtil {
    private static final String[] UNICODE_PARTS = new String[]{"\u042e\u043d\u0438\u043a\u043e\u0434\u0435", "\u00da\u00f1\u00ed\u00e7\u00f8d\u00ea"};

    private IoTestUtil() {
    }

    @Nullable
    public static String getUnicodeName() {
        return IoTestUtil.filterParts(PathUtil::isValidFileName);
    }

    @Nullable
    public static String getUnicodeName(String forEncoding) {
        return IoTestUtil.filterParts(Charset.forName(forEncoding).newEncoder()::canEncode);
    }

    private static String filterParts(Predicate<String> predicate) {
        return StringUtil.nullize((String)Stream.of(UNICODE_PARTS).filter(predicate).collect(Collectors.joining("_")));
    }

    @NotNull
    public static File getTempDirectory() {
        File dir = new File(FileUtil.getTempDirectory());
        dir = IoTestUtil.expandWindowsPath(dir);
        return dir;
    }

    private static File expandWindowsPath(File file) {
        if (SystemInfo.isWindows && file.getPath().indexOf(126) > 0) {
            try {
                return file.getCanonicalFile();
            }
            catch (IOException iOException) {
                // empty catch block
            }
        }
        return file;
    }

    @NotNull
    public static File createSymLink(@NotNull String target, @NotNull String link) {
        return IoTestUtil.createSymLink(target, link, true);
    }

    @NotNull
    public static File createSymLink(@NotNull String target, @NotNull String link, boolean shouldExist) {
        File linkFile = IoTestUtil.getFullLinkPath(link);
        try {
            Files.createSymbolicLink(linkFile.toPath(), FileSystems.getDefault().getPath(target, new String[0]), new FileAttribute[0]);
        }
        catch (IOException e) {
            throw new RuntimeException(e);
        }
        Assert.assertEquals((String)("target=" + target + ", link=" + linkFile), (Object)shouldExist, (Object)linkFile.exists());
        return linkFile;
    }

    @NotNull
    public static File createJunction(@NotNull String target, @NotNull String junction) {
        Assert.assertTrue((boolean)SystemInfo.isWindows);
        File targetFile = new File(target);
        Assert.assertTrue((String)targetFile.getPath(), (boolean)targetFile.isDirectory());
        File junctionFile = IoTestUtil.getFullLinkPath(junction);
        IoTestUtil.runCommand("cmd", "/C", "mklink", "/J", junctionFile.getPath(), targetFile.getPath());
        Assert.assertTrue((String)("target=" + targetFile + ", link=" + junctionFile), (boolean)junctionFile.isDirectory());
        return junctionFile;
    }

    public static void deleteJunction(@NotNull String junction) {
        Assert.assertTrue((boolean)SystemInfo.isWindows);
        Assert.assertTrue((boolean)new File(junction).delete());
    }

    @NotNull
    public static File createSubst(@NotNull String target) {
        Assert.assertTrue((boolean)SystemInfo.isWindows);
        File targetFile = new File(target);
        Assert.assertTrue((String)targetFile.getPath(), (boolean)targetFile.isDirectory());
        String substRoot = IoTestUtil.getFirstFreeDriveLetter() + ":";
        IoTestUtil.runCommand("subst", substRoot, targetFile.getPath());
        File rootFile = new File(substRoot + "\\");
        Assert.assertTrue((String)("target=" + targetFile + ", subst=" + rootFile), (boolean)rootFile.isDirectory());
        return rootFile;
    }

    public static void deleteSubst(@NotNull String substRoot) {
        IoTestUtil.runCommand("subst", StringUtil.trimEnd((String)substRoot, (String)"\\"), "/d");
    }

    private static char getFirstFreeDriveLetter() {
        Set roots = ContainerUtil.map2Set((Object[])File.listRoots(), root -> Character.valueOf(root.getPath().toUpperCase(Locale.US).charAt(0)));
        for (char c = 'E'; c <= 'Z'; c = (char)(c + '\u0001')) {
            if (roots.contains(Character.valueOf(c))) continue;
            return c;
        }
        throw new RuntimeException("No free roots");
    }

    private static File getFullLinkPath(String link) {
        File linkFile = new File(link);
        if (!linkFile.isAbsolute()) {
            linkFile = new File(IoTestUtil.getTempDirectory(), link);
        }
        Assert.assertTrue((String)link, (!linkFile.exists() || linkFile.delete() ? 1 : 0) != 0);
        File parentDir = linkFile.getParentFile();
        Assert.assertTrue((String)("link=" + link + ", parent=" + parentDir), (parentDir != null && (parentDir.isDirectory() || parentDir.mkdirs()) ? 1 : 0) != 0);
        return linkFile;
    }

    private static void runCommand(String ... command) {
        try {
            ProcessBuilder builder = new ProcessBuilder(command);
            builder.redirectErrorStream(true);
            Process process2 = builder.start();
            StringBuilder output = new StringBuilder();
            Thread thread = new Thread(() -> {
                try (BufferedReader reader = new BufferedReader(new InputStreamReader(process2.getInputStream()));){
                    String line;
                    while ((line = reader.readLine()) != null) {
                        output.append(line).append('\n');
                    }
                }
                catch (IOException e) {
                    throw new RuntimeException(e);
                }
            }, "io test");
            thread.start();
            int ret = process2.waitFor();
            thread.join();
            if (ret != 0) {
                throw new RuntimeException(builder.command() + "\nresult: " + ret + "\noutput:\n" + output);
            }
        }
        catch (IOException | InterruptedException e) {
            throw new RuntimeException(e);
        }
    }

    public static void assertTimestampsEqual(long expected, long actual) {
        long roundedExpected = expected / 1000L * 1000L;
        long roundedActual = actual / 1000L * 1000L;
        Assert.assertEquals((String)("expected: " + expected + ", actual: " + actual), (long)roundedExpected, (long)roundedActual);
    }

    public static void assertTimestampsNotEqual(long expected, long actual) {
        long roundedExpected = expected / 1000L * 1000L;
        long roundedActual = actual / 1000L * 1000L;
        Assert.assertTrue((String)("(un)expected: " + expected + ", actual: " + actual), (roundedExpected != roundedActual ? 1 : 0) != 0);
    }

    @NotNull
    public static File createTestJar() {
        try {
            File jarFile = IoTestUtil.expandWindowsPath(FileUtil.createTempFile((String)"test.", (String)".jar"));
            return IoTestUtil.createTestJar(jarFile);
        }
        catch (IOException e) {
            throw new RuntimeException(e);
        }
    }

    @NotNull
    public static File createTestJar(File jarFile) {
        return IoTestUtil.createTestJar(jarFile, "META-INF/MANIFEST.MF", "");
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    @NotNull
    public static File createTestJar(@NotNull File jarFile, String ... namesAndTexts) {
        try (ZipOutputStream stream = new ZipOutputStream(new FileOutputStream(jarFile));){
            for (int i = 0; i < namesAndTexts.length; i += 2) {
                stream.putNextEntry(new ZipEntry(namesAndTexts[i]));
                if (namesAndTexts[i + 1] != null) {
                    stream.write(namesAndTexts[i + 1].getBytes(CharsetToolkit.UTF8_CHARSET));
                }
                stream.closeEntry();
            }
            File file = jarFile;
            return file;
        }
        catch (IOException e) {
            throw new RuntimeException(e);
        }
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    @NotNull
    public static File createTestJar(@NotNull File jarFile, @NotNull Collection<Pair<String, byte[]>> namesAndContents) {
        try (ZipOutputStream stream = new ZipOutputStream(new FileOutputStream(jarFile));){
            for (Pair<String, byte[]> p : namesAndContents) {
                String name2 = (String)p.first;
                byte[] content = (byte[])p.second;
                stream.putNextEntry(new ZipEntry(name2));
                stream.write(content);
                stream.closeEntry();
            }
            File file = jarFile;
            return file;
        }
        catch (IOException e) {
            throw new RuntimeException(e);
        }
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    @NotNull
    public static File createTestJar(@NotNull File jarFile, @NotNull File root) {
        try (ZipOutputStream stream = new ZipOutputStream(new FileOutputStream(jarFile));){
            FileUtil.visitFiles((File)root, file -> {
                if (file.isFile()) {
                    String path = FileUtil.toSystemIndependentName((String)((String)ObjectUtils.assertNotNull((Object)FileUtil.getRelativePath((File)root, (File)file))));
                    try {
                        stream.putNextEntry(new ZipEntry(path));
                        try (FileInputStream is = new FileInputStream((File)file);){
                            FileUtil.copy((InputStream)is, (OutputStream)stream);
                        }
                        stream.closeEntry();
                    }
                    catch (IOException e) {
                        throw new RuntimeException(e);
                    }
                }
                return true;
            });
            File file2 = jarFile;
            return file2;
        }
        catch (IOException e) {
            throw new RuntimeException(e);
        }
    }

    @NotNull
    public static File createTestDir(@NotNull String name2) {
        return IoTestUtil.createTestDir(IoTestUtil.getTempDirectory(), name2);
    }

    @NotNull
    public static File createTestDir(@NotNull File parent, @NotNull String name2) {
        File dir = new File(parent, name2);
        Assert.assertTrue((String)dir.getPath(), (boolean)dir.mkdirs());
        return dir;
    }

    @NotNull
    public static File createTestFile(@NotNull String name2) {
        return IoTestUtil.createTestFile(name2, null);
    }

    @NotNull
    public static File createTestFile(@NotNull String name2, @Nullable String content) {
        return IoTestUtil.createTestFile(IoTestUtil.getTempDirectory(), name2, content);
    }

    @NotNull
    public static File createTestFile(@NotNull File parent, @NotNull String name2) {
        return IoTestUtil.createTestFile(parent, name2, null);
    }

    @NotNull
    public static File createTestFile(@NotNull File parent, @NotNull String name2, @Nullable String content) {
        try {
            Assert.assertTrue((String)parent.getPath(), (parent.isDirectory() || parent.mkdirs() ? 1 : 0) != 0);
            File file = new File(parent, name2);
            Assert.assertTrue((String)file.getPath(), (boolean)file.createNewFile());
            if (content != null) {
                FileUtil.writeToFile((File)file, (String)content);
            }
            return file;
        }
        catch (IOException e) {
            throw new RuntimeException(e);
        }
    }

    public static void delete(File ... files) {
        for (File file : files) {
            if (file == null) continue;
            FileUtil.delete((File)file);
        }
    }

    public static void updateFile(@NotNull File file, String content) {
        try {
            FileUtil.writeToFile((File)file, (String)content);
        }
        catch (IOException e) {
            throw new RuntimeException(e);
        }
    }
}

