/*
 * Decompiled with CFR 0.152.
 */
package com.intellij.openapi.application;

import com.intellij.openapi.application.IdeDirectoriesInfo;
import com.intellij.openapi.util.text.StringUtil;
import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStreamReader;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.text.MessageFormat;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.jar.JarEntry;
import java.util.jar.JarFile;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

public class InstallationDetector {
    private final Map<Path, InstallationInfo> myPathToInstallationInfoCache = new HashMap<Path, InstallationInfo>();
    private final String myApplicationInfoFilePath;
    private static final List<Pattern> INSTALLATION_HOME_PATTERNS = Stream.of(Pattern.compile("\\s-Xbootclasspath/a:([^.].*?boot\\.jar)"), Pattern.compile("\\s-Djb.vmOptionsFile=(.*?vmoptions)"), Pattern.compile("\\sext\\s*:\\s*(.*)\\s*:\\s*\\[.*?\\.jar"), Pattern.compile("\\sStarting file watcher\\s*:\\s*(.*fsnotifier)")).collect(Collectors.toList());
    private static final Pattern ATTR_VAL_PATTERN = Pattern.compile("([-:a-zA-Z]+)\\s*=\\s*\"([^\"]*)\"");
    private static final List<String> INTERESTING_ATTRIBUTES = Stream.of("major", "minor", "micro", "patch", "full", "eap", "number", "date", "product", "fullname").collect(Collectors.toList());

    public InstallationDetector(@NotNull String applicationInfoFilePath) {
        this.myApplicationInfoFilePath = applicationInfoFilePath;
    }

    @Nullable
    public InstallationInfo detectInstallation(@NotNull IdeDirectoriesInfo dirsInfo) {
        SelectorInfo selectorInfo = new SelectorInfo(dirsInfo.getSelector());
        HashSet<Path> installationHomeCandidatesChecked = new HashSet<Path>();
        Path homeLocatorFile = dirsInfo.findHomeLocatorFile();
        InstallationInfo installationInfo = this.detectInstallationUsingHomeLocatorFile(homeLocatorFile, installationHomeCandidatesChecked, selectorInfo);
        if (installationInfo == null) {
            Path ideaLogFile = dirsInfo.findIdeaLogFile();
            installationInfo = this.detectInstallationUsingIdeaLogFile(ideaLogFile, installationHomeCandidatesChecked, selectorInfo);
        }
        return installationInfo;
    }

    @Nullable
    private InstallationInfo detectInstallationUsingHomeLocatorFile(Path homeLocatorFile, Set<Path> installationHomeCandidatesChecked, SelectorInfo selectorInfo) {
        InstallationInfo installationInfo;
        if (homeLocatorFile == null || !Files.isRegularFile(homeLocatorFile, new LinkOption[0])) {
            return null;
        }
        Path installationHomeCandidate = null;
        try (BufferedReader reader2 = Files.newBufferedReader(homeLocatorFile);){
            String line = reader2.readLine();
            if (line != null && !line.isEmpty()) {
                installationHomeCandidate = Paths.get(line, new String[0]);
            }
        }
        catch (IOException | SecurityException reader2) {
            // empty catch block
        }
        if (installationHomeCandidate != null && installationHomeCandidate.isAbsolute() && installationHomeCandidatesChecked.add(installationHomeCandidate) && InstallationDetector.matchInfo(installationInfo = this.extractInstallationInfo(installationHomeCandidate), selectorInfo)) {
            return installationInfo;
        }
        return null;
    }

    @Nullable
    private InstallationInfo detectInstallationUsingIdeaLogFile(Path ideaLogFile, Set<Path> installationHomeCandidatesChecked, SelectorInfo selectorInfo) {
        if (ideaLogFile == null || !Files.isRegularFile(ideaLogFile, new LinkOption[0])) {
            return null;
        }
        Path ideaLogDir = ideaLogFile.getParent();
        for (int i = 0; i < 13; ++i) {
            Path logFile = ideaLogDir.resolve(i == 0 ? "idea.log" : "idea.log." + i);
            InstallationInfo installationInfo = this.detectInstallationUsingIdeaLogFileInner(logFile, installationHomeCandidatesChecked, selectorInfo);
            if (installationInfo == null) continue;
            return installationInfo;
        }
        return null;
    }

    @Nullable
    private InstallationInfo detectInstallationUsingIdeaLogFileInner(Path ideaLogFile, Set<Path> installationHomeCandidatesChecked, SelectorInfo selectorInfo) {
        if (ideaLogFile == null || !Files.isRegularFile(ideaLogFile, new LinkOption[0])) {
            return null;
        }
        ArrayList<Path> candidates = new ArrayList<Path>(5);
        try (BufferedReader reader2 = Files.newBufferedReader(ideaLogFile);){
            String line;
            while ((line = reader2.readLine()) != null) {
                for (Pattern pattern : INSTALLATION_HOME_PATTERNS) {
                    Path candidate = InstallationDetector.findInstallationHomeCandidate(pattern, line);
                    if (candidate == null || !candidate.isAbsolute()) continue;
                    candidates.add(candidate);
                }
            }
        }
        catch (IOException | SecurityException reader2) {
            // empty catch block
        }
        for (int i = candidates.size() - 1; i >= 0; --i) {
            InstallationInfo installationInfo;
            Path candidate = (Path)candidates.get(i);
            if (!installationHomeCandidatesChecked.add(candidate) || !InstallationDetector.matchInfo(installationInfo = this.extractInstallationInfo(candidate), selectorInfo)) continue;
            return installationInfo;
        }
        return null;
    }

    @Nullable
    private static Path findInstallationHomeCandidate(@NotNull Pattern installationHomePattern, @NotNull String line) {
        String installationHomeCandidate;
        Matcher matcher = installationHomePattern.matcher(line);
        if (matcher.find() && (installationHomeCandidate = matcher.group(1)) != null && !installationHomeCandidate.isEmpty()) {
            return Paths.get(installationHomeCandidate, new String[0]);
        }
        return null;
    }

    private static boolean matchInfo(@Nullable InstallationInfo installationInfo, @Nullable SelectorInfo selectorInfo) {
        if (installationInfo == null || selectorInfo == null) {
            return false;
        }
        return installationInfo.getMajor().equals(selectorInfo.getMajor()) && installationInfo.getMinorMainPart().equals(selectorInfo.getMinor()) && (!selectorInfo.isAndroidStudio() || installationInfo.getEap().equals(selectorInfo.getEap()));
    }

    @Nullable
    private InstallationInfo extractInstallationInfo(@Nullable Path installationHomeCandidate) {
        if (installationHomeCandidate == null) {
            return null;
        }
        if (this.myPathToInstallationInfoCache.containsKey(installationHomeCandidate)) {
            return this.myPathToInstallationInfoCache.get(installationHomeCandidate);
        }
        Path installationHome = InstallationDetector.getInstallationHome(installationHomeCandidate);
        if (this.myPathToInstallationInfoCache.containsKey(installationHome)) {
            return this.myPathToInstallationInfoCache.get(installationHome);
        }
        InstallationInfo installationInfo = InstallationDetector.getInstallationInfo(installationHome, this.myApplicationInfoFilePath);
        this.myPathToInstallationInfoCache.put(installationHomeCandidate, installationInfo);
        this.myPathToInstallationInfoCache.put(installationHome, installationInfo);
        return installationInfo;
    }

    @Nullable
    private static Path getInstallationHome(@Nullable Path installationHomeCandidate) {
        if (installationHomeCandidate == null) {
            return null;
        }
        Path resourcesJar = Paths.get("lib", "resources.jar");
        while (installationHomeCandidate != null) {
            if (Files.isRegularFile(installationHomeCandidate.resolve(resourcesJar), new LinkOption[0])) {
                return installationHomeCandidate;
            }
            installationHomeCandidate = installationHomeCandidate.getParent();
        }
        return null;
    }

    @Nullable
    private static InstallationInfo getInstallationInfo(@Nullable Path installationHome, @NotNull String applicationInfoFilePath) {
        if (installationHome == null) {
            return null;
        }
        Path resourcesJar = installationHome.resolve("lib").resolve("resources.jar");
        try {
            JarFile jarFile = new JarFile(resourcesJar.toFile());
            JarEntry applicationInfoEntry = jarFile.getJarEntry(applicationInfoFilePath);
            return InstallationDetector.getInstallationInfo(jarFile, applicationInfoEntry);
        }
        catch (IOException iOException) {
            return null;
        }
    }

    @Nullable
    private static InstallationInfo getInstallationInfo(@NotNull JarFile resourcesJar, @Nullable JarEntry applicationInfoEntry) throws IOException {
        String patch;
        String micro;
        String minor;
        String major;
        if (applicationInfoEntry == null) {
            return null;
        }
        HashMap<String, String> attributeValueMap = new HashMap<String, String>();
        INTERESTING_ATTRIBUTES.forEach(attribute -> attributeValueMap.put((String)attribute, ""));
        try (BufferedReader reader = new BufferedReader(new InputStreamReader(resourcesJar.getInputStream(applicationInfoEntry)));){
            String line;
            while ((line = reader.readLine()) != null) {
                Matcher matcher = ATTR_VAL_PATTERN.matcher(line);
                while (matcher.find()) {
                    String attribute2 = matcher.group(1);
                    String value = matcher.group(2);
                    if (!attributeValueMap.containsKey(attribute2)) continue;
                    attributeValueMap.put(attribute2, value);
                }
            }
        }
        String full = (String)attributeValueMap.get("full");
        if (full.indexOf(123) >= 0) {
            major = (String)attributeValueMap.get("major");
            minor = (String)attributeValueMap.get("minor");
            micro = (String)attributeValueMap.get("micro");
            patch = (String)attributeValueMap.get("patch");
            attributeValueMap.put("full", MessageFormat.format(full, major, minor, micro, patch));
        } else if (full.isEmpty()) {
            major = (String)attributeValueMap.get("major");
            minor = (String)attributeValueMap.get("minor");
            micro = (String)attributeValueMap.get("micro");
            patch = (String)attributeValueMap.get("patch");
            attributeValueMap.put("full", InstallationDetector.joinNonEmpty(".", major, minor, micro, patch));
        }
        return new InstallationInfo(attributeValueMap);
    }

    private static String joinNonEmpty(@NotNull String delimiter, String ... elements) {
        return Arrays.stream(elements).reduce("", (a, b) -> b == null || b.isEmpty() ? a : (a == null || a.isEmpty() ? b : a + delimiter + b));
    }

    public static class InstallationInfo {
        private final String fullProductName;
        private final String fullVersion;
        private final String major;
        private final String minor;
        private final String micro;
        private final String patch;
        private final String eap;

        public InstallationInfo(Map<String, String> attributeValueMap) {
            this.fullProductName = attributeValueMap.get("fullname");
            this.fullVersion = attributeValueMap.get("full");
            this.major = attributeValueMap.get("major");
            this.minor = attributeValueMap.get("minor");
            this.micro = attributeValueMap.get("micro");
            this.patch = attributeValueMap.get("patch");
            this.eap = attributeValueMap.get("eap");
        }

        public String getFullProductName() {
            return this.fullProductName;
        }

        public String getFullVersion() {
            return this.fullVersion;
        }

        public String getMajor() {
            return this.major;
        }

        public String getMinor() {
            return this.minor;
        }

        public String getMinorMainPart() {
            int i = this.minor.indexOf(46);
            return i >= 0 ? this.minor.substring(0, i) : this.minor;
        }

        public String getMicro() {
            return this.micro;
        }

        public String getPatch() {
            return this.patch;
        }

        public String getEap() {
            return this.eap;
        }

        public String toString() {
            return this.getFullProductName() + " " + this.getFullVersion();
        }
    }

    public static class SelectorInfo {
        private final String prefix;
        private final String major;
        private final String minor;
        private final String eap;
        private static final Pattern SELECTOR_PATTERN = Pattern.compile("([a-zA-Z]+)((\\d+)(\\.(\\d+))?(\\.\\d+)*)?");

        public SelectorInfo(String selector) {
            Matcher matcher = SELECTOR_PATTERN.matcher(selector);
            if (matcher.matches()) {
                this.prefix = matcher.group(1);
                this.major = matcher.group(3);
                this.minor = matcher.group(5);
                this.eap = this.prefix.contains("Preview") ? "true" : "false";
            } else {
                this.prefix = "";
                this.major = "";
                this.minor = "";
                this.eap = "";
            }
        }

        public String getPrefix() {
            return this.prefix;
        }

        public String getMajor() {
            return this.major;
        }

        public String getMinor() {
            return this.minor;
        }

        public String getEap() {
            return this.eap;
        }

        public boolean isAndroidStudio() {
            return StringUtil.containsIgnoreCase((String)this.prefix, (String)"AndroidStudio");
        }
    }
}

