// *************************************************************************
//
//  COPYRIGHT 1996-2000 DIGIGRAM. ALL RIGHTS RESERVED.
//
//  DIGIGRAM
//
// **************************************************************************

#include "apidef.h"

#include "ifapidrv.h"  // private interface between driver and API
#include "pcxapi_e.h"    // prototypes of functions
#include "pcxerr_e.h"  // shared list of errror codes

#include "common.h"     // common objects shared between all API family
                        // functions
#include "genasync.h"   // shared driver commands of general/async family
#include "util.h"


// Constants
// *********

#define TIMEOUT_EVENT   5000            // in ms (i.e. 5 seconds)


// Local Functions
// ***************

// ****************************************************************************
// STATIC WORD SamplingClock()
// ***************************
//
// Input parameters :
// ****************
//
// PCX_HANDLE PmMyProcess: Handle of the calling process.
//
// DWORD PmPipeOutMask: Pipe outputs mask.
//
// DWORD PmPipeInMask: Pipe inputs mask.
//
// DWORD PmFrequencyValue: frequency value
//
// DWORD PmSource: source of the clock
//
// DWORD PmSyncBoardNum: board number on which the clock is taken
//
// DWORD PmSyncInputNum: input number on which the clock is taken
//
// WORD  PmBoardMask   : mask of boards on which the clock is taken
//
// Return value :
// **************
// 0 if no error occured, an error code otherwise.
//
// ****************************************************************************
//
// This function Set no sampling clock
//
// PmSource            | Relevant parameters
// --------------------+--------------------------------------------------------------
// CLOCK_TYPE_NONE            |
// CLOCK_TYPE_INTERNAL     | PmFrequencyValue
// CLOCK_TYPE_UER_SYNCHRO            |                   PmSyncBoardNum  PmSyncInputNum
// CLOCK_TYPE_WORD_CLOCK          |
// CLOCK_TYPE_PROGRAMMABLE_CLOCK  | PmFrequencyValue                                  PmBoardMask
//
// ****************************************************************************

STATIC WORD SamplingClock(
    IN PCX_HANDLE   PmMyProcess,
    IN DWORD        PmPipeOutMask,
    IN DWORD        PmPipeInMask,
    IN DWORD        PmFrequencyValue,
    IN BYTE         PmSource,
    IN DWORD        PmSyncBoardNum,
    IN DWORD        PmSyncInputNum,
    IN WORD         PmBoardMask,
	IN BYTE         PmClockFormat,
	IN DWORD        PmValidationMask)
{
    LPSAMPLE_CLK_REQ_INFO   LcPReq          = 0;
    LPRESP_HEADER_INFO      LcPResp         = 0;
    WORD                    LcRetValue      ;

    // Take Exmut
    // ----------
    if ( !COMWaitExmut() ) {
        return(EA_CANT_TAKE_MUTEX) ;
    }

    // Fill the header of the request block
    //
    COMFillBcHeader(
        PIPE_CTRL_FAM,
        SAMPLE_CLOCK_CMD,
        (APP_HANDLE) PmMyProcess,
        sizeof( SAMPLE_CLK_REQ_INFO ),
        sizeof( RESP_HEADER_INFO ),
        (BC_HEADER_HANDLE) &LcPReq );

    // Fill the other fields of the request block
    //
    LcPReq->scqOutputMask64.QuadPart = PmPipeOutMask;
    LcPReq->scqInputMask64.QuadPart  = PmPipeInMask;
    LcPReq->scqSource           = PmSource;

    LcPReq->scqClockBoardNum    = PmSyncBoardNum;
    LcPReq->scqFrequency        = PmFrequencyValue;
    LcPReq->scqSyncInputNum     = PmSyncInputNum;
    LcPReq->scqClockBoardMask   = PmBoardMask;
	LcPReq->scqClockFormat      = PmClockFormat;
	LcPReq->scqValidationMask   = PmValidationMask;

    // Send the request to the driver
    //
    COMCallDriver(
        (LPBC_HEADER_INFO) LcPReq,
        0,
        &LcPResp,
        0);

    LcRetValue = LcPResp->rhCptr ;

    COMReleaseExmut();

    // Return the error code
    //
    return(LcRetValue);
}


// Exported functions
// ******************

// ****************************************************************************
// WORD PCXSetPipeNoSamplingClock()
// ***************************
//
// Input parameters :
// ****************
//
// PCX_HANDLE PmMyProcess: Handle of the calling process.
//
// DWORD PmPipeOutMask: Pipe output mask.
//
// DWORD PmPipeInMask: Pipe input mask.
//
// PTIME_INFO PmDelayCondition: Points to a PTIME_INFO structure.
//
//
// Return value :
// **************
// 0 if no error occured, an error code otherwise.
//
// ****************************************************************************
//
// This function Unset the sampling clock
//
// ****************************************************************************

WORD _CDECL_ PCXSetPipeNoSamplingClock(
    IN PCX_HANDLE PmMyProcess,
    IN DWORD PmPipeOutMask,
    IN DWORD PmPipeInMask,
    IN PTIME_INFO PmDelayCondition )
{
    WORD LcValret;

    if( (PmMyProcess == 0) || (PmMyProcess > MAX_PCX_HANDLE) )
        return EA_INVALID_PCX_HANDLE;

    LcValret = SamplingClock( PmMyProcess,
                          PmPipeOutMask,
                          PmPipeInMask,
                          0,
                          CLOCK_TYPE_NONE,
                          0,
                          0,
                          0,
						  0,
						  0);
    return LcValret;
}


// ****************************************************************************
// WORD PCXSetPipeInternalClock()
// ***************************
//
// Input parameters :
// ****************
//
// PCX_HANDLE PmMyProcess: Handle of the calling process.
//
// DWORD PmPipeOutMask: Pipe outputs mask.
//
// DWORD PmPipeInMask: Pipe inputs mask.
//
// PTIME_INFO PmDelayCondition: points to a TIME_INFO structure.
//
// PINTERNAL_CLOCK_INFO PmInternalClockInfo: points to a INTERNAL_CLOCK_INFO
//                                           structure.
//
// Return value :
// **************
// 0 if no error occured, an error code otherwise.
//
// ****************************************************************************
//
// This function is used to select an internal sampling clock for
// audio inputs and outputs, to set them synchronized or
// independent.
//
// ****************************************************************************

WORD _CDECL_ PCXSetPipeInternalClock(
    IN PCX_HANDLE PmMyProcess,
    IN DWORD PmPipeOutMask,
    IN DWORD PmPipeInMask,
    IN PTIME_INFO PmDelayCondition,
    IN PINTERNAL_CLOCK_INFO PmInternalClockInfo )
{
    WORD LcValret;

    if( (PmMyProcess == 0) || (PmMyProcess > MAX_PCX_HANDLE) )
        return EA_INVALID_PCX_HANDLE;

    if( PmInternalClockInfo->icClock == 0 )
		return EA_BAD_FREQUENCY_VALUE;

    LcValret = SamplingClock( PmMyProcess,
                          PmPipeOutMask,
                          PmPipeInMask,
                          PmInternalClockInfo->icClock,
                          CLOCK_TYPE_INTERNAL,
                          0,
                          0,
                          0,
						  0,
						  PmInternalClockInfo->icValidationMask1);
    return LcValret;
}


// ****************************************************************************
// WORD PCXSetPipeExternalClock()
// ***************************
//
// Input parameters :
// ****************
//
// PCX_HANDLE PmMyProcess: Handle of the calling process.
//
// DWORD PmPipeOutMask: Pipe outputs mask.
//
// DWORD PmPipeInMask: Pipe inputs mask.
//
// PTIME_INFO PmDelayCondition: Points to a TIME_INFO structure.
//
// PEXTERNAL_CLOCK_INFO PmExternalClockInfo: External Clock description.
//
//
// Return value :
// **************
// 0 if no error occured, an error code otherwise.
//
// ****************************************************************************
//
// ****************************************************************************
WORD _CDECL_ PCXSetPipeExternalClock(
    IN PCX_HANDLE PmMyProcess,
    IN DWORD PmPipeOutMask,
    IN DWORD PmPipeInMask,
    IN PTIME_INFO PmDelayCondition,
    IN PEXTERNAL_CLOCK_INFO PmExternalClockInfo )
{
    WORD LcValret;

    if( (PmMyProcess == 0) || (PmMyProcess > MAX_PCX_HANDLE) )
        return EA_INVALID_PCX_HANDLE;

    switch( PmExternalClockInfo->ecType )
    {
    case CLOCK_TYPE_ETHERSOUND:
        LcValret = SamplingClock( PmMyProcess,
                              PmPipeOutMask,
                              PmPipeInMask,
                              0,
                              CLOCK_TYPE_ETHERSOUND,
                              (PmExternalClockInfo->ecInfo).tEthersound.ecCardNumber,
                              0,
                              0,
							  0,
                              (PmExternalClockInfo->ecInfo).tEthersound.ecValidationMask);
        break;

    case CLOCK_TYPE_WORD_CLOCK:

		if( PmExternalClockInfo->ecInfo.tWCl.wcWordClockInputNumber != 0 )
			return EA_BAD_SYNC_SOURCE;

        LcValret = SamplingClock( PmMyProcess,
                              PmPipeOutMask,
                              PmPipeInMask,
                              0,
                              CLOCK_TYPE_WORD_CLOCK,
                              (PmExternalClockInfo->ecInfo).tWCl.wcCardNumber,
                              0,
                              0,
							  0,
							  (PmExternalClockInfo->ecInfo).tWCl.wcValidationMask1 );
        break;

    default:
        LcValret = EA_BAD_SYNC_SOURCE ;
    }

    return LcValret;
}


// ****************************************************************************
// WORD PCXSetPipeProgrammableClock()
// ***************************
//
// Input parameters :
// ****************
//
// PCX_HANDLE PmMyProcess: Handle of the calling process.
//
// DWORD PmPipeOutMask: Pipe outputs mask.
//
// DWORD PmPipeInMask: Pipe inputs mask.
//
// PTIME_INFO PmDelayCondition: Points to a TIME_INFO structure.
//
// PPROGRAMMABLE_CLOCK_INFO PmProgrammableClockInfo: Programmable Clock description.
//
//
// Return value :
// **************
// 0 if no error occured, an error code otherwise.
//
// ****************************************************************************
//
// ****************************************************************************
WORD _CDECL_ PCXSetPipeProgrammableClock(
    IN PCX_HANDLE PmMyProcess,
    IN DWORD PmPipeOutMask,
    IN DWORD PmPipeInMask,
    IN PTIME_INFO PmDelayCondition,
    IN PPROGRAMMABLE_CLOCK_INFO PmProgrammableClockInfo )
{
    WORD LcValret;

    if( (PmMyProcess == 0) || (PmMyProcess > MAX_PCX_HANDLE) )
        return EA_INVALID_PCX_HANDLE;

    if( PmProgrammableClockInfo->pcClock == 0 )
		return EA_BAD_FREQUENCY_VALUE;

    LcValret = SamplingClock( PmMyProcess,
                              PmPipeOutMask,
                              PmPipeInMask,
                              PmProgrammableClockInfo->pcClock,
                              CLOCK_TYPE_PROGRAMMABLE_CLOCK,
                              0,
                              0,
                              PmProgrammableClockInfo->pcBoardMask,
							  0,
							  0 );
    return LcValret;
}

