// *************************************************************************
//
//  COPYRIGHT 1996-2000 DIGIGRAM. ALL RIGHTS RESERVED.
//
//  DIGIGRAM
//
// **************************************************************************

// Include files
// *************

#include "shared.h"
#include "lxeswdm.h"
#include "xx_protocol.h"


extern	BOARD_INFO		APH_Board_Info_Array[];
extern	CProtocol*	    APH_Protocol_Array[];
extern	CPIOCommands*	APH_Commands_Array[];

static LIST_ENTRY  EventQueueHead; // Queue where all the user notification requests are queued
static KSPIN_LOCK  QueueLock;


static PMDL    ES_Mdl_Array[MAX_BOARD];
static PVOID   ES_UVA_Array[MAX_BOARD];

typedef struct _ES_NOTIFY_RECORD
{
    LONG            CommandState;   // use enum ES_COMMAND_STATE
    PIRP            PendingIrp;
    LIST_ENTRY      ListEntry;
    KDPC            Dpc;
    LONG            BoardNum;
    DWORD           ReadResp[4];    // response data read in the interrupt

} ES_NOTIFY_RECORD, *PES_NOTIFY_RECORD;


// values used for ES_NOTIFY_RECORD.CommandState
typedef enum _ES_COMMAND_STATE
{
    ES_cmd_free         = 0,    // no command executing
    ES_cmd_processing   = 1,    // execution of a read/write command
    ES_read_pending     = 2,    // a asynchron read command is pending
    ES_read_finishing   = 3,    // a read command has finished waiting (set by Interrupt or CancelIrp)
} ES_COMMAND_STATE;


static ES_NOTIFY_RECORD ES_Async_Array[MAX_BOARD];


static VOID ESReadDPC(
    IN PKDPC Dpc,
    IN PVOID DeferredContext,
    IN PVOID SystemArgument1,
    IN PVOID SystemArgument2
    );

#pragma code_seg("PAGE")

VOID ESCmdsInit( PDSOUND_DEVICE_EXTENSION LcDevExt )
{
    PAGED_CODE();

    (void) LcDevExt;

    InitializeListHead( &EventQueueHead );
    KeInitializeSpinLock( &QueueLock );

    RtlZeroMemory(ES_Mdl_Array, sizeof(ES_Mdl_Array));
    RtlZeroMemory(ES_UVA_Array, sizeof(ES_UVA_Array));
    RtlZeroMemory(ES_Async_Array, sizeof(ES_Async_Array));

    for(LONG i=0; i<MAX_BOARD; i++)
    {
        PES_NOTIFY_RECORD notifyRecord = &ES_Async_Array[i];

        notifyRecord->BoardNum = i;

        InitializeListHead( &notifyRecord->ListEntry ); //not required for listentry but it's a good practice.

        KeInitializeDpc( &notifyRecord->Dpc,            // Dpc
                                    ESReadDPC,          // DeferredRoutine
                                    notifyRecord        // DeferredContext
                                    );
    }
}

VOID ESCmdsUnload( PDSOUND_DEVICE_EXTENSION LcDevExt )
{
    PAGED_CODE();

    (void) LcDevExt;

    if(!IsListEmpty(&EventQueueHead))
    {
        DOUT(DBG_ERROR, ("Event Queue is not empty\n"));
    }

    /*
    for(LONG i=0; i<MAX_BOARD; i++)
    {
        PES_NOTIFY_RECORD notifyRecord = &ES_Async_Array[i];

        KeRemoveQueueDpc( &notifyRecord->Dpc ) ;
    }
    */
}


VOID ESFreeUserMappedMemory(int board)
{
    PAGED_CODE();

    PMDL    mdl = ES_Mdl_Array[board];
    PVOID   userVAToReturn = ES_UVA_Array[board];
    if((mdl != NULL) || (userVAToReturn != NULL))
    {
        MmUnmapLockedPages(userVAToReturn, mdl);
        IoFreeMdl(mdl);
        ES_Mdl_Array[board] = NULL;
        ES_UVA_Array[board] = NULL;
        DOUT(DBG_PRINT, ("ESFreeUserMappedMemory : UserVA = 0x%0x\n", userVAToReturn));
    }
}

PVOID ESMapMemory2UserMode(PVOID buffer, int board)
{
    PAGED_CODE();

    PMDL  mdl;
    PVOID userVAToReturn;

    if((!buffer) || (board >= MAX_BOARD))  {
        return(NULL);
    }

    // free old mdl !
    //
    ESFreeUserMappedMemory(board);

    // Allocate and initalize an MDL that describes the buffer
    //
    mdl = IoAllocateMdl(buffer,
                        sizeof(ULONG),
                        FALSE,
                        FALSE,
                        NULL);
    if(!mdl)    {
        return(NULL);
    }
    // Finish building the MDL -- Fill in the "page portion"
    //
    MmBuildMdlForNonPagedPool(mdl);

    // The preferred V5 way to map the buffer into user space
    //
    userVAToReturn =
         MmMapLockedPagesSpecifyCache(mdl,          // MDL
                                      UserMode,     // Mode
                                      MmCached,     // Caching
                                      NULL,         // Address
                                      FALSE,        // Bugcheck?
                                      NormalPagePriority); // Priority

    // If we get NULL back, the request didn't work.
    // I'm thinkin' that's better than a bug check anyday.
    //
    if(!userVAToReturn) {
        IoFreeMdl(mdl);
        return(NULL);
    }

    // Store away both the mapped VA and the MDL address, so that
    // later we can call MmUnmapLockedPages(StoredPointer, StoredMdl)
    //
    ES_UVA_Array[board] = userVAToReturn;
    ES_Mdl_Array[board] = mdl;

    DOUT(DBG_PRINT, ("ESMapMemory2UserMode(%d) : UserVA = 0x%p\n", board, userVAToReturn));

    return(userVAToReturn);
}


// TODO - IRP_MJ_CLEANUP


#pragma code_seg()


static VOID ESCancelRoutine(
    IN PDEVICE_OBJECT   DeviceObject,
    IN PIRP             Irp
    )
{
    PDSOUND_DEVICE_EXTENSION    deviceExtension = (PDSOUND_DEVICE_EXTENSION)DeviceObject->DeviceExtension;
    KIRQL                       oldIrql ;
    BOOLEAN                     bCancelTheIrp = FALSE;
    PES_NOTIFY_RECORD           notifyRecord;

    DOUT (DBG_WARNING, ("==>EventCancelRoutine %p\n", Irp));

    // Release the cancel spinlock
    //
    IoReleaseCancelSpinLock( Irp->CancelIrql);

    // Acquire the queue spinlock 
    //
    KeAcquireSpinLock(&QueueLock, &oldIrql);

    notifyRecord = (PES_NOTIFY_RECORD)InterlockedExchangePointer(&Irp->Tail.Overlay.DriverContext[3], NULL);

    if( NULL == notifyRecord )
    {
        // DPC or Cleanup routine is in the process of completing this IRP.
        // So don't touch it.
        DOUT (DBG_WARNING, ("ESCancelRoutine notifyRecord == NULL !!!!\n"));
    }
    else
    {
        LONG oldstate = InterlockedExchange(&notifyRecord->CommandState, ES_read_finishing);

        // Remove the entry from the list and cancel the IRP only
        // if the command was pending. Otherwise the Dpc will take care.
        //
        ASSERT(oldstate == ES_read_pending);
        ASSERT(notifyRecord->PendingIrp == Irp);

        RemoveEntryList(&notifyRecord->ListEntry);                      

        bCancelTheIrp = TRUE;
        notifyRecord->PendingIrp = NULL;
    }

    KeReleaseSpinLock(&QueueLock, oldIrql);

    if( bCancelTheIrp )
    {
        // free entry
        //
        InterlockedExchange( &notifyRecord->CommandState, ES_cmd_free);

        DOUT (DBG_WARNING, ("\t canceled IRP %p\n", Irp));
        Irp->IoStatus.Status = STATUS_CANCELLED;
        Irp->IoStatus.Information = 0;
        IoCompleteRequest (Irp, IO_NO_INCREMENT);
    }

    DOUT (DBG_WARNING, ("<==EventCancelRoutine %p\n", Irp));
    return;
}


VOID ESCMDSInterruptService(LONG boardnum, BOOL purgedata)
{
    ASSERT(boardnum < MAX_BOARD);
    PES_NOTIFY_RECORD notifyRecord = &ES_Async_Array[boardnum];

    // read the interrupt response data (ack interrupt in all cases !)
    APH_Commands_Array[boardnum]->PIOReadCRES(notifyRecord->ReadResp, 4 );

    if( purgedata )
    {
        DOUT(DBG_WARNING, ("ESCMDSInterruptService : PURGE response data\n"));
        RtlZeroMemory(notifyRecord->ReadResp, sizeof(notifyRecord->ReadResp));
    }

    LONG oldstate = InterlockedCompareExchange(&notifyRecord->CommandState, ES_read_finishing, ES_cmd_processing);

    if( oldstate == ES_cmd_processing )
    {
        // sync command will finish now (new state = ES_read_finishing)
        return;
    }
    if( oldstate == ES_read_pending )
    {
        // fire DPC
        KeInsertQueueDpc(
            &notifyRecord->Dpc,
            (PVOID) boardnum,   //SystemArgument1
            NULL
            );

        return;
    }
    else
    {
        DOUT(DBG_WARNING, ("ESCMDSInterruptService : nothing to do\n"));
    }
}


static VOID ESReadDPC(
    IN PKDPC Dpc,
    IN PVOID DeferredContext,
    IN PVOID SystemArgument1,
    IN PVOID SystemArgument2
    )
{
    PES_NOTIFY_RECORD      notifyRecord = (PES_NOTIFY_RECORD)DeferredContext;

    DOUT(DBG_PRINT, ("==> ESReadDPC \n"));

    ASSERT(notifyRecord); // can't be NULL

    KeAcquireSpinLockAtDpcLevel(&QueueLock);

    PIRP PendingIrp = notifyRecord->PendingIrp;

    if( PendingIrp )
    {
        RemoveEntryList(&notifyRecord->ListEntry);           

        notifyRecord->PendingIrp = NULL;
        PendingIrp->Tail.Overlay.DriverContext[3] = NULL;

        // Even if the cancel routine is called at this point, it wouldn't
        // complete the IRP becuase we have cleared the DriverContext[3]
        //
        IoSetCancelRoutine (PendingIrp, NULL);
    }

    KeReleaseSpinLockFromDpcLevel(&QueueLock);

    if( PendingIrp == NULL )
    {
        // Cancel routine has run and completed the IRP
        // So don't touch it.
        DOUT (DBG_WARNING, ("ESReadDPC PendingIrp == NULL !!!!\n"));
        return;
    }

    LONG oldstate = InterlockedCompareExchange(&notifyRecord->CommandState, ES_read_finishing, ES_read_pending);

    ASSERT(oldstate == ES_read_pending);

        // fill in response data
        //
        LONG boardnum = notifyRecord->BoardNum;

        ASSERT(boardnum == (LONG)SystemArgument1);

        PES_DRV_RESP response = (PES_DRV_RESP)PendingIrp->AssociatedIrp.SystemBuffer;

        RtlZeroMemory(response, sizeof(*response));
        //response->ulStatus = SUCCESS;

        // copy the contents of the 4 registers read in the interrupt
        MEMCPY(&response->ulResp1, notifyRecord->ReadResp, sizeof(notifyRecord->ReadResp));

        // free entry
        //
        InterlockedExchange( &notifyRecord->CommandState, ES_cmd_free);

        // complete the Irp
        //
        PendingIrp->IoStatus.Status = STATUS_SUCCESS;
        PendingIrp->IoStatus.Information = sizeof(ES_DRV_RESP);
        IoCompleteRequest( PendingIrp, IO_NETWORK_INCREMENT );

    DOUT(DBG_PRINT, ("<== ESReadDPC\n"));

    return;
}


static NTSTATUS ESReadWriteDiffered(
    IN PIRP Irp,
    IN PDSOUND_DEVICE_EXTENSION deviceExtension,
    IN PES_DRV_CMD command,
    IN PES_DRV_RESP response
    )
{
    DOUT(DBG_PRINT, ("ESReadWriteDiffered\n"));

    ULONG boardnum = command->ulParam1;   // already checked valid
    ULONG timeout  = command->ulParam2;

    PES_NOTIFY_RECORD notifyRecord = &ES_Async_Array[boardnum];

    BOOL bBroadcast = ((command->ulParam5 & 0x0001) != 0);
    BOOL bAsyncCmd;

    if( (command->ulCmd == ES_CMD_ID_WRITE_ES) || (timeout != 0) )
    {
            bAsyncCmd = FALSE;  // sync commands
    }
    else    bAsyncCmd = TRUE;   // async command, timeout == 0

    if( ES_cmd_free != InterlockedCompareExchange(
                    &notifyRecord->CommandState,
                    ES_cmd_processing,
                    ES_cmd_free) )
    {
        // for instance refuse parallel access (otherwise spinlock required)
        //
        return  STATUS_DEVICE_BUSY;
    }

    // Xilinx ready ?
    //
    if( APH_Commands_Array[boardnum]->PIOCheckESPipeline() == FALSE )
    {
        InterlockedExchange( &notifyRecord->CommandState, ES_cmd_free);
        return  STATUS_DEVICE_BUSY;
    }

    // async command : prepare DPC
    // attention : the DPC could take place directly after the PIOWriteCRES !
    if( bAsyncCmd )
    {
        KIRQL   oldIrql;

        IoMarkIrpPending(Irp);

        notifyRecord->PendingIrp = Irp;

        // ready for DPC
        //
        InterlockedExchange( &notifyRecord->CommandState, ES_read_pending);

        // We will set the cancel routine and TimerDpc within the
        // lock so that they don't modify the list before we are
        // completely done.
        //
        KeAcquireSpinLock(&QueueLock, &oldIrql);

        // Set the cancel routine. This is required if the app decides to
        // exit or cancel the event prematurely.
        //
        IoSetCancelRoutine (Irp, ESCancelRoutine);

        InsertTailList( &EventQueueHead, 
                        &notifyRecord->ListEntry);

        // We will save the record pointer in the IRP so that we can get to it directly in the
        // CancelRoutine. (Nar, this eliminates the need to match IRP pointers)
        //
        Irp->Tail.Overlay.DriverContext[3] = notifyRecord;

        KeReleaseSpinLock(&QueueLock, oldIrql);
    }

    // write the command
    //                                                   CRESMsb            CRESLsb
    APH_Commands_Array[boardnum]->PIOWriteCRES( command->ulParam3, command->ulParam4, bBroadcast );

    if( bAsyncCmd )
    {
        // Async command must return STATUS_PENDING.
        //
        return STATUS_PENDING;
    }

    // a write command is finished here
    if( command->ulCmd == ES_CMD_ID_WRITE_ES )
    {
        InterlockedExchange( &notifyRecord->CommandState, ES_cmd_free);

        RtlZeroMemory(response, sizeof(*response));
        // response->ulStatus = SUCCESS;

        return  STATUS_SUCCESS;
    }
    ASSERT( command->ulCmd == ES_CMD_ID_READ_ES );

    // if synchron command, timeout != 0
    // wait for interrupt and return (no DPC)
    //
    RtlZeroMemory(response, sizeof(*response));
    response->ulStatus = ED_DSP_TIMED_OUT;

    for( ULONG i=0; i<timeout; i++ )
    {
        // wait for ES_read_finishing and switch back to ES_cmd_processing !
        //
        LONG oldstate = InterlockedCompareExchange(&notifyRecord->CommandState, ES_cmd_processing, ES_read_finishing);
        if( oldstate == ES_read_finishing )
        {
            // copy the contents of the 4 registers read in the interrupt
            MEMCPY(&response->ulResp1, notifyRecord->ReadResp, sizeof(notifyRecord->ReadResp));
            response->ulStatus = SUCCESS;

            // TEST
            PULONG test = (PULONG)&response->ulString[0];
            *test = i;  // save polling iterations
            // END TEST

            DOUT(DBG_PRINT, ("ESReadWriteDiffered READ SYNC polling = %d\n", i));

            break;
        }
        PCX_WAIT_AT_LEAST_MICRO_SEC(1);
    }

    // free entry
    //
    InterlockedExchange( &notifyRecord->CommandState, ES_cmd_free);
    return STATUS_SUCCESS;
}

#pragma code_seg("PAGE")


// *************************************************************************
//
// NTSTATUS APHDispatchES( INOUT LPDWORD, INOUT LPDWORD, IN PDWORD )
//
// *************************************************************************

static  ULONG   ESMapCSUF( PDSOUND_DEVICE_EXTENSION pDevExt, DWORD CurrentBoard, PULONG_PTR mapped_ptr );
static  ULONG   ESResetBoard( PDSOUND_DEVICE_EXTENSION PDevExt, DWORD CurrentBoard ) ;


NTSTATUS ESCmdsDispatch( IN PIRP PmIrp, IN PDSOUND_DEVICE_EXTENSION PmDevExt)
{
    PAGED_CODE();

    PIRP irp = (PIRP)PmIrp;

    // attention : buffer entre/sortie sont les mmes !
    ES_DRV_CMD  *LcCMD = (ES_DRV_CMD *)irp->AssociatedIrp.SystemBuffer;
    ES_DRV_RESP *LcRESP = (ES_DRV_RESP *)LcCMD;

    DWORD LcCurrentBoard = LcCMD->ulParam1;
    NTSTATUS status = STATUS_SUCCESS;

    if( (LcCurrentBoard >= MAX_BOARD) || (APH_Board_Info_Array[LcCurrentBoard].biCarac != CARAC_USED) )
    {
        DOUT (DBG_WARNING, ("ESCmdsDispatch : ED_UNKNOWN_BOARD !\n"));
        LcRESP->ulStatus = ED_UNKNOWN_BOARD;
    }
    else
    {
        DWORD LcCommand = LcCMD->ulCmd;
        DWORD LcParam2 = LcCMD->ulParam2;

        switch(LcCommand)
        {
        case ES_CMD_ID_INFO:        // VERSION of available board
            LcRESP->ulResp1 = APH_Board_Info_Array[LcCurrentBoard].biType;
            LcRESP->ulResp2 = APH_Board_Info_Array[LcCurrentBoard].biTypeNoAlias;
            LcRESP->ulResp3 = APH_Board_Info_Array[LcCurrentBoard].biTbDspInfo[0].dsDspVersion;
            LcRESP->ulResp4 = ((PCX_VERSION_NUMBER << 16) | (PCX_VERSION_RELEASE << 8) |  PCX_BUILD_NUMBER);	// version du driver, dans le meme format que l'embarqu

            if ( 0 == (LcParam2 & LXES_CMD_INFO_GET_REAL_FW_DRV_VERSION) )
            {
                // fool ES driver : get only min version of both : driver and firmware version
                //
                if (LcRESP->ulResp3 > LcRESP->ulResp4)
                {
                    LcRESP->ulResp3 = LcRESP->ulResp4;
                }
                else
                {
                    LcRESP->ulResp4 = LcRESP->ulResp3;
                }
            }

            LcRESP->ulResp5=  ES_DRV_IF_VERSION;	// version de l'interface ES_CMD

			// on remonte la conf ES.
			APH_Commands_Array[LcCurrentBoard]->PIOGetESConfig( &(LcRESP->ulResp6)); 

            STRNCPY(LcRESP->ulString, APH_Board_Info_Array[LcCurrentBoard].biBoardName, sizeof(LcRESP->ulString));
            LcRESP->ulStatus = SUCCESS;
            break;
        case ES_CMD_ID_WRITE_ES:    // Commande Ecriture ES (Write)
        case ES_CMD_ID_READ_ES:     // Commande Lecture ES (Read synchrone et asynchrone)

            status = ESReadWriteDiffered( PmIrp, PmDevExt, LcCMD, LcRESP );

            break;
        case ES_CMD_ID_MAP_CSUF:    // get adresse virtuelle du registre CSUF

            LcRESP->ulStatus = ESMapCSUF( PmDevExt, LcCurrentBoard,  &(LcRESP->ulResp1));

			break;			


        case ES_CMD_ID_FREE_CSUF:   // liberer l'adresse virtuelle (MDL) du registre CSUF

			APH_Commands_Array[LcCurrentBoard]->PIOEnableSendMessage( TRUE );

            ESFreeUserMappedMemory(LcCurrentBoard);
            LcRESP->ulStatus = SUCCESS;
            break;

		case ES_CMD_ID_RESET_BOARD:  // reseter la carte !

            LcRESP->ulStatus = ESResetBoard( PmDevExt, LcCurrentBoard);

            break;
        default:
            status = STATUS_NOT_IMPLEMENTED;
        }
    }
    return status;
} // ESCmdsDispatch


/**
*		map l'adresse du CSUF dans l'espace user, et protge la carte des acces message et uBlaze
*/
ULONG	ESMapCSUF( PDSOUND_DEVICE_EXTENSION pDevExt, DWORD CurrentBoard, PULONG_PTR mapped_ptr )
{
            PAGED_CODE();

			KIRQL   LcOldIrql;
			DWORD   l_ESConfig = 0;
			ULONG   l_ret = SUCCESS;

            // map 
			//
			*mapped_ptr =
				(ULONG_PTR) ESMapMemory2UserMode(
                APH_Commands_Array[CurrentBoard]->PIOGetCSUFAddr(),
                CurrentBoard);

			// get boardlock ( pas avant le map -> bugcheck C4 ) 
			//
			KeAcquireSpinLock( pDevExt->dsdeSpinLockPtr, &LcOldIrql );
			
			// park microblaze, cancel+disable PCI DMA actions
			//
			APH_Protocol_Array[CurrentBoard]->IDiag_HaltEmbedded();

			// disable PIOSendMessage
			//
			APH_Commands_Array[CurrentBoard]->PIOEnableSendMessage( FALSE );

			// Release BoardLock
			//
            KeReleaseSpinLock( pDevExt->dsdeSpinLockPtr, LcOldIrql );
			
			return l_ret;

}; // ESMapCSUF

/**
*		reset de la carte, protg. 
*/
ULONG	ESResetBoard( PDSOUND_DEVICE_EXTENSION pDevExt, DWORD CurrentBoard )
{
            PAGED_CODE();

			KIRQL   LcOldIrql;
			DWORD   LcDspVersion;
			DWORD   l_ESConfig = 0;
			ULONG   l_ret = SUCCESS;

            KeAcquireSpinLock( pDevExt->dsdeSpinLockPtr, &LcOldIrql );

			APH_Board_Info_Array[CurrentBoard].biTbDspInfo[0].dsEtat = 1 ;	// and "dsp" state

			APH_Commands_Array[CurrentBoard]->PIOEnableSendMessage( FALSE );
            APH_Commands_Array[CurrentBoard]->PIODisableIrq( );

				APH_Commands_Array[CurrentBoard]->PIODefaultConfig( );    // reset Xilinx
			
			APH_Commands_Array[CurrentBoard]->PIOEnableIrq( );
			APH_Commands_Array[CurrentBoard]->PIOEnableSendMessage( TRUE );

			APH_Protocol_Array[CurrentBoard]->IInit_GetVersionAndFeatures( 0, pDevExt->dsdeGeneralInfo, &LcDspVersion );
			APH_Board_Info_Array[CurrentBoard].biTbDspInfo[0].dsDspVersion = LcDspVersion ;

			// apply ES Conf again
			//
            APH_Commands_Array[CurrentBoard]->PIOGetESConfig(&l_ESConfig );
            APH_Commands_Array[CurrentBoard]->PIOUpdateESConfig(l_ESConfig);

			// apply granularity again

            IBL_RESP_INFO	IBL_Info;
							IBL_Info.iblSamples		=  (WORD) pDevExt->dsdeGeneralInfo->PCMOnlyGranularity;
							IBL_Info.iblMin			=  0;
							IBL_Info.iblMax			=  0;
							IBL_Info.iblGranularity	=  0;

			APH_Protocol_Array[CurrentBoard]->IDiag_HandleBoardIBL( &IBL_Info );

            // BOARD UNLOCK
            // ReleaseBoardLock
            KeReleaseSpinLock( pDevExt->dsdeSpinLockPtr, LcOldIrql );

			return l_ret; 

}; // ESResetBoard

#pragma code_seg()


