// *************************************************************************
//
//  COPYRIGHT 1996-2000 DIGIGRAM. ALL RIGHTS RESERVED.
//  Portions Copyright (c) 1998-1999 Intel Corporation
//  Portions Copyright (c) 1998-1999 Microsoft Corporation. 
//
// **************************************************************************

#include "shared.h"
#include "xx_protocol.h"
#include "piocommands.h"

#include "minwave.h"
#include "ichwave.h"

#include "pcxerr_e.h"

#include "if_drv_mb.h"


#pragma code_seg("PAGE")

// These macros should not be used in paged code
//
#undef BOARD_LOCK
#undef BOARD_UNLOCK

#define DMA_USE_TIMER   FALSE

/*****************************************************************************
 * CreateMiniportWaveICHStream
 *****************************************************************************
 * Creates a wave miniport stream object for the ICH audio adapter. This is
 * (nearly) like the macro STD_CREATE_BODY_ from STDUNK.H.
 */
NTSTATUS CreateMiniportWaveICHStream
(
    OUT CMiniportWaveICHStream  **WaveIchStream,
    IN  PUNKNOWN                pUnknownOuter,
    IN  POOL_TYPE               PoolType
)
{
    PAGED_CODE ();

    DOUT (DBG_STREAM, ("[CreateMiniportWaveICHStream]\n"));

    //
    // This is basically like the macro at stdunk with the change that we
    // don't cast to interface unknown but to interface WaveIchStream.
    //
    *WaveIchStream = new (PoolType, PCX_ICHW_NEW_TAG)
                        CMiniportWaveICHStream (pUnknownOuter);
    if (*WaveIchStream)
    {
        (*WaveIchStream)->AddRef ();
        return STATUS_SUCCESS;
    }

    return STATUS_INSUFFICIENT_RESOURCES;
}


/*****************************************************************************
 * CMiniportWaveICHStream::NonDelegatingQueryInterface
 *****************************************************************************
 * Obtains an interface.  This function works just like a COM QueryInterface
 * call and is used if the object is not being aggregated.
 */
STDMETHODIMP_(NTSTATUS) CMiniportWaveICHStream::NonDelegatingQueryInterface
(
    IN  REFIID  Interface,
    OUT PVOID * Object
)
{
    PAGED_CODE ();

    ASSERT (Object);

    DOUT (DBG_STREAM, ("[CMiniportWaveICHStream::NonDelegatingQueryInterface]\n"));

    //
    // Convert for IID_IMiniportWavePciStream
    //
    if (IsEqualGUIDAligned (Interface, IID_IMiniportWavePciStream))
    {
        *Object = (PVOID)(PMINIPORTWAVEPCISTREAM)this;
    }
    //
    // Convert for IID_IServiceSink
    //
    else if (IsEqualGUIDAligned (Interface, IID_IServiceSink))
    {
        *Object = (PVOID)(PSERVICESINK)this;
    }
    //
    // Convert for IID_IDrmAudioStream
    //
    else if (IsEqualGUIDAligned (Interface, IID_IDrmAudioStream))
    {
        *Object = (PVOID)(PDRMAUDIOSTREAM)this;
    }
    //
    // Convert for IID_IUnknown
    //
    else if (IsEqualGUIDAligned (Interface, IID_IUnknown))
    {
        *Object = (PVOID)(PUNKNOWN)(PMINIPORTWAVEPCISTREAM)this;
    }
    else
    {
        *Object = NULL;
        return STATUS_INVALID_PARAMETER;
    }

    ((PUNKNOWN)*Object)->AddRef ();
    return STATUS_SUCCESS;
}


/*****************************************************************************
 * CMiniportWaveICHStream::GetAllocatorFraming
 *****************************************************************************
 * Returns the framing requirements for this device.
 * That is sample size (for one sample) and preferred frame (buffer) size.
 */
STDMETHODIMP_(NTSTATUS) CMiniportWaveICHStream::GetAllocatorFraming
(
    OUT PKSALLOCATOR_FRAMING AllocatorFraming
)
{
    PAGED_CODE ();

    DOUT (DBG_STREAM, ("[CMiniportWaveICHStream::GetAllocatorFraming]\n"));

    //
    // Report the minimum requirements.
    // same as AC97 !
    //

    AllocatorFraming->RequirementsFlags =
        KSALLOCATOR_REQUIREMENTF_SYSTEM_MEMORY |
        KSALLOCATOR_REQUIREMENTF_PREFERENCES_ONLY;

	AllocatorFraming->Frames = 8;
	AllocatorFraming->FrameSize = m_SampleSize * m_DataFormat->WaveFormatEx.nSamplesPerSec / 100;

    AllocatorFraming->FileAlignment = FILE_LONG_ALIGNMENT;
    AllocatorFraming->PoolType = NonPagedPool;

    DOUT (DBG_PRINT,  ("-->AllocatorFraming->FrameSize = %d\n", AllocatorFraming->FrameSize));

    return STATUS_SUCCESS;
}


/*****************************************************************************
 * Non paged code begins here
 *****************************************************************************
 */

#pragma code_seg()

// These macros are only defined below
//
#define BOARD_LOCK()\
    KIRQL   LcOldBoardIrql; \
    \
    DOUT (DBG_SYSINFO, ("[Wave BOARD_LOCK]"));\
    m_Wave->m_AdapterCommon->AcquireBoardLock(&LcOldBoardIrql);\
    DOUT (DBG_SYSINFO, ("[Wave BOARD_LOCKED]"))

#define BOARD_LOCK_AGAIN()\
    DOUT (DBG_SYSINFO, ("[Wave BOARD_LOCK]"));\
    m_Wave->m_AdapterCommon->AcquireBoardLock(&LcOldBoardIrql);\
    DOUT (DBG_SYSINFO, ("[Wave BOARD_LOCKED]"))

#define BOARD_UNLOCK()\
    DOUT (DBG_SYSINFO, ("[Wave BOARD_UNLOCK]"));\
    m_Wave->m_AdapterCommon->ReleaseBoardLock(&LcOldBoardIrql)

#define BOARD_LOCK_DPC()\
    DOUT (DBG_SYSINFO, ("[BOARD_LOCK_DPC]"));\
    m_Wave->m_AdapterCommon->AcquireBoardLockAtDpcLevel();\
    DOUT (DBG_SYSINFO, ("[BOARD_LOCKED_DPC]"))

#define BOARD_UNLOCK_DPC()\
    DOUT (DBG_SYSINFO, ("[BOARD_UNLOCK_DPC]"));\
    m_Wave->m_AdapterCommon->ReleaseBoardLockAtDpcLevel()


/*****************************************************************************
 * CMiniportWaveICHStream::PowerChangeNotify
 *****************************************************************************
 * This functions saves and maintains the stream state through power changes.
 */
NTSTATUS CMiniportWaveICHStream::PowerChangeNotify
(
    IN  POWER_STATE NewState
)
{
    KIRQL   OldIrql;
    DOUT (DBG_PRINT, ("[CMiniportWaveICHStream::PowerChangeNotify]\n"));

    //
    // Check to see if this is the current power state.
    //
    if (NewState.DeviceState == m_PowerState)
    {
        DOUT (DBG_POWER, ("New device state equals old state\n"));
        return STATUS_SUCCESS;
    }

    DOUT (DBG_POWER, ("Changing state to D%d\n",
                     (ULONG)NewState.DeviceState - (ULONG)PowerDeviceD0));

    switch (NewState.DeviceState)
    {
    case PowerDeviceD0:
        // If we are coming from D3 or D2 we have to restore the registers cause
        // there might have been a power loss.
        //
        if ( (m_PowerState == PowerDeviceD3) || (m_PowerState == PowerDeviceD2) )
        {
            NTSTATUS ntStatus = PrepareHardware( FALSE );

            if (!NT_SUCCESS (ntStatus))
                return STATUS_UNSUCCESSFUL;

	        KeAcquireSpinLock (&m_MapLock, &OldIrql);

	        // do not re-enter in this fct
	        if( m_InGetNewMap )
	        {
		        DOUT (DBG_ERROR, ("DO NOT ENTER PowerChangeNotify when GetNewMappings (chan=%d capt=%d)\n", m_ChannelNumber, m_Capture));
		        KeReleaseSpinLock (&m_MapLock,OldIrql);
		        return STATUS_UNSUCCESSFUL;
	        }
	        m_InGetNewMap = TRUE;
	        // do not re-enter in this fct
	        if( m_InReleaseMap )
	        {
		        DOUT (DBG_ERROR, ("DO NOT ENTER PowerChangeNotify when ReleaseUsedMappings (chan=%d capt=%d)\n", m_ChannelNumber, m_Capture));
		        KeReleaseSpinLock (&m_MapLock,OldIrql);
                m_InGetNewMap = FALSE;
		        return STATUS_UNSUCCESSFUL;
	        }
	        m_InReleaseMap = TRUE;

            LONG LcUpdateMappingIdx = m_ReqInfoReleaseIndex;

            while(LcUpdateMappingIdx != m_ReqInfoGetNewIndex)
            {
		        PREQUEST_INFO LcReqInfoPtr = &m_RequestInfo[LcUpdateMappingIdx];
		        ASSERT( LcReqInfoPtr->Mapping.QuadPart != 0 );

		        DOUT (DBG_PRINT, ("WAKE UP MAPPING %d  (chan=%d capt=%d Idx=%d p@=%p size=%d)\n", LcReqInfoPtr->Tag, m_ChannelNumber, m_Capture, LcUpdateMappingIdx,
			        LcReqInfoPtr->Mapping.LowPart, LcReqInfoPtr->Size));

                // check sample frame alignement for first mapping
                //
                if( LcUpdateMappingIdx == m_ReqInfoReleaseIndex )
                {
                    ASSERT( m_SampleSize != 0 );

                    ULONG ulSampleRemainder = (ULONG)(m_TotalBytesReleased % m_SampleSize);

                    if( ulSampleRemainder )
                    {
                        ULONG ulSampleOffset = (m_SampleSize - ulSampleRemainder);

                        if( LcReqInfoPtr->Size > ulSampleOffset)
                        {
                            LcReqInfoPtr->Size -= ulSampleOffset;
                            LcReqInfoPtr->Mapping.QuadPart += ulSampleOffset;
                            m_PositionOffset += ulSampleOffset; // later used for position calculation
                            m_TotalBytesReleased += ulSampleOffset;

		                    DOUT (DBG_PRINT, ("############### sample aligned by %d to p@=%p size=%d\n", ulSampleOffset, LcReqInfoPtr->Mapping.LowPart, LcReqInfoPtr->Size));
                        }
                        else
                        {
                            // should not happen
		                    DOUT (DBG_ERROR, ("############### ERROR align mapping\n"));
                        }
                    }
                }

		        WORD LcRet = m_pDsp->vio_StreamGiveBufferLX(    &m_Target,
                                                                LcReqInfoPtr->Size,
                                                                LcReqInfoPtr->Mapping.LowPart,
                                                                LcReqInfoPtr->Mapping.HighPart,
                                                                0,
                                                                ( ! DMA_USE_TIMER ),    // notify buffer if no timer
                                                                0,
                                                                &LcReqInfoPtr->IndexLX);

		        if( LcRet != SUCCESS )
		        {
			        DOUT (DBG_ERROR, ("[PowerChangeNotify] vio_StreamGiveBufferLX ERROR\n"));
			        ntStatus = STATUS_UNSUCCESSFUL;
			        break;
		        }

		        LcUpdateMappingIdx = (LcUpdateMappingIdx + 1) & BDL_MASK;
            }
		    //LcRet = ResetDMA ();

	        m_InGetNewMap = FALSE;
	        m_InReleaseMap = FALSE;

		    KeReleaseSpinLock (&m_MapLock, OldIrql);

            if (!NT_SUCCESS (ntStatus))
                return STATUS_UNSUCCESSFUL;
        }
        break;

    case PowerDeviceD1:
        // Here we do nothing. The device has still enough power to keep all
        // it's register values.
        break;

    case PowerDeviceD2:
    case PowerDeviceD3:
        //
        // If we power down to D2 or D3 we might loose power, so we have to be
        // aware of the DMA engine resetting. In that case a play would start
        // with scatter gather entry 0 (the current index is read only).
        // We just rearrange the scatter gather list (like we do on RevokeMappings)
        // so that the current buffer which is played is at entry 0.

	    KeAcquireSpinLock (&m_MapLock, &OldIrql);

        // Just in case.
        PauseDMA ();

        // after wakeup, the DSP stream position will re-start with 0 !
        // save the offset here !
        m_PositionOffset = m_TotalBytesReleased;

		KeReleaseSpinLock (&m_MapLock, OldIrql);

        // mark the pipe as paused to start it correctly after wakeup with PrepareHardware() !
        //
        m_pDsp->IFlow_PipePause(&m_Target, 1, START_PAUSE_IMMEDIATE );

        break;
    }

    //
    // Save the new state.  This local value is used to determine when to
    // cache property accesses and when to permit the driver from accessing
    // the hardware.
    //
    m_PowerState = NewState.DeviceState;

    DOUT (DBG_POWER, ("ICHWAVE Entering D%d\n",(ULONG)m_PowerState - (ULONG)PowerDeviceD0));

    return STATUS_SUCCESS;
}


/*****************************************************************************
 * CMiniportWaveICHStream::Service
 *****************************************************************************
 * This routine is called by the port driver in response to the interrupt
 * service routine requesting service on the stream's service group.
 * Requesting service on the service group results in a DPC being scheduled
 * that calls this routine when it runs.
 */
STDMETHODIMP_(void) CMiniportWaveICHStream::Service (void)
{
	DOUT (DBG_SYSINFO, ("[CMiniportWaveICHStream::Service]\n"));

	if(m_DMAEngineState != DMA_ENGINE_ON)
		return;

	// release all mappings
	ReleaseUsedMappings ();

	// get new mappings
	GetNewMappings (FALSE);
}


/*****************************************************************************
 * CMiniportWaveICHStream::NormalizePhysicalPosition
 *****************************************************************************
 * Given a physical position based on the actual number of bytes transferred,
 * this function converts the position to a time-based value of 100ns units.
 */
STDMETHODIMP_(NTSTATUS) CMiniportWaveICHStream::NormalizePhysicalPosition
(
    IN OUT PLONGLONG PmPhysicalPosition
)
{
    DOUT (DBG_STREAM, ("NormalizePhysicalPosition\n"));

    //
    // Calculate the time in 100ns steps.
    //
    *PmPhysicalPosition = (_100NS_UNITS_PER_SECOND / m_SampleSize *
                           *PmPhysicalPosition) / m_CurrentRate;

    return STATUS_SUCCESS;
}


/*****************************************************************************
 * CMiniportWaveICHStream::GetPosition
 *****************************************************************************
 * Gets the stream position. This is a byte count of the current position of
 * a stream running on a particular DMA engine.  We must return a sample
 * accurate count or the WaveDrv32 wave drift tests (35.2 & 36.2) will fail.
 *
 * The position is the sum of three parts:
 *     1) The total number of bytes in released buffers
 *     2) The position in the current buffer.
 *     3) The total number of bytes in played but not yet released buffers
 */
STDMETHODIMP_(NTSTATUS) CMiniportWaveICHStream::GetPosition
(
    OUT PULONGLONG   Position
)
{
    ASSERT (Position);

    // only get position from DSP when at Power D0 !!!
    //
    if( m_PowerState != PowerDeviceD0 )
    {
        KIRQL   OldIrql;

        *Position = m_PositionOffset;

        DOUT (DBG_WARNING, ("[CMiniportWaveICHStream::GetPosition] USE TOTAL RELEASED = %I64u\n", *Position));
    }
    else
    {
        CPIOCommands* pPIO;
        pPIO = (CPIOCommands*)m_Wave->m_AdapterCommon->GetIOCommandsPtrAsVoid();

        ASSERT(pPIO);  // get the position from the DSP

        WORD LcRet;
        PCX_TIME  LcStreamPos = 123; // 123 will mean byte_count
	    CProtocol *pDSP;

        // call without BOARD_LOCK();

        pDSP = (CProtocol*)m_Wave->m_AdapterCommon->GetProtocolPtrAsVoid();

        LcRet = pDSP->IFlow_StreamSampleCount(&m_Target, 1, &LcStreamPos);

        // call without BOARD_UNLOCK();

        if(LcRet == SUCCESS)
        {
            // position in bytes = sampleposition * channels * byte/sample

            *Position = m_PositionOffset + LcStreamPos;
        }
        else
            *Position = 0;
    }

    DOUT (DBG_STREAM, ("[GetPosition] = %I64u  (total released = %I64u ) diff = %d  bufsize = %d\n", *Position, m_TotalBytesReleased,
        (LONG)(*Position - m_TotalBytesReleased), m_RequestInfo[m_ReqInfoReleaseIndex].Size));

    return STATUS_SUCCESS;
}


/*****************************************************************************
 * CMiniportWaveICHStream::RevokeMappings
 *****************************************************************************
 * This routine is used by the port to revoke mappings previously delivered
 * to the miniport stream that have not yet been unmapped.  This would
 * typically be called in response to an I/O cancellation request.
 */
STDMETHODIMP_(NTSTATUS) CMiniportWaveICHStream::RevokeMappings
(
    IN  PVOID     FirstTag,
    IN  PVOID     LastTag,
    OUT PULONG    MappingsRevoked
)
{
    ASSERT (MappingsRevoked);

    KIRQL   OldIrql;
    ULONG   ulOldDMAEngineState;
    int     nNumMappings;
    ULONG   nFirst = PtrToUlong( FirstTag );
    ULONG   nLast  = PtrToUlong( LastTag );
	ULONG   LcMappingsRevoked;

    //
    // print information about the scatter gather list.
    //
    DOUT (DBG_PRINT, ("RevokeMappings: First(%u) Last(%u) nextTag(%d)\n", nFirst, nLast, m_TagCounter));

    // Start accessing the mappings.
    //
    KeAcquireSpinLock (&m_MapLock, &OldIrql);

    //
    // Save old DMA engine state.
    //
    ulOldDMAEngineState = m_DMAEngineState;

    //
    // First stop the DMA engine so it won't process the next buffer in the
    // scatter gather list which might be one of the revoked buffers.
    //
    PauseDMA ();

    //
    // We always rearrange the scatter gather list. That means we reset the DMA
    // engine and move the BD list so that the entry where the current index
    // pointed to is located at position 0.
    //
    // ResetDMA ();

    //
    // Calculate the entries between the indizes.
    //
    if (nLast < nFirst)
    {
        nNumMappings = ((nLast + MAX_BDL_ENTRIES) - nFirst) + 1;
    }
    else
    {
        nNumMappings = (nLast - nFirst) + 1;
    }

	LcMappingsRevoked = 0;

    for ( int i = 0 ; i < nNumMappings ; i++ ) {

		PREQUEST_INFO LcPReqInf;
		WORD LcRet;
        int j;

		for ( j=0; j<MAX_BDL_ENTRIES; j++ )
		{
			LcPReqInf = &m_RequestInfo[j];

			if( (LcPReqInf->Tag == nFirst) && (LcPReqInf->Mapping.QuadPart != NULL) )
			{
				DOUT (DBG_PRINT, ( "REVOKE MAPPING %d  (chan=%d capt=%d Idx=%d p@=%p size=%d)\n",nFirst , m_ChannelNumber, m_Capture, j,
					LcPReqInf->Mapping.LowPart, LcPReqInf->Size));

				LcRet = m_pDsp->vio_StreamCancelBuffer(	&m_Target,
														0,
														LcPReqInf->Size,
														LcPReqInf->Mapping.LowPart
														);   // cancel the buffer itself
				LcPReqInf->Mapping.QuadPart = 0;
				LcMappingsRevoked++;

				if( LcRet != SUCCESS )
				{
					DOUT (DBG_ERROR, ("[RevokeMappings] vio_StreamCancelBuffer ERROR\n"));
					// do not reset LcPReqInf->Tag and move m_ReqInfoReleaseIndex, etc.
					// vio_InitSoundTransferNP() will do this later on !
					break;	// j
				}

				LcPReqInf->Tag = 0 ;
				m_ReqInfoPending-- ;

				if(j == m_ReqInfoReleaseIndex)
				{
					// revoked last mapping held :
					m_ReqInfoReleaseIndex = (m_ReqInfoReleaseIndex + 1)  & BDL_MASK ;
					DOUT (DBG_PRINT, ("[RevokeMappings] BLD set ReleaseIdx from %d to %d (GetNewIdx = %d)\n", j, m_ReqInfoReleaseIndex, m_ReqInfoGetNewIndex));
				}
				else
				{
					int x = m_ReqInfoGetNewIndex;
					if( j > m_ReqInfoGetNewIndex ) x += MAX_BDL_ENTRIES;

					// re-arrange our array : fill the whole of the revoked mapping
					for (int k=j; k<x; k++)
					{
						m_RequestInfo[k & BDL_MASK].Mapping = m_RequestInfo[(k+1) & BDL_MASK].Mapping;
						m_RequestInfo[k & BDL_MASK].Size    = m_RequestInfo[(k+1) & BDL_MASK].Size;
						m_RequestInfo[k & BDL_MASK].Tag     = m_RequestInfo[(k+1) & BDL_MASK].Tag;
						DOUT (DBG_PRINT, ("[RevokeMappings] BLD copy %d to %d (GetNewIdx = %d)\n", (k+1) & BDL_MASK, k & BDL_MASK, m_ReqInfoReleaseIndex));
					}
					m_ReqInfoGetNewIndex = (m_ReqInfoGetNewIndex - 1) & BDL_MASK;
					DOUT (DBG_PRINT, ("[RevokeMappings] BLD set GetNewIdx from %d to %d\n", x & BDL_MASK, m_ReqInfoGetNewIndex));
				}

				break;	// j
			}
		}
		if(j == MAX_BDL_ENTRIES)
		{
			DOUT (DBG_ERROR, ("REVOKE MAPPING (%d) TAG NOT FOUND\n", nFirst));
			break;	// i
		}

        nFirst++;
    }

    //
    // Just un-pause the DMA engine if it was running before and there are
    // still entries left and tail != 0.
    //
    if (ulOldDMAEngineState == DMA_ENGINE_ON)
    {
        ResumeDMA ();
    }

    //
    // Release the mapping spin lock and return the number of mappings we 
    // revoked.
    //
    KeReleaseSpinLock (&m_MapLock, OldIrql);
    *MappingsRevoked = LcMappingsRevoked;

    return STATUS_SUCCESS;
}


/*****************************************************************************
 * CMiniportWaveICHStream::MappingAvailable
 *****************************************************************************
 * This routine is called by the port driver to notify the stream that there
 * are new mappings available.  Note that this is ONLY called after the stream
 * has previously had a GetMapping() call fail due to lack of available
 * mappings.
 */
STDMETHODIMP_(void) CMiniportWaveICHStream::MappingAvailable (void)
{
    DOUT (DBG_DMA, ("MappingAvailable\n"));
	KIRQL  OldIrql;

	KeRaiseIrql(DISPATCH_LEVEL, &OldIrql);

	// Release processed mappings.
	//
	ReleaseUsedMappings ();

	// Process the new mappings.
	//
	GetNewMappings (FALSE);	// TRUE does not work with VISTA

	KeLowerIrql(OldIrql);
}

/*****************************************************************************
 * CMiniportWaveICHStream::GetNewMappings
 *****************************************************************************
 * This routine is called when new mappings are available from the port driver.
 * The routine places mappings into the input mapping queue. ICH can handle up
 * to 32 entries (descriptors). We program the DMA registers if we have at least
 * one mapping in the queue. The mapping spin lock must be held when calling
 * this routine.
 */
NTSTATUS CMiniportWaveICHStream::GetNewMappings (BOOL PmOnlyOne)
{
    KIRQL		OldIrql;
    NTSTATUS    ntStatus = STATUS_SUCCESS;
    ULONG       ulBytesMapped = 0;

    DOUT (DBG_DMA, ("[GetNewMappings]\n"));

	// acquire the mapping spin lock
    KeAcquireSpinLock (&m_MapLock,&OldIrql);

	// do not re-enter in this fct
	if( m_InGetNewMap )
	{
		DOUT (DBG_WARNING, ("DO NOT REENTER GetNewMappings (chan=%d capt=%d)\n", m_ChannelNumber, m_Capture));
		KeReleaseSpinLock (&m_MapLock,OldIrql);
		return STATUS_SUCCESS;
	}
	m_InGetNewMap = TRUE;

#ifdef DBG_VIEWER
	if( (m_Capture == FALSE) && (m_ReqInfoPending >= MAX_STREAM_BUFFER) )
	{
		DOUT (DBG_DMA, ("MAX_R_BUFFERS(%d) reached (PLAY chan=%d)\n", MAX_STREAM_BUFFER, m_ChannelNumber));
	}
#endif
	// Get available mappings up to the max that we can hold in the BDL.
    //
    while ( m_ReqInfoPending < MAX_STREAM_BUFFER )
    {
        PREQUEST_INFO LcReqInfoPtr;
		WORD  LcRet;

		// Get the information from the list.
		//
		ULONG				Flags;
		ULONG				ulTag = m_TagCounter++;
		ULONG				ulBufferLength;
		PHYSICAL_ADDRESS	PhysAddr;
		PVOID				VirtAddr;
		BOOLEAN				bBufferInterrupt;

        // Release the mapping spin lock
        KeReleaseSpinLock (&m_MapLock,OldIrql);

		// Try to get the mapping from the port.
		// Here comes the problem: When calling GetMapping or ReleaseMapping we
		// cannot hold our spin lock. So we need to buffer the return values and
		// stick the information into the structure later when we have the spin
		// lock acquired.
		//
		ntStatus = m_PortStream->GetMapping((PVOID)ULongToPtr(ulTag),
											(PPHYSICAL_ADDRESS)&PhysAddr,
											&VirtAddr,
											&ulBufferLength,
											&Flags);
		// Acquire the mapping spin lock
		KeAcquireSpinLock (&m_MapLock,&OldIrql);

		// Quit this loop when we run out of mappings.
		//
		if (!NT_SUCCESS (ntStatus))
		{
			break;
		}

		// Sanity check: The audio stack will not give you data
		// that you cannot handle, but an application could use
		// DirectKS and send bad buffer down.

		// One mapping needs to be <0x1FFFF bytes for mono
		// streams on the ICH (PCXHR same thing!)
		//
		if (ulBufferLength > MAX_DSBUFF_SIZE)
		{
			// That is a little too long. That should never happen.
			DOUT (DBG_ERROR, ("[GetNewMappings] Buffer length too long!"));
			ulBufferLength = MAX_DSBUFF_SIZE;
		}

#if ( DMA_USE_TIMER )
		bBufferInterrupt = FALSE;
#else
		ulBytesMapped += ulBufferLength;
        //
        // Generate an interrupt when portcls tells us to or roughly every 10ms.
        //
        if ((Flags != 0) || (ulBytesMapped > (m_CurrentRate * m_SampleSize) / 100))
		{
			bBufferInterrupt = TRUE;
            ulBytesMapped = 0;
		}
		else
		{
			bBufferInterrupt = FALSE;
		}
#endif

		LcReqInfoPtr = &m_RequestInfo[m_ReqInfoGetNewIndex];
		ASSERT( LcReqInfoPtr->Mapping.QuadPart == 0 );

        BYTE ucIndexLX;
		LcRet = m_pDsp->vio_StreamGiveBufferLX( &m_Target,
                                                ulBufferLength,
                                                PhysAddr.LowPart,
                                                PhysAddr.HighPart,
                                                0,
                                                bBufferInterrupt,
                                                0,
                                                &ucIndexLX);

		DOUT (DBG_DMA, ("GET NEW MAPPING %d  (chan=%d capt=%d Idx=%d p@=%p size=%d -> %d  IT=%d) = %x\n",ulTag , m_ChannelNumber, m_Capture, m_ReqInfoGetNewIndex,
			PhysAddr.LowPart, ulBufferLength, ucIndexLX, bBufferInterrupt, LcRet));

		if( LcRet != SUCCESS )
		{
			DOUT (DBG_ERROR, ("[GetNewMappings] vio_StreamGiveBufferLX ERROR\n"));
			ntStatus = STATUS_UNSUCCESSFUL;
			break;
		}
		LcReqInfoPtr->Mapping.QuadPart = PhysAddr.QuadPart;
		LcReqInfoPtr->Size = ulBufferLength;
		LcReqInfoPtr->Tag = ulTag;
        LcReqInfoPtr->IndexLX = ucIndexLX;

		m_ReqInfoGetNewIndex = (m_ReqInfoGetNewIndex + 1) & BDL_MASK;
		m_ReqInfoPending++;

		if( (PmOnlyOne != FALSE) && (Flags != 0) ) break;
	}

	m_InGetNewMap = FALSE;

	// Release the mapping spin lock
	KeReleaseSpinLock (&m_MapLock,OldIrql);

    return ntStatus;
}


/*****************************************************************************
 * CMiniportWaveICHStream::ReleaseUsedMappings
 *****************************************************************************
 * This routine unmaps previously mapped memory that the hardware has 
 * completed processing on.  This routine is typically called at DPC level 
 * from the stream deferred procedure call that results from a stream 
 * interrupt. The mapping spin lock must be held when calling this routine.
 */
NTSTATUS CMiniportWaveICHStream::ReleaseUsedMappings (void)
{
	KIRQL OldIrql;

    DOUT (DBG_DMA, ("[ReleaseUsedMappings]\n"));

    // acquire the mapping spin lock
    KeAcquireSpinLock (&m_MapLock,&OldIrql);

	// do not re-enter in this fct
	if( m_InReleaseMap )
	{
		DOUT (DBG_WARNING, ("DO NOT REENTER ReleaseUsedMappings (chan=%d capt=%d)\n", m_ChannelNumber, m_Capture));
		KeReleaseSpinLock (&m_MapLock,OldIrql);
		return STATUS_SUCCESS;
	}
	m_InReleaseMap = TRUE;

    // Clean up everything to that index.
    //
    while (m_ReqInfoPending)
    {
		DWORD LcNeededBuff;
		DWORD LcFreedBuff;
		DWORD LcFreedBuffLen[MAX_STREAM_BUFFER];
        PREQUEST_INFO LcReqInfoPtr;

		// get freed buffer and do "Stream has place in RBuffers ?"
		//
        WORD LcRet = m_pDsp->vio_InitSoundTransferLX(&m_Target, &LcNeededBuff, &LcFreedBuff, LcFreedBuffLen);

		if( LcRet != SUCCESS )
		{
			DOUT (DBG_ERROR, ("[GetAndReleaseMappings] vio_InitSoundTransferLX ERROR\n"));
			break;
		}

        ASSERT( LcFreedBuff <= MAX_STREAM_BUFFER );
        ASSERT( LcFreedBuff <= (DWORD)m_ReqInfoPending );

		for( DWORD i=0; i<LcFreedBuff; i++ )
		{
			LcReqInfoPtr = &m_RequestInfo[m_ReqInfoReleaseIndex];

			if( LcReqInfoPtr->Mapping.QuadPart )
			{
				// Release the mapping spin lock
				KeReleaseSpinLock (&m_MapLock,OldIrql);

				// Release this entry.
				m_PortStream->ReleaseMapping ((PVOID)ULongToPtr(LcReqInfoPtr->Tag));

				DOUT (DBG_DMA, ("RELEASE MAPPING %d  (chan=%d capt=%d Idx=%d IdxLX=%d p@=%p size=%d of %d)\n",LcReqInfoPtr->Tag , m_ChannelNumber, m_Capture, m_ReqInfoReleaseIndex, LcReqInfoPtr->IndexLX,
					LcReqInfoPtr->Mapping.LowPart, LcFreedBuffLen[LcReqInfoPtr->IndexLX], LcReqInfoPtr->Size));

                m_TotalBytesReleased += LcReqInfoPtr->Size;

				// acquire the mapping spin lock
				KeAcquireSpinLock (&m_MapLock,&OldIrql);

				LcReqInfoPtr->Mapping.QuadPart = 0;
			}
			else
			{
				// when we want to notify a mapping, that was refused to be revoked
				// before (vio_StreamCancelBuffer())
				DOUT (DBG_ERROR, ("DO NOT RELEASE THE REVOKED MAPPING %d (chan=%d capt=%d Idx=%d)\n",LcReqInfoPtr->Tag , m_ChannelNumber, m_Capture, m_ReqInfoReleaseIndex));
			}
			m_ReqInfoReleaseIndex = (m_ReqInfoReleaseIndex + 1) & BDL_MASK;
			m_ReqInfoPending--;
		}
        break;  // while only 0 or 1 times !
    }

	m_InReleaseMap = FALSE;

	// Release the mapping spin lock
	KeReleaseSpinLock (&m_MapLock,OldIrql);

    return STATUS_SUCCESS;
}


/*****************************************************************************
 * CMiniportWaveICHStream::ResetDMA
 *****************************************************************************
 * This routine resets the Run/Pause bit in the control register. In addition, it
 * resets all DMA registers contents.
 */
NTSTATUS CMiniportWaveICHStream::ResetDMA (void)
{
    if ( m_DMAEngineState == DMA_ENGINE_OFF )
    {
        return STATUS_SUCCESS;
    }

    DOUT (DBG_PRINT, ("ResetDMA\n"));

    WORD LcRet = m_pDsp->IFlow_StreamStopBrutal( &m_Target );

    if ( LcRet != SUCCESS )
    {
        DOUT (DBG_ERROR, ("(IFlow_StreamStopBrutal) <- 0x%x\n", LcRet));
        return STATUS_UNSUCCESSFUL;
    }

    m_DMAEngineState = DMA_ENGINE_OFF;

    return STATUS_SUCCESS;
}


/*****************************************************************************
 * CMiniportWaveICHStream::PauseDMA
 *****************************************************************************
 * This routine pauses a hardware stream by reseting the Run/Pause bit in the
 * control registers, leaving DMA registers content intact so that the stream
 * can later be resumed.
 * You need to have the spin lock "MapLock" acquired.
 */
NTSTATUS CMiniportWaveICHStream::PauseDMA (void)
{
    //
    // Only pause if we're actually on.
    //
    if ( m_DMAEngineState != DMA_ENGINE_ON )
    {
        return STATUS_SUCCESS;
    }

    DOUT (DBG_PRINT, ("PauseDMA\n"));

#if( DMA_USE_TIMER )
	m_pDsp->IDiag_DspSetTimerInterrupt( 0 );
#endif

    WORD LcRet =  m_pDsp->IFlow_StreamPause( &m_Target, FALSE ); //drop stream

    if ( LcRet != SUCCESS )
    {
        DOUT (DBG_ERROR, ("(IFlow_StreamPause, Drop)  <- 0x%x\n", LcRet));
        return STATUS_UNSUCCESSFUL;
    }

    m_DMAEngineState = DMA_ENGINE_PAUSE;

    return STATUS_SUCCESS;
}


/*****************************************************************************
 * CMiniportWaveICHStream::ResumeDMA
 *****************************************************************************
 * This routine sets the Run/Pause bit for the particular DMA engine to resume
 * it after it's been paused. This assumes that DMA registers content have 
 * been preserved.
 * You need to have the spin lock "MapLock" acquired.
 */
NTSTATUS CMiniportWaveICHStream::ResumeDMA (void)
{
    if ( m_DMAEngineState == DMA_ENGINE_ON )
    {
        return STATUS_SUCCESS;
    }

    DOUT (DBG_PRINT, ("ResumeDMA\n"));

    WORD  LcRet = m_pDsp->IFlow_StreamStart( &m_Target );

    if ( LcRet != SUCCESS )
    {
        DOUT (DBG_ERROR, ("(IFlow_StreamStart) <- 0x%x\n", LcRet));
        return STATUS_UNSUCCESSFUL;
    }

#if( DMA_USE_TIMER )
    // use K=1
    WORD LcTimerSamples = m_pDsp->IDiag_GetPCMOnlyGranularity();

	m_pDsp->IDiag_DspSetTimerInterrupt((WORD)LcTimerSamples);
#endif

    m_DMAEngineState = DMA_ENGINE_ON;

    return STATUS_SUCCESS;
}


/*****************************************************************************
 * CMiniportWaveICHStream::SetState
 *****************************************************************************
 * This routine sets/changes the DMA engine state to play, stop, or pause
 */
STDMETHODIMP_(NTSTATUS) CMiniportWaveICHStream::SetState
(
    IN  KSSTATE State
)
{
    KIRQL		OldIrql;
    DOUT (DBG_STREAM, ("[CMiniportWaveICHStream::SetState]\n"));

    //
    // TODO: if this doesn't change, make a nice "if" statement.
    //
    if(State != KSSTATE_RUN)
    {
        DOUT (DBG_PRINT, ("SetState KSSTATE_%s chan=%d rec=%d\n",
			State==KSSTATE_STOP?"STOP":State==KSSTATE_ACQUIRE?"ACQUIRE":"PAUSE",
			m_ChannelNumber, m_Capture));

        // acquire the mapping spin lock
        KeAcquireSpinLock (&m_MapLock,&OldIrql);

        // Just pause DMA. If we have mappings left in our queue,
        // portcls will call RevokeMappings to destroy them.
        PauseDMA ();

        // release the mapping spin lock
        KeReleaseSpinLock (&m_MapLock,OldIrql);

        // Release processed mappings
        ReleaseUsedMappings( );
	}
	else
	{
        DOUT (DBG_PRINT, ("SetState KSSTATE_RUN chan=%d rec=%d\n", m_ChannelNumber, m_Capture));

		// Release processed mappings.
		ReleaseUsedMappings ();

		// Get new mappings.
		GetNewMappings (FALSE);

        // acquire the mapping spin lock
        KeAcquireSpinLock (&m_MapLock,&OldIrql);

        // Let's rock.
        ResumeDMA ();

        // release the mapping spin lock
        KeReleaseSpinLock (&m_MapLock,OldIrql);
	}

    return STATUS_SUCCESS;
}



/*****************************************************************************
 * CMiniportWaveICHStream::Init
 *****************************************************************************
 * This routine initializes the stream object, sets up the BDL, and programs
 * the buffer descriptor list base address register for the pin being
 * initialized.
 */
#define _OPER(c)    (c) ? OPER_REC : OPER_PLAY


NTSTATUS CMiniportWaveICHStream::Init
(
    IN  CMiniportWaveICH        *Miniport_,
    IN  PPORTWAVEPCISTREAM      PortStream_,
    IN  ULONG                   Pin_,
    IN  BOOLEAN                 Capture_,
    IN  PKSDATAFORMAT           DataFormat_,
    IN  ULONG                   Entry_,
    IN  WORD                    ChannelNumber_,
    OUT PSERVICEGROUP           *ServiceGroup_
)
{
    DOUT (DBG_STREAM, ("[ --> CMiniportWaveICHStream::Init]\n"));

    ASSERT (Miniport_);
    ASSERT (PortStream_);
    ASSERT (DataFormat_);
    ASSERT (ServiceGroup_);

    //
    // The rule here is that we return when we fail without a cleanup.
    // The destructor will relase the allocated memory.
    //
    NTSTATUS		ntStatus                = STATUS_SUCCESS;
	PWAVEFORMATEX   LcWaveFormatEx			= NULL;
    WORD			LcBitsPerSample ;
    WORD			LcChannels      ;

    // Clear target info for this stream
    //   
    RtlZeroMemory( &m_Target, sizeof(TARGET_INFO) );

    // Save miniport pointer and addref it.
    //
    m_Wave = Miniport_;
    m_Wave->AddRef ();

    // Save portstream interface pointer and addref it.
    //
    m_PortStream = PortStream_;
    m_PortStream->AddRef();

    // Save pin ID, capture flag and channel number.
    //
    m_Pin           = Pin_              ;
    m_Capture       = Capture_          ;
    m_ChannelNumber = ChannelNumber_    ;

    // Save data format and current sample rate.
	// DriverVerifier does not like when we touch LcWaveFormatEx after the BOARD_LOCK !
    // because LcWaveFormatEx seems to be paged memory
	//
    m_DataFormat  = (PKSDATAFORMAT_WAVEFORMATEX) DataFormat_    ;
    LcWaveFormatEx = &(m_DataFormat->WaveFormatEx) ;
    LcBitsPerSample = LcWaveFormatEx->wBitsPerSample;
    LcChannels      = LcWaveFormatEx->nChannels;
    m_SampleSize    = LcChannels * LcBitsPerSample / 8;

	//
	//	Check if Clock is available or locked by DSound
	//
    DWORD   LcCurrentSampleRate;
	DWORD   LcNewSampleRate = LcWaveFormatEx->nSamplesPerSec;
    BOOL    LcFrequencyIsLocked;

    // look if there is already a locked clock
    if( m_Wave->m_AdapterCommon->GetCurrentClock(m_MyInPipeMask, m_MyOutPipeMask, &LcCurrentSampleRate, &LcFrequencyIsLocked, NULL, NULL))
	{
		// does it match ?
		if( LcFrequencyIsLocked && (LcCurrentSampleRate != LcNewSampleRate) )
		{
			DOUT (DBG_ERROR, ("CMiniportWaveICHStream::Init LcCurrentSampleRate = %d / new = %d\n",LcCurrentSampleRate, LcNewSampleRate));
			return STATUS_UNSUCCESSFUL;
		}
	}

	m_CurrentRate = LcNewSampleRate;

    // build the frame information
    // 
    if (   ( LcWaveFormatEx->wFormatTag != WAVE_FORMAT_PCM )
        && ( LcWaveFormatEx->wFormatTag != WAVE_FORMAT_EXTENSIBLE) )
    {
        // Unsupported format
        //
        DOUT (DBG_ERROR, ("(Unsupported format) <- 0x%x\n", LcWaveFormatEx->wFormatTag));
        //RtlZeroMemory( &m_Target, sizeof(TARGET_INFO) );
        return(STATUS_UNSUCCESSFUL) ;    // buffers are freed in the destructor.
    }

    //
    // Initialize the BDL spinlock.
    //
    KeInitializeSpinLock (&m_MapLock);

    RtlZeroMemory(m_RequestInfo, MAX_BDL_ENTRIES * sizeof(REQUEST_INFO));

    // Reset the position pointers.
    //
    m_TotalBytesReleased = 0;
    m_PositionOffset = 0;
	m_ReqInfoGetNewIndex = 0;
	m_ReqInfoReleaseIndex = 0;
	m_ReqInfoPending = 0;
	m_InGetNewMap = 0;
	m_InReleaseMap = 0;

    // PrepareHardware
    //
    m_DMAEngineState = DMA_ENGINE_OFF;

    ntStatus = PrepareHardware( TRUE );

    if (!NT_SUCCESS(ntStatus)) {
        DOUT (DBG_ERROR, ("Failed to prepare and start Pipe!\n"));
        return ntStatus;
    }

    // Create a service group (a DPC abstraction/helper) to help with
    // interrupts.
    //
    ntStatus = PcNewServiceGroup (&m_ServiceGroup, NULL);

    if (!NT_SUCCESS(ntStatus)) {
        DOUT (DBG_ERROR, ("Failed to create a service group!\n"));
        return ntStatus;
    }

    // Pass the ServiceGroup pointer to portcls.
    //
    *ServiceGroup_ = m_ServiceGroup;
    m_ServiceGroup->AddRef ();

    // Initialize the device state.
    //
    m_PowerState = PowerDeviceD0;

	if ( ! m_Capture )	// PreFetchOffset only useful for playback streams
	{
		PPREFETCHOFFSET PreFetchOffset;
		//
		// Query for the new interface "PreFetchOffset" and use
		// function offered there in case the interface is offered.
		//
		if (NT_SUCCESS(m_PortStream->QueryInterface(IID_IPreFetchOffset, (PVOID *)&PreFetchOffset)))
		{
			DWORD LcLatency;

			LcLatency = m_pDsp->IDiag_GetPCMOnlyGranularity();

			DOUT (DBG_PRINT, ("IBL = %d\n", LcLatency));

			// keep ahead of next burst
			PreFetchOffset->SetPreFetchOffset(LcLatency * m_SampleSize);
			PreFetchOffset->Release();
		}
	}

    //
    // Store the stream pointer, it is used by the ISR.
    //
    m_Wave->Streams[m_Pin] = this;

    return STATUS_SUCCESS;
}


NTSTATUS CMiniportWaveICHStream::PrepareHardware( BOOL PmReservePipe )
{
    WORD        LcRet = 0;
    FRAME_INFO  LcFrameInfo;
    WORD        LcNbAllocatedStreams = 0;
    WORD        LcBoardNum = m_Wave->m_sBoardIndex ;
    WORD        LcChannels = m_DataFormat->WaveFormatEx.nChannels;
    WORD        LcBitsPerSample = m_DataFormat->WaveFormatEx.wBitsPerSample;

    if ( LcChannels == 0) {
        // Unsupported format
        //
        DOUT (DBG_ERROR, ("(Unsupported number of channels) <- %d\n", LcChannels));
        //RtlZeroMemory( &m_Target, sizeof(TARGET_INFO) );
        return(STATUS_UNSUCCESSFUL) ;    // buffers are freed in the destructor.
    }

    // test sample resolution
    //

    switch ( LcBitsPerSample ) {
    case 8:  LcFrameInfo.fiFormat = STREAM_FMT_LIN_8  ; break ;
    case 16: LcFrameInfo.fiFormat = STREAM_FMT_LIN_16 ; break ;
    case 24: LcFrameInfo.fiFormat = STREAM_FMT_LIN_24 ; break ;
    default:
        DOUT (DBG_ERROR, ("(Unsupported sample size) <- %d\n", LcBitsPerSample));
        return(STATUS_UNSUCCESSFUL) ;    // buffers are freed in the destructor.
    }

    LcFrameInfo.fiHSize = LcChannels
                        * LcBitsPerSample
                        * m_CurrentRate ;

    // Fill in target structure
    //
    m_Target.tgCarte    = LcBoardNum      ;
    m_Target.tgAudio    = m_ChannelNumber ;
    m_Target.tgPipe     = m_ChannelNumber ;
    m_Target.tgMaskFlux = 0x01 ;

    BOARD_LOCK();

    if ( m_Capture )
    {
        m_Target.tgCaracPipeVoie  = OPER_REC;

        if( PmReservePipe )
        {
        APHPrepareFakedPipeEntry( &m_Target,
                                  DS_FAKED_APPINDEX,
                                  LcChannels,
                                  &m_MyInPipeMask );
        }
        else
        {
            ASSERT(m_MyInPipeMask != 0);
        }
		m_MyOutPipeMask = 0;
    }
    else
    {
        m_Target.tgCaracPipeVoie  = OPER_PLAY;

        if( PmReservePipe )
        {
        APHPrepareFakedPipeEntry( &m_Target,
                                  DS_FAKED_APPINDEX,
                                  LcChannels,
                                  &m_MyOutPipeMask );
        }
        else
        {
            ASSERT(m_MyOutPipeMask != 0);
        }
		m_MyInPipeMask = 0;
    }

    if ( (m_MyInPipeMask == NULL) && (m_MyOutPipeMask == NULL) )
    {
        DOUT (DBG_ERROR, ("APHPrepareFakedPipeEntry failed\n"));
        BOARD_UNLOCK();
        return(STATUS_INSUFFICIENT_RESOURCES) ;
    }

    // if the allocation suceeds, the VioHandle is directly written to m_Target
    //
    LcRet = m_pDsp->IFlow_PipeDef(
                        m_Target.tgCaracPipeVoie,
                        m_ChannelNumber,  // LcFirstAudio
                        1, // 1 Stream
						LcChannels,                     // AudioNum
                        0,                              // LcChannelMask,
                        P_PCM_ONLY_MASK,                // Management
                        &LcFrameInfo,
                        FALSE,                          // not from APINP
                        &m_Target.tgVioHandle,
                        &LcNbAllocatedStreams	);

    if ( LcRet != SUCCESS )
    {
        APHRemoveFakedPipeEntry(&m_Target, DS_FAKED_APPINDEX );
        DOUT (DBG_ERROR, ("(IFlow_PipeDef) <- 0x%x\n", LcRet));
        BOARD_UNLOCK();
        return(STATUS_INSUFFICIENT_RESOURCES) ;
    }

    if( PmReservePipe )
    {
        if( m_Capture )
        {
            APHFillInFakedPipeEntry( &m_Target, m_MyInPipeMask, m_Wave); 
        }
        else
        {
            APHFillInFakedPipeEntry( &m_Target, m_MyOutPipeMask, m_Wave); 
        }
    }

    BOARD_UNLOCK();

    // Tell the topology to actually program the corresponding node settings
	// if this succeeds for a pipe, it a clock is reserved for it.
	//
    NTSTATUS    ntStatus;

	ntStatus = m_Wave->m_AdapterCommon->SetBoardClock(	m_CurrentRate, m_MyInPipeMask, m_MyOutPipeMask );
    if (!NT_SUCCESS (ntStatus))
    {
        DOUT (DBG_ERROR, ("SetBoardClock call failed!\n"));
        return ntStatus;
    }

    ntStatus = m_Wave->m_pTopology->StartActualNodeSettings ( m_ChannelNumber, m_Capture);

    if (!NT_SUCCESS (ntStatus))
    {
        DOUT (DBG_ERROR, ("StartActualNodeSettings call failed!\n"));
        return ntStatus;
    }

    BOARD_LOCK_AGAIN();

    // if one of this audios is controlled by DHS,
    // adjust its digital and monitoring levels and mutes
    for ( WORD i = 0 ; i < LcChannels; i ++ )
    {
        WORD LcChannelNumber = m_ChannelNumber + i;

        if( APHChkIsAudioDHSControlled( LcBoardNum, LcChannelNumber, m_Capture ) )
        {
            APHNewPipeSetDHSLevels( LcBoardNum, LcChannelNumber, (m_Capture != FALSE) );
        }
    }

    // Forward format to DSP
    //
	LcRet = m_pDsp->IFlow_FastSetFormat
						(
							&m_Target,
							m_CurrentRate,
							LcBitsPerSample,
							TRUE, /* PmIsIntel*/
							LcChannels
						);

	DOUT (DBG_PRINT, ("IFlow_FastSetFormat %s stream, %d Hz, %d bits %d channels\n", m_Capture ? "IN":"OUT", m_CurrentRate,LcBitsPerSample, LcChannels ));

	if ( LcRet != SUCCESS ) {
		BOARD_UNLOCK();
		DOUT (DBG_ERROR, ("(IFlow_FastSetFormat) <- 0x%x\n", LcRet));
		return STATUS_UNSUCCESSFUL;
	}

    // ----------------------------------------
    // Step #6: start the pipe
    // ----------------------------------------

    LcRet = m_pDsp->IFlow_PipeStart( &m_Target, 1, START_PAUSE_IMMEDIATE, NULL );

    BOARD_UNLOCK();

    if(LcRet == SUCCESS) {
        LcRet = m_pDsp->IEvent_WaitEndOfStartPipe( &m_Target );
    }

    if ( LcRet != SUCCESS ) {
        DOUT (DBG_ERROR, ("ERROR IFlow_PipeStart(%d) <- 0x%x\n", m_Target.tgPipe, LcRet));
        return (STATUS_UNSUCCESSFUL);
    }

    DOUT (DBG_PRINT, ("IFlow_PipeStart SUCCESS\n"));

    return STATUS_SUCCESS;
}


/*****************************************************************************
 * CMiniportWaveICHStream::SetFormat
 *****************************************************************************
 * This routine tests for proper data format (calls wave miniport) and sets
 * or changes the stream data format.
 * To figure out if the codec supports the sample rate, we just program the
 * sample rate and read it back. If it matches we return happy, if not then
 * we restore the sample rate and return unhappy.
 * We fail this routine if we are currently running (playing or recording).
 *  
 * reponse a KSPROPERTY_CONNECTION_DATAFORMAT, peut arriver en cour de play, car 
 * kmixer ajuste dynamiquement pour la plus haute freq d'ech.
 * 
 */
STDMETHODIMP_(NTSTATUS) CMiniportWaveICHStream::SetFormat
(
    IN  PKSDATAFORMAT   PmFormat
)
{
    DWORD   LcSamplesPerSec;
    WORD    LcBitsPerSample;
    WORD    LcChannels;
    BOOLEAN LcSetStreamFormat = FALSE;  // it is most probably just a freq setting
	DWORD   LcFrequency;

    ASSERT (PmFormat);

    DOUT (DBG_PRINT, ("[CMiniportWaveICHStream::SetFormat]\n"));
    
    //
    // Change sample rate when we are in the stop or pause states - not 
    // while running!
    //
    if (m_DMAEngineState == DMA_ENGINE_ON)
    {
        return STATUS_UNSUCCESSFUL;
    }

    //
    // Retrieve wave format portion.
    //
    PWAVEFORMATEX waveFormat = (PWAVEFORMATEX)(PmFormat + 1);
    
    // DriverVerifier does not like when we touch waveFormat after the BOARD_LOCK !
    // because waveFormat seems to be paged memory
    LcSamplesPerSec = waveFormat->nSamplesPerSec;
    LcBitsPerSample = waveFormat->wBitsPerSample;
    LcChannels      = waveFormat->nChannels;

    if( LcChannels == 0 ) {
            DOUT (DBG_ERROR, ("MinWaveICHStream::SetFormat %d channels not allowed\n", LcChannels));
            return STATUS_UNSUCCESSFUL;
    }

    // Do not send a Request to DSP, if the stream format doesn't need to be set,
    // which will be the case, until kmixer handles bitspersamples adaption.
    // 
    if  (      (m_DataFormat == NULL)
            || (m_DataFormat->WaveFormatEx.wBitsPerSample != LcBitsPerSample)
            || (m_DataFormat->WaveFormatEx.nChannels != LcChannels))
            LcSetStreamFormat = TRUE;

    // Ensure format falls in proper range and is supported.
    //
    NTSTATUS ntStatus = m_Wave->TestDataFormat (PmFormat, (WavePins)(m_Pin));
    if (!NT_SUCCESS (ntStatus)) {
        DOUT (DBG_VSR, ("MinWaveICHStream::SetFormat => TestDataFormat failed\n"));
        return ntStatus;
    }

	// is the clock locked by someone else ?
    BOOL LcFrequencyLocked;
	BOOL LcClockLocked = m_Wave->m_AdapterCommon->GetCurrentClock(m_MyInPipeMask, m_MyOutPipeMask, &LcFrequency, &LcFrequencyLocked, NULL, NULL);
    if( LcClockLocked && LcFrequencyLocked )
	{
		if( LcFrequency != LcSamplesPerSec )
		{
			DOUT (DBG_ERROR, ("MinWaveICHStream::SetFormat(%d Hz) failed, board locked at %d Hz\n", LcSamplesPerSec, LcFrequency));
			return STATUS_UNSUCCESSFUL;
		}
	}
	else // the clock is not locked, need to be higher than 12000
	{
		if( LcSamplesPerSec < 12000 )
		{
			DOUT (DBG_ERROR, ("MinWaveICHStream::SetFormat(f < 12000 Hz) fail!\n"));
			return STATUS_UNSUCCESSFUL;
		}
	}

    if ( m_Target.tgVioHandle )
    {
        // Set the bit rate accordingly to the new format
        // Do not change clock source
		//
		ntStatus = m_Wave->m_AdapterCommon->SetBoardClock(	LcSamplesPerSec, m_MyInPipeMask, m_MyOutPipeMask );
		
		// what ever happened, ask AdapterCommon what the current rate is.
		//
		m_CurrentRate = 0;
        m_Wave->m_AdapterCommon->GetCurrentClock(m_MyInPipeMask, m_MyOutPipeMask, &m_CurrentRate, NULL, NULL, NULL);

        DOUT (DBG_PRINT, ("[CMiniportWaveICHStream::SetFormat ] %s stream, %d Hz\n", m_Capture ? "IN":"OUT", LcSamplesPerSec));

		if (NT_SUCCESS (ntStatus) && (LcSetStreamFormat != FALSE))
		{
			 // Forward format to DSP
			 //
             BOARD_LOCK();

			 WORD LcRet =  m_pDsp->IFlow_FastSetFormat(
													&m_Target,
													LcSamplesPerSec,
													LcBitsPerSample,
                                                    TRUE, // intel
													LcChannels
													);
             BOARD_UNLOCK();

             DOUT (DBG_PRINT, ("[CMiniportWaveICHStream::SetFormat ] %s stream, %d bits %d channels\n", m_Capture ? "IN":"OUT", LcBitsPerSample, LcChannels ));

			 // Setformat to DSP failed
			 //
			if ( LcRet != SUCCESS ) 
			{
				ntStatus = STATUS_NOT_SUPPORTED;
			} 

		} // setformat
	} // tgVioHandle

    if (NT_SUCCESS (ntStatus))
    {
        // Update current format
        //
        m_DataFormat  = (PKSDATAFORMAT_WAVEFORMATEX) PmFormat ;
        m_SampleSize  = m_DataFormat->WaveFormatEx.nChannels *
                        m_DataFormat->WaveFormatEx.wBitsPerSample / 8;
    }
	else
	{
		DOUT (DBG_VSR, ("MinWaveICHStream::SetFormat => SetBoardClock failed\n"));
	}

    return(ntStatus);
}


#pragma code_seg("PAGE")


/*****************************************************************************
 * CMiniportWaveICHStream::SetContentId
 *****************************************************************************
 * This routine gets called by drmk.sys to pass the content to the driver.
 * The driver has to enforce the rights passed.
 */
STDMETHODIMP_(NTSTATUS) CMiniportWaveICHStream::SetContentId
(
    IN  ULONG       contentId,
    IN  PCDRMRIGHTS drmRights
)
{
    PAGED_CODE ();

    DOUT (DBG_PRINT, ("[CMiniportWaveICHStream::SetContentId]\n"));

    //
    // If "drmRights->DigitalOutputDisable" is set, we need to disable S/P-DIF.
    // Currently, we don't have knowledge about the S/P-DIF interface. However,
    // in case you expanded the driver with S/P-DIF features you need to disable
    // S/P-DIF or fail SetContentId. If you have HW that has S/P-DIF turned on
    // by default and you don't know how to turn off (or you cannot do that)
    // then you must fail SetContentId.
    //
    // Store the copyright flag. We have to disable PCM recording if it's set.
    //
    m_Wave->m_pTopology->SetDrmFlags( drmRights->CopyProtect, drmRights->DigitalOutputDisable );

    //
    // We assume that if we can enforce the rights, that the old content
    // will be destroyed. We don't need to store the content id since we
    // have only one playback channel, so we are finished here.
    //
    
    return STATUS_SUCCESS;
}


#pragma code_seg()


/*****************************************************************************
 * CMiniportWaveICHStream::~CMiniportWaveICHStream
 *****************************************************************************
 * Destructor
 */
CMiniportWaveICHStream::~CMiniportWaveICHStream ()
{
    DOUT (DBG_PRINT, ("[CMiniportWaveICHStream::~CMiniportWaveICHStream]\n"));

    // Remove self from miniport Streams array.
    //
    if (m_Wave->Streams[m_Pin] == this)
    {
        m_Wave->Streams[m_Pin] = NULL;
    }

    // Release the service group.
    //
    if ( m_ServiceGroup )
    {
        m_ServiceGroup->Release ();
        m_ServiceGroup = NULL;
    }

    m_Wave->m_AdapterCommon->ReleaseClockSettings(	m_MyInPipeMask, m_MyOutPipeMask ) ;

    BOARD_LOCK();

    WORD LcRet = SUCCESS;
    if ( m_Target.tgVioHandle )
	{
		DWORD LcGranularity;
		DWORD LcWait;

        LcRet = m_pDsp->IFlow_PipePause(&m_Target, 1, START_PAUSE_IMMEDIATE );
        DOUT (DBG_PRINT, ("IFlow_PipePause = %x\n", LcRet));

		BOARD_UNLOCK();

        // do not hold board lock to wait !
        //
		m_pDsp->IEvent_WaitEndOfCoding( &m_Target );

		BOARD_LOCK_AGAIN();

        LcRet = m_pDsp->IFlow_PipeStop(&m_Target);
        DOUT (DBG_PRINT, ("IFlow_PipeStop = %x\n", LcRet));

		LcRet = m_pDsp->IFlow_PipeRelease(&m_Target, TRUE );
        DOUT (DBG_PRINT, ("IFlow_PipeRelease (%s) = %x\n", m_Capture ? "REC" : "PLAY", LcRet));

		APHRemoveFakedPipeEntry(&m_Target, DS_FAKED_APPINDEX );
    }
 
    BOARD_UNLOCK();

    // Tell again the topology we are dead
    //
    m_Wave->m_pTopology->StopActualNodeSettings (m_ChannelNumber, m_Capture);

    if( m_Capture )
    {
        WORD LcChannels = m_DataFormat->WaveFormatEx.nChannels;

		BOARD_LOCK_AGAIN();

        for( WORD i=0; i<LcChannels; i+=2 )
        {
            // do not change audio input settings if DHS controlled
            //
            if( APHChkIsAudioDHSControlled( m_Wave->m_sBoardIndex, m_ChannelNumber + i, TRUE ) )
                LcRet = SUCCESS;
            else
                LcRet = m_pDsp->IResource_Source_Release_Audios( (DWORDLONG)3 << (m_ChannelNumber + i) );
        }

		BOARD_UNLOCK();

		if( ERROR_MASK & LcRet )
		{
			DOUT (DBG_ERROR, ("MinTopo::Source Release Audios NOK\n"));
		}
    }

    // Release the port stream.
    //
    if ( m_PortStream )
    {
        m_PortStream->Release ();
        m_PortStream = NULL;
    }

    // Release the miniport.
    //
    if ( m_Wave )
    {
        m_Wave->Release ();
        m_Wave = NULL;
    }
}
