// ***************************************************************************
//
// COPYRIGHT 1996-2000 DIGIGRAM. ALL RIGHT RESERVED.
//
// DIGIGRAM
//
// ***************************************************************************
 
//
// All the GUIDs from portcls and your own defined GUIDs end up in this object.
//
#define PUT_GUIDS_HERE

#define DEFINE_DEBUG_VARS


// Include files
// *************

#include "shared.h"
#include "lxeswdm.h"
#include "log.h"
#include "pci_conf.h"
#include "es_cmds.h"
#include "if_drv_mb.h"
#include <ntstrsafe.h>



// Local constants, types, macros definitions
// ******************************************

// Size of registry query table ( 8 entries + 1 NULL-entry )
// -------------------------------------------------------
#define QUERY_TAB_SIZE          9

// Path to driver specific keys
// ----------------------------
#define PCX_PARAMETERS_KEY      L"\\Parameters"

// Names of subkeys
// ----------------
#define PCX_BINARIES_ROOT_KEY   L"BinariesRootPath"
#define PCX_BUFFERS_ROOT_KEY    L"Buffers"
#define PCX_BUF_SIZE_ROOT_KEY   L"BufferSize"
#define PCX_LOWLAT_MIX_ROOT_KEY L"PCMOnlyGranularity"
#define PCX_PREFERRED_FW_V_KEY  L"PreferredFirmwareVersion"
#define PCX_CHECK_ROOT_KEY      L"Check"
#define PCX_DEBUG_ROOT_KEY      L"Debug"
#define PCX_HIDE_DEVICE_KEY     L"HideDeviceMask"

static PDRIVER_UNLOAD       m_PortClassUnload         = NULL ;

static NTSTATUS    NP88DispatchWrapIoctl  (IN PDEVICE_OBJECT,IN PIRP);
static VOID        NP88WrapUnload         (IN PDRIVER_OBJECT);

       UNICODE_STRING       m_RegistryPathName  ;

       GENERAL_INFO         m_GeneralInfo ;


#pragma code_seg("PAGE")

/*****************************************************************************
 * InstallSubdevice
 *****************************************************************************
 * This function creates and registers a subdevice consisting of a port
 * driver, a minport driver and a set of resources bound together.  It will
 * also optionally place a pointer to an interface on the port driver in a
 * specified location before initializing the port driver.  This is done so
 * that a common ISR can have access to the port driver during initialization,
 * when the ISR might fire.
 * This function is internally used and validates no parameters.
 */
STATIC NTSTATUS InstallSubdevice
(
    IN      PDEVICE_OBJECT      DeviceObject,
    IN      PIRP                Irp,
    IN      REFGUID             PortClassId,
    IN      REFGUID             MiniportClassId,
    IN      PFNCREATEINSTANCE   MiniportCreate      OPTIONAL,
    IN      PUNKNOWN            UnknownAdapter      OPTIONAL,
    IN      PRESOURCELIST       /* ResourceList */,
    IN      REFGUID             PortInterfaceId,
    OUT     PMINIPORT *         OutMiniport         OPTIONAL,
    OUT     PPORT *             OutPort             OPTIONAL,
    OUT     PUNKNOWN *          OutPortUnknown      OPTIONAL
)
{
    PAGED_CODE ();

    NTSTATUS    ntStatus;
    PPORT       port;
    PMINIPORT   miniport;

	(void)PortInterfaceId;
	(void)UnknownAdapter;
	(void)Irp;
	(void)DeviceObject;

    //
    // Create the port driver object
    //
    ntStatus = PcNewPort (&port,PortClassId);

    //
    // return immediately in case of an error
    //
    if (!NT_SUCCESS (ntStatus))
       return ntStatus;

    //
    // Create the miniport object
    //
    if (MiniportCreate)
    {
        ntStatus = MiniportCreate ((PUNKNOWN*)&miniport,
                                    MiniportClassId,
                                    NULL,
                                    NonPagedPool);
    }
    else
    {
        ntStatus = PcNewMiniport (&miniport,MiniportClassId);
    }

    //
    // return immediately in case of an error
    //
    if (!NT_SUCCESS (ntStatus))
    {
        port->Release ();
        return ntStatus;
    }


    //
    // Deposit the port as an unknown if it's needed.
    //
    if (OutPortUnknown && NT_SUCCESS (ntStatus))
    {
      ntStatus = port->QueryInterface (IID_IUnknown,
                       (PVOID *)OutPortUnknown);
    }

    //
    // Deposit the port as an IPort if it's needed.
    //
    if ( OutPort && NT_SUCCESS (ntStatus) )
    {
      ntStatus = port->QueryInterface (IID_IPort,
                       (PVOID *)OutPort);
    }

    //
    // Deposit the miniport as an IMiniport if it's needed.
    //
    if ( OutMiniport && NT_SUCCESS (ntStatus) )
    {
      ntStatus = miniport->QueryInterface (IID_IMiniport,
                       (PVOID *)OutMiniport);
    }

    //
    // Release the reference for the port and miniport. This is the right
    // thing to do, regardless of the outcome.
    //
    miniport->Release ();
    port->Release ();

    return ntStatus;
}


#pragma code_seg("INIT")

// ****************************************************************************
// STATIC NTSTATUS      GetParametersFromRegistry()
// **************************************
//
// Input parameters :
// ******************
//
//      PUNICODE_STRING     PmRegistryPath  :   Path to the driver keys
//
// Output parameters :
// *******************
//
//      PGENERAL_INFO       PmGeneralInfo   :   pointer to driver structures
//
// Return value :
// **************
//
//      a NTSTATUS error code
//
// ****************************************************************************
//
// Fills in driver structures in order to know its needs
//
// ****************************************************************************
//
// Retrieves the binaries root path from the registry, open and read the
// pcxcfg_e.ini file and lookup the setup string, then fills in the data
// structures according to this string
//
// ****************************************************************************
static  NTSTATUS        GetParametersFromRegistry(
    IN  PUNICODE_STRING     PmRegistryPath,
    OUT PGENERAL_INFO       PmGeneralInfoPtr )
{
    PAGED_CODE();

    UNICODE_STRING      LcParamPath   ;
    UNICODE_STRING      LcRootString  ;
    ANSI_STRING         LcAnsiString  ;
    NTSTATUS            LcNtStatus  = STATUS_SUCCESS  ;

    DWORD               LcParamBuffers;
    DWORD               LcParamBuffersDefault     = 150;
    DWORD               LcParamBufferSize;
    DWORD               LcParamBufferSizeDefault  = 128*1024;
    DWORD               LcParamCheck;
    DWORD               LcParamCheckDefault       = 'R';	// do not check revision and build number
    DWORD               LcParamDebug;
    DWORD               LcParamDebugDefault       = DBG_DEFAULT;
    DWORD               LcPCMOnlyGranularity;
    DWORD               LcPCMOnlyGranularityDefault  = MICROBLAZE_IBL_DEFAULT ; // le DSP ajuste en fonction de son min 
    DWORD               LcPreferredFirmware;
    DWORD               LcPreferredFirmwareDefault  = MICROBLAZE_VERSION;
    DWORD               LcHideDeviceMask;
    DWORD               LcHideDeviceMaskDefault = 0;

    RTL_QUERY_REGISTRY_TABLE LcParamTable[QUERY_TAB_SIZE];
                                              // Used for BinariesRootPath, ...
                                              // + NULL-terminating entry

    // Default value returned by registry if a key is not found
    // --------------------------------------------------------
    static WCHAR DefaultKeyValue[]  = L"!NOT FOUND!";

    // Initialize empty strings
    // -------------------------
    LcRootString.Buffer = NULL ;

    LcParamPath.MaximumLength     = PmRegistryPath->Length + 50 ;
    LcParamPath.Length            = 0                         ;
    LcParamPath.Buffer            = (PWSTR) DDK_ALLOC( sizeof(WCHAR) * LcParamPath.MaximumLength );

    if ( LcParamPath.Buffer == NULL )
    {
        LcNtStatus = STATUS_NO_MEMORY ;
        goto clean_exit ;
    }

    if ( !UNCAllocateUnicodeString( &LcRootString ) )
    {
        LcNtStatus = STATUS_NO_MEMORY ;
        goto clean_exit               ;
    }

    // Points to driver parameters
    // ----------------------------
    RtlAppendUnicodeStringToString( &LcParamPath, PmRegistryPath );
    RtlAppendUnicodeToString( &LcParamPath, PCX_PARAMETERS_KEY );

    // BINARY ROOT PATH
    LcParamTable[0].QueryRoutine = (PRTL_QUERY_REGISTRY_ROUTINE) NULL;
    LcParamTable[0].Flags        = RTL_QUERY_REGISTRY_DIRECT;
    LcParamTable[0].Name         = PCX_BINARIES_ROOT_KEY;
    LcParamTable[0].EntryContext  = (PVOID)&LcRootString;
    LcParamTable[0].DefaultType   = REG_SZ;
    LcParamTable[0].DefaultData   = DefaultKeyValue;
    LcParamTable[0].DefaultLength = 0;

    // PmPtrGeneral->NbOfBuffers
    LcParamTable[1].QueryRoutine  = (PRTL_QUERY_REGISTRY_ROUTINE) NULL;
    LcParamTable[1].Flags         = RTL_QUERY_REGISTRY_DIRECT;
    LcParamTable[1].Name          = PCX_BUFFERS_ROOT_KEY;
    LcParamTable[1].EntryContext  = (PVOID)&LcParamBuffers;
    LcParamTable[1].DefaultType   = REG_DWORD;
    LcParamTable[1].DefaultData   = (PVOID)&LcParamBuffersDefault ;
    LcParamTable[1].DefaultLength = sizeof (LcParamBuffersDefault);

    // PmPtrGeneral->SizeOfBuffers
    LcParamTable[2].QueryRoutine  = (PRTL_QUERY_REGISTRY_ROUTINE) NULL;
    LcParamTable[2].Flags         = RTL_QUERY_REGISTRY_DIRECT;
    LcParamTable[2].Name          = PCX_BUF_SIZE_ROOT_KEY;
    LcParamTable[2].EntryContext  = (PVOID)&LcParamBufferSize;
    LcParamTable[2].DefaultType   = REG_DWORD;
    LcParamTable[2].DefaultData   = (PVOID)&LcParamBufferSizeDefault ;
    LcParamTable[2].DefaultLength = sizeof (LcParamBufferSizeDefault);

    // PCMOnlyGranularity
    LcParamTable[3].QueryRoutine  = (PRTL_QUERY_REGISTRY_ROUTINE) NULL;
    LcParamTable[3].Flags         = RTL_QUERY_REGISTRY_DIRECT;
    LcParamTable[3].Name          = PCX_LOWLAT_MIX_ROOT_KEY;
    LcParamTable[3].EntryContext  = (PVOID)&LcPCMOnlyGranularity;
    LcParamTable[3].DefaultType   = REG_DWORD;
    LcParamTable[3].DefaultData   = (PVOID)&LcPCMOnlyGranularityDefault ;
    LcParamTable[3].DefaultLength = sizeof (LcPCMOnlyGranularityDefault);

    // PmPtrGeneral->PDriverInfo->drDoNotCheckForVersionState
    LcParamTable[4].QueryRoutine  = (PRTL_QUERY_REGISTRY_ROUTINE) NULL;
    LcParamTable[4].Flags         = RTL_QUERY_REGISTRY_DIRECT;
    LcParamTable[4].Name          = PCX_CHECK_ROOT_KEY;
    LcParamTable[4].EntryContext  = (PVOID)&LcParamCheck;
    LcParamTable[4].DefaultType   = REG_DWORD;
    LcParamTable[4].DefaultData   = (PVOID)&LcParamCheckDefault ;
    LcParamTable[4].DefaultLength = sizeof (LcParamCheckDefault);

    // g_ulDebugOut
    LcParamTable[5].QueryRoutine  = (PRTL_QUERY_REGISTRY_ROUTINE) NULL;
    LcParamTable[5].Flags         = RTL_QUERY_REGISTRY_DIRECT;
    LcParamTable[5].Name          = PCX_DEBUG_ROOT_KEY;
    LcParamTable[5].EntryContext  = (PVOID)&LcParamDebug;
    LcParamTable[5].DefaultType   = REG_DWORD;
    LcParamTable[5].DefaultData   = (PVOID)&LcParamDebugDefault ;
    LcParamTable[5].DefaultLength = sizeof (LcParamDebugDefault);

    // LcPreferredFirmware
    //
    LcParamTable[6].QueryRoutine  = (PRTL_QUERY_REGISTRY_ROUTINE) NULL;
    LcParamTable[6].Flags         = RTL_QUERY_REGISTRY_DIRECT;
    LcParamTable[6].Name          = PCX_PREFERRED_FW_V_KEY;
    LcParamTable[6].EntryContext  = (PVOID)&LcPreferredFirmware;
    LcParamTable[6].DefaultType   = REG_DWORD;
    LcParamTable[6].DefaultData   = (PVOID)&LcPreferredFirmwareDefault ;
    LcParamTable[6].DefaultLength = sizeof (LcPreferredFirmwareDefault);

    // LcHideDeviceMask
    //
    LcParamTable[7].QueryRoutine  = (PRTL_QUERY_REGISTRY_ROUTINE) NULL;
    LcParamTable[7].Flags         = RTL_QUERY_REGISTRY_DIRECT;
    LcParamTable[7].Name          = PCX_HIDE_DEVICE_KEY;
    LcParamTable[7].EntryContext  = (PVOID)&LcHideDeviceMask;
    LcParamTable[7].DefaultType   = REG_DWORD;
    LcParamTable[7].DefaultData   = (PVOID)&LcHideDeviceMaskDefault ;
    LcParamTable[7].DefaultLength = sizeof (LcHideDeviceMaskDefault);

    // NULL-terminate the ParamTable
    // ------------------------------
    ASSERT( 8 < QUERY_TAB_SIZE);
    BZERO2( &LcParamTable[8], LcParamTable[8], 1 );

    LOGFileTrace("Querying registry\n");

    LcNtStatus = RtlQueryRegistryValues(
                        RTL_REGISTRY_ABSOLUTE | RTL_REGISTRY_OPTIONAL,
                        LcParamPath.Buffer,
                        LcParamTable,
                        NULL,
                        NULL );

    if ( !NT_SUCCESS( LcNtStatus ) )
    {
        LOGFileTrace("Failed to query registry (0x%lx)\n", LcNtStatus);
        goto clean_exit ;
    }

    // convert our registry unicode string to an ANSI (narrow) string
    // ---------------------------------------------------------------
    LcAnsiString.MaximumLength = MAX_PCX_PATH_LEN ;
    LcAnsiString.Length = 0 ;
    LcAnsiString.Buffer = PmGeneralInfoPtr->PDriverInfo->drPath ;
    if ( LcAnsiString.Buffer == NULL)
    {
        LcNtStatus = STATUS_NO_MEMORY;
        goto clean_exit;
    }

    LcNtStatus = RtlUnicodeStringToAnsiString( &LcAnsiString,
                                               &LcRootString,
                                               FALSE );
    if ( !NT_SUCCESS( LcNtStatus ) )
    {
        LOGFileTrace("String conversion failed (0x%lx)\n", LcNtStatus);
        goto clean_exit ;
    }

    LOGFileTrace("-> BinariesRootPath = \"%s\"\n", LcAnsiString.Buffer);

    PmGeneralInfoPtr->PDriverInfo->drResetAnalogLevelsAction = RESET_GAIN_MUTE  ;

    LOGFileTrace("-> Setup : Buffers(%d) BufferSize(%d)\n", LcParamBuffers, LcParamBufferSize);
    if(LcParamCheck != LcParamCheckDefault)
        LOGFileTrace("-> *BEWARE* Check against version match disabled !!!!\n");
    if(LcParamDebug != LcParamDebugDefault)
        LOGFileTrace("-> Debug = %x\n", LcParamDebug);
    if(LcPreferredFirmware != LcPreferredFirmwareDefault)
    {
        LcPreferredFirmware &= 0xffffff;
        LOGFileTrace("-> Preferred Firmware = V%02d.%02d # %03d\n", (LcPreferredFirmware >> 16),
                                                                    (LcPreferredFirmware >> 8)&0xff,
                                                                    (LcPreferredFirmware & 0xff)  );
    }
    LOGFileTrace("********************************************\n");
    LOGFileTrace("	PCM granularity set to %d samples\n" , LcPCMOnlyGranularity);
    LOGFileTrace("********************************************\n");

    PmGeneralInfoPtr->PCMOnlyGranularity = LcPCMOnlyGranularity;
    PmGeneralInfoPtr->PreferredFirmwareVersion = LcPreferredFirmware;
    PmGeneralInfoPtr->NbOfBuffers = LcParamBuffers;
    PmGeneralInfoPtr->SizeOfBuffers = LcParamBufferSize;
	PmGeneralInfoPtr->PDriverInfo->drDoNotCheckForVersionState = (BYTE)LcParamCheck;
    PmGeneralInfoPtr->HideDeviceMask = LcHideDeviceMask;
#ifdef DBG_VIEWER
    g_ulDebugOut = LcParamDebug;
#endif

clean_exit:
    if(LcParamPath.Buffer)  DDK_FREE( LcParamPath.Buffer );
    UNCFreeUnicodeString( &LcRootString )  ;

    return( LcNtStatus ) ;
}

#pragma code_seg("PAGE")

#define CONFIG_QUERY_TAB_SIZE 3

NTSTATUS GetSerialnumConfigFromRegistry(
    IN  PUNICODE_STRING PmSerialnum,
    OUT PDWORD          PmConfES)
{
    PAGED_CODE();

    UNICODE_STRING      LcParamPath   ;
    DWORD               LcConfES, LcConfESDefault;
    RTL_QUERY_REGISTRY_TABLE LcParamTable[CONFIG_QUERY_TAB_SIZE];
    NTSTATUS            LcNtStatus  = STATUS_SUCCESS  ;

    LcParamPath.MaximumLength     = m_RegistryPathName.Length + 128;
    LcParamPath.Length            = 0;
    LcParamPath.Buffer            = (PWSTR) DDK_ALLOC( LcParamPath.MaximumLength );

    if ( LcParamPath.Buffer == NULL )
    {
        return STATUS_NO_MEMORY ;
    }

    // Defaults ConfES
    LcConfESDefault =   (64 << IOCR_OUTPUTS_OFFSET) |
                        (64 << IOCR_INPUTS_OFFSET) |
                        (FREQ_RATIO_SINGLE_MODE << FREQ_RATIO_OFFSET);

    // Points to driver parameters
    // ----------------------------
    RtlAppendUnicodeStringToString( &LcParamPath, &m_RegistryPathName );
    RtlAppendUnicodeToString( &LcParamPath, L"\\Parameters\\" );
    RtlAppendUnicodeStringToString( &LcParamPath, PmSerialnum );

    // ConfES
    LcParamTable[0].QueryRoutine  = (PRTL_QUERY_REGISTRY_ROUTINE) NULL;
    LcParamTable[0].Flags         = RTL_QUERY_REGISTRY_DIRECT;
    LcParamTable[0].Name          = L"ConfES";
    LcParamTable[0].EntryContext  = (PVOID)&LcConfES;
    LcParamTable[0].DefaultType   = REG_DWORD;
    LcParamTable[0].DefaultData   = (PVOID)&LcConfESDefault ;
    LcParamTable[0].DefaultLength = sizeof (LcConfESDefault);

    // IOMR
    BZERO2( &LcParamTable[1], LcParamTable[1], 1 ); // no more supported

    // NULL-terminate the ParamTable
    // ------------------------------
    ASSERT( 2 < CONFIG_QUERY_TAB_SIZE);
    BZERO2( &LcParamTable[2], LcParamTable[2], 1 );

    LOGFileTrace("-> Querying registry for SerialNumber %ws\n", PmSerialnum->Buffer);

    LcNtStatus = RtlQueryRegistryValues(
                        RTL_REGISTRY_ABSOLUTE | RTL_REGISTRY_OPTIONAL,
                        LcParamPath.Buffer,
                        LcParamTable,
                        NULL,
                        NULL );

    if ( !NT_SUCCESS( LcNtStatus ) )
    {
        LOGFileTrace("-> Registry : No config found; use defaults\n");
        LcConfES = LcConfESDefault;
    }

    if(PmConfES)    *PmConfES = LcConfES;

    if(LcParamPath.Buffer)  DDK_FREE( LcParamPath.Buffer );

    return( LcNtStatus ) ;
}


/*****************************************************************************
 * ValidateResources
 *****************************************************************************
 * This function validates the list of resources for the various functions on
 * the card.  This code is specific to the adapter.
 * This function doesn't check the ResourceList parameter and returns
 * STATUS_SUCCESS when the resources are valid.
 */
NTSTATUS ValidateResources
(
    IN      PDEVICE_OBJECT  DeviceObject,   // Device object.
    IN      PRESOURCELIST   ResourceList    // All resources.
)
{
    PAGED_CODE ();

    PDSOUND_DEVICE_EXTENSION LcMyDeviceExtension;
    PGENERAL_INFO           LcGeneralInfo;
    NTSTATUS                ntStatus   = STATUS_SUCCESS ;
    PWCHAR                  deviceName = NULL;
    PBOARD_INFO             LcPtrBoardInfo = NULL;
    ULONG                   nameLength = 0 ;
    WORD                    LcBoardIndex = -1;

    static WCHAR        LcPciInit1[]    = L"PCI\\";
    static WCHAR        LcPciInit2[]    = L"pci/";
    static WCHAR        LcIDInit3[]     = L"0123";

    LOGFileTrace( "-> Validating new board\n" );

    LcMyDeviceExtension = (PDSOUND_DEVICE_EXTENSION) DeviceObject->DeviceExtension;
    LcGeneralInfo = LcMyDeviceExtension->dsdeGeneralInfo;

    //
    // Get counts for the types of resources.
    //
    ULONG countIO  = ResourceList->NumberOfPorts ();
    ULONG countIRQ = ResourceList->NumberOfInterrupts ();
    ULONG countDMA = ResourceList->NumberOfDmas ();
    ULONG countMemory = ResourceList->NumberOfMemories();

     // validate resources
    if (   (countMemory != 2)
        || (countIO != 1)
        || (countIRQ != 1)
        || (countDMA != 0))
    {
        LOGFileTrace("Unknown configuration:\n");
        LOGFileTrace("   IO  count: %d\n", countIO);
        LOGFileTrace("   IRQ count: %d\n", countIRQ);
        LOGFileTrace("   DMA count: %d\n", countDMA);
        LOGFileTrace("   Mem count: %d\n", countMemory);
        return STATUS_DEVICE_CONFIGURATION_ERROR;
    }

    // Let's retrieve the device name in order to
    // make sure this is one of ours
    //
    ntStatus = PcGetDeviceProperty (DeviceObject,
                                   DevicePropertyHardwareID,
                                   0,
                                   NULL,
                                   &nameLength);

    if ((nameLength != 0) && (ntStatus == STATUS_BUFFER_TOO_SMALL)) {
        deviceName = (PWCHAR) DDK_ALLOC_WITH_TAG (NonPagedPool, nameLength, PCX_HRWDM_TAG);

        if (NULL == deviceName) {
           ntStatus = STATUS_NO_MEMORY;
           goto clean_exit;
        }

        ntStatus = PcGetDeviceProperty (DeviceObject,
                                        DevicePropertyHardwareID,
                                        nameLength, deviceName, &nameLength);
    }
    if ( ntStatus != STATUS_SUCCESS ) goto clean_exit;

    // Now we get the device name, let's verify which board this is
    // in order to do the appropriate init stuff
    //
    UNICODE_STRING      LcUnicodePciNameRef;
    UNICODE_STRING      LcUnicodeDevName;
    UNICODE_STRING      LcUnicodePciName;
    UNICODE_STRING      LcUnicodeNumString;

    RtlZeroMemory(&LcUnicodePciNameRef,sizeof(UNICODE_STRING));
    RtlInitUnicodeString(&LcUnicodePciNameRef, LcPciInit1);
    ASSERT(LcUnicodePciNameRef.Length==8);

    RtlZeroMemory(&LcUnicodePciName,sizeof(UNICODE_STRING));
    RtlInitUnicodeString(&LcUnicodePciName, LcPciInit2);
    ASSERT(LcUnicodePciName.Length==8);

    RtlZeroMemory(&LcUnicodeDevName,sizeof(UNICODE_STRING));
    RtlInitUnicodeString(&LcUnicodeDevName, deviceName);

    // We copy the full device name into a string initialized
    // to "PCI\\" (hence sized to the length "PCI\\". The
    // copy will have the same maximum length than "PCI\\",
    // but may be not NULL-terminated.
    //
    // Let's manually Null-terminate the copied string in order
    // to get a chance to have a match
    //
    RtlCopyUnicodeString(&LcUnicodePciName, &LcUnicodeDevName);
    RtlZeroMemory(&(LcUnicodePciName.Buffer[(LcUnicodePciName.Length/2)-1]),2);
    LcUnicodePciName.Length -= 2;

    if ( RtlEqualUnicodeString(&LcUnicodePciNameRef,&LcUnicodePciName,FALSE) )
    {
        // ==============================================================
        //
        // PCI board
        //
        // ==============================================================
        MY_PCI_CONFIG           LcPciConfig;
        ULONG                   i;

        // Let's find a free slot for the board
        //
        for ( i = 0 ; i < MAX_BOARD ; i ++ )
        {
            LcPtrBoardInfo = &(LcGeneralInfo->PBoardInfo[i]);
            LcBoardIndex = (WORD) i;
            if (LcPtrBoardInfo->biCarac == CARAC_FREE ) break;
        }

        if (LcPtrBoardInfo->biCarac != CARAC_FREE )
        {
            LcBoardIndex = -1;
            ntStatus = STATUS_DEVICE_CONFIGURATION_ERROR;
            goto clean_exit;
        }

        // Initialize the MY_PCI_CONFIG
        //
        RtlZeroMemory(&LcPciConfig,sizeof(MY_PCI_CONFIG));

        // Retrieve information for the device name string
        // PCI\VEN_xxxx&DEV_yyyy&SUBSYS_ssssvvvv&...
        //         8-11     17-20       29  33-36
        ASSERT(LcUnicodeDevName.Length >= (36 * sizeof(WCHAR)));

        //
        // Initialize a 4-char string
        //
        RtlZeroMemory(&LcUnicodeNumString,sizeof(UNICODE_STRING));
        RtlInitUnicodeString(&LcUnicodeNumString, LcIDInit3);
        ASSERT(LcUnicodeNumString.Length==8);

        // Get VEN_xxxx
        //
        for ( i = 0; i < 4 ; i++ )
        {
            LcUnicodeNumString.Buffer[i]   = LcUnicodeDevName.Buffer[8+i];
        }
        RtlUnicodeStringToInteger( &LcUnicodeNumString, 16, &i );
        LcPciConfig.VendorID = (USHORT) i ;

        // Get DEV_yyyy
        //
        for ( i = 0; i < 4 ; i++ )
        {
            LcUnicodeNumString.Buffer[i] = LcUnicodeDevName.Buffer[17+i];
        }
        RtlUnicodeStringToInteger( &LcUnicodeNumString, 16, &i );
        LcPciConfig.DeviceID = (USHORT) i ;

        // Get vvvv in SUBSYS_ssssvvvv
        //
        for ( i = 0; i < 4 ; i++ )
        {
            LcUnicodeNumString.Buffer[i]   = LcUnicodeDevName.Buffer[33+i];
        }
        RtlUnicodeStringToInteger( &LcUnicodeNumString, 16, &i );
        LcPciConfig.SubVendorID = (USHORT) i ;

        // Get SUBSYS_ssss
        //
        for ( i = 0; i < 4 ; i++ )
        {
            LcUnicodeNumString.Buffer[i]   = LcUnicodeDevName.Buffer[29+i];
        }
        RtlUnicodeStringToInteger( &LcUnicodeNumString, 16, &i );
        LcPciConfig.SubSystemID = (USHORT) i ;

        // Retrieve all information last in the ResourceList
        //
        LOGFileTrace( "-> Indentifying PCI\\VEN_%X&DEV_%X&SUBSYS_%X%X\n",
                            LcPciConfig.VendorID, LcPciConfig.DeviceID,
                            LcPciConfig.SubSystemID, LcPciConfig.SubVendorID);

        // 1st base address ALWAYS hold a memory-mapped space
        //
        if (countMemory >= 2)
        {
            PCM_PARTIAL_RESOURCE_DESCRIPTOR LcMem = ResourceList->FindTranslatedMemory(0);

            // PLX memory mapped (not used)

            LcPciConfig.BaseAddresses[0] = (PBYTE)LcMem->u.Memory.Start.QuadPart;
            LcPciConfig.WindowSize[0]    = LcMem->u.Memory.Length;

            LOGFileTrace( "-> MEM window : @%Xh size=%Xh\n",
                            LcMem->u.Memory.Start.LowPart,
                            LcMem->u.Memory.Length );

            LcMem = ResourceList->FindTranslatedMemory(1);

            // Xilinx/MicroBlaze/DSP

            LcPciConfig.BaseAddresses[2] = (PBYTE)LcMem->u.Memory.Start.QuadPart;
            LcPciConfig.WindowSize[2]    = LcMem->u.Memory.Length;

            LOGFileTrace( "-> MEM window : @%Xh size=%Xh\n",
                            LcMem->u.Memory.Start.LowPart,
                            LcMem->u.Memory.Length );
        }

        // subsequent base addresses ALWAYS hold IO-mapped space
        //
        if ( countIO )
        {
            i = 0;
            PCM_PARTIAL_RESOURCE_DESCRIPTOR pPort = ResourceList->FindTranslatedPort(i);

            LcPciConfig.BaseAddresses[1+i] = (PBYTE)pPort->u.Port.Start.QuadPart;
            LcPciConfig.WindowSize[1+i]    = pPort->u.Port.Length;

            LOGFileTrace( "-> PIO window : @%Xh size=%Xh\n",
                            pPort->u.Port.Start.LowPart,
                            pPort->u.Port.Length );
        }

        if(countIRQ)
        {
            PCM_PARTIAL_RESOURCE_DESCRIPTOR LcIrq = ResourceList->FindTranslatedInterrupt(0);

            // under WinNT/2000, an interrupt vector is interrupt line + 0x30,
            // under Win98, an interrupt vector is interrupt line + 0x50, but
            // WDM model is correctly emulated as far as this aspect is concerned
            LcPciConfig.InterruptLine = LcIrq->u.Interrupt.Vector - 0x30;

            LOGFileTrace( "-> Interrupt %d  (%s and %s Mode)\n",
                          LcPciConfig.InterruptLine,
                          LcIrq->Flags == CM_RESOURCE_INTERRUPT_LATCHED ? "Latched" : "LevelSensitive",
                          LcIrq->ShareDisposition == CmResourceShareShared ? "Shared" : "Private" );
        }

        PCIExtractBoardInformation(
                                &LcPciConfig,
                                LcPtrBoardInfo) ;

        if ( LcPtrBoardInfo->biType == UNKNOWN_BOARD )
        {
            ntStatus = STATUS_DEVICE_CONFIGURATION_ERROR;
            goto clean_exit;
        }
    }
    else
    {
            DOUT (DBG_ERROR, ("Match NOT found"));
            ntStatus = STATUS_NO_SUCH_DEVICE;
            goto clean_exit;
    }

    LOGFileTrace( "-> Board #%d is a %s\n", LcBoardIndex, LcPtrBoardInfo->biBoardName );

    LcMyDeviceExtension->dsdeBoardIndex = LcBoardIndex ;

clean_exit:
    if (deviceName != NULL) {
       DDK_FREE_WITH_TAG(deviceName, PCX_HRWDM_TAG);
    }

    if ( ntStatus != STATUS_SUCCESS )
    {
        LOGFileTrace("... failed (0x%x)\n", ntStatus);
    }

    return ntStatus;
}


#define MAX_DEVICES 32

static PWCHAR g_wave_info[MAX_DEVICES] =
{
    L"Wave01",  L"Wave02",  L"Wave03",  L"Wave04",   L"Wave05",   L"Wave06",   L"Wave07",   L"Wave08",
    L"Wave09",  L"Wave10",  L"Wave11",  L"Wave12",   L"Wave13",   L"Wave14",   L"Wave15",   L"Wave16",
    L"Wave17",  L"Wave18",  L"Wave19",  L"Wave20",   L"Wave21",   L"Wave22",   L"Wave23",   L"Wave24",
    L"Wave25",  L"Wave26",  L"Wave27",  L"Wave28",   L"Wave29",   L"Wave30",   L"Wave31",   L"Wave32"
};

static PWCHAR g_topo_info[MAX_DEVICES] =
{
    L"Topology01", L"Topology02", L"Topology03", L"Topology04", L"Topology05", L"Topology06", L"Topology07", L"Topology08",
    L"Topology09", L"Topology10", L"Topology11", L"Topology12", L"Topology13", L"Topology14", L"Topology15", L"Topology16",
    L"Topology17", L"Topology18", L"Topology19", L"Topology20", L"Topology21", L"Topology22", L"Topology23", L"Topology24",
    L"Topology25", L"Topology26", L"Topology27", L"Topology28", L"Topology29", L"Topology30", L"Topology31", L"Topology32"
};

// get the device configuration for the directsound devices
// from the registry :
static NTSTATUS InitDSConfigurationArray(
    IN      DWORD               boardIndex,
    IN      DWORD               maxBoardAudio,
    INOUT   LPDWORD             subDeviceCount,
    OUT     PSUB_DEVICE_INFO    subDeviceInfoArray)
{
    PAGED_CODE();

    if(*subDeviceCount > MAX_DEVICES)
        return STATUS_UNSUCCESSFUL;

    NTSTATUS ntStatus = STATUS_SUCCESS;
    UNICODE_STRING      uncParamPath;
    UNICODE_STRING      uncDefaultStr;
    static const GUID deviceNameGuid = LXES_DSDEVNAME_GUIDS;

    ASSERT(m_RegistryPathName.Buffer);

    uncParamPath.Length         = 0;
    uncParamPath.MaximumLength  = m_RegistryPathName.Length + 64;    // bytes !!!
    uncParamPath.Buffer         = (PWSTR) DDK_ALLOC( uncParamPath.MaximumLength );

    uncDefaultStr.Length        = 0;
    uncDefaultStr.MaximumLength = 64;
    uncDefaultStr.Buffer        = (PWSTR) DDK_ALLOC( uncDefaultStr.MaximumLength );

    if ( (uncParamPath.Buffer == NULL) || (uncDefaultStr.Buffer == NULL) )
    {
        ntStatus = STATUS_NO_MEMORY;
        goto clean_exit;
    }

    RtlAppendUnicodeStringToString( &uncParamPath, &m_RegistryPathName );
    RtlAppendUnicodeToString( &uncParamPath, L"\\Parameters\\" );
    RtlIntegerToUnicodeString( boardIndex, 10, &uncDefaultStr );
    RtlAppendUnicodeToString( &uncParamPath, uncDefaultStr.Buffer );

    LOGFileTrace("Querying registry for DirectSound configuration\n");

    DWORD dwCount = 0;  // resulting devices

    for( DWORD n=0; n < *subDeviceCount; n++ )
    {
        WCHAR szDeviceXX_Channels[20];
        WCHAR szDeviceXX_Origin[20];
        WCHAR szDeviceXX_Validity[20];

        DWORD   dwParamChannels;
        DWORD   dwParamChannelsDefault = 0;     // invalid value !
        DWORD   dwParamOrigin;
        DWORD   dwParamOriginDefault = 0;       // invalid value !
        DWORD   dwParamValidity;
        DWORD   dwParamValidityDefault = 0;     // invalid !

        // init Wave%d and Topology%d names
        subDeviceInfoArray[dwCount].waveName = g_wave_info[dwCount];
        subDeviceInfoArray[dwCount].topoName = g_topo_info[dwCount];

        RtlStringCbPrintfW( szDeviceXX_Channels,
                            SIZEOF_ARRAY(szDeviceXX_Channels) * sizeof(WCHAR),
                            L"Device%02d_Channels", n);
        RtlStringCbPrintfW( szDeviceXX_Origin,
                            SIZEOF_ARRAY(szDeviceXX_Origin) * sizeof(WCHAR),
                            L"Device%02d_Origin", n);
        RtlStringCbPrintfW( szDeviceXX_Validity,
                            SIZEOF_ARRAY(szDeviceXX_Validity) * sizeof(WCHAR),
                            L"Device%02d_Validity", n);

        RTL_QUERY_REGISTRY_TABLE LcParamTable[4];

        // szDeviceXX_Channels
        LcParamTable[0].QueryRoutine  = (PRTL_QUERY_REGISTRY_ROUTINE) NULL;
        LcParamTable[0].Flags         = RTL_QUERY_REGISTRY_DIRECT;
        LcParamTable[0].Name          = szDeviceXX_Channels;
        LcParamTable[0].EntryContext  = (PVOID)&dwParamChannels;
        LcParamTable[0].DefaultType   = REG_DWORD;
        LcParamTable[0].DefaultData   = (PVOID)&dwParamChannelsDefault ;
        LcParamTable[0].DefaultLength = sizeof (dwParamChannelsDefault);

        // szDeviceXX_Origin
        LcParamTable[1].QueryRoutine  = (PRTL_QUERY_REGISTRY_ROUTINE) NULL;
        LcParamTable[1].Flags         = RTL_QUERY_REGISTRY_DIRECT;
        LcParamTable[1].Name          = szDeviceXX_Origin;
        LcParamTable[1].EntryContext  = (PVOID)&dwParamOrigin;
        LcParamTable[1].DefaultType   = REG_DWORD;
        LcParamTable[1].DefaultData   = (PVOID)&dwParamOriginDefault ;
        LcParamTable[1].DefaultLength = sizeof (dwParamOriginDefault);

        // szDeviceXX_Validity
        LcParamTable[2].QueryRoutine  = (PRTL_QUERY_REGISTRY_ROUTINE) NULL;
        LcParamTable[2].Flags         = RTL_QUERY_REGISTRY_DIRECT;
        LcParamTable[2].Name          = szDeviceXX_Validity;
        LcParamTable[2].EntryContext  = (PVOID)&dwParamValidity;
        LcParamTable[2].DefaultType   = REG_DWORD;
        LcParamTable[2].DefaultData   = (PVOID)&dwParamValidityDefault ;
        LcParamTable[2].DefaultLength = sizeof (dwParamValidityDefault);

        // NULL-terminate the ParamTable
        // ------------------------------
        BZERO2( &LcParamTable[3], LcParamTable[3], 1 );

        ntStatus = RtlQueryRegistryValues(
                        RTL_REGISTRY_ABSOLUTE | RTL_REGISTRY_OPTIONAL,
                        uncParamPath.Buffer,
                        LcParamTable,
                        NULL,
                        NULL );

        if ( !NT_SUCCESS( ntStatus ) )
        {
            DOUT(DBG_ERROR, ("Failed to query registry (0x%lx)\n", ntStatus));
            break ;
        }
        if( (dwParamChannels == 0) || (dwParamOrigin == 0) )
        {
            // wrong or invalid entry. Stop enumeration
            break;
        }
        dwParamOrigin--;    // 1-based in registry (and application) but 0-based in the driver

        if( (dwParamChannels + dwParamOrigin) > maxBoardAudio )
        {
            // wrong or invalid entry. Stop enumeration
            break;
        }

        if( dwParamValidity )
        {
            subDeviceInfoArray[dwCount].channelCount = (WORD) dwParamChannels;
            subDeviceInfoArray[dwCount].channelFirst = (WORD) dwParamOrigin;

            // create corresponding GUID for KSCOMPONENTID.Name requests:
            subDeviceInfoArray[dwCount].deviceNameGuid = deviceNameGuid;
            subDeviceInfoArray[dwCount].deviceNameGuid.Data1 += ((boardIndex * LXES_MAX_GUID_PER_CARD) + n);

            dwCount++;
        }
    }
    if( dwCount == 0 )    // no config found : apply default setting
    {
        subDeviceInfoArray[0].channelCount = 2;
        subDeviceInfoArray[0].channelFirst = 0;

        if(!IoIsWdmVersionAvailable(6, 0x00))   // Windows XP
        {
            subDeviceInfoArray[0].deviceNameGuid = GUID_NULL;   // use friendly name of .inf file
        }
        else    // VISTA
        {
            // create corresponding GUID for Pin names and KSCOMPONENTID.Name requests:
            subDeviceInfoArray[0].deviceNameGuid = deviceNameGuid;
            subDeviceInfoArray[0].deviceNameGuid.Data1 += (boardIndex * LXES_MAX_GUID_PER_CARD);
        }

        LOGFileTrace("No DirectSound configuration entries found. Use 1 default stereo device\n");

        dwCount=1;
    }
    else
    {
        LOGFileTrace("Found %d valid DirectSound configuration entries\n", dwCount);
    }

    ntStatus = STATUS_SUCCESS;
    *subDeviceCount = dwCount;

clean_exit:
    if(uncParamPath.Buffer)  DDK_FREE( uncParamPath.Buffer );
    if(uncDefaultStr.Buffer) DDK_FREE( uncDefaultStr.Buffer );

    return ntStatus;
}


/*****************************************************************************
 * StartDevice
 *****************************************************************************
 * This function is called by the operating system when the device is started.
 * It is responsible for starting the miniports.  This code is specific to
 * the adapter because it calls out miniports for functions that are specific
 * to the adapter.
 */
NTSTATUS StartDevice
(
    IN  PDEVICE_OBJECT  DeviceObject,   // Device object.
    IN  PIRP            Irp,            // IO request packet.
    IN  PRESOURCELIST   ResourceList    // List of hardware resources.
)
{
    PAGED_CODE ();

    ASSERT (DeviceObject);
    ASSERT (Irp);
    ASSERT (ResourceList);

    NTSTATUS ntStatus;
    PDSOUND_DEVICE_EXTENSION    LcMyDeviceExtension ;

    LOGFileAppend( (PVOID)&m_RegistryPathName );

    LOGFileTrace("[StartDevice]\n");

    ASSERT (DeviceObject->DeviceExtension);
    LcMyDeviceExtension = (PDSOUND_DEVICE_EXTENSION) DeviceObject->DeviceExtension;

    // Init the extension
    //
    LcMyDeviceExtension->dsdeGeneralInfo = &m_GeneralInfo;
    LcMyDeviceExtension->dsdeBoardIndex  = MAX_BOARD;
    LcMyDeviceExtension->dsdeSpinLockPtr = &m_GeneralInfo.DpcIoctlSpinLock;
    LcMyDeviceExtension->dsdeSemaphorePtr= &m_GeneralInfo.Semaphore ;

    //
    // Validate the resources.
    // We don't have to split the resources into several resource lists cause
    // the topology miniport doesn't need a resource list, the wave pci miniport
    // needs all resources like the adapter common object.
    //
    ntStatus = ValidateResources (DeviceObject, ResourceList);

    //
    // return immediately in case of an error
    //
    if (!NT_SUCCESS (ntStatus))
	{
        LOGFileClose();
        return ntStatus;
	}

    //
    //
    // If the adapter has the right resources...
    //
    PADAPTERCOMMON pAdapterCommon = NULL;
    PUNKNOWN       pUnknownCommon;

    // create a new adapter common object
    ntStatus = NewAdapterCommon (&pUnknownCommon,
                                 IID_IAdapterCommon,
                                 NULL,
                                 NonPagedPool);

    if (NT_SUCCESS (ntStatus))
    {
        // query for the IAdapterCommon interface
        ntStatus = pUnknownCommon->QueryInterface (IID_IAdapterCommon,
                                                   (PVOID *)&pAdapterCommon);
        if (NT_SUCCESS (ntStatus))
        {
            KeWaitForSingleObject(  LcMyDeviceExtension->dsdeSemaphorePtr,
                                    Executive,
                                    KernelMode,
                                    FALSE,
                                    NULL );
            // Initialize the object
            ntStatus = pAdapterCommon->Init (ResourceList,
                                             DeviceObject,
                                             LcMyDeviceExtension->dsdeGeneralInfo,
                                             LcMyDeviceExtension->dsdeBoardIndex );

            KeReleaseSemaphore( LcMyDeviceExtension->dsdeSemaphorePtr,
                                0,          // priority increment
                                1,          // semaphore counter increment
                                FALSE );    // immediate execution of next waiting thread

            if (NT_SUCCESS (ntStatus))
            {
                // register with PortCls for power-management services
                ntStatus = PcRegisterAdapterPowerManagement ((PUNKNOWN)pAdapterCommon,
                                                             DeviceObject);
            }
        }

        // release the IID_IAdapterCommon on adapter common
        pUnknownCommon->Release ();
    }

    // print error message.
    if (!NT_SUCCESS (ntStatus))
    {
        DOUT (DBG_ERROR, ("Could not create or query AdapterCommon."));
    }
    else
    {
        KeWaitForSingleObject(  LcMyDeviceExtension->dsdeSemaphorePtr,
                                Executive,
                                KernelMode,
                                FALSE,
                                NULL );

        ntStatus = pAdapterCommon->InitDSP( DeviceObject,
                                             LcMyDeviceExtension->dsdeGeneralInfo,
                                             LcMyDeviceExtension->dsdeBoardIndex );

        KeReleaseSemaphore( LcMyDeviceExtension->dsdeSemaphorePtr,
                            0,          // priority increment
                            1,          // semaphore counter increment
                            FALSE );    // immediate execution of next waiting thread
    }

    //
    // These are the port driver pointers we are keeping around for registering
    // physical connections.
    //
    PUNKNOWN                *unknownWave            = NULL;
    PUNKNOWN                *unknownTopology        = NULL;

    PMINIPORTTOPOLOGYICH    *pMiniportTopologyICH   = NULL;
    PMINIPORTWAVEICH        *pMiniportWave          = NULL;

    ULONG                   subDeviceCount = 0;
    ULONG                   DeviceIndex    = 0;

    PSUB_DEVICE_INFO        subDeviceInfoArray = NULL;

    ULONG maxBoardAudio = pAdapterCommon->GetMaxWaveInOutNumber() * 2;

    if (!NT_SUCCESS (ntStatus))
    {
        goto    clean_exit;
    }

    // maximum number of possible directsound devices
    //
    subDeviceCount = MAX_DEVICES;

    subDeviceInfoArray = (PSUB_DEVICE_INFO) DDK_ALLOC_WITH_TAG(
                                                PagedPool,
                                                subDeviceCount * sizeof(SUB_DEVICE_INFO),
												PCX_HRWDM_TAG);
    if( !subDeviceInfoArray )
        goto clean_exit;

    RtlZeroMemory(subDeviceInfoArray, subDeviceCount * sizeof(SUB_DEVICE_INFO));

    ntStatus = InitDSConfigurationArray( LcMyDeviceExtension->dsdeBoardIndex, maxBoardAudio, &subDeviceCount, subDeviceInfoArray );
    if (!NT_SUCCESS (ntStatus))
	{
        goto clean_exit;
	}

    DOUT (DBG_PRINT, ("NP88 POWER UP board=%d\n", LcMyDeviceExtension->dsdeBoardIndex));

    unknownWave = (PUNKNOWN *) DDK_ALLOC_WITH_TAG(
                                                NonPagedPool,
                                                subDeviceCount * sizeof(PUNKNOWN),
												PCX_HRWDM_TAG);
    if (unknownWave)
        RtlZeroMemory(unknownWave,subDeviceCount * sizeof(PUNKNOWN));

    unknownTopology = (PUNKNOWN *) DDK_ALLOC_WITH_TAG(
                                                NonPagedPool,
                                                subDeviceCount * sizeof(PUNKNOWN),
												PCX_HRWDM_TAG);
    if (unknownTopology)
        RtlZeroMemory(unknownTopology,subDeviceCount * sizeof(PUNKNOWN));

    pMiniportTopologyICH = (PMINIPORTTOPOLOGYICH *) DDK_ALLOC_WITH_TAG(
                                                NonPagedPool,
                                                subDeviceCount * sizeof(PMINIPORTTOPOLOGYICH),
												PCX_HRWDM_TAG);
    if (pMiniportTopologyICH)
        RtlZeroMemory(pMiniportTopologyICH,subDeviceCount * sizeof(PMINIPORTTOPOLOGYICH));

    pMiniportWave = (PMINIPORTWAVEICH *) DDK_ALLOC_WITH_TAG(
                                                NonPagedPool,
                                                subDeviceCount * sizeof(PMINIPORTWAVEICH),
												PCX_HRWDM_TAG);
    if (pMiniportWave)
        RtlZeroMemory(pMiniportWave,subDeviceCount * sizeof(PMINIPORTWAVEICH));

    if ( ! (unknownWave && unknownTopology && pMiniportTopologyICH && pMiniportWave ) )
    {
        goto clean_exit;
    }

    // Find first Device that will connect the interrupt
    //
    DWORD LcMasterDevice = UTIDWMask2Word(~m_GeneralInfo.HideDeviceMask);

    if(LcMasterDevice >= subDeviceCount)
    {
        DOUT(DBG_ERROR, ("ERROR Registry Value HideDeviceMask = %x : we need min one Device for interrupts!\n", m_GeneralInfo.HideDeviceMask));
        ntStatus = STATUS_UNSUCCESSFUL;
        goto clean_exit;
    }

    for ( DeviceIndex = 0; DeviceIndex < subDeviceCount ; DeviceIndex++)
    {
        //
        // These are the port driver pointers we are keeping around for registering
        // physical connections.
        //
        PMINIPORT               miniTopology            = NULL;
        PMINIPORT               miniWave                = NULL;
        PSUB_DEVICE_INFO        LcSubDeviceInfo         = &subDeviceInfoArray[DeviceIndex];
        PPORT                   LcPort                  = NULL;

        // Hide this device ?
        if( m_GeneralInfo.HideDeviceMask & (1<<DeviceIndex) )
            continue;

        //
        // Start the topology miniport.
        //
        if (NT_SUCCESS (ntStatus))
        {
            ntStatus = InstallSubdevice (DeviceObject,
                                         Irp,
                                         CLSID_PortTopology,
                                         CLSID_PortTopology,
                                         CreateMiniportTopologyICH,
                                         pAdapterCommon,
                                         NULL,
                                         GUID_NULL,
                                         &miniTopology,
                                         &LcPort,
                                         &(unknownTopology[DeviceIndex]));

            if (NT_SUCCESS (ntStatus))
            {
                // query for the IMiniportTopologyICH interface
                ntStatus = miniTopology->QueryInterface (IID_IMiniportTopologyICH,
                                                        (PVOID *)&pMiniportTopologyICH[DeviceIndex]);
                if (NT_SUCCESS (ntStatus))
                {
                    PMINIPORTTOPOLOGYICH LcMyPort = pMiniportTopologyICH[DeviceIndex];
                    //
                    // Init the port driver and miniport in one go.
                    //
                    LcMyPort->MyVery1stInit ( (USHORT) DeviceIndex,
                                              LcMyDeviceExtension->dsdeBoardIndex,
                                              LcSubDeviceInfo->channelFirst,
                                              LcSubDeviceInfo->channelCount,
                                              &LcSubDeviceInfo->deviceNameGuid
                                             );

                    ntStatus = LcPort->Init ( DeviceObject,
                                              Irp,
                                              miniTopology,
                                              pAdapterCommon,
                                              ResourceList );

                    if (NT_SUCCESS (ntStatus))
                    {
                        //
                        // Register the subdevice (port/miniport combination).
                        //
                        ntStatus = PcRegisterSubdevice (DeviceObject,
                                                        LcSubDeviceInfo->topoName,
                                                        LcPort);

                        if (!NT_SUCCESS (ntStatus))
                        {
                            DOUT (DBG_ERROR, ("**** Miniport <<%S>> failed to register (0x%lx)!\n", LcSubDeviceInfo->topoName, ntStatus));
                        }
                    }

                    LcPort->Release ();
                    LcPort = NULL;
                }

                miniTopology->Release ();
                miniTopology = NULL;

            }

            // print error message.
            if (!NT_SUCCESS (ntStatus))
            {
                DOUT (DBG_ERROR, ("Could not create or query TopologyICH"));
            }
            else
            {
                DOUT (DBG_PRINT, ("TopologyICH miniport <<%S>> installed!\n", LcSubDeviceInfo->topoName));
            }
        }

        //
        // Start the wave miniport.
        //
        if (NT_SUCCESS (ntStatus))
        {
            ntStatus = InstallSubdevice (DeviceObject,
                                         Irp,
                                         CLSID_PortWavePci,
                                         CLSID_PortWavePci,
                                         CreateMiniportWaveICH,
                                         pAdapterCommon,
                                         NULL,
                                         GUID_NULL,
                                         &miniWave,
                                         &LcPort,
                                         &(unknownWave[DeviceIndex]));

            if (NT_SUCCESS (ntStatus))
            {
                // query for the IMiniportWaveICH interface
                ntStatus = miniWave->QueryInterface (IID_IMiniportWaveICH,
                                                     (PVOID *)&pMiniportWave[DeviceIndex]);
                if (NT_SUCCESS (ntStatus))
                {
                    PMINIPORTWAVEICH LcMyPort = pMiniportWave[DeviceIndex];
                    //
                    // Init the port driver and miniport in one go.
                    //
                    LcMyPort->MyVery1stInit ( (USHORT) DeviceIndex,
                                              LcMyDeviceExtension->dsdeBoardIndex,
                                              LcSubDeviceInfo->channelFirst,
                                              LcSubDeviceInfo->channelCount,
                                              &LcSubDeviceInfo->deviceNameGuid,
                                              (LcMasterDevice == DeviceIndex)    // ConnectInterrupt ?
                                              );

                    ntStatus = LcPort->Init ( DeviceObject,
                                              Irp,
                                              miniWave,
                                              pAdapterCommon,
                                              ResourceList );

                    if (NT_SUCCESS (ntStatus))
                    {
                        //
                        // Register the subdevice (port/miniport combination).
                        //
                        ntStatus = PcRegisterSubdevice (DeviceObject,
                                                        LcSubDeviceInfo->waveName,
                                                        LcPort);

                        if (!NT_SUCCESS (ntStatus))
                        {
                            DOUT (DBG_ERROR, ("**** Miniport <<%S>> failed to register (0x%lx)!\n", LcSubDeviceInfo->waveName, ntStatus));
                        }
                    }

                    LcPort->Release ();
                    LcPort = NULL;
                }

                miniWave->Release ();
                miniWave = NULL;

            }

            // print error message.
            if (!NT_SUCCESS (ntStatus))
            {
                DOUT (DBG_ERROR, ("Could not create or query Wave"));
            }
            else {
                DOUT (DBG_PRINT, ("WaveICH miniport <<%S>> installed!\n", LcSubDeviceInfo->waveName));

                // ## FS (02/25/2000) -- Remember the topology x-ref.
                //
                pMiniportWave[DeviceIndex]->SetTopologyRef(pMiniportTopologyICH[DeviceIndex]) ;
            }
        }

        //
        // Establish physical connections between filters as shown.
        //
        //              +---------------------+    +----------------+
        //              |         Wave        |    |       Topo     |
        //  Capture <---|2     (WAVEIN_BRIDGE)|<===|(WAVE_IN)       |<--- Line in
        //              |                     |    |                |
        //   Render --->|0    (WAVEOUT_BRIDGE)|===>|(WAVE_OUT)      |---> Line Out
        //              |                     |    |                |
        //              +---------------------+    +----------------+
        //

        // Note that the pin numbers for the nodes to be connected
        // may vary depending on the hardware/codec configuration.
        // Also, the mic input may or may not be present.
        //
        // So,
        //      Do a QI on unknownTopology to get an interface to call
        //          a method on to get the topology miniport pin IDs.

        if (NT_SUCCESS (ntStatus))
        {
            ULONG   l_ulPinTopoNumber;
            ULONG   l_ulPinWaveNumber;

            // get the pin numbers.
            DOUT (DBG_PRINT, ("Connecting topo and wave\n"));

            // register all wave render connection
            if (NT_SUCCESS (ntStatus))
            {
                // get topo pin number
                ntStatus = pMiniportTopologyICH[DeviceIndex]->GetPhysicalConnectionPins (
                                                    PIN_WAVEOUT_SOURCE,
                                                    &l_ulPinTopoNumber
                                                    );
                if (   NT_SUCCESS (ntStatus)
                    && (l_ulPinTopoNumber != -1) )
                {
                    // get wave pin number
                    ntStatus = pMiniportWave[DeviceIndex]->GetPhysicalConnectionPins (
                                                        WAVE_PIN_WAVEOUT_BRIDGE,
                                                        &l_ulPinWaveNumber
                                                        );

                    ntStatus = PcRegisterPhysicalConnection (DeviceObject,
                                                             unknownWave[DeviceIndex],
                                                             l_ulPinWaveNumber,
                                                             unknownTopology[DeviceIndex],
                                                             l_ulPinTopoNumber);
                }
                else
                {
                    if ( DeviceIndex != 0 )
                    {
                        // print information message.
                        DOUT (DBG_PRINT, ("Skipping connection between topology and wave miniport"
                                          " (render) for device %d!\n", DeviceIndex));
                    }
                    else
                    {
                        // print error message.
                        DOUT (DBG_ERROR, ("Cannot connect topology and wave miniport"
                                          " (render)! Status = 0x%lx", ntStatus));
                    }
                }

            }

            // register wave capture connection
            if (NT_SUCCESS (ntStatus))
            {
                // get topo pin number
                ntStatus = pMiniportTopologyICH[DeviceIndex]->GetPhysicalConnectionPins (
                                                    PIN_WAVEIN_DEST,
                                                    &l_ulPinTopoNumber
                                                    );

                if (   NT_SUCCESS (ntStatus)
                    && (l_ulPinTopoNumber != -1) )
                {
                    // get wave pin number
                    ntStatus = pMiniportWave[DeviceIndex]->GetPhysicalConnectionPins (
                                                        WAVE_PIN_WAVEIN_BRIDGE,
                                                        &l_ulPinWaveNumber
                                                        );

                    // register wave capture connection
                    ntStatus = PcRegisterPhysicalConnection (DeviceObject,
                                                             unknownTopology[DeviceIndex],
                                                             l_ulPinTopoNumber,
                                                             unknownWave[DeviceIndex],
                                                             l_ulPinWaveNumber);
                }
                else
                {
                    if ( DeviceIndex != 0 )
                    {
                        // print information message.
                        DOUT (DBG_PRINT, ("Skipping connection between topology and wave miniport"
                                          " (capture) for device %d!\n", DeviceIndex));
                    }
                    else
                    {
                        // print error message.
                        DOUT (DBG_ERROR, ("Cannot connect topology and wave miniport"
                                          " (capture)! Status = 0x%lx", ntStatus));
                    }
                }
            }

        }

    } // end for DeviceIndex

clean_exit:

    //
    // Release the adapter common object.  It either has other references,
    // or we need to delete it anyway.
    //
    if (pAdapterCommon)
        pAdapterCommon->Release ();

    //
    // Release the unknowns.
    //
    if (unknownTopology)
    {
        for ( DeviceIndex = 0; DeviceIndex < subDeviceCount ; DeviceIndex++)
        {
            if (unknownTopology[DeviceIndex])
                unknownTopology[DeviceIndex]->Release ();
        }
        DDK_FREE_WITH_TAG(unknownTopology, PCX_HRWDM_TAG);
    }
    if (unknownWave)
    {
        for ( DeviceIndex = 0; DeviceIndex < subDeviceCount ; DeviceIndex++)
        {
            if (unknownWave[DeviceIndex])
                unknownWave[DeviceIndex]->Release ();
        }
        DDK_FREE_WITH_TAG(unknownWave, PCX_HRWDM_TAG);
    }

    // and the ICH miniport.
    if (pMiniportTopologyICH)
    {
        for ( DeviceIndex = 0; DeviceIndex < subDeviceCount ; DeviceIndex++)
        {
            if (pMiniportTopologyICH[DeviceIndex])
                pMiniportTopologyICH[DeviceIndex]->Release ();
        }
        DDK_FREE_WITH_TAG(pMiniportTopologyICH, PCX_HRWDM_TAG);
    }
    if (pMiniportWave)
    {
        for ( DeviceIndex = 0; DeviceIndex < subDeviceCount ; DeviceIndex++)
        {
            if (pMiniportWave[DeviceIndex])
                pMiniportWave[DeviceIndex]->Release ();
        }
        DDK_FREE_WITH_TAG(pMiniportWave, PCX_HRWDM_TAG);
    }
    if(subDeviceInfoArray)
    {
        DDK_FREE_WITH_TAG(subDeviceInfoArray, PCX_HRWDM_TAG);
    }

    LOGFileClose();

    return ntStatus;    // whatever this is ...
}

/*****************************************************************************
 * AddDevice
 *****************************************************************************
 * This function is called by the operating system when the device is added.
 * All adapter drivers can use this code without change.
 */
NTSTATUS AddDevice
(
    IN PDRIVER_OBJECT   DriverObject,
    IN PDEVICE_OBJECT   PhysicalDeviceObject
)
{
    PAGED_CODE ();

    //
    // Tell the class driver to add the device.
    //
    ASSERT(sizeof(DSOUND_DEVICE_EXTENSION) >= PORT_CLASS_DEVICE_EXTENSION_SIZE);

    return PcAddAdapterDevice(  DriverObject,
                                PhysicalDeviceObject,
                                (PCPFNSTARTDEVICE)StartDevice,
                                MAX_MINIPORTS,
                                sizeof(DSOUND_DEVICE_EXTENSION) );
}

#pragma code_seg()

// ****************************************************************************
// STATIC VOID      ProtectAndSync()
// ************************************
//
// Input parameters :
// ******************
//
//      PDEVICE_EXTENSION_INFO  PmDeviceExtensionPtr:   the device extension
//      PKIRQL                  PmOldIrqlPtr        :   storage for saving IRQL
//
// ****************************************************************************
//
// Protect Hardware interrupts to disturb processing from this point
//
// ****************************************************************************
extern "C" VOID ProtectAndSync(
    PDSOUND_DEVICE_EXTENSION    PmDeviceExtensionPtr,
    PKIRQL                  PmOldIrqlPtr        )
{
    // Note :
    //  KeRaiseIrql( DISPATCH_LEVEL, &LcOldIrql ) will not work
    // properly on a multiprocessor node. KeAcquireSpinLock raises
    // the IRQL to DISPATCH_LEVEL and allow to define a critical section
    // -----------------------------------------------------------------
    KeAcquireSpinLock( PmDeviceExtensionPtr->dsdeSpinLockPtr, PmOldIrqlPtr );
}

// ****************************************************************************
// STATIC VOID      UnProtectAndSync()
// **************************************
//
// Input parameters :
// ******************
//
//      PDEVICE_EXTENSION_INFO  PmDeviceExtensionPtr:   the device extension
//      PKIRQL                  PmOldIrqlPtr        :   storage for saving IRQL
//
// ****************************************************************************
//
// Allow again Hardware interrupts to disturb processing at this point
//
// ****************************************************************************
extern "C" VOID UnProtectAndSync(
    PDSOUND_DEVICE_EXTENSION    PmDeviceExtensionPtr,
    PKIRQL                  PmOldIrqlPtr        )
{
    KeReleaseSpinLock( PmDeviceExtensionPtr->dsdeSpinLockPtr, *PmOldIrqlPtr );
}


#pragma code_seg("PAGE")


/*****************************************************************************
 * NP88Dispatch* wrappers
 *****************************************************************************/

static VOID
NP88WrapUnload(IN PDRIVER_OBJECT pDriverObject)
{
    PAGED_CODE ();

    DOUT (DBG_PRINT, ("[NP88WrapUnload called]"));

    // free resources allocated in DriverEntry
    if ( m_RegistryPathName.Buffer )
    {
        DDK_FREE_WITH_TAG(m_RegistryPathName.Buffer, PCX_HRWDM_TAG);
		m_RegistryPathName.Buffer = NULL;
    }
    if( m_GeneralInfo.ReplyBuffer )
    {
        DDK_FREE_WITH_TAG(m_GeneralInfo.ReplyBuffer, PCX_HRWDM_TAG);
		m_GeneralInfo.ReplyBuffer = NULL;
    }

    // Unmap and release memory space
    // ------------------------------
    DOUT(DBG_PRINT, ("-> Unmapping buffers\n"));

    BUFFreeAllBuffers( &m_GeneralInfo );

    ESCmdsUnload( NULL );

    if( m_PortClassUnload != NULL )
    {
        (*m_PortClassUnload)(pDriverObject);
    }
}


#pragma code_seg()


static NTSTATUS
NP88Dispatch(
    IN PDEVICE_OBJECT   PmDeviceObjectPtr,
    IN PIRP             PmIrpPtr,
    IN DWORD            PmReplySize
)
{
    NTSTATUS LcNtStatus;
    LPDWORD                 LcRequestBlock  ;
    BOOLEAN                 LcSyncCmd       = FALSE ;
    KIRQL                   LcOldIrql       ;

    PmIrpPtr->IoStatus.Status      = STATUS_SUCCESS;
    PmIrpPtr->IoStatus.Information = 0;

    // Synchronize execution of DPCs and others incoming DevIoctl
    // ----------------------------------------------------------
    PDSOUND_DEVICE_EXTENSION LcDeviceExtension = (PDSOUND_DEVICE_EXTENSION) PmDeviceObjectPtr->DeviceExtension;

    KeWaitForSingleObject(  LcDeviceExtension->dsdeSemaphorePtr,
                            Executive,
                            KernelMode,
                            FALSE,
                            NULL );

    DOUT (DBG_SYNCNPAPI, ("[NP88DispatchWrapIoctl] -- Acces NP88"));

    ProtectAndSync( LcDeviceExtension, &LcOldIrql );

    {
        // pick out parameters from irp
        // FS - 28/02/1997
        // if DIRECT_IO get address from
        //               MmGetSystemAddressForMdl(PmIrp->MdlAddress)
        // if BUFFERED_IO then use Irp->AssociatedIrp.SystemBuffer
        // ----------------------------
        LcRequestBlock = (LPDWORD) PmIrpPtr->AssociatedIrp.SystemBuffer;
        //LcRequestSize  = PmIrpStack->Parameters.DeviceIoControl.InputBufferLength ;
        //LcReplySize    = PmIrpStack->Parameters.DeviceIoControl.OutputBufferLength;

        PVOID ReplyBuffer = LcDeviceExtension->dsdeGeneralInfo->ReplyBuffer;

        // Let's call the software interrupt
        // BEWARE: request and reply blocks
        // are mixed together
        // ---------------------------------
        // We'd better clean up the reply buffer to make sure
        // the APHDispatch2(..) will not see rubbish data if an
        // error occurs before the reply block is correctly initialized
        // fix for FA #107
        // -----------------------------------------------------------

        RtlZeroMemory( ReplyBuffer, PmReplySize );
        LcSyncCmd = APHDispatch( LcRequestBlock                       ,
                         (LPDWORD) ReplyBuffer ,
                         (PDWORD) PmIrpPtr                            );

        // Copy back the reply block into the exchange block
        // for synchronous commands
        // -------------------------------------------------
        if (LcSyncCmd)
        {
            RtlCopyBytes( (PVOID) LcRequestBlock             ,
                          ReplyBuffer ,
                          PmReplySize                        );
        }

        // say how much we have written back
        // ---------------------------------
        PmIrpPtr->IoStatus.Information = PmReplySize ;

    }

    // set Irp's status
    // ----------------
    LcNtStatus = PmIrpPtr->IoStatus.Status;

	if( LcNtStatus == STATUS_PENDING )
	{
		// Mark the IRP as pending
		// -----------------------
		IoMarkIrpPending( PmIrpPtr );
	}

    // allow DPC again
    // ---------------
    UnProtectAndSync( LcDeviceExtension, &LcOldIrql );

    KeReleaseSemaphore( LcDeviceExtension->dsdeSemaphorePtr,
                        0,          // priority increment
                        1,          // semaphore counter increment
                        FALSE );    // immediate execution of next waiting thread

    // called here for info functions where the result is already in the parameter block
    // or called in the DPC for recording and playback coz the Irp disappears after this call
    // -----------------------------------------------
    if ( LcSyncCmd )
    {
        DOUT (DBG_SYNCNPAPI, ("[SYNC Dispatch2]"));

        // FS _ 28/01/97
        // Eventually handle posted mapping requests
        // -----------------------------------------
        APHDispatch2( LcRequestBlock );

        IoCompleteRequest( PmIrpPtr, IO_NO_INCREMENT );
    }

    DOUT (DBG_SYNCNPAPI, ("[NP88DispatchWrapIoctl done]"));

    return (LcNtStatus);
}


static NTSTATUS ESDispatch( IN PDEVICE_OBJECT   PmDeviceObjectPtr,
                            IN PIRP             PmIrpPtr,
                            IN DWORD            PmRequestSize,
                            IN DWORD            PmReplySize     )
{
    NTSTATUS    status;
    DWORD       reply_size = 0;

    // Synchronize execution of DPCs and others incoming DevIoctl
    // ----------------------------------------------------------
    PDSOUND_DEVICE_EXTENSION LcDeviceExtension = (PDSOUND_DEVICE_EXTENSION) PmDeviceObjectPtr->DeviceExtension;

    DOUT( DBG_SYNCNPAPI, ("==> ESDispatch\n"));

    if ( (PmRequestSize != sizeof(ES_DRV_CMD)) || (PmReplySize != sizeof(ES_DRV_RESP)) )
    {
        status = STATUS_INVALID_PARAMETER;
    }
    else
    {
        status = ESCmdsDispatch( PmIrpPtr, LcDeviceExtension );

        // say how much we have written back
        // ---------------------------------
        if( status == STATUS_SUCCESS )
        {
            reply_size = sizeof(ES_DRV_RESP);
        }
    }

    if( status != STATUS_PENDING )
    {
        // complete the Irp
        //
        PmIrpPtr->IoStatus.Status = status;
        PmIrpPtr->IoStatus.Information = reply_size;
        IoCompleteRequest( PmIrpPtr, IO_NO_INCREMENT );
    }

    DOUT( DBG_SYNCNPAPI, ("<== ESDispatch\n"));

    return status;
}


#pragma code_seg("PAGE")

static NTSTATUS
NP88DispatchWrapIoctl(
    IN PDEVICE_OBJECT   pDeviceObject,
    IN PIRP             pIrp
)
{
    PAGED_CODE();

    ASSERT(pDeviceObject);
    ASSERT(pIrp);

    NTSTATUS ntStatus = STATUS_SUCCESS;

    PIO_STACK_LOCATION pIrpStack = IoGetCurrentIrpStackLocation(pIrp);

    if( pIrpStack->MajorFunction == IRP_MJ_DEVICE_CONTROL )
    {
        ULONG   LcIoControlCode = pIrpStack->Parameters.DeviceIoControl.IoControlCode;

        if( LcIoControlCode == IOCTL_PCX_COMMAND )
        {
            DWORD LcReplySize = pIrpStack->Parameters.DeviceIoControl.OutputBufferLength;

            ntStatus = NP88Dispatch(pDeviceObject, pIrp, LcReplySize);

            return ntStatus;
        }
        else if( LcIoControlCode == IOCTL_ES_COMMAND )
        {
            DWORD LcRequestSize = pIrpStack->Parameters.DeviceIoControl.InputBufferLength;
            DWORD LcReplySize = pIrpStack->Parameters.DeviceIoControl.OutputBufferLength;

            ntStatus = ESDispatch(pDeviceObject, pIrp, LcRequestSize, LcReplySize);

            return ntStatus;
        }
    }

    //
    // Pass the IRPs on to PortCls
    //
    ntStatus = PcDispatchIrp( pDeviceObject, pIrp );

    return ntStatus;
}


#pragma code_seg("INIT")


/*****************************************************************************
 * DriverEntry
 *****************************************************************************
 * This function is called by the operating system when the driver is loaded.
 * All adapter drivers can use this code without change.
 */
extern "C" NTSTATUS DriverEntry
(
    IN PDRIVER_OBJECT   DriverObject,
    IN PUNICODE_STRING  RegistryPathName
)
{
    PAGED_CODE ();

    // now minimum is Windows XP or higher
    // TEST W2K !!!
    if (!IoIsWdmVersionAvailable(1,0x10))
    {
        return STATUS_UNSUCCESSFUL;
    }

    LOGFileOpen( (PVOID) RegistryPathName );

    LOGFileTrace("[DriverEntry]\n") ;

    LOGFileTrace("-> Driver version %s\n", PCX_PRODUCTVERSION_STR ) ;

    m_RegistryPathName.MaximumLength = 0;
    m_RegistryPathName.Length = 0;
    m_RegistryPathName.Buffer = (PWCHAR) DDK_ALLOC_WITH_TAG(
                                                PagedPool,
                                                RegistryPathName->Length + sizeof(WCHAR),
												PCX_HRWDM_TAG);
    if ( m_RegistryPathName.Buffer == NULL )
    {
        return STATUS_NO_MEMORY;
    }
    m_RegistryPathName.MaximumLength = RegistryPathName->Length + sizeof(WCHAR);

    RtlCopyUnicodeString( &m_RegistryPathName, RegistryPathName );
    m_RegistryPathName.Buffer[RegistryPathName->Length / 2] = 0;    // add a null terminator


    // init our global static arrays and structs
    //
    BZERO2 ( &m_GeneralInfo, GENERAL_INFO, 1 );

    BUFInit( &m_GeneralInfo.PGeneralBuffer,
             &m_GeneralInfo.PBufferInfo );
    APHInit( &m_GeneralInfo.PDriverInfo,
             &m_GeneralInfo.PBoardInfo );

    KeInitializeSpinLock(&m_GeneralInfo.DpcIoctlSpinLock);
    KeInitializeSemaphore(&m_GeneralInfo.Semaphore,1,1) ;  // init count , max

    // Retrieve information about resources need according to the driver
    // configuration
    //
    NTSTATUS ntStatus = GetParametersFromRegistry( &m_RegistryPathName, &m_GeneralInfo );
    if ( !NT_SUCCESS( ntStatus ) )
    {
        goto _clean_exit;
    }

    // Initialize the WindowsNT-specific library
    // -----------------------------------------
    WNTLibInit();

    ESCmdsInit(NULL);

    m_GeneralInfo.ReplyBuffer = DDK_ALLOC_WITH_TAG( NonPagedPool,
                                                    MAX_RESP_SIZE,
                                                    PCX_HRWDM_TAG );
    if(m_GeneralInfo.ReplyBuffer == NULL)
    {
        ntStatus = STATUS_NO_MEMORY;
        goto _clean_exit;
    }

    LOGFileTrace("-> Allocating buffers\n");

    ntStatus = (NTSTATUS) BUFAllocateAllBuffers( &m_GeneralInfo );
    if ( !NT_SUCCESS( ntStatus ) )
    {
        goto _clean_exit;
    }

    //
    // Tell the class driver to initialize the driver.
    //
    ntStatus = PcInitializeAdapterDriver (DriverObject,
                                      RegistryPathName,
                                      AddDevice);
_clean_exit:

    if ( NT_SUCCESS( ntStatus ) )
    {
        DriverObject->MajorFunction[IRP_MJ_DEVICE_CONTROL] = NP88DispatchWrapIoctl;

		m_PortClassUnload = DriverObject->DriverUnload;
		DriverObject->DriverUnload  = NP88WrapUnload;
    }
    else
    {
        if( m_RegistryPathName.Buffer )
        {
            DDK_FREE_WITH_TAG(m_RegistryPathName.Buffer, PCX_HRWDM_TAG);
		    m_RegistryPathName.Buffer = NULL;
        }
        if( m_GeneralInfo.ReplyBuffer )
        {
            DDK_FREE_WITH_TAG(m_GeneralInfo.ReplyBuffer, PCX_HRWDM_TAG);
		    m_GeneralInfo.ReplyBuffer = NULL;
        }
    }

    LOGFileClose();

    return ntStatus;
}

#pragma code_seg("PAGE")


NTSTATUS AllocApiNpResources
(
    IN  PDEVICE_OBJECT  DeviceObject
)
{
    PAGED_CODE ();

    // apinp - Register
    //
    #define DSOUND_APP_NAME L"DirectSound\0"

    REG_REQ_INFO   LcRegistrationRequest;
    REG_REP_INFO   LcRegistrationReply;

    RtlZeroMemory(&LcRegistrationRequest,sizeof(LcRegistrationRequest));
    RtlZeroMemory(&LcRegistrationReply,sizeof(LcRegistrationReply));

    LcRegistrationRequest.Header.hdFamily   = GENE_FAM;
    LcRegistrationRequest.Header.hdComNum   = REG_CMD;
    LcRegistrationRequest.Header.hdBrSize   = sizeof(REG_REP_INFO);
    LcRegistrationRequest.Header.hdHandle   = REGISTER_CMD_CDE;

    LcRegistrationRequest.rgqAttributes     = APP_PRIVATE ;
    LcRegistrationRequest.rgqNameSize       = sizeof(DSOUND_APP_NAME)/2;
    (LcRegistrationRequest.rgqAppName).asStringType
                                            = APP_UNICODE_STRING ;
    RtlCopyMemory( (LcRegistrationRequest.rgqAppName).asUnicodeString,
                   DSOUND_APP_NAME,
                   sizeof(DSOUND_APP_NAME) );

    APHDispatch((LPDWORD)&LcRegistrationRequest,
                (LPDWORD)&LcRegistrationReply,
                NULL);

    return STATUS_SUCCESS;
}


NTSTATUS ReleaseApiNpResources
(
)
{
    PAGED_CODE ();

    // cleanup async commands in TbAppliInfo.
    //
    APH_CleanUpAppliInfo();

    return STATUS_SUCCESS;
}


#pragma code_seg()

