// *************************************************************************
//
//  COPYRIGHT 1996-2000 DIGIGRAM. ALL RIGHTS RESERVED.
//  Portions Copyright (c) 1998-1999 Intel Corporation
//  Portions Copyright (c) 1998-1999 Microsoft Corporation. 
//
// **************************************************************************

#ifndef _MINTOPO_H_
#define _MINTOPO_H_


/*****************************************************************************
 * Structures and Definitions
 */

// This structure is used to translate the TopoPins member to a (registered)
// system pin member or vice versa.
typedef struct
{
    TopoPins    PinDef;
    int         PinNr;
} tPinTranslationTable;


// This structure is used to translate the TopoNodes member to a (registered)
// system node member or vice versa.
typedef struct
{
    TopoNodes   NodeDef;
    int         NodeNr;
} tNodeTranslationTable;


// We must cache the values for the volume and tone controls. If we don't,
// the controls will "jump".
// We have around 50% volume controls in the nodes. One way would be to
// create a separate structure for the node cache and a second array for
// the mapping. But we just use one big array (that could cache mute controls
// too) and in that way simplify the code a little.
// We use the bLeftValid and bRightValid to indicate whether the values stored
// in lLeft and lRight are true - remember at startup we only write and
// initialize the AC97 registers and not this structure. But as soon as there
// is a property get or a property set, this element (node) will be stored here
// and marked valid.
// We don't have to save/restore the virtual controls for WaveIn and MonoOut
// in the registry, cause WDMAUD will do this for every node that is exposed
// in the topology.
typedef struct
{
    long    lLeft;
    long    lRight;
    BOOL    bLeftValid;
    BOOL    bRightValid;
} tNodeCache;

typedef struct
{
    long    lvalue[32];
    DWORD   dwValidityMask;
} tMultichannelNodeCache;

/*****************************************************************************
 * Classes
 */

/*****************************************************************************
 * CMiniportTopologyICH
 *****************************************************************************
 * ICH topology miniport.  This object is associated with the device and is
 * created when the device is started.  The class inherits IMiniportTopology
 * so it can expose this interface and CUnknown so it automatically gets
 * reference counting and aggregation support.
 */
class CMiniportTopologyICH : public IMiniportTopologyICH,
                             public CUnknown
{
private:
    PADAPTERCOMMON           AdapterCommon;          // Adapter common object
    PPORTTOPOLOGY            Port;                   // Callback interface
    PPCFILTER_DESCRIPTOR     FilterDescriptor;       // Filter Descriptor
    PPCPIN_DESCRIPTOR        PinDescriptors;         // Pin Descriptors
    PPCNODE_DESCRIPTOR       NodeDescriptors;        // Node Descriptors
    PPCCONNECTION_DESCRIPTOR ConnectionDescriptors;  // Connection Descriptors

    tPinTranslationTable  *pstPinTrans;   // pin translation table
    tNodeTranslationTable *pstNodeTrans; // node translation table
    tMultichannelNodeCache  stMultiNodeCache[NODE_TOP_ELEMENT]; // cache for multichann nodes.

    BOOL                  m_bCopyProtectFlag;            // Copy protect flag for DRM.
    BOOL                  m_bDigitalOutputDisable;       // DigitalOutputDisable flag for DRM.

    BOOL                    m_bIsOutputPipeReady;
    BOOL                    m_bIsInputPipeReady;

    WORD                    m_sDeviceNumber;
    WORD                    m_wBoardIndex;

    WORD                    m_wChannelNumber;    // ## FM (07/02/2000)
    WORD                    m_wChannelCount;

    GUID                    m_guidDevName;

    /*************************************************************************
     * CMiniportTopologyICH methods
     *
     * These are private member functions used internally by the object.  See
     * MINIPORT.CPP for specific descriptions.
     */

    // builds the topology.
    NTSTATUS BuildTopology
    (
        IN      ULONG   PmWaveInNumber,
        IN      ULONG   PmWaveOutNumber
    );

    // registers (builds) the pins
    NTSTATUS BuildPinDescriptors
    (
        IN      ULONG   PmWaveInNumber,
        IN      ULONG   PmWaveOutNumber
    );

    // registers (builds) the nodes
    NTSTATUS BuildNodeDescriptors
    (
        IN      ULONG   PmWaveInNumber,
        IN      ULONG   PmWaveOutNumber
    );

    // registers (builds) the connection between the pin, nodes.
    NTSTATUS BuildConnectionDescriptors
    (
        IN      ULONG   PmWaveInNumber,
        IN      ULONG   PmWaveOutNumber
    );

#if (DBG)
    // dumps the topology. you can specify if you want to dump pins, nodes,
    // connections (see debug.h).
    void DumpTopology (void);
#endif

/*************************************************************************

    TransNodeNrToNodeDef

*************************************************************************/

    // translates the system node id to a TopoNode.
    TopoNodes TransNodeNrToNodeDef (IN int Node)
    {
#if (DBG)
        TopoNodes   n;

        n = pstNodeTrans[Node].NodeDef;
        // check for invalid translation. could be caused by a connection
        // to a not registered node or wrong use of nodes.
        if (n == NODE_INVALID)
        {
            DOUT (DBG_ERROR, ("Invalid node nr %u.", Node));
        }
        return n;
#else
        return pstNodeTrans[Node].NodeDef;
#endif
    };

/*************************************************************************

    TransTopoNodeDefToNodeNr

*************************************************************************/

    // translates a TopoNode to a system node id.
    int TransTopoNodeDefToNodeNr (IN TopoNodes Node)
    {
#if (DBG)
        int n;

        // check for invalid translation. could be caused by a connection
        // to a not registered node or wrong use of nodes.
        n = pstNodeTrans[Node].NodeNr;
        if (n == -1)
        {
            DOUT (DBG_ERROR, ("Invalid TopoNode %u.", Node));
        }
        return n;
#else
        return pstNodeTrans[Node].NodeNr;
#endif
    };

/*************************************************************************

    TransPinNrToPinDef

*************************************************************************/

    // translates a system pin id to a TopoPin.
    TopoPins TransPinNrToPinDef (IN int Pin)
    {
#if (DBG)
        TopoPins p;

        p = pstPinTrans[Pin].PinDef;
        // check for invalid translation. could be caused by a connection
        // to a not registered pin or wrong use of pins.
        if (p == PIN_INVALID)
        {
            DOUT (DBG_ERROR, ("Invalid pin nr %u.", Pin));
        }
        return p;
#else
        return pstPinTrans[Pin].PinDef;
#endif
    };

/*************************************************************************

    TransTopoPinDefToPinNr

*************************************************************************/

    // translates a TopoPin to a system pin id.
    int TransTopoPinDefToPinNr (IN TopoPins Pin)
    {
#if (DBG)
        int p;

        p = pstPinTrans[Pin].PinNr;
        // check for invalid translation. could be caused by a connection
        // to a not registered pin or wrong use of pins.
        if (p == -1)
        {
            DOUT (DBG_ERROR, ("Invalid TopoPin %u.", Pin));
        }
        return p;
#else
        return pstPinTrans[Pin].PinNr;
#endif
    };

/*************************************************************************

    SetNodeTranslation

*************************************************************************/

    // sets one table entry for translation.
    void SetNodeTranslation (IN int NodeNr, IN TopoNodes NodeDef , IN ULONG line)
    {
        pstNodeTrans[NodeNr].NodeDef = TopoNodes(NodeDef + line * NODE_TOP_ELEMENT);
        pstNodeTrans[NodeDef + line * NODE_TOP_ELEMENT].NodeNr = NodeNr;
    };

/*************************************************************************

    SetPinTranslation

*************************************************************************/

    // sets one table entry for translation.
    void SetPinTranslation (IN int PinNr, IN TopoPins PinDef, IN ULONG line)
    {
        pstPinTrans[PinNr].PinDef = TopoPins(PinDef + line * PIN_TOP_ELEMENT);
        pstPinTrans[PinDef + line * PIN_TOP_ELEMENT].PinNr = PinNr;
    };

// **********************************************************************************
//
// Digigram provided adapter methods
//
// **********************************************************************************
//

    NTSTATUS SetDigitalLevel
    (
        IN      TopoNodes   PmNode,
        IN      LONG        PmChannel,
        IN      PLONG       PmPLevel
    );

    NTSTATUS SetMasterMute
    (
        IN      TopoNodes   PmNode,
        IN      PBOOL       PmPOnOff
    );

// **********************************************************************************
//
// End of Digigram provided adapter methods
//
// **********************************************************************************
//

public:
    /*************************************************************************
     * The following two macros are from STDUNK.H.  DECLARE_STD_UNKNOWN()
     * defines inline IUnknown implementations that use CUnknown's aggregation
     * support.  NonDelegatingQueryInterface() is declared, but it cannot be
     * implemented generically.  Its definition appears in MINIPORT.CPP.
     * DEFINE_STD_CONSTRUCTOR() defines inline a constructor which accepts
     * only the outer unknown, which is used for aggregation.  The standard
     * create macro (in MINIPORT.CPP) uses this constructor.
     */
    DECLARE_STD_UNKNOWN ();
    DEFINE_STD_CONSTRUCTOR (CMiniportTopologyICH);

    ~CMiniportTopologyICH ();

    /*************************************************************************
     * include IMiniportTopology (public/exported) methods
     */
    IMP_IMiniportTopology;

    /*************************************************************************
     * IMiniportTopologyICH methods
     */

    STDMETHODIMP_(NTSTATUS) StartActualNodeSettings
    (
        IN      WORD    PmChannelNum,
        IN      BOOLEAN PmCapture
    );

    STDMETHODIMP_(void) StopActualNodeSettings
    (
        IN      WORD    PmChannelNum,
        IN      BOOLEAN PmCapture
    );

    // returns the system pin id's for wave out, wave in .
    STDMETHODIMP_(NTSTATUS) GetPhysicalConnectionPins
    (
        IN      ULONG  Index,
        OUT     PULONG  Wave
    );

    // sets the copy protect flag.
    STDMETHODIMP_(void) SetDrmFlags
    (
        IN      BOOL CopyProtect,
        IN      BOOL DigitalOutputDisable
    );

    /*************************************************************************
     * IMyPort methods
     */

    STDMETHODIMP_(void) MyVery1stInit
    (
        IN      USHORT  DeviceNumber,
        IN      USHORT  BoardIndex,
        IN      USHORT  PmChannelFirst,
        IN      USHORT  PmChannelCount,
        IN      LPGUID  PmDeviceNameGuid
    );

    /*************************************************************************
     * static functions for the property handler.
     * They are not part of an COM interface and are called directly.
     */

    // called by the other property handlers to get the data range of volume or
    // tone controls in dB.
    static NTSTATUS GetDBValues
    (
        IN  PADAPTERCOMMON,
        IN  TopoNodes       Node,
        OUT LONG            *plMinimum,
        OUT LONG            *plMaximum,
        OUT ULONG           *puStep
    );

    // property handler for mute controls of checkboxes like Loudness.
    static NTSTATUS PropertyHandler_OnOff
    (
        IN      PPCPROPERTY_REQUEST PropertyRequest
    );


    // This routine is the basic support handler for volume or tone controls.
    static NTSTATUS BasicSupportHandler
    (
        IN      PPCPROPERTY_REQUEST PropertyRequest
    );

    static NTSTATUS PropertyHandler_Level
    (
        IN      PPCPROPERTY_REQUEST PropertyRequest
    );

    // property handler for muxer. we have just two muxer, one for recording
    // and one for mono out.
    static NTSTATUS PropertyHandler_Ulong
    (
        IN      PPCPROPERTY_REQUEST PropertyRequest
    );

    // save and validate values to a nod cache.
    static void Property_ValidateValues
	(
        CMiniportTopologyICH*   PmThat,
        TopoNodes               PmNodeDef,
		LONG                    PmChannel,
		LONG                    PmValue
	);

    // this says that audio is played and processed without CPU resources.
    static NTSTATUS PropertyHandler_CpuResources
    (
        IN      PPCPROPERTY_REQUEST PropertyRequest
    );

//  protected:
//	CNPProtocol* m_pDsp;
//  CPIOCommands *m_PIO;
};

#endif  // _MINTOPO_H_
