// *************************************************************************
//
//  COPYRIGHT 1996-2000 DIGIGRAM. ALL RIGHTS RESERVED.
//
//  DIGIGRAM
//
// **************************************************************************

#ifndef _XX_PROTOCOL_H
#define _XX_PROTOCOL_H

#include "pcxini_e.h"
#include "pcxerr_e.h"
#include "if_es_audio.h"
#include "pIOCommands.h"


#define MAX_OUTSTREAM       MAX_BUFFER
#define MAX_INSTREAM        MAX_DRV_INPIPE


typedef struct  _DSP_CMD_INFO {
    DWORD               dcCodeOp;		/**< Op Code of the command (usually 1st 24-bits word).*/
    WORD                dcCmdLength;	/**< Command length in words of 24 bits.*/
    WORD                dcStatusType;	/**< Status type: 0 for fixed length, 1 for random.*/
    WORD                dcStatusLength;	/**< Status length (if fixed).*/
    LPSTR               dcOpName;
} DSP_CMD_INFO, *PDSP_CMD_INFO;



typedef struct _FRAME_INFO {
    DWORD       fiFormat ;
    DWORD       fiHSize ;
} FRAME_INFO, *PFRAME_INFO ;

// Structure only used for get stream state command
// ------------------------------------------------
typedef struct _STATE_STREAM_INFO {
    BYTE        Start;
    BYTE        zero[3];
    PCX_TIME    Hour;
} STATE_STREAM_INFO, *PSTATE_STREAM_INFO;


// Structure only used for IO Audio numeric level command
// ------------------------------------------------------
typedef struct _LEVEL_AUDIO_INFO {
    BOOLEAN     gaiHasDigitalLevel;
    BOOLEAN     gaiHasMonitorLevel;
    BOOLEAN     gaiHasMuteLevel;
    BOOLEAN     gaiHasM1Level;
    BOOLEAN     gaiHasM2Level;

    WORD        gaiDigitalLevel;
    WORD        gaiMonitorLevel;
    WORD        gaiMuteLevel;
    WORD        gaiM1Level;
    WORD        gaiM2Level;
} LEVEL_AUDIO_INFO, *PLEVEL_AUDIO_INFO ;


// Structure only used for output stream numeric level command
// -----------------------------------------------------------
typedef struct _STREAM_LEVEL_INFO {
    BOOLEAN     sliHasPanLevelCgV1;
    BOOLEAN     sliHasPanLevelCgV2;
    BOOLEAN     sliHasPanLevelCdV1;
    BOOLEAN     sliHasPanLevelCdV2;
    BOOLEAN     sliHasLevelStream1;
    BOOLEAN     sliHasLevelStream2;
    BOOLEAN     sliHasMuteStream1;
    BOOLEAN     sliHasMuteStream2;

    union   _LEVEL_VALUES_UNION {
        struct _STREAM_ALL_LEVELS_INFO {
            WORD        saliPanLevelCgV1;
            WORD        saliPanLevelCgV2;
            WORD        saliPanLevelCdV1;
            WORD        saliPanLevelCdV2;
            WORD        saliLevelStream1;
            WORD        saliLevelStream2;
            WORD        saliMuteStream1;
            WORD        saliMuteStream2;
        } lvuAllLevels;

#define sliPanLevelCgV1 u.lvuAllLevels.saliPanLevelCgV1
#define sliPanLevelCgV2 u.lvuAllLevels.saliPanLevelCgV2
#define sliPanLevelCdV1 u.lvuAllLevels.saliPanLevelCdV1
#define sliPanLevelCdV2 u.lvuAllLevels.saliPanLevelCdV2
#define sliLevelStream1 u.lvuAllLevels.saliLevelStream1
#define sliLevelStream2 u.lvuAllLevels.saliLevelStream2
#define sliMuteStream1  u.lvuAllLevels.saliMuteStream1
#define sliMuteStream2  u.lvuAllLevels.saliMuteStream2

        struct _STREAM_LEVEL_CURVE_INFO {
            WORD        slcCurveNbLevels;
            DWORD       slcCurveStep;
            LPWORD      slcCurveValues;
        } lvuLevelCurve;

#define sliCurveNbLevels    u.lvuLevelCurve.slcCurveNbLevels
#define sliCurveStep        u.lvuLevelCurve.slcCurveStep
#define sliCurveValues      u.lvuLevelCurve.slcCurveValues

   } u;

} LEVEL_STREAM_INFO, *PLEVEL_STREAM_INFO ;


// Structure only used for get audio IO vu-meter, Pic-meter command
// ----------------------------------------------------------------
typedef struct _VU_PIC_METER_INFO {
    DWORD       vpmiVuMeter;
    DWORD       vpmiPicMeter;
    BOOLEAN     vpmiSaturation;
} VU_PIC_METER_INFO, *PVU_PIC_METER_INFO;


// TAG_DEBUG_DRIVER : DEB

#define SI_MAX_STREAM_BUFFER	8

typedef struct  _STREAM_INFO {

    LPBC_HEADER_INFO    siAdrBuffer;
    LPBC_HEADER_INFO    siAdrNextBufferToGive;
    LONG                siReleasedBuffers;
    BOOLEAN             siAPINP;			// a APINP stream (uses its own buffer pool)
    BYTE                siUnused1[3];       // fill up to multiple of 4 Bytes
    struct _STREAM_INFO *siAdrNextStream;

    // FS - 06/03/1997 - make a double-chain used by Sound Transfer
    // ************************************************************
    struct _STREAM_INFO *siAdrPreviousStream;
    WORD			siFirstAudioNum;		// number of first board audio ( valid after pipe allocation)
    BOOLEAN 		siFirstAudioValid;		// number of first board audio may be zero, only first stream get this number
    BYTE			siUnused;				// fill up to multiple of 4 Bytes
	DWORD			siBufferAddr2Index[SI_MAX_STREAM_BUFFER];
    DWORD           siPCMSampleSize;
} STREAM_INFO, *PSTREAM_INFO;

// TAG_DEBUG_DRIVER : FIN

/// Stream Format Header Defines (for LIN and IEEE754)
#define HEADER_FMT_BASE			HEADER_FMT_BASE_LIN
#define HEADER_FMT_BASE_LIN		0xFED00000
#define HEADER_FMT_BASE_FLOAT	0xFAD00000
#define HEADER_FMT_MONO			0x00000080 /**< bit 23 in header_lo. WARNING : old bit 22 is ignored in float format*/
#define HEADER_FMT_INTEL		0x00008000 
#define HEADER_FMT_16BITS		0x00002000
#define HEADER_FMT_24BITS		0x00004000
#define HEADER_FMT_UPTO11		0x00000200	/**< frequency is less or equ. to 11k.*/
#define HEADER_FMT_UPTO32		0x00000100	/**< frequency is over 11k and less then 32k.*/

// Pipe / Stream state
// check pcxapi_e.h for compatibility
// --------------------
#define START_STATE             1
#define PAUSE_STATE             0


// Values for a user delay
#define DC_DIFFERED_DELAY       (1<<BIT_DIFFERED_COMMAND)
#define DC_NOTIFY_DELAY         (1<<BIT_NOTIFIED_COMMAND)
#define DC_HBUFFER_DELAY        (1<<BIT_TIME_RELATIVE_TO_BUFFER)
#define DC_MULTIPLE_DELAY       (1<<BIT_RESERVED)
#define DC_STREAM_TIME_DELAY    (1<<BIT_STREAM_TIME)
#define DC_CANCELLED_DELAY      (1<<BIT_CANCELLED_COMMAND)
#define DC_BYTECOUNT_DELAY      ((1<<BIT_AFTER_BUFFER_TIME)|(1<<BIT_STREAM_TIME))




// Global typedef
//***************

typedef PDWORD   HANDLE_PIPE, * PHANDLE_PIPE;


class CPIOCommands;

/// VX Protocol Implementation Class.
/**
*	@class CProtocol 
*	This Class Encapsulates the generation of DSP commands.
*	It accesses the NP Board Type abstraction class IOCommands.
*	This Class provides Several "Interfaces", in fact families of methodes
*	for the management of all of the Audio Processing Features of a NP Board DSP.
*	It should be instancied by an objet representing either the owner of the DSP, 
*	like IOAudioEngine in MacOsX's IOKit, or CAdapterCommon is the AC'97-based WDM Driver.
*
*/
class CProtocol 
{
protected:

			PDSP_INFO		m_pDSPInfo;

			STREAM_INFO		m_Out_Stream_Info_Array[MAX_OUTSTREAM];		/**<  Output Stream descriptors, up to 376.*/
			PSTREAM_INFO	m_StreamOutFreeList;						/**<  1st free Output Stream descriptors.*/
			STREAM_INFO		m_In_Stream_Info_Array[MAX_INSTREAM];		/**<  Input Stream descriptors, up to 64.*/
			PSTREAM_INFO	m_StreamInFreeList;							/**<  1st free Input Stream descriptors.*/

			PKSPIN_LOCK		m_MsgLock;

			CMD_RMH_INFO	m_Rmh;
	
			CPIOCommands*	m_PIO;									/**<  IOCommands board type abstraction and bus access object .*/

            // Features
            //
            DWORD m_PlayFormats;
            DWORD m_RecordFormats;
            WORD m_PCMOnlyGranularity;

			LONG  m_TimerIT_Refs;
			DWORD m_TimerIT;
			LONG  m_SrcIT;

			DWORDLONG m_AudioInMask_Reserving_Src;
			DWORDLONG m_AudioInMask_UER_SRC_Def;
			DWORDLONG m_AudioInMask_UER_SRC_Async;
	public:
			
			WORD  m_FatalDspError;

	
	virtual		WORD IDiag_DspSetTimerInterrupt( IN WORD PmSamples) = 0; 
	
	/**
	*	Constructor, 
	*	one and only one protocol object must be instancied for each board.
	*	@param 	PmpDspInfo	(for compatibility with VX32)
	*	@param  pPIO		(Board type abstraction and Bus accesses, typically instancied by adapter object)
	*	@see	VXWDM project : CAdapterCommon::InitDsp()
	*/
				CProtocol( PDSP_INFO PmpDspInfo, CPIOCommands* PmpPIO);

    /**
	*	Destructor 
	*	@see	VXWDM project : ~CAdapterCommon()
	*/
	virtual ~CProtocol();

    // IIIIIIIII    Interface RESOURCE MANAGER     IIIIIIIIIIIIII//

/**
 * \defgroup Methodes for Pipe resource reservation 
 */
/*@{*/
			WORD   IResource_Source_Release_Audios(IN DWORDLONG PmAudioMask) ;

			WORD    IResource_Source_TestClock_Audio(	IN WORD PmAudio,
												IN BYTE PmSourceType,
												IN WORD PmClockType,
												IN DWORD PmSyncInputNum,
												IN DWORD PmFrequency,
												INOUT PBYTE PmUER_SRC);

			WORD    IResource_Source_Check_Audios_New_Clock(	IN WORD PmClockType,
													IN DWORD PmSyncInputNum,
													IN DWORD PmFrequency,
													IN DWORDLONG PmAudioMask,
													IN BYTE PmUER_SRC);

			WORD    IResource_Source_Change_Audios(	IN DWORDLONG PmAudioMask,
											IN BYTE PmSourceType,
											IN BYTE PmUER_SRC);
/*@}*/



    // IIIIIIIII    Interface INIT et DIAG      IIIIIIIIIIIIII//

/**
 * \defgroup Init_and_Diag Initialisation and Diagnostic Interface
 */
/*@{*/

    /**
	*	DIAG Interface:	check DSP state.
    *   Return TRUE only if Dsp is loaded AND there is no fatal error
	*
	*/
		BOOLEAN	IDiag_IsDspRunning( )
	{
		BOOLEAN LcOK = ( ( m_pDSPInfo->dsEtat ) && !( m_pDSPInfo->dsFatalError ) );
#ifdef DBG_VIEWER
        if( LcOK == FALSE ) DOUT(DBG_ERROR, ("LXESWDM : FatalError (%x)\n", m_pDSPInfo->dsFatalError));
#endif
		return LcOK;
	};

	
	virtual	WORD	IDiag_HaltEmbedded() = 0;

	
	/**
	*	INIT Interface : Check DSP version and relevance, set default Latency.
	*	@param	PmSoftNum	because on NP, this may be different from DSP_INFO->dsEtat
	*	@param	PmCheckVersion  do not check BUILD:'B' or RELEASE:'R'
	*	@see	CAdapterCommon::InitDSP()
	*	@return SUCCESS or ED_DSP_CORRUPTED, if a wrong SoftWare Sign was submitted to the DSP
	*/
	virtual WORD    IInit_GetVersionAndFeatures( BYTE PmSoftNum, PGENERAL_INFO PmGenInfo, PDWORD PmDspVersion ) = 0;

	/**
	*	INIT Interface : Retrieve this Board's DSP Software Number
	*/	
			BYTE	IInit_GetDspSoftNum( void) { return m_pDSPInfo->dsEtat; };

    /**
	*	DIAG Interface : Retrieve the Mask of Physical mono Inputs and Outputs (Audios)
	*	@param	PmPtrTg			IN board/dsp
	*	@param	PmPlayFmts		OUT supported playback formats.
	*	@param	PmPlayEffects	OUT supported playback effects.
	*	@param	PmRecordFmts	OUT supported record formats.
	*	@return SUCCESS or an error code.
	*/
			VOID    IDiag_GetSupportedPRFeatures(  OUT PDWORD PmPlayFmts, OUT PDWORD PmRecordFmts )
                                        {
                                            ASSERT(m_PlayFormats);

                                            *PmPlayFmts     = m_PlayFormats;
                                            *PmRecordFmts   = m_RecordFormats;
                                        };

			WORD	IDiag_GetPCMOnlyGranularity(){return m_PCMOnlyGranularity;}

	virtual	WORD	IDiag_HandleBoardIBL(	INOUT LPIBL_RESP_INFO	PmPtrIBLRep) = 0;
	
	virtual	WORD	IDiag_DspResourcesUsage(OUT LPDSP_RESOURCES_RESP_INFO   PmDspResourcesRep) { return ED_UNAVAILABLE_FEATURE; };
	
	/**
	*	DIAG Interface : Retrieve DSP Structures Information for debug purpose
	*	@see	API : xxx-code\addon4.c for samples. or use vxdiag dll.
	*/
	virtual	WORD	IDiag_DSPTrace(	IN WORD     PmRequest,
							IN WORD     PmParam1,
							IN WORD     PmParam2,
							IN LPDWORD  PmInBlocPtr,
							OUT LPDWORD PmOutBlocPtr ) { return ED_UNAVAILABLE_FEATURE; };
										
	/**
	*	DIAG Interface : Retrieve STREAM_INFO data.
	*	@param	PmAdr	IN	address of the record in the TbStreamOut or TbStreamIn array.
	*	@param	PmPPStreamInfo	OUT	PmAdr
	*	@param	PmPSize	OUT	size of the record
	*	@see	ah_gene.c/DriverDebugFct
	*	@return SUCCESS or an error code.
	*/
		WORD	IDiag_StreamGetInfo(
								IN  DWORD   PmAdr,
								OUT PBYTE   *PmPPStreamInfo,
								OUT PWORD   PmPSize );

    /**
    *	DIAG Interface : Audio Availability test
    *	@param	PmAudioNum		Audio Number on the board ( 0 to N )
    *	@param	PmAudioAttrib	AUDIO_IN or AUDIO_OUT
    *	@see	api_hdl.c : LookupBoardAudio(..)
    *	@return SUCCESS if the Audio is available for a new allocation, 
    *	or a error if it is already reserved, or the dsp is crashed.
    */
		WORD    IDiag_AudioCheckAvailability(   IN  BYTE    PmAudioNum,     // numero d'audio sur la carte
                                            IN  WORD    PmAudioAttrib); // in ou out

/*@{*/


    // IIIIIIIII    Interface CLOCK      IIIIIIIIIIIIII//

/**
 * \defgroup Clock and Synchro Signal Settings Interface
 */
/*@{*/


	/**
	*	CLOCK Interface : Read the current UER clock 
	*	@see	TraitHardItBoard
	*	@return SUCCESS or error code.
	*/
	virtual	WORD	IClock_BoardGetClockFrequency(	OUT PDWORD          PmFrequencyPtr,
													OUT PBOOLEAN        PmPFirstChange,
													OUT PDWORD			PmPRawData	= NULL ) = 0;

	/**
	*	CLOCK Interface : Pipe Allocation Method
	*
	*	@param PmSyncFifo		IN	the DSP shall resync. it's input FIFOs
	*	@param PmComputeFreq	IN  recompute frequency
	*	@return		SUCCESS or error code.
	*/ 
	virtual	WORD	IClock_BoardModifyClock(	IN BOOLEAN          PmSyncFifo,
										IN BOOLEAN          PmComputeFreq_ON ) { return ED_UNAVAILABLE_FEATURE; };

	/**
	*	CLOCK Interface : 	the DSP shall resync. it's FIFOs
	*
	*	@param PmAudioNum		IN	Audio Number
	*	@return		SUCCESS or error code.
	*/ 
	virtual	WORD	IClock_BoardModifyInputs( IN WORD PmAudioNum ) { return ED_UNAVAILABLE_FEATURE; };

	/**
	*	CLOCK Interface : Used for start on timecode features
	*	@see	API, if needed
	*	@return SUCCESS or error code.
	*/
    virtual	WORD	IClock_TimeCodeConfig(	IN  BOOLEAN         PmCancel,	
									IN  WORD            PmNbOfTimeCode,
									IN  DWORD           PmSamplesCount,
									IN  PPCX_TIME_CODE  PmPTimeCode ) { return ED_UNAVAILABLE_FEATURE; };

	/**
	*	CLOCK Interface : Retrieve the Last decoded LTC value
	*	@param	 PmPNewTimeCode		TRUE if this is a new LTC value
	*	@param	 PmPBackward		TRUE if LTC direction is backward
	*	@param	 PmPWaitingTimeCode	TRUE if a specific LTC value is beeing scanned by the board 
	*	@param	 PmPPcxTime			Reference Sample Count 
	*	@param	 PmPTimeCode		Current LTC
	*	@return SUCCESS or error code.
	*/
	virtual	WORD	IClock_TimeCodeGet(		OUT PBOOLEAN        PmPNewTimeCode,
									OUT PBOOLEAN        PmPBackward,
									OUT LPBOOLEAN       PmPWaitingTimeCode,
									OUT PPCX_TIME       PmPPcxTime,
									OUT PPCX_TIME_CODE  PmPTimeCode ) { return ED_UNAVAILABLE_FEATURE; };

	/**
	*	CLOCK Interface : Retrieve the Last decoded LTC value
	*	@param	 PmSignal	set if LTC shall be managed by the DSP
	*	@return SUCCESS or error code.
	*/
	virtual	WORD	IClock_ManageSignal( IN  BYTE  PmSignal ) { return ED_UNAVAILABLE_FEATURE; };

/*@{*/

	// IIIIIIIII    Interface LEVELS & Meters      IIIIIIIIIIIIII//
	
/**
 * \defgroup Levels Levels and Meters Management Interface
 */
/*@{*/
	
	/**
	*	LEVEL Interface :	Get VU and Peak Meters
	*
	*	@param	 PmIsInput	    
	*	@param	 PmAudioMask	target audios 
	*	@param	 PmNbAudio		number of structures to fill out 
	*	@param	 PmPtrVuPicMeterInf pointer to the Array of meter infos
	*
	*	@see API : GetAudioVuMeterSubFct
	*	@return SUCCESS or error code.
	*/  
    virtual	WORD    ILevel_AudioGetVuPicMeter(	IN BOOLEAN      PmIsInput,
										IN DWORDLONG            PmAudioMask,
										IN WORD                 PmNbAudio,
										OUT PVU_PIC_METER_INFO  PmPtrVuPicMeterInf ) =0;

    /**
	*	LEVEL Interface :	Set Levels and Mutes
	*
	*	Setting a Mute :
	*			LcLevelAudioInfo.gaiHasMuteLevel = TRUE;
	*			LcLevelAudioInfo.gaiMuteLevel = (WORD)( *PmPOnOff ? 1 : 0 );
	*
	*	Setting a Monitoring Level :
    *			LcLevelAudioInfo.gaiHasMonitorLevel = TRUE;
	*			LcLevelAudioInfo.gaiMonitorLevel = (WORD) LcDriverLevel;
	*
	*	Setting a Level :
    *			LcLevelAudioInfo.gaiHasDigitalLevel = TRUE;
    *			LcLevelAudioInfo.gaiDigitalLevel = (WORD) LcDriverLevel;
	*
	*	@param	 PmPtrTgPipe	target Pipe
	*	@param	 PmAudioMask	target audios 
	*	@param	 PmPtrLevel		Level Info Array for this/ese audio/s
	*
	*	@see Topology, PLEVEL_AUDIO_INFO
	*	@return SUCCESS or error code.
	*/
	virtual	WORD	ILevel_AudioSetDigitalLevel(	IN PTARGET_INFO        PmPtrTgPipe,
													IN DWORDLONG           PmAudioMask,
													IN PLEVEL_AUDIO_INFO    PmPtrLevel ) =0; 		
	/**
	*	LEVEL Interface :	Get Audio Levels
	*
	*	@param	 PmIsInput	
	*	@param	 PmAudioMask	IN	target Audios
	*	@param	 PmNbAudio		IN	number of structures to fill out
	*	@param	 PmPtrLevelInf  OUT	pointer to the array of Info Structures
	*
	*	@see API : GetOutAudioLevelsSubFct
	*	@return SUCCESS or error code.
	*/  
    virtual	WORD    ILevel_AudioGetLevels(	IN BOOLEAN              PmIsInput,
											IN DWORDLONG            PmAudioMask,
											IN WORD                 PmNbAudio,
											OUT PLEVEL_AUDIO_INFO    PmPtrLevelInf )  =0;


	/**
	*	LEVEL Interface :	Get Stream Levels
	*	@param PmPtrTgPipe		IN	target pipe (board/dsp/pipe)
	*	@param PmNbStream		IN	Number of streams
	*	@param PmPtrStreamInf	OUT	Levels for each stream
	*	@return		SUCCESS or error code.
	*/  
    virtual	WORD    ILevel_StreamOutGetLevels(	IN  PTARGET_INFO		PmPtrTgPipe     ,
												IN  WORD				PmNbStream      ,
												OUT PLEVEL_STREAM_INFO	PmPtrStreamInf  ) { return ED_UNAVAILABLE_FEATURE; };


	/**
	*	LEVEL Interface :	Get Audio Levels
	*	@param		PmPtrTgPipe		IN	target Pipe
	*	@param		PmPtrLevelInf	IN	Audios Levels to be set
	*	@see API : SetOutStreamLevels
	*	@return SUCCESS or error code.
	*/  
    virtual	WORD    ILevel_StreamOutSetLevels(	IN  PTARGET_INFO        PmPtrTgPipe,
												IN  PLEVEL_STREAM_INFO  PmPtrLevelInf   ) { return ED_UNAVAILABLE_FEATURE; };

	/**
	*	LEVEL Interface :	formats and sends a request to set multiple levels on streams
	*	@param		PmPtrTgPipe		IN	target Pipe
	*	@param		PmPtrLevelInf	IN	Audios Levels to be set
	*	@return SUCCESS or error code.
	*/
	virtual	WORD    ILevel_StreamOutSetLevelCurve(	IN  PTARGET_INFO        PmPtrTgPipe,
											IN  PLEVEL_STREAM_INFO  PmPtrLevelInf   ) { return ED_UNAVAILABLE_FEATURE; }; 
	
	/**
	*	LEVEL Interface :	Get Stream VU Meters
	*
	*	@param	 PmPtrTgPipe			IN	target Pipe
	*	@param	 PmNbStream				IN	Number of streams
	*	@param	 PmPtrVuPicMeterInf		OUT	VU Meters for each stream
	*
	*	@see API : GetOutAudioLevelsSubFct
	*	@return SUCCESS or error code.
	*/
	virtual	WORD    ILevel_StreamGetVuMeter(	IN PTARGET_INFO         PmPtrTgPipe,
												IN WORD                 PmNbMeters,
							                    IN DWORDLONG            PmPipeAudiosMask,
												OUT PSTREAM_VU_METER_RESP_INFO  PmPtrStreamVuPicMeterInf ) { return ED_UNAVAILABLE_FEATURE; }; 
	
	
/*@}*/
	
	// IIIIIIIII    Interface FLOW      IIIIIIIIIIIIII//

/**
 * \defgroup FLow Sound Transport Management Interface
 */
/*@{*/

	
	virtual	WORD	IFlow_AudioConnect(	IN  WORD            PmAudioOut,
										IN  WORD            PmAudioIn   ) { return ED_UNAVAILABLE_FEATURE; }; 

    
 	/**
	*	FLOW IFlow_AudioInputEffect
    *  Set or Get effect parameter, or remove an effect from stack.
    *
	*  @param   PmAudioNum    IN	    Identifies the target Input Audio.
	*  @param   PmpEffectInfo INOUT	Sets or Gets the effect parameters.
    *  @return  SUCCESS or error code.
	*/
    WORD   IFlow_AudioInputEffect( IN WORD PmAudioNum, INOUT LPIN_AUDIO_EFFECT_REQ_INFO PmpEffectInfo) { return ED_UNAVAILABLE_FEATURE; }; 

	/**
	*	FLOW Interface :	Start the Pipe's unique stream
	*
	*	@param		PmPtrTgPipe		IN	tgPipe,tgCaracPipeVoie,tgVioHandle are relevant
	*	@return SUCCESS or error code.
	*	@warning On VX boards, starting/pause or stopping a stream while the pipe is running
	*	may cause the board to output noise. This won't lead to a malfunction, it is just noisy.
	*	This means that one should always care to follow this sequence : a) send buffers b) start stream
	*	c) start pipe.....[n) pause pipe/ n+1) start pipe]....z) stop pipe.
	*/
    virtual	WORD	IFlow_StreamStart( IN PTARGET_INFO PmPtrTgPipe )=0;

    /**
	*	FLOW Interface :	Stop the Pipe's unique stream
	*	StopPipe should be used alone.
	*
	*	@param		PmPtrTgPipe		IN	tgPipe,tgCaracPipeVoie,tgVioHandle are relevant
	*	@return SUCCESS or error code.
	*	@warning On VX boards, starting/pause or stopping a stream while the pipe is running
	*	may cause the board to output noise. This won't lead to a malfunction, it is just noisy.
	*	This means that one should always care to follow this sequence : a) send buffers b) start stream
	*	c) start pipe.....[n) pause pipe/ n+1) start pipe]....z) stop pipe.
	*/  
    virtual	WORD	IFlow_StreamStop( IN PTARGET_INFO PmPtrTgPipe ) = 0;

    /**
	*	FLOW Interface :	Stop the Pipe's unique stream
	*	StopPipe should be used alone.
	*
	*	@param		PmPtrTgPipe		IN	tgPipe,tgCaracPipeVoie,tgVioHandle are relevant
	*	@return SUCCESS or error code.
	*	@warning The Buffer cueue is not handled! and no buffers will be notified
    *   with the command.
	*/  
    WORD	IFlow_StreamStopBrutal( IN PTARGET_INFO PmPtrTgPipe ) { return ED_UNAVAILABLE_FEATURE; }; 

    /**
	*	FLOW Interface :	Pause the Pipe's unique stream
	*
	*	@param		PmPtrTgPipe		IN	tgPipe,tgCaracPipeVoie,tgVioHandle are relevant
	*	@return SUCCESS or error code.
	*	@warning On VX boards, starting/pause or stopping a stream while the pipe is running
	*	may cause the board to output noise. This won't lead to a malfunction, it is just noisy.
	*	This means that one should always care to follow this sequence : a) send buffers b) start stream
	*	c) start pipe.....[n) pause pipe/ n+1) start pipe]....z) stop pipe.
	*/  
    virtual	WORD IFlow_StreamPause( IN PTARGET_INFO PmPtrTgPipe, BOOLEAN PmDrain = FALSE ) = 0;


	/**
	*	FLOW Interface :	Set the Output Stream playback Format
	*
	*	@return		SUCCESS or error code.
	*	@warning	No default format should be assumed. Must match DSP mixing capabilities
	*/  
    virtual	WORD  IFlow_StreamOutSetFormat(     IN PTARGET_INFO PmPtrTgPipe ,
                                        IN DWORD        PmFmtParams,
                                        IN DWORD        PmHeaderHiLo,
                                        IN DWORD        PmChannelMask,
										IN DWORD        PmSamplesPerChannel
                                        ) = 0;
    
	/**
	*	FLOW Interface :	Set a Pipe's stream Format simple Methode
	*	Simplified Stream Format Setting Methode, for Linear formats.
	*	@param		PmPtrTgPipe		IN	tgPipe,tgCaracPipeVoie,tgVioHandle are relevant
	*	@param		PmFrequency		IN	sample frequency, default is 44100 
	*	@param		PmLinearBits	IN	sample bitwidth, default is 16
	*	@param		PmIsMono		IN	Mono, default is FALSE
	*	@param		PmIsIntel		IN	Intel (little endian), default is TRUE

	*	@return		SUCCESS or error code.
	*	@see		#IFlow_SetOutStreamFormat, ichwave.cpp
	*	@see		#IFlow_SetInStreamFormat, ichwave.cpp
	*/  
		WORD  IFlow_FastSetFormat(	IN PTARGET_INFO PmPtrTgPipe ,
									IN DWORD      PmFrequency	= 44100,
									IN WORD	      PmLinearBits	= 16,
									IN BOOLEAN    PmIsIntel		= TRUE,
									IN DWORD      PmChannels	= 2 )
	{
		// Build header 
		DWORD LcdwHeader = HEADER_FMT_BASE;
		DWORD LcdwParams = PmChannels & 0x1FF;
        switch (PmChannels) {
        case 1:
            LcdwHeader |= HEADER_FMT_MONO;
            break;
        case 2:
            break;
        default:
           LcdwParams |= (1<<BIT_FMP_MULTICHANNEL);
           break;
        }

        if (PmIsIntel)	LcdwHeader |= HEADER_FMT_INTEL;

        if ((PmFrequency < 32000L) && (PmFrequency > 11025L))
            LcdwHeader |= HEADER_FMT_UPTO32;
        else if (PmFrequency <= 11025L)
            LcdwHeader |= HEADER_FMT_UPTO11;

        switch ( PmLinearBits ) {
        case 8  : break;
        case 16 : LcdwHeader |= HEADER_FMT_16BITS; break;
        case 24 : LcdwHeader |= HEADER_FMT_24BITS; break;
		default : ; //			DOUT(DBG_ERROR,("IFlow_SetOutStreamFormat: linear bit %d unsupported (Forced to 8)",PmLinearBits));
        };

		if (PmPtrTgPipe->tgCaracPipeVoie)	// Record ?
		{
			return IFlow_StreamInSetFormat(
					  PmPtrTgPipe,
                      LcdwParams,
					  LcdwHeader,	//PmHeaderHiLo
                      0 ,            //PmChannelMask
					  0
					  );
		}

			return IFlow_StreamOutSetFormat(
					  PmPtrTgPipe,
                      LcdwParams,
					  LcdwHeader,	//PmHeaderHiLo
                      0 ,            //PmChannelMask
					  0
					  );
	};

	/**
	*	FLOW Interface :	Set the actual Input Pipe recording Format
	*
	*	@return		SUCCESS or error code.
	*	@warning	No default format should be assumed.
	*/  
    virtual	WORD    IFlow_StreamInSetFormat(   IN PTARGET_INFO PmPtrTgPipe ,
                                       IN DWORD        PmFmtParams,
                                       IN DWORD        PmHeaderHiLo,
                                       IN DWORD        PmChannelMask,
									   IN DWORD        PmSamplesPerChannel) = 0;

	/**
	*	FLOW Interface :	Get the State of many Streams
	*	
	*	@param PmPtrTgPipe		IN	target pipe (board/dsp/pipe)
	*	@param PmNbStream		IN	Number of streams
	*	@param PmPtrStreamInf	OUT	States for each stream
	*	@return		SUCCESS or error code.
	*/ 
	virtual	WORD	IFlow_StreamGetState(	IN PTARGET_INFO         PmPtrTgPipe,
									IN WORD                 PmNbStream,
									OUT PSTATE_STREAM_INFO  PmPtrStreamInf ) = 0;
	
   /**
	*	FLOW Interface :	Get Stream Sample Position
	*	@param PmPtrTgPipe		IN	target pipe (board/dsp/pipe)
	*	@param PmNbStream		IN	Number of streams
	*	@param PmPtrStreamInf	OUT	sample position being played for each stream
	*	@return		SUCCESS or error code.
	*/  
    virtual	WORD    IFlow_StreamSamplePosition(	IN  PTARGET_INFO    PmPtrTgPipe     ,
										IN  WORD            PmNbStream      ,
										OUT PPCX_TIME       PmPtrStreamInf  )= 0;	
	/**
	*	FLOW Interface :	Get Stream Sample Count
	*	@param PmPtrTgPipe		IN	target pipe (board/dsp/pipe)
	*	@param PmNbStream		IN	Number of streams
	*	@param PmPtrStreamInf	OUT	sample count de/coded for each stream
	*	@return		SUCCESS or error code.
	*/  
    virtual	WORD    IFlow_StreamSampleCount(	IN  PTARGET_INFO    PmPtrTgPipe     ,
												IN  WORD            PmNbStream      ,
												OUT PPCX_TIME       PmPtrStreamInf  ) 
				{
					// call into SamplePos, if not implemented.
					return IFlow_StreamSamplePosition(	PmPtrTgPipe, PmNbStream, PmPtrStreamInf);
				};
	
 

	/**
	*	FLOW Interface : Audio Connection for Monitoring
	*	This methode is only useful, when a  monitoring audio different
	*	to the default one has to be specified.
	*	By default, Audio IN n is monitored on Audio OUT n.
	*
	*	@return		SUCCESS or error code.
	*/ 
	WORD	IFlow_ChangeMonitoringSource(	IN  WORD            PmAudioOut,
											IN  WORD            PmAudioIn   )  { return ED_UNAVAILABLE_FEATURE; }; 

	/**
	*	FLOW Interface :	Get Stream Parameters
	*	@param PmPtrTgPipe		IN	target pipe (board/dsp/pipe)
	*	@param PmSetAutomation	IN Automation (synchronous LEVEL_1) parameter
	*	@return		SUCCESS or error code.
	*/  
	WORD	IFlow_StreamOutSetParameters(	IN PTARGET_INFO PmPtrTgPipe ,
											IN BOOLEAN      PmSetAutomation )  { return ED_UNAVAILABLE_FEATURE; }; 

	/**
	*	FLOW Interface :	Get Stream EQ coefficients
	*	@param PmPtrTgPipe			IN	target pipe (board/dsp/pipe)
	*	@param PCX_EFFECT_HANDLE	IN	Effect ID (EFFECTID_MPEG_EQUALIZATION)
	*	@param PmParam				OUT	Array of MAX_EQUALIZATION_COEFF parameters
	*	@return		SUCCESS or error code.
	*/
	WORD    IFlow_StreamOutGetEffects(	IN      PTARGET_INFO        PmPtrTgPipe,
										IN      PCX_EFFECT_HANDLE   PmEffectId,
										INOUT   PDWORD              PmParam   )  { return ED_UNAVAILABLE_FEATURE; }; 
	
	/**
	*	FLOW Interface :	Set Stream EQ coefficients
	*	@param PmPtrTgPipe			IN	target pipe (board/dsp/pipe)
	*	@param PCX_EFFECT_HANDLE	IN	Effect ID (EFFECTID_MPEG_EQUALIZATION)
	*	@param PmParam				IN	Array of MAX_EQUALIZATION_COEFF parameters
	*	@return		SUCCESS or error code.
	*/
	WORD    IFlow_StreamOutSetEffects(	IN  PTARGET_INFO        PmPtrTgPipe,
										IN  PCX_EFFECT_HANDLE   PmEffectId,													
										IN  PDWORD              PmParam   )  { return ED_UNAVAILABLE_FEATURE; }; 




	/**
	*	FLOW Interface : Pipe Allocation Method
	*
	*	@param PmNbMaxFlux	IN	nb of streams the Caller wishes
	*	@param PmNbAudioIO	IN	nb of audios supported by the pipe (output is always stereo)
	*	@param PmManagement	IN	Management options : see PIPE_DECL_INFO, pcxall_e.h
	*	@param PmFrameInfo	IN	bitrate information (optional, may be left empty)
	*	@param PmPtrTgPipe	IN	pipe description structure (see code for relevant fields)
	*	@param PmPHdlPipe	OUT	pointer to a vioHandle  (in fact STREAM_INFO) structure
	*	@param PmStreamFifoSizeArray	OUT	Size of stream FIFO, for each stream (slave boards)
	*	@param PmNbAllocatedStreams		OUT	number of streams availeble on this pipe
	*
	*	@return		SUCCESS or error code.
	*/     
	virtual	WORD	IFlow_PipeDef(		IN BYTE PmAttributes,
								IN WORD PmFirstAudio,
                                IN BYTE PmNbMaxFlux,
								IN WORD PmNbAudioIO,
								IN DWORD PmChannelMask,
								IN WORD PmManagement,
								IN PFRAME_INFO PmFrameInfo,
                                IN BOOLEAN PmDirectSound,
								OUT PHANDLE_PIPE PmPHdlPipe,
								OUT PWORD PmNbAllocatedStreams ) = 0;

	/**
	*	FLOW Interface : Start/Restart 1 or many Pipes.
	*	
	*	@param PmNbPipe		IN	number of pipes to pause
	*	@param PmPtrTgPipe	IN	tgCaracPipeVoie, tgVioHandle and tgPipe are relevant
	*	@param PmCondition	IN	START_PAUSE_IMMEDIATE or START_PAUSE_ON_TIME_CODE
	*	@param PmPTimeCodeStart	IN	array of LTC start info structures
	*
	*	@return		SUCCESS or "DSP BUSY", if the pipe wasn't able to generate enought samples to start
	*	if the accepted delay. A VX-Pipe needs a least 126 samples to be sent to the board, prior to a start
	*	command.
	*	@see		#vio_TogglePipeState .
	*/     
    WORD    IFlow_PipeStart(	IN PTARGET_INFO             PmPtrTgPipe,
								IN BYTE                     PmNbPipe,
								IN BYTE                     PmCondition,
								IN PTIME_CODE_START_INFO    PmPTimeCodeStart)
	{
		return vio_TogglePipeState( START_STATE, PmCondition, PmNbPipe, PmPtrTgPipe, PmPTimeCodeStart );
	}

 	/**
	*	FLOW Interface : Pause 1 or many Pipes.
	*	
	*	@param PmNbPipe		IN	number of pipes to pause
	*	@param PmPtrTgPipe	IN	tgCaracPipeVoie, tgVioHandle and tgPipe are relevant
	*	@param PmCondition	IN	START_PAUSE_IMMEDIATE or START_PAUSE_ON_TIME_CODE
	*
	*	@return		SUCCESS or error code.
	*	@see		#vio_TogglePipeState .
	*/     
    WORD    IFlow_PipePause(	IN PTARGET_INFO     PmPtrTgPipe,
								IN BYTE             PmNbPipe,
								IN BYTE             PmCondition)
	{
    	return vio_TogglePipeState( PAUSE_STATE, PmCondition, PmNbPipe, PmPtrTgPipe, NULL );
	}

    /**
	*	FLOW Interface :	Stop One Pipe
	*
	*	@param		PmPtrTgPipe		IN	tgPipe,tgCaracPipeVoie,tgVioHandle are relevant
	*	@return SUCCESS or error code.
	*	@warning While the API makes usage of this function, StopPipeFct is not meant for pipes
	*	belonging to different boards anymore (by opposition to the real np API), erratical behave may result.
	*/  
	virtual	WORD	IFlow_PipeStop( IN  PTARGET_INFO    PmPtrTgPipe) = 0;

	
	/**
	*	FLOW Interface : Get a Pipe's State.
	*	
	*	@param PmPtrTgPipe	IN	tgCaracPipeVoie and tgPipe are relevant
	*	@param PmState		OUT	1 if pipe is started, 0 if paused/stop
	*
	*	@return		SUCCESS or error code.
	*	@warning	don't poll too much on this,
	*	the StartPipe function already insures that a Pipe is started, if no "DSP BUSY" occured.
	*	@see		#vio_TogglePipeState .
	*/     
    virtual	WORD	IFlow_PipeState(	IN  PTARGET_INFO PmPtrTgPipe,
								OUT PBYTE        PmState ) = 0;

	/**
	*	FLOW Interface : Remove 1 Pipe
	*
	*  @param PmPtrTgPipe	IN	Identies the target pipe.
    *  @param PmCommit		IN	Whether to forward the request to the board or not
    *  @return  SUCCESS or error code.
	*/     
	virtual	WORD	IFlow_PipeRelease(	IN  PTARGET_INFO PmPtrTgPipe,
                                IN  BOOLEAN      PmCommit ) = 0;
    
	/**
	*	FLOW Interface :	Retrieve the Pipe Time in samples
	*
	*	@param		PmPtrTgPipe		IN	tgPipe,tgCaracPipeVoie,tgVioHandle are relevant
	*	@param		PmSampleCount	OUT	returned Sample Count
	*	@return SUCCESS or error code.
	*	@warning While the API makes usage of this function, StopPipeFct is not meant for pipes
	*	belonging to different boards anymore (by opposition to the real np API), erratical behave may result.
	*/  
    virtual	WORD	IFlow_PipeSampleCount(  IN	PTARGET_INFO    PmPtrTgPipe     ,
									OUT PPCX_TIME       PmSampleCount   ) = 0;

	/**
	*	FLOW Interface :	Settings for a Pipe Effect
	*	@brief VX Supports the following effects :
	*
	*	- <b> EFFECT_SCRUB </b> :
	*		-# <b>Scrub Ratio</b>: 16384*r, where r ranges from -2.0 to 2.0 
	*
	*	- <b> EFFECT_EQUALIZATION_PIPE </b>
	*		-# <b>Setup</b>		: [Phase | (Type << 8)], where
	*							-  Phase is the phase to set (may be 1 2 or 3)
	*							-  Type  is the type of the phase (may be 1 for LP 2 for BP or 3 for HP)
	*		-# <b>Gain</b>		: the Gain in dB/4 shifted by an offset on +18, i.e. 0 => -18dB, 144 => +18dB
	*		-# <b>Wo</b>		: the Cut-off pulsation, ranges from 0 to PI/4 in DSP fractional form.
	*							-	=Pi/2.Fc/Fe
	*		-# <b>Wb_slope</b>	: depending on the type on filter, may be an order (1/2/3) and hence a slope of -6/-12-18dB per octave,
	*							-	or a BandWidth, when the filter is a BP. The bandwidth ranges from 0 to PI/4 in DSP fractional form.
	*							-	=Wo/Q
	*
	*	- <b> EFFECT_MAXIMIZER </b>
	*
	*	@param		PmTargetPtr		IN	Target Pipe	
	*	@param		PmEffectID		IN	one value amongst the supported effects	
	*	@param		PmNbParams		IN	Number of passed parameters		
	*	@param		PmpParamsArray	IN	Array of passed parameters
	*	@warning	the caller must make sure that the EXACT number of parameters are available.
	*	@return		SUCCESS or error code.
	*	@see		ah_ctrl.c pcxall_e.h	
	*/ 
	WORD	IFlow_PipeSetEffect(
        IN	PTARGET_INFO		PmTargetPtr,
		IN	PCX_EFFECT_HANDLE 	PmEffectID,
		IN	WORD				PmNbParams,
        IN	PDWORD				PmpParamsArray ) { return ED_UNAVAILABLE_FEATURE; }; 

	
	/**
	*	FLOW Interface : Download a context for an effect (Freq Change).
	*	
	*	@param PmTargetPtr			IN	tgCarte at least is relevant
	*	@param PmNbBuffers			IN	The number of buffer to download
	*	@param PmFirmwareBuffArray	IN	array of FIRMWARE_BUFFER describing address and size of those
	*	@return		SUCCESS or error code.
	*/   
	WORD	IFlow_LoadContext(  IN WORD					PmContextID,
								IN PCX_EFFECT_HANDLE	PmEffectID,
								IN WORD					PmNbBuffers,
								IN LPFIRMWARE_BUFFER	PmFirmwareBuffArray ) { return ED_UNAVAILABLE_FEATURE; }; 



	/**
	*	FLOW Interface :	Retrieve the Pipe Time in samples
	*
	*	@param		PmTgPipe		IN	Identify the target pipe.
	*	@param		PmHasMpeg		IN	TRUE if PmMeg is valid.
	*	@param		PmMpeg			IN	1 to set Mpeg format for each stream of the pipe.
	*	@param		PmHasBackward	IN	TRUE if PmBackward is valid.
	*	@param		PmBackward		IN	1 to set the pipe in bacward mode, 0 otherwise.
	*	@return SUCCESS or error code.
	*/ 
	WORD	IFlow_PipeOutSetParam(	IN PTARGET_INFO PmPtrTgPipe,
									IN BOOLEAN      PmHasMpeg,
									IN BYTE         PmMpeg,
									IN BOOLEAN      PmHasBackward,
									IN BYTE         PmBackward ) { return ED_UNAVAILABLE_FEATURE; }; 

/*@}*/


    // IIIIIIIII    Interface BUFFER      IIIIIIIIIIIIII//

/**
 * \defgroup Buffer Buffer Management Interface
 */
/*@{*/

	/**
	*	BUFFER Interface : Post a Buffer to be queued to the Target Stream's buffer queue
	*	@param	PmTarget	fields tgCaracPipeVoie and  tgVioHandle are mandatory
	*	@param	PmPtBuffer	points to the REQUETS_INFO to be posted
	*	@return SUCCESS or an error code.
	*/
		WORD	IBuffer_GiveStreamBuffer(   PTARGET_INFO        PmTarget,
										LPBC_HEADER_INFO    PmPtBuffer,
										WORD                PmNbStreams );

	/**
	*	BUFFER Interface :	function added to manage IRP cancellation : this function simply removes the buffer from the stream, with no extra processing.
	*
	*	@param		PmPtrTgPipe			IN	tgPipe,tgCaracPipeVoie,tgDsp,tgVioHandle are relevant
	*	@param		PmEndOfCondition	IN	BOOLEAN TRUE if a end of play or end of record for the pipe has been received.
	*	@param		PmNbStreams			IN	Number of streams of the pipe.
	*	@return SUCCESS or error code.
	*/  
	virtual	VOID    IBuffer_CancelStreamBuffer( IN  PTARGET_INFO	PmPtrTgPipe  ,
												IN  LPBC_HEADER_INFO    PmPtBuffer  ) = 0;

	/**
	*	BUFFER Interface :	Transfer the sound for all the streams of a pipe. 
	*	If the pipe is in play, manages the pause bit and the end of play., In record, manage the
	*	end of record.
	*
	*	@param		PmPtrTgPipe			IN	tgPipe,tgCaracPipeVoie,tgDsp,tgVioHandle are relevant
	*	@param		PmEndOfCondition	IN	BOOLEAN TRUE if a end of play or end of record for the pipe has been received.
	*	@param		PmNbStreams			IN	Number of streams of the pipe.
	*	@return SUCCESS or error code.
	*/  
		WORD	IBuffer_TransferSoundPipe(  IN  PTARGET_INFO    PmPtrTgPipe,
										IN  BOOLEAN         PmEndOfCondition,
										IN  WORD            PmNbStreams);

	/**
	*	BUFFER Interface :	Removes sound buffers for a stream. 
	*	If the pipe is in play, manages the pause bit and the end of play., In record, manage the
	*	end of record.
	*
	*	@param		PmPtrTgPipe			IN	tgPipe,tgCaracPipeVoie,tgDsp,tgVioHandle are relevant
	*	@param		PmFirstBuffer		IN	The Number of the buffer where to cut the queue.
	*	@return SUCCESS or error code.
	*/  
		WORD	IBuffer_BypassPauseAtBufferEnd(	IN  PTARGET_INFO        PmPtrTgPipe,
											IN  DWORD               PmBufferNum );
	
	/**
	*	BUFFER Interface :	Removes sound buffers for a stream. 
	*	If the pipe is in play, manages the pause bit and the end of play., In record, manage the
	*	end of record.
	*
	*	@param		PmPtrTgPipe			IN	tgPipe,tgCaracPipeVoie,tgDsp,tgVioHandle are relevant
	*	@param		PmFirstBuffer		IN	The Number of the buffer where to cut the queue.
	*	@return SUCCESS or error code.
	*/  
		WORD	IBuffer_CutBuffersQueued(   IN  PTARGET_INFO        PmPtrTgPipe  ,
											IN  DWORD               PmFirstBuffer,
											OUT  PDWORD              PmNbBuffers );

	/**
	*	BUFFER Interface : In answer to PortCls.sys, unqueue the REQUEST_INFO with the matching tag.
	*	@param	PmTargetPtr	the target stream
	*	@param	PmTag		the DirectSound tag to be searched for, and canceled.
	*	@see	CMiniportWaveICHStream::RevokeMappings()
	*	@return SUCCESS or error code.
	*/
		WORD	IBuffer_RemoveBufferFromQueue(  IN  PTARGET_INFO        PmTargetPtr,
												IN  DWORD               PmBuffNum);

/*@{*/


    // IIIIIIIII    Interface EVENT      IIIIIIIIIIIIII//

/**
 * \defgroup Events/Notifications and Interrupt Management Interface
 */
/*@{*/

			BOOLEAN	IEvent_AsynchronousEventPending() { return (m_SrcIT != 0); }

	virtual	BOOLEAN	IEvent_AcknowledgeIRQ(BOOLEAN *PmTimerOccured, BOOLEAN *PmAsyncPending, BOOLEAN* PmAsyncESCMD) = 0;
	

	/**
	*	EVENT Interface : Retrieve the IRQ source, and Asynchrounous Events details.
	*	@param PmPEndStartOnTimeCode	OUT	is TRUE if a timecode event was received (pipe start on LTC).
	*	@param PmPFrequencyChange	OUT	is TRUE if the UER clock changed.
    *   @param   PmPNotifiedOutPipeMask
    *	@param   PmPNotifiedInPipeMask
	*	@param	 PmES_Response
	*	@return  SUCCESS or error code.
	*	@see	#IEvent_TestITSrc	
	*/     
	virtual WORD	IEvent_AsynchronousEvents(  OUT PBOOLEAN         PmPEndStartOnTimeCode,
												OUT PBOOLEAN         PmPFrequencyChange,
												OUT PDWORDLONG       PmPNotifiedOutPipeMask,
												OUT PDWORDLONG       PmPNotifiedInPipeMask,
												OUT PES_DRV_RESP	 PmES_Response) = 0;

	/**
	*	EVENT Interface : Retrieve the IRQ source.
	*	The IRQ service routine should always prompt the board for the reason
	*	why it sent an Interrupt to the PC. IEvent_AsynchronousEvents calls this methode on DSP cards, and processes the returned value,
	*	to retrieve complementary data.
	*	@param PmpWord	OUT	receives the cause of the irq
	*	@return  SUCCESS or error code.
	*	@see	#IEvent_AsynchronousEvents	
	*/     
	virtual	WORD	IEvent_TestITSrc( OUT PDWORD   PmWord )
					{
					 *PmWord = m_SrcIT;
					 return SUCCESS;
					};  


	/**
	*	EVENT Interface : retrieve the notification data for a given pipe.
	*	@param		PmTgPipe	PTARGET_INFO	:  Identify the target pipe
	*	@param      PmCodePtr	PWORD			: the notified command code or 0 if none
	*	@param      PmTimePtr	PPCX_TIME		: the execution time
	*	@param      PmDelayPtr	PBYTE			: the type of differed delay
	*	@param      PmParam1Ptr	PDWORD			: 1st param of the notified command
	*	@param      PmParam2Ptr	PDWORD			: 2nd param of the notified command
	*	@param      PmCancelled	PBOOLEAN		: the notified command was cancelled.
	*	@return  SUCCESS or error code.
	*/ 
	virtual	WORD	IEvent_PipeNotification(	IN  PTARGET_INFO    PmTargetPtr ,
													OUT PWORD           PmCodePtr   ,
													OUT PPCX_TIME       PmTimePtr   ,
													OUT PBYTE           PmDelayPtr  ,
													OUT PDWORD          PmParam1Ptr ,
													OUT PDWORD          PmParam2Ptr ,
													OUT PBOOLEAN        PmLast      )  { return ED_UNAVAILABLE_FEATURE; }; 
	/**
	*	EVENT Interface : differed command, pipes to notify at a given STREAM time.
	*	@param PmTgPipe	IN	Target Pipe/Stream ARRAY.
	*	@param PmNbPipe	IN	the number of targets in the above table.
	*	@return  SUCCESS or error code.
	*/ 
	virtual	WORD    IEvent_StreamNotifyTime(	IN  PTARGET_INFO    PmTgPipe,
										IN  WORD            PmNbPipe    ) { return ED_UNAVAILABLE_FEATURE; }; 

	/**
	*	EVENT Interface : differed command, pipes to notify at a given time.
	*	@param PmTgPipe	IN	Target Pipe ARRAY.
	*	@param PmNbPipe	IN	the number of targets in the above table.
	*	@return  SUCCESS or error code.
	*/ 
	virtual	WORD	IEvent_PipeNotifyTime(	IN  PTARGET_INFO    PmTgPipe,
									IN  WORD            PmNbPipe    ) { return ED_UNAVAILABLE_FEATURE; }; 

	/**
	*	EVENT Interface : Purge Pipe differed commands.
	*	@return  SUCCESS or error code.
	*/ 
	virtual	WORD    IEvent_PipePurgeDCmds(	IN PTARGET_INFO         PmPtrTgPipe,
									IN BYTE                 PmDiffered,
									IN WORD                 PmCommandCode,
									IN DWORD                PmParam2 ) { return ED_UNAVAILABLE_FEATURE; }; 
	/**
	*	EVENT Interface : Purge Stream differed commands.
	*	@return  SUCCESS or error code.
	*/ 
	virtual	WORD    IEvent_StreamPurgeDCmds(	IN PTARGET_INFO         PmPtrTgPipe,
										IN BYTE                 PmDiffered,
										IN WORD                 PmCommandCode,
										IN DWORD                PmParam2 ) { return ED_UNAVAILABLE_FEATURE; }; 

	/**	
	*	EVENT Interface :	This command is used to guarantee the DSP (on a NP board) has finished any
	* coding process before purging it. Otherwise, the purge process is guaranteed
	* to crash the DSP.
	* Note that this command (i.e. wait for 30 ms) must be called after a pause
	* cmd (pause stream or pause pipe) but before the call to VIOTransferSoundPipe().
	* Initialy, this was done in VIOTransferSoundPipe() between the buffers request
	* (needed/freed...) and the buffer purge. However, we were frequently in
	* trouble, because during the wait, the DSP was sending an IRQ (meaning it has
	* freed a buffer and needs a new one) but, as we were not able to handle the
	* IRQ (because we were still in the driver), we were ending loosing a buffer.
	*
	*	@param   IsNbBoard: TRUE if the board is a NP one, FALSE otherwise.
	*	@return  SUCCESS or error code.
	*/
	virtual	VOID	IEvent_WaitEndOfCoding( IN PTARGET_INFO PmTarget ) {
				PCX_WAIT_AT_LEAST_MILLI_SEC(30) ;
	}

	virtual	WORD	IEvent_WaitEndOfStartPipe( IN PTARGET_INFO PmTarget ) {
				return SUCCESS ;
    }
/*@}*/

		VOID    vio_FreeBuffer(         IN  BYTE                PmCaracPipeVoie,
			                            IN  PSTREAM_INFO        PmStreamPtr,
				                        IN  LPBC_HEADER_INFO    PmReqPtr );

	virtual	WORD	vio_InitSoundTransferNP(	IN PTARGET_INFO PmPtrTarget,
										IN WORD         PmNbOfStreams,
										OUT PDWORD      PmNeededBuffMaskPtr,
										OUT PDWORD      PmFreedBuffMaskPtr,
										OUT PDWORD      PmBufferSizeArray,
										OUT PDWORD      PmFreeMoreBuff) 
										{ return ED_UNAVAILABLE_FEATURE; }; 
	
	/**
	*	This will ask for the buffer status on the given pipe.
	*	@warning	on LXES, all finished buffers are freed at once with this command
	*				and must be notified to the application (no "More" bit)
	*
	*	@param	PmNbNeededPtr		IN Pointer to the variable that receives the number of Needed buffers.
	*	@param	PmNbFreedPtr		IN Pointer to the variable that receives the number of Freed buffers.
	*	@param	PmBufferSizeArray	IN array of at least MAX_STREAM_BUFFERS DWORDS, that receive the size of the freed capture buffers.
	*/
	virtual	WORD	vio_InitSoundTransferLX(	IN PTARGET_INFO PmPtrTarget,
										OUT PDWORD      PmNbNeededPtr,
										OUT PDWORD      PmNbFreedPtr,
										OUT PDWORD      PmBufferSizeArray
										) { return ED_UNAVAILABLE_FEATURE; }; 

	/**
	*	Private Method : Purge any waiting record buffer on the specified stream.
	*
	*	@param	PmPtStream	IN Pointer onto the stream descriptor.
	*/
		VOID    vio_StreamPurgeRecBuffers(	IN  PSTREAM_INFO    PmPtStream  )
	{
		LPRECORD_REQ_INFO LcBuffer, LcNextBuffer ;
		DWORD           LcBufferCount          = 0 ;   // ## FS - Security add

		if ( PmPtStream != NULL )
		{
			LcBuffer = (LPRECORD_REQ_INFO) PmPtStream->siAdrBuffer ;

			while ( (LcBuffer != NULL ) && ( LcBufferCount < MAX_BUFFER ) )
			{
				// Next buffer if any
				// ------------------
				LcNextBuffer = (LPRECORD_REQ_INFO) LcBuffer->rcqForDriver.dprAdrNextBlk;

                // MBR(26/11/2002) no more APHFreeRecordBuffer( LcBuffer );
                vio_FreeBuffer(OPER_REC, PmPtStream, (LPBC_HEADER_INFO)LcBuffer) ;

				LcBufferCount ++ ;

				LcBuffer = LcNextBuffer ;
			}

			PmPtStream->siAdrBuffer = NULL ;
			PmPtStream->siAdrNextBufferToGive = NULL  ;
		}
	};


	
	/**
	*	Private Method : Purge any waiting buffer on the specified stream.
	*
	*	@param	PmPtStream	IN Pointer onto the stream descriptor.
	*/
		VOID    vio_StreamPurgeBuffers(		IN  PSTREAM_INFO    PmPtStream  )
	{
		LPPLAY_REQ_INFO LcBuffer, LcNextBuffer ;
		DWORD           LcBufferCount          = 0 ;   // ## FS - Security add

		if ( PmPtStream != NULL )
		{
			LcBuffer = (LPPLAY_REQ_INFO) PmPtStream->siAdrBuffer ;

			while ( (LcBuffer != NULL ) && ( LcBufferCount < MAX_BUFFER ) )
			{
				// Next buffer if any
				// ------------------
				LcNextBuffer = (LPPLAY_REQ_INFO) LcBuffer->plqForDriver.dprAdrNextBlk;

                // MBR(26/11/2002) no more APHFreeBuffer( LcBuffer );
                vio_FreeBuffer( OPER_PLAY,
                                PmPtStream,
                                (LPBC_HEADER_INFO)LcBuffer );

				LcBufferCount ++ ;

				LcBuffer = LcNextBuffer ;
			}

			PmPtStream->siAdrBuffer = NULL ;
			PmPtStream->siAdrNextBufferToGive = NULL; 
		}
	};


	virtual	WORD	vio_StreamGiveBufferNP(	IN PTARGET_INFO     PmPtrTarget,
											IN WORD             PmCurrentStream,
											IN DWORD            PmBuffDataLength,
											IN DWORD            PmBuffAddress,
											IN BYTE             PmPauseRequest,
											IN BOOLEAN          PmNotifyBuffer,
											IN WORD             Pm9BitBufferTag,
											IN DWORD            PmMisc
											) { return ED_UNAVAILABLE_FEATURE; }; 

		/**
		*	Send a buffer for play or capture on a LXES card
		*
		*   @param	IN	PmPtrTarget
		*   @param	IN	PmBuffDataLength
		*   @param	IN	PmBuffAddressLo
		*   @param	IN	PmBuffAddressHi		set to zero if 64bit transfer not supported, or HIGH part is zero
		*   @param	IN	PmPauseRequest
		*   @param	IN	PmNotifyBuffer
		*   @param	IN	PmMisc
		*	@param	OUT PmBufferIndex
		*	@see	vio_StreamGiveBufferNP
		*/
	virtual	WORD vio_StreamGiveBufferLX(IN	PTARGET_INFO     PmPtrTarget,
										IN	DWORD            PmBuffDataLength,
										IN	DWORD            PmBuffAddressLo,
										IN	DWORD            PmBuffAddressHi,
										IN	BYTE             PmPauseRequest,
										IN	BOOLEAN          PmNotifyBuffer,
										IN	DWORD            PmMisc,
										OUT	PBYTE			 PmBufferIndex
										) { return ED_UNAVAILABLE_FEATURE; }; 

	virtual	WORD	vio_StreamCancelBuffer  (   IN PTARGET_INFO     PmPtrTarget,
										IN WORD             PmCurrentStream,
										IN DWORD            PmBuffDataLength,
										IN DWORD            PmBuffAddress) = 0;

	virtual	WORD	vio_FreeBufferAndGetSizeNP( IN  PTARGET_INFO    PmPtrTarget,
										IN  WORD            PmCurrentStream,
										OUT PDWORD          PmBufferSizePtr) = 0;


	virtual	WORD	vio_SoundTransferPlayNP(	IN  PTARGET_INFO    PmTarget,
										IN  BOOLEAN         PmEndOfPlayRequested,
										IN  WORD            PmNbStreams) = 0;

	virtual	WORD	vio_SoundTransferRecordNP(  IN  PTARGET_INFO    PmTarget,
										IN  BOOLEAN         PmFinishing,
										IN  WORD            PmNbStreams)= 0;

    virtual WORD    vio_SupportedPRFeatures();

    virtual	WORD	vio_TogglePipeState(
    IN BYTE         PmState,
    IN BYTE         PmCondition,
    IN BYTE         PmNbPipe,
    IN PTARGET_INFO PmPtrTgPipe,
    IN PTIME_CODE_START_INFO PmPTimeCodeStart) = 0;

};


#endif // _NPPROTOCOL_H





