// *************************************************************************
//
//  COPYRIGHT 1996-2000 DIGIGRAM. ALL RIGHTS RESERVED.
//
//  DIGIGRAM
//
// **************************************************************************

#include "PIOCommands.h"


// STATIC MEMBERS
//                                                     liMinReal  liMinCoded  liMaxCoded
const LEVEL_INFO  CPIOCommands::m_driverLevelInfo = { -10975,     0x0000,     0x01b7 };
      

//////////////////////////////////////////////////////////////////////
// Wait a number of MicroSeconds
//////////////////////////////////////////////////////////////////////

VOID CPIOCommands::WaitForN( ULONG PmDuration )
{
	if( (KeGetCurrentIrql() != PASSIVE_LEVEL) || (PmDuration < 50) )
	{
#ifdef DBG_VIEWER
		// normalement pas plus que 50 MicroSec avec cette fonction
		if(PmDuration > 200) DOUT(DBG_WARNING, ("WARNING PCXHR KeStallExecutionProcessor(%d)\n", PmDuration));
#endif
		KeStallExecutionProcessor(PmDuration);
	}
	else
	{
		LARGE_INTEGER WaitTime;
		// 100 ns base; negative value is relative time.
		WaitTime.QuadPart = (LONGLONG)(-10) * (LONGLONG)PmDuration;
		KeDelayExecutionThread (KernelMode, FALSE, &WaitTime);
	}
}

//////////////////////////////////////////////////////////////////////
// Construction/Destruction
//////////////////////////////////////////////////////////////////////

CPIOCommands::CPIOCommands(IOPCIDevice* pIOPCIDevice)
{
    m_piXilinxTestDone = FALSE;

    m_piSn              = 1;
    m_piNbWcl           = 0;
	m_piNbVideoSync     = 0;
    m_piEtherSoundSync  = 0;
    m_piMiscFeatures    = 0;    // Presence of SMPTE...
    m_piOptions = 0;            // VX_PRO_MODE_OPTION, PCXVX_V2_OPTION, ...
    m_piType = 0;               // board family PCXVX_BOARD, PCXVX_2_BOARD

	m_piMACAddressMSB = 0;
	m_piMACAddressLSB = 0;

    BZERO2(m_wOutAnalogLevels, WORD, MAX_BOARD_OUTPUTS);
    BZERO2(m_wInAnalogLevels,  WORD, MAX_BOARD_INPUTS);
    BZERO2(m_wInMicroLevels,   WORD, MAX_BOARD_INPUTS);
    BZERO2(m_bInAnalogMute,    BYTE, MAX_BOARD_INPUTS);
    BZERO2(m_bInMicroMute,     BYTE, MAX_BOARD_INPUTS);

    BZERO2(m_piDescAudioOut,   WORD, MAX_BOARD_OUTPUTS / 2);    // piDescAudioOut describes stereo audio
    BZERO2(m_piDescAudioIn,    WORD, MAX_BOARD_INPUTS / 2);     // piDescAudioIn  describes stereo audio
    BZERO2(m_bPhysInFeature2,  BYTE, MAX_BOARD_INPUTS / 2);     // contains CARD_FEATURES_INFO.ciPhysicalInFeature2 parameters
    BZERO2(m_bPhysOutFeature2, BYTE, MAX_BOARD_OUTPUTS / 2);    // contains CARD_FEATURES_INFO.ciPhysicalOutFeature2 parameters

    m_piStereoOutNumber = 0;
    m_piStereoInNumber = 0;

    BZERO2(&m_LevelInfo, BOARD_LEVEL_INFO, 1);

    m_DriverCodedAnalogBoardValueFor0dB = 0;

    MEMSET2(m_piTypeSrc, DATA_FROM_NO_INPUT, BYTE, MAX_BOARD_INPUTS);   // set to no input

    MEMSET2(m_ucUerByte, UER_MODE_UNKNOWN, BYTE, MAX_BOARD_INPUTS / 2); // set to unknown
    BZERO2(m_ucUerCBitFreq, DWORD, MAX_BOARD_INPUTS / 2);

    m_piActualClkFrequency  = 0;                    // unit is Hz
    m_piActualClkSource     = CLOCK_TYPE_NONE;      // CLOCK_TYPE_INTERNAL/CLOCK_TYPE_UER_SYNCHRO...
    m_piActualClkInputNum   = 0;                    // relevant only if ClockSource == UER_SYNC | WORD_CLOCK

    m_piActualClkType       = INDEPENDANT_CLOCK;    // SLAVE_CLOCK/MASTER_CLOCK...
    m_piActualClkMasterNum  = 0;                    // relevant only if ClkType == SLAVE
    m_piActualClkSlaveMask  = 0;                    // relevant only if ClkType == MASTER

    m_piIsFrequencyFirstChange = FALSE;

    // Initialize Object

    BZERO2(m_aIOOffset, PBYTE, eMaxPort);

    // is it a VX222, VX222_MIC, PCX924, PCX924_MIC or PCX22 ?
    m_dwBoardVersion = pIOPCIDevice->dwBoardVersion;
    m_ucBoardFlavor  = pIOPCIDevice->ucBoardFlavor;

    m_EnableSendMessage = TRUE;

    m_SendMessageLocked = 0;

    // Init address offset and MemoryMap for each register!

    /* voir doc PLX */
    m_aIOOffset[ePLX_PCICR]     = pIOPCIDevice->AddressPLX + 0x04;      // PCICR
    m_aIOOffset[ePLX_MBOX0]     = pIOPCIDevice->AddressPLX + 0x40;		
    m_aIOOffset[ePLX_MBOX1]     = pIOPCIDevice->AddressPLX + 0x44;		
    m_aIOOffset[ePLX_MBOX2]     = pIOPCIDevice->AddressPLX + 0x48;		
    m_aIOOffset[ePLX_MBOX3]     = pIOPCIDevice->AddressPLX + 0x4c;		
    m_aIOOffset[ePLX_MBOX4]     = pIOPCIDevice->AddressPLX + 0x50;		
    m_aIOOffset[ePLX_MBOX5]     = pIOPCIDevice->AddressPLX + 0x54;		
    m_aIOOffset[ePLX_MBOX6]     = pIOPCIDevice->AddressPLX + 0x58;		
    m_aIOOffset[ePLX_MBOX7]     = pIOPCIDevice->AddressPLX + 0x5c;		
    m_aIOOffset[ePLX_L2PCIDB]   = pIOPCIDevice->AddressPLX + 0x64;		// L2PDBELL
    m_aIOOffset[ePLX_IRQCS]     = pIOPCIDevice->AddressPLX + 0x68;		// INTCSR
    m_aIOOffset[ePLX_CHIPSC]    = pIOPCIDevice->AddressPLX + 0x6C;	    // CNTRL
}

CPIOCommands::~CPIOCommands()
{
}


// ######################################################################################################################
// physio.c



LONG CPIOCommands::PIOConvert( LONG PmX1, LONG PmY1, LONG PmX2, LONG PmY2, LONG PmX )
{
    LONG LcA;
    LONG LcB;
    LONG LcY;

    LcA = PmY2 - PmY1;
    LcB = PmX2 * PmY1 - PmX1 * PmY2;

    LcY = ( LcA * PmX + LcB ) / ( PmX2 - PmX1 );

    return( LcY );
}


LONG CPIOCommands::PIOConvertLevelDriverCoded2BoardCoded( PLEVEL_INFO PmPBoardLevelInfo, LONG PmDriverCodedLevel )
{
    LONG LcRealLevel;
    LONG LcBoardCodedLevel;
    LONG LcMin;
    LONG LcMax;

    // Convert the driver coded level into the real level.
    //
    LcRealLevel = PIOConvert( m_driverLevelInfo.liMinCoded,  // X1
                              m_driverLevelInfo.liMinReal,   // Y1
                              m_driverLevelInfo.liMaxCoded,  // X2
                              0,                           // Y2
                              PmDriverCodedLevel );        // X

    // Convert the real level into the board coded level.
    //
    if(LcRealLevel <= m_driverLevelInfo.liMinReal) // let's mute the board if smaller than -109.5dB
    {
        LcBoardCodedLevel = PmPBoardLevelInfo->liMinCoded;
    }
    else
    {
        LcBoardCodedLevel = PIOConvert( PmPBoardLevelInfo->liMinReal,  // X1
                                        PmPBoardLevelInfo->liMinCoded, // Y1
                                        0,                             // X2
                                        PmPBoardLevelInfo->liMaxCoded, // Y2
                                        LcRealLevel );                 // X
    }

    // The board coded level must belong to the board coded level range.
    //
    LcMin = MIN( PmPBoardLevelInfo->liMinCoded, PmPBoardLevelInfo->liMaxCoded );
    LcMax = MAX( PmPBoardLevelInfo->liMinCoded, PmPBoardLevelInfo->liMaxCoded );
    LcBoardCodedLevel = MAX( LcBoardCodedLevel, LcMin );
    LcBoardCodedLevel = MIN( LcBoardCodedLevel, LcMax );

    return LcBoardCodedLevel ;
}



WORD CPIOCommands::PIOSetAudioAnalogLevel(WORD PmBoardAudioNum, BYTE PmCaracPipeVoie, WORD PmDriverCodedLevel )
{
    WORD        LcDescAudio;
    WORD        LcNewBoardLevel;
    PLEVEL_INFO LcPLevelInfo;
    PWORD       LcPOldBoardLevel;

    DOUT(DBG_PIO_CMD, ("CPIOCommands::PIOSetAudioAnalogLevel\n"));
    // Verify that the board has analog audio input/output
    //
    if  ( PmCaracPipeVoie == OPER_PLAY )
    {
        // Beware: piDescAudioOut describes stereo audio
        //
        LcDescAudio = m_piDescAudioOut[PmBoardAudioNum/2];

        // Check if the feature is supported
        //
        if (    ( !(LcDescAudio & AUDIO_ANALOG_MSK) )            // Analog input not present
             || ( !(LcDescAudio & AUDIO_LEVEL_MSK) ) )           // Analog levels not presents
        {
            EXIT_PCX_ERROR( WD_NO_HARDWARE_SUPPORT );
        }

        LcPLevelInfo = & ( m_LevelInfo.bliOutputLevel );
        LcPOldBoardLevel = &m_wOutAnalogLevels[PmBoardAudioNum];
    }
    else    // OPER_REC
    {
        // Verify that the board has analog audio input/output
        //
        if (!CheckAnalogLevelsSupport( PmBoardAudioNum ))
        {
                        EXIT_PCX_ERROR( WD_NO_HARDWARE_SUPPORT );
        }

        LcPLevelInfo = & ( m_LevelInfo.bliInputLevel );
        LcPOldBoardLevel = &m_wInAnalogLevels[PmBoardAudioNum];
    }

    // Convert the driver coded level into the board coded level.
    //
    LcNewBoardLevel = (WORD) PIOConvertLevelDriverCoded2BoardCoded(
                                LcPLevelInfo,
                                (LONG) PmDriverCodedLevel );

    // check if current level is already set, as this
    // may take some time
    if (*LcPOldBoardLevel == LcNewBoardLevel) {
        //DbgPrint("Level CACHE HIT !\n");
        return SUCCESS;
    }

    // Save the board coded level.
    //
    *LcPOldBoardLevel = LcNewBoardLevel;

    HALSetAudioAnalogLevel( PmBoardAudioNum, (PmCaracPipeVoie != OPER_PLAY) );

    return SUCCESS;
}

WORD CPIOCommands::PIOSetInAudioLevels(WORD PmBoardAudioNum, LPIN_AUDIO_SET_LEVEL_REQ_INFO PmPLevels )
{
    WORD    LcRet = SUCCESS;
    BOOL    LcChange = FALSE;
    WORD    LcNew;

    DOUT(DBG_PIO_CMD, ("CPIOCommands::PIOSetInAudioLevels\n"));

    // Verify that the board has analog audio input/output
    //
    if (!CheckAnalogLevelsSupport( PmBoardAudioNum )) 
        EXIT_PCX_ERROR( WD_NO_HARDWARE_SUPPORT );

    // Convert the driver coded level into the board coded level.
    //
    if( PmPLevels->iasliPar1 & IN_AUDIO_SET_LEVEL_ANALOG_MASK )
    {
        LcNew = (WORD) PIOConvertLevelDriverCoded2BoardCoded(
                                &m_LevelInfo.bliInputLevel ,
                                PmPLevels->iasliAnalogLevel );

        if( m_wInAnalogLevels[PmBoardAudioNum] != LcNew )
        {
            m_wInAnalogLevels[PmBoardAudioNum] = LcNew;
            LcChange = TRUE;
        }
    }
    if( PmPLevels->iasliPar1 & IN_AUDIO_SET_LEVEL_MUTE_ANALOG_MASK )
    {
        if( m_bInAnalogMute[PmBoardAudioNum] != PmPLevels->iasliAnalogMute )
        {
            m_bInAnalogMute[PmBoardAudioNum] = PmPLevels->iasliAnalogMute;
            LcChange = TRUE;
        }
    }

    if( m_bPhysInFeature2[PmBoardAudioNum/2] & AUDIO_MIXED_MICRO_MSK )
    {
        if( PmPLevels->iasliPar1 & IN_AUDIO_SET_LEVEL_MICRO_MASK )
        {
            LcNew = (WORD) PIOConvertLevelDriverCoded2BoardCoded(
                                &m_LevelInfo.bliInputLevel ,
                                PmPLevels->iasliMicroLevel );

            if( m_wInMicroLevels[PmBoardAudioNum] != LcNew )
            {
                m_wInMicroLevels[PmBoardAudioNum] = LcNew;
                LcChange = TRUE;
            }
        }
        if( PmPLevels->iasliPar1 & IN_AUDIO_SET_LEVEL_MUTE_MICRO_MASK )
        {
            if( m_bInMicroMute[PmBoardAudioNum] != PmPLevels->iasliMicroMute )
            {
                m_bInMicroMute[PmBoardAudioNum] = PmPLevels->iasliMicroMute;
                LcChange = TRUE;
            }
        }
    }
    else    // other cards than VX222-Mic or PCX924-Mic cannot do this! Give warning, but set the Analog values!
    {
        if( PmPLevels->iasliPar1 & (IN_AUDIO_SET_LEVEL_MUTE_MICRO_MASK | IN_AUDIO_SET_LEVEL_MICRO_MASK) )
        {
            LOAD_PCX_ERROR( LcRet, WD_NO_HARDWARE_SUPPORT );
        }
    }

    // Set the board coded level on the board
    //
    if( LcChange )
    {
        HALSetAudioAnalogLevel( PmBoardAudioNum, TRUE );
    }

    return LcRet;
}

WORD CPIOCommands::PIOSetInAudioEffects( WORD PmBoardAudioNum, LPIN_AUDIO_EFFECT_REQ_INFO PmPInAudioEffects )
{
    WORD    LcDescAudio;

    DOUT(DBG_PIO_CMD, ("CPIOCommands::PIOSetInAudioEffects\n"));
    // Verify that the board has analog audio input/output
    //
    ASSERT(PmBoardAudioNum < MAX_BOARD_INPUTS);

    LcDescAudio = m_piDescAudioIn[PmBoardAudioNum/2];

    // Exit if hardware input audio effect is not present
    if( ! ( LcDescAudio & AUDIO_MIC_IN_COMPR_MSK ) )
    {
        EXIT_PCX_ERROR( WD_NO_HARDWARE_SUPPORT) ;
    }

    if( (PmPInAudioEffects->iaqIdentifier != HW_EFFECT_COMPRESSOR_LIMITER)  &&
        (PmPInAudioEffects->iaqIdentifier != HW_EFFECT_NONE)                )
    {
        EXIT_PCX_ERROR( WD_NO_HARDWARE_SUPPORT) ;
    }

    if( HALSetInAudioEffect( PmBoardAudioNum,
                            PmPInAudioEffects->iaqIdentifier != HW_EFFECT_NONE,
                            PmPInAudioEffects->iaqParams.hw.iaqCompThreshold,
                            PmPInAudioEffects->iaqParams.hw.iaqCompRatio,
                            PmPInAudioEffects->iaqParams.hw.iaqNoiseThreshold,
                            PmPInAudioEffects->iaqParams.hw.iaqOutputGain    ) )
    {
        return SUCCESS;
    }

    return WD_NO_HARDWARE_SUPPORT;
}


VOID CPIOCommands::PIODefaultConfig( )
{
    DWORD LcRet = SUCCESS;

    DOUT(DBG_PIO_CMD ,("Reset board alias: %d\n", PIOGetTypeNoAlias()));

    LcRet = HALDefaultConfig( );

    DOUT(DBG_BOARD ,("HALDefaultConfig: = %d\n", LcRet));

    // Reset Clock Control members to 48000
    //
    if(LcRet == SUCCESS)
    {
        m_piActualClkType = INDEPENDANT_CLOCK;
        // done in HALDefaultConfig : m_piActualClkSource = CLOCK_TYPE_NONE;
        // done in HALDefaultConfig : m_piActualClkFrequency = 48000;
        m_piActualClkInputNum = 0;
        m_piActualClkMasterNum = 0;
        m_piActualClkSlaveMask = 0;
    }
} // PIODefaultConfig


BOOL CPIOCommands::PIOXilinxLoad( IN PBYTE PmXilinx1File, DWORD PmXilinx1Length, IN PBYTE PmXilinx2File, DWORD PmXilinx2Length )
{
    DWORD           i;

    if( ( PmXilinx1File == NULL )   ||
        ( PmXilinx1Length == 0 )    )
    {
        return FALSE;
    }

	// only one file here
	//
	ASSERT(PmXilinx2File == NULL);

    // reset
    //  
    HALXilinxReset(TRUE);

#ifdef DBG_VIEWER
    LARGE_INTEGER LcPerfFreq;
    LARGE_INTEGER LcTime1 = KeQueryPerformanceCounter(&LcPerfFreq);
#endif

    // Send the data
    //
	HALXlxOutFile( PmXilinx1File, PmXilinx1Length);

#ifdef DBG_VIEWER
    LARGE_INTEGER LcTime2 = KeQueryPerformanceCounter(NULL);
    LONGLONG LcTime = LcTime2.QuadPart - LcTime1.QuadPart;
    if(LcTime)
    {
        LcTime *= 1000000;
        LcTime /= LcPerfFreq.QuadPart;
        DOUT(DBG_ERROR, ("PIOXilinxLoad( %d ) = %I64d us\n", PmXilinx1Length , LcTime));
    }
#endif

    // Test the Xilinx 
    // the dsp awaits at least one irqC whithin 10ms
    // IrqC freq = 48kHz/((k+1)*8) => 6khz if everything ok.
    //
    if( HALTestXilinx() == FALSE)
	{
		DOUT(DBG_ERROR, ("ERROR TestXilinx\n"));
        return FALSE;
	}

    m_piXilinxTestDone = TRUE;

    return TRUE;
}



LONG CPIOCommands::PIOConvertLevelBoardCoded2DriverCoded( PLEVEL_INFO PmPBoardLevelInfo, LONG PmBoardCodedLevel )
{
    LONG LcRealLevel;
    LONG LcDriverCodedLevel;
    LONG LcMin;
    LONG LcMax;

    // Convert the board coded level into the real level.
    //
    LcRealLevel = PIOConvert( (LONG) ( PmPBoardLevelInfo->liMinCoded ),  // X1
                              (LONG) ( PmPBoardLevelInfo->liMinReal ),   // Y1
                              (LONG) ( PmPBoardLevelInfo->liMaxCoded ),  // X2
                              (LONG) 0,                                  // Y2
                              (LONG) PmBoardCodedLevel );                // X

    // Convert the real level into the driver coded level.
    //
    LcDriverCodedLevel = PIOConvert( m_driverLevelInfo.liMinReal,  // X1
                                     m_driverLevelInfo.liMinCoded, // Y1
                                     0,                          // X2
                                     m_driverLevelInfo.liMaxCoded, // Y2
                                     LcRealLevel );              // X

    // The driver coded level must belong to the driver coded level range.
    //
    LcMin = MIN( m_driverLevelInfo.liMinCoded, m_driverLevelInfo.liMaxCoded );
    LcMax = MAX( m_driverLevelInfo.liMinCoded, m_driverLevelInfo.liMaxCoded );
    LcDriverCodedLevel = MAX( LcDriverCodedLevel, LcMin );
    LcDriverCodedLevel = MIN( LcDriverCodedLevel, LcMax );

    return LcDriverCodedLevel ;
}

WORD CPIOCommands::PIOGetAudioAnalogLevel(WORD PmBoardAudioNum, BYTE PmCaracPipeVoie, OUT PWORD PmPDriverCodedLevel )
{
    WORD        LcDescAudio;
    PLEVEL_INFO LcPLevelInfo;
    LONG        LcBoardLevel;

    DOUT(DBG_PIO_CMD, ("CPIOCommands::PIOGetAudioAnalogLevel\n"));

    *PmPDriverCodedLevel = 0 ;

    // Verify that the board has analog audio input/output
    //
    if  ( PmCaracPipeVoie == OPER_PLAY )
    {
        // Beware: piDescAudioOut describes stereo audio
        //
        LcDescAudio = m_piDescAudioOut[PmBoardAudioNum/2];

        // Check if the feature is supported
        //
        if (    ( !(LcDescAudio & AUDIO_ANALOG_MSK) )            // Analog input not present
             || ( !(LcDescAudio & AUDIO_LEVEL_MSK) ) )           // Analog levels not presents
        {
            EXIT_PCX_ERROR( WD_NO_HARDWARE_SUPPORT );
        }
    }
    else    // OPER_REC
    {
        // Verify that the board has analog audio input/output
        //
        if (!CheckAnalogLevelsSupport( PmBoardAudioNum )) 
                        EXIT_PCX_ERROR( WD_NO_HARDWARE_SUPPORT );
    }

    if ( PmCaracPipeVoie == OPER_PLAY )
    {
        LcPLevelInfo = & ( m_LevelInfo.bliOutputLevel );
        LcBoardLevel = m_wOutAnalogLevels[PmBoardAudioNum] ;
    }
    else
    {
        LcPLevelInfo = & ( m_LevelInfo.bliInputLevel );
        LcBoardLevel = m_wInAnalogLevels[PmBoardAudioNum] ;
    }

    // Convert the board coded level into the driver coded level.
    //
    *PmPDriverCodedLevel = (WORD) PIOConvertLevelBoardCoded2DriverCoded(
                                      LcPLevelInfo,
                                      LcBoardLevel );
    return SUCCESS;
}


WORD CPIOCommands::PIOGetInAudioLevels(WORD PmBoardAudioNum, OUT PWORD PmPDriverCodedAnalogLevel, OUT PBYTE PmPAnalogMute, OUT PWORD PmPDriverCodedMicroLevel, OUT PBYTE PmPMicroMute)
{
    DOUT(DBG_PIO_CMD, ("CPIOCommands::PIOGetInAudioLevels\n"));

    // Verify that the board has analog audio input/output
    //
    if (!CheckAnalogLevelsSupport( PmBoardAudioNum )) 
        EXIT_PCX_ERROR( WD_NO_HARDWARE_SUPPORT );

    
    // Convert the board coded level into the driver coded level.
    //
    *PmPDriverCodedAnalogLevel = (WORD) PIOConvertLevelBoardCoded2DriverCoded(
                                      & ( m_LevelInfo.bliInputLevel ),
                                      m_wInAnalogLevels[PmBoardAudioNum] );

    *PmPAnalogMute = m_bInAnalogMute[PmBoardAudioNum];

    // support for VX222-Mic and PCX24-Mic
    if( m_bPhysInFeature2[PmBoardAudioNum/2] & AUDIO_MIXED_MICRO_MSK )
    {
        *PmPDriverCodedMicroLevel = (WORD) PIOConvertLevelBoardCoded2DriverCoded(
                                      & ( m_LevelInfo.bliInputLevel ),
                                      m_wInMicroLevels[PmBoardAudioNum] );
        *PmPMicroMute = m_bInMicroMute[PmBoardAudioNum];
    }
    else
    {
        *PmPDriverCodedMicroLevel = (WORD)m_driverLevelInfo.liMinCoded;
        *PmPMicroMute = TRUE;
    }
    return SUCCESS;
}


/*
* WARNING : no check upon resource reservation is done here, only hardware settings !
*/
WORD CPIOCommands::PIOSetOneAudioSource( IN WORD PmAudioNum, BYTE PmSource )
{
    WORD LcRet = SUCCESS;

    DOUT(DBG_PIO_CMD, ("CPIOCommands::PIOSetOneAudioSource\n"));

    ASSERT (PmAudioNum < MAX_BOARD_INPUTS);
    ASSERT (PmSource != m_piTypeSrc[PmAudioNum]);

    if (  PmSource != DATA_FROM_NO_INPUT) {
        // check HW support
        if (( (PmSource == DATA_FROM_ANALOG ) &&  !( m_piDescAudioIn[PmAudioNum/2] & AUDIO_ANALOG_MSK ) )
            ||
            ( ( PmSource == DATA_FROM_MICRO ) && !( m_piDescAudioIn[PmAudioNum/2] & AUDIO_MICRO_MSK ) )
            ||
            ( ( PmSource == DATA_FROM_DIGITAL_SYNCHRO ) && !( m_piDescAudioIn[PmAudioNum/2] & AUDIO_DIGITAL1_MSK ) )
            ||
            ( ( PmSource == DATA_FROM_DIGITAL_DATA ) && !( m_piDescAudioIn[PmAudioNum/2] & AUDIO_DIGITAL2_MSK ) )
            ||
            ( ( PmSource == DATA_FROM_SYNC_CONNECTOR ) && !( m_bPhysInFeature2[PmAudioNum/2] & AUDIO_SYNC_CONNECTOR_MSK ) )
			)
        {
            return ED_INVALID_BOARD_AUDIO ;
        }
        LcRet = HALSetAudioSource( PmAudioNum, PmSource );
        if ( LcRet != SUCCESS ) return LcRet;
    }
	else
	{
		// if DATA_FROM_NO_INPUT then reset the source to DATA_FROM_ANALOG (even for digital only boards)
		// this avoids strange behaviour with monitoring, if it was DATA_FROM_DIGITAL before.
		// -> defect #3978
		//
		if( m_piTypeSrc[PmAudioNum] != DATA_FROM_ANALOG )
		{
			HALSetAudioSource( PmAudioNum, DATA_FROM_ANALOG );
		}
	}

    m_piTypeSrc[PmAudioNum] = PmSource ;

    return SUCCESS;
}


DWORD  CPIOCommands::PIOGetSourceAudioDependency( IN WORD PmAudioNum )
{
	if( PmAudioNum >= (2 * m_piStereoInNumber) )
		return 0;	// invalid PmAudioNum !

	// result is a stereo mask (0x03 for audio 0 and 1; 0x0C for audio 2 and 3, etc...)
	return (0x03 << (PmAudioNum & ~0x01));
}



WORD CPIOCommands::PIOGetAudioUer( WORD PmBoardAudio, OUT PBYTE PmDataPtr, OUT PBYTE PmModePtr, OUT PDWORD PmCBitFrequ, OUT PBYTE PmUerExtra )
{
	WORD LcRet;
    WORD LcSyncInputNumber;

	if( (PmBoardAudio < 2) && (m_piTypeSrc[PmBoardAudio] == DATA_FROM_SYNC_CONNECTOR))
		LcSyncInputNumber = 0;
	else
		LcSyncInputNumber = (PmBoardAudio/2) + 1;

    // Reset all the returned values (security)
    // ------------------------------
    *PmDataPtr = UER_DATA_UNKNOWN ;
    *PmModePtr = UER_MODE_UNKNOWN ;
	*PmCBitFrequ = 0;
	*PmUerExtra = 0;

    DOUT(DBG_PIO_CMD, ("CPIOCommands::PIOGetAudioUer\n"));

    // Check the audio exists and is an digital input
    if( ( PmBoardAudio >= MAX_BOARD_INPUTS ) ||
        ( ! ( m_piDescAudioIn[PmBoardAudio/2] & (AUDIO_DIGITAL1_MSK | AUDIO_DIGITAL2_MSK) ) ) )
    {
        EXIT_PCX_ERROR( ED_INVALID_BOARD_AUDIO );
    }

    LcRet = HALGetAudioUer( LcSyncInputNumber, NULL, PmDataPtr, NULL );

	if( (LcRet == SUCCESS) && (*PmDataPtr == UER_DATA_PRESENT) )
	{
		HALReadCBitsBasic( LcSyncInputNumber, PmModePtr, PmCBitFrequ, PmUerExtra );
	}
	return LcRet;
}

WORD CPIOCommands::PIOGetAudioCByte( WORD PmBoardAudio, BYTE PmByteID, OUT PBYTE PmValidity, OUT PBYTE PmCByte )
{
	WORD LcRet;
    WORD LcSyncInputNumber;

	if( (PmBoardAudio < 2) && (m_piTypeSrc[PmBoardAudio] == DATA_FROM_SYNC_CONNECTOR))
		LcSyncInputNumber = 0;
	else
		LcSyncInputNumber = (PmBoardAudio/2) + 1;

    DOUT(DBG_PIO_CMD, ("CPIOCommands::PIOGetAudioCByte\n"));

    // Check the audio exists and is an digital input
    if( ( PmBoardAudio >= MAX_BOARD_INPUTS ) ||
        ( ! ( m_piDescAudioIn[PmBoardAudio/2] & (AUDIO_DIGITAL1_MSK | AUDIO_DIGITAL2_MSK) ) ) )
    {
        EXIT_PCX_ERROR( ED_INVALID_BOARD_AUDIO );
    }

    LcRet = HALGetAudioUer( LcSyncInputNumber, NULL, PmValidity, NULL );

	if( (LcRet == SUCCESS) && (*PmValidity == UER_DATA_PRESENT) )
	{
		if( !HALReadCByte( LcSyncInputNumber, PmByteID, PmCByte ) )
		{
			*PmValidity = UER_NO_DATA;
			*PmCByte = 0;
		}
	}

	return LcRet;
}


WORD CPIOCommands::PIOSetParamOutPipe( WORD PmBoardAudio, DWORD PmFrequency, BYTE PmUerByte /*default 0xff */ )
{
    BYTE	LcProUerMode;
	DWORD	LcCBitsFrequency;
	BOOL	LcModeChanged = FALSE;
	
    DOUT(DBG_PIO_CMD, ("CPIOCommands::PIOSetParamOutPipe : freq=%ld, PmUerByte: %d\n", PmFrequency, PmUerByte));

    // Return a warning if the board does not have numerical outputs
    if( ( PmBoardAudio >= MAX_BOARD_OUTPUTS ) ||
        (!( m_piDescAudioOut[PmBoardAudio/2] & AUDIO_DIGITAL_PRESENT_MASK) ) )
    {
		DOUT(DBG_PRINT, ("PIOSetParamOutPipe : !AUDIO_DIGITAL_PRESENT_MASK\n"));
        return WD_NO_HARDWARE_SUPPORT;
    }

    // handle case with same cbits, but different frequency
    //
    if( PmUerByte == 0xff )
        PmUerByte =  m_ucUerByte[PmBoardAudio/2];

	// if never or not defined, set default settings :
	//
	if((PmUerByte & UER_MODE_MASK) == UER_MODE_UNKNOWN)
		PmUerByte =	UER_PROFESSIONAL      |
					UER_IS_AUDIO_MODE     |
					UER_IS_NO_COPYRIGHT   |
					UER_IS_CHANNEL_STEREO ;

    // Bit 0 = Consumer  1 = Professional
    LcProUerMode = ((PmUerByte & UER_MODE_MASK) == UER_PROFESSIONAL);

	// do we change mode from PROFESSIONAL to CONSUMER or vice versa,
	// or was it UER_MODE_UNKNOWN before :
	//
	if ( (PmUerByte & UER_MODE_MASK) != (m_ucUerByte[PmBoardAudio/2] & UER_MODE_MASK) )
	{
		if ( LcProUerMode )
		{
			// PROFESSIONAL mode selected, we must reset the CONSUMER bits
			HALWriteOneCBit(PmBoardAudio,UER_CON_COPYRIGHT_OFFSET,0);       // Reset Copyright Bit! Otherwise modif of Emphasis in Professional Mode!
			HALWriteOneCBit(PmBoardAudio,UER_CON_CATEGORY_OFFSET,0);        // Reset Category Bit9
			HALWriteOneCBit(PmBoardAudio,UER_CON_FREQ_BIT_OFFSET,0) ;       // Reset Frequ Bits !!!
			HALWriteOneCBit(PmBoardAudio,UER_CON_FREQ_BIT_OFFSET+1,0) ;     // Reset Frequ Bits !!!
		}
		else
		{
			// CONSUMER mode selected, we must reset the PROFESSIONAL bits
			HALWriteOneCBit(PmBoardAudio,UER_PRO_FREQ_BIT_OFFSET,0) ;       // Reset Frequ Bits !!!
			HALWriteOneCBit(PmBoardAudio,UER_PRO_FREQ_BIT_OFFSET+1,0);      // Reset Frequ Bits !!!
			HALWriteOneCBit(PmBoardAudio,UER_PRO_STEREO_BIT_OFFSET,0);      // Reset Stereo bit for Consumer Mode !!!!
			// Reset Professional sound data width
			HALWriteOneCBit(PmBoardAudio,UER_PRO_SWL3_BIT_OFFSET,0);        // Source Word Length bits 3-5 = 101 (24 bits)
			HALWriteOneCBit(PmBoardAudio,UER_PRO_SWL5_BIT_OFFSET,0);        //
			HALWriteOneCBit(PmBoardAudio,UER_PRO_AUX1_BIT_OFFSET,0);        // Use of aux data bits 0-2 = 001 (main audio 24 bits)

			// reset of the frequency bits in Byte 4 is done below !
		}

		// write the new mode (PROFESSIONAL or CONSUMER)
		HALWriteOneCBit(PmBoardAudio,UER_MODE_BIT_OFFSET,LcProUerMode);

		if ( LcProUerMode ) // PROFESSIONAL mode selected
		{
			HALWriteOneCBit(PmBoardAudio,UER_PRO_AUX1_BIT_OFFSET,1);        // Use of aux data bits 0-2 = 001 (main audio 24 bits)
			HALWriteOneCBit(PmBoardAudio,UER_PRO_SWL5_BIT_OFFSET,1);        // Source Word Length bits 3-5 = 101 (24 bits)
			HALWriteOneCBit(PmBoardAudio,UER_PRO_SWL3_BIT_OFFSET,1);        // 
		}
		else                // CONSUMER mode selected
		{
			HALWriteOneCBit(PmBoardAudio,UER_CON_CATEGORY_OFFSET,1);        // Category Bit9 = PCM enc/dec !
		}

		LcModeChanged = TRUE;
	}

	// do some extended info or the mode change ?
	// then write the extended info :
	//
	if( PmUerByte != m_ucUerByte[PmBoardAudio/2] )
	{
		BYTE LcDataMode = ((PmUerByte & UER_AUDIO_DATA_MASK) == UER_IS_DATA_MODE);
		HALWriteOneCBit(PmBoardAudio,UER_DATA_MODE_OFFSET,LcDataMode);			// Set Data Mode (1) : Set Audio Mode (0) = default

		if(LcProUerMode)    // write PROFESSIONAL sound data width
		{   
			BYTE LcStereo = ((PmUerByte & UER_CHANNEL_MODE_MASK) == UER_IS_CHANNEL_STEREO);
			HALWriteOneCBit(PmBoardAudio,UER_PRO_STEREO_BIT_OFFSET,LcStereo);	// Set Stereo bit for Professional Mode !!!!
		}
		else                // write CONSUMER Mode Copyright Bit
		{
			BYTE LcRights = ((PmUerByte & UER_COPYRIGHT_MASK) == UER_IS_NO_COPYRIGHT);
			HALWriteOneCBit(PmBoardAudio,UER_CON_COPYRIGHT_OFFSET,LcRights);	// Set Copyright (0) : Set No Copyright (1) = default
		}

		m_ucUerByte[PmBoardAudio/2] = PmUerByte;        // save new UER Type
	}

 	if(      PmFrequency > 196000 )	LcCBitsFrequency = 0;       // off limits
	else if( PmFrequency > 188000 )	LcCBitsFrequency = 192000;
	else if( PmFrequency > 160000 )	LcCBitsFrequency = 176400;
	else if( PmFrequency > 100000 )	LcCBitsFrequency = 0 ;      // 128k not supported 
    else if( PmFrequency >  92000 ) LcCBitsFrequency =  96000;
	else if( PmFrequency >  84000 )	LcCBitsFrequency =  88200;
	else if( PmFrequency >  50000 )	LcCBitsFrequency =  0;      // 64k not supported
	else if( PmFrequency >  46000 )	LcCBitsFrequency =  48000;
    else if( PmFrequency >  38000 )	LcCBitsFrequency =  44100;
	else if( PmFrequency >  28000 )	LcCBitsFrequency =  32000;
	else if( PmFrequency >  23025 )	LcCBitsFrequency =  24000;
	else							LcCBitsFrequency =  22050;

	// call to HALHandleCodecSpeedModes() removed (call it in APHUpdateCBitsOnBoard)

	// if the mode or the frequency change, write the frequency bits :
	//
	if( LcModeChanged || (LcCBitsFrequency != m_ucUerCBitFreq[PmBoardAudio/2]))
	{
		BYTE	LcUerFreq;
		WORD	LcUerFreqBitOffset;
		BOOL	LcWriteByte4 = FALSE;

		switch( LcCBitsFrequency ) {
		case 48000:
			LcUerFreq = LcProUerMode ? UER_PRO_48K_CBITS_VAL : UER_CON_48K_CBITS_VAL; break;
		case 44100:
			LcUerFreq = LcProUerMode ? UER_PRO_44K_CBITS_VAL : UER_CON_44K_CBITS_VAL; break;
		case 32000:
			LcUerFreq = LcProUerMode ? UER_PRO_32K_CBITS_VAL : UER_CON_32K_CBITS_VAL; break;
		default:
			LcUerFreq = LcProUerMode ? UER_PRO_XXK_CBITS_VAL : UER_CON_48K_CBITS_VAL;
		}

		if ( LcProUerMode )	LcUerFreqBitOffset = UER_PRO_FREQ_BIT_OFFSET;
		else				LcUerFreqBitOffset = UER_CON_FREQ_BIT_OFFSET;

		HALWriteOneCBit(PmBoardAudio,LcUerFreqBitOffset,(BYTE)(LcUerFreq>>1));
		HALWriteOneCBit(PmBoardAudio,LcUerFreqBitOffset+1,(BYTE)(LcUerFreq&1));

		if( LcProUerMode && ( LcUerFreq == UER_PRO_XXK_CBITS_VAL ) )
		{
			switch( LcCBitsFrequency ) {
			case  24000: LcUerFreq = UER_EXT_FREQ__24000_VAL; break;
			case  96000: LcUerFreq = UER_EXT_FREQ__96000_VAL; break;
			case 192000: LcUerFreq = UER_EXT_FREQ_192000_VAL; break;
			case  22050: LcUerFreq = UER_EXT_FREQ__22050_VAL; break;
			case  88200: LcUerFreq = UER_EXT_FREQ__88200_VAL; break;
			case 176400: LcUerFreq = UER_EXT_FREQ_176400_VAL; break;
			default: LcUerFreq = 0;
			}
			LcWriteByte4 = TRUE;	// Write CBits in Byte 4 !!!
		}
		else if( (m_ucUerCBitFreq[PmBoardAudio/2] != 0 ) &&
				((m_ucUerCBitFreq[PmBoardAudio/2] < 32000) ||
				 (m_ucUerCBitFreq[PmBoardAudio/2] > 48000)) )
		{
			// if mode changed and we changed to consumer mode, or
			// if professional mode and we didn't change the mode
			//
			if( LcModeChanged == (LcProUerMode == 0) )
			{
				LcUerFreq = 0;
				LcWriteByte4 = TRUE;	// Reset CBits from Byte 4
			}
		}

		// write the frequency bits in Byte 4
		//
		if( LcWriteByte4 )
		{
			BYTE LcMask = UER_EXT_FREQ_BYTE4_MASK;
			WORD LcOffset = 8*4;
			ASSERT( (UER_EXT_FREQ_BIT_OFFSET/8) == 4 ); // assert that it's really byte 4

			while( LcMask ) {
				if(LcMask & 0x80)
				{
					if( ! HALWriteOneCBit(PmBoardAudio,LcOffset,((LcUerFreq&0x80) != 0)) )
						break;	// Byte 4 offset not supported by this card !
				}
				LcMask <<= 1;
				LcUerFreq <<= 1;
				LcOffset++;
			}
		}

		m_ucUerCBitFreq[PmBoardAudio/2] = LcCBitsFrequency;  // save UER Frequ
	}

    return SUCCESS;
}


//**************************************************************************
//
// This function selects the right input of the board in order to detect
// a sync signal from the bits C of UER IN
//
//**************************************************************************
WORD CPIOCommands::PIOReadUerSync( WORD PmSynchroNum, OUT PBYTE PmSync, OUT PDWORD PmFrequency, IN BOOL PmWaitLong /* = FALSE*/ )
{
    DWORD   LcFrequency         = 0 ;
    WORD    LcRet               = SUCCESS ;
    BYTE    LcSync ;

    DOUT(DBG_PIO_CMD, ("CPIOCommands::PIOReadUerSync\n"));

    // Select UER Synchro input if it exists
    if( PmSynchroNum >= m_piSn )
    {
        LOAD_PCX_ERROR( LcRet, ED_INVALID_BOARD_AUDIO );
        *PmSync = DRV_SYNC_UNKNOWN ;
    }
    else
    {
        // in case of np boards this function shall retrieve the frequency from the DSP, is available
        LcRet = HALGetAudioUer(PmSynchroNum, &LcSync, NULL, &LcFrequency, PmWaitLong) ;
        *PmSync = LcSync;

        // if HALGetAudioUer cannot detect the frequency, simply return the actual one if it's the captured clock
        //
        if ( !PIOIsFrequencyAutoComputed() )
        {
            if( (m_piActualClkSource == CLOCK_TYPE_UER_SYNCHRO) &&
                (m_piActualClkInputNum == PmSynchroNum)          )
            LcFrequency = m_piActualClkFrequency;
        }
    }

    *PmFrequency = LcFrequency;

    return( LcRet );
}

WORD CPIOCommands::PIOReadActualSyncForBoard( OUT PBYTE  PmSourcePtr, OUT PBYTE  PmSyncPtr, OUT PDWORD PmFrequencyPtr, IN BOOL PmWaitLong /* = FALSE*/ )
{
    WORD        LcRet       = SUCCESS ;

    // Safety inits
    // ------------
    *PmSyncPtr      = DRV_SYNC_UNKNOWN  ;
    *PmFrequencyPtr = 0                 ;
    *PmSourcePtr    = (BYTE)m_piActualClkSource ;

    switch ( m_piActualClkSource )
    {
    case CLOCK_TYPE_INTERNAL:
    case CLOCK_TYPE_PROGRAMMABLE_CLOCK:
    case CLOCK_TYPE_ETHERSOUND:
        // if the clock source is a programmable clock, tell the user
        // that sync signal is present.
        //
        *PmSyncPtr      = DRV_SYNC_PRESENT;
        *PmFrequencyPtr = m_piActualClkFrequency;
        break;

    case CLOCK_TYPE_UER_SYNCHRO:
        // Ask the board for presence of signal
        //
        PIOReadUerSync( (WORD)m_piActualClkInputNum,
                        PmSyncPtr,
                        PmFrequencyPtr,
                        PmWaitLong);
        break;

    case CLOCK_TYPE_WORD_CLOCK:
        // Ask the board for presence of signal
        //
        PIOReadWordClockSync(   (WORD) m_piActualClkInputNum,
                                PmSyncPtr,
                                PmFrequencyPtr);
        break;

    case CLOCK_TYPE_VIDEO:
        // Ask the board for presence of signal
        //
        PIOReadVideoSync(   (WORD) m_piActualClkInputNum,
			PmSyncPtr,
			PmFrequencyPtr);
        break;
		
    default:
        DOUT(DBG_ERROR, ("ERROR  CPIOCommands::PIOReadActualSyncForBoard : wrong ClkSource\n"));
    }

    DOUT(DBG_PIO_CMD, ("CPIOCommands::PIOReadActualSyncForBoard (Source:%d Sync:%d, Frequ:%ld)\n", *PmSourcePtr, *PmSyncPtr, *PmFrequencyPtr));

    return( LcRet );
}


BOOL CPIOCommands::CheckClockCompatibility(WORD PmClockType, WORD PmSource, DWORD PmFrequency, DWORD PmSyncInputNum)
{
    BOOL    LcIsCompatible = TRUE;

    if( m_piActualClkType != PmClockType )
    {
        LcIsCompatible = FALSE ;
    }
    else
    if( m_piActualClkSource == CLOCK_TYPE_NONE )
    {
        LcIsCompatible = FALSE ;
    }
    else
    if( m_piActualClkSource != PmSource )
    {
        LcIsCompatible = FALSE ;
    }
    else
    if( ( m_piActualClkSource == CLOCK_TYPE_INTERNAL ) && ( m_piActualClkFrequency != PmFrequency ) )
    {
        LcIsCompatible = FALSE ;
    }
    else
    if( ( m_piActualClkSource == CLOCK_TYPE_UER_SYNCHRO ) && ( m_piActualClkInputNum != PmSyncInputNum ) )
    {
        LcIsCompatible = FALSE ;
    }
    else
    if( ( m_piActualClkSource == CLOCK_TYPE_WORD_CLOCK ) && ( m_piActualClkInputNum != PmSyncInputNum ) )
    {
        LcIsCompatible = FALSE ;
    }
    else
    if( ( m_piActualClkSource == CLOCK_TYPE_VIDEO ) &&
		((m_piActualClkFrequency != PmFrequency) || (m_piActualClkInputNum != PmSyncInputNum)) )
    {
        LcIsCompatible = FALSE ;
    }
    
    DOUT(DBG_PIO_CMD, ("CPIOCommands::CheckClockCompatibility = %d\n", LcIsCompatible));

    return  LcIsCompatible;
}; // CheckClockCompatibility


BOOL   CPIOCommands::CheckAnalogLevelsSupport( IN WORD PmBoardAudioNum)
{
        // Beware: piDescAudioIn describes stereo audio
        //
        WORD LcDescAudio = m_piDescAudioIn[PmBoardAudioNum/2];

        // Exit if analog inputs are not present
        if ( !(LcDescAudio & AUDIO_ANALOG_MSK) ) return FALSE;

        // Exit if the analogs levels of the current audio source are not present
        if (   ((m_piTypeSrc[PmBoardAudioNum] == DATA_FROM_ANALOG) && !(LcDescAudio & AUDIO_LINE_IN_LEVEL_MSK))
            || ((m_piTypeSrc[PmBoardAudioNum] == DATA_FROM_MICRO ) && !(LcDescAudio & AUDIO_MICRO_IN_LEVEL_MSK))
            || ((m_piTypeSrc[PmBoardAudioNum] == DATA_FROM_CD)     && !(LcDescAudio & AUDIO_CD_IN_LEVEL_MSK ))
           ) 
        {
        return FALSE;
        }

        return TRUE;
} // CheckAnalogLevelsSupport



WORD CPIOCommands::PIOSetClockDependency( WORD PmClockType )
{
    WORD    LcRet = HALSetClockDependency( (BYTE) PmClockType );

    if( LcRet == SUCCESS )
    {
        m_piActualClkType = PmClockType;
    }
    return LcRet;
}

WORD CPIOCommands::PIODefClock(
        WORD            PmClockType,
        WORD            PmSource,
        DWORD           PmFrequency,
        DWORD           PmSyncInputNum,
		BYTE            PmClockFormat)
{
    WORD    LcRet               = SUCCESS       ;
    BOOL    LcResyncStallNeeded = FALSE         ;

    DOUT(DBG_PIO_CMD, ("CPIOCommands::PIODefClock\n"));

    // Check for a necessary reset of boards' clock
    // --------------------------------------------
    if ( CheckClockCompatibility(
            PmClockType     ,
            PmSource        ,
            PmFrequency     ,
            PmSyncInputNum  ) )
    {
        // ## FS (08/04/1999) -- The compatibility test may complete successfully
        // for a programmable clock, but we still have to update clock settings
        // to the new programmed frequency
        //
        if ( PmSource == CLOCK_TYPE_PROGRAMMABLE_CLOCK )
        {
            LcRet = PIOSetClock( PmFrequency,
                                 CLOCK_TYPE_PROGRAMMABLE_CLOCK,
                                 0,
								 PmClockFormat);
			return LcRet;
        }
        return SUCCESS ;
    }

    // ----------------------------------------------------------------
    // Perform all the code that initializes some variables used later
    // ----------------------------------------------------------------

    switch( PmSource )
    {
    case CLOCK_TYPE_UER_SYNCHRO:

        // Check that the synchro input number is correct
        // ----------------------------------------------
        if( PmSyncInputNum >= m_piSn )  
        {
            EXIT_PCX_ERROR( ED_CLOCK_UER_BOARD );
        }

        // On all boards, when we switch from an internal quartz
        // to UER sync, we must wait a while before resynchronizing
        // the audio input/output on DSP except full digital boards
        // -------------------------------------------------------
        if( (m_piActualClkSource != CLOCK_TYPE_UER_SYNCHRO) ||
			(m_piActualClkInputNum != PmSyncInputNum ) )
        {
            LcResyncStallNeeded = TRUE ;
        }

        if ( PIOIsFrequencyAutoComputed() )
        {
            // An IRQ is generated when the driver switch to the UER clock and resynchronizes
            // We must specify it's a first frequency change not to resynchronize again in the ISR
            //
            PIOSetFrequencyFirstChange(TRUE);
        }

        break;

    case CLOCK_TYPE_WORD_CLOCK:

        // Check that the synchro input number is correct
        // ----------------------------------------------
        if ( PmSyncInputNum >= m_piNbWcl )
        {
            EXIT_PCX_ERROR( ED_CLOCK_WORD_CLOCK_BOARD );
        }

        break;

    case CLOCK_TYPE_VIDEO:

        if ( PmSyncInputNum >= m_piNbVideoSync )
        {
            EXIT_PCX_ERROR( ED_SET_CLOCK_REFUSED );
        }
        break;
		
    case CLOCK_TYPE_ETHERSOUND:

        if ( PmSyncInputNum >= m_piEtherSoundSync )
        {
            EXIT_PCX_ERROR( ED_SET_CLOCK_REFUSED );
        }
        break;

    case CLOCK_TYPE_INTERNAL:
    case CLOCK_TYPE_PROGRAMMABLE_CLOCK:
    default:
        break;
    }

    // Set the clock on the master board
    //
    LcRet = PIOSetClock(
            PmFrequency,
            PmSource,
            PmSyncInputNum,
			PmClockFormat);
    
    if ( LcRet != SUCCESS ) return( LcRet );


    // Update dependency of board
    //
    if( m_piActualClkType != PmClockType )
    {
        LcRet = HALSetClockDependency( (BYTE)PmClockType ); // should be INDEPENDANT_CLOCK or MASTER_CLOCK
        if ( LcRet != SUCCESS ) return( LcRet );
    }

    // Update actual settings of board
    //
    m_piActualClkSource    = PmSource ;
    m_piActualClkInputNum  = PmSyncInputNum;
    m_piActualClkType      = PmClockType;

    if ( LcResyncStallNeeded != FALSE )
    {
        PCX_WAIT_AT_LEAST_MILLI_SEC(6);    // i.e. 192 samples at 32 KHz
    }

    return( LcRet );
}


WORD CPIOCommands::PIOGetBoardLevelInfo( OUT PBYTE *PmPPBoardLevelInfo, OUT PWORD PmPSize )
{
    DOUT(DBG_PIO_CMD, ("CPIOCommands::PIOGetBoardLevelInfo\n"));

    *PmPPBoardLevelInfo = (PBYTE) &m_LevelInfo;
    *PmPSize            = sizeof( BOARD_LEVEL_INFO );

    return SUCCESS;
}

WORD CPIOCommands::PIOWriteGPIO( DWORD PmPIOWriteMask, DWORD PmPIODataToWrite )
{
    WORD    LcRetStatus;

    DOUT(DBG_PIO_CMD, ("CPIOCommands::PIOWriteGPIO\n"));

    if( HALWriteGPIO( PmPIOWriteMask, PmPIODataToWrite ) )
        LcRetStatus = SUCCESS;
    else
        LOAD_PCX_ERROR( LcRetStatus, ED_UNAVAILABLE_FEATURE );

    return LcRetStatus;
}


WORD CPIOCommands::PIOReadGPIO( DWORD PmPIOReadMask, OUT PDWORD PmPIOData )
{
    WORD    LcRetStatus;

    DOUT(DBG_PIO_CMD, ("CPIOCommands::PIOReadGPIO\n"));

    if( HALReadGPIO( PmPIOReadMask, PmPIOData ) )
        LcRetStatus = SUCCESS;
    else
        LOAD_PCX_ERROR( LcRetStatus, ED_UNAVAILABLE_FEATURE );

    return LcRetStatus;
}


WORD CPIOCommands::PIOInitGPIO( DWORD PmConfigMaskGPIO, DWORD PmConfigMaskDIR )
{
    EXIT_PCX_ERROR( ED_UNAVAILABLE_FEATURE );
}


WORD CPIOCommands::PIOGetGPIOFeatures( OUT PDWORD PmReadAccess, OUT PDWORD PmWriteAccess, OUT PDWORD PmProgrammable)
{
    EXIT_PCX_ERROR( ED_UNAVAILABLE_FEATURE );
}


BOOL CPIOCommands::DefineClockRegisterValue(IN DWORD PmFreq,OUT PDWORD PmRealFreqPtr)
{
// Sample clock
// The formula is as follows:
//
//    HexFreq = (dword) ((double) ((double) 28224000 / (double) Frequency))
//    switch ( HexFreq & 0x00000F00 )
//    case 0x00000100: ;
//    case 0x00000200:
//    case 0x00000300: HexFreq -= 0x00000201 ;
//    case 0x00000400:
//    case 0x00000500:
//    case 0x00000600:
//    case 0x00000700: HexFreq = (dword) (((double) 28224000 / (double) (Frequency*2)) - 1)
//    default        : HexFreq = (dword) ((double) 28224000 / (double) (Frequency*4)) + 0x000001FF
//

	*PmRealFreqPtr = 0x0000004B;  // default 48k

	// protect division !
	//
	if(PmFreq == 0)	return FALSE;

    DWORD LcHexFreq = (28224000 * 10) / PmFreq ;
    LcHexFreq = (LcHexFreq + 5) / 10 ;

	if( LcHexFreq <= 0x00000200)	// max freq = 55125 Hz
		return FALSE;

	if( LcHexFreq <= 0x000003FF)
	{
		*PmRealFreqPtr = LcHexFreq - 0x00000201;
		return TRUE;
	}
	if( LcHexFreq <= 0x000007FF)
	{
		*PmRealFreqPtr = (LcHexFreq / 2) - 1;
		return TRUE;
	}
	if( LcHexFreq <= 0x00000FFF)
	{
		*PmRealFreqPtr = (LcHexFreq / 4) + 0x000001FF;
		return TRUE;
	}

	// min freq = 6893 Hz
	return FALSE;
}

