// **************************************************************************
//
//  COPYRIGHT 1996-2000 DIGIGRAM. ALL RIGHTS RESERVED.
//
//  DIGIGRAM
//
// ***************************************************************************

// Included files
//****************

// Header file
// ***********
#include "buff_hdl.h"

#include "pcxerr_e.h"

#include "drvdbg.h"
#include "log.h"

#include "ntbuffer.h"


#define BUF_MASK_SIZE   ((WORD) sizeof(DWORD)*8)

#ifndef min
#   define min(a,b)    ( ((a) < (b)) ? (a) : (b) )
#endif

#ifndef max
#   define max(a,b)    ( ((a) > (b)) ? (a) : (b) )
#endif

// Local variable
// ***************
STATIC BUFFER_INFO      TbBuffers[MAX_BUFFER];
STATIC GEN_BUFFER_INFO  GeneralBuffer;


//**************************************************************************
//  Internal functions
//**************************************************************************

// ****************************************************************************
// STATIC VOID AddBuffer2Handle( )
// ***************************
//
// Input parameters :
// ****************
//      WORD        PmBufferIndex   : the index of the buffer to add to this set
//      HANDLE_BUFFER PmHandleBuffers: the handle of the buffer set
//
// Output parameters :
// *******************
//      HANDLE_BUFFER PmHandleBuffers: the handle of the buffer set
//
// Return value :
// **************
//      VOID
//
// ****************************************************************************
//
// Add a buffer internally identified by its index in TbBuffers, to a buffer
// set identified externally by the handle.
//
// ****************************************************************************
STATIC VOID AddBuffer2Handle(
    IN  WORD            PmBufferIndex,
    INOUT PHANDLE_BUFFER PmPHandleBuffers )
{
    WORD    k;

    k = ( PmBufferIndex / BUF_MASK_SIZE );
    PmPHandleBuffers->hbMask[k] |=
                     UTIMaskDWord( (WORD)(PmBufferIndex % BUF_MASK_SIZE) );
}


// ****************************************************************************
// STATIC BOOLEAN DecodeHandleBuffer( )
// ***************************
//
// Input parameters :
// ****************
//      DWORD         PmBufferNumber : the number of the buffer to decode
//      HANDLE_BUFFER PmHandleBuffers: the handle of the buffer set
//
// Output parameters :
// *******************
//      PWORD       PmPBufferIndex  : the internal index of the buffer
//
// Return value :
// **************
//      TRUE if the buffer PmBufferNumber exists within the buffer set
//  identified by PmHandleBuffers
//
// ****************************************************************************
//
//  Scan the handle buffer mask to retrieve each buffer declared in the
// set of this handle, until the requested buffer number is encountered
//
// ****************************************************************************
STATIC BOOLEAN DecodeHandleBuffer(
    IN  DWORD            PmBufferNumber,
    IN  PHANDLE_BUFFER   PmPHandleBuffers,
    OUT PDWORD           PmPBufferIndex )
{
    DWORD   i, k, LcBufferNum;
    BOOLEAN LcFound = FALSE;

    // Decode the handle and buffer number into an
    // internal buffer index
    // -------------------------------------------
    for ( i = 0, LcBufferNum = 0; i < MAX_BUFFER; i++ )
    {
        k = ( i / BUF_MASK_SIZE );
        if (   PmPHandleBuffers->hbMask[k]
             & UTIMaskDWord( (WORD)( i % BUF_MASK_SIZE ) ) )
        {
            // this buffer belongs to the set specified by the
            // handle
            // -------------------------------------------------------
            if ( LcBufferNum == PmBufferNumber )
            {
                // Here is the buffer we are looking for
                // -------------------------------------
                *PmPBufferIndex = i;
                LcFound = TRUE;
                break;
            }

            LcBufferNum++;
        }  // if (this buffer belongs to the set of the handle)
    }

    // Verify the buffer has been found actually
    // -----------------------------------------
    return LcFound;

}


//**************************************************************************
//  Exported functions
//**************************************************************************

// ****************************************************************************
// VOID BUFInit(  )
// ***************************
//
// Input parameters :
// ****************
//      NONE
//
// Output parameters :
// *******************
//      GEN_BUFFER_HANDLE  PmGeneralBuffer: a pointer that will hold
//                          the address of GeneralBuffer
//      BUFFER_INFO_HANDLE PmGeneralBuffer: a pointer that will hold
//                          the address of TbBuffer
//
// Return value :
// **************
//      NONE
//
// ****************************************************************************
//
// This function is intended to be called only during the driver startup
//
// ****************************************************************************
VOID BUFInit(
    OUT GEN_BUFFER_INFO_HANDLE    PmHGeneralBuffer,
    OUT BUFFER_INFO_HANDLE        PmHBufferTab )
{
    *PmHGeneralBuffer = (LPGEN_BUFFER_INFO) &GeneralBuffer;
    *PmHBufferTab = (LPBUFFER_INFO) TbBuffers;

    BZERO2( &GeneralBuffer, GEN_BUFFER_INFO, 1 );
    BZERO2( TbBuffers, BUFFER_INFO, MAX_BUFFER );

    NTBInit();
}

// ****************************************************************************
// WORD BUFAllocateBuffers(  )
// ***************************
//
// Input parameters :
// ****************
//      WORD    PmNbBuffers             : the requested number of buffers
//
// Output parameters :
// *******************
//      PWORD          PmPNbBuffers     : the number of buffers allocated
//      PHANDLE_BUFFER PmHandleBuffers  : the handle mask of buffers
//                                        actually allocated
//
// Return value :
// **************
//      SUCCESS if the call completes successfully,
//      a warning if the number of buffers allocated is less than
//      the number requested (but at least > 0), an error code otherwise.
//
// ****************************************************************************
//
// Reserves full or partial amount of requested buffer. Gives back
// an handle that is to be used together with a buffer number in order
// to identify a specific buffer within the set of buffers allocated
//
// ****************************************************************************
WORD BUFAllocateBuffers(
    IN  DWORD           PmNbBuffers,
    IN  DWORD           PmBufferSize,
    OUT PDWORD          PmPNbBuffers,
    OUT PHANDLE_BUFFER  PmPHandleBuffers )
{
    BOOLEAN     LcWarn = FALSE;     // Flag set if we can allocate
                                    // only a partial amount of buffers
    WORD        i;

    // First of all, check that there is as many buffers available
    // as requested
    // -----------------------------------------------------------
    if ( PmNbBuffers > GeneralBuffer.gbBuffAvailNb )
    {
        LcWarn = TRUE;
        PmNbBuffers = GeneralBuffer.gbBuffAvailNb;
    }

    // Start allocation process:
    // Linear scan the table of buffers and allocate
    // free entries encountered until the full amount
    // of requested buffers is reached, or the end
    // of the table
    // ---------------------------------------------
    BZERO2( PmPHandleBuffers, HANDLE_BUFFER, 1 ) ;

    for ( i = 0 , *PmPNbBuffers = 0 ;
          (i < MAX_BUFFER) && (*PmPNbBuffers < PmNbBuffers) ;
          i++ )
    {
        if ( (TbBuffers[i].biCarac == 1) && (PmBufferSize <= NTBGetBufferSize(i)) )
        {
            // This was a free entry
            TbBuffers[i].biCarac = 0;
            GeneralBuffer.gbBuffAvailNb --;

            // Update both the number of buffers allocated
            // and the handle mask returned
            // -------------------------------------------
            (*PmPNbBuffers)++;

            AddBuffer2Handle( i, PmPHandleBuffers );

            DOUT(DBG_BUFFER, ("BUFAllocateBuffers(size %d) = index %d\n", PmBufferSize, i));

        } // if (free_entry)

    } // for (table_of_managed_buffers)

    // let's return :
    //  - an error if NO buffer has been allocated,
    //  - SUCCESS if the full amount requested has been granted
    //  - a warning otherwise
    // -----------------------------------------------------
    // ## FS (04/08/1998) -- Allow allocation of 0 buffers
    // ----------------------------------------------------
    if (   ( PmNbBuffers != 0 )
        && ( *PmPNbBuffers == 0 ) )
    {
        EXIT_PCX_ERROR( ED_ALLOCATE_BUFFER_IMPOSSIBLE );
    }

    if ( !( LcWarn || ( *PmPNbBuffers < PmNbBuffers ) ) )
        return SUCCESS;

    EXIT_PCX_ERROR( WD_UNDER_ALLOCATE_BUFFER );
}


// ****************************************************************************
// VOID BUFFreeBuffers(  )
// ***************************
//
// Input parameters :
// ****************
//      WORD            PmHandleBuffers: the handle of allocated buffers
//      LPADD_DESC_INFO PmAddrDescrTab:  the address descriptors in reply
//
// Output parameters :
// *******************
//      NONE
//
// Return value :
// **************
//      VOID
//
// ****************************************************************************
//
//  Release all the buffers allocated and granted to this handle
//
// ****************************************************************************
VOID BUFFreeBuffers(
    IN  PHANDLE_BUFFER  PmPHandleBuffers ,
    IN  LPADD_DESC_INFO PmAddrDescrTab  )
{
    DWORD   LcBufferIndex, LcBufferNum ;
    DWORD   LcUnmapIndex;

    // Loop and free the n-th buffer (n = LcBufferNum)
    // of the handle set till there is no more buffer
    // in this set
    // -----------------------------------------------
    LcBufferNum = 0;

    // Decode the handle and buffer number into an
    // internal buffer index
    // -------------------------------------------
    while ( DecodeHandleBuffer( LcBufferNum, PmPHandleBuffers, &LcBufferIndex ) )
    {
        if ( TbBuffers[LcBufferIndex].biCarac == 0 )
        {
            // This #ifdef directive can be replaced
            // by a if ( GeneralBuffer.gbTypeMem == MEM_NT_NONPAGED )
            // test, but NTBUnMapBuffer must then be defined even
            // over other systems
            // ------------------------------------------------------
            if (PmAddrDescrTab) {
                LcUnmapIndex = PmAddrDescrTab[0].adAddr1 ;

                // Under WindowsNT, we must unmap the buffers
                // from user space
                // ------------------------------------------
                NTBUnMapBuffer(LcBufferIndex,&(PmAddrDescrTab[LcUnmapIndex]) );
                PmAddrDescrTab[0].adAddr1 ++ ;
            }
			else
			{
				if(KeGetCurrentIrql() == PASSIVE_LEVEL)
				{
					ADD_DESC_INFO LcAddrInfo;
			        DOUT(DBG_BUFFER, ("BUFFreeBuffers : %d\n", LcBufferIndex));
					LcAddrInfo.adAddr1 = 1;
					LcAddrInfo.adAddr2 = POST_UNMAP_BUFFER;
					LcAddrInfo.adAddr3 = LcBufferIndex;
					NTBProceedMappings( 1, &LcAddrInfo );
				}
			}

            // Simply mark the buffer as available
            // -----------------------------------
            TbBuffers[LcBufferIndex].biCarac = 1;
            GeneralBuffer.gbBuffAvailNb++;
        }

        // Let's try n+1-th buffer
        LcBufferNum++;
    }
}


//**************************************************************************
//
// WORD BUFGetBufferInfoForTransNP( )
//
//Input Parameters:
//*****************
// DWORD  PmNumBuffer   : buffer number in the allocated buffer table
// HANDLE_BUFFER PmHandleBuffers : the handle of a set of buffers
// DWORD  PmOffset      : the offset of data requested within the buffer
// DWORD  PmLength      : the length of data requested within the buffer
//
//Output Paramaters:
//******************
// DWORD  PmLength      : the length of data actually retrieved
// DWORD  PmAddress     : the physical address of the retrieved buffer
//
//Return value:
//*************
//
//  SUCCESS if the retrieval completes successfully, an error otherwise
//
//**************************************************************************
//
// Decode the number of buffer and the handle, and returns the real address
// of the corresponding buffer
//
//**************************************************************************
WORD BUFGetBufferInfoForTransNP(
    IN DWORD            PmBufferNumber,
    IN PHANDLE_BUFFER   PmPHandleBuffers,
    IN DWORD            PmOffset,
    IN DWORD            PmLength,
    OUT PDWORD          PmPLength,
    OUT PDWORD          PmAddress )
{
    DWORD LcBufferIndex;

    // Decode the handle and buffer number into an
    // internal buffer index
    // -------------------------------------------
    if ( !DecodeHandleBuffer( PmBufferNumber, PmPHandleBuffers, &LcBufferIndex ) )
        EXIT_PCX_ERROR( ED_INVALID_BUFFER );

    // request must stay within buffer bounds
    // --------------------------------------
    if ( ( PmLength + PmOffset ) > NTBGetBufferSize( LcBufferIndex ) )
        EXIT_PCX_ERROR( ED_INVALID_BUFFER );

    *PmPLength = PmLength;

    *PmAddress = NTBGetBufferPhysicalAddress( LcBufferIndex ) + PmOffset;

    return SUCCESS;
}


//**************************************************************************
//
// VOID BUFFillAddressInfo( )
//
//Input Parameters:
//*****************
// WORD   PmNumBuffer   : buffer number in the allocated buffer table
// HANDLE_BUFFER PmHandleBuffers : the handle of a set of buffers
// WORD   PmWorld       : WORLD_16BITS or WORLD_32BITS
//
//Output Paramaters:
//******************
// LPADD_DESC_INFO      : the address descriptor to fill in
//
//Return value:
//*************
//      NONE
//
//**************************************************************************
//
//  Fill an address descriptor for the given buffer
//
//**************************************************************************
VOID BUFFillAddressInfo(
    IN  WORD            PmBufferNumber  ,
    IN  PHANDLE_BUFFER  PmPHandleBuffers ,
    IN  WORD            PmWorld,
    OUT LPADD_DESC_INFO PmPAddressInfo  )
{
    DWORD LcBufferIndex;

    if ( !DecodeHandleBuffer( PmBufferNumber, PmPHandleBuffers, &LcBufferIndex ) )
    {
        // Should NEVER go here !
        DOUT(DBG_ERROR, ("!!BUFFFillAddressInfo!!"));
        return;
    }

    NTBMapBuffer( LcBufferIndex, PmPAddressInfo );
}

//**************************************************************************
//
// VOID BUFGetFeatures( )
//
//Input Parameters:
//*****************
//      NONE
//
//Output Paramaters:
//******************
//  WORD        PmPTotalBuff: the total number of buffers managed by the driver
//  WORD        PmPAvailBuff: the number of buffers currently available
//  DWORD       PmPBuffSize : the size in bytes of a buffer
//  BYTE        PmPTypeMem  : the memory type used for the buffer
//
//Return value:
//*************
//      NONE
//
//**************************************************************************
//
//  Gives back the general features about buffer management
//
//**************************************************************************
VOID BUFGetFeatures(
    OUT LPDWORD PmPTotalBuff,
    OUT LPDWORD PmPAvailBuff,
    OUT LPDWORD PmPBuffSize )
{
    if ( PmPTotalBuff )
        *PmPTotalBuff = GeneralBuffer.gbBuffTotalNb;

    if ( PmPAvailBuff )
        *PmPAvailBuff = GeneralBuffer.gbBuffAvailNb;

    if ( PmPBuffSize )
        *PmPBuffSize  = GeneralBuffer.gbBuffSize;
}


//******************************************************************************
// DWORD BUFAllocateAllBuffers(INOUT LPGENERAL_INFO)
//*******************************************
//
// INPUT PARAMETERS     :
//***********************
//
//  LPGENERAL_INFO   PmGeneralInfoPtr  :  a pointeur to the GENERAL_INFO struct
//
// OUTPUT PARAMETERS    :
//***********************
//
//  LPGENERAL_INFO   PmGeneralInfoPtr  :  a pointeur to the GENERAL_INFO struct
//
// RETURN VALUE         :
//***********************
//
//   ERROR_SUCCESS if no error
//   <> ERROR_SUCCESS if error
//
//******************************************************************************
// Allocate all the required buffers
//******************************************************************************
//
DWORD BUFAllocateAllBuffers(INOUT LPVOID PmVoidPtr)
{
    LPGENERAL_INFO  LcGeneralInfoPtr= (LPGENERAL_INFO) PmVoidPtr    ;
    DWORD           LcBufferSize    = LcGeneralInfoPtr->SizeOfBuffers ;
    DWORD           LcNbOfBuffers   = LcGeneralInfoPtr->NbOfBuffers ;

    LcGeneralInfoPtr->PGeneralBuffer->gbBigBuffNb = 8;
    LcGeneralInfoPtr->PGeneralBuffer->gbBigBuffSize = 5000*64*4;

    // Get actual buffer size: user-specified of default value
    // --------------------------------------------------------
    if ( LcBufferSize > 0 )
    {
        LcGeneralInfoPtr->PGeneralBuffer->gbBuffSize = max( min( LcBufferSize, MAX_BUFFER_SIZE ), MIN_BUFFER_SIZE ) ;
    }
    else
    {
        LcGeneralInfoPtr->PGeneralBuffer->gbBuffSize  = DEFAULT_BUFFER_SIZE   ;
    }

    if ( LcNbOfBuffers >= MAX_BUFFER ) LcNbOfBuffers = MAX_BUFFER ;

    if( LcNbOfBuffers < LcGeneralInfoPtr->PGeneralBuffer->gbBigBuffNb)
        LcNbOfBuffers = LcGeneralInfoPtr->PGeneralBuffer->gbBigBuffNb;

    LcGeneralInfoPtr->PGeneralBuffer->gbBuffTotalNb  = LcNbOfBuffers ;

    // Proceed to actual allocation. This operation is system-dependent
    // Note that the filed gbBuffAvailableNb will be updated when allocation completes
    // -------------------------------------------------------------------------------
    return((DWORD) NTBAllocateAllBuffers(LcGeneralInfoPtr)) ;

}


//******************************************************************************
// VOID BUFFreeAllBuffers(INOUT PGENERAL_INFO)
//*******************************************
//
// INPUT PARAMETERS     :
//***********************
//
//  PGENERAL_INFO   PmGeneralInfoPtr  :  a pointeur to the GENERAL_INFO struct
//
// OUTPUT PARAMETERS    :
//***********************
//
// RETURN VALUE         :
//***********************
//
//******************************************************************************
// Free all buffers
//******************************************************************************
//
VOID BUFFreeAllBuffers(INOUT LPVOID PmVoidPtr)
{
    LPGENERAL_INFO  LcGeneralInfoPtr = (LPGENERAL_INFO) PmVoidPtr ;

    NTBFreeAllBuffers(LcGeneralInfoPtr) ;

    LcGeneralInfoPtr->PGeneralBuffer->gbBuffTotalNb = 0  ;
    LcGeneralInfoPtr->PGeneralBuffer->gbBuffAvailNb = 0  ;
}

//******************************************************************************
// WORD BUFGetBufferInfoForDebug(...)
//
// INPUT PARAMETERS     :
//***********************
//
//  WORD PmIndex : the index of the buffer in the array TbBuffers.
//
// OUTPUT PARAMETERS    :
//***********************
//
//  PBYTE *PmPPBufferInfo : the address the record describing the buffer.
//  PWORD PmPSize         : the size of the record.
//
// RETURN VALUE         :
//***********************
//
//  An error code.
//
//******************************************************************************
// Return the record describing a buffer.
//******************************************************************************
//
EXTERN WORD BUFGetBufferInfoForDebug(
    IN  DWORD   PmIndex,
    OUT PBYTE   *PmPPBufferInfo,
    OUT PWORD   PmPSize )
{
    WORD LcRet = SUCCESS;

    if ( PmIndex >= MAX_BUFFER )
    {
        LOAD_PCX_ERROR( LcRet, ED_INVALID_ADDRESS );
    }
    else
    {
        *PmPPBufferInfo = (PBYTE) ( TbBuffers + PmIndex );
        *PmPSize        = sizeof( BUFFER_INFO );
    }

    return( LcRet );
}


//******************************************************************************
// WORD BUFGetPhysMapInfoForDebug(...)
//
// INPUT PARAMETERS     :
//***********************
//
//  WORD PmIndex : the index of the buffer in the array TbPhysMapInfo.
//
// OUTPUT PARAMETERS    :
//***********************
//
//  PBYTE *PmPPPhysMapInfo : the address the record describing the buffer.
//  PWORD PmPSize          : the size of the record.
//
// RETURN VALUE         :
//***********************
//
//  An error code.
//
//******************************************************************************
// Return the record describing a buffer.
//******************************************************************************
//
EXTERN WORD BUFGetPhysMapInfoForDebug(
    IN  WORD    PmIndex,
    OUT PBYTE   *PmPPBufferInfo,
    OUT PWORD   PmPSize )
{
    return( NTBGetPhysMapInfoForDebug ( PmIndex, PmPPBufferInfo, PmPSize ) );
}


