/*******************************************************************/
/* Copyright 1995 Crystal Semiconductor Corp., ALL RIGHTS RESERVED */
/*   Program name: WAVE.EXE                                        */
/*   Purpose:      Wave Play/Record with DMA or PIO.               */
/*                 Reads/Writes WAVE format files direct to disk   */
/*                 Unlimited length.                               */
/*   Written by:   Richard Vail for Crystal Semiconductor          */
/*   History: 1.0  02/17/95: Created by Dick Vail                  */
/*            1.01 03/10/95: Added TIO - timer int driven pio.     */
/*            1.1  03/15/95: Added volume hot keys.                */
/*            1.2  05/03/95: Improve ADPCM support, FFT added.     */
/*            1.3  06/06/95: Add ThinkPad and OPTi929 support,     */
/*                           Improve 4232 handling,                */
/*                           Added support for Crystal ADPCM,      */
/*                           Mode 1 support added,                 */
/*                           Improved ASIC support.                */
/*            1.4  06/20/95: Fixed bug in Calibration bit defines, */
/*                           Fixed blocked ADPCM handling,         */
/*                           Add code to keep both crystals on.    */
/*            1.5  06/20/95: Fix full-duplex dma defaults.         */
/*                 06/28/95: Remove interrupt chaining(fix MCarpet)*/
/*                           Enable full-duplex FFT on ADPCM files,*/
/*                           by using 16bit linear capture format. */
/*                 06/30/95: Add KARAOKE play/record for 4232.     */
/*                                                                 */
/*******************************************************************/
#include <ctype.h>
#include <io.h>
#include <fcntl.h>
#include <conio.h>
#include <dos.h>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <malloc.h>
#include <math.h>
#include <graph.h>

char heading1[]="\nCrystalWare(tm) Wave Play/Record Ver. 1.5.1";
char heading2[]="\nCopyright(c) 1995 Crystal Semiconductor Corp.";
char heading3[]="\nAll rights reserved.\n";

typedef unsigned char BYTE;
typedef unsigned int  WORD;
typedef unsigned long DWORD;

#define CS4248          0xFFFF
#define CS4248A         0xFFFE
#define CS4231          0
#define CS4231A         1
#define CS4232          2
#define CS4232A         3

#define TRUE            1
#define FALSE           0

#define LEFT_COLOR      9               //blue
#define RIGHT_COLOR     12              //red

#define FFT_POWER_LOW   4
#define FFT_POWER_HIGH  13

#define PLAY            1
#define RECORD          2
#define CAPTURE         3
#define KARAOKE			4
#define TEST				5

#define MONO            1
#define STEREO          2

#define ONEUSEC         3L
#define TENUSEC         30L
#define ONEMSEC         3000L
#define TWOMSEC         6000L

#define FILETAG "File created by WAVE.EXE, written by Richard Vail for Crystal Semiconductor, 1995."

WORD IOADDR=0;	//must default to zero for CRYSTAL_CHECK
WORD IRQ=5;
WORD DMA_Play=1;
WORD DMA_Capture=0;

WORD IRQtoInt[] = {0x08, 0x09, 0x0A, 0x0B, 0x0C, 0x0D, 0x0E, 0x0F,
				  0x70, 0x71, 0x72, 0x73, 0x74, 0x75, 0x76, 0x77};
int IRQmasks[] = {1,2,4,8,16,32,64,128,1,2,4,8,16,32,64,128};
int dma_address[] = {0, 2, 4, 6};
int dma_page[] = {0x87, 0x83, 0x81, 0x82};
int dma_count[] = {1, 3, 5, 7};
int dma_status_req[] = {0x10, 0x20, 0x40, 0x80};

/* INTERRUPT CONTROLLER REGISTER DEFINITIONS  */
#define PIC1            0x20
#define PIC1MASKREG     0x21
#define PIC2            0xa0
#define PIC2MASKREG     0xa1
#define EOI             0x20     /* non specific end of interrupt */

// DMA CONTROLLER REGISTER DEFINITIONS
#define DMA1_COMMAND    0x08
#define DMA1_STATUS     0x08
#define DMA1_MASKREG    0x0A
#define DMA1_MODEREG    0x0B
#define DMA1_CLEAR_FF   0x0C
#define DMA1_MASTER_DISABLE	0x0D
#define DMA1_RW_MASK    0x0F

// DMA CONTROLLER MODE DEFINITIONS
#define DMA_SETMASK     0x04
#define DMA_DEMAND      0x00
#define DMA_SINGLE      0x40
#define DMA_AUTOINIT    0x10
#define DMA_PLAYBACK    0x08
#define DMA_CAPTURE     0x04

//CRYSTAL HARDWARE SUPPORTED WAVE DATA TYPES
#define  WAVE_FORMAT_PCM                   0x0001 /* industry standard */
#define  WAVE_FORMAT_ALAW                  0x0006 /* Microsoft Corporation */
#define  WAVE_FORMAT_MULAW                 0x0007 /* Microsoft Corporation */
#define  WAVE_FORMAT_CS_IMAADPCM           0x0039 /* Crystal Semiconductor IMA ADPCM */

//OTHER DATA TYPES
#define  WAVE_FORMAT_UNKNOWN               0x0000 /* Microsoft Corporation */
#define  WAVE_FORMAT_ADPCM                 0x0002 /* Microsoft Corporation */
#define  WAVE_FORMAT_IBM_CVSD              0x0005 /* IBM Corporation */
#define  WAVE_FORMAT_OKI_ADPCM             0x0010 /* OKI  */
#define  WAVE_FORMAT_DVI_ADPCM             0x0011 /* Intel Corporation */
#define  WAVE_FORMAT_IMA_ADPCM  (WAVE_FORMAT_DVI_ADPCM) /*  Intel Corporation */
#define  WAVE_FORMAT_MEDIASPACE_ADPCM      0x0012 /* Videologic */
#define  WAVE_FORMAT_SIERRA_ADPCM          0x0013 /* Sierra Semiconductor Corp */
#define  WAVE_FORMAT_G723_ADPCM            0x0014 /* Antex Electronics Corporation */
#define  WAVE_FORMAT_DIGISTD               0x0015 /* DSP Solutions, Inc. */
#define  WAVE_FORMAT_DIGIFIX               0x0016 /* DSP Solutions, Inc. */
#define  WAVE_FORMAT_DIALOGIC_OKI_ADPCM    0x0017 /* Dialogic Corporation */
#define  WAVE_FORMAT_YAMAHA_ADPCM          0x0020 /* Yamaha Corporation of America */
#define  WAVE_FORMAT_SONARC                0x0021 /* Speech Compression */
#define  WAVE_FORMAT_DSPGROUP_TRUESPEECH   0x0022  /*  DSP Group, Inc */
#define  WAVE_FORMAT_ECHOSC1               0x0023 /* Echo Speech Corporation */
#define  WAVE_FORMAT_AUDIOFILE_AF36        0x0024 /*    */
#define  WAVE_FORMAT_APTX                  0x0025 /* Audio Processing Technology */
#define  WAVE_FORMAT_AUDIOFILE_AF10        0x0026 /*    */
#define  WAVE_FORMAT_DOLBY_AC2             0x0030 /* Dolby Laboratories */
#define  WAVE_FORMAT_GSM610                0x0031 /* Microsoft Corporation */
#define  WAVE_FORMAT_ANTEX_ADPCME          0x0033 /* Antex Electronics Corporation  */
#define  WAVE_FORMAT_CONTROL_RES_VQLPC     0x0034 /* Control Resources Limited */
#define  WAVE_FORMAT_DIGIREAL              0x0035 /* DSP Solutions, Inc. */
#define  WAVE_FORMAT_DIGIADPCM             0x0036 /* DSP Solutions, Inc. */
#define  WAVE_FORMAT_CONTROL_RES_CR10      0x0037 /* Control Resources Limited */
#define  WAVE_FORMAT_NMS_VBXADPCM          0x0038 /* Natural MicroSystems */
#define  WAVE_FORMAT_G721_ADPCM            0x0040 /* Antex Electronics Corporation */
#define  WAVE_FORMAT_MPEG                  0x0050 /* Microsoft Corporation */
#define  WAVE_FORMAT_CREATIVE_ADPCM        0x0200 /* Creative Labs, Inc */
#define  WAVE_FORMAT_CREATIVE_FASTSPEECH8  0x0202 /* Creative Labs, Inc */
#define  WAVE_FORMAT_CREATIVE_FASTSPEECH10 0x0203 /* Creative Labs, Inc */
#define  WAVE_FORMAT_FM_TOWNS_SND          0x0300 /* Fujitsu Corp. */
#define  WAVE_FORMAT_OLIGSM                0x1000 /* Ing C. Olivetti & C., S.p.A. */
#define  WAVE_FORMAT_OLIADPCM              0x1001 /* Ing C. Olivetti & C., S.p.A. */
#define  WAVE_FORMAT_OLICELP               0x1002 /* Ing C. Olivetti & C., S.p.A. */
#define  WAVE_FORMAT_OLISBC                0x1003 /* Ing C. Olivetti & C., S.p.A. */
#define  WAVE_FORMAT_OLIOPR                0x1004 /* Ing C. Olivetti & C., S.p.A. */
#define  WAVE_FORMAT_DEVELOPMENT          (0xFFFF)

/* CS4231 direct register bit definitions */
#define INDEX           0
#define DATA            1
#define STATUS          2
#define IODATA          3
//index reg bits
#define MCE             0x40    //Mode Change Enable
#define TRD             0x20    //Transfer Request Disable
#define INIT            0x80    //Initialization state
#define INDEX_MASK      0xE0
//status reg bits
#define INT             0x01    //Interrupt pending
#define PRDY            0x02    //Ready for Playback sample
#define CRDY            0x20    //Capture sample Ready
#define SER             0x10    //Sample Error
/* CS4231 indirect register bit definitions */
#define MUTE            0x80
//Codec Reg 0
#define LMGE            0x20    //Left Mic Gain Enable
//Codec Reg 1
#define RMGE            0x20    //Right Mic Gain Enable
//Codec Reg 9
#define CPIO            0x80    //Capture
#define PPIO            0x40    //Playback
#define SDC             0x04    //Single DMA Channel
#define CEN             0x02    //Capture Enable
#define PEN             0x01    //Playback Enable
#define NO_CAL          0x00
#define CONVERTER_CAL   0x08
#define DAC_CAL         0x10	//which is full cal on 4231
#define FULL_CAL        0x18
#define CAL_MASK        0x18    //mask for calibration bits
//Codec Reg 10
#define IEN             0x02    //Interrupt enable
#define DTM             0x04    //DMA timing
#define DEN             0x08    //Dither Enable
//Codec Reg 11
#define ACI             0x20    //AutoCal Inprogress
//Codec Reg 12
#define MODE2           0x40    //Mode 2 enable
//Codec Reg 13
#define LBE             0x01    //LoopBack Enable
//Codec Reg 16
#define DACZ            0x01    //DAC return to Zero
#define SPE             0x02    //Serial Port Enable
#define PMCE            0x10    //Playback Mode Change Enable
#define CMCE            0x20    //Capture Mode Change Enable
#define TE              0x40    //Timer Enable
#define OLB             0x80    //Output Level Bit
//Codec Reg 17
#define HPF             0x01    //High Pass Filter
#define XTALE           0x02    //Crystal Enable
#define APAR            0x08    //ADPCM Playback Accumulator Reset
//Codec Reg 22, for 4232 only
#define SRE             0x80    //SampleRate Enable
//Codec Reg 23
#define ACF             0x01    //ADPCM Capture Freeze
//Codec Reg 24
#define PI              0x10    //Playback Interrupt
#define CI              0x20    //Capture Interrupt
#define TI              0x40    //Timer Interrupt

void playdma(void);
void capturedma(void);
void capturepio(void);
void playpio(void);
void capturetio(void);
void playtio(void);
void karaokedma(void);//play and record in full duplex mode w/mic input
void testmode(void);//play and record in full duplex mode
int CSiread(int);               //indirect regs
void CSiwrite(int, int);        //indirect regs
void CSiset(int, int);          //indirect regs
void CSireset(int, int);        //indirect regs
int CSread(int);                //direct regs
void CSwrite(int, int);         //direct regs
void CSset(int, int);           //direct regs
void CSreset(int, int);         //direct regs
int wait_for_init(void);        //wait for init period after changing crystals
int wait_for_ACI(void);         //wait for calibrate after removing MCE
int read_str(char *, int);
int read_size(DWORD *);
int find_playfreq(long);        //find actual sample rate closest to request
void ExitErrorMsg(char *);
void writeWAVheader(void);      //create WAVE header
void readWAVheader(void);       //read and decode WAVE header
void setupCODECformat(void);    //calculate I8 format register
void setupWAVheader(void);      //make sure valid format combination
void printWAVheader(void);
int _kbhit(void);
int findCRYSTALcodec(void);
int isCRYSTALcodec(int addr);
void maskIRQ(int);              //disable IRQ
void unmaskIRQ(int);            //enable IRQ
void (__interrupt __far Mode2ISR) (void);
void (__interrupt __far Mode2ISRTest) (void);
void (__interrupt __far Mode1ISR) (void);
void (__interrupt __far CtlBrkISR) (void);
void (__interrupt __far KeybdISR) (void);
void clearPIC(int IRQline);
int ASICread(int reg);            //read ASIC regs 0-3
void ASICwrite(int reg, int data);//write ASIC regs 0-7
int MAD16_asic(void);
int OPTi929_asic(void);
int ThinkPad(void);
int getSBparams(void);          //read BLASTER environment variable
void startDMA(WORD, DWORD, WORD, BYTE); //setup and enable DMA
void delayMS(long);             //wait for n milliseconds
void delayUS(WORD);             //wait for n microseconds
void saveCODECregs(void);
void restoreCODECregs(void);
void parsecommandline(int argc, char *argv[], int start);
void register_dump(void);       //debug dump of CODEC regs to screen
void display_syntax(void);      //dsiplay comand line syntax
WORD getpeaks(BYTE _far *buffer, WORD length);
void PEAKdisplay(WORD peaks, int nChannels);
void enableCODECaccess(int access);//enable MAD16 access to CODEC
void CleanExit(int code);
int set_FORMAT(char *s);
void enable_outputs(void);
void enable_inputs(void);
void displaydatatime(char *prefix);//display time for WAVE_data_length
void print_heading(void);
int get_key(void);
int FFT(int LeftRight, BYTE _far *data, int SampleSize, int nChannels);
void FFTdisplay(int LeftRight, int nChannels);
void FFTfree(void);
void FFTalloc(void);

//assembly language routines
void NIBBLE_SWAP(BYTE _far *buffer, WORD length);//reverse nibbles in ADPCM byte
void OUT_SAMPLE(BYTE _far *samples, WORD samplesize, WORD ioport);
void IN_SAMPLE(BYTE _far *samples, WORD samplesize, WORD ioport);
WORD GET_US_CLK(void);

// ASIC registers
#define ASICbase 0x0F8C
#define ASIC_R0 ASICbase + 3
#define ASIC_R1 ASICbase + 1
#define ASIC_R2 ASICbase + 2
#define ASIC_R3 ASICbase + 3
#define ASIC_R4 ASICbase + 4
#define ASIC_R5 ASICbase + 5
#define ASIC_R6 ASICbase + 6

#define MAD16password 0xE2
#define OPTi929password 0xE3
int ASICpassword=MAD16password;

int MAD16found=FALSE;
static int MCSCirq[] = {7,11,5,-1};
static int MCSCdrq[] = {1,2,3,-1};

int OPTi929found=FALSE;
static int OPTiSCirq[] = {7,10,5,-1};
static int OPTiSCdrq[] = {1,0,3,-1};

static int WSSbase[] = {0x530, 0xE80, 0xF40, 0x604};
int CS4232_control=0x538;

#define SB 0
#define WSS 1
int SBioaddr=0x220;
int SBirq=7;
int SBdmachan=1;

// Table containing possible sampling rates
// and corresponding crystal and divisor field values.
#define FTABCNT 14
static struct
{	long frequency;
	int crystal;
	int divisor;
}  freq_table[FTABCNT] =
{	{5512L,  1, 0},
 	{6615L,  1, 7},
 	{8000L,  0, 0},
 	{9600L,  0, 7},
 	{11025L, 1, 1},
 	{16000L, 0, 1},
	{18900L, 1, 2},
	{22050L, 1, 3},
	{27428L, 0, 2},
	{32000L, 0, 3},
	{33075L, 1, 6},
 	{37800L, 1, 4},
 	{44100L, 1, 5},
 	{48000L, 0, 6}
};

/*
 *  this pragma disables the warning issued by the Microsoft C compiler
 *  when using a zero size array as place holder when compiling for
 *  C++ or with -W4.
 *
 */
#pragma warning(disable:4200)

struct
{	WORD  wFormatTag;
	WORD  nChannels;
	DWORD nSamplesPerSec;
	DWORD nAvgBytesPerSec;
	WORD  nBlockAlign;
	WORD  wBitsPerSample;
	WORD  cbSize;                 // BYTE count/size of extra information (after cbSize)
	WORD  wSamplesPerBlock;
	WORD  wNumCoef;
	struct
	{	short   iCoef1;
		short   iCoef2;
	}  aCoef[1];
	WORD  wPadding[16];
}  wavefmt =
{	{WAVE_FORMAT_PCM},              //PCM
	{2},                            //stereo
	{44100L},                       //44.1 kHz
	{176400L},                      //4 * 44.1 kHz
	{4},                            //4 bytes per sample
	{16},                           //16 bits per sample per channel
	{0},
	{0},
	{0},
	{0},
	{0}
};

long WaveIN_RIFF_data_count=36L;	//normally 36 bytes plus WaveIN_data_length
long WaveIN_fmtdata_size=16L;		//header size, normally 16 bytes
long WaveIN_data_length=0L;			//actual raw wave data length

long WaveOUT_RIFF_data_count=36L;	//normally 36 bytes plus WaveIN_data_length
long WaveOUT_fmtdata_size=16L;		//header size, normally 16 bytes
long WaveOUT_data_length=0L;			//actual raw wave data length

long Record_limit=0x7FFFFFFFL;

BYTE codec_regs[32];

char path_buffer_in[_MAX_PATH]={"wav.wav"};
char path_buffer_out[_MAX_PATH]={"wav.wav"};
char drive[_MAX_DRIVE];
char dir[_MAX_DIR];
char fname[_MAX_FNAME];
char ext[_MAX_EXT];

#define LINE            0
#define AUX             1
#define MIC             2
#define LOOP            3
char *inputs[4]=
{	{"LINE"},
	{"AUX "},
	{"MIC "},
	{"LOOP"}
};
int input_select=AUX;
int input_leftvol=0;
int input_rightvol=0;

BYTE _far *PlayBuffer_ptr=NULL;
BYTE _far *PlayBuffer1=NULL;
BYTE _far *PlayBuffer2=NULL;
DWORD PlayBuffer_physical_address;
WORD PlayBuffer_using=0;
WORD PlayBuffer1_read=0;
WORD PlayBuffer2_read=0;
int PlayBuffer1_service=FALSE;
int PlayBuffer2_service=FALSE;

BYTE _far *CaptureBuffer_ptr=NULL;
BYTE _far *CaptureBuffer1=NULL;
BYTE _far *CaptureBuffer2=NULL;
DWORD CaptureBuffer_physical_address;
WORD CaptureBuffer_using=0;
WORD CaptureBuffer1_read=0;
WORD CaptureBuffer2_read=0;
int CaptureBuffer1_service=FALSE;
int CaptureBuffer2_service=FALSE;

BYTE _far *ADPCMBuffer_ptr=NULL;
WORD ADPCMBuffer_read=0;

WORD buffer_size=0;
WORD bytes_read=0;
WORD bytes_to_read=0;
WORD buffer_index=0;

int sample_size=0;
int samples_per_buffer=0;
int sample_time=0;
long time_sample_size=0L;

WORD CSindex=0;
WORD CSdata=0;
WORD CSstatus=0;
WORD CSiodata=0;
WORD CSversion=0;

BYTE format_reg=0;
BYTE capture_format_reg=0;
char format_name[9];

int CRYSTAL_check=1;	//look for codec?
int codec_ASIC_access=FALSE;
int regs_saved=0;
int output_leftvol=0;
int output_rightvol=0;
int program_mode=0;
int pio=FALSE;			//programmed I/O
int tio=FALSE;			//timer interrupt driven PIO
int verbose=FALSE;	//display information
int debug=FALSE;		//display debug info
int clipleft=0;		//ADC detected clip
int clipright=0;		//ADC detected clip
int underrun=FALSE;	//starved of data error
int overrun=FALSE;	//too much data error
int swap_nibbles=0;	//ADPCM fixup for old chips
int peakscrolling=0;
int peakmonitor=0;	//VU meter display
int equalizer=0;		//FFT display
int timemonitor=0;	//time display
int digital_loopback=0;
int single_mode=0;	//DMA mode
int CtlBrkHooked=0;
int CtlBrk=0;
int ISRHooked=0;
int full_duplex=0;	//2 dma channels
int blocked_adpcm=0;
int snapshot=0;		//debug info
short savecursor;
int FFTpower=9;                 // power(2,?) = FFTsize
int FFTsize=512;                // Number of points in FFT
int EQNumBars=16;               // 16 bar EQ display
int FFTSampleSize=2;
int LR=0;
int ThinkPad750found=0;
int Mode1=FALSE;
int ClockRunning=0;

int KeybdHooked=0;
int KeybdInt=0x09;
int Extended=0;

void (__interrupt __far *OldISR)();
void (__interrupt __far *OldInt23ISR)();
void (__interrupt __far *OldInt1BISR)();
void (__interrupt __far *OldKeybdISR)();

int toprow, leftcol, bottomrow, rightcol;
struct _videoconfig vc;

BYTE silence=0;
FILE *wavefilein=NULL;
FILE *wavefileout=NULL;
BYTE _far *filebuf=NULL;

/******************************************************************/
void main(int argc, char *argv[], char **envp)
{	struct _diskfree_t driveinfo;
	char *p;
	int i, start=1;
	WORD T;

	print_heading();
	T = GET_US_CLK();
	if (GET_US_CLK() > T)	//if Timer0 is running
		ClockRunning = 1;
	else
		printf("\n*****TIMER 0 IS NOT RUNNING AT NORMAL RATE*****");

	if (argc > 1)//now get any user specifications
	{	p=argv[1];
		if (*p == '/')
		{	p++;
			if (strnicmp(p, "CS", 2) == 0)
			{	if (strnicmp(&p[2], "4232A", 5) == 0)
					CSversion = CS4232A;
				else if (strnicmp(&p[2], "4232", 4) == 0)
					CSversion = CS4232;
				else if (strnicmp(&p[2], "4231A", 5) == 0)
					CSversion = CS4231A;
				else if (strnicmp(&p[2], "4231", 4) == 0)
					CSversion = CS4231;
				else if (strnicmp(&p[2], "4248A", 4) == 0)
					CSversion = CS4248A;
				else
				{	printf("\nCommand line error.");
					exit(-2);
				}
				CRYSTAL_check=0;
				start = 2;//ignore first argument during normal command line parse
			}
		}
		else if (*p == '?')
		{	display_syntax();
			exit(0);
		}
		else if ((*p == 'X') || (*p == 'x'))
		{	debug = 1;
			start = 2;//ignore first argument during normal command line parse
		}
	}
	if (CRYSTAL_check)
	{	if (ThinkPad())
			ThinkPad750found = TRUE;
		else if(MAD16_asic())
			MAD16found = TRUE;
		else if(OPTi929_asic())
			OPTi929found = TRUE;

		if (!findCRYSTALcodec())
			ExitErrorMsg("\nCrystal CODEC not found.\n");
	}

	if ((CSversion == CS4232) || (CSversion == CS4232A))
	{	if (getSBparams() == TRUE)
		{	IRQ = SBirq;
			DMA_Play = SBdmachan;
		}
	}

	if (argc > 1)//now get any user specifications
		parsecommandline(argc, argv, start);
	else
	{	display_syntax();
		CleanExit(0);
	}

	if (!findCRYSTALcodec())
		ExitErrorMsg("\nCrystal CODEC not found.\n");

	if (debug==-1)
	{	register_dump();
		if (peakmonitor || equalizer)
			_gettextwindow(&toprow, &leftcol, &bottomrow, &rightcol);
		CleanExit(-1);
	}
	if (program_mode == 0)
		ExitErrorMsg("\nNo operation specified.");
	if (DMA_Play >= 4)
		ExitErrorMsg("\nInvalid DMA channnel.");
	if (IRQ > 15)
		ExitErrorMsg("\nInvalid IRQ.");
	if ((pio==TRUE) || (tio==TRUE))
		full_duplex=FALSE;
	if (CSversion == CS4248A)
	{	//full_duplex=FALSE;
		single_mode=TRUE;
		tio=FALSE;
	}
	if (full_duplex && (DMA_Play == DMA_Capture))
		ExitErrorMsg("\nSeparate DMA channels must be specified for full duplex mode.");

	CtlBrkHooked=1;
	OldInt1BISR=_dos_getvect(0x1B);       //save current control break
	_dos_setvect(0x1B, CtlBrkISR);  //install our control break handler
	OldInt23ISR=_dos_getvect(0x23);       //save current control break
	_dos_setvect(0x23, CtlBrkISR);  //install our control break handler
	KeybdHooked=1;
	OldKeybdISR=_dos_getvect(KeybdInt);   //save current keyboard int
	_dos_setvect(KeybdInt, KeybdISR);       //install our keyboard handler

	if (program_mode == PLAY)
	{	_splitpath(path_buffer_in, drive, dir, fname, ext);
		if (strlen(ext) == 0)
			strcpy(ext, "wav");
		_makepath(path_buffer_in, drive, dir, fname, ext );

		if ((wavefilein=fopen(path_buffer_in, "rb")) == NULL)
		{	printf ("\nFile \"%s\" not found.\n", path_buffer_in);
			CleanExit(1);
		}
		if ((filebuf=(BYTE _far *) _halloc(16384L, 1)) == NULL)
			ExitErrorMsg("\nError: No memory available for filebuf.\n");
		setvbuf(wavefilein, filebuf, _IOFBF, sizeof(filebuf));
		readWAVheader();
	}
	else if (program_mode == RECORD)
	{	_splitpath(path_buffer_out, drive, dir, fname, ext);
		if (strlen(ext) == 0)
			strcpy(ext, "wav");
		_makepath(path_buffer_out, drive, dir, fname, ext );

		if (_access(path_buffer_out, 0) != -1)
		{	printf("\nFile %s exists: Replace? (Y/N) ", path_buffer_out);
			switch(get_key())
			{	case 'y':
				case 'Y':
					break;
				case 'n':
				case 'N':
				default:
					CleanExit(1);
			}
		}
		if (_dos_getdiskfree(toupper(drive[0]) - 'A' + 1, &driveinfo) == 0)
			Record_limit=(long)(driveinfo.avail_clusters-20) * (long)driveinfo.sectors_per_cluster * (long)driveinfo.bytes_per_sector;
		if ((wavefileout=fopen(path_buffer_out, "wb")) == NULL)
		{	printf ("\nFile \"%s\" cannot be created.\n", path_buffer_out);
			CleanExit(1);
		}
		if ((filebuf=(BYTE _far *) _halloc(16384L, 1)) == NULL)
			ExitErrorMsg("\nError: No memory available for filebuf.\n");
		setvbuf(wavefileout, filebuf, _IOFBF, sizeof(filebuf));
		setupWAVheader();
		writeWAVheader();
	}
	else if (program_mode == CAPTURE)
		setupWAVheader();
	else if (program_mode == KARAOKE)
	{	_splitpath(path_buffer_in, drive, dir, fname, ext);
		if (strlen(ext) == 0)
			strcpy(ext, "wav");
		_makepath(path_buffer_in, drive, dir, fname, ext );
		if ((wavefilein=fopen(path_buffer_in, "rb")) == NULL)
		{	printf ("\nFile \"%s\" not found.\n", path_buffer_in);
			CleanExit(1);
		}
		if ((filebuf=(BYTE _far *) _halloc(16384L, 1)) == NULL)
			ExitErrorMsg("\nError: No memory available for filebuf.\n");
		setvbuf(wavefilein, filebuf, _IOFBF, sizeof(filebuf));
		readWAVheader();

		_splitpath(path_buffer_out, drive, dir, fname, ext);
		if (strlen(ext) == 0)
			strcpy(ext, "wav");
		_makepath(path_buffer_out, drive, dir, fname, ext );
		if (stricmp(path_buffer_in, path_buffer_out) == 0)
			ExitErrorMsg("\nInput and output files cannot be the same.");
		if (_access(path_buffer_out, 0) != -1)
		{	printf("\nFile %s exists: Replace? (Y/N) ", path_buffer_out);
			switch(get_key())
			{	case 'y':
				case 'Y':
					break;
				case 'n':
				case 'N':
				default:
					CleanExit(1);
			}
		}
		if (_dos_getdiskfree(toupper(drive[0]) - 'A' + 1, &driveinfo) == 0)
			Record_limit=(long)(driveinfo.avail_clusters-20) * (long)driveinfo.sectors_per_cluster * (long)driveinfo.bytes_per_sector;
		if ((wavefileout=fopen(path_buffer_out, "wb")) == NULL)
		{	printf ("\nFile \"%s\" cannot be created.\n", path_buffer_out);
			CleanExit(1);
		}
		if ((filebuf=(BYTE _far *) _halloc(16384L, 1)) == NULL)
			ExitErrorMsg("\nError: No memory available for filebuf.\n");
		setvbuf(wavefileout, filebuf, _IOFBF, sizeof(filebuf));
		writeWAVheader();
	}
	else if (program_mode == TEST)
	{	_splitpath(path_buffer_in, drive, dir, fname, ext);
		if (strlen(ext) == 0)
			strcpy(ext, "wav");
		_makepath(path_buffer_in, drive, dir, fname, ext );
		if ((wavefilein=fopen(path_buffer_in, "rb")) == NULL)
		{	printf ("\nFile \"%s\" not found.\n", path_buffer_in);
			CleanExit(1);
		}
		if ((filebuf=(BYTE _far *) _halloc(16384L, 1)) == NULL)
			ExitErrorMsg("\nError: No memory available for filebuf.\n");
		setvbuf(wavefilein, filebuf, _IOFBF, sizeof(filebuf));
		readWAVheader();

		_splitpath(path_buffer_out, drive, dir, fname, ext);
		if (strlen(ext) == 0)
			strcpy(ext, "wav");
		_makepath(path_buffer_out, drive, dir, fname, ext );
		if (stricmp(path_buffer_in, path_buffer_out) == 0)
			ExitErrorMsg("\nInput and output files cannot be the same.");
		if (_access(path_buffer_out, 0) != -1)
		{	printf("\nFile %s exists: Replace? (Y/N) ", path_buffer_out);
			switch(get_key())
			{	case 'y':
				case 'Y':
					break;
				case 'n':
				case 'N':
				default:
					CleanExit(1);
			}
		}
		if (_dos_getdiskfree(toupper(drive[0]) - 'A' + 1, &driveinfo) == 0)
			Record_limit=(long)(driveinfo.avail_clusters-20) * (long)driveinfo.sectors_per_cluster * (long)driveinfo.bytes_per_sector;
		if ((wavefileout=fopen(path_buffer_out, "wb")) == NULL)
		{	printf ("\nFile \"%s\" cannot be created.\n", path_buffer_out);
			CleanExit(1);
		}
		if ((filebuf=(BYTE _far *) _halloc(16384L, 1)) == NULL)
			ExitErrorMsg("\nError: No memory available for filebuf.\n");
		setvbuf(wavefileout, filebuf, _IOFBF, sizeof(filebuf));
		writeWAVheader();
	}
	setupCODECformat();//setup format specific parameters

	if (buffer_size == 0)
	{	if (pio==TRUE)
			buffer_size=(WORD)(wavefmt.nAvgBytesPerSec / 1024L / 8L) * 1024;
		else if (tio==TRUE)
			buffer_size=(equalizer ? 4096 : 1024);
		else
			buffer_size=(WORD)(wavefmt.nAvgBytesPerSec / 1024L / 16L) * 1024;
	}
	if (equalizer)
	{	if (buffer_size < (WORD)FFTsize)
			buffer_size=FFTsize;
	}
	if (buffer_size < 1024)
		buffer_size=1024;
	if (buffer_size > (31*1024))
		buffer_size=(31*1024);

	if ((program_mode == PLAY) || full_duplex)
	{	if ((PlayBuffer_ptr=(BYTE _far *) _halloc((long)buffer_size * 2L, 2)) == NULL)
			ExitErrorMsg("\nError: No memory available for PlayBuffer.\n");
		PlayBuffer_physical_address=((long)FP_OFF(PlayBuffer_ptr)) + (((long)FP_SEG(PlayBuffer_ptr)) << 4);
		if ((PlayBuffer_physical_address & 0xFFFF0000L) != ((PlayBuffer_physical_address + (long)(buffer_size * 2)) & 0xFFFF0000L))
		{	PlayBuffer_physical_address += 0x0FFFFL;
			PlayBuffer_physical_address &= 0xF0000L;
		}
		PlayBuffer1=_MK_FP(PlayBuffer_physical_address >> 4, 0);
		PlayBuffer2=_MK_FP(PlayBuffer_physical_address + buffer_size >> 4, 0);
	 }
	if ((program_mode == RECORD) || (program_mode == CAPTURE) || full_duplex)
	{	if ((CaptureBuffer_ptr=(BYTE _far *) _halloc((long)buffer_size * 2L, 2)) == NULL)
			ExitErrorMsg("\nError: No memory available for CaptureBuffer.\n");
		CaptureBuffer_physical_address=((long)FP_OFF(CaptureBuffer_ptr)) + (((long)FP_SEG(CaptureBuffer_ptr)) << 4);
		if ((CaptureBuffer_physical_address & 0xFFFF0000L) != ((CaptureBuffer_physical_address + (long)(buffer_size * 2)) & 0xFFFF0000L))
		{	CaptureBuffer_physical_address += 0x0FFFFL;
			CaptureBuffer_physical_address &= 0xF0000L;
		}
		CaptureBuffer1=_MK_FP(CaptureBuffer_physical_address >> 4, 0);
		CaptureBuffer2=_MK_FP(CaptureBuffer_physical_address + buffer_size >> 4, 0);
	}
	if (blocked_adpcm)
	{	if ((ADPCMBuffer_ptr=(BYTE _far *) _halloc((long)wavefmt.nBlockAlign, 2)) == NULL)
			ExitErrorMsg("\nError: No memory available for Blocked ADPCMBuffer.\n");
	}
	if (wavefmt.wBitsPerSample == 4)
	{	sample_size=4;
		time_sample_size=1L;
		samples_per_buffer=(int)(buffer_size / 4);
		if (CSversion==CS4231)
			swap_nibbles=1;
	}
	else
	{	sample_size=(wavefmt.wBitsPerSample * wavefmt.nChannels) / 8;
		time_sample_size=(long)sample_size;
		samples_per_buffer=(int) (buffer_size / sample_size);
	}
	sample_time=(int)(1000000L / wavefmt.nSamplesPerSec);//in microseconds

	if (pio==TRUE)
		equalizer=0;
	if ((full_duplex==FALSE) && (wavefmt.wBitsPerSample==4))
		equalizer=0;
	if (equalizer && peakmonitor)
		peakmonitor=0;

	if (peakmonitor)
	{	_getvideoconfig(&vc);
		_gettextwindow(&toprow, &leftcol, &bottomrow, &rightcol);
		savecursor=_settextcursor(0x2000);
		_clearscreen(_GCLEARSCREEN);
		_settextwindow(toprow, leftcol, bottomrow, rightcol);
		if (peakmonitor==3)
			_settextposition(3, 1);
		else
		{	_settextposition(2, 1);
			if (peakscrolling)
				_settextposition(bottomrow, 1);
		}
		print_heading();
	}
	else if (equalizer)
	{	_getvideoconfig(&vc);
		savecursor=_settextcursor(0x2000);
		_clearscreen(_GCLEARSCREEN);
		print_heading();
		if (full_duplex)
			FFTSampleSize = 2;
		else
			FFTSampleSize = wavefmt.wBitsPerSample/8;

		FFTalloc();
	}

	printf("\nCrystal CODEC found at IO=0x%X, IRQ=%d, PDMA=%d", CSindex, IRQ, DMA_Play);
	if (full_duplex)
		printf(", CDMA=%d (Full Duplex)", DMA_Capture);
	if (verbose)
	{	if (debug)
		{	printf("\nCODEC type=");
			if (CSversion == CS4248)
				printf("CS4248");
			else if (CSversion == CS4231)
				printf("CS4231");
			else if (CSversion == CS4231A)
				printf("CS4231A");
			else if (CSversion == CS4232)
				printf("CS4232");
			else if (CSversion == CS4232A)
				printf("CS4232A");
			else
				printf("Unknown");
		}
		if (MAD16found)
			printf("\nMAD16 detected, ASIC_R1=0x%2.2X ASIC_R3=0x%2.2X", ASICread(ASIC_R1), ASICread(ASIC_R3));
		if (OPTi929found)
			printf("\nOPTi929 detected, ASIC_R1=0x%2.2X ASIC_R3=0x%2.2X", ASICread(ASIC_R1), ASICread(ASIC_R3));
		if (pio==TRUE)
			printf("\nTransfer method: Programmed I/O");
		else if (tio==TRUE)
			printf("\nTransfer method: Timer Interrupt driven Programmed I/O");
		else
			printf("\nTransfer method: DMA in %s mode", (single_mode==TRUE ? "single" : "demand"));
		printf("\nBuffer size=%d (0x%0X) bytes", buffer_size, buffer_size);
		if ((program_mode == RECORD) || (program_mode == CAPTURE))
		{	printf("\nInput selected=%s", inputs[input_select]);
			if ((input_leftvol!=0) || (input_rightvol!=0))
				printf("\nInput Volume=%d,%d", input_leftvol, input_rightvol);
		}
		if ((output_leftvol!=0) || (output_rightvol!=0))
			printf("\nOutput Volume=%d,%d", output_leftvol, output_rightvol);
		if (full_duplex)
			printf("\nPlay format=0x%2.2X Capture format=0x%2.2X", format_reg, capture_format_reg);
		if (debug)
		{	if ((program_mode == PLAY) || full_duplex)
			{	printf("\nPlayBuffer=0x%0X:0x%0X", FP_SEG(PlayBuffer_ptr), FP_OFF(PlayBuffer_ptr));
				printf("\nPlayBuffer1=0x%0X:0x%0X", FP_SEG(PlayBuffer1), FP_OFF(PlayBuffer1));
				printf("\nPlayBuffer2=0x%0X:0x%0X", FP_SEG(PlayBuffer2), FP_OFF(PlayBuffer2));
				printf("\nPlayBuffer_physical_address=0x%8.8lX", PlayBuffer_physical_address);
			}
			if ((program_mode == RECORD) || (program_mode == CAPTURE) || full_duplex)
			{	printf("\nCaptureBuffer=0x%0X:0x%0X", FP_SEG(CaptureBuffer_ptr), FP_OFF(CaptureBuffer_ptr));
				printf("\nCaptureBuffer1=0x%0X:0x%0X", FP_SEG(CaptureBuffer1), FP_OFF(CaptureBuffer1));
				printf("\nCaptureBuffer2=0x%0X:0x%0X", FP_SEG(CaptureBuffer2), FP_OFF(CaptureBuffer2));
				printf("\nCaptureBuffer_physical_address=0x%8.8lX", CaptureBuffer_physical_address);
			}
			if (peakmonitor || equalizer)
				printf("\nVideo mode=%d adapter=%d monitor=%d memory=%d rows=%d cols=%d",
					vc.mode, vc.adapter, vc.monitor, vc.memory, vc.numtextrows, vc.numtextcols);
		}
		printf("\nSample Size=%d %s, Time=%d usec, Avg Bytes/Second=%ld",
			sample_size, (sample_size == 1 ? "byte" : "bytes"),
			sample_time, wavefmt.nAvgBytesPerSec);
		if (program_mode == RECORD)
		{	printf("\nAvailable disk space=%ld Bytes", Record_limit);
		}
		if (peakmonitor && (wavefmt.wBitsPerSample == 4))
			printf("\nADPCM peak monitoring is an approximation.");
		if (blocked_adpcm)
			printf("\nWAV file in BLOCKED ADPCM format - playing as unblocked.");
		if (equalizer)
			printf("\nFFT size=%d, power=%d", FFTsize, FFTpower);
		if (ClockRunning==0)
			printf("\n*****TIMER 0 IS NOT RUNNING AT NORMAL RATE*****");
	}
	saveCODECregs();
	maskIRQ(IRQ);   //just in case

	if (pio == FALSE)
	{	OldISR=_dos_getvect(IRQtoInt[IRQ]);   //save current ISR
		if ((CSversion == CS4248) || (CSversion == CS4248A))
		{	_dos_setvect(IRQtoInt[IRQ], Mode1ISR);  //install our ISR
			Mode1=TRUE;
			if (verbose)
				printf("\nISR=Mode1");
		}
		else
		{	_dos_setvect(IRQtoInt[IRQ], Mode2ISR);	//install our ISR
			if (verbose)
				printf("\nISR=Mode2ISR");
		}
		ISRHooked=1;
	}

	if (program_mode == PLAY)
	{	printf("\nPlaying %s (%ldHz %dbps %s %s)",
			path_buffer_in,
			wavefmt.nSamplesPerSec,
			wavefmt.wBitsPerSample,
			wavefmt.nChannels == 1 ? "Mono" : "Stereo",
			format_name);
		displaydatatime("\nTotal Playing Time=");
		printf("\nPress any key to stop...\n");
		if (pio==TRUE)
			playpio();
		else if (tio==TRUE)
			playtio();
		else
			playdma();
		if (peakscrolling)
			_scrolltextwindow(2);
		if (peakmonitor)
			_settextposition(bottomrow-3, 1);
		if (_kbhit() || CtlBrk)
			printf("\nPlay stopped by keystroke.");
		if (_kbhit())
			get_key();      //throw away
		if (underrun)
			printf("\nPlayback underrun occurred.");
	}
	else if (program_mode == RECORD)
	{	printf("\nRecording %s (%ldHz %dbps %s %s)",
			path_buffer_out,
			wavefmt.nSamplesPerSec,
			wavefmt.wBitsPerSample,
			wavefmt.nChannels == 1 ? "Mono" : "Stereo",
			format_name);
		i=(int)(Record_limit / wavefmt.nAvgBytesPerSec);
		printf("\nMaximum recording time=");
		if (i<3600)
			printf("%2.2d:%2.2d", i/60, i%60);
		else
			printf("%2.2d:%2.2d:%2.2d", i/3600, (i%3600)/60, i%60);
		printf("\nPress any key to stop...\n");
		if (pio==TRUE)
			capturepio();
		else if (tio==TRUE)
			capturetio();
		else
			capturedma();
		fseek(wavefileout, 0L, SEEK_SET);
		writeWAVheader();
		if (peakscrolling)
			_scrolltextwindow(2);
		if (peakmonitor)
			_settextposition(bottomrow-3, 1);
		if (_kbhit())
			get_key();      //throw away
		displaydatatime("\nTotal Recorded Time=");
		if (overrun)
			printf("\nCapture overrun occurred.");
	}
	else if (program_mode == CAPTURE)
	{	printf("\nCapturing (%ldHz %dbps %s %s)",
			wavefmt.nSamplesPerSec,
			wavefmt.wBitsPerSample,
			wavefmt.nChannels == 1 ? "Mono" : "Stereo",
			format_name);
		printf("\nPress any key to stop...\n");
		if (pio==TRUE)
			capturepio();
		else if (tio==TRUE)
			capturetio();
		else
			capturedma();
		if (peakscrolling)
			_scrolltextwindow(2);
		if (peakmonitor)
			_settextposition(bottomrow-3, 1);
		if (_kbhit())
			get_key();      //throw away
		displaydatatime("\nTotal Captured Time=");
		if (overrun)
			printf("\nCapture overrun occurred.");
	}
	else if (program_mode == KARAOKE)
	{	printf("\nKARAOKE: Playing %s Recording %s (%ldHz %dbps %s %s)",
			path_buffer_in,
			path_buffer_out,
			wavefmt.nSamplesPerSec,
			wavefmt.wBitsPerSample,
			wavefmt.nChannels == 1 ? "Mono" : "Stereo",
			format_name);
		_outp(CS4232_control+1, (_inp(CS4232_control+1) & 0xCF) | 0x20);	//set karaoke mode
		i=(int)(Record_limit / wavefmt.nAvgBytesPerSec);
		printf("\nMaximum recording time=");
		if (i<3600)
			printf("%2.2d:%2.2d", i/60, i%60);
		else
			printf("%2.2d:%2.2d:%2.2d", i/3600, (i%3600)/60, i%60);
		printf("\nPress any key to stop...\n");
		karaokedma();
		fseek(wavefileout, 0L, SEEK_SET);
		writeWAVheader();
		if (peakscrolling)
			_scrolltextwindow(2);
		if (peakmonitor)
			_settextposition(bottomrow-3, 1);
		if (_kbhit())
			get_key();      //throw away
		displaydatatime("\nTotal Recorded Time=");
		if (overrun)
			printf("\nCapture overrun occurred.");
		_outp(CS4232_control+1, (_inp(CS4232_control+1) & 0xCF));	//reset karaoke mode
	}
	else if (program_mode == TEST)
	{	printf("\nTEST: Playing %s Recording %s (%ldHz %dbps %s %s)",
			path_buffer_in,
			path_buffer_out,
			wavefmt.nSamplesPerSec,
			wavefmt.wBitsPerSample,
			wavefmt.nChannels == 1 ? "Mono" : "Stereo",
			format_name);
		i=(int)(Record_limit / wavefmt.nAvgBytesPerSec);
		printf("\nMaximum recording time=");
		if (i<3600)
			printf("%2.2d:%2.2d", i/60, i%60);
		else
			printf("%2.2d:%2.2d:%2.2d", i/3600, (i%3600)/60, i%60);
		printf("\nPress any key to stop...\n");
		testmode();
		fseek(wavefileout, 0L, SEEK_SET);
		writeWAVheader();
		if (peakscrolling)
			_scrolltextwindow(2);
		if (peakmonitor)
			_settextposition(bottomrow-3, 1);
		if (_kbhit())
			get_key();      //throw away
		displaydatatime("\nTotal Recorded Time=");
		if (overrun)
			printf("\nCapture overrun occurred.");
		_outp(CS4232_control+1, (_inp(CS4232_control+1) & 0xCF));	//reset karaoke mode
	}
	CleanExit(0);
}

/******************************************************************/
void TimerPlay(void)	//called by timer interrupt routine
							//to play samples until FIFO full
{
	while (_inp(CSstatus) & PRDY)
	{	if ((PlayBuffer_using==1) && (PlayBuffer1_service==FALSE))
		{	OUT_SAMPLE(&PlayBuffer1[buffer_index], sample_size, CSiodata);
			if ((buffer_index += sample_size) >= PlayBuffer1_read)
			{	PlayBuffer1_service=TRUE;
				PlayBuffer_using=2;
				buffer_index=0;
			}
		}
		else if ((PlayBuffer_using==2) && (PlayBuffer2_service==FALSE))
		{	OUT_SAMPLE(&PlayBuffer2[buffer_index], sample_size, CSiodata);
			if ((buffer_index += sample_size) >= PlayBuffer2_read)
			{	PlayBuffer2_service=TRUE;
				PlayBuffer_using=1;
				buffer_index=0;
			}
		}
		else break;
	}
}

/******************************************************************/
void TimerCapture(void)	//called by timer interrupt routine
								//to capture samples until FIFO empty
{
	while (_inp(CSstatus) & CRDY)
	{	if ((CaptureBuffer_using==1) && (CaptureBuffer1_service==FALSE))
		{	IN_SAMPLE(&CaptureBuffer1[buffer_index], sample_size, CSiodata);
			if ((buffer_index += sample_size) >= CaptureBuffer1_read)
			{	CaptureBuffer1_service=TRUE;
				CaptureBuffer_using=2;
				buffer_index=0;
			}
		}
		else if ((CaptureBuffer_using==2) && (CaptureBuffer2_service==FALSE))
		{	IN_SAMPLE(&CaptureBuffer1[buffer_index], sample_size, CSiodata);
			if ((buffer_index += sample_size) >= CaptureBuffer2_read)
			{	CaptureBuffer2_service=TRUE;
				CaptureBuffer_using=1;
				buffer_index=0;
			}
		}
		else break;
	}
}

/******************************************************************/
void (__interrupt __far Mode2ISR) (void)
{	register int SaveIndex;
	void TimerPlay(void);
	void TimerCapture(void);

	SaveIndex=_inp(CSindex);
	CSireset(10, IEN);   //no more ints from part until done
	_enable();
	if (debug==-4)
		printf("\nCODEC Interrupt=0x%2.2X", _inp(CSstatus));
	while (_inp(CSstatus) & INT)
	{	if (CSiread(24) & TI)
		{	if (_inp(CSstatus) & PRDY)
				TimerPlay();
			if (_inp(CSstatus) & CRDY)
				TimerCapture();
			CSireset(24, TI);       // Clear the CS423x TIMER int
			clearPIC(IRQ);
		}
		if (CSiread(24) & PI)
		{	if (PlayBuffer_using == 1)
			{	PlayBuffer_using=2;
				PlayBuffer1_service=TRUE;
			}
			else
			{	PlayBuffer_using=1;
				PlayBuffer2_service=TRUE;
			}
			CSireset(24, PI);       // Clear the CS423x PLAYBACK int
			clearPIC(IRQ);
		}
		if (CSiread(24) & CI)
		{	if (CaptureBuffer_using == 1)
			{	CaptureBuffer_using=2;
				CaptureBuffer1_service=TRUE;
			}
			else
			{	CaptureBuffer_using=1;
				CaptureBuffer2_service=TRUE;
			}
			CSireset(24, CI);       // Clear the CS423x CAPTURE int
			clearPIC(IRQ);
		}
	}
	_disable();
	CSiset(10, IEN);  //now ready for more ints
	_outp(CSindex, SaveIndex);
//	_chain_intr(OldISR);            //just in case someone else is using this int
}

/******************************************************************/
void (__interrupt __far Mode2ISRTest) (void)	//timer interrupt routine
{	register int SaveIndex;
	register BYTE CurrentInt;
	static BYTE PendingInt=0;
	static int IntInProgress=0;
	void TimerPlay(void);
	void TimerCapture(void);

	SaveIndex=_inp(CSindex);
	CurrentInt=(CSiread(24) & (PI | CI | TI));
	_outp(CSindex, SaveIndex);
	if (CurrentInt)
	{	PendingInt|=CurrentInt;
		_outp(CSstatus, 0);
		clearPIC(IRQ);
		if (!IntInProgress)
		{	IntInProgress=1;
			do
			{	if (PendingInt & TI)
				{	PendingInt&=~TI;
					_enable();
					if (_inp(CSstatus) & PRDY)
						TimerPlay();
					if (_inp(CSstatus) & CRDY)
						TimerCapture();
					_disable();
				}
				if (PendingInt & PI)
				{	PendingInt&=~PI;
					if (PlayBuffer_using == 1)
					{	PlayBuffer_using=2;
						PlayBuffer1_service=TRUE;
					}
					else
					{	PlayBuffer_using=1;
						PlayBuffer2_service=TRUE;
					}
				}
				if (PendingInt & CI)
				{	PendingInt&=~CI;
					if (CaptureBuffer_using == 1)
					{	CaptureBuffer_using=2;
						CaptureBuffer1_service=TRUE;
					}
					else
					{	CaptureBuffer_using=1;
						CaptureBuffer2_service=TRUE;
					}
				}
			}  while (PendingInt);
			IntInProgress=0;
		}
	}
//	_chain_intr(OldISR);            //just in case someone else is using this int
}

/******************************************************************/
void (__interrupt __far Mode1ISR) (void) //mode 1 only (no timer)
{
	_enable();
	if (_inp(CSstatus) & INT)
	{	if (full_duplex)
		{	if (PlayBuffer_using == 1)
			{	PlayBuffer_using=2;
				PlayBuffer1_service=TRUE;
			}
			else
			{	PlayBuffer_using=1;
				PlayBuffer2_service=TRUE;
			}
			if (CaptureBuffer_using == 1)
			{	CaptureBuffer_using=2;
				CaptureBuffer1_service=TRUE;
			}
			else
			{	CaptureBuffer_using=1;
				CaptureBuffer2_service=TRUE;
			}
		}
		else
		{	if (program_mode == PLAY)
			{	if (PlayBuffer_using == 1)
				{	PlayBuffer_using=2;
					PlayBuffer1_service=TRUE;
				}
				else
				{	PlayBuffer_using=1;
					PlayBuffer2_service=TRUE;
				}
			}
			else
			{	if (CaptureBuffer_using == 1)
				{	CaptureBuffer_using=2;
					CaptureBuffer1_service=TRUE;
				}
				else
				{	CaptureBuffer_using=1;
					CaptureBuffer2_service=TRUE;
				}
			}
		}
	}
	_outp(CSstatus, 0);
	clearPIC(IRQ);
//	_chain_intr(OldISR);            //just in case someone else is using this int
}

/******************************************************************/
void (__interrupt __far CtlBrkISR) (void)
{
	CtlBrk=1;
}

/******************************************************************/
int get_key(void)
{	int c;

	c=_getch();
	if ((c==0) || (c==0xE0))
		c=_getch();
	return(c);
}

/******************************************************************/
void enable_inputs(void)	//select input and set mixer
{
	if ((input_leftvol!=0) || (input_rightvol!=0))
	{	CSiwrite(0, (input_select << 6) | ((input_leftvol * 15 / 100) & 0x1F));
		CSiwrite(1, (input_select << 6) | ((input_rightvol * 15 / 100) & 0x1F));
		if ((program_mode==RECORD) || (program_mode==CAPTURE))
		{	if (input_select == LOOP)
			{	CSiwrite(4, 31 - (input_leftvol * 31 / 100));//set AUX2 vol
				CSiwrite(5, 31 - (input_rightvol* 31 / 100));
			}
		}
		if (digital_loopback)//digital loopback enable
			CSiwrite(13, ((63 - (input_leftvol * 63 / 100)) << 2) | LBE);
		if (input_select == MIC)
		{	CSiset(0, LMGE);
			CSiset(1, RMGE);
		}
	}
	else    //just select input
	{	CSiwrite(0, (input_select << 6) | (CSiread(0) & 0x3F));
		CSiwrite(1, (input_select << 6) | (CSiread(1) & 0x3F));
		input_leftvol=(CSiread(0) & 0x0F) * 100 / 15;
		input_rightvol=(CSiread(1) & 0x0F) * 100 / 15;
		if ((program_mode==RECORD) || (program_mode==CAPTURE))
		{	if (input_select == LOOP)
			{	CSireset(4, MUTE);
				CSireset(5, MUTE);
			}
		}
		if (digital_loopback)//digital loopback enable
			CSiset(13, LBE);
	}
}

/******************************************************************/
void enable_outputs(void)	//set mixer
{
	if ((output_leftvol!=0) || (output_rightvol!=0))
	{	CSiwrite(6, 63 - (output_leftvol * 63 / 100));
		CSiwrite(7, 63 - (output_rightvol* 63 / 100));
		if ((CSversion == CS4232) || (CSversion == CS4232A))//set master volume
		{	CSiset(12, MODE2);                  //set MODE 2
			CSiwrite(27, 15 - (output_leftvol * 15 / 100));//left master
			CSiwrite(29, 15 - (output_rightvol * 15 / 100));//right master
			if (Mode1==TRUE)
				CSireset(12, MODE2);                  //set MODE 1
		}
		if ((program_mode==RECORD) || (program_mode==CAPTURE))
		{	if (input_select == AUX)
			{	CSiwrite(2, 31 - (output_leftvol * 31 / 100));//set AUX1 vol
				CSiwrite(3, 31 - (output_rightvol* 31 / 100));
			}
			else if (input_select == LINE)
			{	CSiset(12, MODE2);                  //set MODE 2
				CSiwrite(18, 31 - (output_leftvol * 31 / 100));//set LINE vol
				CSiwrite(19, 31 - (output_rightvol* 31 / 100));
				if (Mode1==TRUE)
					CSireset(12, MODE2);                  //set MODE 1
			}
		}
	}
	else
	{	CSireset(6, MUTE);      //unmute DAC's
		CSireset(7, MUTE);
		output_leftvol=(63 - (CSiread(6) & 0x3F)) * 100 / 63;
		output_rightvol=(63 - (CSiread(7) & 0x3F)) * 100 / 63;
		if ((CSversion == CS4232) || (CSversion == CS4232A))
		{	CSiset(12, MODE2);                  //set MODE 2
			CSireset(27, MUTE);     //unmute left master
			CSireset(29, MUTE);     //unmute right master
			if (Mode1==TRUE)
				CSireset(12, MODE2);                  //set MODE 1
		}
		if ((program_mode==RECORD) || (program_mode==CAPTURE))
		{	if (input_select == AUX)
			{	CSireset(2, MUTE);
				CSireset(3, MUTE);
			}
			else if (input_select == LINE)
			{	CSiset(12, MODE2);                  //set MODE 2
				CSireset(18, MUTE);
				CSireset(19, MUTE);
				if (Mode1==TRUE)
					CSireset(12, MODE2);                  //set MODE 1
			}
		}
	}
}

/******************************************************************/
void displaydatatime(char *prefix)	//display play/record time
{	int i;

	if ((program_mode == RECORD) || (program_mode == KARAOKE) || (program_mode == TEST))
		i=(int)((WaveOUT_data_length / time_sample_size) / (long)wavefmt.nSamplesPerSec);
	else
		i=(int)((WaveIN_data_length / time_sample_size) / (long)wavefmt.nSamplesPerSec);
	if (i<3600)
		printf("%s[%2.2d:%2.2d]", prefix, i/60, i%60);
	else
		printf("%s[%2.2d:%2.2d:%2.2d]", prefix, i/3600, (i%3600)/60, i%60);
}

/******************************************************************/
int ADPCMFillBuffer(BYTE _far *Buffer, WORD BufferSize)
{	register int i, j;

	for (j=BufferSize; j > 0; )
	{	if (ADPCMBuffer_read)
		{	if (ADPCMBuffer_read < BufferSize)
			{	memcpy(Buffer, ADPCMBuffer_ptr+wavefmt.nBlockAlign-ADPCMBuffer_read, ADPCMBuffer_read);
				j -= ADPCMBuffer_read;
				ADPCMBuffer_read = 0;
			}
			else
			{	memcpy(Buffer, ADPCMBuffer_ptr+wavefmt.nBlockAlign-ADPCMBuffer_read, BufferSize);
				ADPCMBuffer_read -= BufferSize;
				j = 0;
			}
		}
		while ((j > 0) && (WaveIN_data_length != 0))
		{	ADPCMBuffer_read=fread(ADPCMBuffer_ptr, 1, wavefmt.nBlockAlign, wavefilein);
			WaveIN_data_length -= ADPCMBuffer_read;
			if (wavefmt.nChannels==1)
				i=4;
			else
				i=8;
			if (j >= (int)(ADPCMBuffer_read - i))
			{	memcpy(Buffer, ADPCMBuffer_ptr+i, ADPCMBuffer_read-i);
				j -= (ADPCMBuffer_read-i);
				ADPCMBuffer_read=0;
			}
			else
			{	memcpy(Buffer, ADPCMBuffer_ptr+i, j);
				ADPCMBuffer_read -= j;
				j=0;
			}
		}
	}
	return(BufferSize-j);
}

/******************************************************************/
void playdma(void)//play in half or full duplex mode (capture for peak detect)
{	register int i, j, bytesleft;

	bytes_to_read=(int)min((long)buffer_size, WaveIN_data_length);
	if (blocked_adpcm)
	{	bytes_read=ADPCMFillBuffer(PlayBuffer1, bytes_to_read);
		if (swap_nibbles)
			NIBBLE_SWAP(PlayBuffer1, bytes_read);
		PlayBuffer_using=1;
		PlayBuffer1_service=FALSE;
	}
	else if (bytes_to_read != 0)
	{	bytes_read=fread(PlayBuffer1, 1, bytes_to_read, wavefilein);
		WaveIN_data_length -= bytes_read;
		if (swap_nibbles)
			NIBBLE_SWAP(PlayBuffer1, bytes_read);
		PlayBuffer_using=1;
		PlayBuffer1_service=FALSE;
	}
	else PlayBuffer1_service=TRUE;
	PlayBuffer2_service=TRUE;
	if (full_duplex)
	{	CaptureBuffer_using=1;
		CaptureBuffer1_service=FALSE;
		CaptureBuffer2_service=FALSE;
	}

	CSreset(INDEX, TRD);
	CSiset(12, MODE2);      //set MODE 2
	CSireset(9, PEN | CEN);
	_outp(DMA1_MASKREG, DMA_SETMASK | DMA_Play);        //mask channel
	if (full_duplex)
		_outp(DMA1_MASKREG, DMA_SETMASK | DMA_Capture);        //mask channel
	if ((CSversion!=CS4248) && (CSversion!=CS4231))
		CSireset(16, TE | SPE | PMCE | CMCE);
	if ((CSversion == CS4232) || (CSversion == CS4232A))
	{	CSiset(10, DTM);		//must have correct DMA timing
		CSiset(17, XTALE);	//keep both crystals on
		CSireset(22, SRE);	//must reset to allow I8 changes
	}
	if (Mode1==TRUE)
		CSireset(12, MODE2); //set MODE 1

	CSset(INDEX, MCE);      //set ModeChangeEnable
	CSiwrite(15, samples_per_buffer - 1);        //set play lower base count
	CSiwrite(14, (samples_per_buffer - 1) >> 8); //set play upper base count
	CSireset(9, PPIO | CPIO | CAL_MASK);  //clear CAL bits
	CSiset(9, SDC | DAC_CAL);	//set CAL method and single DMA channel mode
	if (full_duplex)
	{	CSireset(9, SDC);
		if (Mode1==FALSE)
		{	CSiwrite(31, samples_per_buffer - 1);        //set capture lower base count
			CSiwrite(30, (samples_per_buffer - 1) >> 8); //set capture upper base count
			CSiwrite(28, capture_format_reg);	//set capture format to linear 16 bit stereo
		}
	}
	CSiwrite(8, format_reg);  //set format and rate
	wait_for_init();
	CSreset(INDEX, MCE);    //reset ModeChangeEnable
	wait_for_ACI();

	CSiset(10, IEN);
	clearPIC(IRQ);       //clear any leftover interrupt
	CSwrite(STATUS, 0);  //clear STATUS
	unmaskIRQ(IRQ);

	if (full_duplex)
	{	CSiset(9, CEN | PEN);	//same time
		if (peakmonitor)
			PEAKdisplay(0, 0);
	}
	else
		CSiset(9, PEN);
	if (CSversion == CS4232)   //special fix for rev C only
	{	CSset(INDEX, MCE);      //set ModeChangeEnable
		CSreset(INDEX, MCE);    //reset ModeChangeEnable
		wait_for_ACI();
	}

	startDMA(DMA_Play, PlayBuffer_physical_address, (buffer_size * 2) - 1, DMA_PLAYBACK | DMA_AUTOINIT);
	if (full_duplex)
		startDMA(DMA_Capture, CaptureBuffer_physical_address, (buffer_size * 2) - 1, DMA_CAPTURE | DMA_AUTOINIT);

	if (full_duplex)
		enable_inputs();
	enable_outputs();
	if (debug)
		register_dump();
	while ((WaveIN_data_length > 0L) && !feof(wavefilein))
	{	if (PlayBuffer1_service == TRUE)
		{	if ((bytes_to_read=(int)min((long)buffer_size, WaveIN_data_length)) != 0)
			{	bytes_read=fread(PlayBuffer1, 1, bytes_to_read, wavefilein);
				WaveIN_data_length -= bytes_read;
				if (swap_nibbles)
					NIBBLE_SWAP(PlayBuffer1, bytes_read);
				if (bytes_read < buffer_size)
					memset(PlayBuffer1+bytes_read, silence, buffer_size - bytes_read);
				else
				{	if (full_duplex == FALSE)
					{	if (peakmonitor)
							PEAKdisplay(getpeaks(PlayBuffer1, buffer_size), wavefmt.nChannels);
						if (equalizer)
							FFTdisplay(FFT(LR++, PlayBuffer1, FFTSampleSize, wavefmt.nChannels), wavefmt.nChannels);
						if (timemonitor)
							displaydatatime("\r");
					}
				}
				PlayBuffer1_service=FALSE;
			}
		}
		if (PlayBuffer2_service == TRUE)
		{	if ((bytes_to_read=(int)min((long)buffer_size, WaveIN_data_length)) != 0)
			{	bytes_read=fread(PlayBuffer2, 1, bytes_to_read, wavefilein);
				WaveIN_data_length -= bytes_read;
				if (swap_nibbles)
					NIBBLE_SWAP(PlayBuffer2, bytes_read);
				if (bytes_read < buffer_size)
					memset(PlayBuffer2+bytes_read, silence, buffer_size - bytes_read);
				else
				{	if (full_duplex == FALSE)
					{	if (peakmonitor)
							PEAKdisplay(getpeaks(PlayBuffer2, buffer_size), wavefmt.nChannels);
						if (equalizer)
							FFTdisplay(FFT(LR++, PlayBuffer2, FFTSampleSize, wavefmt.nChannels), wavefmt.nChannels);
						if (timemonitor)
							displaydatatime("\r");
					}
				}
				PlayBuffer2_service=FALSE;
			}
		}
		if (full_duplex && (CaptureBuffer1_service==TRUE))
		{	if (peakmonitor)
				PEAKdisplay(getpeaks(CaptureBuffer1, buffer_size), ((capture_format_reg & 0x10) ? 2 : 1));
			if (equalizer)
				FFTdisplay(FFT(LR++, CaptureBuffer1, FFTSampleSize, ((capture_format_reg & 0x10) ? 2 : 1)), ((capture_format_reg & 0x10) ? 2 : 1));
			if (timemonitor)
				displaydatatime("\r");
			CaptureBuffer1_service=FALSE;
		}
		if (full_duplex && (CaptureBuffer2_service==TRUE))
		{	if (peakmonitor)
				PEAKdisplay(getpeaks(CaptureBuffer2, buffer_size), ((capture_format_reg & 0x10) ? 2 : 1));
			if (equalizer)
				FFTdisplay(FFT(LR++, CaptureBuffer2, FFTSampleSize, ((capture_format_reg & 0x10) ? 2 : 1)), ((capture_format_reg & 0x10) ? 2 : 1));
			if (timemonitor)
				displaydatatime("\r");
			CaptureBuffer2_service=FALSE;
		}
		if (_kbhit() || CtlBrk)
		{	bytes_read=0;
			break;
		}
		if (_inp(CSstatus) & SER)
			underrun=1;
		if (full_duplex && (CSiread(11) & 0x02))
			clipleft=1;
		if (full_duplex && (CSiread(11) & 0x08))
			clipright=1;
		if (debug==-2)
			printf("\nI11=0x%2.2X, DMA Status=0x%2.2X ", CSiread(11), _inp(DMA1_STATUS));
	}

	if (bytes_read != 0)
	{	for (i=0; i<samples_per_buffer; i++)    //wait up to one buffer time
		{	for (j=0; j<sample_time; j++)           //for the current to be empty
			{	if ((PlayBuffer1_service==TRUE) || (PlayBuffer2_service==TRUE))
					break;
			}
			if ((PlayBuffer1_service==TRUE) || (PlayBuffer2_service==TRUE))
				break;
		}
		if (PlayBuffer1_service == TRUE)
			memset(PlayBuffer1, silence, buffer_size);
		if (PlayBuffer2_service == TRUE)
			memset(PlayBuffer2, silence, buffer_size);
		//get current DMA position
		bytesleft=(_inp(dma_count[DMA_Play]) + (_inp(dma_count[DMA_Play]) << 8)) % buffer_size;
		//wait for playing of the last buffer
		for (i=(int)(bytesleft/sample_size); i>0; i--)
		{	for (j=0; j<sample_time; j++)
				;
			_inp(CSindex);
		}
	}

	CSiset(6, MUTE);        //mute DAC's
	CSiset(7, MUTE);
	if (CSversion==CS4232)  //special fix for rev C only
	{	for(i=0;i<65535;i++)
			if (_inp(DMA1_STATUS) & dma_status_req[DMA_Play])
				break;
	}
	CSireset(9, PEN);
	_outp(DMA1_MASKREG, DMA_SETMASK | DMA_Play);        //mask channel
	if (full_duplex)
	{	if (CSversion==CS4232)       //special fix for rev C only
		{	for(i=0;i<65535;i++)
				if (_inp(DMA1_STATUS) & dma_status_req[DMA_Capture])
					break;
		}
		CSireset(9, CEN);
		_outp(DMA1_MASKREG, DMA_SETMASK | DMA_Capture);        //mask channel
	}
	if (peakmonitor)
		PEAKdisplay(0, 0);
	if (equalizer)
		FFTdisplay(0, 0);
}

/******************************************************************/
void capturedma(void)   //half duplex capture (using play buffers)
{	WORD bytesleft, i;

	CaptureBuffer_using=1;
	CaptureBuffer1_service=FALSE;
	CaptureBuffer2_service=FALSE;

	CSreset(INDEX, TRD);
	CSiset(12, MODE2);      	//set MODE 2
	CSireset(9, PEN | CEN);
	_outp(DMA1_MASKREG, DMA_SETMASK | DMA_Play);        //mask channel
	if ((CSversion!=CS4248) && (CSversion!=CS4231))
		CSireset(16, TE | SPE | PMCE | CMCE);
	if ((CSversion == CS4232) || (CSversion == CS4232A))
	{	CSiset(10, DTM);        //must have correct DMA timing
		CSiset(17, XTALE);		//keep both crystals on
		CSireset(22, SRE);      //must reset to allow I8 changes
	}
	if (Mode1==TRUE)
		CSireset(12, MODE2);		//set MODE 1
//	if ((wavefmt.wBitsPerSample == 8) && (wavefmt.wFormatTag == WAVE_FORMAT_PCM))
	if ((capture_format_reg & 0xE0) == 0)
		CSireset(10, DEN);      //forces inverted dither enable for 8 bit linear

	CSset(INDEX, MCE);      //set ModeChangeEnable
	CSireset(9, CPIO | PPIO | CAL_MASK);  //clear CAL bits
	CSiset(9, SDC | DAC_CAL);	//set CAL method and single DMA channel mode
	CSiwrite(15, samples_per_buffer - 1);        //set capture lower base count
	CSiwrite(14, (samples_per_buffer - 1) >> 8); //set capture upper base count
	if (Mode1==FALSE)
		CSiwrite(28, capture_format_reg);    //set capture format
	CSiwrite(8, format_reg);            //set format and rate
	wait_for_init();
	CSreset(INDEX, MCE);    //reset ModeChangeEnable
	wait_for_ACI();

	CSiset(10, IEN);
	CSwrite(STATUS, 0);  //clear STATUS
	clearPIC(IRQ);
	unmaskIRQ(IRQ);

	CSiset(9, CEN);
	startDMA(DMA_Play, CaptureBuffer_physical_address, (buffer_size * 2) - 1, DMA_CAPTURE | DMA_AUTOINIT);

	enable_inputs();
	enable_outputs();
	if (debug)
		register_dump();
	while ((_kbhit()==0) && (CtlBrk==0))
	{	if (CaptureBuffer1_service == TRUE)
		{	if (swap_nibbles)
				NIBBLE_SWAP(CaptureBuffer1, buffer_size);
			if (program_mode == RECORD)
			{	WaveOUT_data_length += (long)fwrite(CaptureBuffer1, 1, buffer_size, wavefileout);
				if (WaveOUT_data_length > Record_limit)
					break;
			}
			else
				WaveOUT_data_length += buffer_size;
			if (peakmonitor)
				PEAKdisplay(getpeaks(CaptureBuffer1, buffer_size), wavefmt.nChannels);
			if (equalizer)
				FFTdisplay(FFT(LR++, CaptureBuffer1, FFTSampleSize, wavefmt.nChannels), wavefmt.nChannels);
			if (timemonitor)
				displaydatatime("\r");
			CaptureBuffer1_service=FALSE;
		}
		if (CaptureBuffer2_service == TRUE)
		{	if (swap_nibbles)
				NIBBLE_SWAP(CaptureBuffer2, buffer_size);
			if (program_mode == RECORD)
			{	WaveOUT_data_length += (long)fwrite(CaptureBuffer2, 1, buffer_size, wavefileout);
				if (WaveOUT_data_length > Record_limit)
					break;
			}
			else
				WaveOUT_data_length += buffer_size;
			if (peakmonitor)
				PEAKdisplay(getpeaks(CaptureBuffer2, buffer_size), wavefmt.nChannels);
			if (equalizer)
				FFTdisplay(FFT(LR++, CaptureBuffer2, FFTSampleSize, wavefmt.nChannels), wavefmt.nChannels);
			if (timemonitor)
				displaydatatime("\r");
			CaptureBuffer2_service=FALSE;
		}
		if (_inp(CSstatus) & SER)
			overrun=1;
		if (CSiread(11) & 0x02)
			clipleft=1;
		if (CSiread(11) & 0x08)
			clipright=1;
		if (debug==-2)
			printf("\nI11=0x%2.2X, DMA Status=0x%2.2X ", CSiread(11), _inp(DMA1_STATUS));
	}
	if (CSversion==CS4232)  //special fix for rev C only
	{	for(i=0;i<65535;i++)
			if (_inp(DMA1_STATUS) & dma_status_req[DMA_Play])
				break;
		CSireset(9, CEN);
	}
	_outp(DMA1_MASKREG, DMA_SETMASK | DMA_Play);        //mask channel

	bytesleft=((_inp(dma_count[DMA_Play]) + (_inp(dma_count[DMA_Play]) << 8)) % buffer_size) & 0xFFFE;

	if (CaptureBuffer1_service == TRUE)
	{	if (swap_nibbles)
			NIBBLE_SWAP(CaptureBuffer1, buffer_size);
		if (program_mode == RECORD)
			WaveOUT_data_length += (long)fwrite(CaptureBuffer1, 1, buffer_size, wavefileout);
		else
			WaveOUT_data_length += buffer_size;
	}
	else if (CaptureBuffer_using == 1)
	{	if (swap_nibbles)
			NIBBLE_SWAP(CaptureBuffer1, bytesleft);
		if (program_mode == RECORD)
			WaveOUT_data_length += (long)fwrite(CaptureBuffer1, 1, bytesleft, wavefileout);
		else
			WaveOUT_data_length += buffer_size;
	}

	if (CaptureBuffer2_service == TRUE)
	{	if (swap_nibbles)
			NIBBLE_SWAP(CaptureBuffer2, buffer_size);
		if (program_mode == RECORD)
			WaveOUT_data_length += (long)fwrite(CaptureBuffer2, 1, buffer_size, wavefileout);
		else
			WaveOUT_data_length += buffer_size;
	}
	else if (CaptureBuffer_using == 2)
	{	if (swap_nibbles)
			NIBBLE_SWAP(CaptureBuffer2, bytesleft);
		if (program_mode == RECORD)
			WaveOUT_data_length += (long)fwrite(CaptureBuffer2, 1, bytesleft, wavefileout);
		else
			WaveOUT_data_length += buffer_size;
	}
	WaveOUT_RIFF_data_count += WaveOUT_data_length;

	if (WaveOUT_data_length != 0L)
	{	memset(CaptureBuffer1, silence, buffer_size);
		if (program_mode == RECORD)
		{	WaveOUT_data_length += (long)fwrite(CaptureBuffer1, 1, buffer_size, wavefileout);
			WaveOUT_RIFF_data_count += fprintf(wavefileout, "%s ", format_name);
			WaveOUT_RIFF_data_count += fprintf(wavefileout, FILETAG);
		}
		else
			WaveOUT_data_length += buffer_size;
	}
	if (peakmonitor)
		PEAKdisplay(0, 0);
	if (equalizer)
		FFTdisplay(0, 0);
}

/******************************************************************/
void karaokedma(void)//play and record in full duplex mode w/mic input
{	register int i, j, bytesleft;

	bytes_to_read=(int)min((long)buffer_size, WaveIN_data_length);
	if (bytes_to_read != 0)
	{	bytes_read=fread(PlayBuffer1, 1, bytes_to_read, wavefilein);
		WaveIN_data_length -= bytes_read;
		if (swap_nibbles)
			NIBBLE_SWAP(PlayBuffer1, bytes_read);
		PlayBuffer_using=1;
		PlayBuffer1_service=FALSE;
	}
	else PlayBuffer1_service=TRUE;
	PlayBuffer2_service=TRUE;
	CaptureBuffer_using=1;
	CaptureBuffer1_service=FALSE;
	CaptureBuffer2_service=FALSE;

	CSreset(INDEX, TRD);
	CSiset(12, MODE2);      //set MODE 2
	CSireset(9, PEN | CEN);
	_outp(DMA1_MASKREG, DMA_SETMASK | DMA_Play);        //mask channel
	_outp(DMA1_MASKREG, DMA_SETMASK | DMA_Capture);        //mask channel
	CSireset(16, TE | SPE | PMCE | CMCE);
	CSiset(10, DTM);		//must have correct DMA timing
	CSiset(17, XTALE);	//keep both crystals on
	CSireset(22, SRE);	//must reset to allow I8 changes
	if ((capture_format_reg & 0xE0) == 0)
		CSireset(10, DEN);   //forces inverted dither enable for 8 bit linear

	CSset(INDEX, MCE);      //set ModeChangeEnable
	CSiwrite(15, samples_per_buffer - 1);        //set play lower base count
	CSiwrite(14, (samples_per_buffer - 1) >> 8); //set play upper base count
	CSireset(9, PPIO | CPIO | CAL_MASK);  //clear CAL bits
	CSiset(9, DAC_CAL);	//set CAL method
	CSireset(9, SDC);
	CSiwrite(31, samples_per_buffer - 1);        //set capture lower base count
	CSiwrite(30, (samples_per_buffer - 1) >> 8); //set capture upper base count
	CSiwrite(28, capture_format_reg);	//set capture format to linear 16 bit stereo
	CSiwrite(8, format_reg);  //set format and rate
	wait_for_init();
	CSreset(INDEX, MCE);    //reset ModeChangeEnable
	wait_for_ACI();

	CSiset(10, IEN);
	clearPIC(IRQ);       //clear any leftover interrupt
	CSwrite(STATUS, 0);  //clear STATUS
	unmaskIRQ(IRQ);

	CSiset(9, CEN | PEN);	//same time
	if (peakmonitor)
		PEAKdisplay(0, 0);
	if (CSversion == CS4232)   //special fix for rev C only
	{	CSset(INDEX, MCE);      //set ModeChangeEnable
		CSreset(INDEX, MCE);    //reset ModeChangeEnable
		wait_for_ACI();
	}

	startDMA(DMA_Play, PlayBuffer_physical_address, (buffer_size * 2) - 1, DMA_PLAYBACK | DMA_AUTOINIT);
	startDMA(DMA_Capture, CaptureBuffer_physical_address, (buffer_size * 2) - 1, DMA_CAPTURE | DMA_AUTOINIT);

	enable_inputs();
	enable_outputs();
	if (debug)
		register_dump();
	while ((WaveIN_data_length > 0L) && !feof(wavefilein))
	{	if (PlayBuffer1_service == TRUE)
		{	if ((bytes_to_read=(int)min((long)buffer_size, WaveIN_data_length)) != 0)
			{	bytes_read=fread(PlayBuffer1, 1, bytes_to_read, wavefilein);
				WaveIN_data_length -= bytes_read;
				if (bytes_read < buffer_size)
					memset(PlayBuffer1+bytes_read, silence, buffer_size - bytes_read);
				PlayBuffer1_service=FALSE;
			}
		}
		if (PlayBuffer2_service == TRUE)
		{	if ((bytes_to_read=(int)min((long)buffer_size, WaveIN_data_length)) != 0)
			{	bytes_read=fread(PlayBuffer2, 1, bytes_to_read, wavefilein);
				WaveIN_data_length -= bytes_read;
				if (bytes_read < buffer_size)
					memset(PlayBuffer2+bytes_read, silence, buffer_size - bytes_read);
				PlayBuffer2_service=FALSE;
			}
		}
		if (CaptureBuffer1_service==TRUE)
		{	WaveOUT_data_length += (long)fwrite(CaptureBuffer1, 1, buffer_size, wavefileout);
			if (WaveOUT_data_length > Record_limit)
				break;
			if (peakmonitor)
				PEAKdisplay(getpeaks(CaptureBuffer1, buffer_size), ((capture_format_reg & 0x10) ? 2 : 1));
			if (equalizer)
				FFTdisplay(FFT(LR++, CaptureBuffer1, FFTSampleSize, ((capture_format_reg & 0x10) ? 2 : 1)), ((capture_format_reg & 0x10) ? 2 : 1));
			if (timemonitor)
				displaydatatime("\r");
			CaptureBuffer1_service=FALSE;
		}
		if (CaptureBuffer2_service==TRUE)
		{	WaveOUT_data_length += (long)fwrite(CaptureBuffer2, 1, buffer_size, wavefileout);
			if (WaveOUT_data_length > Record_limit)
				break;
			if (peakmonitor)
				PEAKdisplay(getpeaks(CaptureBuffer2, buffer_size), ((capture_format_reg & 0x10) ? 2 : 1));
			if (equalizer)
				FFTdisplay(FFT(LR++, CaptureBuffer2, FFTSampleSize, ((capture_format_reg & 0x10) ? 2 : 1)), ((capture_format_reg & 0x10) ? 2 : 1));
			if (timemonitor)
				displaydatatime("\r");
			CaptureBuffer2_service=FALSE;
		}
		if (_kbhit() || CtlBrk)
		{	bytes_read=0;
			break;
		}
		if (_inp(CSstatus) & SER)
			underrun=1;
		if (CSiread(11) & 0x02)
			clipleft=1;
		if (CSiread(11) & 0x08)
			clipright=1;
	}

	if (bytes_read != 0)	//wait for play to finish
	{	for (i=0; i<samples_per_buffer; i++)    //wait up to one buffer time
		{	for (j=0; j<sample_time; j++)           //for the current to be empty
			{	if ((PlayBuffer1_service==TRUE) || (PlayBuffer2_service==TRUE))
					break;
			}
			if ((PlayBuffer1_service==TRUE) || (PlayBuffer2_service==TRUE))
				break;
		}
		if (PlayBuffer1_service == TRUE)
			memset(PlayBuffer1, silence, buffer_size);
		if (PlayBuffer2_service == TRUE)
			memset(PlayBuffer2, silence, buffer_size);
		//get current DMA position
		bytesleft=(_inp(dma_count[DMA_Play]) + (_inp(dma_count[DMA_Play]) << 8)) % buffer_size;
		//wait for playing of the last buffer
		for (i=(int)(bytesleft/sample_size); i>0; i--)
		{	for (j=0; j<sample_time; j++)
				;
			_inp(CSindex);
		}
	}

	CSiset(6, MUTE);        //mute DAC's
	CSiset(7, MUTE);
	if (CSversion==CS4232)  //special fix for rev C only
	{	for(i=0;i<65535;i++)
			if (_inp(DMA1_STATUS) & dma_status_req[DMA_Play])
				break;
	}
	CSireset(9, PEN);
	_outp(DMA1_MASKREG, DMA_SETMASK | DMA_Play);        //mask play channel

	if (CSversion==CS4232)       //special fix for rev C only
	{	for(i=0;i<65535;i++)
			if (_inp(DMA1_STATUS) & dma_status_req[DMA_Capture])
				break;
	}
	CSireset(9, CEN);
	_outp(DMA1_MASKREG, DMA_SETMASK | DMA_Capture);        //mask capture channel

	bytesleft=((_inp(dma_count[DMA_Capture]) + (_inp(dma_count[DMA_Capture]) << 8)) % buffer_size) & 0xFFFE;

	if (CaptureBuffer1_service == TRUE)
		WaveOUT_data_length += (long)fwrite(CaptureBuffer1, 1, buffer_size, wavefileout);
	else if (CaptureBuffer_using == 1)
		WaveOUT_data_length += (long)fwrite(CaptureBuffer1, 1, bytesleft, wavefileout);

	if (CaptureBuffer2_service == TRUE)
		WaveOUT_data_length += (long)fwrite(CaptureBuffer2, 1, buffer_size, wavefileout);
	else if (CaptureBuffer_using == 2)
		WaveOUT_data_length += (long)fwrite(CaptureBuffer2, 1, bytesleft, wavefileout);
	WaveOUT_RIFF_data_count += WaveOUT_data_length;

	if (WaveOUT_data_length != 0L)
	{	memset(CaptureBuffer1, silence, buffer_size);
		WaveOUT_data_length += (long)fwrite(CaptureBuffer1, 1, buffer_size, wavefileout);
		WaveOUT_RIFF_data_count += fprintf(wavefileout, "%s ", format_name);
		WaveOUT_RIFF_data_count += fprintf(wavefileout, FILETAG);
	}
	if (peakmonitor)
		PEAKdisplay(0, 0);
	if (equalizer)
		FFTdisplay(0, 0);
}

/******************************************************************/
void testmode(void)//play and record in full duplex mode
{	register int i, j, bytesleft;

	bytes_to_read=(int)min((long)buffer_size, WaveIN_data_length);
	if (bytes_to_read != 0)
	{	bytes_read=fread(PlayBuffer1, 1, bytes_to_read, wavefilein);
		WaveIN_data_length -= bytes_read;
		if (swap_nibbles)
			NIBBLE_SWAP(PlayBuffer1, bytes_read);
		PlayBuffer_using=1;
		PlayBuffer1_service=FALSE;
	}
	else PlayBuffer1_service=TRUE;
	PlayBuffer2_service=TRUE;
	CaptureBuffer_using=1;
	CaptureBuffer1_service=FALSE;
	CaptureBuffer2_service=FALSE;

	CSreset(INDEX, TRD);
	CSiset(12, MODE2);      //set MODE 2
	CSireset(9, PEN | CEN);
	_outp(DMA1_MASKREG, DMA_SETMASK | DMA_Play);        //mask channel
	_outp(DMA1_MASKREG, DMA_SETMASK | DMA_Capture);        //mask channel
	CSireset(16, TE | SPE | PMCE | CMCE);
	CSiset(10, DTM);		//must have correct DMA timing
	CSiset(17, XTALE);	//keep both crystals on
	CSireset(22, SRE);	//must reset to allow I8 changes
	if ((capture_format_reg & 0xE0) == 0)
		CSireset(10, DEN);   //forces inverted dither enable for 8 bit linear

	CSset(INDEX, MCE);      //set ModeChangeEnable
	CSiwrite(15, samples_per_buffer - 1);        //set play lower base count
	CSiwrite(14, (samples_per_buffer - 1) >> 8); //set play upper base count
	CSireset(9, PPIO | CPIO | CAL_MASK);  //clear CAL bits
	CSiset(9, DAC_CAL);	//set CAL method
	CSireset(9, SDC);
	CSiwrite(31, samples_per_buffer - 1);        //set capture lower base count
	CSiwrite(30, (samples_per_buffer - 1) >> 8); //set capture upper base count
	CSiwrite(28, capture_format_reg);	//set capture format to linear 16 bit stereo
	CSiwrite(8, format_reg);  //set format and rate
	wait_for_init();
	CSreset(INDEX, MCE);    //reset ModeChangeEnable
	wait_for_ACI();

	CSiset(10, IEN);
	clearPIC(IRQ);       //clear any leftover interrupt
	CSwrite(STATUS, 0);  //clear STATUS
	unmaskIRQ(IRQ);

	CSiset(9, CEN | PEN);	//same time
	if (peakmonitor)
		PEAKdisplay(0, 0);
	if (CSversion == CS4232)   //special fix for rev C only
	{	CSset(INDEX, MCE);      //set ModeChangeEnable
		CSreset(INDEX, MCE);    //reset ModeChangeEnable
		wait_for_ACI();
	}

	startDMA(DMA_Play, PlayBuffer_physical_address, (buffer_size * 2) - 1, DMA_PLAYBACK | DMA_AUTOINIT);
	startDMA(DMA_Capture, CaptureBuffer_physical_address, (buffer_size * 2) - 1, DMA_CAPTURE | DMA_AUTOINIT);

	enable_inputs();
	enable_outputs();
	if (debug)
		register_dump();
	while ((WaveIN_data_length > 0L) && !feof(wavefilein))
	{	if (PlayBuffer1_service == TRUE)
		{	if ((bytes_to_read=(int)min((long)buffer_size, WaveIN_data_length)) != 0)
			{	bytes_read=fread(PlayBuffer1, 1, bytes_to_read, wavefilein);
				WaveIN_data_length -= bytes_read;
				if (bytes_read < buffer_size)
					memset(PlayBuffer1+bytes_read, silence, buffer_size - bytes_read);
				PlayBuffer1_service=FALSE;
			}
		}
		if (PlayBuffer2_service == TRUE)
		{	if ((bytes_to_read=(int)min((long)buffer_size, WaveIN_data_length)) != 0)
			{	bytes_read=fread(PlayBuffer2, 1, bytes_to_read, wavefilein);
				WaveIN_data_length -= bytes_read;
				if (bytes_read < buffer_size)
					memset(PlayBuffer2+bytes_read, silence, buffer_size - bytes_read);
				PlayBuffer2_service=FALSE;
			}
		}
		if (CaptureBuffer1_service==TRUE)
		{	WaveOUT_data_length += (long)fwrite(CaptureBuffer1, 1, buffer_size, wavefileout);
			if (WaveOUT_data_length > Record_limit)
				break;
			if (peakmonitor)
				PEAKdisplay(getpeaks(CaptureBuffer1, buffer_size), ((capture_format_reg & 0x10) ? 2 : 1));
			if (equalizer)
				FFTdisplay(FFT(LR++, CaptureBuffer1, FFTSampleSize, ((capture_format_reg & 0x10) ? 2 : 1)), ((capture_format_reg & 0x10) ? 2 : 1));
			if (timemonitor)
				displaydatatime("\r");
			CaptureBuffer1_service=FALSE;
		}
		if (CaptureBuffer2_service==TRUE)
		{	WaveOUT_data_length += (long)fwrite(CaptureBuffer2, 1, buffer_size, wavefileout);
			if (WaveOUT_data_length > Record_limit)
				break;
			if (peakmonitor)
				PEAKdisplay(getpeaks(CaptureBuffer2, buffer_size), ((capture_format_reg & 0x10) ? 2 : 1));
			if (equalizer)
				FFTdisplay(FFT(LR++, CaptureBuffer2, FFTSampleSize, ((capture_format_reg & 0x10) ? 2 : 1)), ((capture_format_reg & 0x10) ? 2 : 1));
			if (timemonitor)
				displaydatatime("\r");
			CaptureBuffer2_service=FALSE;
		}
		if (_kbhit() || CtlBrk)
		{	bytes_read=0;
			break;
		}
		if (_inp(CSstatus) & SER)
			underrun=1;
		if (CSiread(11) & 0x02)
			clipleft=1;
		if (CSiread(11) & 0x08)
			clipright=1;
	}

	if (bytes_read != 0)	//wait for play to finish
	{	for (i=0; i<samples_per_buffer; i++)    //wait up to one buffer time
		{	for (j=0; j<sample_time; j++)           //for the current to be empty
			{	if ((PlayBuffer1_service==TRUE) || (PlayBuffer2_service==TRUE))
					break;
			}
			if ((PlayBuffer1_service==TRUE) || (PlayBuffer2_service==TRUE))
				break;
		}
		if (PlayBuffer1_service == TRUE)
			memset(PlayBuffer1, silence, buffer_size);
		if (PlayBuffer2_service == TRUE)
			memset(PlayBuffer2, silence, buffer_size);
		//get current DMA position
		bytesleft=(_inp(dma_count[DMA_Play]) + (_inp(dma_count[DMA_Play]) << 8)) % buffer_size;
		//wait for playing of the last buffer
		for (i=(int)(bytesleft/sample_size); i>0; i--)
		{	for (j=0; j<sample_time; j++)
				;
			_inp(CSindex);
		}
	}

	CSiset(6, MUTE);        //mute DAC's
	CSiset(7, MUTE);
	if (CSversion==CS4232)  //special fix for rev C only
	{	for(i=0;i<65535;i++)
			if (_inp(DMA1_STATUS) & dma_status_req[DMA_Play])
				break;
	}
	CSireset(9, PEN);
	_outp(DMA1_MASKREG, DMA_SETMASK | DMA_Play);        //mask play channel

	if (CSversion==CS4232)       //special fix for rev C only
	{	for(i=0;i<65535;i++)
			if (_inp(DMA1_STATUS) & dma_status_req[DMA_Capture])
				break;
	}
	CSireset(9, CEN);
	_outp(DMA1_MASKREG, DMA_SETMASK | DMA_Capture);        //mask capture channel

	bytesleft=((_inp(dma_count[DMA_Capture]) + (_inp(dma_count[DMA_Capture]) << 8)) % buffer_size) & 0xFFFE;

	if (CaptureBuffer1_service == TRUE)
		WaveOUT_data_length += (long)fwrite(CaptureBuffer1, 1, buffer_size, wavefileout);
	else if (CaptureBuffer_using == 1)
		WaveOUT_data_length += (long)fwrite(CaptureBuffer1, 1, bytesleft, wavefileout);

	if (CaptureBuffer2_service == TRUE)
		WaveOUT_data_length += (long)fwrite(CaptureBuffer2, 1, buffer_size, wavefileout);
	else if (CaptureBuffer_using == 2)
		WaveOUT_data_length += (long)fwrite(CaptureBuffer2, 1, bytesleft, wavefileout);
	WaveOUT_RIFF_data_count += WaveOUT_data_length;

	if (WaveOUT_data_length != 0L)
	{	memset(CaptureBuffer1, silence, buffer_size);
		WaveOUT_data_length += (long)fwrite(CaptureBuffer1, 1, buffer_size, wavefileout);
		WaveOUT_RIFF_data_count += fprintf(wavefileout, "%s ", format_name);
		WaveOUT_RIFF_data_count += fprintf(wavefileout, FILETAG);
	}
	if (peakmonitor)
		PEAKdisplay(0, 0);
	if (equalizer)
		FFTdisplay(0, 0);
}

/******************************************************************/
void playpio(void)	//play in programmed I/O mode
{	register BYTE c;

	if (Mode1==TRUE)
		CSireset(12, MODE2);    //set MODE 1
	else
		CSiset(12, MODE2);      //set MODE 2
	CSireset(9, CEN | PEN);
	CSireset(10, IEN);
	if ((CSversion!=CS4248) && (CSversion!=CS4231))
	{	CSiset(12, MODE2);                  //set MODE 2
		CSireset(16, TE | SPE | PMCE | CMCE);
		if (Mode1==TRUE)
			CSireset(12, MODE2); //set MODE 1
	}
	if ((CSversion == CS4232) || (CSversion == CS4232A))
	{	CSiset(12, MODE2);      //set MODE 2
		CSiset(17, XTALE);		//keep both crystals on
		CSireset(22, SRE);      //must reset to allow I8 changes
		if (Mode1==TRUE)
			CSireset(12, MODE2); //set MODE 1
	}

	CSset(INDEX, MCE);      //set ModeChangeEnable
	CSiset(9, PPIO);
	CSireset(9, SDC);
	CSiwrite(8, format_reg);//set format
	wait_for_init();
	CSreset(INDEX, MCE);    //reset ModeChangeEnable
	wait_for_ACI();

	PlayBuffer1_read=fread(PlayBuffer1, 1, (int)min((long)buffer_size, WaveIN_data_length), wavefilein);
	buffer_index=0;
	enable_outputs();
	if (debug)
		register_dump();

	CSwrite(STATUS, 0);  //clear STATUS
	CSiset(9, PEN);
	while ((_kbhit()==0) && (CtlBrk==0))
	{	if (swap_nibbles)
		{	while ((buffer_index < PlayBuffer1_read) && (_kbhit()==0) && (CtlBrk==0))
			{	if (_inp(CSstatus) & PRDY)
				{	c=PlayBuffer1[buffer_index++];
					_outp(CSiodata, (BYTE)((c >> 4) + (c << 4)));
				}
			}
		}
		else
		{	while ((buffer_index < PlayBuffer1_read) && (_kbhit()==0) && (CtlBrk==0))
			{	if (_inp(CSstatus) & PRDY)
				{	OUT_SAMPLE(&PlayBuffer1[buffer_index], sample_size, CSiodata);
					buffer_index += sample_size;
				}
			}
		}
		if (_kbhit() || feof(wavefilein) || CtlBrk)
			break;
		if (_inp(CSstatus) & SER)
			underrun=1;
		PlayBuffer1_read=fread(PlayBuffer1, 1, (int)min((long)buffer_size, WaveIN_data_length), wavefilein);
		WaveIN_data_length -= PlayBuffer1_read;
		buffer_index=0;
		displaydatatime("\r");
	}
	CSiset(6, MUTE);        //mute DAC's
	CSiset(7, MUTE);
	CSireset(9, PEN);
	CSset(INDEX, MCE);      //set ModeChangeEnable
	CSireset(9, PPIO);
	CSreset(INDEX, MCE);    //reset ModeChangeEnable
	wait_for_ACI();
}

/******************************************************************/
void capturepio(void)	//capture in programmed I/O mode
{	register BYTE c;

	if (Mode1==TRUE)
		CSireset(12, MODE2);    //set MODE 1
	else
		CSiset(12, MODE2);      //set MODE 2
	CSireset(9, CEN | PEN);
	CSireset(10, IEN);
	if ((CSversion!=CS4248) && (CSversion!=CS4231))
	{	CSiset(12, MODE2);                  //set MODE 2
		CSireset(16, TE | SPE | PMCE | CMCE);
		if (Mode1==TRUE)
			CSireset(12, MODE2); //set MODE 1
	}
	if ((CSversion == CS4232) || (CSversion == CS4232A))
	{	CSiset(12, MODE2);      //set MODE 2
		CSiset(17, XTALE);		//keep both crystals on
		CSireset(22, SRE);      //must reset to allow I8 changes
		if (Mode1==TRUE)
			CSireset(12, MODE2);                  //set MODE 1
	}
	if ((wavefmt.wBitsPerSample == 8) && (wavefmt.wFormatTag == WAVE_FORMAT_PCM))
		CSireset(10, DEN);      //sets inverted dither enable for 8 bit linear

	CSset(INDEX, MCE);      //set ModeChangeEnable
	CSiset(9, CPIO);
	CSireset(9, SDC);
	if (Mode1==FALSE)
		CSiwrite(28, capture_format_reg);//set capture format
	CSiwrite(8, format_reg);//set format and rate
	wait_for_init();
	CSreset(INDEX, MCE);    //reset ModeChangeEnable
	wait_for_ACI();

	buffer_index=0;
	enable_inputs();
	enable_outputs();
	if (debug)
		register_dump();

	CSwrite(STATUS, 0);  //clear STATUS
	CSiset(9, CEN);
	while ((_kbhit()==0) && (CtlBrk==0))
	{	if (swap_nibbles)
		{	while ((buffer_index < buffer_size) && (_kbhit()==0) && (CtlBrk==0))
			{	if (_inp(CSstatus) & CRDY)
				{	c=_inp(CSiodata);
					CaptureBuffer1[buffer_index++]=(BYTE)((c >> 4) + (c << 4));
				}
			}
		}
		else
		{	while ((buffer_index < buffer_size) && (_kbhit()==0) && (CtlBrk==0))
			{	if (_inp(CSstatus) & CRDY)
				{	IN_SAMPLE(&CaptureBuffer1[buffer_index], sample_size, CSiodata);
					buffer_index += sample_size;
				}
			}
		}
		if (_inp(CSstatus) & SER)
			overrun=1;
		if (program_mode == RECORD)
		{	WaveOUT_data_length += (long)fwrite(CaptureBuffer1, 1, buffer_index, wavefileout);
			if (WaveOUT_data_length > Record_limit)
				break;
		}
		else
			WaveOUT_data_length += buffer_index;
		buffer_index=0;
		displaydatatime("\r");
	}
	CSireset(9, CEN);
	CSset(INDEX, MCE);      //set ModeChangeEnable
	CSireset(9, CPIO);
	CSreset(INDEX, MCE);    //reset ModeChangeEnable
	wait_for_ACI();

	WaveOUT_RIFF_data_count += WaveOUT_data_length;
	if (WaveOUT_data_length != 0L)
	{	memset(CaptureBuffer1, silence, buffer_size);
		if (program_mode == RECORD)
		{	WaveOUT_data_length += (long)fwrite(CaptureBuffer1, 1, buffer_size, wavefileout);
			WaveOUT_RIFF_data_count += fprintf(wavefileout, FILETAG);
		}
		else
			WaveOUT_data_length += buffer_size;
	}
}

/******************************************************************/
void playtio(void)	//play programmed I/O during timer interrupts
							//MODE 2 ONLY
{	long l;

	PlayBuffer1_read=fread(PlayBuffer1, 1, (int)min((long)buffer_size, WaveIN_data_length), wavefilein);
	WaveIN_data_length -= PlayBuffer1_read;
	PlayBuffer1_service=FALSE;
	PlayBuffer_using=1;
	PlayBuffer2_service=TRUE;

	CSiset(12, MODE2);				//set MODE 2
	CSireset(9, CEN | PEN);
	CSireset(10, IEN);
	CSireset(16, TE | SPE | PMCE | CMCE);
	if ((CSversion == CS4232) || (CSversion == CS4232A))
	{	CSiset(17, XTALE);		//keep both crystals on
		CSireset(22, SRE);      //must reset to allow I8 changes
	}

	CSiwrite(15, 0xFF);     //just to keep sample ints from interferring
	CSiwrite(14, 0xFF);
	CSiwrite(21, (sample_time/10) >> 8);
	CSiwrite(20, max(sample_time/10, 16));

	CSset(INDEX, MCE);      //set ModeChangeEnable
	CSiset(9, PPIO);
	CSiwrite(8, format_reg);//set format
	wait_for_init();
	CSreset(INDEX, MCE);    //reset ModeChangeEnable
	wait_for_ACI();

	CSwrite(STATUS, 0);  //clear STATUS
	clearPIC(IRQ);
	unmaskIRQ(IRQ);

	enable_outputs();

	CSiset(10, IEN);
	CSiset(9, PEN);
	CSiset(16, TE);

	if (debug)
		register_dump();

	while (!feof(wavefilein) && (WaveIN_data_length > 0L))
	{	if (PlayBuffer1_service==TRUE)
		{	PlayBuffer1_read=fread(PlayBuffer1, 1, (int)min((long)buffer_size, WaveIN_data_length), wavefilein);
			WaveIN_data_length -= PlayBuffer1_read;
			if (swap_nibbles)
				NIBBLE_SWAP(PlayBuffer1, PlayBuffer1_read);
			PlayBuffer1_service=FALSE;
			if (peakmonitor)
				PEAKdisplay(getpeaks(PlayBuffer1, buffer_size), wavefmt.nChannels);
			if (equalizer)
				FFTdisplay(FFT(LR++, PlayBuffer1, FFTSampleSize, wavefmt.nChannels), wavefmt.nChannels);
			if (timemonitor)
				displaydatatime("\r");
		}
		if (PlayBuffer2_service==TRUE)
		{	PlayBuffer2_read=fread(PlayBuffer2, 1, (int)min((long)buffer_size, WaveIN_data_length), wavefilein);
			WaveIN_data_length -= PlayBuffer2_read;
			if (swap_nibbles)
				NIBBLE_SWAP(PlayBuffer2, PlayBuffer2_read);
			PlayBuffer2_service=FALSE;
			if (peakmonitor)
				PEAKdisplay(getpeaks(PlayBuffer2, buffer_size), wavefmt.nChannels);
			if (equalizer)
				FFTdisplay(FFT(LR++, PlayBuffer2, FFTSampleSize, wavefmt.nChannels), wavefmt.nChannels);
			if (timemonitor)
				displaydatatime("\r");
		}
		if (_kbhit() || CtlBrk)
		{	WaveIN_data_length=0;
			PlayBuffer1_service=TRUE;
			PlayBuffer2_service=TRUE;
			break;
		}
		if (_inp(CSstatus) & SER)
			underrun=1;
	}

	for (l=(long)sample_time*(long)buffer_size;(PlayBuffer1_service=FALSE) && l; l--);
	for (l=(long)sample_time*(long)buffer_size;(PlayBuffer2_service=FALSE) && l; l--);

	CSiset(6, MUTE);        //mute DAC's
	CSiset(7, MUTE);
	CSireset(16, TE);
	CSireset(9, PEN);
	CSireset(10, IEN);
	clearPIC(IRQ);  //clear any leftover interrupt
	maskIRQ(IRQ);   //turn off
	CSset(INDEX, MCE);      //set ModeChangeEnable
	CSireset(9, PPIO);
	CSreset(INDEX, MCE);    //reset ModeChangeEnable
	wait_for_ACI();

	if (peakmonitor)
		PEAKdisplay(0, 0);
	if (equalizer)
		FFTdisplay(0, 0);
}

/******************************************************************/
void capturetio(void)	//capture programmed I/O during timer interrupts
								//MODE 2 ONLY
{
	CSiset(12, MODE2);                  //set MODE 2
	CSireset(9, CEN | PEN);
	CSireset(10, IEN);
	CSireset(16, TE | SPE | PMCE | CMCE);
	if ((CSversion == CS4232) || (CSversion == CS4232A))
	{	CSiset(17, XTALE);		//keep both crystals on
		CSireset(22, SRE);      //must reset to allow I8 changes
	}
	if ((wavefmt.wBitsPerSample == 8) && (wavefmt.wFormatTag == WAVE_FORMAT_PCM))
		CSireset(10, DEN);      //sets inverted dither enable for 8 bit linear

	CSiwrite(15, 0xFF);     //just to keep sample ints from interferring
	CSiwrite(14, 0xFF);
	CSiwrite(31, 0xFF);     //just to keep sample ints from interferring
	CSiwrite(30, 0xFF);
	CSiwrite(21, (sample_time/10) >> 8);
	CSiwrite(20, max(sample_time/10, 16));

	CSset(INDEX, MCE);      //set ModeChangeEnable
	CSiset(9, CPIO);
	CSiwrite(28, capture_format_reg);//set capture format
	CSiwrite(8, format_reg);//set format and rate
	wait_for_init();
	CSreset(INDEX, MCE);    //reset ModeChangeEnable
	wait_for_ACI();

	CSwrite(STATUS, 0);  //clear STATUS
	clearPIC(IRQ);
	unmaskIRQ(IRQ);

	enable_inputs();
	enable_outputs();

	CSiset(10, IEN);
	CSiset(9, CEN);
	CSiset(16, TE);

	if (debug)
		register_dump();

	while ((_kbhit()==0) && (CtlBrk==0))
	{	if (CaptureBuffer1_service == TRUE)
		{	if (swap_nibbles)
				NIBBLE_SWAP(CaptureBuffer1, buffer_size);
			if (program_mode == RECORD)
			{	WaveOUT_data_length += (long)fwrite(CaptureBuffer1, 1, buffer_size, wavefileout);
				if (WaveOUT_data_length > Record_limit)
					break;
			}
			else
				WaveOUT_data_length += buffer_size;
			if (peakmonitor)
				PEAKdisplay(getpeaks(CaptureBuffer1, buffer_size), wavefmt.nChannels);
			if (equalizer)
				FFTdisplay(FFT(LR++, CaptureBuffer1, FFTSampleSize, wavefmt.nChannels), wavefmt.nChannels);
			if (timemonitor)
				displaydatatime("\r");
			CaptureBuffer1_service=FALSE;
		}
		if (CaptureBuffer2_service == TRUE)
		{	if (swap_nibbles)
				NIBBLE_SWAP(CaptureBuffer2, buffer_size);
			if (program_mode == RECORD)
			{	WaveOUT_data_length += (long)fwrite(CaptureBuffer2, 1, buffer_size, wavefileout);
				if (WaveOUT_data_length > Record_limit)
					break;
			}
			else
				WaveOUT_data_length += buffer_size;
			if (peakmonitor)
				PEAKdisplay(getpeaks(CaptureBuffer2, buffer_size), wavefmt.nChannels);
			if (equalizer)
				FFTdisplay(FFT(LR++, CaptureBuffer2, FFTSampleSize, wavefmt.nChannels), wavefmt.nChannels);
			if (timemonitor)
				displaydatatime("\r");
			CaptureBuffer2_service=FALSE;
		}
		if (_inp(CSstatus) & SER)
			overrun=1;
		if (CSiread(11) & 0x02)
			clipleft=1;
		if (CSiread(11) & 0x08)
			clipright=1;
	}
	CSiset(6, MUTE);        //mute DAC's
	CSiset(7, MUTE);
	CSireset(9, CEN);
	CSireset(16, TE);
	CSireset(10, IEN);
	clearPIC(IRQ);  //clear any leftover interrupt
	maskIRQ(IRQ);   //turn off
	CSset(INDEX, MCE);      //set ModeChangeEnable
	CSireset(9, CPIO);
	CSreset(INDEX, MCE);    //reset ModeChangeEnable
	wait_for_ACI();

	WaveOUT_RIFF_data_count += WaveOUT_data_length;
	if (WaveOUT_data_length != 0L)
	{	memset(CaptureBuffer1, silence, buffer_size);
		if (program_mode == RECORD)
		{	WaveOUT_data_length += (long)fwrite(CaptureBuffer1, 1, buffer_size, wavefileout);
			WaveOUT_RIFF_data_count += fprintf(wavefileout, FILETAG);
		}
		else
			WaveOUT_data_length += buffer_size;
	}
	if (peakmonitor)
		PEAKdisplay(0, 0);
	if (equalizer)
		FFTdisplay(0, 0);
}

/******************************************************************/
void readWAVheader(void)	//read and decode WAVE file header
{	long chunksize;
//  .WAV file format
//   4 bytes 'RIFF'
//   4 bytes <length>
//   4 bytes 'WAVE'
//   4 bytes 'fmt '
//   4 bytes  <length>  ; 10h - length of 'data' block
//   2 bytes  01        ; format tag
//   2 bytes  01        ; channels (1=mono, 2=stereo)
//   4 bytes  xxxx      ; samples per second
//   4 bytes  xxxx      ; average bytes per second
//   2 bytes  01/02/04  ; block alignment
//   2 bytes  04/08/16  ; bits per sample

//  in ADPCM formats there are more fields here
//   2 bytes  ?         ; extra byte count
//   2 bytes  ?         ; samples per block
//   2 bytes  ?         ; number of coefficients
//   2 bytes  ?         ; ADPCM coefficient
//   2 bytes  ?         ; ADPCM coefficient

//   4 bytes 'data'
//   4 bytes <length>
//   n bytes <sample data>

	if (read_str("RIFF", 4) != 0)				//file should be at beginning
		ExitErrorMsg("\nRIFF header not found.");
	if (fread(&WaveIN_RIFF_data_count, 1, 4, wavefilein) != 4)		//length of rest of file
		ExitErrorMsg("\nRIFF data count not found.");

	if (read_str("WAVE", 4) != 0)				//only wave files supported
		ExitErrorMsg("\nWAVE chunk not found.");
	if ((WaveIN_RIFF_data_count -= 4) <= 0L) // keep track of remaining bytes in file
		ExitErrorMsg("\nInvalid RIFF file.");

	if (read_str("fmt ", 4) != 0)				//format header must be first
		ExitErrorMsg("\nWAVE \"fmt \" header not found.");
	if ((WaveIN_RIFF_data_count -= 4) <= 0L) // keep track of remaining bytes in file
		ExitErrorMsg("\nInvalid RIFF file.");

	if (fread(&WaveIN_fmtdata_size, 1, 4, wavefilein) != 4)	//size of header
		ExitErrorMsg("\nWAVE \"fmt \" data size not found.");
	if ((WaveIN_RIFF_data_count -= 4) <= 0L) // keep track of remaining bytes in file
		ExitErrorMsg("\nInvalid RIFF file.");

	if (fread(&wavefmt, 1, (unsigned int)WaveIN_fmtdata_size, wavefilein) != (unsigned int)WaveIN_fmtdata_size)	//read header
		ExitErrorMsg("\nError reading WAV format.");
	if ((WaveIN_RIFF_data_count -= WaveIN_fmtdata_size) <= 0L)	//account for header size
		ExitErrorMsg("\nInvalid RIFF file.");

	if ((wavefmt.nChannels != 1) && (wavefmt.nChannels != 2))
	{	printf("\nNumber of channels (%d) greater than two.", wavefmt.nChannels);
		CleanExit(-1);
	}
	if (debug==-3)
		printWAVheader();
	switch (wavefmt.wFormatTag)
	{	case WAVE_FORMAT_PCM:
			if ((wavefmt.wBitsPerSample!=8) && (wavefmt.wBitsPerSample!=16))
			{	printf("\nInvalid Bits Per Sample (%d) found.", wavefmt.wBitsPerSample);
				CleanExit(-1);
			}
			if (wavefmt.nBlockAlign != (wavefmt.nChannels * wavefmt.wBitsPerSample / 8))
			{	printf("\nInvalid Block Alignment (%d) found.", wavefmt.nBlockAlign);
				if (wavefmt.nBlockAlign > 4)
					CleanExit(-1);
			}
			if (wavefmt.nAvgBytesPerSec != (wavefmt.nSamplesPerSec * (DWORD)wavefmt.nBlockAlign))
			{	printf("\nInvalid header field (AvgBPS=%ld <> SPS=%ld * BA=%d).",
					wavefmt.nAvgBytesPerSec, wavefmt.nSamplesPerSec, wavefmt.nBlockAlign);
				CleanExit(-1);
			}
			break;
		case WAVE_FORMAT_ALAW:
		case WAVE_FORMAT_MULAW:
			if (wavefmt.wBitsPerSample != 8)
			{	printf("\nInvalid Bits Per Sample (%d) found.", wavefmt.wBitsPerSample);
				CleanExit(-1);
			}
			if (wavefmt.nBlockAlign != wavefmt.nChannels)
			{	printf("\nInvalid block alignment (%d) found.", wavefmt.nBlockAlign);
				if (wavefmt.nBlockAlign > 4)
					CleanExit(-1);
			}
			break;
		case WAVE_FORMAT_CS_IMAADPCM:
			if (wavefmt.wBitsPerSample != 4)
			{	printf("\nInvalid Bits Per Sample (%d) found.", wavefmt.wBitsPerSample);
				CleanExit(-1);
			}
			if (wavefmt.nBlockAlign != wavefmt.nChannels)
			{	if (wavefmt.cbSize == 2)
				{	printf("\nBlocked ADPCM, will not play correctly.");
					blocked_adpcm=1;
				}
				else
				{	printf("\nInvalid Block Alignment (%d) found.", wavefmt.nBlockAlign);
					CleanExit(-1);
				}
			}
			break;
		case WAVE_FORMAT_IMA_ADPCM:
			if (wavefmt.wBitsPerSample != 4)
			{	printf("\nInvalid Bits Per Sample (%d) found.", wavefmt.wBitsPerSample);
				CleanExit(-1);
			}
			if (wavefmt.nBlockAlign != wavefmt.nChannels)
			{	if (wavefmt.cbSize == 2)
				{	printf("\nBlocked ADPCM, will not play correctly.");
					blocked_adpcm=1;
				}
				else
				{	printf("\nInvalid Block Alignment (%d) found.", wavefmt.nBlockAlign);
					CleanExit(-1);
				}
			}
			break;
		default:
			printf("\nUnknown wave format (tag=%d) found.", wavefmt.wFormatTag);
			CleanExit(-1);

	}
	while (WaveIN_RIFF_data_count)	//look for DATA chunk, ignore others
	{	if (read_str("data", 4) == 0)
			break;
		if ((WaveIN_RIFF_data_count -= 4) <= 0L)               // keep track of remaining bytes
			ExitErrorMsg("\nInvalid RIFF file.");
		if (fread(&chunksize, 1, 4, wavefilein) != 4)
			ExitErrorMsg("\nError reading \"chunk\" length.");
		if ((WaveIN_RIFF_data_count -= 4) <= 0L)               // keep track of remaining bytes
			ExitErrorMsg("\nInvalid RIFF file.");
		if (fseek(wavefilein, chunksize, SEEK_CUR) != 0)
			ExitErrorMsg("\nError reading chunk.");
	}
	if ((WaveIN_RIFF_data_count -= 4) <= 0L)                  // keep track of remaining bytes
		ExitErrorMsg("\nError finding \"data\".");
	if (fread(&WaveIN_data_length, 1, 4, wavefilein) != 4)
		ExitErrorMsg("\nError reading \"data\" length.");
	if ((WaveIN_RIFF_data_count -= 4) <= 0L)                  // keep track of remaining bytes
		ExitErrorMsg("\nNo data found.");
	//returns with file positioned ready to read audio WAVE data
}

/******************************************************************/
int read_str(char *id, int len)	//read string entries
{	int i, result=0;

	for (i=0;i<len;i++)
	{	if (fgetc(wavefilein) != id[i])
			result++;
	}
	return(result | feof(wavefilein));
}

/******************************************************************/
void writeWAVheader(void)	//write a WAVE file header
{
//  .WAV file format
//   4 bytes 'RIFF'
//   4 bytes <length>
//   4 bytes 'WAVE'
//   4 bytes 'fmt '
//   4 bytes  <length>  ; 10h - length of 'data' block
//   2 bytes  01        ; format tag
//   2 bytes  01        ; channels (1=mono, 2=stereo)
//   4 bytes  xxxx      ; samples per second
//   4 bytes  xxxx      ; average samples per second
//   2 bytes  01/02/04  ; block alignment
//   2 bytes  08/16     ; bits per sample
//   4 bytes 'data'
//   4 bytes <length>
//   n bytes <sample data>

	fwrite("RIFF", 1, 4, wavefileout);
	fwrite(&WaveOUT_RIFF_data_count, 1, 4, wavefileout);
	fwrite("WAVEfmt ", 1, 8, wavefileout);
	fwrite(&WaveOUT_fmtdata_size, 1, 4, wavefileout);
	fwrite(&wavefmt.wFormatTag, 1, 2, wavefileout);
	fwrite(&wavefmt.nChannels, 1, 2, wavefileout);
	fwrite(&wavefmt.nSamplesPerSec, 1, 4, wavefileout);
	fwrite(&wavefmt.nAvgBytesPerSec, 1, 4, wavefileout);
	fwrite(&wavefmt.nBlockAlign, 1, 2, wavefileout);
	fwrite(&wavefmt.wBitsPerSample, 1, 2, wavefileout);
	fwrite("data", 1, 4, wavefileout);
	fwrite(&WaveOUT_data_length, 1, 4, wavefileout);
	//returns with file positioned ready to write audio WAVE data
}

/******************************************************************/
void printWAVheader(void)	//display WAVE file header
{
	printf("\nWAVE chunk, format data size=%ld", WaveIN_fmtdata_size);
	printf("\nFormat type=%d", wavefmt.wFormatTag);
	printf("\nNumber of channels=%d", wavefmt.nChannels);
	printf("\nSample Rate=%ld", wavefmt.nSamplesPerSec);
	printf("\nAverage data rate %ld bytes/sec", wavefmt.nAvgBytesPerSec);
	printf("\nBlock alignment=%d", wavefmt.nBlockAlign);
	printf("\nBits per Sample=%d", wavefmt.wBitsPerSample);
}

/******************************************************************/
void setupWAVheader(void)	//initialize wav header for record/capture
{
	switch (wavefmt.wFormatTag)
	{	case WAVE_FORMAT_PCM:
			if ((wavefmt.wBitsPerSample != 8) &&
				 (wavefmt.wBitsPerSample != 16))
				ExitErrorMsg("\nInvalid audio data format.");
			break;
		case WAVE_FORMAT_ALAW:
			if (wavefmt.wBitsPerSample != 8)
				ExitErrorMsg("\nInvalid audio data format.");
			break;
		case WAVE_FORMAT_MULAW:
			if (wavefmt.wBitsPerSample != 8)
				ExitErrorMsg("\nInvalid audio data format.");
			break;
		case WAVE_FORMAT_CS_IMAADPCM:
			if (wavefmt.wBitsPerSample != 4)
				ExitErrorMsg("\nInvalid audio data format.");
			break;
		default:
			ExitErrorMsg("\nInvalid audio data format.");
	}
	if (wavefmt.wBitsPerSample == 4)
	{	wavefmt.nBlockAlign=wavefmt.nChannels;
		wavefmt.nAvgBytesPerSec=wavefmt.nSamplesPerSec;
	}
	else
	{	wavefmt.nBlockAlign=(wavefmt.wBitsPerSample * wavefmt.nChannels) / 8;
		wavefmt.nAvgBytesPerSec=wavefmt.nSamplesPerSec * (DWORD)wavefmt.nBlockAlign;
	}
}

/******************************************************************/
void setupCODECformat(void)	//initialize CODEC format reg
{	int i;

	capture_format_reg = format_reg = 0;
	if (wavefmt.nChannels == 2)	//stereo?
		capture_format_reg = format_reg = 0x10;
	i=find_playfreq(wavefmt.nSamplesPerSec);
	if (wavefmt.nSamplesPerSec != (DWORD)(freq_table[i].frequency))
	{	printf("\nExact sample rate %ld not available, using %ld.\n", wavefmt.nSamplesPerSec, freq_table[i].frequency);
		wavefmt.nSamplesPerSec=freq_table[i].frequency;
	}
	format_reg |= freq_table[i].crystal;
	format_reg |= freq_table[i].divisor << 1;
	switch (wavefmt.wFormatTag)
	{	case WAVE_FORMAT_PCM:
			strcpy(format_name, "PCM");
			if (wavefmt.wBitsPerSample == 8)
				silence=0x80;
			else if (wavefmt.wBitsPerSample == 16)
			{	format_reg |= 0x40;
				silence=0;
			}
			break;
		case WAVE_FORMAT_ALAW:
			if (wavefmt.wBitsPerSample != 8)
				ExitErrorMsg("\nInvalid audio data format.");
			strcpy(format_name, "ALaw");
			format_reg |= 0x60;
			silence=0x55;
			break;
		case WAVE_FORMAT_MULAW:
			if (wavefmt.wBitsPerSample != 8)
				ExitErrorMsg("\nInvalid audio data format.");
			strcpy(format_name, "MULaw");
			format_reg |= 0x20;
			silence=0x7F;
			break;
		case WAVE_FORMAT_IMA_ADPCM:
			if (wavefmt.wBitsPerSample != 4)
				ExitErrorMsg("\nInvalid audio data format.");
			strcpy(format_name, "IMAADPCM");
			format_reg |= 0xA0;
			silence=0;
			break;
		case WAVE_FORMAT_CS_IMAADPCM:
			if (wavefmt.wBitsPerSample != 4)
				ExitErrorMsg("\nInvalid audio data format.");
			strcpy(format_name, "CSADPCM");
			format_reg |= 0xA0;
			silence=0;
			break;
		case WAVE_FORMAT_ADPCM:
			if (wavefmt.wBitsPerSample != 4)
				ExitErrorMsg("\nInvalid audio data format.");
			strcpy(format_name, "MSADPCM");
			format_reg |= 0xA0;
			silence=0;
			break;
		default:
			ExitErrorMsg("\nInvalid audio data format.");
	}
	if ((program_mode != TEST) && (program_mode != KARAOKE) && (full_duplex && (wavefmt.wBitsPerSample == 4)))
		capture_format_reg |= 0x40;	//add linear 16 bit
	else
		capture_format_reg = (format_reg & 0xF0);
}

/******************************************************************/
int find_playfreq(long freq)	//find the nearest freq in the table
{	int  i, j;

	for (i=0; i < FTABCNT; i++ )
		if (freq <= freq_table[i].frequency)
			break;
	if (freq == freq_table[i].frequency)
		return(i);
	if (i == 0)
		return(i);
	if ((j=i--) == FTABCNT)
		return(i);
	if ((freq - freq_table[i].frequency) < (freq_table[j].frequency - freq))
		return(i);
	else
		return(j);
	return(0);
}

/******************************************************************/
void parsecommandline(int argc, char *argv[], int start)
{	int count;
	char *p, *s;
	void set_SR(long samplerate);
	void set_BPS(int bitspersec);

	for (count=start; count < argc; count++)
	{	p=argv[count];
		tryagain:
		switch (*p++)
		{	case 'p'://play
			case 'P':
				timemonitor=1;
				if (*p == '=')
				{	program_mode=PLAY;
					if ((s=strchr(p, ',' )) != NULL)
					{	*s++='\0';
						if (tolower(*s) == 'f')
						{	full_duplex=1;
							input_select=LOOP;
						}
						else goto parse_err_invarg;
					}
					if (_fullpath(path_buffer_in, ++p, _MAX_PATH) == NULL)
					{	printf("\nInvalid path or filename specified \"%s\"", p);
						goto parse_err_invarg;
					}
					break;
				}
				else if (strnicmp(p, "io", 2) == 0)
				{	pio=TRUE;
					buffer_size=16 * 1024;
					break;
				}
				goto parse_err_invarg;
			case 'r'://record
			case 'R':
				program_mode=RECORD;
				timemonitor=1;
				if (*p++ == '=')
				{	if ((s=strchr(p, ',' )) != NULL)
						*s++='\0';
					if (_fullpath(path_buffer_out, p, _MAX_PATH) == NULL)
					{	printf("\nInvalid path or filename specified \"%s\"", p);
						goto parse_err_invarg;
					}
					while ((p=s) != NULL)
					{	if ((s=strchr(p, ',' )) != NULL)
							*s++='\0';
						if (isdigit(*p))
						{	if ((WORD)atoi(p) < 100)        //if small, might be BPS
								set_BPS(atoi(p));
							else
								set_SR(atol(p));
						}
						else
							set_FORMAT(p);
					}
					break;
				}
				goto parse_err_invarg;
			case 'c'://capture only
			case 'C':
				program_mode=CAPTURE;
				timemonitor=1;
				if (*p++ == '=')
				{	s=p;
					while ((p=s) != NULL)
					{	if ((s=strchr(p, ',' )) != NULL)
							*s++='\0';
						if (isdigit(*p))
						{	if ((WORD)atoi(p) < 100)        //if small, might be BPS
								set_BPS(atoi(p));
							else
								set_SR(atol(p));
						}
						else
							set_FORMAT(p);
					}
				}
				break;
			case 'k'://KARAOKE play and record w/mic (records same format as play)
			case 'K':
				if ((CSversion != CS4232) && (CSversion != CS4232A))
					ExitErrorMsg("\nKaraoke mode only available on CS4232");
				program_mode=KARAOKE;
				input_select = MIC;
				full_duplex = TRUE;
				timemonitor=1;
				if (*p++ == '=')
				{	if ((s=strchr(p, ',' )) != NULL)
					{	*s++='\0';
						if (_fullpath(path_buffer_in, p, _MAX_PATH) == NULL)
						{	printf("\nInvalid path or filename specified \"%s\"", p);
							goto parse_err_invarg;
						}
						p=s;
					}
					if ((s=strchr(p, ',' )) != NULL)
					{	*s++='\0';
						if (_fullpath(path_buffer_out, p, _MAX_PATH) == NULL)
						{	printf("\nInvalid path or filename specified \"%s\"", p);
							goto parse_err_invarg;
						}
						p=s;
					}
					if (*p == 0)
						goto parse_err_invarg;
					sscanf(p, "0x%x", &CS4232_control);
					break;
				}
				goto parse_err_invarg;
			case 't':
			case 'T':
				if (*p++ == '=')	//special test play and record mode
				{	if ((s=strchr(p, ',' )) != NULL)
					{	*s++='\0';
						if (_fullpath(path_buffer_in, p, _MAX_PATH) == NULL)
						{	printf("\nInvalid path or filename specified \"%s\"", p);
							goto parse_err_invarg;
						}
						p=s;
					}
					if (_fullpath(path_buffer_out, p, _MAX_PATH) == NULL)
					{	printf("\nInvalid path or filename specified \"%s\"", p);
						goto parse_err_invarg;
					}
					program_mode = TEST;
					full_duplex = TRUE;
					timemonitor=1;
					break;
				}
				else if (strnicmp(p, "io", 2) == 0)
				{	tio=TRUE;
					break;
				}
				goto parse_err_invarg;
			case 'a'://set I/O address
			case 'A':
				if (*p++ == '=')
				{	if (sscanf(p, "0x%x", &IOADDR) != 1)
						goto parse_err_invarg;
					break;
				}
				goto parse_err_invarg;
			case 'i'://set IRQ channel
			case 'I':
				if (*p++ == '=')
				{	IRQ=atoi(p);
					break;
				}
				goto parse_err_invarg;
			case 'd'://set DMA channel
			case 'D':
				if (*p++ == '=')
				{	if ((s=strchr(p, ',' )) != NULL)
					{	*s++='\0';
						DMA_Play = atoi(p);
						if ((s=strchr(p=s, ',' )) != NULL)
						{	*s++='\0';    // 3 parms
							DMA_Capture = atoi(p);
							if (tolower(*s) == 's')
								single_mode=TRUE;
							else
								goto parse_err_invarg;
						}
						else    //only 2 parms
						{	if (isdigit(*p))
							{	DMA_Capture = atoi(p);

							}
							else if (tolower(*p) == 's')
								single_mode=TRUE;
							else
								goto parse_err_invarg;
						}
					}
					else    //only 1 parm
					{	if (*p != 0)
							DMA_Play = atoi(p);
						else
							goto parse_err_invarg;
					}
					break;
				}
				goto parse_err_invarg;
			case 'b'://set buffer size
			case 'B':
				if (*p++ == '=')
				{	buffer_size=atoi(p) * 1024;
					break;
				}
				goto parse_err_invarg;
			case 'l'://set digital loopback bit
			case 'L':
				digital_loopback=1;
				if (*p != 0)
					goto parse_err_invarg;
				break;
			case 'm'://set time and peak monitors
			case 'M':
				peakmonitor=1;
				if (*p++ == '=')
				{	if ((s=strchr(p, ',' )) != NULL)
					{	*s++='\0';
						if (tolower(*s) == 's')
							peakscrolling=1;
						else goto parse_err_invarg;
					}
					peakmonitor=atoi(p);
				}
				break;
			case 'f'://FFT
			case 'F':
				equalizer=1;
				if (*p++ == '=')
				{	if ((s=strchr(p, ',' )) != NULL)
					{	*s++='\0';
						if ((EQNumBars=atoi(p)) == 0)
							EQNumBars=16;
						if (isdigit(*s))
							FFTpower=atoi(s);
						else
							goto parse_err_invarg;
					}
					else    //only 1 parm
					{	if (*p != 0)
							EQNumBars=atoi(p);
						else
							goto parse_err_invarg;
					}
				}
				if ((EQNumBars!=8) && (EQNumBars!=16) && (EQNumBars!=32))
				{	printf("\nInvalid equalizer size (%d) specified", EQNumBars);
					goto parse_err_invarg;
				}
				if ((FFTpower < FFT_POWER_LOW) || (FFTpower > FFT_POWER_HIGH))
				{	printf("\nInvalid FFT power (%d) specified", FFTpower);
					goto parse_err_invarg;
				}
				FFTsize=(1 << FFTpower);
				break;
			case 's'://select input for capture
			case 'S':
				if (*p++ == '=')
				{	if ((s=strchr(p, ',' )) != NULL)
						*s++='\0';
					if (strnicmp(p, inputs[input_select=LINE], 4) == 0);
					else if (strnicmp(p, inputs[input_select=AUX], 3) == 0);
					else if (strnicmp(p, inputs[input_select=MIC], 3) == 0);
					else if (strnicmp(p, inputs[input_select=LOOP], 4) != 0)
						ExitErrorMsg("\nInvalid input specified");
					if ((p=s) != NULL)
					{	input_rightvol=input_leftvol=atoi(p) % 101;
						if ((s=strchr(p, ',' )) != NULL)
							*s++='\0';
						if ((p=s) != NULL)
							input_rightvol=atoi(p) % 101;
					}
					break;
				}
				goto parse_err_invarg;
			case 'o'://set volume or verbose flag
			case 'O':
				if (*p++ == '=')
				{	if ((s=strchr(p, ',' )) != NULL)
						*s++='\0';
					output_rightvol=output_leftvol=atoi(p) % 101;
					if ((p=s) != NULL)
					{	if ((s=strchr(p, ',' )) != NULL)
							*s++='\0';
						output_rightvol=atoi(p) % 101;
					}
					break;
				}
				goto parse_err_invarg;
			case 'v'://set verbose flag
			case 'V':
				verbose=TRUE;
				if (*p != 0)
					goto parse_err_invarg;
				break;
			case 'x'://debug register dump
			case 'X':
				if (*p++ == '=')
				{	if ((debug=atoi(p)) == 0)
						debug=1;
				}
				else
					debug=1;
				break;
			case '/':
				goto tryagain;
			case '?':
			case 'h':
			case 'H':
				display_syntax();
				CleanExit(0);
			default:
				goto parse_err_invarg;
		}
	}
	return;

parse_err_invarg:
	printf("\nCommand line argument number %d is invalid.\n", count);
	CleanExit(1);
}

/******************************************************************/
void display_syntax(void)	//command line syntax
{
	printf("\nCommand line arguments are:");
	printf("\n   P=<filename>[,F] Play from disk, opt. full duplex mode");
	printf("\n   R=<filename>[,<sr>,<bps>,<s/m>,<f>] Record to disk");
	printf("\n   C[=<sr>,<bps>,<s/m>,<f>] Capture without storing");
	printf("\n       sr=sample rate [5510...48000] (Default=44100)");
	printf("\n       bps=bits per sample [4,8,16] (Default=16)");
	printf("\n       m=Mono or s=Stereo (Default=Stereo)");
	printf("\n       f=format [PCM,ALAW,MULAW,ADPCM] (Default=PCM)");
	printf("\n   K=<play filename>,<record filename>,<control port>");
	printf("\n   T=<play filename>,<record filename>");
	printf("\n   A=<I/O address 0x534,etc> (Default=search)");
	printf("\n   I=<IRQ line 5,7,9,11,etc> (Default=5)");
	printf("\n   D=<DMA play ch>[,<DMA rec ch>][,S] S=single mode (Default=1,0)");
	printf("\n   S=<LINE,AUX,MIC,LOOP>[,<lvol>[,<rvol>]] (Default=LINE) {regs=0,1}");
	printf("\n   O=<left volume>[,<right volume>] (1-100) (Default=no chg) {regs=6,7}");
	printf("\n   B=<buffer size in KBytes 1-31> (Default based on average BPS)");
	printf("\n   M[=<Peak level display type 1-3>][,S] (Default=none)");
	printf("\n   F[=<# of Equilizer bars>,[<FFT power>]] (defaults=16,9)");
	printf("\n   V   Verbose output (Default=no display)");
	printf("\n   L   Digital loopback enable (Default=disable) {reg=13}");
	printf("\n   PIO Programmed I/O {reg=9}");
	printf("\n   TIO Timer interrupt programmed I/O {reg=9,10,16}");
	printf("\n");
}

/******************************************************************/
void set_SR(long samplerate)	//set wave sample rate paramter
{	int index;


	index=find_playfreq(samplerate);
	if (samplerate != freq_table[index].frequency)
		printf("\nExact sample rate %ld not available, using %ld", samplerate, freq_table[index].frequency);
	wavefmt.nSamplesPerSec=freq_table[index].frequency;
}

/******************************************************************/
void set_BPS(int bitspersec)	//set wave sample size parameter
{
	if (!((bitspersec == 4)||
			(bitspersec == 8)||
			(bitspersec == 16)))
	{	printf("\nInvalid sample size \"%d\" entered.", bitspersec);
		CleanExit(1);
	}
	wavefmt.wBitsPerSample=bitspersec;
}

/******************************************************************/
int set_FORMAT(char *s)	//set wave format parameters
{
	if (strnicmp(s, "PCM", 3) == 0)
		wavefmt.wFormatTag=WAVE_FORMAT_PCM;
	else if (strnicmp(s, "ALAW", 4) == 0)
	{	wavefmt.wFormatTag=WAVE_FORMAT_ALAW;
		wavefmt.wBitsPerSample=8;
	}
	else if (strnicmp(s, "MULAW", 5) == 0)
	{	wavefmt.wFormatTag=WAVE_FORMAT_MULAW;
		wavefmt.wBitsPerSample=8;
	}
	else if (strnicmp(s, "ADPCM", 5) == 0)
	{	wavefmt.wFormatTag=WAVE_FORMAT_CS_IMAADPCM;
		wavefmt.wBitsPerSample=4;
	}
	else if (tolower(*s) == 's')
		wavefmt.nChannels=2;
	else if (tolower(*s) == 'm')
		wavefmt.nChannels=1;
	else
	{	printf("\nInvalid entry in capture parameters \"%s\".", s);
		return(1);
	}
	return(0);
}

/******************************************************************/
int wait_for_init(void)	//wait for CODEC initialization period
{	WORD t;

	delayUS(10);//in Rev.C & Rev.D, 80H's show up < 10 usec late and leave < 1 msec early

	for (t=0; (_inp(CSindex)==INIT) && (t < 20); t++)       //wait for INIT
		delayUS(100);   // wait .1 ms
	if (t == 20)      // in any rev, if 80H's are present more than 2ms
		ExitErrorMsg("\nWait for init failed.");

	if (debug)
		printf("\nWaiting for INIT required %d us", t*100);

	delayMS(1);

	return 0;
}

/******************************************************************/
int wait_for_ACI(void)	//wait for CODEC calibration period
								//DAC CAL is 40 sample periods on 4232
								//full cal on 4231 is 168 sample periods
{	WORD t, tb;//time in tenths of milliseconds

	if (sample_time != 0)
	{	if ((CSversion==CS4248) || (CSversion==CS4231))
			t = ((168 * sample_time) / 100);	//FULL CAL
		else
			t = ((40 * sample_time) / 100);	//DAC CAL
	}
	else
		t = 200;
	if (debug)
		printf("\nWaiting %d sample periods for ACI", t);
	else
		t+=10;
	for (tb = t; (CSiread(11) & ACI) && (t != 0); t--)	//wait for ACI
		delayUS(100);   // wait .1 ms
	if (t == 0)
		ExitErrorMsg("\nWait for ACI failed.");
	if (debug)
		printf(" (Actual=%d)", tb - t);

	return 0;
}

/******************************************************************/
void delayMS(long howlong)	//delay in milliseconds
{
	if (ClockRunning)
	{	while (howlong--)
			delayUS(1194);  //1ms == 1194 * 838ns timer 0 ticks
	}
	else
	{	while (howlong--)
			delayUS(1050);  //1ms == 1050 * 950ns I/O's
	}
}

/******************************************************************/
void delayUS(WORD howlong)	//delay in microseconds
{	register WORD t;

	//if greater than overhead and clock is running
	if ((howlong > 20) && ClockRunning)
	{	t=GET_US_CLK();
		while ((GET_US_CLK() - t) < howlong);
	}
	else	//use good'ol i/o instr. (speed may vary from bus to bus)
	{	while (howlong--)
			_inp(0x80);     // 1us each (about 950ns on Pentium 90 PCI)
	}
}

/******************************************************************/
int findCRYSTALcodec()	//search for CODEC at several I/O locations
{	int i;
	static WORD CSbases[4]={0x530,0x604,0xE80,0xF40};

	if (IOADDR != 0)
		if (isCRYSTALcodec(IOADDR - 4))
			return(1);
	if (CSindex != 0)
		if (isCRYSTALcodec(CSindex - 4))
			return(1);

	for (i=0; i<4; i++)
		if (isCRYSTALcodec(CSbases[i]))
			return(1);
	return(0);
}

/******************************************************************/
int isCRYSTALcodec(int addr)	//check for legal CODEC
{	int indexsave, modesave;

	if (_inp(addr + 4) == 0xFF)
		return(0);//fail

	CSindex=addr + 4;
	CSdata=CSindex + 1;
	CSstatus=CSindex + 2;
	CSiodata=CSindex + 3;

	if (CRYSTAL_check == 0)				//special switch allows for no codec detect
		return(1);

	indexsave=_inp(CSindex);
	modesave=CSiread(12);
	CSiwrite(12, MODE2);            	//try to clear CRYSTAL bit and set mode 2
	if ((CSiread(12) & 0x8F) == 0x81)//Crystal bit should be on plus ID
	{	CSiset(12, MODE2);      		//try to set mode 2
		if ((CSiread(25) & 0xE7) == 0x80)
		{	CSversion=CS4248;
			goto CRYSTALgood;
		}
		else
			goto CRYSTALfail;
	}
	if ((CSiread(12) & 0x8F) != 0x8A)//Crystal bit should be on plus ID
		goto CRYSTALfail;
	if ((CSiread(12) & MODE2) == 0)	//fail mode 2?
	{	_outp(CSindex, 0x99);   		//try to access mode 2 reg 17 in mode 1
		if ((_inp(CSindex) & 0x10) && ((_inp(CSdata) & 0x07) == 0x01))
		{	CSversion=CS4248A;
			goto CRYSTALReservedBits;
		}
		else
			goto CRYSTALfail;
	}
	CSireset(12, MODE2);			//make sure mode 1
	_outp(CSindex, 0x99);		//try to access mode 2 reg 25 in mode 1
	if (_inp(CSindex) & 0x10)	//did the address bit turn on?
	{	if ((_inp(CSdata) & 0xE7) == 0xA0)
			CSversion=CS4231A;
		else if ((_inp(CSdata) & 0xE7) == 0xA2)
			CSversion=CS4232;
		else if ((_inp(CSdata) & 0xE7) == 0xC2)
			CSversion=CS4232A;
		else
			goto CRYSTALfail;
	}
	else    //no
	{	CSiset(12, MODE2);		//try to set mode 2
		if ((CSiread(25) & 0xE7) == 0x80)
			CSversion=CS4231;
		else
			goto CRYSTALfail;
	}
CRYSTALReservedBits:
	CSiset(12, MODE2);
	switch (CSversion)
	{	case CS4231:
			if ((CSiread(22) != 0x80) || (CSiread(23) != 0x80) || (CSiread(27) != 0x80) || (CSiread(29) != 0x80))
			{	if (debug)
					printf("\nCS4231 failed reserved bit test\n");
				goto CRYSTALfail;
			}
			break;
		case CS4248A:
			if ((CSiread(22) != 0) || (CSiread(23) != 0) || (CSiread(27) != 0) || (CSiread(29) != 0))
			{	if (debug)
					printf("\nCS4248A failed reserved bit test\n");
				goto CRYSTALfail;
			}
			break;
		case CS4231A:
			if ((CSiread(22) != 0) || (CSiread(27) != 0) || (CSiread(29) != 0))
			{	if (debug)
					printf("\nCS4231A failed reserved bit test\n");
				goto CRYSTALfail;
			}
			break;
		case CS4232:
			if (((CSiread(23) & 0xFE) != 0) || ((CSiread(28) & 0x0F) != 0))
			{	if (debug)
					printf("\nCS4232 failed reserved bit test\n");
				goto CRYSTALfail;
			}
			break;
		case CS4232A:
			if (((CSiread(23) & 0xFE) != 0) || ((CSiread(28) & 0x07) != 0))
			{	if (debug)
					printf("\nCS4232A failed reserved bit test\n");
				goto CRYSTALfail;
			}
			break;
	}
CRYSTALgood:
	CSiwrite(12, modesave);
	_outp(CSindex, indexsave);
	return(1);//good return

CRYSTALfail:
	CSiwrite(12, modesave);
	_outp(CSindex, indexsave);
	CSindex=0;
	CSdata=0;
	CSstatus=0;
	CSiodata=0;
	return(0);//fail return
}

/******************************************************************/
int getSBparams(void)/* Read parameters from BLASTER enivronment variable */
{	char *t, *blaster;

	t=getenv("BLASTER");
	if (!t)
		return(FALSE);
	blaster=strdup(t);             // Get a copy
	if (t=strtok(blaster, " \t"))
	{	do                          // Parse the BLASTER variable
		{	switch (t[0])
			{	case 'A':
				case 'a':             // I/O address
					sscanf(t + 1, "%x", &SBioaddr);
					break;
				case 'I':
				case 'i':             // IRQ
					SBirq=atoi(t + 1);
					break;
				case 'D':
				case 'd':             // DMA channel
					SBdmachan=atoi(t + 1);
					break;
				case 'T':
				case 't':             // SB type
					break;
				default:
					printf("Unknown BLASTER option %c\n",t[0]);
					break;
			}
		}  while (t=strtok(NULL," \t"));
	}
	else
	{	free(blaster);
		return(FALSE);
	}
	free(blaster);
	return(TRUE);
}

/******************************************************************/
int ASICread(int reg) /* read access to ASIC regs 0-3 */
{
	_outp(ASIC_R0, ASICpassword);
	return(_inp(reg));
}

/******************************************************************/
void ASICwrite(int reg, int value) /* write access to ASIC regs 0-7 */
{
	_outp(ASIC_R0, ASICpassword);
	_outp(reg, value);
}

/******************************************************************/
int CSiread(int reg)    //read any indirect register
{
	_outp(CSindex, (_inp(CSindex) & INDEX_MASK) | reg);
	return(_inp(CSdata));
}

/******************************************************************/
void CSiwrite(int reg, int value)       //write any indirect register
{
	_outp(CSindex, (_inp(CSindex) & INDEX_MASK) | reg);
	_outp(CSdata, value);
}

/******************************************************************/
void CSiset(int reg, int bits)  //turn on one or more bits in indirect reg
{
	_outp(CSindex, (_inp(CSindex) & INDEX_MASK) | reg);
	_outp(CSdata, _inp(CSdata) | bits );
}

/******************************************************************/
void CSireset(int reg, int bits)        //turn off one or more bits in indirect reg
{
	_outp(CSindex, (_inp(CSindex) & INDEX_MASK) | reg);
	_outp(CSdata, _inp(CSdata) & ~bits );
}

/******************************************************************/
int CSread(int reg)     //read CODEC reg
{
	return(_inp(CSindex + reg));
}

/******************************************************************/
void CSwrite(int reg, int value)	//write CODEC reg
{
	_outp(CSindex + reg, value);
}

/******************************************************************/
void CSset(int reg, int bits)	//set specified bits
{
	_outp(CSindex + reg, _inp(CSindex + reg) | bits);
}

/******************************************************************/
void CSreset(int reg, int bits)	//clear specified bits
{
	_outp(CSindex + reg, _inp(CSindex + reg) & ~bits);
}

/******************************************************************/
void clearPIC(int IRQline)	//send non-specific EOI to PIC
{
	if (IRQline >= 8)
		_outp(PIC2, EOI);
	_outp(PIC1, EOI);
}

/******************************************************************/
void maskIRQ(int IRQline)	//turn off specified IRQ
{
	if (IRQline < 8)
		_outp(PIC1MASKREG, _inp(PIC1MASKREG) | IRQmasks[IRQline]);
	else
		_outp(PIC2MASKREG, _inp(PIC2MASKREG) | IRQmasks[IRQline]);
}

/******************************************************************/
void unmaskIRQ(int IRQline)	//turn on specified IRQ
{
	if (IRQline < 8)
		_outp(PIC1MASKREG, _inp(PIC1MASKREG) & ~IRQmasks[IRQline]);
	else
		_outp(PIC2MASKREG, _inp(PIC2MASKREG) & ~IRQmasks[IRQline]);
}

/******************************************************************/
void startDMA(WORD dma_ch, DWORD phys_addr, WORD count, BYTE mode)
{
	_disable();
	if ((dma_ch >= 0) && (dma_ch < 4))
	{	_outp(DMA1_MASKREG, DMA_SETMASK | dma_ch);   //mask channel
		_outp(DMA1_CLEAR_FF, 0);        //clear first/last flip-flop
		if (single_mode==TRUE)
			_outp(DMA1_MODEREG, mode | dma_ch);  //set mode
		else
			_outp(DMA1_MODEREG, DMA_DEMAND | mode | dma_ch);  //set mode

		_outp(dma_page[dma_ch], (BYTE) (phys_addr >> 16));      //set page register
		_outp(dma_address[dma_ch], (BYTE) phys_addr);   //set base and current address
		_outp(dma_address[dma_ch], (BYTE) (phys_addr >> 8));

		_outp(dma_count[dma_ch], (BYTE) count); //set base and current count
		_outp(dma_count[dma_ch], (BYTE) (count >> 8));

		_outp(DMA1_MASKREG, dma_ch);  // unmask
	}
	_enable();
}

/******************************************************************/
void saveCODECregs(void)	//store a copy of CODEC regs
{	int i;

	for (i=0;i<32;i++)
		codec_regs[i]=CSiread(i);
	regs_saved=1;
}

/******************************************************************/
void restoreCODECregs(void)	//reload CODEC regs
{	int i;

	CSset(INDEX, MCE);      //set ModeChangeEnable
	for (i=0;i<32;i++)
	{	CSiwrite(i, codec_regs[i]);
		delayMS(5);				//just in case any init's are caused
	}
	CSreset(INDEX, MCE);    //set ModeChangeEnable
	delayMS(50);
	regs_saved=0;
}

/******************************************************************/
void register_dump(void)	//CODEC register dump
{	unsigned int i;

	printf("\nCODEC REGISTERS: Index=0x%2.2X Status=0x%2.2X",
		CSread(INDEX), CSread(STATUS));
	for (i=0;i<32;i+=8)
	{	printf("\nI%2.2d=0x%2.2X", i, CSiread(i));
		printf(" I%2.2d=0x%2.2X", i+1, CSiread(i+1));
		printf(" I%2.2d=0x%2.2X", i+2, CSiread(i+2));
		printf(" I%2.2d=0x%2.2X", i+3, CSiread(i+3));
		printf(" I%2.2d=0x%2.2X", i+4, CSiread(i+4));
		printf(" I%2.2d=0x%2.2X", i+5, CSiread(i+5));
		printf(" I%2.2d=0x%2.2X", i+6, CSiread(i+6));
		printf(" I%2.2d=0x%2.2X", i+7, CSiread(i+7));
	}
	printf("\n");
}

/******************************************************************/
void enableCODECaccess(int access)	//control ASIC to access CODEC
{
	codec_ASIC_access=access;
	_disable();
	if (ASICpassword == MAD16password)
	{	ASICwrite(ASIC_R3, (ASICread(ASIC_R3) & 0xFC) | 1);//set access to ASIC_R5
		if (access==TRUE)
			ASICwrite(ASIC_R5, (ASICread(ASIC_R5) & 0xC0) | 0x1A);//enable access to codec
		else
			ASICwrite(ASIC_R5, (ASICread(ASIC_R5) & 0xC0) | 0x2A);//disable access to codec
	}
	else if (ASICpassword == OPTi929password)
	{	if (access==TRUE)
			ASICwrite(ASIC_R5, 0xBF);	//enable access to codec
		else
			ASICwrite(ASIC_R5, 0xAF);	//disable access to codec
	}
	_enable();
}

/******************************************************************/
void print_heading(void)	//prints the three heading lines defined above
{
	printf(heading1);
	printf(heading2);
	printf(heading3);
}

/******************************************************************/
int ThinkPad(void)	//detects ThinkPad750 if present
{	_segment romseg, ramseg;
	BYTE __based(romseg) *bROM;
	WORD __based(romseg) *wROM;
	WORD __based(ramseg) *wRAM;

	romseg=0xF000;
	bROM=(BYTE __based(romseg) *)(0x16);
	if (strncmp(bROM, "IBM", 3) == 0)
	{	_outp(0x15E8, 0x1C);
		_outp(0x15E9, _inp(0x15E9) | 2);
		ramseg=0x0000;
		wRAM=(WORD __based(ramseg) *)(0x40E);
		romseg=*wRAM;
		bROM=(BYTE __based(romseg) *)(0x14C);
		wROM=(WORD __based(romseg) *)(0x14D);
		IOADDR=*wROM;
		IRQ=*bROM;
		DMA_Play = 0;
		DMA_Capture = 1;
		if (isCRYSTALcodec(IOADDR - 4))
		{	if (verbose)
				printf("\nCrystal CODEC found in ThinkPad750 at IO=0x%0X", IOADDR);
			return(1);
		}
	}
	return(0);
}

/******************************************************************/
int MAD16_asic(void)	//detects MAD16 if present
{
	ASICpassword = MAD16password;
	if (_inp(ASIC_R1) != ASICread(ASIC_R1))
	{	ASICwrite(ASIC_R1, ASICread(ASIC_R1) | 0x80);//set to SB mode
		IRQ=MCSCirq[ASICread(ASIC_R3) >> 6];
		DMA_Play=MCSCdrq[(ASICread(ASIC_R3) >> 4) & 3];
		if ((ASICread(ASIC_R1) & 0x80) == 0)
			return(0);
		enableCODECaccess(TRUE);
		if (!isCRYSTALcodec(WSSbase[(ASICread(ASIC_R1) & 0x30) >> 4]))
		{	printf("\nCrystal CODEC not found at IO=0x%0X MAD16_R1=0x%2.2X",
				WSSbase[(ASICread(ASIC_R1) & 0x30) >> 4],
				ASICread(ASIC_R1));
			exit(-1);
		}
		return(1);
	}
	return(0);
}

/******************************************************************/
int OPTi929_asic(void)	//detects OPTi929 if present
{
	ASICpassword = OPTi929password;
	if (_inp(ASIC_R1) != ASICread(ASIC_R1))
	{	ASICwrite(ASIC_R1, ASICread(ASIC_R1) | 0x80);//set to SB mode
		IRQ=OPTiSCirq[ASICread(ASIC_R3) >> 6];
		DMA_Play=OPTiSCdrq[(ASICread(ASIC_R3) >> 4) & 3];
		if ((ASICread(ASIC_R1) & 0x80) == 0)
			return(0);
		enableCODECaccess(TRUE);
		if (!isCRYSTALcodec(WSSbase[(ASICread(ASIC_R1) & 0x30) >> 4]))
		{	printf("\nCrystal CODEC not found at IO=0x%0X OPTi_R1=0x%2.2X",
				WSSbase[(ASICread(ASIC_R1) & 0x30) >> 4],
				ASICread(ASIC_R1));
			exit(-1);
		}
		return(1);
	}
	return(0);
}

/******************************************************************/
void ExitErrorMsg(char *msg)	//display error msg and exit
{
	if (msg != NULL)
		printf(msg);
	CleanExit(-1);
}

/******************************************************************/
void CleanExit(int code)	//clean up to exit
{
	maskIRQ(IRQ);   //turn off
	clearPIC(IRQ);  //clear any leftover interrupt
	if (program_mode == KARAOKE)
		_outp(CS4232_control+1, (_inp(CS4232_control+1) & 0xCF));	//reset karaoke mode
	if (regs_saved)
		restoreCODECregs();
	if ((MAD16found || OPTi929found) && codec_ASIC_access)
		enableCODECaccess(FALSE);
	if (wavefilein != NULL)
		fclose(wavefilein);
	if (wavefileout != NULL)
		fclose(wavefileout);
	if (filebuf != NULL)
		_hfree(filebuf);
	if (PlayBuffer_ptr != NULL)
		_hfree(PlayBuffer_ptr);
	if (CaptureBuffer_ptr != NULL)
		_hfree(CaptureBuffer_ptr);
	if (CtlBrkHooked)
	{	_dos_setvect(0x1B, OldInt1BISR);//un-install ISR
		_dos_setvect(0x23, OldInt23ISR);//un-install ISR
	}
	if (KeybdHooked)
		_dos_setvect(KeybdInt, OldKeybdISR);
	if (ISRHooked)
		_dos_setvect(IRQtoInt[IRQ], OldISR);//un-install ISR
	if (equalizer)
	{	FFTdisplay(0, wavefmt.nChannels);//clear display
		FFTfree();
		_settextcursor(savecursor);
	}
	if (peakmonitor)
	{	_settextcursor(savecursor);
		_settextposition(bottomrow-1, 1);
	}
	exit(code);
}

/******************************************************************/
void (__interrupt __far KeybdISR) (void)	//monitor keystrokes to control mixer
{	BYTE key;

	void DAClouder(void);
	void ADClouder(void);
	void DACsofter(void);
	void ADCsofter(void);
	void Looplouder(void);
	void Loopsofter(void);

// __segment       romseg;
// BYTE __based(romseg) *keystate;
// romseg=0x40;
// keystate=(BYTE __based(romseg) *)(0x17);
// if ((*keystate & 0x0C) != 0x0C)

	key=_inp(0x60);
	if (key==0xE0)
	{	Extended=1;
		_chain_intr(OldKeybdISR);
	}
	if (Extended==0)
		_chain_intr(OldKeybdISR);
	Extended=0;
	switch (key)
	{	case 0x49://master louder    //gray page up
			DAClouder();
			DAClouder();
			output_leftvol=(63 - (CSiread(6) & 0x3F)) * 100 / 63;
			output_rightvol=(63 - (CSiread(7) & 0x3F)) * 100 / 63;
			ADClouder();
			input_leftvol=(CSiread(0) & 0x0F) * 100 / 15;
			input_rightvol=(CSiread(1) & 0x0F) * 100 / 15;
			if (input_select == LOOP)
				Looplouder();
			break;
		case 0x51://master softer       //gray page down
			DACsofter();
			DACsofter();
			output_leftvol=(63 - (CSiread(6) & 0x3F)) * 100 / 63;
			output_rightvol=(63 - (CSiread(7) & 0x3F)) * 100 / 63;
			ADCsofter();
			input_leftvol=(CSiread(0) & 0x0F) * 100 / 15;
			input_rightvol=(CSiread(1) & 0x0F) * 100 / 15;
			if (input_select == LOOP)
				Loopsofter();
			break;
		case 0x47://DAC's louder        //gray home key
			DAClouder();
			output_leftvol=(63 - (CSiread(6) & 0x3F)) * 100 / 63;
			output_rightvol=(63 - (CSiread(7) & 0x3F)) * 100 / 63;
			break;
		case 0x4F://DAC's softer        //gray end key
			DACsofter();
			output_leftvol=(63 - (CSiread(6) & 0x3F)) * 100 / 63;
			output_rightvol=(63 - (CSiread(7) & 0x3F)) * 100 / 63;
			break;
		case 0x52://ADC's louder        //gray insert key
			ADClouder();
			input_leftvol=(CSiread(0) & 0x0F) * 100 / 15;
			input_rightvol=(CSiread(1) & 0x0F) * 100 / 15;
			break;
		case 0x53://ADC's softer        //gray delete key
			ADCsofter();
			input_leftvol=(CSiread(0) & 0x0F) * 100 / 15;
			input_rightvol=(CSiread(1) & 0x0F) * 100 / 15;
			break;
		case 0x35://FFT buffer snapshot     // gray slash
			snapshot=1;
	}
	_outp(0x20, 0x20);
}

/******************************************************************/
void DAClouder(void)    //DAC's are attenuators
{	int i, k;

	i=_inp(CSindex);
	if (((k=CSiread(6)) & 0x3F) > 0)
		CSiwrite(6, --k);
	if (((k=CSiread(7)) & 0x3F) > 0)
		CSiwrite(7, --k);
	_outp(CSindex, i);
}

/******************************************************************/
void DACsofter(void)    //DAC's are attenuators
{	int i, k;

	i=_inp(CSindex);
	if (((k=CSiread(6)) & 0x3F) < 0x3F)
		CSiwrite(6, ++k);
	if (((k=CSiread(7)) & 0x3F) < 0x3F)
		CSiwrite(7, ++k);
	_outp(CSindex, i);
}

/******************************************************************/
void ADClouder(void)    //ADC's are gains
{	int i, k;

	i=_inp(CSindex);
	if (((k=CSiread(0)) & 0xF) < 0xF)
		CSiwrite(0, ++k);
	if (((k=CSiread(1)) & 0xF) < 0xF)
		CSiwrite(1, ++k);
	_outp(CSindex, i);
}

/******************************************************************/
void ADCsofter(void)    //ADC's are gains
{	int i, k;

	i=_inp(CSindex);
	if (((k=CSiread(0)) & 0xF) > 0)
		CSiwrite(0, --k);
	if (((k=CSiread(1)) & 0xF) > 0)
		CSiwrite(1, --k);
	_outp(CSindex, i);
}

/******************************************************************/
void Looplouder(void)	//higher loopback volume
{	int i, k;

	i=_inp(CSindex);
	if (((k=CSiread(4)) & 0x1F) > 0)
		CSiwrite(4, --k);
	if (((k=CSiread(5)) & 0x1F) > 0)
		CSiwrite(5, --k);
	_outp(CSindex, i);
}

/******************************************************************/
void Loopsofter(void)	//lower loopback volume
{	int i, k;

	i=_inp(CSindex);
	if (((k=CSiread(4)) & 0x1F) < 0x1F)
		CSiwrite(4, ++k);
	if (((k=CSiread(5)) & 0x1F) < 0x1F)
		CSiwrite(5, ++k);
	_outp(CSindex, i);
}