//*****************************************************************************
//* Name:
//*      1212card.c
//*
//* Project:
//*      1212 I/O VxD
//*
//* Author:
//*      Bill Jenkins
//*
//* Description:
//*      This file contains the definition of the card configuration structure
//*      functions.
//*
//* Modification Log:
//*
//*      1.7   11/17/99 Bill
//*      (v1.2) We now use CONFIGMG_Get_Bus_Info() to determine the bus and device numbers.
//*
//*      1.6   08/20/97 Bill
//*      (v1.0B6) Sync box variables on the wave devices are initialized to zero -
//*      even if they have non-zero entries in the system registry.  This prevents
//*      the card from getting locked out if the user checks the sync box of an
//*      output device other than the system default and stores this in the registry.
//*
//*      1.5   08/01/97 Bill
//*      (v1.0B5) Added message box call when the card bombs out.  Plus, new
//*      ISR handling of card stoppage.
//*
//*      1.4   06/17/97 Bill
//*      (v1.11) Added increment of new record buffer index variable.
//*
//*      1.3   06/16/97 Bill
//*      (v1.10) Removed code to open and close the debug file.
//*
//*      1.2   06/03/97 Bill
//*      (v1.05) Added code to open and close the debug file.
//*
//*      1.1   11/14/96 Bill
//*      Initial version created.
//*
//*
//* Copyright (c) 1996 Korg, Inc.
//* All rights reserved.
//*
//* This program is protected as an unpublished work under the U.S.
//* copyright laws.  The above copyright notice is not intended to
//* effect a publication of this work.
//*
//* This program is the confidential and proprietary information of
//* Korg and all its subsidiaries.
//*****************************************************************************

#include <vtoolsc.h>
#include <vmm.h>
#include <debug.h>
#include <vmmreg.h>
#include <configmg.h>
#include <vpicd.h>
#include <vxdsvc.h>

#ifndef  K1212API_H
#include "1212api.h"
#endif
#ifndef  K1212IFPR_H
#include "1212ifpr.h"
#endif
#ifndef  K1212INTF_H
#include "1212intf.h"
#endif
#ifndef  K1212CARD_H
#include "1212card.h"
#endif
#ifndef  K1212CFG_H
#include "1212cfg.h"
#endif
#ifndef  K1212DNLD_H
#include "1212dnld.h"
#endif
#ifndef  K1212REGISTRY_H
#include "1212registry.h"
#endif
#ifndef K1212WAVE_H
#include "1212wave.h"
#endif

// ----------------------------------------------------------------------------
// global declarations
// ----------------------------------------------------------------------------
extern DDB   The_DDB;
extern WORD  ClockSourceSelector[];


// ----------------------------------------------------------------------------
// function declarations
// ----------------------------------------------------------------------------
Event_THUNK        k1212Glbl_Event_Thunk;       // Thunk for global event handler
Event_THUNK        k1212StopCard_Event_Thunk;   // Thunk for global card error event handler
Event_THUNK        k1212StopCardAck_Event_Thunk;// Thunk for global card stop event handler
VPICD_HWInt_THUNK  k1212Int_Thunk;              // Thunk for interrupt handler
ASYNCTIMEOUT_THUNK k1212DSPBoot_Timeout_Thunk;  // Thunk for waiting for card to boot
TIMEOUT_THUNK      k1212CardBoot_Timeout_Thunk; // Thunk for waiting for card reset

RestrictedEvent_THUNK   k1212Download_Thunk;     // Thunk for downloading the DSP microcode

void  ShowErrorBox(void);
void _cdecl ErrorBoxCallback      (PVOID RefData, DWORD flags);
void _cdecl ProcessDSPBootComplete(PVOID RefData, DWORD flags);




// ----------------------------------------------------------------------------
// the following array of card configuration data structures stores all
// configuration data for each 1212 I/O card in the system.
// ----------------------------------------------------------------------------
cardConfig cardData[k1212MaxCards];


// ----------------------------------------------------------------------------
// getBufferCount
//
//    This function returns the number of record and playback buffers used by
//    the card.
// ----------------------------------------------------------------------------
DWORD  getBufferCount(void)
{
   return kNumBuffers;
}


// ----------------------------------------------------------------------------
// createCardBuffers
//
//    This function creates the KorgSharedBuffer and KorgMiscBuffer for the
//    card.  These spaces are accessed by ring 3 applications that interface
//    with this driver.
//
//    Returns: TRUE if all is well, or FALSE if the allocation failed.
// ----------------------------------------------------------------------------
BOOL  CreateCardBuffers(DWORD cardIndex)
{
   DWORD      totalBufSize;
   DWORD      numBufPages;
   PVOID      linAddress;
   PVOID      physAddress;
   MEMHANDLE  memHandle;

   // ---------------------------------------------------------------
   // get the size of the structures and then allocate enough memory
   // for them.  Set the pointers in the card data structure.
   // ---------------------------------------------------------------
   totalBufSize = sizeof(KorgSharedBuffer);
   numBufPages  = ((totalBufSize / PAGESIZE) + 1);   // plus 1 for fractional part

   if (_PageAllocate(numBufPages,     // number of pages to allocate
                     PG_SYS,          // allocate in system shared area
                     NULL,            // no VM specified for sys memory
                     0,               // 4K alignment
                     0,               // min acceptable page number
                     MAXSYSTEMPAGE,   // max acceptable page number
                     &physAddress,    // pointer for physical address returned
                     (PAGECONTIG   |  // specify: contiguous memory
                      PAGEFIXED    |  //          can't be moved
                      PAGEUSEALIGN |  //          use align - required for contig
                      PAGEZEROINIT),  //          initialize to all zeroes
                     &memHandle,      // receiver for the handle to this memory block
                     &linAddress      // receiver for the linear address of the block
       )) {

      setSharedBufferPtr(cardIndex, (KorgSharedBuffer*)linAddress);
      setSharedPhyBufPtr(cardIndex, (KorgSharedBuffer*)physAddress);
      setSharedBufferHdl(cardIndex, memHandle);

      #ifdef DEBUG
         Debug_Printf_Service("Korg 1212 Card %d: shared buffer handle 0x%x, phys 0x%x, lin 0x%x\n",
                              cardIndex,
                              getSharedBufferHdl(cardIndex),
                              getSharedPhyBufPtr(cardIndex),
                              getSharedBufferPtr(cardIndex)
         );
      #endif // DEBUG

      return TRUE;
   }
   else {

      #ifdef DEBUG
         Debug_Printf_Service("Korg 1212 Card %d: Shared buffer allocation failed.\n",
                              cardIndex
         );
      #endif // DEBUG

      return FALSE;
   }
}


// ----------------------------------------------------------------------------
// deleteCardBuffers
//
//    This function deletes the KorgSharedBuffer and KorgMiscBuffer for the
//    card.
//
//    Returns: TRUE if all is well, or FALSE if the allocation failed.
// ----------------------------------------------------------------------------
BOOL  DeleteCardBuffers(DWORD cardIndex)
{
   if (getSharedBufferPtr(cardIndex)) {
      #ifdef DEBUG
         Debug_Printf_Service("Korg 1212 Card %d: Freeing shared buffer. address = 0x%x\n",
                              cardIndex,
                              getSharedBufferPtr(cardIndex)
         );
      #endif // DEBUG
      _PageFree(getSharedBufferHdl(cardIndex), 0);
   }
   return TRUE;
}


void printCardStuff(DWORD cardIndex)
{
   int statRegVal;
   int controlRegVal;
   int idRegVal;

   #ifdef DEBUG
//      Debug_Printf_Service("Korg 1212 Card %d status reg = 0x%08x\n",
//                           cardIndex,
//                           readStatusReg(cardIndex)
//      );
//      Debug_Printf_Service("Korg 1212 Card %d control reg = 0x%08x\n",
//                           cardIndex,
//                           readControlReg(cardIndex)
//      );
//      Debug_Printf_Service("Korg 1212 Card %d ID reg = 0x%08x\n",
//                           cardIndex,
//                           readIdReg(cardIndex)
//      );
      Debug_Printf_Service("Current time slice granularity is %d milliseconds\n",
                           Get_Time_Slice_Granularity()
      );
   #endif // DEBUG
}


// ----------------------------------------------------------------------------
// startConfiguration
//
//    This function begins the card configuration process.
//
//    Returns: CR_SUCCESS if all is well, or one of the CONFIGRET error codes.
// ----------------------------------------------------------------------------
CONFIGRET startConfiguration(DEVNODE devNode)
{
   CONFIGRET   returnCode;
   CMCONFIG    cmcfgData;
   char*       devIdBuffer;
   ULONG       devIdLength;
   DWORD       busNum;
   DWORD       deviceNum;
   DWORD       cardIndex;
   sPCIAccess  pciAccess;
   CMBUSTYPE   busType;
   DWORD       busInfoLength;
   BYTE        devNumber;

   // --------------------------------------------------------------------
   // Get the resource configuration information from the configuration
   // manager.
   // --------------------------------------------------------------------
   if ((returnCode = CONFIGMG_Get_Alloc_Log_Conf(&cmcfgData,
                                                 devNode,
                                                 CM_GET_ALLOC_LOG_CONF_ALLOC
                     )) != CR_SUCCESS) {

      return(returnCode);
   }

   // --------------------------------------------------------------------
   // Find a free card configuration data block locally.
   // --------------------------------------------------------------------
   for (cardIndex = 0; cardIndex < k1212MaxCards; cardIndex++) {
      if (getCardState(cardIndex) == K1212_STATE_NONEXISTENT) {
         break;   // found a free element
      }
   }
   if (cardIndex >= k1212MaxCards) {
      return(CR_FAILURE);
   }

   // ----------------------------------------------------------------------
   // fill in the configuration data
   // ----------------------------------------------------------------------
   setDevNode      (cardIndex, devNode);  // store the device node in the card data structure
   setFillThreadHdl(cardIndex, 0);        // make note - has not yet been set by an application

   setIrqNum     (cardIndex, cmcfgData.bIRQRegisters[0]);
   setPhysMemBase(cardIndex, (char*)cmcfgData.dMemBase[0]);
   setIoBase     (cardIndex, cmcfgData.wIOPortBase[0]);
   setMemLength  (cardIndex, cmcfgData.dMemLength[0]);
   setCardState  (cardIndex, K1212_STATE_UNINITIALIZED);

   // ----------------------------------------------------------------------
   // make the physical memory available to us for reads and writes
   // ----------------------------------------------------------------------
   setMemBase(cardIndex,
              (char*)MapPhysToLinear(getPhysMemBase(cardIndex),
                                     getMemLength(cardIndex),
                                     0
                     )
   );

   // ----------------------------------------------------------------------
   // Get the device ID string to obtain the card's bus and device numbers.
   // Find out how big the string is, allocate space to store it, get the
   // string, and parse the bus and device numbers out of it.
   // ----------------------------------------------------------------------
#if 0
   if ((returnCode = CONFIGMG_Get_Device_ID_Size(&devIdLength,
                                                 devNode,
                                                 0
                     )) != CR_SUCCESS) {
      return(returnCode);
   }

   if (!(devIdBuffer = malloc(devIdLength+1))) {
      return(CR_OUT_OF_MEMORY);
   }

   if ((returnCode = CONFIGMG_Get_Device_ID(devNode,
                                            devIdBuffer,
                                            devIdLength+1,
                                            0
                     )) != CR_SUCCESS) {
      return(returnCode);
   }

   if ((returnCode = ParseDevIdString(devIdBuffer,
                                      &busNum,
                                      &deviceNum
                     )) != CR_SUCCESS) {
      return(returnCode);
   }
   setBusNum   (cardIndex, busNum);
   setDeviceNum(cardIndex, deviceNum);
   free(devIdBuffer);                     // done with the buffer
#endif

	busInfoLength = sizeof(pciAccess);
	if (returnCode = CONFIGMG_Get_Bus_Info
							(
								devNode,
								&busType,
								&busInfoLength,
								&pciAccess,
								0
							) != CR_SUCCESS) {
		return (returnCode);
	}

	devNumber = (pciAccess.bDevFuncNumber >> 3);
   setBusNum   (cardIndex, pciAccess.bBusNumber);
   setDeviceNum(cardIndex, devNumber);

   // -----------------------------------------------------------------------
   // Get the registry path for future access to stored configuration values.
   // -----------------------------------------------------------------------
   GetRegistryPath(&The_DDB,
                   getRegPathPtr(cardIndex),
                   MAX_REGPATH_SIZE
   );

   // -----------------------------------------------------------------------
   // Now that the configuration data has been stored, we can setup for
   // communication to the card.
   // -----------------------------------------------------------------------
   SetupCardInterface(cardIndex);
   CreateCardBuffers (cardIndex);

   // --------------------------------------------------------------------
   // Hook the card's IRQ.
   // --------------------------------------------------------------------
   if ((returnCode = hookCardIRQ(cardIndex,
                                 getIrqNum(cardIndex)
                     )) != CR_SUCCESS) {
      return(returnCode);
   }

   // --------------------------------------------------------------------
   // Now that all the preliminary setup is done, send the card the reboot
   // command and then finish up the configuration.
   // --------------------------------------------------------------------
   return RebootCard(cardIndex);
}


// ----------------------------------------------------------------------------
// RebootCard
//
//    This function reboots the card and schedules the final configuration
//    steps.
//
//    Returns: CR_SUCCESS if all is well, or one of the CONFIGRET error codes.
// ----------------------------------------------------------------------------
CONFIGRET RebootCard(DWORD cardIndex)
{
   PVOID       refdata;

   if (cardIndex > k1212MaxCards) {
      return CR_INVALID_DEVICE_ID;
   }
   if (getCardState(cardIndex) == K1212_STATE_NONEXISTENT) {
      return CR_INVALID_DEVICE_ID;
   }

   // --------------------------------------------------------------------
   // (re)init debug variables
   // --------------------------------------------------------------------
   setOpenCount      (cardIndex, 0);
   clearIrqCount     (cardIndex);
   clearCmdRetryCount(cardIndex);

   // --------------------------------------------------------------------
   // Send the card the reboot command, so we know it is in a good state.
   // We'll give it a few milliseconds to come back up, then finish the
   // configuration process.
   // --------------------------------------------------------------------
   refdata = HeapAllocate(sizeof(DWORD),
                          HEAPLOCKEDIFDP
   );
   if (refdata == NULL) {
      return CR_OUT_OF_MEMORY;
   }

   *((DWORD*)refdata) = cardIndex;

   // ----------------------------------------------------
   // reboot the card and update its state
   // ----------------------------------------------------
   Send1212Command(cardIndex,
                   K1212_DB_RebootCard,
                   DUMMY_PARAMETER,
                   DUMMY_PARAMETER,
                   DUMMY_PARAMETER,
                   DUMMY_PARAMETER
   );
   setCardState(cardIndex, K1212_STATE_UNINITIALIZED);


   if (Set_Global_Time_Out(K1212CARD_BOOT_DELAY_IN_MS,
                           refdata,
                           finishConfiguration,
                           &k1212CardBoot_Timeout_Thunk
       )) {
      return CR_SUCCESS;
   }
   else {
      return CR_FAILURE;
   }
}


void __stdcall DoDSPDownload
(
   VMHANDLE       hVM,
   THREADHANDLE   hThread,
   PVOID          Refdata,
   PCLIENT_STRUCT pRegs
)
{
   DWORD cardIndex;
   cardIndex = (DWORD)Refdata;
   downloadDSPCode(cardIndex);
}


// ---------------------------------------------------------------------------
// finishConfiguration
//
//    Once the card has been reset by the startConfiguration routine, this
//    routine gets called after a period of time.  Here, we finish the
//    configuration process.
// ---------------------------------------------------------------------------
void __stdcall finishConfiguration(VMHANDLE       hVM,
                                   PCLIENT_STRUCT pcrs,
                                   PVOID          refData,
                                   DWORD          extra)
{
   DWORD cardIndex;

   cardIndex = *((DWORD*)refData);
   HeapFree(refData, 0);            // free up the memory now that we're done with it

   // --------------------------------------------------------------------
   // Now that the IRQ is hooked, and all the configuration data is setup,
   // we can program the card's PCI command and interrupt control
   // registers to allow it to do DMAs and interrupt us.
   // --------------------------------------------------------------------
   writeStatusReg(cardIndex,
                  (PCI_INT_ENABLE_BIT            |
                   PCI_DOORBELL_INT_ENABLE_BIT   |
                   LOCAL_INT_ENABLE_BIT          |
                   LOCAL_DOORBELL_INT_ENABLE_BIT |
                   LOCAL_DMA1_INT_ENABLE_BIT)
   );


   #ifdef DEBUG
      // ----------------------------------------------------------------
      // print stuff from the configuration manager
      // ----------------------------------------------------------------
      Debug_Printf_Service("Korg 1212 Card %d: state variable address is 0x%x \n",
                           cardIndex,
                           &(cardData[cardIndex].state)
      );
      Debug_Printf_Service("Korg 1212 Card %d: reports IRQ %d \n",
                           cardIndex,
                           getIrqNum(cardIndex)
      );
      Debug_Printf_Service("Korg 1212 Card %d: reports physical memBase 0x%x \n",
                           cardIndex,
                           getPhysMemBase(cardIndex)
      );
      Debug_Printf_Service("Korg 1212 Card %d: reports linear memBase 0x%x \n",
                           cardIndex,
                           getMemBase(cardIndex)
      );
      Debug_Printf_Service("Korg 1212 Card %d: reports memLength 0x%x \n",
                           cardIndex,
                           getMemLength(cardIndex)
      );
      Debug_Printf_Service("Korg 1212 Card %d: reports bus number    = %d \n",
                           cardIndex,
                           getBusNum(cardIndex)
      );
      Debug_Printf_Service("Korg 1212 Card %d: reports device number = %d \n",
                           cardIndex,
                           getDeviceNum(cardIndex)
      );
   #endif // DEBUG

   printCardStuff(cardIndex);

   // -----------------------------------------------------------------------
   // Get the DSP code download to the card started if we're past system
   // init.  Otherwise, let the init complete message handler get it started.
   // -----------------------------------------------------------------------
   if (VMM_GetSystemInitState() >= SYSSTATE_PREINITCOMPLETE) {
      Call_Restricted_Event(0,                                          // no boost
                            Get_Cur_VM_Handle(),                        // current VM
                            (PEF_WAIT_CRIT | PEF_WAIT_NOT_NESTED_EXEC), // suggested values for
                                                                        //  avoiding deadlock
                            (PVOID)cardIndex,                           // pass in card index
                            DoDSPDownload,                              // pointer to callback function
                            0,                                          // no timeout specified
                            &k1212Download_Thunk                        // the thunk for the function
      );
   }
}


// ----------------------------------------------------------------------------
// stopConfiguration
//
//    This function makes sure any remaining dynamically allocated resources
//    for the card have been freed up.
//
//    Returns: CR_SUCCESS if all is well, or one of the CONFIGRET error codes.
// ----------------------------------------------------------------------------
CONFIGRET stopConfiguration(DEVNODE devNode)
{
   DWORD       cardIndex;
   DWORD       tempIndex;
   DWORD       count;
   CONFIGRET   returnCode = CR_SUCCESS;
   ULONG       intHandle;
   WORD        error;                     // v1.05 for debug file closing

   // --------------------------------------------------------------------------
   // find the card index corresponding to this devnode.
   // --------------------------------------------------------------------------
   for (cardIndex = 0; cardIndex < k1212MaxCards; cardIndex++) {
      if (getDevNode(cardIndex) == devNode) {
         break;      // found the matching entry
      }
   }

   // --------------------------------------------------------------------------
   // take the card out of monitor mode - if it was there
   // --------------------------------------------------------------------------
   K1212TurnOffIdleMonitor(cardIndex);

   // --------------------------------------------------------------------------
   // if this is the last card that is virtualizing this IRQ, then remove the
   // ISR from this IRQ.  In either case, set the interrupt handle to 0.
   // --------------------------------------------------------------------------
   intHandle = getIrqHandle(cardIndex);
   count     = 0;
   for (tempIndex = 0; tempIndex < k1212MaxCards; tempIndex++) {
      if (getIrqHandle(tempIndex) == intHandle) {
         if (tempIndex != cardIndex) {
            count++;
         }
      }
   }
   if (!count) {
      VPICD_Force_Default_Behavior(intHandle);
   }
   setIrqHandle(cardIndex, 0);

   // --------------------------------------------------------------------------
   // now free up allocated resources.
   // --------------------------------------------------------------------------
   freeDSPucodeImageMemory(cardIndex);


   DeleteCardBuffers(cardIndex);

   #ifdef DEBUG_MSG
      R0_CloseFile(getFileHandle(cardIndex), &error);    // v1.05 - v1.10 removed
   #endif DEBUG_MSG
   initCardData(cardIndex);

   return returnCode;
}


// ----------------------------------------------------------------------------
// freeDSPucodeImageMemory
//
//    This function makes sure that the memory allocated for the DSP microcode
//    image has been freed up.
// ----------------------------------------------------------------------------
void freeDSPucodeImageMemory(DWORD cardIndex)
{
   if (getDSPMemPtr(cardIndex)) {
      #ifdef DEBUG
         Debug_Printf_Service("Korg 1212 Card %d: Card DSP microcode image manually freed\n",
                              cardIndex
         );
      #endif // DEBUG
      _PageFree(getDSPMemHdl(cardIndex), 0);
   }
}


// ----------------------------------------------------------------------------
// k1212IrqHandler
//
//    This function handles hardware interrupts from our card.  Since the IRQ
//    may be shared with another device, we first check the doorbell to make
//    sure we were interrupted by the card.
//
//    Returns: TRUE (thunk will clear carry to mark that we processed the
//                   interrupt), or FALSE (thunk sets carry)
// ----------------------------------------------------------------------------
BOOL __stdcall k1212IrqHandler(VMHANDLE  hVM,
                               IRQHANDLE hIRQ)
{
   DWORD doorbellValue;
   DWORD cardIndex;

   // -------------------------------------------------------------------------
   // We first need to determine which card has interrupted us - if any.
   // To do this, we spin through the array of cards looking for cards that
   // have the same IRQ handle.  For each one found, we check its status
   // register to see if it interrupted us.
   // -------------------------------------------------------------------------
   for (cardIndex = 0; cardIndex < k1212MaxCards; cardIndex++) {

      if (getIrqHandle(cardIndex) == hIRQ) {
         doorbellValue = readInDoorbell(cardIndex);

         if (doorbellValue != 0) {                  // did the card interrupt us?
            writeInDoorbell(cardIndex,              // yes, clear the interrupt on the card
                            doorbellValue
            );
            incIrqCount(cardIndex);                 // make note that we had an(other) interrupt

            // -----------------------------------------------------------------
            // We've determined this card to have interrupted us, so it's time
            // to process the interrupt.
            // -----------------------------------------------------------------
            switch (doorbellValue) {

               case K1212_DB_DSPDownloadDone:
                  // -----------------------------------------------------------
                  // the DSP download has completed.  We can now free up the
                  // resources used for downloading, & update the card's state.
                  // -----------------------------------------------------------
                  if (getCardState(cardIndex) == K1212_STATE_DSP_IN_PROCESS) {
                     OnDSPDownloadComplete(cardIndex);
                  }
                  break;

               default:
                  // -----------------------------------------------------------
                  // If the card is currently in play mode, then this must be a
                  // request for data.  The MSB of the doorbell value is used
                  // to indicate errors, so this is also passed to the handling
                  // routine.
                  // -----------------------------------------------------------
                  if ((getCardState(cardIndex)         > K1212_STATE_SETUP) ||
                      (getIdleMonitorState(cardIndex) == TRUE)) {
                     OnRequestForData(cardIndex,
                                      doorbellValue
                     );
                  }
                  break;
            }

            VPICD_Phys_EOI(hIRQ);                   // allow interrupts at this level again
            return TRUE;                            // let VPICD know we processed the interrupt
         }
      }
   }

   // --------------------------------------------------------------------------
   // if we got here, we did not find a card with this interrupt handle, that
   // has interrupted us.  return FALSE to let VPICD know we didn't process the
   // interrupt.
   // --------------------------------------------------------------------------
   return FALSE;
}




// ----------------------------------------------------------------------------
// hookCardIRQ
//
//    This function hooks the card's IRQ.  Since we now support multiple cards,
//    and the interrupts can be shared, we check the existing cards to see if
//    the requested interrupt level has already been hooked.  If so, the
//    requesting card is marked as having the same handle as the first card
//    to have requested this interrupt level.
//
//    Returns: CR_SUCCESS if all goes well, or a CONFIGRET error code.
// ----------------------------------------------------------------------------
CONFIGRET hookCardIRQ(DWORD cardIndex,
                      int   irqNum)
{
   VID   irqDesc;
   HIRQ  irqHandle;
   int   index;

   // ----------------------------------------------------------------------
   // first check and see if we've already hooked this level.  If so, then
   // this card will use the same handle.
   // ----------------------------------------------------------------------
   for (index = 0; index < k1212MaxCards; index++) {
      if ((getIrqNum(index) == irqNum) &&                // if same IRQ
          (getIrqHandle(index) != 0)) {                  // and this one already assigned
         setIrqHandle(cardIndex,
                      getIrqHandle(index)
         );
         return CR_SUCCESS;
      }
   }

   // ----------------------------------------------------------------------
   // fill in the IRQ descriptor and then call the VPICD service to install
   // our handler.  We should allow sharing for PCI interrupts.
   // ----------------------------------------------------------------------
   irqDesc.VID_IRQ_Number       = irqNum;                // IRQ level to intercept
   irqDesc.VID_Options          = VPICD_OPT_CAN_SHARE;   // allow sharing

   irqDesc.VID_Hw_Int_Proc      = (DWORD)VPICD_Thunk_HWInt(k1212IrqHandler,
                                                           &k1212Int_Thunk
                                         );
   irqDesc.VID_Virt_Int_Proc    = 0;                     // we only handle H/W interrupts
   irqDesc.VID_EOI_Proc         = 0;                     //  default EOI handling
   irqDesc.VID_Mask_Change_Proc = 0;                     //  mask will not change
   irqDesc.VID_IRET_Proc        = 0;                     //  default IRET handling
   irqDesc.VID_IRET_Time_Out    = 500;                   // 1/2 second timeout
   irqDesc.VID_Hw_Int_Ref       = 0;                     // no reference data to the handler

   irqHandle = VPICD_Virtualize_IRQ(&irqDesc);           // install our handler
   setIrqHandle(cardIndex,
                irqHandle
   );                                                    // store the assigned handle

   #ifdef DEBUG
      Debug_Printf_Service("Korg 1212 Card %d: IRQ handler assigned handle 0x%x\n",
                           cardIndex,
                           getIrqHandle(cardIndex)
      );
   #endif // DEBUG

   // ----------------------------------------------------------------------
   // Make sure the IRQ is unmasked on the PIC. Otherwise, the interrupt
   // will never occur.
   // ----------------------------------------------------------------------
   VPICD_Physically_Unmask(irqHandle);

   return CR_SUCCESS;
}



// -------------------------------------------------------------------------------------
// This function gets called when the card has requested more data.  The registered
// fill routine is ultimately called via the following process.
// -------------------------------------------------------------------------------------
#define K1212_ISRCODE_DMAERROR      0x00000080
#define K1212_ISRCODE_CARDSTOPPED   0x00000081

void OnRequestForData(DWORD cardIndex,
                      DWORD doorbellValue)
{
   // ------------------------------------------------------------------------------
   // First determine the reason for this interrupt.
   // ------------------------------------------------------------------------------
   switch (doorbellValue) {
      case K1212_ISRCODE_DMAERROR:
         // ------------------------------------------------------------------------
         // an error occurred - stop the card
         // ------------------------------------------------------------------------
         setCardCommand(cardIndex, 0);
         Schedule_Global_Event(StopCardOnErrorRoutine,
                               (PVOID)cardIndex,
                               &k1212StopCard_Event_Thunk
         );
         break;

      case K1212_ISRCODE_CARDSTOPPED:
         // ------------------------------------------------------------------------
         // the card has stopped by our request.  Clear the command word and signal
         // the semaphore in case someone is waiting for this.
         // ------------------------------------------------------------------------
         setCardCommand(cardIndex, 0);
         break;

      default:
         // ------------------------------------------------------------------------
         // the card needs more data.  Set a global event.  The global event causes
         // a callback function to execute.  The callback function makes sure the
         // fill routine will be executed in a timely fashion.
         // ------------------------------------------------------------------------
         Schedule_Global_Event(ScheduleFillRoutine,
                               (PVOID)cardIndex,
                               &k1212Glbl_Event_Thunk
         );
         break;
  }
}


// -------------------------------------------------------------------------------------
// This function gets called when the card has encountered an error and must be stopped.
// -------------------------------------------------------------------------------------
void __stdcall StopCardOnErrorRoutine(VMHANDLE       hVM,
                                      PVOID          refdata,
                                      PCLIENT_STRUCT pRegs)
{
   DWORD cardIndex;
   DWORD deviceId;

   cardIndex = (DWORD)refdata;

   // --------------------------------------------------------------------------------
   // let the user know what happened.  (v1.0B5)
   // --------------------------------------------------------------------------------
   ShowErrorBox();

   // --------------------------------------------------------------------------------
   // If the card is in the native API mode, then simply call the stop card function.
   // Otherwise, we have to reset every wave device.
   // --------------------------------------------------------------------------------
   setCardState(cardIndex, K1212_STATE_ERRORSTOP);
   if (getApiMode(cardIndex) != K1212_NATIVE_API_MODE) {
      for (deviceId = 0; deviceId < k1212NumWaveDevices; deviceId++) {
         ResetWaveInDevice ((WORD)cardIndex, (WORD)deviceId);
         ResetWaveOutDevice((WORD)cardIndex, (WORD)deviceId);
      }
   }
//   setCardState(cardIndex, K1212_STATE_OPEN);
   setCardCommand(cardIndex, 0);
}


// -------------------------------------------------------------------------------------
// This function gets called when the card has requested more data.  The registered
// fill routine is ultimately called via the following process.
// -------------------------------------------------------------------------------------
void __stdcall ScheduleFillRoutine(VMHANDLE       hVM,
                                   PVOID          refdata,
                                   PCLIENT_STRUCT pRegs)
{
   DWORD cardIndex;
   cardIndex = (DWORD)refdata;

   // ---------------------------------------------------------------------------
   // now that a fill routine will be called, update the current fill buffer #.
   // v1.11 - added separate record buffer number.
   // ---------------------------------------------------------------------------
   incCurFillBufNum(cardIndex);
   incCurRecBufNum(cardIndex);

   // ---------------------------------------------------------------------------
   // If the card is in the native API mode, then queue an APC to the ring 3
   // fill routine.  Otherwise, call the ring 0 wave device fill routine.
   // ---------------------------------------------------------------------------
   if (getApiMode(cardIndex) == K1212_NATIVE_API_MODE) {

      // ------------------------------------------------------------------------
      // first, queue the APC, then boost the thread's priority so that it will
      // hopefully run real soon.
      // ------------------------------------------------------------------------
      if (getFillFunc(cardIndex)) {
         _VWIN32_QueueUserApc(getFillFunc(cardIndex),
                              getFillParam(cardIndex),
                              getFillThreadHdl(cardIndex)
         );
         Adjust_Thread_Exec_Priority(getFillThreadHdl(cardIndex),
                                     HIGH_PRI_DEVICE_BOOST
         );
      }
   }
   else {

      // ------------------------------------------------------------------------
      // the card is in wave device driver mode.  Call the wave driver fill
      // function to move data between the wave and the card shared buffers.
      // ------------------------------------------------------------------------
      WaveDeviceFillRoutine((WORD)cardIndex);
   }
}


// -------------------------------------------------------------------------------------
// This function gets called when the DSP download has completed.  It instructs the
// card to boot from page 4.
// -------------------------------------------------------------------------------------
void OnDSPDownloadComplete(DWORD cardIndex)
{
   PVOID  Refdata;
   HANDLE tempHandle;
   WORD   error;
   BYTE   action;

   // ----------------------------------------------------
   // free up memory resources used for the DSP download.
   // ----------------------------------------------------
   if (getDSPMemPtr(cardIndex)) {

      #ifdef DEBUG
         Debug_Printf_Service("Korg 1212 Card %d: DSP microcode successfully downloaded\n",
                              cardIndex
         );
      #endif // DEBUG

      _PageFree(getDSPMemHdl(cardIndex), 0);
      setDSPMemHdl(cardIndex, 0);
      setDSPMemPtr(cardIndex, 0);
   }

   // ----------------------------------------------------
   // tell the card to boot
   // ----------------------------------------------------
   setCardState(cardIndex, K1212_STATE_DSP_COMPLETE);
   Send1212Command(cardIndex,
                   K1212_DB_BootFromDSPPage4,
                   DUMMY_PARAMETER,
                   DUMMY_PARAMETER,
                   DUMMY_PARAMETER,
                   DUMMY_PARAMETER
   );

   // ----------------------------------------------------
   // give the card enough time to finish booting, and
   // then let it know where the buffers are.  We set up
   // the reference data for the timeout handler to
   // contain the cardIndex.
   // ----------------------------------------------------
   Set_Async_Time_Out(K1212DSP_BOOT_DELAY_IN_MS,
                      (PVOID)cardIndex,
                      OnDSPBootComplete,
                      &k1212DSPBoot_Timeout_Thunk
   );

   // ----------------------------------------------------------
   // create a file for posting messages to.  (v1.10) - removed
   // ----------------------------------------------------------
   #ifdef DEBUG_MSG
      tempHandle = R0_OpenCreateFile(FALSE,
                                     "C:\\MESSAGES",
                                     (OPEN_ACCESS_READWRITE | OPEN_SHARE_COMPATIBLE),
                                     ATTR_NORMAL,
                                     (ACTION_IFEXISTS_TRUNCATE | ACTION_IFNOTEXISTS_CREATE),
                                     R0_NO_CACHE,
                                     &error,
                                     &action
                   );
      setFileHandle(cardIndex, tempHandle);
   #endif // DEBUG_MSG
}

// -------------------------------------------------------------------------------------
// This function gets called when the card has successfully booted from the DSP
// microcode.  It lets the card know where all of its buffers are.
// -------------------------------------------------------------------------------------
void __stdcall OnDSPBootComplete(PVOID refdata, DWORD Extra)
{
   DWORD cardIndex;

   cardIndex = (DWORD)refdata;

   // --------------------------------------------------------------------------------
   // Let the card know where all the buffers are.
   // --------------------------------------------------------------------------------
   Send1212Command(cardIndex,
                   K1212_DB_ConfigureBufferMemory,
                   LowerWordSwap((DWORD)getPlayDataPhy(cardIndex)),
                   LowerWordSwap((DWORD)getRecDataPhy(cardIndex)),
                   ((kNumBuffers * kPlayBufferFrames) / 2),   // size given to the card
                                                              // is based on 2 buffers
                   DUMMY_PARAMETER
   );

   WaitRTCTicks(INTERCOMMAND_DELAY);

   Send1212Command(cardIndex,
                   K1212_DB_ConfigureMiscMemory,
                   LowerWordSwap((DWORD)getVolumeTablePhy(cardIndex)),
                   LowerWordSwap((DWORD)getRoutingTablePhy(cardIndex)),
                   LowerWordSwap((DWORD)getAdatTimeCodePhy(cardIndex)),
                   DUMMY_PARAMETER
   );

   // ----------------------------------------------------
   // we need to read the registry, so do it at a time
   // that is guaranteed to be safe?
   // ----------------------------------------------------
   SHELL_CallAtAppyTime(ProcessDSPBootComplete,
                        (PVOID)refdata,
                        CAAFL_RING0,
                        0
   );
}

void _cdecl ProcessDSPBootComplete(PVOID RefData, DWORD flags)
{
   DWORD cardIndex;

   cardIndex = (DWORD)RefData;

   // --------------------------------------------------------------------------------
   // Initialize the routing and volume tables, then update the card's state.
   // --------------------------------------------------------------------------------
   WaitRTCTicks(INTERCOMMAND_DELAY);
   InitSettings(cardIndex);
   WaitRTCTicks(INTERCOMMAND_DELAY);
   Send1212Command(cardIndex,
                   K1212_DB_SetClockSourceRate,
                   ClockSourceSelector[getClkSrcRate(cardIndex)],
                   DUMMY_PARAMETER,
                   DUMMY_PARAMETER,
                   DUMMY_PARAMETER
   );
   setCardState(cardIndex, K1212_STATE_READY);
}



// -------------------------------------------------------------------------------------
// This function parses the device ID string in order to obtain the bus and device
// numbers of the card.  The device ID string is assumed to have the device number
// immediately following the bus number.
// -------------------------------------------------------------------------------------
CONFIGRET ParseDevIdString(char*  devIdBuffer,
                           DWORD* busNum,
                           DWORD* deviceNum)
{
   char* busStr;
   char* devStr;
   char  strNum[3];     // used for converting two character hex numbers to integers.

   // -----------------------------------------------------------------------------
   // get the bus number first
   // -----------------------------------------------------------------------------
   busStr =  strstr(devIdBuffer, "BUS_");
   busStr += 4;                              // scoot past to the actual digits
   strncpy(strNum, busStr, 2);               // there should be only two digits in this field
   strNum[2] = '\0';                         // null terminate before converting
   if ((*busNum = HexStrToInt(strNum)) == (DWORD)-1) {
      return CR_INVALID_DEVICE_ID;
   }

   // -----------------------------------------------------------------------------
   // now get the device number.  busStr points to the substring of the device ID
   // string that should contain the device number string.
   // -----------------------------------------------------------------------------
   devStr =  strstr(busStr, "DEV_");
   devStr += 4;                              // scoot past to the actual digits
   strncpy(strNum, devStr, 2);               // there should be only two digits in this field
   strNum[2] = '\0';                         // null terminate before converting
   if ((*deviceNum = HexStrToInt(strNum)) == (DWORD)-1) {
      return CR_INVALID_DEVICE_ID;
   }

   return CR_SUCCESS;
}


// -------------------------------------------------------------------------------------
// This function converts the passed in string, containing hexadecimal characters, to
// its decimal value.  It assumes that the string contains only valid hex characters.
// If it does not, then -1 is returned.
// -------------------------------------------------------------------------------------
DWORD HexStrToInt(char* hexStr)
{
   int  place;
   int  length;
   char c;
   int  value;

   place  = 1;                // start in one's place, then 16's, then 256's, etc...
   value  = 0;                // keeps a running total
   length = strlen(hexStr);

   for (length--; length >= 0; length--, place *= 16) {
      c = hexStr[length];
      if ((c >= '0') && (c <= '9')) {
         c = c - '0';
      }
      else if ((c >= 'A') && (c <= 'F')) {
         c = c - 'A';
         c += 10;
      }
      else if ((c >= 'a') && (c <= 'f')) {
         c = c - 'a';
         c += 10;
      }
      else {
         return (DWORD)-1;
      }
      value += (c * place);
   }
   return value;
}

// -------------------------------------------------------------------------------------
// This function converts the passed in decimal value (ranging from 0x0 - 0xFF), to its
// equivalent string characters.  If the value is out of range, then FALSE is returned.
// twoHexChars must point to a three character array (zero terminated string).
// -------------------------------------------------------------------------------------
void charTo2HexChars(char* lpszTwoHexChars, char decValue)
{
   DWORD firstDigitValue;

   lpszTwoHexChars[2] = '\0';

   firstDigitValue = (decValue >> 4);
   if (firstDigitValue > 0x9) {
      lpszTwoHexChars[0] = ((firstDigitValue - 0xA) + 'A');
   }
   else {
      lpszTwoHexChars[0] = firstDigitValue + '0';
   }

   decValue &= 0x0F;
   if (decValue > 0x9) {
      lpszTwoHexChars[1] = ((decValue - 0xA) + 'A');
   }
   else {
      lpszTwoHexChars[1] = decValue + '0';
   }
}


// -------------------------------------------------------------------------------------
// InitSettings
//
// This function initializes the volume and routing tables.  It also initializes the
// ADC sensitivity values and has them written to the card.
// -------------------------------------------------------------------------------------
void  InitSettings(DWORD cardIndex)
{
   WORD  channel;
   WORD* volTable;
   WORD* routeTable;
   WORD  deviceId;
   k1212WaveInDevice*  waveInDevPtr;
   k1212WaveOutDevice* waveOutDevPtr;

   // ------------------------------------------------------------------------------
   // first set the default values.
   // ------------------------------------------------------------------------------
   volTable   = getVolumeTable(cardIndex);
   routeTable = getRoutingTable(cardIndex);

   for (channel = 0; channel < kAudioChannels; channel++) {
      volTable[channel]   = k1212MaxVolume;
      routeTable[channel] = channel;
   }

   // ------------------------------------------------------------------------------
   // retrieve saved settings from the registry, if any exist.  Then write the
   // input sensitivity values to the card.
   // ------------------------------------------------------------------------------
   if (RetrieveCardSettings(cardIndex) == FALSE) {
      WriteADCSensitivity(cardIndex);
      SaveCardSettings(cardIndex);        // try to create an entry if one doesn't exist
   }

   // ------------------------------------------------------------------------------
   // retrieve saved wave device settings from the registry, if any exist.
   // Now, to avoid problems of the card lockout variety, we should clear out
   // the sync boxes for now.  (v1.0B6)
   // ------------------------------------------------------------------------------
   RetrieveWaveSettings(cardIndex);
   for (deviceId = 0; deviceId < k1212NumWaveDevices; deviceId++) {
      waveOutDevPtr = getWaveOutDevPtr(cardIndex,
                                       deviceId
                      );
      if (waveOutDevPtr != 0) {
         waveOutDevPtr->chainLinkSync = FALSE;
      }
      waveInDevPtr = getWaveInDevPtr(cardIndex,
                                     deviceId
                     );
      if (waveInDevPtr != 0) {
         waveInDevPtr->chainLinkSync = FALSE;
      }
   }

}


// -------------------------------------------------------------------------------------
// SaveCardSettings
//
// This function saves the card's clock source/rate, volume, routing, and sensitivity
// settings to the system registry.  The bus and device numbers are used to identify
// the card for future retrieval.
// -------------------------------------------------------------------------------------
void  SaveCardSettings(DWORD cardIndex)
{
   k1212RegistryData registryData;

   if (PrepareRegistryDataStruct(cardIndex, &registryData) == TRUE) {
      if (!WriteRegistryValues(cardIndex, &registryData)) {
         #ifdef DEBUG
            Debug_Printf_Service("Korg 1212 Card %d: SaveCardSettings failed\n",
                                 cardIndex
            );
         #endif // DEBUG
      }
   }
}


// -------------------------------------------------------------------------------------
// RetrieveCardSettings
//
// This function retrieves the card's clock source/rate, volume, routing, and sensitivity
// settings from the system registry.  If a registry entry is not found, this function
// does nothing.
// -------------------------------------------------------------------------------------
BOOL  RetrieveCardSettings(DWORD cardIndex)
{
   k1212RegistryData registryData;

   if (ReadRegistryValues(cardIndex, &registryData) == TRUE) {
      if (!RestoreRegistryValues(cardIndex, &registryData)) {
         #ifdef DEBUG
            Debug_Printf_Service("Korg 1212 Card %d: RetrieveCardSettings failed\n",
                                 cardIndex
            );
         #endif // DEBUG
         return FALSE;
      }
      else {
         return TRUE;
      }
   }

   return FALSE;
}


// -------------------------------------------------------------------------------------
// SaveWaveSettings
//
// This function saves the card's wave device settings to the system registry.  The
// bus and device numbers are used to identify the card for future retrieval.
// -------------------------------------------------------------------------------------
void  SaveWaveSettings(DWORD cardIndex)
{
   k1212RegistryWaveData registryData;

   if (PrepareRegistryWaveDataStruct(cardIndex, &registryData) == TRUE) {
      if (!WriteRegistryWaveValues(cardIndex, &registryData)) {
         #ifdef DEBUG
            Debug_Printf_Service("Korg 1212 Card %d: SaveWaveSettings failed\n",
                                 cardIndex
            );
         #endif // DEBUG
      }
   }
}


// -------------------------------------------------------------------------------------
// RetrieveWaveSettings
//
// This function retrieves the card's wave device settings from the system registry.
// If a registry entry is not found, this function does nothing.
// -------------------------------------------------------------------------------------
BOOL  RetrieveWaveSettings(DWORD cardIndex)
{
   k1212RegistryWaveData registryData;

   if (ReadRegistryWaveValues(cardIndex, &registryData) == TRUE) {
      if (!RestoreRegistryWaveValues(cardIndex, &registryData)) {
         #ifdef DEBUG
            Debug_Printf_Service("Korg 1212 Card %d: RetrieveWaveSettings failed\n",
                                 cardIndex
            );
         #endif // DEBUG
         return FALSE;
      }
      else {
         return TRUE;
      }
   }

   return FALSE;
}



// -------------------------------------------------------------------------------------
// ShowErrorBox
// ErrorBoxCallback
//
// These functions execute our dialog program to inform the user that an error occurred.
// -------------------------------------------------------------------------------------
void  ShowErrorBox(void)
{
   SHELL_CallAtAppyTime(ErrorBoxCallback,
                        (PVOID)k1212_DMA_ERROR,
                        CAAFL_RING0,
                        0
   );
}

#define K1212_ERRBOX_MAXFILENAMESIZE 13      // 12 plus null termination
#define K1212_ERRBOX_MAXPARAMSTRSIZE 16

void _cdecl ErrorBoxCallback(PVOID RefData, DWORD flags)
{
   struct _shex_ex
   {
      SHEXPACKET shex;
      char       filename[K1212_ERRBOX_MAXFILENAMESIZE];
      char       cmdParams[K1212_ERRBOX_MAXPARAMSTRSIZE];
   } my_shex;

   strcpy(my_shex.filename, "K1212ERR.EXE");
   strcpy(my_shex.cmdParams, " /1");            // specifies PCI error - fixed for now, could
                                                //    be variable if more dialogs are defined.

   my_shex.shex.dwTotalSize = (sizeof(SHEXPACKET) +
                               K1212_ERRBOX_MAXFILENAMESIZE +
                               strlen(my_shex.cmdParams));
   my_shex.shex.dwSize      = sizeof(SHEXPACKET);
   my_shex.shex.ibOp        = 0;
   my_shex.shex.ibFile      = sizeof(SHEXPACKET);
   my_shex.shex.ibParams    = (sizeof(SHEXPACKET) + K1212_ERRBOX_MAXFILENAMESIZE);
   my_shex.shex.ibDir       = 0;
   my_shex.shex.dwReserved  = 0;
   my_shex.shex.nCmdShow    = 1;            // SW_SHOWNORMAL
   SHELL_ShellExecute(&(my_shex.shex));
}


// -------------------------------------------------------------------------------------
// WaitForCardStopAck
//
// This function waits for up to 2 seconds for the card to acknowledge a stop request
// in the card command register.  If the card does not acknowledge within this time,
// we clear the command register and return.
// -------------------------------------------------------------------------------------
void WaitForCardStopAck(DWORD cardIndex)
{
   DWORD  startTime;

   startTime = Get_System_Time();

   do {
      if (getCardCommand(cardIndex) == 0) {
         return;
      }
   } while ((startTime + 2000) > Get_System_Time());

   setCardCommand(cardIndex, 0);
}
