//*****************************************************************************
//* Name:
//*      1212intf.c
//*
//* Project:
//*      1212 I/O VxD
//*
//* Author:
//*      Bill Jenkins
//*
//* Description:
//*      This file contains the definition of the driver to card interface 
//*      functions.
//*
//* Modification Log:
//*
//*      1.4   07/08/97 Bill
//*      addToCmdRetryCount() calls added when the card has never acknowledged
//*      a command.  (1.0B4)
//*
//*      1.3   06/30/97 Bill
//*      Added DEBUG_CARD_MSG option for logging messages sent to the card
//*      in the debug file.  
//*
//*      1.2   12/30/96 Bill
//*      Modified to support multiple cards.  
//*
//*      1.1   11/15/96 Bill
//*      Initial version created.  
//*
//*
//* Copyright (c) 1996 Korg, Inc.
//* All rights reserved.
//*
//* This program is protected as an unpublished work under the U.S.
//* copyright laws.  The above copyright notice is not intended to
//* effect a publication of this work.
//*
//* This program is the confidential and proprietary information of
//* Korg and all its subsidiaries.
//*****************************************************************************

#include <vtoolsc.h>
#include <vmm.h>
#include <debug.h>
#include <vmmreg.h>

#ifndef  K1212CARD_H
#include "1212card.h"
#endif
#ifndef  K1212INTF_H
#include "1212intf.h"
#endif
#ifndef  K1212IFPR_H
#include "1212ifpr.h"
#endif
#ifdef DEBUG_CARD_MSG
   #ifndef  K1212WAVE_H
   #include "1212wave.h"
   #endif
#endif // DEBUG_CARD_MSG



// --------------------------------------------------------------------------
// private functions for reading/writing the card interface registers
// --------------------------------------------------------------------------
void     writeStatusReg(DWORD cardIndex, DWORD value)
{
   *(statusRegAddress(cardIndex)) = value;
}

DWORD    readStatusReg(DWORD cardIndex)
{
   return *(statusRegAddress(cardIndex));
}

void     writeOutDoorbell(DWORD cardIndex, DWORD value)
{
   *(outDoorbellAddress(cardIndex)) = value;
}

DWORD    readOutDoorbell(DWORD cardIndex)
{
   return *(outDoorbellAddress(cardIndex));
}

void     writeInDoorbell(DWORD cardIndex, DWORD value)
{
   *(inDoorbellAddress(cardIndex)) = value;
}

DWORD    readInDoorbell(DWORD cardIndex)
{
   return *(inDoorbellAddress(cardIndex));
}

void     writeMailbox0(DWORD cardIndex, DWORD value)
{
   *(mailbox0Address(cardIndex)) = value;
}

DWORD    readMailbox0(DWORD cardIndex)
{
   return *(mailbox0Address(cardIndex));
}

void     writeMailbox1(DWORD cardIndex, DWORD value)
{
   *(mailbox1Address(cardIndex)) = value;
}

DWORD    readMailbox1(DWORD cardIndex)
{
   return *(mailbox1Address(cardIndex));
}

void     writeMailbox2(DWORD cardIndex, DWORD value)
{
   *(mailbox2Address(cardIndex)) = value;
}

DWORD    readMailbox2(DWORD cardIndex)
{
   return *(mailbox2Address(cardIndex));
}

void     writeMailbox3(DWORD cardIndex, DWORD value)
{
   *(mailbox3Address(cardIndex)) = value;
}

DWORD    readMailbox3(DWORD cardIndex)
{
   return *(mailbox3Address(cardIndex));
}

DWORD    readControlReg(DWORD cardIndex)
{
   return *(controlRegAddress(cardIndex));
}

void     writeSensReg(DWORD cardIndex, WORD value)
{
   *(sensRegAddress(cardIndex)) = value;   
}

DWORD    readIdReg(DWORD cardIndex)
{
   return *(idRegAddress(cardIndex));
}



// --------------------------------------------------------------------------
// SetupCardInterface
//
//    This function is called after the card configuration has been read in
//    from the configuration manager.  It sets up the interface structure
//    based on the configuration data so that the card interface function
//    can correctly communicate with the card.  A flag is used to indicate
//    whether this setup has been performed, so that the interface functions
//    can fail and return before trying to write based on uninitialized
//    pointers.  This flag is implemented with the doorbellPtr member of the
//    cardIntfRegs structure.  This pointer is initialized to zero during
//    SysDynDevInit and then set to its usable value here.
// --------------------------------------------------------------------------
void  SetupCardInterface(DWORD cardIndex)
{
   char* memBase = getMemBase(cardIndex);   

   setStatusRegAddress  (cardIndex, (DWORD*)(memBase + STATUS_REG_OFFSET));
   setOutDoorbellAddress(cardIndex, (DWORD*)(memBase + OUT_DOORBELL_OFFSET));
   setInDoorbellAddress (cardIndex, (DWORD*)(memBase + IN_DOORBELL_OFFSET));
   setMailbox0Address   (cardIndex, (DWORD*)(memBase + MAILBOX0_OFFSET));
   setMailbox1Address   (cardIndex, (DWORD*)(memBase + MAILBOX1_OFFSET));
   setMailbox2Address   (cardIndex, (DWORD*)(memBase + MAILBOX2_OFFSET));
   setMailbox3Address   (cardIndex, (DWORD*)(memBase + MAILBOX3_OFFSET));
   setControlRegAddress (cardIndex, (DWORD*)(memBase + PCI_CONTROL_OFFSET));
   setSensRegAddress    (cardIndex, (WORD*) (memBase + SENS_CONTROL_OFFSET));
   setIdRegAddress      (cardIndex, (DWORD*)(memBase + DEV_VEND_ID_OFFSET));

   #ifdef DEBUG
      Debug_Printf_Service("Korg 1212 Card %d: card registers assigned as...\n",
                           cardIndex
      );
      Debug_Printf_Service("     Status register = 0x%x\n", statusRegAddress(cardIndex));
      Debug_Printf_Service("     OutDoorbell     = 0x%x\n", outDoorbellAddress(cardIndex));
      Debug_Printf_Service("     InDoorbell      = 0x%x\n", inDoorbellAddress(cardIndex));
      Debug_Printf_Service("     Mailbox0        = 0x%x\n", mailbox0Address(cardIndex));
      Debug_Printf_Service("     Mailbox1        = 0x%x\n", mailbox1Address(cardIndex));
      Debug_Printf_Service("     Mailbox2        = 0x%x\n", mailbox2Address(cardIndex));
      Debug_Printf_Service("     Mailbox3        = 0x%x\n", mailbox3Address(cardIndex));

   #endif // DEBUG
}

WORD ByteSwap(WORD swappee)
{
   unsigned char retVal[2];
   unsigned char swapper[2];
   
   *(WORD*)swapper = swappee;
   retVal[1] = swapper[0];
   retVal[0] = swapper[1];
   
   return *(DWORD*)retVal;
}

DWORD UpperWordSwap(DWORD swappee)
{
   unsigned char retVal[4];
   unsigned char swapper[4];
   
   *(DWORD*)swapper = swappee;
   retVal[2] = swapper[3];
   retVal[3] = swapper[2];
   retVal[1] = swapper[1];
   retVal[0] = swapper[0];
   
   return *(DWORD*)retVal;
}

DWORD LowerWordSwap(DWORD swappee)
{
   unsigned char retVal[4];
   unsigned char swapper[4];
   
   *(DWORD*)swapper = swappee;
   retVal[2] = swapper[2];
   retVal[3] = swapper[3];
   retVal[1] = swapper[0];
   retVal[0] = swapper[1];
   
   return *(DWORD*)retVal;
}

DWORD EndianSwap(DWORD swappee)
{
   unsigned char retVal[4];
   unsigned char swapper[4];
   
   *(DWORD*)swapper = swappee;
   retVal[0] = swapper[3];
   retVal[1] = swapper[2];
   retVal[2] = swapper[1];
   retVal[3] = swapper[0];
   
   return *(DWORD*)retVal;
}

// ---------------------------------------------------------------------------
// Send1212Command
//
//    This function sends a command to the card by writing the passed mailbox
//    parameters (which contain command specific information) to the card's
//    mailbox, and then notifying the card of the request by writing the
//    command value to it's doorbell.  The current implementation is SLOW
//    because it uses function calls to write to the card's registers rather
//    than writing directly.  Since writing to the card is an infrequent
//    event, this shouldn't be a problem.
//
//
//    Returns: True or False depending on whether the register addresses
//             have been assigned.
// ---------------------------------------------------------------------------
K1212CmdRet Send1212Command(DWORD            cardIndex,
                            DoorbellConstant doorbellVal,   // doorbell value
                            DWORD            mailBox0Val,   // value for mailbox 0
                            DWORD            mailBox1Val,   // value for mailbox 1
                            DWORD            mailBox2Val,   // value for mailbox 2
                            DWORD            mailBox3Val    // value for mailbox 3
            )
{
   #ifdef DEBUG_CARD_MSG
      k1212PMIn reqInputBuf;
   #endif // DEBUG_CARD_MSG

   DWORD retryCount;
   WORD  mailBox3Lo;
      
   if (outDoorbellAddress(cardIndex)) {

      for (retryCount = 0; retryCount < MAX_COMMAND_RETRIES; retryCount++) {

         writeMailbox3   (cardIndex, mailBox3Val);   
         writeMailbox2   (cardIndex, mailBox2Val);
         writeMailbox1   (cardIndex, mailBox1Val);
         writeMailbox0   (cardIndex, mailBox0Val);
         writeOutDoorbell(cardIndex, doorbellVal);  // interrupt the card

         // --------------------------------------------------------------
         // the reboot command will not give an acknowledgement.
         // --------------------------------------------------------------
         switch (doorbellVal) {
            case K1212_DB_RebootCard:
            case K1212_DB_BootFromDSPPage4:
            case K1212_DB_StartDSPDownload:
               return K1212_CMDRET_Success;
         }
         
         // --------------------------------------------------------------
         // See if the card acknowledged the command.  Wait a bit, then
         // read in the low word of mailbox3.  If the MSB is set and the
         // low byte is equal to the doorbell value, then it ack'd.
         // --------------------------------------------------------------
         WaitRTCTicks(COMMAND_ACK_DELAY);
         mailBox3Lo = (WORD)readMailbox3(cardIndex);
         if (mailBox3Lo & COMMAND_ACK_MASK) {
            if ((mailBox3Lo & DOORBELL_VAL_MASK) == (doorbellVal & DOORBELL_VAL_MASK)) {
               addToCmdRetryCount(cardIndex, retryCount);

               #ifdef DEBUG_CARD_MSG
                  reqInputBuf.cardIndex = cardIndex;
                  sprintf(reqInputBuf.string, "Card <- %08x %08x\n", 
                          doorbellVal, 
                          mailBox0Val
                  );
                  K1212PostMessageToFile(&reqInputBuf, NULL);
               #endif // DEBUG_CARD_MSG

               return K1212_CMDRET_Success;
            }
         }
      }
      addToCmdRetryCount(cardIndex, retryCount);
      return K1212_CMDRET_NoAckFromCard;
   }
   else {
      return K1212_CMDRET_CardUninitialized;
   }
}


