//*****************************************************************************
//* Name:
//*      1212registry.c - System registry access module for VxD KORG1212
//*
//* Project:
//*      1212 I/O VxD
//*
//* Author:
//*      Bill Jenkins
//*
//* Description:
//*      This file implements the system registry access functions.
//*
//*
//* Modification Log:
//*
//*      1.3   09/28/97 Bill
//*      (v1.0B8) Added additional statements to close an opened key if
//*      an error condition is encountered.
//*
//*      1.2   08/20/97 Bill
//*      (v1.0B6) Removed assanine check for a null wave device pointer
//*      that then went ahead and tried to dereference it (DOH!!!).  Added
//*      auto sync setting.
//*
//*      1.1   02/14/97 Bill
//*      Initial version created.  
//*
//*
//* Copyright (c) 1997 Korg, Inc.
//* All rights reserved.
//*
//* This program is protected as an unpublished work under the U.S.
//* copyright laws.  The above copyright notice is not intended to
//* effect a publication of this work.
//*
//* This program is the confidential and proprietary information of
//* Korg and all its subsidiaries.
//*****************************************************************************

#include <vtoolsc.h>
#include <vmm.h>
#include <debug.h>
#include <vmmreg.h>

#ifndef  K1212CFG_H
#include "1212cfg.h"
#endif
#ifndef  K1212CARD_H
#include "1212card.h"
#endif
#ifndef  K1212REGISTRY_H
#include "1212registry.h"
#endif
#ifndef  K1212WAVE_H
#include "1212wave.h"
#endif
#ifndef  K1212INTF_H
#include "1212intf.h"
#endif


extern WORD ClockSourceSelector[];     // defined in korg1212.c


// ------------------------------------------------------------------------
// PrepareRegistryDataStruct
//
//    This function, given a card index, sets up the k1212RegistryData 
//    structure based on the card's current values.  This call should 
//    be made prior to writing the data to the system registry.  This
//    call returns TRUE if the data was successfully copied.
// ------------------------------------------------------------------------
BOOL PrepareRegistryDataStruct(DWORD              cardIndex,
                               k1212RegistryData* registryData)
{
   DWORD                channelCount;
   WORD*                rteTable;
   short*               volTable;

   if (registryData == NULL) {
      return FALSE;
   }
   
   // -------------------------------------------------------------------
   // copy over volume and routing table settings
   // -------------------------------------------------------------------
   rteTable = getRoutingTable(cardIndex);
   volTable = getVolumeTable(cardIndex);
   if ((volTable == NULL) || (rteTable == NULL)) {
      return FALSE;
   }
   
   for (channelCount = 0; channelCount < kAudioChannels; channelCount++) {
      registryData->routing[channelCount] = rteTable[channelCount];
      registryData->volume[channelCount]  = volTable[channelCount];
   }

   // -------------------------------------------------------------------
   // copy over input sensitivity and clock source rate settings
   // -------------------------------------------------------------------
   registryData->leftADCSens  = getLeftADCInSens(cardIndex);
   registryData->rightADCSens = getRightADCInSens(cardIndex);
   registryData->clkSrcRate   = getClkSrcRate(cardIndex);
   return TRUE;
}


// ------------------------------------------------------------------------
// PrepareRegistryWaveDataStruct
//
//    This function, given a card index, sets up the k1212RegistryWaveData 
//    structure based on the card's current values.  This call should 
//    be made prior to writing the data to the system registry.  This
//    call returns TRUE if the data was successfully copied.
// ------------------------------------------------------------------------
BOOL PrepareRegistryWaveDataStruct(DWORD                  cardIndex,
                                   k1212RegistryWaveData* registryData)
{
   DWORD                deviceId;
   k1212WaveOutDevice*  waveOutDevPtr;
   k1212WaveInDevice*   waveInDevPtr;

   if (registryData == NULL) {
      return FALSE;
   }
   
   // -------------------------------------------------------------------
   // copy over wave device settings
   // -------------------------------------------------------------------
   for (deviceId = 0; deviceId < k1212NumWaveDevices; deviceId++) {
      waveOutDevPtr = getWaveOutDevPtr(cardIndex,
                                       deviceId
                      );
      if (waveOutDevPtr == 0) {  
         // -------------------------------------------------------------
         // just write in defaults if we can't get the device pointer
         // -------------------------------------------------------------
         registryData->leftWaveVolume   [deviceId] = k1212MaxWaveVolume;
         registryData->rightWaveVolume  [deviceId] = k1212MaxWaveVolume;
         registryData->volumeEnableFlags[deviceId] = FALSE;
         registryData->muteFlags        [deviceId] = (WORD)FALSE;
         registryData->playSyncFlags    [deviceId] = FALSE;
      }
      else {
         registryData->leftWaveVolume   [deviceId] = waveOutDevPtr->leftVolumeLevel;
         registryData->rightWaveVolume  [deviceId] = waveOutDevPtr->rightVolumeLevel;
         registryData->volumeEnableFlags[deviceId] = waveOutDevPtr->volumeControlEnabled;
         registryData->muteFlags        [deviceId] = (WORD)waveOutDevPtr->muted;
         registryData->playSyncFlags    [deviceId] = waveOutDevPtr->chainLinkSync;
      }
      waveInDevPtr = getWaveInDevPtr(cardIndex,
                                     deviceId
                     );
      if (waveInDevPtr == 0) {  
         // -------------------------------------------------------------
         // just write in defaults if we can't get the device pointer
         // -------------------------------------------------------------
         registryData->recSyncFlags[deviceId] = FALSE;
      }
      else {
         registryData->recSyncFlags[deviceId] = waveInDevPtr->chainLinkSync;
      }
   }   

   // -------------------------------------------------------------------
   // set the auto sync setting and extra words.
   // -------------------------------------------------------------------
   registryData->autoSync = getAutoWaveSyncMode(cardIndex);
   registryData->extra1   = 0;
   registryData->extra2   = 0;
   registryData->extra3   = 0;
   registryData->extra4   = 0;
   registryData->extra5   = 0;

   return TRUE;
}


// ------------------------------------------------------------------------
// WriteRegistryValues
//
//    This function, given a card index, writes the card's configuration
//    data to the system registry.  If the information cannot be stored, 
//    FALSE is returned.
// ------------------------------------------------------------------------
BOOL WriteRegistryValues(DWORD              cardIndex,
                         k1212RegistryData* registryData)
{
   int   retResult;
   char  keyString[MAX_REGPATH_SIZE];
   char  tempNumStr[3];
   HKEY  keyHandle;

   // ------------------------------------------------------------------------------
   // First, create the subkey name.  The subkey is CardXXYY, where XX is the bus
   // number, and YY is the device number.
   // ------------------------------------------------------------------------------
   strcpy(keyString, getRegistryPath(cardIndex));        // keyString = regPath
   strcat(keyString, "\\Card");                          // keyString = regPath\Card
   charTo2HexChars(tempNumStr, (char)getBusNum(cardIndex));
   strcat(keyString, tempNumStr);                        // keyString = regPath\CardXX
   charTo2HexChars(tempNumStr, (char)getDeviceNum(cardIndex));
   strcat(keyString, tempNumStr);                        // keyString = regPath\CardXXYY
   strcat(keyString, "\\defaults");                      // keyString = regPath\CardXXYY\defaults
   
   #ifdef DEBUG
      Debug_Printf_Service("Korg 1212 Card %d: keyString = %s\n",
                           cardIndex,
                           keyString
      );
   #endif // DEBUG

   // ------------------------------------------------------------------------------
   // try to open the subkey.  If this fails, create the key
   // ------------------------------------------------------------------------------
   retResult = RegOpenKey(HKEY_LOCAL_MACHINE, 
                          keyString, 
                          &keyHandle);

   #ifdef DEBUG
      Debug_Printf_Service("Korg 1212 Card %d: RegOpenKey returned %d\n",
                           cardIndex,
                           retResult
      );
   #endif // DEBUG

   switch (retResult) {
      case ERROR_SUCCESS:
         break;

      case ERROR_FILE_NOT_FOUND:
         if ((retResult = RegCreateKey(HKEY_LOCAL_MACHINE, 
                                       keyString, 
                                       &keyHandle
                          )) != ERROR_SUCCESS) {

            #ifdef DEBUG
               Debug_Printf_Service("Korg 1212 Card %d: RegCreateKey failed - %d\n",
                                    cardIndex,
                                    retResult
               );
            #endif // DEBUG
            return FALSE;
         }
         break;

      default:
         #ifdef DEBUG
            Debug_Printf_Service("Korg 1212 Card %d: WriteReg - RegOpenKey failed - %d\n",
                                 cardIndex,
                                 retResult
            );
         #endif // DEBUG
         return FALSE;
   }

   // ------------------------------------------------------------------------------
   // write the data to the registry
   // ------------------------------------------------------------------------------
   if ((retResult = RegSetValueEx(keyHandle,
                                  K1212_SETTINGS_STRING,
                                  0,
                                  REG_BINARY,
                                  (unsigned char*)registryData,
                                  sizeof(k1212RegistryData)
                    )) != ERROR_SUCCESS) {

      #ifdef DEBUG
         Debug_Printf_Service("Korg 1212 Card %d: RegSetValueEx failed - %d\n",
                              cardIndex,
                              retResult
         );
      #endif // DEBUG
      RegCloseKey(keyHandle);
      return FALSE;
   }
   else {
      #ifdef DEBUG
         Debug_Printf_Service("Korg 1212 Card %d: RegSetValueEx succeeded\n",
                              cardIndex
         );
      #endif // DEBUG
      RegCloseKey(keyHandle);
      return TRUE;
   }
}


// ------------------------------------------------------------------------
// WriteRegistryWaveValues
//
//    This function, given a card index, writes the card's wave device
//    configuration data to the system registry.  If the information 
//    cannot be stored, FALSE is returned.
// ------------------------------------------------------------------------
BOOL WriteRegistryWaveValues(DWORD                  cardIndex,
                             k1212RegistryWaveData* registryData)
{
   int   retResult;
   char  keyString[MAX_REGPATH_SIZE];
   char  tempNumStr[3];
   HKEY  keyHandle;
   DWORD retVal;

   // ------------------------------------------------------------------------------
   // First, create the subkey name.  The subkey is CardXXYY, where XX is the bus
   // number, and YY is the device number.
   // ------------------------------------------------------------------------------
   strcpy(keyString, getRegistryPath(cardIndex));        // keyString = regPath
   strcat(keyString, "\\Card");                          // keyString = regPath\Card
   charTo2HexChars(tempNumStr, (char)getBusNum(cardIndex));
   strcat(keyString, tempNumStr);                        // keyString = regPath\CardXX
   charTo2HexChars(tempNumStr, (char)getDeviceNum(cardIndex));
   strcat(keyString, tempNumStr);                        // keyString = regPath\CardXXYY
   strcat(keyString, "\\waveinfo");                      // keyString = regPath\CardXXYY\waveinfo
   
   #ifdef DEBUG
      Debug_Printf_Service("Korg 1212 Card %d: keyString = %s\n",
                           cardIndex,
                           keyString
      );
   #endif // DEBUG

   // ------------------------------------------------------------------------------
   // try to open the subkey.  If this fails, create the key
   // ------------------------------------------------------------------------------
   retResult = RegOpenKey(HKEY_LOCAL_MACHINE, 
                          keyString, 
                          &keyHandle);

   #ifdef DEBUG
      Debug_Printf_Service("Korg 1212 Card %d: RegOpenKey returned %d\n",
                           cardIndex,
                           retResult
      );
   #endif // DEBUG

   switch (retResult) {
      case ERROR_SUCCESS:
         break;

      case ERROR_FILE_NOT_FOUND:
         if ((retVal = RegCreateKey(HKEY_LOCAL_MACHINE, 
                                    keyString, 
                                    &keyHandle
                       )) != ERROR_SUCCESS) {

            #ifdef DEBUG
               Debug_Printf_Service("Korg 1212 Card %d: RegCreateKey failed - %d\n",
                                    cardIndex,
                                    retVal
               );
            #endif // DEBUG
            return FALSE;
         }
         break;

      default:
         return FALSE;
   }

   // ------------------------------------------------------------------------------
   // write the data to the registry
   // ------------------------------------------------------------------------------
   if ((retVal = RegSetValueEx(keyHandle,
                               K1212_SETTINGS_STRING,
                               0,
                               REG_BINARY,
                               (unsigned char*)registryData,
                               sizeof(k1212RegistryWaveData)
                 )) != ERROR_SUCCESS) {

      #ifdef DEBUG
         Debug_Printf_Service("Korg 1212 Card %d: RegSetValueEx failed - %d\n",
                              cardIndex,
                              retVal
         );
      #endif // DEBUG
      RegCloseKey(keyHandle);
      return FALSE;
   }
   else {
      #ifdef DEBUG
         Debug_Printf_Service("Korg 1212 Card %d: RegSetValueEx succeeded\n",
                              cardIndex
         );
      #endif // DEBUG
      RegCloseKey(keyHandle);
      return TRUE;
   }
}


// ------------------------------------------------------------------------
// ReadRegistryValues
//
//    This function, given a card index, reads any stored configuration 
//    data from the system registry for that card.  If the information
//    cannot be retrieved, FALSE is returned.
// ------------------------------------------------------------------------
BOOL ReadRegistryValues(DWORD              cardIndex,
                        k1212RegistryData* registryData)
{
   char  keyString[MAX_REGPATH_SIZE];
   char  tempNumStr[3];
   HKEY  keyHandle;
   DWORD type, size, retVal;

   // ------------------------------------------------------------------------------
   // First, create the subkey name.  The subkey is CardXXYY, where XX is the bus
   // number, and YY is the device number.
   // ------------------------------------------------------------------------------
   strcpy(keyString, getRegistryPath(cardIndex));        // keyString = regPath
   strcat(keyString, "\\Card");                           // keyString = regPath\Card
   charTo2HexChars(tempNumStr, (char)getBusNum(cardIndex));
   strcat(keyString, tempNumStr);                        // keyString = regPath\CardXX
   charTo2HexChars(tempNumStr, (char)getDeviceNum(cardIndex));
   strcat(keyString, tempNumStr);                        // keyString = regPath\CardXXYY
   strcat(keyString, "\\defaults");                      // keyString = regPath\CardXXYY\defaults

   #ifdef DEBUG
      Debug_Printf_Service("Korg 1212 Card %d: keyString = %s\n",
                           cardIndex,
                           keyString
      );
   #endif // DEBUG

   // ------------------------------------------------------------------------------
   // try to open the subkey.  If this fails, return FALSE.
   // ------------------------------------------------------------------------------
   if ((retVal = RegOpenKey(HKEY_LOCAL_MACHINE, 
                            keyString, 
                            &keyHandle)) != ERROR_SUCCESS) {

      #ifdef DEBUG
         Debug_Printf_Service("Korg 1212 Card %d: RegOpenKey failed - %d\n",
                              cardIndex,
                              retVal
         );
      #endif // DEBUG
      return FALSE;
   }

   size = sizeof(k1212RegistryData);

   if ((retVal = RegQueryValueEx(keyHandle,
                                 K1212_SETTINGS_STRING,
                                 0,
                                 &type,
                                 (unsigned char*)registryData,
                                 &size
                 )) != ERROR_SUCCESS) {

      #ifdef DEBUG
         Debug_Printf_Service("Korg 1212 Card %d: RegQueryValueEx failed - %d\n",
                              cardIndex,
                              retVal
         );
      #endif // DEBUG
      RegCloseKey(keyHandle);
      return FALSE;
   }
   #ifdef DEBUG
      Debug_Printf_Service("Korg 1212 Card %d: RegQueryValueEx succeeded with type %d and size %d\n",
                           cardIndex,
                           type,
                           size
      );
   #endif // DEBUG
   RegCloseKey(keyHandle);
   return TRUE;
}


// ------------------------------------------------------------------------
// ReadRegistryWaveValues
//
//    This function, given a card index, reads the stored wave device
//    configuration data from the system registry for that card.  If the 
//    information cannot be retrieved, FALSE is returned.
// ------------------------------------------------------------------------
BOOL ReadRegistryWaveValues(DWORD                  cardIndex,
                            k1212RegistryWaveData* registryData)
{
   char  keyString[MAX_REGPATH_SIZE];
   char  tempNumStr[3];
   HKEY  keyHandle;
   DWORD type, size, retVal;

   // ------------------------------------------------------------------------------
   // First, create the subkey name.  The subkey is CardXXYY, where XX is the bus
   // number, and YY is the device number.
   // ------------------------------------------------------------------------------
   strcpy(keyString, getRegistryPath(cardIndex));        // keyString = regPath
   strcat(keyString, "\\Card");                           // keyString = regPath\Card
   charTo2HexChars(tempNumStr, (char)getBusNum(cardIndex));
   strcat(keyString, tempNumStr);                        // keyString = regPath\CardXX
   charTo2HexChars(tempNumStr, (char)getDeviceNum(cardIndex));
   strcat(keyString, tempNumStr);                        // keyString = regPath\CardXXYY
   strcat(keyString, "\\waveinfo");                      // keyString = regPath\CardXXYY\waveinfo

   #ifdef DEBUG
      Debug_Printf_Service("Korg 1212 Card %d: keyString = %s\n",
                           cardIndex,
                           keyString
      );
   #endif // DEBUG

   // ------------------------------------------------------------------------------
   // try to open the subkey.  If this fails, return FALSE.
   // ------------------------------------------------------------------------------
   if ((retVal = RegOpenKey(HKEY_LOCAL_MACHINE, 
                            keyString, 
                            &keyHandle)) != ERROR_SUCCESS) {

      #ifdef DEBUG
         Debug_Printf_Service("Korg 1212 Card %d: RegOpenKey failed - %d\n",
                              cardIndex,
                              retVal
         );
      #endif // DEBUG
      return FALSE;
   }

   size = sizeof(k1212RegistryWaveData);

   if ((retVal = RegQueryValueEx(keyHandle,
                                 K1212_SETTINGS_STRING,
                                 0,
                                 &type,
                                 (unsigned char*)registryData,
                                 &size
                  )) != ERROR_SUCCESS) {

      #ifdef DEBUG
         Debug_Printf_Service("Korg 1212 Card %d: RegQueryValueEx failed - %d\n",
                              cardIndex,
                              retVal
         );
      #endif // DEBUG
      RegCloseKey(keyHandle);
      return FALSE;
   }
   #ifdef DEBUG
      Debug_Printf_Service("Korg 1212 Card %d: RegQueryValueEx succeeded with type %d and size %d\n",
                           cardIndex,
                           type,
                           size
      );
   #endif // DEBUG
   RegCloseKey(keyHandle);
   return TRUE;
}


// ------------------------------------------------------------------------
// RestoreRegistryValues
//
//    This function, given a card index, restores the values contained in
//    the k1212RegistryData structure to the card.
// ------------------------------------------------------------------------
BOOL RestoreRegistryValues(DWORD                    cardIndex,
                           const k1212RegistryData* registryData)
{
   DWORD                channelCount;
   WORD*                rteTable;
   short*               volTable;

   if (registryData == NULL) {
      return FALSE;
   }
   
   // -------------------------------------------------------------------
   // copy over volume and routing table settings
   // -------------------------------------------------------------------
   rteTable = getRoutingTable(cardIndex);
   volTable = getVolumeTable(cardIndex);
   if ((volTable == NULL) || (rteTable == NULL)) {
      return FALSE;
   }
   
   for (channelCount = 0; channelCount < kAudioChannels; channelCount++) {
      rteTable[channelCount] = registryData->routing[channelCount];
      volTable[channelCount] = registryData->volume[channelCount];
   }

   // -------------------------------------------------------------------
   // copy over input sensitivity and clock source rate settings
   // -------------------------------------------------------------------
   setLeftADCInSens(cardIndex, registryData->leftADCSens);
   setRightADCInSens(cardIndex, registryData->rightADCSens);

   // -------------------------------------------------------------------
   // set the clock source and rate, and then write the input sensitivity 
   // settings to the card
   // -------------------------------------------------------------------
   if (setClkSrcRate(cardIndex,
                     registryData->clkSrcRate
       )) {
      Send1212Command(cardIndex,
                      K1212_DB_SetClockSourceRate,
                      ClockSourceSelector[getClkSrcRate(cardIndex)],
                      DUMMY_PARAMETER,
                      DUMMY_PARAMETER,
                      DUMMY_PARAMETER
      );
   }
   return (WriteADCSensitivity(cardIndex));
}


// ------------------------------------------------------------------------
// RestoreRegistryWaveValues
//
//    This function, given a card index, restores the values contained in
//    the k1212RegistryWaveData structure to the card's wave devices.
// ------------------------------------------------------------------------
BOOL RestoreRegistryWaveValues(DWORD                        cardIndex,
                               const k1212RegistryWaveData* registryData)
{
   DWORD                deviceId;
   k1212WaveOutDevice*  waveOutDevPtr;
   k1212WaveInDevice*   waveInDevPtr;

   if (registryData == NULL) {
      return FALSE;
   }
   
   // -------------------------------------------------------------------
   // copy over wave device settings
   // -------------------------------------------------------------------
   for (deviceId = 0; deviceId < k1212NumWaveDevices; deviceId++) {
      waveOutDevPtr = getWaveOutDevPtr(cardIndex,
                                       deviceId
                      );
      if (waveOutDevPtr != 0) {
         waveOutDevPtr->leftVolumeLevel      = registryData->leftWaveVolume   [deviceId];
         waveOutDevPtr->rightVolumeLevel     = registryData->rightWaveVolume  [deviceId];
         waveOutDevPtr->volumeControlEnabled = registryData->volumeEnableFlags[deviceId];
         waveOutDevPtr->muted                = (BOOL)registryData->muteFlags  [deviceId];
//         waveOutDevPtr->chainLinkSync        = registryData->playSyncFlags    [deviceId];
      }

      waveInDevPtr = getWaveInDevPtr(cardIndex,
                                     deviceId
                     );
      if (waveInDevPtr != 0) {
//         waveInDevPtr->chainLinkSync = registryData->recSyncFlags[deviceId];
      }
   }

   // -------------------------------------------------------------------
   // copy over the auto sync setting
   // -------------------------------------------------------------------
   setAutoWaveSyncMode(cardIndex,
                       registryData->autoSync
   );
   
   return TRUE;
}
