//*****************************************************************************
//* Name:
//*      Mixer.cpp
//*
//* Project:
//*      1212 I/O Driver
//*
//* Author:
//*      Bill Jenkins
//*
//* Description:
//*               NOTE: This entire implementation is based on only supporting
//*               volume control for each channel of each device.  Beyond that,
//*               the architecture should be changed so that we keep arrays of
//*               controls like the DDK samples do.  The VxD would then need to
//*               be updated to allow control changes to be transferred to the card.
//*
//* Modification Log:
//*
//*      1.1   4/1/97 Bill
//*      Initial version.  
//*
//*
//* Copyright (c) 1997 Korg, Inc.
//* All rights reserved.
//*
//* This program is protected as an unpublished work under the U.S.
//* copyright laws.  The above copyright notice is not intended to
//* effect a publication of this work.
//*
//* This program is the confidential and proprietary information of
//* Korg and all its subsidiaries.
//*
//* Note: This file was originally adapted from the Microsoft Windows
//*       95 DDK.  Copyright (c) 1991 - 1995  Microsoft Corporation.
//*****************************************************************************


#include <windows.h>
#include <string.h>

#include "gsound.h"
#include "12pmapic.h"
#include "driver.h"
#include "mixer.h"
#include "1212WavDefs.h"
#include "1212stat.h"
#include "12pmctrl.h"


// ----------------------------------------------------------
// the following enum and string table provide strings for
// various mixer control labels.
// ----------------------------------------------------------
typedef enum {
               MY_VOL_CTRL = 0,
               MASTER_VOL
} mixerStringsId;

char __far*  mixerStrings[] = {"My volume control",
                               "Master Volume"
             };


//==========================================================================;
//==========================================================================;
//==========================================================================;
//==========================================================================;
//==========================================================================;
//==========================================================================;

//--------------------------------------------------------------------------;
//
//  VOID mxdGetDevCaps
//
//  Description:
//      This function handles the MXDM_GETDEVCAPS message for mxdMessage.
//      This message is sent by the Mixer Manager when the caller calls
//      mixerGetDevCaps.
//
//--------------------------------------------------------------------------;

VOID FAR PASCAL mxdGetDevCaps
(
   WORD                 cardRefNum,
   WORD                 uDevId,
   MDEVICECAPSEX FAR*   lpCaps
)
{
   MIXERCAPS   mc;
   char        mixerName[MAXPNAMELEN];

   mc.wMid           = MM_KORG;
   mc.wPid           = MM_KORG_1212IO_MSWAVEOUT;
   mc.vDriverVersion = DRV_VERSION;
   mc.fdwSupport     = NULL;                       // (currently all reserved)
   mc.cDestinations  = NUM_MIX_DESTINATIONS;       // one master destination (6 devices connected)

   // --------------------------------------------------------------
   // create the device name, and copy it into the dev cap structure
   // --------------------------------------------------------------
   pK1212Driver->GetMixerName(mixerName);
   lstrcpyn(mc.szPname, 
            mixerName, 
            (MAXPNAMELEN-1)
   );

   _fmemcpy(lpCaps->pCaps, 
            &mc,
            min((UINT)lpCaps->cbSize, sizeof(mc)));

} // end of mxdGetDevCaps()


//--------------------------------------------------------------------------
//
//  DWORD MxdOpen
//
//  Description:
//      This function handles the MXDM_OPEN message for mxdMessage. This
//      message is sent when a client calls mixerOpen. A client calls
//      mixerOpen to get one or more of following:
//
//      1.  Locks the mixer device such that it will not be removed until
//          the client closes the device.
//
//      2.  Allows a client to receive notifications of changes in the
//          mixer device state.
//
//      A mixer driver should be written to allow multiple opens on a
//      single device (thus allowing more than one client to receive
//      notifications).
//
//      Note that the Mixer Manager may or may NOT coalesce opens and closes
//      on a mixer device (depends on the situation) from applications.
//      This should make NO difference to the mixer device (and there is no
//      way for the mixer device to know). But if you are debugging your
//      driver and notice that only one MXDM_OPEN message gets sent even
//      if more than one application open the device, you now know why.
//
//  Arguments:
//      LPDWORD pdwUser: Pointer to _mixer device instance data store_.
//      The value that the mixer device places in this location will be
//      returned to the device as the dwUser argument in mxdMessage.
//      This value can be anything that the mixer device chooses.
//
//      LPMIXEROPENDESC pmxod: Pointer to a MIXEROPENDESC structure that
//      contains callback and other information for the client.
//
//      DWORD fdwOpen: Flags passed to mixerOpen. See the list of valid
//      flags for mixerOpen to determine what this argument may be.
//
//  Return (DWORD):
//      Returns zero (MMSYSERR_NOERROR) if successfully opened. Otherwise
//      returns a non-zero error code.
//
//
//--------------------------------------------------------------------------
DWORD NEAR PASCAL MxdOpen
(
   LPDWORD             pdwUser,
   WORD                cardIndex,
   UINT                uDevId,
   LPMIXEROPENDESC     pmxod,
   DWORD               fdwOpen
)
{
   PMXDCLIENT   pmxdc;

   // --------------------------------------------------------------------
   //  if we cannot allocate our instance structure, then we must fail
   //  the open request.
   // --------------------------------------------------------------------
   pmxdc = (PMXDCLIENT)LocalAlloc(LPTR, 
                                  sizeof(MXDCLIENT)
                       );
   if (pmxdc == NULL) {
       return (MMSYSERR_NOMEM);
   }

   // --------------------------------------------------------------------
   //  fill in our instance structure... save a bunch of stuff for
   //  callbacks.
   // --------------------------------------------------------------------
   pmxdc -> fdwOpen      = fdwOpen;
   pmxdc -> hmx          = pmxod->hmx;
   pmxdc -> dwCallback   = pmxod->dwCallback;
   pmxdc -> fuCallback   = (UINT)HIWORD(fdwOpen & CALLBACK_TYPEMASK);
   pmxdc -> dwInstance   = pmxod->dwInstance;
   pmxdc -> cardIndex    = cardIndex;
   pmxdc -> deviceIndex  = uDevId;
   pmxdc -> lpNext       = NULL;

   // --------------------------------------------------------------------
   //  add to the device's client list
   // --------------------------------------------------------------------
   if (pK1212Driver) {
      if (pK1212Driver->AddMixerClient(cardIndex,
                                       uDevId,
                                       (LPMXDCLIENT)pmxdc
                        ) != K1212_CMDRET_Success) {
         return MMSYSERR_ERROR;
      }
   } 
   else {
      return MMSYSERR_NODRIVER;
   }

   // --------------------------------------------------------------------
   //  return our instance data pointer--this will be returned in the
   //  dwUser argument for all other mixer driver messages that are
   //  on behalf of an opened instance..
   // --------------------------------------------------------------------
   *pdwUser = (DWORD)(UINT)pmxdc;

    return MMSYSERR_NOERROR;

} // MxdOpen()


//--------------------------------------------------------------------------;
//
//  DWORD MxdClose
//
//  Description:
//      This function is called to handle the MXDM_CLOSE from mxdMessage.
//      This message is generated by a client calling mixerClose on a
//      previously mixerOpen'd mixer device handle. This function will never
//      be called unless it is for previously _successful_ call to
//      mixerOpen.
//
//  Arguments:
//
//      PMXDCLIENT pmxdc
//         Pointer to mixer device instance data allocated
//         by MxdOpen. This argument is passed as the dwUser
//         argument to mxdMessage.
//
//  Return (DWORD):
//      Should always succeed. Returns zero (MMSYSERR_NOERROR).
//
//
//--------------------------------------------------------------------------;
DWORD NEAR PASCAL MxdClose
(
   PMXDCLIENT  pmxdc
)
{
   WORD  cardIndex;
   UINT  uDevId;

   cardIndex = pmxdc->cardIndex;
   uDevId    = pmxdc->deviceIndex;

   // --------------------------------------------------------------
   // Note: pmxdc will always be valid - we will not get a close
   //       from msmixmgr (or mmsystem) unless an open has
   //       succeeded.
   // --------------------------------------------------------------
   if (pK1212Driver) {
      pK1212Driver->RemoveMixerClient(cardIndex,
                                      uDevId,
                                      (LPMXDCLIENT)pmxdc
                    );
   }
   LocalFree((HLOCAL)pmxdc);

   return MMSYSERR_NOERROR;

} // MxdClose()


//--------------------------------------------------------------------------;
//
//  DWORD MxdGetLineInfo
//
//  Description:
//      This function handles the MXDM_GETLINEINFO message for mxdMessage.
//      The message is sent in response to a client calling mixerGetLineInfo.
//
//      There are currently five different query types that the caller can
//      use to get information on a line (specified in fdwInfo):
//
//      1.  MIXER_GETLINEINFOF_DESTINATION: caller wants information on
//          the MIXERLINE.dwDestination line.
//
//      2.  MIXER_GETLINEINFOF_SOURCE: caller wants information on the
//          MIXERLINE.dwSource associated with MIXERLINE.dwDestination.
//
//      3.  MIXER_GETLINEINFOF_LINEID: caller wants information on the
//          the MIXERLINE.dwLineID line.
//
//      4.  MIXER_GETLINEINFOF_COMPONENTTYPE: caller wants information on
//          the _first_ MIXERLINE.dwComponentType.
//
//      5.  MIXER_GETLINEINFOF_TARGETTYPE: caller wants information on the
//          the MIXERLINE.Target.dwType (wMid, wPid, etc) line.
//
//      All mixer drivers must support these four queries. If a query
//      is sent that the mixer device does not know how to handle, then
//      MMSYSERR_NOTSUPPORTED must be returned.
//
//  Arguments:
//      PMXDCLIENT pmxdc
//         Pointer to mixer device instance data allocated
//         by MxdOpen (if the mixer device was opened!).
//
//      LPMIXERLINE pmxl
//         Pointer to caller's receiving buffer for the line information.
//         This pointer has been validated by the Mixer Manager to be at
//         least big enough for all input arguments and big enough to
//         hold MIXERLINE.cbStruct bytes of information.
//
//      DWORD fdwInfo
//         Flags passed from mixerGetLineInfo.
//
//  Return (DWORD):
//      Returns zero (MMSYSERR_NOERROR) if successfull. Otherwise
//      returns a non-zero error code.
//
//
//--------------------------------------------------------------------------;
DWORD NEAR PASCAL MxdGetLineInfo
(
   PMXDCLIENT   pmxdc,
   LPMIXERLINE  pmxl,
   DWORD        fdwInfo
)
{
   WORD              cardRefNum;
   WORD              deviceId;
   UINT              uDestination;
   UINT              uSource;
   UINT              cbmxl;
   
   if (pmxdc == NULL) {
      return MMSYSERR_INVALHANDLE;
   }
   if (!pK1212Driver) {
      return MMSYSERR_NODRIVER;   
   }

   cardRefNum = pmxdc->cardIndex;
   deviceId   = pmxdc->deviceIndex;

   // --------------------------------------------------------------------
   //  we will fill in the following elements dynamically:
   //
   //      dwDestination
   //      dwSource
   //      dwLineID
   //
   // note, we just want to support volume control for each of the wave
   // devices, so a lot of the info provided is common to all cases.
   // --------------------------------------------------------------------

   pmxl->fdwLine           = MIXERLINE_LINEF_ACTIVE;
   pmxl->dwUser            = (DWORD)(WORD)pmxdc;

   pmxl->Target.dwType         = MIXERLINE_TARGETTYPE_WAVEOUT;
   pmxl->Target.dwDeviceID     = deviceId;
   pmxl->Target.wMid           = MM_KORG;
   pmxl->Target.wPid           = MM_KORG_1212IO_MSWAVEOUT;
   pmxl->Target.vDriverVersion = DRV_VERSION;
   lstrcpyn(pmxl->Target.szPname, 
            mixerStrings[MY_VOL_CTRL], 
            (MAXPNAMELEN-1)
   );

   cbmxl  = (UINT)min(pmxl->cbStruct, sizeof(MIXERLINE));
   pmxl->cbStruct = cbmxl;

   // --------------------------------------------------------------------
   //  determine what line to get the information for. a mixer driver
   //  MUST support the following four queries:
   //
   //      MIXER_GETLINEINFOF_DESTINATION
   //      MIXER_GETLINEINFOF_SOURCE
   //      MIXER_GETLINEINFOF_LINEID
   //      MIXER_GETLINEINFOF_COMPONENTTYPE
   //
   //
   //  others (no others are defined for V1.00 of MSMIXMGR) can optionally
   //  be supported. if this mixer driver does NOT support a query, then
   //  MMSYSERR_NOTSUPPORTED must be returned.
   // --------------------------------------------------------------------

   switch (fdwInfo & MIXER_GETLINEINFOF_QUERYMASK)
   {
      // -----------------------------------------------------------------
      //  MIXER_GETLINEINFOF_DESTINATION
      //
      //  this query specifies that the caller is interested in the
      //  line information for MIXERLINE.dwDestination. this index can
      //  range from 0 to MIXERCAPS.cDestinations - 1.
      //
      //  valid elements of MIXERLINE:
      //      cbStruct
      //      dwDestination
      //
      //  all other MIXERLINE elements are undefined.
      // -----------------------------------------------------------------
      case MIXER_GETLINEINFOF_DESTINATION:
         mixer_getlineinfof_destination:
         if (pmxl->dwDestination >= NUM_MIX_DESTINATIONS) {
            return (MIXERR_INVALLINE);
         }

         // --------------------------------------------------------------
         //  return the dwLineID - master volume only for a destination
         //    control
         // --------------------------------------------------------------
         pK1212Driver->GetDestinationShortName(cardRefNum,
                                               pmxl->szShortName
                       );
         pK1212Driver->GetDestinationLongName(cardRefNum,
                                              pmxl->szName
                       );
         pmxl->dwComponentType = pK1212Driver->GetDestType(cardRefNum);
         pmxl->dwLineID        = MAKELONG(pmxl->dwDestination, DEST_WORD_MASK);
         pmxl->cControls       = pK1212Driver->GetNumDestMixControls(cardRefNum);
         pmxl->cChannels       = 1;    // disables panning for the master volume control
         pmxl->cConnections    = NUM_MIX_CONNECTIONS;
         return (MMSYSERR_NOERROR);


      // -----------------------------------------------------------------
      //  MIXER_GETLINEINFOF_SOURCE
      //
      //  this query specifies that the caller is interested in the
      //  line information for MIXERLINE.dwSource associated with
      //  MIXERLINE.dwDestination.
      //
      //  valid elements of MIXERLINE:
      //      cbStruct
      //      dwDestination
      //      dwSource
      //
      //  all other MIXERLINE elements are undefined.
      // -----------------------------------------------------------------
      case MIXER_GETLINEINFOF_SOURCE:
         mixer_getlineinfof_source:
         uDestination = (UINT)pmxl->dwDestination;
         if (uDestination >= NUM_MIX_DESTINATIONS) {
            return (MIXERR_INVALLINE);
         }

         uSource = (UINT)pmxl->dwSource;
         if (uSource >= NUM_MIX_SOURCES) {
            return (MIXERR_INVALLINE);
         }
         
         pK1212Driver->GetSourceName(cardRefNum,
                                     uSource,
                                     pmxl->szShortName
                       );
         pK1212Driver->GetSourceName(cardRefNum,
                                     uSource,
                                     pmxl->szName
                       );

         // --------------------------------------------------------------
         //  return the dwLineID
         // --------------------------------------------------------------
         pmxl->dwComponentType = pK1212Driver->GetSourceType(cardRefNum,
                                                             uSource
                                               );
         pmxl->dwLineID        = MAKELONG(uDestination, uSource);
         pmxl->cControls       = pK1212Driver->GetNumSourceMixControls(cardRefNum, 
                                                                       uSource
                                               );
         pmxl->cChannels       = NUM_MIX_CHANNELS;
         pmxl->cConnections    = NUM_MIX_CHANNELS;
         return (MMSYSERR_NOERROR);


      // -----------------------------------------------------------------
      //  MIXER_GETLINEINFOF_LINEID
      //
      //  this query specifies that the caller is interested in the
      //  line information for MIXERLINE.dwLineID. the dwLineID is
      //  completely mixer driver dependent, so this driver must validate
      //  the ID.
      //
      //  valid elements of MIXERLINE:
      //      cbStruct
      //      dwLineID
      //
      //  all other MIXERLINE elements are undefined.
      // -----------------------------------------------------------------
      case MIXER_GETLINEINFOF_LINEID:
         uDestination = LOWORD(pmxl->dwLineID);
         uSource      = HIWORD(pmxl->dwLineID);

         pmxl->dwDestination = uDestination;
         pmxl->dwSource      = uSource;

         if (uSource == DEST_WORD_MASK) {
            goto mixer_getlineinfof_destination;
         }
         else {
            goto mixer_getlineinfof_source;
         }
         break;

      // -----------------------------------------------------------------
      //  MIXER_GETLINEINFOF_COMPONENTTYPE
      //
      //  this query specifies that the caller is interested in the
      //  line information for MIXERLINE.dwComponentType
      //
      //  valid elements of MIXERLINE:
      //      cbStruct
      //      dwComponentType
      //
      //  all other MIXERLINE elements are undefined.
      // -----------------------------------------------------------------
      case MIXER_GETLINEINFOF_COMPONENTTYPE:
         // ----------------------------------------------------
         // check the destination first for a match
         // ----------------------------------------------------
         if (pmxl->dwComponentType == pK1212Driver->GetDestType(cardRefNum)) {
            goto mixer_getlineinfof_destination;
         }

         // ----------------------------------------------------
         // now check the sources for a match
         // ----------------------------------------------------
         for (uSource = 0; uSource < NUM_MIX_CONNECTIONS; uSource++) {
            if (pmxl->dwComponentType == pK1212Driver->GetSourceType(cardRefNum, uSource)) {
               break;
            }
         }
         if (uSource < NUM_MIX_CONNECTIONS) {
            pmxl->dwDestination = uDestination;
            pmxl->dwSource      = uSource;
            goto mixer_getlineinfof_source;
         }
         return (MIXERR_INVALLINE);


      // -----------------------------------------------------------------
      //  MIXER_GETLINEINFOF_TARGETTYPE
      //
      //  this query specifies that the caller is interested in the
      //  line information for MIXERLINE.Target.
      //
      //  valid elements of MIXERLINE:
      //      cbStruct
      //      Target.dwType
      //      Target.wMid
      //      Target.wPid
      //      Target.vDriverVersion
      //      Target.szPname
      //
      //  all other MIXERLINE elements are undefined.
      // -----------------------------------------------------------------
      case MIXER_GETLINEINFOF_TARGETTYPE:
         switch (pmxl->Target.dwType) {
            case MIXERLINE_TARGETTYPE_WAVEOUT:
               return (MMSYSERR_NOERROR);

            case MIXERLINE_TARGETTYPE_WAVEIN:
               return (MIXERR_INVALLINE);

            case MIXERLINE_TARGETTYPE_MIDIOUT:
               return (MIXERR_INVALLINE);

            case MIXERLINE_TARGETTYPE_MIDIIN:
               return (MIXERR_INVALLINE);

            case MIXERLINE_TARGETTYPE_AUX:
               return (MIXERR_INVALLINE);

            default:
               return (MIXERR_INVALLINE);
         }  // end switch target type
         return (MIXERR_INVALLINE);

      // -----------------------------------------------------------------
      //  if the query type is not something this driver understands, then
      //  return MMSYSERR_NOTSUPPORTED.
      // -----------------------------------------------------------------
      default:
         return (MMSYSERR_NOTSUPPORTED);
   }
} // MxdGetLineInfo()

//--------------------------------------------------------------------------;
//
//  DWORD MxdGetLineControls
//
//  Description:
//      This function handles the MXDM_GETLINECONTROLS message for
//      mxdMessage. The message is sent in response to a client calling
//      mixerGetLineControls.
//
//      There are currently three different query types that the caller can
//      use to get information on a mixer's controls (specified in
//      fdwControls):
//
//      1.  MIXER_GETLINECONTROLSF_ALL: caller wants all controls for
//          a specified line (dwLineID).
//
//      2.  MIXER_GETLINECONTROLSF_ONEBYID: caller wants one control
//          specified by dwControlID.
//
//      3.  MIXER_GETLINECONTROLSF_ONEBYTYPE: caller wants the FIRST control
//          on the specified line (dwLineID) of type dwControlType.
//
//      All mixer drivers must support these three queries. If a query
//      is sent that the mixer device does not know how to handle, then
//      MMSYSERR_NOTSUPPORTED must be returned.
//
//  Arguments:
//      PMXDCLIENT pmxdc: Pointer to mixer device instance data allocated
//      by MxdOpen (if the mixer device was opened!). This argument MAY be
//      NULL if the caller did not open the mixer device.
//
//      LPMIXERLINECONTROLS pmxlc: Pointer to line control information
//      header. This pointer has been validated by the Mixer Manager to
//      be at least sizeof(MIXERLINECONTROLS) in size (currently the only
//      valid size for the header).
//
//      Also note that cControls has been validated to be >= 1. cbmxctrl
//      is at least sizeof(MIXERCONTROL) and pmxctrl points to a memory
//      location big enough to hold (cControls * cbmxctrl) bytes.
//
//      DWORD fdwControls: Flags passed from mixerGetLineControls.
//
//  Return (DWORD):
//      Returns zero (MMSYSERR_NOERROR) if successfull. Otherwise
//      returns a non-zero error code.
//
//
//--------------------------------------------------------------------------;

DWORD NEAR PASCAL MxdGetLineControls
(
    PMXDCLIENT              pmxdc,
    LPMIXERLINECONTROLS     pmxlc,
    DWORD                   fdwControls
)
{
   WORD           cardRefNum;
   WORD           deviceId;
   LPMIXERCONTROL pmxctrl;
   UINT           uDestination;
   UINT           uSource;
   UINT           uCurControl;
   DWORD          uNumControlsToCopy;
   MIXERCONTROL   mixControl;

   const CKorg1212MixControl __far* lpMixControl;

   if (pmxdc == NULL) {
      return MMSYSERR_INVALHANDLE;
   }
   if (!pK1212Driver) {
      return MMSYSERR_NODRIVER;   
   }

   cardRefNum = pmxdc->cardIndex;
   deviceId   = pmxdc->deviceIndex;

   pmxctrl    = pmxlc->pamxctrl;

   // -----------------------------------------------------------------------
   // for now, we only support volume controls for each channel.  So create
   // a local template to be copied into the caller's structures.
   // -----------------------------------------------------------------------
   mixControl.cbStruct        = sizeof(mixControl);
   mixControl.fdwControl      = 0;
   mixControl.cMultipleItems  = 0;

   // -----------------------------------------------------------------------
   //  Determine for which control(s) to get the information.
   //  A mixer driver MUST support the following three queries:
   //
   //      MIXER_GETLINECONTROLSF_ALL
   //      MIXER_GETLINECONTROLSF_ONEBYID
   //      MIXER_GETLINECONTROLSF_ONEBYTYPE
   //
   //  Others (no others are defined for V1.00 of MSMIXMGR) can optionally
   //  be supported. If this mixer driver does NOT support a query, then
   //  MMSYSERR_NOTSUPPORTED must be returned.
   // -----------------------------------------------------------------------

   switch (fdwControls & MIXER_GETLINECONTROLSF_QUERYMASK) {
      // --------------------------------------------------------------------
      //  MIXER_GETLINECONTROLSF_ALL
      //
      //  This query specifies that the caller is interested in ALL
      //  controls for a line.
      //
      //  Valid elements of MIXERLINECONTROLS:
      //      cbStruct
      //      dwLineID
      //      cControls
      //      cbmxctrl
      //      pamxctrl
      //
      //  All other MIXERLINECONTROLS elements are undefined.
      // --------------------------------------------------------------------
      case MIXER_GETLINECONTROLSF_ALL:

         uDestination = LOWORD(pmxlc->dwLineID);
         if (uDestination >= NUM_MIX_DESTINATIONS) {
            return (MIXERR_INVALLINE);
         }

         uSource = (UINT)(int)HIWORD(pmxlc->dwLineID);
         if (uSource == DEST_WORD_MASK) {
            uNumControlsToCopy = pK1212Driver->GetNumDestMixControls(cardRefNum);
         }
         else {
            if (uSource >= NUM_MIX_SOURCES) {
               return (MIXERR_INVALLINE);
            }
            uNumControlsToCopy = pK1212Driver->GetNumSourceMixControls(cardRefNum,
                                                                       uSource
                                               );
         }

         if (uNumControlsToCopy > pmxlc->cControls) {
            return (MMSYSERR_INVALPARAM);
         }

         // ----------------------------------------------------------------
         // Copy the requested controls
         // ----------------------------------------------------------------
         for (uCurControl = 0; uCurControl < uNumControlsToCopy; uCurControl++) {

            if (uSource == DEST_WORD_MASK) {
               lpMixControl = pK1212Driver->GetMasterVolumeControl(cardRefNum);
               mixControl.dwControlID = MAKELONG(uCurControl, DEST_WORD_MASK);
            }
            else {
               lpMixControl = pK1212Driver->GetMixControl(cardRefNum,
                                                          uSource,
                                                          uCurControl
                                            );
               mixControl.dwControlID = MAKELONG(uCurControl, uSource);
            }
            
            lstrcpyn(mixControl.szShortName, 
                     lpMixControl->GetShortName(),
                     (MIXER_SHORT_NAME_CHARS-1)
            );
            lstrcpyn(mixControl.szName, 
                     lpMixControl->GetName(),
                     (MIXER_LONG_NAME_CHARS-1)
            );
            mixControl.dwControlType   = lpMixControl->GetControlType();
            mixControl.Bounds.lMinimum = lpMixControl->GetMinValue();
            mixControl.Bounds.lMaximum = lpMixControl->GetMaxValue();
            mixControl.Metrics.cSteps  = lpMixControl->GetNumSteps();

            _fmemcpy(pmxctrl, 
                     &mixControl,
                     (WORD)mixControl.cbStruct
            );

            // ----------------------------------------------------------
            //  jump to the next structure--the step size is based on
            //  the caller's specified structure size..
            // ----------------------------------------------------------
            pmxctrl = (LPMIXERCONTROL)((LPBYTE)pmxctrl + pmxlc->cbmxctrl);
         }

         // -------------------------------------------------------------
         //  tell the caller what is really valid in the structure
         // -------------------------------------------------------------
         pmxlc->cbmxctrl = (uNumControlsToCopy * sizeof(MIXERCONTROL));
         return (MMSYSERR_NOERROR);


      // --------------------------------------------------------------------
      //  MIXER_GETLINECONTROLSF_ONEBYID
      //
      //  This query specifies that the caller is interested in ONE
      //  control specified by dwControlID.
      //
      //  valid elements of MIXERLINECONTROLS:
      //      cbStruct
      //      dwControlID
      //      cbmxctrl
      //      pamxctrl
      //
      //  all other MIXERLINECONTROLS elements are undefined.
      // --------------------------------------------------------------------
      case MIXER_GETLINECONTROLSF_ONEBYID:

         // -------------------------------------------------------------
         // Make sure the control ID they gave us is OK.
         // -------------------------------------------------------------
         uDestination = LOWORD(pmxlc->dwLineID);
         if (uDestination >= NUM_MIX_DESTINATIONS) {
            return (MIXERR_INVALLINE);
         }
         uSource = (UINT)(int)HIWORD(pmxlc->dwLineID);
         uCurControl = LOWORD(pmxlc->dwControlID);

         if (uSource == DEST_WORD_MASK) {
            cardRefNum = HIWORD(pmxlc->dwControlID);
            if (uCurControl >= pK1212Driver->GetNumDestMixControls(cardRefNum)) {
               return (MMSYSERR_INVALPARAM);
            }
         }
         else {
            uSource = HIWORD(pmxlc->dwControlID);
            if (uSource >= NUM_MIX_SOURCES) {
               return (MIXERR_INVALLINE);
            }
            if (uCurControl >= pK1212Driver->GetNumSourceMixControls(cardRefNum, uSource)) {
               return (MMSYSERR_INVALPARAM);
            }
         }

         // ----------------------------------------------------------
         // get control pointer and data
         // ----------------------------------------------------------
         if (uSource == DEST_WORD_MASK) {
            lpMixControl = pK1212Driver->GetMasterVolumeControl(cardRefNum);
            mixControl.dwControlID = MAKELONG(uCurControl, DEST_WORD_MASK);
         }
         else {
            lpMixControl = pK1212Driver->GetMixControl(cardRefNum,
                                                       uSource,
                                                       uCurControl
                                         );
            mixControl.dwControlID = MAKELONG(uCurControl, uSource);
         }
            
         lstrcpyn(mixControl.szShortName, 
                  lpMixControl->GetShortName(),
                  (MIXER_SHORT_NAME_CHARS-1)
         );
         lstrcpyn(mixControl.szName, 
                  lpMixControl->GetName(),
                  (MIXER_LONG_NAME_CHARS-1)
         );
         mixControl.dwControlType   = lpMixControl->GetControlType();
         mixControl.Bounds.lMinimum = lpMixControl->GetMinValue();
         mixControl.Bounds.lMaximum = lpMixControl->GetMaxValue();
         mixControl.Metrics.cSteps  = lpMixControl->GetNumSteps();

         // ----------------------------------------------------------
         // copy the control information
         // ----------------------------------------------------------
         pmxlc->cbmxctrl = mixControl.cbStruct;
         _fmemcpy(pmxctrl, 
                  &mixControl,
                  (WORD) mixControl.cbStruct
         );
         pmxlc->dwLineID = pmxlc->dwControlID;
         return (MMSYSERR_NOERROR);

      // --------------------------------------------------------------------
      //  MIXER_GETLINECONTROLSF_ONEBYTYPE
      //
      //  This query specifies that the caller is interested in the
      //  FIRST control of type dwControlType on dwLineID.
      //
      //  Valid elements of MIXERLINECONTROLS:
      //      cbStruct
      //      dwLineID
      //      dwControlType
      //      cbmxctrl
      //      pamxctrl
      //
      //  all other MIXERLINECONTROLS elements are undefined.
      // --------------------------------------------------------------------
      case MIXER_GETLINECONTROLSF_ONEBYTYPE:

         uDestination = LOWORD(pmxlc->dwLineID);
         if (uDestination >= NUM_MIX_DESTINATIONS) {
            return (MIXERR_INVALLINE);
         }

         uSource = (UINT)(int)HIWORD(pmxlc->dwLineID);
         if ((uSource != DEST_WORD_MASK) && (uSource >= NUM_MIX_SOURCES)) {
            return (MIXERR_INVALLINE);
         }

         if (uSource == DEST_WORD_MASK) {
            lpMixControl = pK1212Driver->GetMasterVolumeControl(cardRefNum);
            if (pmxlc->dwControlType == lpMixControl->GetControlType()) {
               mixControl.dwControlID = MAKELONG(0, DEST_WORD_MASK);
            }
            else {
               return (MIXERR_INVALCONTROL);               
            }
         }
         else {
            for (uCurControl = 0; 
                 uCurControl < pK1212Driver->GetNumSourceMixControls(cardRefNum, 0);
                 uCurControl++) {

               lpMixControl = pK1212Driver->GetMixControl(cardRefNum,
                                                          uSource,
                                                          uCurControl
                                            );
               if (pmxlc->dwControlType == lpMixControl->GetControlType()) {
                  mixControl.dwControlID = MAKELONG(uCurControl, uSource);
                  break;
               }
            }
            if (uCurControl >= pK1212Driver->GetNumSourceMixControls(cardRefNum, 0)) {
               return (MIXERR_INVALCONTROL);
            }
         }


         // ----------------------------------------------------------
         // get control information
         // ----------------------------------------------------------
         lstrcpyn(mixControl.szShortName, 
                  lpMixControl->GetShortName(),
                  (MIXER_SHORT_NAME_CHARS-1)
         );
         lstrcpyn(mixControl.szName, 
                  lpMixControl->GetName(),
                  (MIXER_LONG_NAME_CHARS-1)
         );
         mixControl.dwControlType   = lpMixControl->GetControlType();
         mixControl.Bounds.lMinimum = lpMixControl->GetMinValue();
         mixControl.Bounds.lMaximum = lpMixControl->GetMaxValue();
         mixControl.Metrics.cSteps  = lpMixControl->GetNumSteps();

         // ------------------------------------------------------------
         // copy information from our local structure into the client's
         // ------------------------------------------------------------
         pmxlc->cbmxctrl = mixControl.cbStruct ;
         _fmemcpy(pmxctrl, 
                  &mixControl,
                  (WORD)mixControl.cbStruct
         );
         return ( MMSYSERR_NOERROR ) ;


      // --------------------------------------------------------------------
      //  If the query type is not something this driver understands,
      //  then return MMSYSERR_NOTSUPPORTED.
      // --------------------------------------------------------------------
      default:
         return (MMSYSERR_NOTSUPPORTED);
   }
} // MxdGetLineControls()


//--------------------------------------------------------------------------;
//
//  MMRESULT MxdGetControlDetails
//
//  Description:
//
//
//  Arguments:
//      PHARDWAREINSTANCE phwi
//
//      LPMIXERCONTROLDETAILS pmxcd
//
//      DWORD fdwDetails
//
//  Return (MMRESULT):
//
//
//--------------------------------------------------------------------------;

MMRESULT FAR PASCAL MxdGetControlDetails
(
   PMXDCLIENT              pmxdc,
   LPMIXERCONTROLDETAILS   pmxcd,
   DWORD                   fdwDetails
)
{
   WORD            cardRefNum;
   WORD            deviceId;
   WORD            volumeLevel;
   WORD            muteValue;
   UINT            uSource;
   UINT            uControlId;

   const CKorg1212MixControl __far* lpMixControl;

   if (pmxdc == NULL) {
      return MMSYSERR_INVALHANDLE;
   }

   cardRefNum = pmxdc->cardIndex;
   deviceId   = pmxdc->deviceIndex;

   if (!pK1212Driver) {
      return MMSYSERR_NODRIVER;   
   }

   // -----------------------------------------------------------------
   //  verify that the control id is not bogus...  If it isn't, get
   //  the control.
   // -----------------------------------------------------------------
   uControlId = LOWORD(pmxcd->dwControlID);
   uSource    = HIWORD(pmxcd->dwControlID);
   if (uSource == DEST_WORD_MASK) {
      if (uControlId >= pK1212Driver->GetNumDestMixControls(cardRefNum)) {
         return (MMSYSERR_INVALPARAM);
      }
      else {
         lpMixControl = pK1212Driver->GetMasterVolumeControl(cardRefNum);
      }
   }
   else {
      if (uSource >= NUM_MIX_SOURCES) {
         return (MMSYSERR_INVALPARAM);
      }
      if (uControlId >= pK1212Driver->GetNumSourceMixControls(cardRefNum, uSource)) {
         return (MMSYSERR_INVALPARAM);
      }
      lpMixControl = pK1212Driver->GetMixControl(cardRefNum,
                                                 uSource,
                                                 uControlId
                                   );
   }

   switch (fdwDetails & MIXER_GETCONTROLDETAILSF_QUERYMASK) {

      // ---------------------------------------------------------------
      // If a multichannel control is queried with cChannels == 1
      // this routine must return a combination of left/right
      // channels. (for volume, we just return the left channel value)
      // ---------------------------------------------------------------
      case MIXER_GETCONTROLDETAILSF_VALUE:
      {

         switch (lpMixControl->GetControlType()) {

            case MIXERCONTROL_CONTROLTYPE_MUTE:
            {
               LPMIXERCONTROLDETAILS_BOOLEAN pmxcd_f;
               
               pmxcd_f = (LPMIXERCONTROLDETAILS_BOOLEAN) pmxcd->paDetails;

               pK1212Driver->QueryWaveDeviceMute(cardRefNum,
                                                 uSource,
                                                 muteValue
                             );
               pmxcd_f->fValue = muteValue;
            }

            case MIXERCONTROL_CONTROLTYPE_VOLUME:
            {
               LPMIXERCONTROLDETAILS_SIGNED    pmxcd_s;

               pmxcd_s = (LPMIXERCONTROLDETAILS_SIGNED)pmxcd->paDetails;

               if (uSource == DEST_WORD_MASK) {
                  pK1212Driver->GetWaveMasterVolume(cardRefNum,
                                                    volumeLevel
                                );
               }
               else {
                  pK1212Driver->GetWaveDeviceVolume(cardRefNum,
                                                    uSource,
                                                    uControlId,
                                                    volumeLevel
                                );
               }
               pmxcd_s->lValue = volumeLevel;
               return (MMSYSERR_NOERROR);
            }

            default:
            {
               return (MMSYSERR_INVALPARAM);
            }
         }
      }
      break;

      case MIXER_GETCONTROLDETAILSF_LISTTEXT:
         // -----------------------------------------------------------
         //  We don't support list text.
         // -----------------------------------------------------------
         return (MIXERR_INVALCONTROL);


      // ------------------------------------------------------------
      //  if the query type is not something this driver understands, then
      //  return MMSYSERR_NOTSUPPORTED.
      // ------------------------------------------------------------
      default:
         return (MMSYSERR_NOTSUPPORTED);
   }
} // MxdGetControlDetails()


//--------------------------------------------------------------------------;
//
//  MMRESULT MxdSetControlDetails
//
//  Description:
//
//
//  Arguments:
//      PHARDWAREINSTANCE phwi
//
//      LPMIXERCONTROLDETAILS pmxcd
//
//      DWORD fdwDetails
//
//  Return (MMRESULT):
//
//
//--------------------------------------------------------------------------;

MMRESULT FAR PASCAL MxdSetControlDetails
(
   PMXDCLIENT              pmxdc,
   LPMIXERCONTROLDETAILS   pmxcd,
   DWORD                   fdwDetails
)
{
   LPMXDCLIENT     lpCur;
   WORD            cardRefNum;
   WORD            deviceId;
   UINT            uSource;
   UINT            uControlId;

   const CKorg1212MixControl __far* lpMixControl;

   if (pmxdc == NULL) {
      return MMSYSERR_INVALHANDLE;
   }

   cardRefNum = pmxdc->cardIndex;
   deviceId   = pmxdc->deviceIndex;

   if (!pK1212Driver) {
      return MMSYSERR_NODRIVER;   
   }

   // ------------------------------------------------------------------
   // For future expandability...
   // ------------------------------------------------------------------
   if (MIXER_SETCONTROLDETAILSF_VALUE !=
         (fdwDetails & MIXER_SETCONTROLDETAILSF_QUERYMASK)) {
         return ( MMSYSERR_NOTSUPPORTED ) ;
   }

   // -----------------------------------------------------------------
   //  verify that the control id is not bogus...  If it isn't, get
   //  the control.
   // -----------------------------------------------------------------
   uControlId = LOWORD(pmxcd->dwControlID);
   uSource    = HIWORD(pmxcd->dwControlID);
   if (uSource == DEST_WORD_MASK) {
      if (uControlId >= pK1212Driver->GetNumDestMixControls(cardRefNum)) {
         return (MMSYSERR_INVALPARAM);
      }
      else {
         lpMixControl = pK1212Driver->GetMasterVolumeControl(cardRefNum);
      }
   }
   else {
      if (uSource >= NUM_MIX_SOURCES) {
         return (MMSYSERR_INVALPARAM);
      }
      if (uControlId >= pK1212Driver->GetNumSourceMixControls(cardRefNum, uSource)) {
         return (MMSYSERR_INVALPARAM);
      }
      lpMixControl = pK1212Driver->GetMixControl(cardRefNum,
                                                 uSource,
                                                 uControlId
                                   );
   }

   switch (lpMixControl->GetControlType()) {

      case MIXERCONTROL_CONTROLTYPE_MUTE:
      {
         LPMIXERCONTROLDETAILS_BOOLEAN pmxcd_f;
               
         pmxcd_f = (LPMIXERCONTROLDETAILS_BOOLEAN) pmxcd->paDetails;

         if (pmxcd_f->fValue) {
            pK1212Driver->MuteWaveDevice(cardRefNum,
                                         uSource
                          );
         }
         else {
            pK1212Driver->UnmuteWaveDevice(cardRefNum,
                                           uSource
                          );
         }
      }
      break;

      case MIXERCONTROL_CONTROLTYPE_VOLUME:
      {
         LPMIXERCONTROLDETAILS_SIGNED  pmxcd_s;

         // --------------------------------------------------------
         //  signed values only.
         // --------------------------------------------------------
         pmxcd_s = (LPMIXERCONTROLDETAILS_SIGNED)pmxcd->paDetails;

         if (uSource == DEST_WORD_MASK) {
            pK1212Driver->SetWaveMasterVolume(cardRefNum,
                                              (WORD)pmxcd_s->lValue
                          );
         }
         else {
            pK1212Driver->SetWaveDeviceVolume(cardRefNum,
                                              uSource,
                                              uControlId,
                                              (WORD)pmxcd_s->lValue
                          );
         }
      }
      break;

      default:
      {
         return (MMSYSERR_INVALPARAM);
      }
   }

   // ----------------------------------------------------------------------
   //  !!! WARNING DANGER WARNING DANGER WARNING DANGER Will Robinson !!!
   //
   //  all notification callbacks for mixer drivers must NEVER occur at
   //  interrupt time. so, even though documenation states that the
   //  DriverCallback() function can be called at interrupt time, this
   //  is NOT true for mixer drivers. ALWAYS perform driver callbacks
   //  for control and line changes at non-interrupt time.
   //
   //  !!! WARNING DANGER WARNING DANGER WARNING DANGER WARNING DANGER !!!
   // ----------------------------------------------------------------------

   // ------------------------------------------------------------
   // Walk the list for the mixer device and notify everyone...
   // ------------------------------------------------------------

   lpCur = pK1212Driver->GetMixerClient(cardRefNum,
                                        deviceId,
                                        0
                         );
   while (lpCur) {
      DriverCallback(lpCur->dwCallback,
                     lpCur->fuCallback,
                     lpCur->hmx,
                     MM_MIXM_CONTROL_CHANGE,
                     lpCur->dwInstance,
                     pmxcd->dwControlID,
                     0L
      );
      lpCur = pK1212Driver->GetMixerClient(cardRefNum,
                                           deviceId,
                                           lpCur
                            );
   }

   return MMSYSERR_NOERROR;

} // MxdSetControlDetails()


//--------------------------------------------------------------------------;
//
//  DWORD mxdMessage
//
//  Description:
//
//
//  Arguments:
//      UINT uMsg:
//
//      DWORD dwUser:
//
//      DWORD dwParam1:
//
//      DWORD dwParam2:
//
//  Return (DWORD):
//
//
//--------------------------------------------------------------------------;
DWORD FAR PASCAL _loadds mxdMessage
(
    UINT            uDevId,
    WORD            uMsg,
    DWORD           dwUser,
    DWORD           dwParam1,
    DWORD           dwParam2
)
{
   PMXDCLIENT    pmxdc;
   k1212CardRef  cardRefNum;

   // --------------------------------------------
   // take care of init time messages...
   // --------------------------------------------
   switch (uMsg)
   {
      case MXDM_INIT:
         // DebugMsg((char __far *)"Got the MXDM_INIT message");
         return (MMSYSERR_NOERROR);

      case DRVM_ENABLE:
         // DebugMsg((char __far *)"Got the MXDM_ENABLE message");
         return (MMSYSERR_NOERROR);

      case DRVM_DISABLE:
         return (MMSYSERR_NOERROR);

      case DRVM_EXIT:
         return (MMSYSERR_NOERROR);

      case MXDM_GETNUMDEVS:
         // -------------------------------------------------------------
         // verify that the device node refers to a card recognized
         // by the VxD.  If it does, then there are k1212NumWaveDevices,
         // otherwise 0.
         // -------------------------------------------------------------
         // DebugMsg((char __far *)"Got the MXDM_GETNUMDEVS message");
         if (verifyDevNode(dwParam1, cardRefNum) == TRUE) {
            return 1;
         }
         else {
            return 0;
         }

      case MXDM_OPEN:
         // DebugMsg((char __far *)"Got the MXDM_OPEN message");
         if (uDevId >= 1) {
            return MMSYSERR_BADDEVICEID;
         }
         if (verifyDevNode(((LPMIXEROPENDESC)dwParam1)->dnDevNode, 
                           cardRefNum
             ) == TRUE) {
            return MxdOpen((LPDWORD)dwUser,
                           (WORD)cardRefNum,
                           uDevId,
                           (LPMIXEROPENDESC) dwParam1,
                           dwParam2
                   );
         }
         else {
            return MMSYSERR_INVALHANDLE;
         }
         break;
         
      case MXDM_GETDEVCAPS:
         // ----------------------------------------------------------
         // verify the device node and device ID are valid.  If they
         // are, fill in the device capabilities structure and return.
         // ----------------------------------------------------------
         // DebugMsg((char __far *)"Got the MXDM_GETDEVCAPS message");
         if (uDevId >= 1) {
            return MMSYSERR_BADDEVICEID;
         }
         if (verifyDevNode(dwParam2, cardRefNum) == TRUE) {
            mxdGetDevCaps((WORD)cardRefNum,
                          uDevId,
                          (MDEVICECAPSEX FAR*)dwParam1
            );
            return MMSYSERR_NOERROR;
         }
         else {
            return MMSYSERR_BADDEVICEID;
         }
         break;
   }

   // ------------------------------------------------------------------
   // make sure we are dealing with good device identification before
   // continuing any further
   // ------------------------------------------------------------------
   if (uDevId >= 1) {
      return MMSYSERR_BADDEVICEID;
   }
   pmxdc = (PMXDCLIENT)LOWORD(dwUser);
   if (pmxdc == NULL) {  
      return MMSYSERR_BADDEVICEID;
   }

   switch (uMsg)
   {
      case MXDM_CLOSE:
         // DebugMsg((char __far *)"Got the MXDM_CLOSE message");
         return (MxdClose(pmxdc));

      case MXDM_GETLINEINFO:
         // DebugMsg((char __far *)"Got the MXDM_GETLINEINFO message");
         return MxdGetLineInfo(pmxdc, 
                               (LPMIXERLINE)dwParam1, 
                               dwParam2
                );

      case MXDM_GETLINECONTROLS:
         // DebugMsg((char __far *)"Got the MXDM_GETLINECONTROLS message");
         return MxdGetLineControls(pmxdc, 
                                   (LPMIXERLINECONTROLS)dwParam1, 
                                   dwParam2
                );

      case MXDM_GETCONTROLDETAILS:
         // DebugMsg((char __far *)"Got the MXDM_GETCONTROLDETAILS message");
         return MxdGetControlDetails(pmxdc, 
                                     (LPMIXERCONTROLDETAILS)dwParam1, 
                                     dwParam2
                );

      case MXDM_SETCONTROLDETAILS:
         // DebugMsg((char __far *)"Got the MXDM_SETCONTROLDETAILS message");
         return MxdSetControlDetails(pmxdc, 
                                     (LPMIXERCONTROLDETAILS)dwParam1, 
                                     dwParam2
                );
   }

   return (MMSYSERR_NOTSUPPORTED);

} // mxdMessage()

