//*****************************************************************************
//* Name:
//*      wid.c
//*
//* Project:
//*      1212 I/O Driver
//*
//* Author:
//*      Bill Jenkins
//*
//* Description: 
//*      Wave input device message handling.
//*
//* Modification Log:
//*
//*      1.3   6/16/97 Bill
//*      (v 1.10)  Commented out DebugMsg lines.
//*
//*      1.3   6/13/97 Bill
//*      (v 1.06)  WIM_DONE procedure now marks the buffers as done and
//*      no longer queued, rather than the VxD.
//*
//*      1.2   6/11/97 Bill
//*      (v 1.04)  Modified Alertbox calls to aid in debugging while
//*      performing MS compatibility tests.
//*
//*      1.1   1/20/97 Bill
//*      Initial version.  
//*
//*
//* Copyright (c) 1997 Korg, Inc.
//* All rights reserved.
//*
//* This program is protected as an unpublished work under the U.S.
//* copyright laws.  The above copyright notice is not intended to
//* effect a publication of this work.
//*
//* This program is the confidential and proprietary information of
//* Korg and all its subsidiaries.
//*
//* Note: This file was originally adapted from the Microsoft Windows
//*       95 DDK.  Copyright (c) 1991 - 1995  Microsoft Corporation.
//*****************************************************************************

#include <windows.h>
#include <string.h>
#include "gsound.h"
#include "12pmapic.h"
#include "driver.h"
#include "1212AppyTime.h"


//--------------------------------------------------------------------------
// Globals
//--------------------------------------------------------------------------
DWORD gdwNumWidOpens  = 0;
DWORD gdwNumWidCloses = 0;

// ------------------------------------------------------
// Forward declarations
// ------------------------------------------------------
DWORD NEAR PASCAL widClose
(
    DWORD   dwUser,
    DWORD   devNode,
    UINT    uDevId
);
DWORD NEAR PASCAL widOpen
(
    DWORD               dwUser,
    DWORD               devNode,
    UINT                uDevId,
    LPWAVEOPENDESC      lpwid,
    DWORD               dwOpenFlags
);
DWORD NEAR PASCAL widAddBuffer
(
    DWORD         cardRefNum,
    UINT          uDevId,
    NPWAVEALLOC   pClient,
    LPWAVEHDR     lpWaveHdr
);
DWORD NEAR PASCAL widStartDevice
(
    DWORD  cardRefNum,
    UINT   uDevId
);
DWORD NEAR PASCAL widStopDevice
(
    DWORD  cardRefNum,
    UINT   uDevId
);
DWORD NEAR PASCAL widResetDevice
(
    DWORD  cardRefNum,
    UINT   uDevId
);
DWORD NEAR PASCAL widGetPosition
(
    DWORD    cardRefNum,
    UINT     uDevId,
    LPMMTIME mmTime
);



//------------------------------------------------------------------------
//  DWORD FAR PASCAL _loadds widMessage( WORD id, WORD msg,
//                                       DWORD dwUser, DWORD dwParam1,
//                                       DWORD dwParam2 )
//
//  Description:
//     This function conforms to the standard wave input driver message
//     procedure (widMessage), which is documented in mmddk.h.
//
//  Parameters:
//     WORD  uDevId
//     WORD  msg
//     DWORD dwUser
//     DWORD dwParam1
//     DWORD dwParam2
//
//  Return Value:
//     DWORD
//        message specific
//
//------------------------------------------------------------------------

DWORD FAR PASCAL _loadds widMessage
(
    UINT   uDevId,
    WORD   msg,
    DWORD  dwUser,
    DWORD  dwParam1,
    DWORD  dwParam2
)
{
   NPWAVEALLOC   pClient;
   k1212CardRef  cardRefNum;
   DWORD         devNode;

   #ifdef DEBUGMSG
      char          string[40];        // for debug strings
   #endif

   // ----------------------------------------
   // Take care of init time messages...
   //-----------------------------------------
   switch (msg)
   {
      case WIDM_INIT:
         #ifdef DEBUGMSG
            DebugMsg((char __far *)"WIDM_INIT \n");
         #endif
         // dwParam2 == PnP DevNode
         return(MMSYSERR_NOERROR);
         break;

      case DRVM_ENABLE:
         #ifdef DEBUGMSG
            DebugMsg((char __far *)"WIDM_ENABLE \n");
         #endif
         // dwParam2 == PnP DevNode
         if (verifyDevNode(dwParam2, cardRefNum)) {
            if (pK1212Driver) {
               if (pK1212Driver->EnableWaveInDevs(cardRefNum) == K1212_CMDRET_Success) {
                  return (MMSYSERR_NOERROR);
               }
               else {
                  return (MMSYSERR_ERROR);
               }
            }
            else {
               return (MMSYSERR_NODRIVER);
            }
         }
         else {
            return (MMSYSERR_BADDEVICEID);
         }
         break;

      case DRVM_DISABLE:
         #ifdef DEBUGMSG
            DebugMsg((char __far *)"WIDM_DISABLE \n");
         #endif
         // dwParam2 == PnP DevNode
         if (verifyDevNode(dwParam2, cardRefNum)) {
            if (pK1212Driver) {
               if (pK1212Driver->DisableWaveInDevs(cardRefNum) == K1212_CMDRET_Success) {
                  return (MMSYSERR_NOERROR);
               }
               else {
                  return (MMSYSERR_ERROR);
               }
            }
            else {
               return (MMSYSERR_NODRIVER);
            }
         }
         else {
            return (MMSYSERR_BADDEVICEID);
         }
         return (MMSYSERR_NOERROR);
         break;

      case DRVM_EXIT:
         #ifdef DEBUGMSG
            DebugMsg((char __far *)"WIDM_EXIT \n");
         #endif
         // dwParam2 == PnP DevNode
         return (MMSYSERR_NOERROR);
         break;

      case WIDM_GETNUMDEVS:
         // ---------------------------------------------------------
         // verify that the devive node refers to a card recognized
         // by the VxD.  If it does, then there are 
         // NUM_1212IO_WAVE_DEVICES, otherwise 0.
         // ---------------------------------------------------------
         if (verifyDevNode(dwParam1, cardRefNum) == TRUE) {
            return k1212NumWaveDevices;
         }
         else {
            return 0;
         }
         break;

      case WIDM_OPEN:
         if (uDevId >= k1212NumWaveDevices) {
            return MMSYSERR_BADDEVICEID;
         }
         devNode = ((LPWAVEOPENDESC)dwParam1)->dnDevNode;
         return widOpen(dwUser,
                        devNode,
                        uDevId,
                        (LPWAVEOPENDESC) dwParam1,
                        dwParam2
                );
         break;

      case WIDM_GETDEVCAPS:
         // ----------------------------------------------------------
         // verify the devive node and device ID are valid.  If they
         // are, fill in the device capabilities structure and return.
         // ----------------------------------------------------------
         if (uDevId >= k1212NumWaveDevices) {
            return MMSYSERR_BADDEVICEID;
         }
         if (verifyDevNode(dwParam2, cardRefNum) == TRUE) {
            widGetDevCaps(uDevId,
                          cardRefNum,
                          (MDEVICECAPSEX FAR*)dwParam1 
            );
            return MMSYSERR_NOERROR;
         }
         else {
            return MMSYSERR_BADDEVICEID;
         }
         break;
   }


   // ------------------------------------------------------------------
   // make sure we are dealing with good device identification before
   // continuing any further
   // ------------------------------------------------------------------
   if (uDevId >= k1212NumWaveDevices) {
      return MMSYSERR_BADDEVICEID;
   }
   pClient = (NPWAVEALLOC)LOWORD(dwUser);
   devNode = pClient->dwDevNode;
   if (verifyDevNode(devNode, cardRefNum) != TRUE) {    // assigns cardRefNum
      return MMSYSERR_BADDEVICEID;
   }


   switch (msg)
   {
      case WIDM_CLOSE:
         #ifdef DEBUGMSG
            DebugMsg((char __far *)"WIDM_CLOSE \n");
         #endif
         return( widClose(dwUser,
                          devNode,
                          uDevId
                 )
         );

      case WIDM_ADDBUFFER:
         #ifdef DEBUGMSG
            wsprintf(string, "WIDM_WRITE %04x%04x \n", HIWORD(dwParam1), LOWORD(dwParam1));
            DebugMsg((char __far *)string);
         #endif
         return (widAddBuffer(cardRefNum,
                              uDevId,
                              pClient,
                              (LPWAVEHDR)dwParam1
                 )
         );

      case WIDM_START:
         #ifdef DEBUGMSG
            DebugMsg((char __far *)"WIDM_START \n");
         #endif
         return (widStartDevice(cardRefNum,
                                uDevId
                 )
         );

      case WIDM_STOP:
         #ifdef DEBUGMSG
            DebugMsg((char __far *)"WIDM_STOP \n");
         #endif
         return (widStopDevice(cardRefNum,
                               uDevId
                 )
         );

      case WIDM_RESET:
         #ifdef DEBUGMSG
            DebugMsg((char __far *)"WIDM_RESET \n");
         #endif
         return (widResetDevice(cardRefNum,
                                uDevId
                 )
         );

      case WIDM_GETPOS:
         #ifdef DEBUGMSG
            DebugMsg((char __far *)"WIDM_GETPOS \n");
         #endif
         return (widGetPosition(cardRefNum,
                                uDevId,
                                (LPMMTIME)dwParam1
                 )
         );


   }

   // --------------------------------------------------------------
   // If we got here, we don't know what they're talking about...
   // --------------------------------------------------------------
   #ifdef DEBUGMSG
      // wsprintf(string, "unknown WIDM %d \n", msg);
      // DebugMsg((char __far *)string);
   #endif

   return ( MMSYSERR_NOTSUPPORTED );

} // end of widMessage()


//------------------------------------------------------------------------
//  VOID widGetDevCaps
//
//  Description:
//     Get the device capabilities.
//
//  Parameters:
//     UINT                uDevId
//        which device on the card to get the capabilities of.
//
//     k1212CardRef        cardRefNum
//        which card to get the capabilities of.  This is used to
//        get the bus and device numbers of the card for identification
//        purposes.
//
//     MDEVICECAPSEX FAR *lpCaps
//        far pointer to a MDEVICECAPSEX structure to receive
//        the information
//
//  Return Value:
//     Nothing.
//
//------------------------------------------------------------------------

VOID FAR PASCAL widGetDevCaps
(
    UINT                uDevId,
    k1212CardRef        cardRefNum,
    MDEVICECAPSEX FAR*  lpCaps
)
{
   WAVEINCAPS           wc;
   char                 devName[MAXPNAMELEN];

   // --------------------------------------------------------------
   // fill in the device capability structure
   // --------------------------------------------------------------
   wc.wMid           = MM_KORG;
   wc.wPid           = MM_KORG_PCIF_MIDIIN;
   wc.vDriverVersion = DRV_VERSION;
   wc.dwFormats      = (WAVE_FORMAT_4S16 | WAVE_FORMAT_4M16 | WAVE_FORMAT_4S08 | WAVE_FORMAT_4M08 |
                        WAVE_FORMAT_2S16 | WAVE_FORMAT_2M16 | WAVE_FORMAT_2S08 | WAVE_FORMAT_2M08 |
                        WAVE_FORMAT_1S16 | WAVE_FORMAT_1M16 | WAVE_FORMAT_1S08 | WAVE_FORMAT_1M08);
   wc.wChannels      = 2;
   
   // --------------------------------------------------------------
   // create the device name, and copy it into the dev cap structure
   // --------------------------------------------------------------
   pK1212Driver->GetDeviceName(cardRefNum,
                               (WORD)uDevId,
                               devName
                 );
   lstrcpyn(wc.szPname, 
            devName, 
            (MAXPNAMELEN-1)
   );

   _fmemcpy(lpCaps->pCaps, 
            &wc,
            min((UINT)lpCaps->cbSize, 
                sizeof(wc) 
            ) 
   );
} // end of widGetDevCaps()


//------------------------------------------------------------------------
//  DWORD widOpen
//
//  Description:
//     Handles the WIDM_OPEN message.
//
//  Parameters:
//     DWORD dwUser
//        pointer to return handle
//
//     DWORD devNode
//        device node of the card
//
//     UINT uDevId
//        specifies which wave device on the card
//
//     LPWAVEOPENDESC lpwid
//        contains a pointer to a WAVEOPENDESC
//
//     DWORD dwOpenFlags
//        LOWORD contains wave driver specific flags
//        HIWORD contains generic driver flags
//
//  Return Value:
//     DWORD
//        MMSYS_NOERROR if no error occurred otherwise
//          specific error code for problem
//
//------------------------------------------------------------------------

DWORD NEAR PASCAL widOpen
(
    DWORD           dwUser,
    DWORD           devNode,
    UINT            uDevId,
    LPWAVEOPENDESC  lpwid,
    DWORD           dwOpenFlags
)
{
   k1212CardRef            cardRefNum;  // reference number of the specified card
   k1212WaveFormatData     formatData;  // passed to the VxD to verify the format
   k1212DeviceOpenData     devOpenData; // passed to the VxD to open the device
   const WAVEFORMATEX FAR* lpFmt;       // pointer to passed format
   NPWAVEALLOC             pClient;     // pointer to client information
                                        //    structure
   BYTE                    bOK;         // TRUE if the format is OK.

   // -------------------------------
   // Assume everything's valid
   // -------------------------------
   bOK = TRUE;

   // ----------------------------------
   // Make sure we can handle the format
   // ----------------------------------
   lpFmt = (WAVEFORMATEX FAR*)lpwid->lpFormat;

   switch (lpFmt->wFormatTag)
   {
      case WAVE_FORMAT_PCM:

         bOK = FALSE;         // only gets set to TRUE if the VxD verifies the format
         if (pK1212Driver) {
            if (verifyDevNode(devNode, cardRefNum) == TRUE) {
               formatData.sampleRate    = lpFmt->nSamplesPerSec;
               formatData.bitsPerSample = ((LPPCMWAVEFORMAT)lpFmt)->wBitsPerSample;
               formatData.nChannels     = lpFmt->nChannels;
               if (pK1212Driver->VerifyWaveInFormat(cardRefNum,
                                                    uDevId,
                                                    formatData
                                 ) == K1212_CMDRET_Success) {
                  bOK = TRUE;
               }
            }
         }
         else {
            return MMSYSERR_NODRIVER;
         }
         break;

      default:
         // unsupported format
         bOK = FALSE;
         break;
   }

   if (!bOK)
      return (WAVERR_BADFORMAT);

   // ---------------------------------------------
   // did they just want format information?
   // ---------------------------------------------
   if (dwOpenFlags & WAVE_FORMAT_QUERY) {
      return MMSYSERR_NOERROR;
   }
   #ifdef DEBUGMSG
      DebugMsg((char __far *)"WIDM_OPEN \n");
   #endif

   // ---------------------------------------------
   // allocate my per-client structure
   // ---------------------------------------------
   pClient = (NPWAVEALLOC)LocalAlloc(LPTR, sizeof(WAVEALLOC));
   if (pClient == NULL) {
      return MMSYSERR_NOMEM;
   }

   // ---------------------------------------------
   // and fill it with info
   // ---------------------------------------------
   pClient->dwCallback = lpwid->dwCallback;
   pClient->dwInstance = lpwid->dwInstance;
   pClient->hWave      = lpwid->hWave;
   pClient->dwFlags    = dwOpenFlags;
   pClient->dwDevNode  = devNode;
   pClient->pcmwf      = *((LPPCMWAVEFORMAT)lpFmt);

   // ---------------------------------------------
   // give the client the driver id
   // ---------------------------------------------
   *((LPDWORD)dwUser) = (DWORD)(LPWAVEALLOC)pClient;

   // ---------------------------------------------
   // open the device with the VxD
   // ---------------------------------------------
   devOpenData.waveFormatParams      = formatData;
   devOpenData.driverCallbackAddress = (DWORD)WidVxDCallback;
   devOpenData.clientHandle          = *((LPDWORD)dwUser);

   switch (pK1212Driver->OpenWaveInDevice(cardRefNum,
                                          uDevId,
                                          devOpenData
                         )) {
      case K1212_CMDRET_Success:
         // ---------------------------------------------
         // call client's callback
         // ---------------------------------------------
         waveCallBack(pClient, WIM_OPEN, 0L);
         break;
         
      case K1212_CMDRET_FailBadState:
         #ifdef DEBUGMSG
            DebugMsg((char __far *)"OPEN failed - state\n");
         #endif
         LocalFree((LOCALHANDLE)pClient);
         return MMSYSERR_ALLOCATED;
      
      default:
         #ifdef DEBUGMSG
            DebugMsg((char __far *)"OPEN failed - default\n");
         #endif
         LocalFree((LOCALHANDLE)pClient);
         return MMSYSERR_ERROR;
   }

   return ( MMSYSERR_NOERROR );
} // end of widOpen()


//------------------------------------------------------------------------
//  DWORD NEAR PASCAL widClose( DWORD dwUser )
//
//  Description:
//     Closes a wave-in device.
//
//  Return Value:
//     DWORD
//        error value
//
//------------------------------------------------------------------------
DWORD NEAR PASCAL widClose
(
    DWORD  dwUser,
    DWORD  devNode,
    UINT   uDevId
)
{
   NPWAVEALLOC         pClient ;
   k1212CardRef        cardRefNum;  // reference number of the specified card

   // --------------------------------------------
   // close the device with the VxD
   // --------------------------------------------
   if (pK1212Driver) {
      if (verifyDevNode(devNode, cardRefNum) == TRUE) {
         switch (pK1212Driver->CloseWaveInDevice(cardRefNum,
                                                 uDevId
                               )) {
            case K1212_CMDRET_Success:
               break;
               
            case K1212_CMDRET_FailBadState:
               return (WAVERR_STILLPLAYING);
               
            default:
               return (MMSYSERR_ERROR);
         }
      }
   }

   // ---------------------------------------------
   // Notify client that it is closing.
   // ---------------------------------------------
   pClient = (NPWAVEALLOC)LOWORD(dwUser);
   waveCallBack(pClient, WIM_CLOSE, 0L);

   // ---------------------------------------------
   // free the allocated memory
   // ---------------------------------------------
   LocalFree((LOCALHANDLE)pClient);

   return MMSYSERR_NOERROR;
} // end of widClose()


//------------------------------------------------------------------------
//  VOID FAR PASCAL widAddBuffer()
//
//  Description:
//     adds a buffer to the wave device's recording queue.
//
//  Return Value:
//     error code based on outcome of VxD call.
//
//------------------------------------------------------------------------
DWORD NEAR PASCAL widAddBuffer
(
    DWORD         cardRefNum,
    UINT          uDevId,
    NPWAVEALLOC   pClient,
    LPWAVEHDR     lpWaveHdr
)
{
   k1212WaveBufferInfo waveBufInfo;

   // -----------------------------------------------------------
   // make sure the header has been prepared
   // -----------------------------------------------------------
   if (!(lpWaveHdr->dwFlags & WHDR_PREPARED)) {
      return WAVERR_UNPREPARED;
   }

   // -----------------------------------------------------------
   // store the pointer to my WAVEALLOC structure and the bufer's
   // address in the wave info structure
   // -----------------------------------------------------------
   waveBufInfo.waveBufferPtr = lpWaveHdr;
   waveBufInfo.lpClient      = (LPVOID)(LPWAVEALLOC)pClient;

   // -----------------------------------------------------------
   // call the VxD to add the buffer to its record buffer queue
   // -----------------------------------------------------------
   if (pK1212Driver) {
      switch (pK1212Driver->AddRecordBuffer(cardRefNum,
                                            uDevId,
                                            waveBufInfo
                            )) {

         case K1212_CMDRET_Success:
            return MMSYSERR_NOERROR;

         case K1212_CMDRET_FailBadState:
            #ifdef DEBUGMSG
               DebugMsg((char __far *)"Write failed - bad state\n");  // v1.03
            #endif
            return MMSYSERR_NOTENABLED;

         default:
            #ifdef DEBUGMSG
               DebugMsg((char __far *)"Write failed - default retcode\n");  // v1.03
            #endif
            return MMSYSERR_ERROR;
      }
   }
   else {
      return MMSYSERR_NODRIVER;
   }
}


//------------------------------------------------------------------------
//  VOID FAR PASCAL widStartDevice()
//
//  Description:
//     starts the wave device recording.
//
//  Return Value:
//     error code based on outcome of VxD call.
//
//------------------------------------------------------------------------
DWORD NEAR PASCAL widStartDevice
(
    DWORD  cardRefNum,
    UINT   uDevId
)
{
   if (pK1212Driver) {
      switch (pK1212Driver->StartWaveInDevice(cardRefNum,
                                              uDevId
                            )) {

         case K1212_CMDRET_Success:
            #ifdef DEBUGMSG
               // DebugMsg((char __far *)"Start succeeded\n");  // v1.11
            #endif
            return MMSYSERR_NOERROR;

         case K1212_CMDRET_FailBadState:
            #ifdef DEBUGMSG
               DebugMsg((char __far *)"Start failed - bad state\n");  // v1.03
            #endif
            return MMSYSERR_NOTENABLED;

         default:
            #ifdef DEBUGMSG
               DebugMsg((char __far *)"Start failed - default retcode\n");  // v1.03
            #endif
            return MMSYSERR_ERROR;
      }
   }
   else {
      return MMSYSERR_NODRIVER;
   }
}


//------------------------------------------------------------------------
//  VOID FAR PASCAL widStopDevice()
//
//  Description:
//     stops the wave device from recording.
//
//  Return Value:
//     error code based on outcome of VxD call.
//
//------------------------------------------------------------------------
DWORD NEAR PASCAL widStopDevice
(
    DWORD  cardRefNum,
    UINT   uDevId
)
{
   if (pK1212Driver) {
      switch (pK1212Driver->StopWaveInDevice(cardRefNum,
                                             uDevId
                            )) {

         case K1212_CMDRET_Success:
            return MMSYSERR_NOERROR;

         default:
            #ifdef DEBUGMSG
               DebugMsg((char __far *)"Stop failed\n");  // v1.03
            #endif
            return MMSYSERR_ERROR;
      }
   }
   else {
      return MMSYSERR_NODRIVER;
   }
}


//------------------------------------------------------------------------
//  VOID FAR PASCAL widResetDevice()
//
//  Description:
//     resets the wave device - recording stops, adn all buffers are released.
//
//  Return Value:
//     error code based on outcome of VxD call.
//
//------------------------------------------------------------------------
DWORD NEAR PASCAL widResetDevice
(
    DWORD  cardRefNum,
    UINT   uDevId
)
{
   if (pK1212Driver) {
      switch (pK1212Driver->ResetWaveDevice(cardRefNum,
                                            uDevId,
                                            K1212_WAVETYPE_INPUT
                            )) {

         case K1212_CMDRET_Success:
            return MMSYSERR_NOERROR;

         default:
            #ifdef DEBUGMSG
               DebugMsg((char __far *)"Reset failed\n");  // v1.03
            #endif
            return MMSYSERR_ERROR;
      }
   }
   else {
      return MMSYSERR_NODRIVER;
   }
}


//------------------------------------------------------------------------
//  VOID FAR PASCAL widGetPosition()
//
//  Description:
//     returns the current recording position into the passed MMTIME 
//     structure.
//
//  Return Value:
//     error code based on outcome of VxD call.
//
//------------------------------------------------------------------------
DWORD NEAR PASCAL widGetPosition
(
    DWORD    cardRefNum,
    UINT     uDevId,
    LPMMTIME mmTime
)
{
   #ifdef DEBUGMSG
      char string[40];
   #endif

   if (pK1212Driver) {
      switch (pK1212Driver->GetWavePosition(cardRefNum,
                                            uDevId,
                                            K1212_WAVETYPE_INPUT,
                                            *mmTime
                            )) {

         case K1212_CMDRET_Success:
            #ifdef DEBUGMSG
               // wsprintf(string, "WID Get Pos %d %u\n", mmTime->wType, mmTime->u.sample);
               // DebugMsg((char __far *)string);
            #endif
            return MMSYSERR_NOERROR;

         default:
            #ifdef DEBUGMSG
               DebugMsg((char __far *)"WID Get Pos failed \n");
            #endif
            return MMSYSERR_ERROR;
      }
   }
   else {
      return MMSYSERR_NODRIVER;
   }
}


//------------------------------------------------------------------------
//  VOID FAR PASCAL WidVxDCallback( WORD request, LPVOID lpRequestParams )
//
//  Description:
//     Callback procedure.  This procedure gets called by the 1212 I/O
//     VxD at AppyTime.
//
//  Parameters:
//     WORD request
//        indicates what the VxD wants from us
//     LPVOID lpRequestParams
//        points to data required by the request (or is the data for
//        requests requiring only a DWORD's worth of data).
//
//  Return Value:
//     none
//
//------------------------------------------------------------------------
VOID FAR PASCAL WidVxDCallback
(
   DWORD       request,
   LPWAVEALLOC lpClient,
   LPVOID      lpRequestParams
)
{
   LPWAVEHDR lpWaveHdr;

   switch (request) {
      case K1212_RETURN_WAVEINBUF:

         // ---------------------------------------------------------
         // mark the buffer as no longer queued and done
         // ---------------------------------------------------------
         lpWaveHdr           = (LPWAVEHDR)lpRequestParams;
         lpWaveHdr->dwFlags &= ~WHDR_INQUEUE;
         lpWaveHdr->dwFlags |= WHDR_DONE;
         
         // ----------------------------------------------
         // call client's callback
         // ----------------------------------------------
         waveCallBack(lpClient, 
                      WIM_DATA,
                      (DWORD)lpRequestParams
         );
         break;

      default:
         // ---------------------------------------------------------------
         // unsupported command from the VxD
         // ---------------------------------------------------------------
         break;
   }
}
