//*****************************************************************************
//* Name:
//*      wod.cpp
//*
//* Project:
//*      1212 I/O Driver
//*
//* Author:
//*      Bill Jenkins
//*
//* Description: 
//*      Wave output device message handling.
//*
//* Modification Log:
//*
//*      1.7   6/16/97 Bill
//*      (v 1.10)  Commented out DebugMsg lines.
//*
//*      1.6   6/13/97 Bill
//*      (v 1.06)  WOM_DONE procedure now marks the buffers as done and
//*      no longer queued, rather than the VxD.
//*
//*      1.5   6/04/97 Bill
//*      (v. 1.05B) Added BREAKLOOP support.  
//*
//*      1.4   6/04/97 Bill
//*      (v. 1.04B) Modifications to meet WHQL compatibility requirements.  
//*
//*      1.3   6/04/97 Bill
//*      (v. 1.02B) Added calls to the VxD for the get and set wave volume
//*      messages.  
//*
//*      1.2   6/03/97 Bill
//*      (v 1.01B) Uncommented DebugMsg messages so that they can be placed
//*      in a file.  This is done to aid in debugging integration issues with
//*      Cakewalk.
//*
//*      1.1   1/20/97 Bill
//*      Initial version.  
//*
//*
//* Copyright (c) 1997 Korg, Inc.
//* All rights reserved.
//*
//* This program is protected as an unpublished work under the U.S.
//* copyright laws.  The above copyright notice is not intended to
//* effect a publication of this work.
//*
//* This program is the confidential and proprietary information of
//* Korg and all its subsidiaries.
//*
//* Note: This file was originally adapted from the Microsoft Windows
//*       95 DDK.  Copyright (c) 1991 - 1995  Microsoft Corporation.
//*****************************************************************************

#include <windows.h>
#include <string.h>
#include "gsound.h"
#include "12pmapic.h"
#include "driver.h"
#include "1212AppyTime.h"

//--------------------------------------------------------------------------
// Globals
//--------------------------------------------------------------------------
DWORD gdwNumWodOpens  = 0;
DWORD gdwNumWodCloses = 0;

// ------------------------------------------------------
// Forward declarations
// ------------------------------------------------------
DWORD NEAR PASCAL wodClose
(
    DWORD dwUser,
    DWORD devNode,
    UINT  uDevId
);
DWORD NEAR PASCAL wodOpen
(
    DWORD          dwUser,
    DWORD          devNode,
    UINT           uDevId,
    LPWAVEOPENDESC lpwod,
    DWORD          dwOpenFlags
);
DWORD NEAR PASCAL wodWrite
(
    DWORD          cardRefNum,
    UINT           uDevId,
    NPWAVEALLOC    pClient,
    LPWAVEHDR      lpWaveHdr
);
DWORD NEAR PASCAL wodPause
(
    DWORD          cardRefNum,
    UINT           uDevId
);
DWORD NEAR PASCAL wodRestart
(
    DWORD          cardRefNum,
    UINT           uDevId
);
DWORD NEAR PASCAL wodReset
(
    DWORD          cardRefNum,
    UINT           uDevId
);
DWORD NEAR PASCAL wodBreakLoop
(
    DWORD          cardRefNum,
    UINT           uDevId
);
DWORD NEAR PASCAL wodGetPos
(
    DWORD          cardRefNum,
    UINT           uDevId,
    LPMMTIME       mmTime
);


//--------------------------------------------------------------------------
//
//  DWORD wodMessage
//
//  Description:
//      This function conforms to the standard wave output driver
//      message procedure (wodMessage), which is documented in mmddk.d.
//
//  Parameters:
//      UINT uDevId
//      WORD msg
//      DWORD dwUser
//      DWORD dwParam1
//      DWORD dwParam2
//
//  Return (DWORD): message dependent
//
//--------------------------------------------------------------------------

DWORD FAR PASCAL _loadds wodMessage
(
    UINT            uDevId,
    WORD            msg,
    DWORD           dwUser,
    DWORD           dwParam1,
    DWORD           dwParam2
)
{
   k1212CardRef  cardRefNum;
   DWORD         devNode;
   NPWAVEALLOC   pClient;
   #ifdef DEBUGMSG
      char          string[40];        // for debug strings
   #endif

   // --------------------------------------------
   // take care of init time messages...
   // --------------------------------------------
   switch (msg)
   {
      case WODM_INIT:
         #ifdef DEBUGMSG
            DebugMsg((char __far *)"WODM_INIT \n");
         #endif
         return (MMSYSERR_NOERROR);
         break ;

      case DRVM_ENABLE:
         #ifdef DEBUGMSG
            DebugMsg((char __far *)"WODM_ENABLE \n");
         #endif
         // dwParam2 == PnP DevNode
         if (verifyDevNode(dwParam2, cardRefNum)) {
            if (pK1212Driver) {
               if (pK1212Driver->EnableWaveOutDevs(cardRefNum) == K1212_CMDRET_Success) {
                  return (MMSYSERR_NOERROR);
               }
               else {
                  return (MMSYSERR_ERROR);
               }
            }
            else {
               return (MMSYSERR_NODRIVER);
            }
         }
         else {
            return (MMSYSERR_BADDEVICEID);
         }
         return (MMSYSERR_NOERROR) ;
         break ;

      case DRVM_DISABLE:
         #ifdef DEBUGMSG
            // DebugMsg((char __far *)"WODM_DISABLE \n");
            // dwParam2 == PnP DevNode
         #endif
         if (verifyDevNode(dwParam2, cardRefNum)) {
            if (pK1212Driver) {
               if (pK1212Driver->DisableWaveOutDevs(cardRefNum) == K1212_CMDRET_Success) {
                  return (MMSYSERR_NOERROR);
               }
               else {
                  return (MMSYSERR_ERROR);
               }
            }
            else {
               return (MMSYSERR_NODRIVER);
            }
         }
         else {
            return (MMSYSERR_BADDEVICEID);
         }
         return (MMSYSERR_NOERROR) ;
         break ;

      case DRVM_EXIT:
         #ifdef DEBUGMSG
            DebugMsg((char __far *)"WODM_EXIT \n");
         #endif
         // dwParam2 == PnP DevNode
         return (MMSYSERR_NOERROR) ;
         break;

      case WODM_GETNUMDEVS:
         // ---------------------------------------------------------
         // verify that the device node refers to a card recognized
         // by the VxD.  If it does, then there are 
         // NUM_1212IO_WAVE_DEVICES, otherwise 0.
         // ---------------------------------------------------------
         if (verifyDevNode(dwParam1, cardRefNum) == TRUE) {
            return k1212NumWaveDevices;
         }
         else {
            return 0;
         }
         break;
         
      case WODM_OPEN:
         if (uDevId >= k1212NumWaveDevices) {
            return MMSYSERR_BADDEVICEID;
         }
         devNode = ((LPWAVEOPENDESC)dwParam1)->dnDevNode;
         return wodOpen(dwUser,
                        devNode,
                        uDevId,
                        (LPWAVEOPENDESC) dwParam1,
                        dwParam2
                );

      case WODM_GETDEVCAPS:
         // ----------------------------------------------------------
         // verify the device node and device ID are valid.  If they
         // are, fill in the device capabilities structure and return.
         // ----------------------------------------------------------
         #ifdef DEBUGMSG
            // DebugMsg((char __far *)"Got the WODM_GETDEVCAPS message\n");
         #endif
         if (uDevId >= k1212NumWaveDevices) {
            return MMSYSERR_BADDEVICEID;
         }
         if (verifyDevNode(dwParam2, cardRefNum) == TRUE) {
            wodGetDevCaps(uDevId,
                          cardRefNum,
                          (MDEVICECAPSEX FAR *)dwParam1
            );
            return MMSYSERR_NOERROR;
         }
         else {
            return MMSYSERR_BADDEVICEID;
         }
         break;

      case WODM_SETVOLUME:
         #ifdef DEBUGMSG
            DebugMsg((char __far *)"Got the WODM_SETVOLUME message\n");
         #endif
         if (pK1212Driver) {                                //  v1.06
            verifyDevNode(dwParam2, cardRefNum);            // get cardRefNum
            pK1212Driver->SetWaveDeviceVolume(cardRefNum,
                                              uDevId,
                                              LOWORD(dwParam1),
                                              HIWORD(dwParam1)
                          );
         }
         return MMSYSERR_NOERROR;
         break;

      case WODM_GETVOLUME:
         WORD leftVol, rightVol;
         #ifdef DEBUGMSG
            // DebugMsg((char __far *)"Got the WODM_GETVOLUME message\n");
         #endif
         if (pK1212Driver) {                                //  v1.06
            verifyDevNode(dwParam2, cardRefNum);            // get cardRefNum
            pK1212Driver->GetWaveDeviceVolume(cardRefNum,
                                              uDevId,
                                              leftVol,
                                              rightVol
                          );
            *((DWORD FAR *) dwParam1) = MAKELONG(leftVol, rightVol);
         }
         return MMSYSERR_NOERROR;
         break;
   }

   // ------------------------------------------------------------------
   // make sure we are dealing with good device identification before
   // continuing any further
   // ------------------------------------------------------------------
   if (uDevId >= k1212NumWaveDevices) {
      return MMSYSERR_BADDEVICEID;
   }
   pClient = (NPWAVEALLOC)LOWORD(dwUser);
   devNode = pClient->dwDevNode;
   if (verifyDevNode(devNode, cardRefNum) != TRUE) {    // assigns cardRefNum
      return MMSYSERR_BADDEVICEID;
   }

   switch (msg)
   {
      case WODM_CLOSE:
         #ifdef DEBUGMSG
            DebugMsg((char __far *)"WODM_CLOSE \n");
         #endif
         return(wodClose(dwUser,
                         devNode,
                         uDevId
                )
         );
         break;
         
      case WODM_WRITE:
         #ifdef DEBUGMSG
            wsprintf(string, "WODM_WRITE %04x%04x \n", HIWORD(dwParam1), LOWORD(dwParam1));
            DebugMsg((char __far *)string);
         #endif
         // -----------------------------------------------------------
         // write it to the playback buffer queue in the VxD
         // -----------------------------------------------------------
         return(wodWrite(cardRefNum,
                         uDevId,
                         pClient,
                         (LPWAVEHDR)dwParam1
                )
         );
         
      case WODM_PAUSE:
         #ifdef DEBUGMSG
            DebugMsg((char __far *)"WODM_PAUSE \n");
         #endif
         return(wodPause(cardRefNum,
                         uDevId
                )
         );

      case WODM_RESTART:
         #ifdef DEBUGMSG
            DebugMsg((char __far *)"WODM_RESTART \n");
         #endif
         return(wodRestart(cardRefNum,
                           uDevId
                )
         );
         
      case WODM_RESET:
         #ifdef DEBUGMSG
            DebugMsg((char __far *)"WODM_RESET \n");
         #endif
         return(wodReset(cardRefNum,
                         uDevId
                )
         );

      case WODM_BREAKLOOP:
         #ifdef DEBUGMSG
            DebugMsg((char __far *)"WODM_BREAKLOOP \n");
         #endif
         return(wodBreakLoop(cardRefNum,
                             uDevId
                )
         );
         break;
         
      case WODM_GETPOS:
         #ifdef DEBUGMSG
            DebugMsg((char __far *)"WODM_GETPOS \n");
         #endif
         return(wodGetPos(cardRefNum,
                          uDevId,
                          (LPMMTIME)dwParam1
                )
         );
         
      default:
         return MMSYSERR_NOTSUPPORTED;
   }

   // -----------------------------------------------
   // should never get here...
   // -----------------------------------------------
   return MMSYSERR_NOTSUPPORTED;

} // wodMessage()


//------------------------------------------------------------------------
//  VOID wodGetDevCaps
//
//  Description:
//     Get the device capabilities.
//
//  Parameters:
//     UINT                uDevId
//        which device on the card to get the capabilities of.
//
//     k1212CardRef        cardRefNum
//        which card to get the capabilities of.  This is used to
//        get the bus and device numbers of the card for identification
//        purposes.
//
//     MDEVICECAPSEX FAR *lpCaps
//        far pointer to a MDEVICECAPSEX structure to receive
//        the information
//
//  Return Value:
//     Nothing.
//
//------------------------------------------------------------------------
VOID FAR PASCAL wodGetDevCaps
(
    UINT                uDevId,
    k1212CardRef        cardRefNum,
    MDEVICECAPSEX FAR*  lpCaps
)
{
   WAVEOUTCAPS wc;
   char        devName[MAXPNAMELEN];

   // ----------------------------------------------------------------
   // fill in the device capability structure.  v1.03 -fixed support
   // and formats to OR the appropriate fields.
   // ----------------------------------------------------------------
   wc.wMid           = MM_KORG;
   wc.wPid           = MM_KORG_PCIF_MIDIOUT;
   wc.vDriverVersion = DRV_VERSION;
   wc.dwFormats      = (WAVE_FORMAT_4S16 | WAVE_FORMAT_4M16 | WAVE_FORMAT_4S08 | WAVE_FORMAT_4M08 |
                        WAVE_FORMAT_2S16 | WAVE_FORMAT_2M16 | WAVE_FORMAT_2S08 | WAVE_FORMAT_2M08 |
                        WAVE_FORMAT_1S16 | WAVE_FORMAT_1M16 | WAVE_FORMAT_1S08 | WAVE_FORMAT_1M08);
   wc.wChannels      = 2;
   wc.dwSupport      = (WAVECAPS_VOLUME | WAVECAPS_LRVOLUME | WAVECAPS_SAMPLEACCURATE);

   // --------------------------------------------------------------
   // create the device name, and copy it into the dev cap structure
   // --------------------------------------------------------------
   pK1212Driver->GetDeviceName(cardRefNum,
                               (WORD)uDevId,
                               devName
                 );
   lstrcpyn(wc.szPname, 
            devName, 
            (MAXPNAMELEN-1)
   );
   
   _fmemcpy(lpCaps -> pCaps, 
            &wc,
            min((UINT)lpCaps->cbSize, 
                sizeof(wc) 
            ) 
   );
} // end of wodGetDevCaps()

//------------------------------------------------------------------------
//  LRESULT wodOpen
//
//  Description:
//
//
//  Parameters:
//     DWORD dwUser
//        pointer to return handle
//
//     DWORD devNode
//        device node of the card
//
//     UINT uDevId
//        specifies which wave device on the card
//
//     LPWAVEOPENDESC lpwid
//        contains a pointer to a WAVEOPENDESC
//
//     DWORD dwOpenFlags
//        LOWORD contains wave driver specific flags
//        HIWORD contains generic driver flags
//
//  Return Value:
//     DWORD
//        MMSYS_NOERROR if no error occurred otherwise
//          specific error code for problem
//
//------------------------------------------------------------------------

DWORD NEAR PASCAL wodOpen
(
    DWORD               dwUser,
    DWORD               devNode,
    UINT                uDevId,
    LPWAVEOPENDESC      lpwod,
    DWORD               dwOpenFlags
)
{
   const WAVEFORMATEX FAR  *lpFmt;          // pointer to passed format
   NPWAVEALLOC             pClient;         // pointer to client information structure
   BYTE                    bOK;
   k1212CardRef            cardRefNum;      // reference number of the specified card
   k1212WaveFormatData     formatData;      // passed to the VxD to verify the format
   k1212DeviceOpenData     devOpenData;     // passed to the VxD to open the device
   K1212CmdRet             openResult;      // result of open request to the VxD

   // --------------------------------------------------------------------
   //  dwOpenFlags contains wave driver specific flags in the LOWORD
   //  and generic driver flags in the HIWORD
   // --------------------------------------------------------------------

   // -------------------------------
   // Assume everything's valid
   // -------------------------------
   bOK = TRUE;

   // ----------------------------------
   // Make sure we can handle the format
   // ----------------------------------
   lpFmt = (WAVEFORMATEX FAR*)lpwod->lpFormat;

   switch (lpFmt -> wFormatTag)
   {
      case WAVE_FORMAT_PCM:
         bOK = FALSE;         // only gets set to TRUE if the VxD verifies the format
         if (pK1212Driver) {
            if (verifyDevNode(devNode, cardRefNum) == TRUE) {
               formatData.sampleRate    = lpFmt->nSamplesPerSec;
               formatData.bitsPerSample = ((LPPCMWAVEFORMAT)lpFmt)->wBitsPerSample;
               formatData.nChannels     = lpFmt->nChannels;
               if (pK1212Driver->VerifyWaveOutFormat(cardRefNum,
                                                     uDevId,
                                                     formatData
                                 ) == K1212_CMDRET_Success) {
                  bOK = TRUE;
               }
            }
         }
         else {
            return MMSYSERR_NODRIVER;
         }
         break;

      default:
         // unsupported format
         bOK = FALSE;
         break;
   }

   if (!bOK) {
      return WAVERR_BADFORMAT;
   }

   // --------------------------------------------
   // did they just want format information?
   // --------------------------------------------
   if (dwOpenFlags & WAVE_FORMAT_QUERY) {
      return MMSYSERR_NOERROR;
   }
   #ifdef DEBUGMSG
      DebugMsg((char __far *)"WODM_OPEN \n");
   #endif

   // --------------------------------------------
   // allocate my per-client structure
   // --------------------------------------------
   pClient = (NPWAVEALLOC)LocalAlloc(LPTR, sizeof(WAVEALLOC));
   if (pClient == NULL) {
      return MMSYSERR_NOMEM;
   }

   // --------------------------------------------
   // and fill it with info
   // --------------------------------------------
   pClient->dwCallback = lpwod->dwCallback;
   pClient->dwInstance = lpwod->dwInstance;
   pClient->hWave      = lpwod->hWave;
   pClient->dwFlags    = dwOpenFlags;
   pClient->dwDevNode  = devNode;
   pClient->pcmwf      = *((LPPCMWAVEFORMAT)lpFmt);

   // ---------------------------------------------------------------
   // give the client my driver dw - this stores the pointer to the
   // pClient structure, so we can later free it.
   // ---------------------------------------------------------------
   *((LPDWORD)dwUser) = (DWORD)(LPWAVEALLOC)pClient;

   // ---------------------------------------------
   // open the device with the VxD
   // ---------------------------------------------
   devOpenData.waveFormatParams      = formatData;
   devOpenData.driverCallbackAddress = (DWORD)WodVxDCallback;
   devOpenData.clientHandle          = *((LPDWORD)dwUser);

   openResult = pK1212Driver->OpenWaveOutDevice(cardRefNum,    // v1.03B
                                                uDevId,
                                                devOpenData
                              );
                              
   switch (openResult) {
   
      case K1212_CMDRET_Success:
         // ---------------------------------------------
         // call client's callback
         // ---------------------------------------------
         waveCallBack(pClient, WOM_OPEN, 0L);
         break;
         
      case K1212_CMDRET_FailBadState:
         LocalFree((LOCALHANDLE)pClient);
         return MMSYSERR_ALLOCATED;
         
      default:
         LocalFree((LOCALHANDLE)pClient);
         return MMSYSERR_ERROR;
   }
   return (MMSYSERR_NOERROR);
} // end of wodOpen()


//--------------------------------------------------------------------------
//
//  DWORD wodClose
//
//  Description:
//      Closes the wave-out device.
//
//  Return (DWORD):
//
//--------------------------------------------------------------------------
DWORD NEAR PASCAL wodClose
(
    DWORD           dwUser,
    DWORD           devNode,
    UINT            uDevId
)
{
   NPWAVEALLOC         pClient;
   k1212CardRef        cardRefNum;  // reference number of the specified card

   // --------------------------------------------
   // close the device with the VxD
   // --------------------------------------------
   if (pK1212Driver) {
      if (verifyDevNode(devNode, cardRefNum) == TRUE) {        // v1.03B
         switch (pK1212Driver->CloseWaveOutDevice(cardRefNum,
                                                  uDevId
                               )) {
            case K1212_CMDRET_Success:
               // all is well, continue...
               break;

            case K1212_CMDRET_FailBadState:
               // not finished playback...
               return (WAVERR_STILLPLAYING);

            default:
               return (MMSYSERR_ERROR);
         }
      }
   }

   // --------------------------------------------
   // call client's callback
   // --------------------------------------------
   pClient = (NPWAVEALLOC)LOWORD(dwUser);
   waveCallBack(pClient, WOM_CLOSE, 0L);

   // --------------------------------------------
   // free the allocated memory
   // --------------------------------------------
   LocalFree((LOCALHANDLE)pClient);

   return MMSYSERR_NOERROR;
} // wodClose()


//------------------------------------------------------------------------
//  DWORD NEAR PASCAL wodWrite()
//
//  Description:
//     This function adds a playback buffer to the wave device's queue.
//
//  Return Value:
//     error code based on the success of the VxD call.
//
//------------------------------------------------------------------------
DWORD NEAR PASCAL wodWrite
(
    DWORD          cardRefNum,
    UINT           uDevId,
    NPWAVEALLOC    pClient,
    LPWAVEHDR      lpWaveHdr
)
{
   k1212WaveBufferInfo waveBufInfo;

   // -----------------------------------------------------------
   // make sure the header has been prepared
   // -----------------------------------------------------------
   if (!(lpWaveHdr->dwFlags & WHDR_PREPARED)) {
      #ifdef DEBUGMSG
         DebugMsg((char __far *)"Write failed - buffer unprepared\n");  // v1.03
      #endif
      return WAVERR_UNPREPARED;
   }

   // -----------------------------------------------------------
   // store the pointer to my WAVEALLOC structure and the bufer's
   // address in the wave info structure
   // -----------------------------------------------------------
   waveBufInfo.waveBufferPtr = lpWaveHdr;
   waveBufInfo.lpClient      = (LPVOID)(LPWAVEALLOC)pClient;

   // -----------------------------------------------------------
   // call the VxD to add the buffer to its playback buffer queue
   // -----------------------------------------------------------
   if (pK1212Driver) {
      switch (pK1212Driver->AddPlayBuffer(cardRefNum,
                                          uDevId,
                                          waveBufInfo
                            )) {

         case K1212_CMDRET_Success:
            #ifdef DEBUGMSG
               // DebugMsg((char __far *)"Write succeeded\n");  // v1.03
            #endif
            return MMSYSERR_NOERROR;

         case K1212_CMDRET_FailBadState:
            #ifdef DEBUGMSG
               DebugMsg((char __far *)"Write failed - bad state\n");  // v1.03
            #endif
            return MMSYSERR_NOTENABLED;

         default:
            #ifdef DEBUGMSG
               DebugMsg((char __far *)"Write failed - default retcode\n");  // v1.03
            #endif
            return MMSYSERR_ERROR;
      }
   }
   else {
      return MMSYSERR_NODRIVER;
   }
}


//------------------------------------------------------------------------
//  DWORD NEAR PASCAL wodPause()
//
//  Description:
//     This function calls the VxD to pause the wave output device.
//
//  Return Value:
//     error code based on the success of the VxD call.
//
//------------------------------------------------------------------------
DWORD NEAR PASCAL wodPause
(
    DWORD          cardRefNum,
    UINT           uDevId
)
{
   // -----------------------------------------------------------
   // call the VxD to pause the device
   // -----------------------------------------------------------
   if (pK1212Driver) {
      switch (pK1212Driver->PauseWaveOutDevice(cardRefNum,
                                               uDevId
                            )) {

         case K1212_CMDRET_Success:
            return MMSYSERR_NOERROR;

         default:
            return MMSYSERR_ERROR;
      }
   }
   else {
      return MMSYSERR_NODRIVER;
   }
}


//------------------------------------------------------------------------
//  DWORD NEAR PASCAL wodRestart()
//
//  Description:
//     This function calls the VxD to restart the wave output device.
//
//  Return Value:
//     error code based on the success of the VxD call.
//
//------------------------------------------------------------------------
DWORD NEAR PASCAL wodRestart
(
    DWORD          cardRefNum,
    UINT           uDevId
)
{
   // -----------------------------------------------------------
   // call the VxD to restart the device
   // -----------------------------------------------------------
   if (pK1212Driver) {
      switch (pK1212Driver->ResumeWaveOutDevice(cardRefNum,
                                                uDevId
                            )) {

         case K1212_CMDRET_Success:
            #ifdef DEBUGMSG
               // DebugMsg((char __far *)"Restart succeeded\n");  // v1.03
            #endif
            return MMSYSERR_NOERROR;

         default:
            return MMSYSERR_ERROR;
      }
   }
   else {
      return MMSYSERR_NODRIVER;
   }
}


//------------------------------------------------------------------------
//  DWORD NEAR PASCAL wodReset()
//
//  Description:
//     This function calls the VxD to reset the wave output device.
//
//  Return Value:
//     error code based on the success of the VxD call.
//
//------------------------------------------------------------------------
DWORD NEAR PASCAL wodReset
(
    DWORD          cardRefNum,
    UINT           uDevId
)
{
   // -----------------------------------------------------------
   // call the VxD to reset the device
   // -----------------------------------------------------------
   if (pK1212Driver) {
      switch (pK1212Driver->ResetWaveDevice(cardRefNum,
                                            uDevId,
                                            K1212_WAVETYPE_OUTPUT
                            )) {

         case K1212_CMDRET_Success:
            return MMSYSERR_NOERROR;

         default:
            return MMSYSERR_ERROR;
      }
   }
   else {
      return MMSYSERR_NODRIVER;
   }
}


//------------------------------------------------------------------------
//  DWORD NEAR PASCAL wodBreakLoop()
//
//  Description:
//     This function calls the VxD to break out of a playback loop.
//
//  Return Value:
//     error code based on the success of the VxD call.
//
//------------------------------------------------------------------------
DWORD NEAR PASCAL wodBreakLoop
(
    DWORD          cardRefNum,
    UINT           uDevId
)
{
   // -----------------------------------------------------------
   // call the VxD to reset the device
   // -----------------------------------------------------------
   if (pK1212Driver) {
      switch (pK1212Driver->BreakLoop(cardRefNum,
                                      uDevId
                            )) {

         case K1212_CMDRET_Success:
            return MMSYSERR_NOERROR;

         default:
            return MMSYSERR_ERROR;
      }
   }
   else {
      return MMSYSERR_NODRIVER;
   }
}


//------------------------------------------------------------------------
//  DWORD NEAR PASCAL wodGetPos()
//
//  Description:
//     This function calls the VxD to get the current playback position 
//     of the wave output device.  The mmTime structure is filled in by
//     the VxD.
//
//  Return Value:
//     error code based on the success of the VxD call.
//
//------------------------------------------------------------------------
DWORD NEAR PASCAL wodGetPos
(
    DWORD          cardRefNum,
    UINT           uDevId,
    LPMMTIME       mmTime
)
{
   #ifdef DEBUGMSG
      // char string[40];
   #endif

   // -----------------------------------------------------------
   // call the VxD to reset the device
   // -----------------------------------------------------------
   if (pK1212Driver) {
      switch (pK1212Driver->GetWavePosition(cardRefNum,
                                            uDevId,
                                            K1212_WAVETYPE_OUTPUT,
                                            *mmTime
                            )) {

         case K1212_CMDRET_Success:
            #ifdef DEBUGMSG
               // wsprintf(string, "WOD Get Pos %d %d\n", mmTime->wType, mmTime->u.sample);
               // DebugMsg((char __far *)string);
            #endif
            return MMSYSERR_NOERROR;

         default:
            #ifdef DEBUGMSG
               DebugMsg((char __far *)"WOD Get Pos failed \n");
            #endif
            return MMSYSERR_ERROR;
      }
   }
   else {
      return MMSYSERR_NODRIVER;
   }
}




//------------------------------------------------------------------------
//  VOID FAR PASCAL WodVxDCallback( WORD request, LPVOID lpRequestParams )
//
//  Description:
//     Callback procedure.  This procedure gets called by the 1212 I/O
//     VxD at AppyTime.
//
//  Parameters:
//     WORD request
//        indicates what the VxD wants from us
//     LPVOID lpRequestParams
//        points to data required by the request (or is the data for
//        requests requiring only a DWORD's worth of data).
//
//  Return Value:
//     none
//
//------------------------------------------------------------------------
VOID FAR PASCAL WodVxDCallback
(
   DWORD       request,
   LPWAVEALLOC lpClient,
   LPVOID      lpRequestParams
)
{
   LPWAVEHDR lpWaveHdr;

   switch (request) {
      case K1212_RETURN_WAVEOUTBUF:

         // ---------------------------------------------------------
         // mark the buffer as no longer queued and done
         // ---------------------------------------------------------
         lpWaveHdr           = (LPWAVEHDR)lpRequestParams;
         lpWaveHdr->dwFlags &= ~WHDR_INQUEUE;
         lpWaveHdr->dwFlags |= WHDR_DONE;
         
         // ----------------------------------------------
         // call client's callback
         // ----------------------------------------------
         waveCallBack(lpClient, 
                      WOM_DONE,
                      (DWORD)lpRequestParams
         );
         break;

      default:
         // ---------------------------------------------------------------
         // unsupported command from the VxD
         // ---------------------------------------------------------------
         break;
   }
}

