//*****************************************************************************
//* Name:
//*      12pmapic.h
//*
//* Project:
//*      1212 I/O VxD
//*
//* Author:
//*      Bill Jenkins
//*
//* Description:
//*      This file contains the declaration of the Korg 1212 I/O driver API class.  
//*      This class provides access to the Korg1212 driver services. 
//*
//* Modification Log:
//*
//*      1.3   06/09/97 Bill
//*      Added support for breaking a playback loop.  
//*
//*      1.2   06/02/97 Bill
//*      Added support for posting messages to a file.  
//*
//*      1.1   02/27/97 Bill
//*      Initial version created.  
//*
//*
//* Copyright (c) 1997 Korg, Inc.
//* All rights reserved.
//*
//* This program is protected as an unpublished work under the U.S.
//* copyright laws.  The above copyright notice is not intended to
//* effect a publication of this work.
//*
//* This program is the confidential and proprietary information of
//* Korg and all its subsidiaries.
//*****************************************************************************

#ifndef K12PMAPIC_H
#define K12PMAPIC_H

#ifndef K1212STAT_H
#include "1212stat.h"
#endif
#ifndef K1212API_H
#include "1212api.h"
#endif
#ifndef K1212WAVDEFS_H
#include "1212Wa~1.h"
#endif
#ifndef MIXER_H
#include "mixer.h"
#endif
#ifndef K12PMCTRL_H
#include "12pmctrl.h"
#endif


// ------------------------------------------------------------------------------
// The following class is defined for the interface to the Korg 1212 I/O VxD.
// The purpose of the class is to ease the job of interfacing to the driver by
// hiding the protected mode calls (for 16 bit applications) within member 
// functions that appear similar to the calls specified in the API document 
// created for the Macintosh driver.  
//
// The class is a singleton class - only one instance is allowed per application.
// To instantiate the class, the application should create a CKorg1212Api pointer,
// and assign the result of the CKorg1212Api::OpenDriver() call to it.  Subsequent
// calls to the driver can then be made using the pointer.  For example:
//
//          CKorg1212Api* pK1212Api = CKorg1212Api::OpenDriver();
//             ...
//          pK1212Api->OpenCard();
//             ...
//
// The following comments are intended to provide the application developer with
// enough information to use the API class to its fullest potential.
//          
//
//    ---------------------------------------------------------------
//    GENERAL USAGE NOTES:
//    ---------------------------------------------------------------
//    The API to the Korg 1212 I/O VxD is provided via a library file -
//    1212pm.lib for the wave device driver.  This library contains
//    the implementation of the API class.  1212pm.lib must be linked 
//    with the wave device driver.
//
//    There are four header files that accompany the library files - 
//    1212api.h, 12pmapic.h, 1212WaveApi.h, and 1212stat.h.  The wave
//    device driver should include 12pmapic.h, which automatically
//    includes the other headers.
//
//    Opening the Driver:
//    -------------------
//    The first step in interfacing to the Korg 1212 I/O card is to 
//    open a connection to the VxD for the card.  This is accomplished
//    through the CKorg1212Api::OpenDriver() call.  The pointer provided
//    by this call is then used for subsequent API calls.  Be sure to 
//    call CKorg1212Api::CloseDriver() before exiting the application to
//    ensure that resources allocated by the API object are properly
//    released back to the system.
//
//    Return Codes:
//    -------------
//    With a few exceptions, all calls will return a value of type
//    K1212CmdRet.  This return value should be checked for each and
//    every call made.  A return code of K1212_CMDRET_Success means
//    that the call succeeded.  Any other value indicates a failure
//    of some sort.  (K1212CmdRet is defined in 1212api.h)
//
//    Utility Functions:
//    ------------------
//    There are several utility functions provided that will convert
//    return values into descriptive strings.  These may be used for
//    debugging purposes, or for application dialogs, messages, etc.
//    The string utilities are:
//
//                      GetStateString()
//                      GetCmdRetString()
//                      GetClockSelString()
//                      GetStopModeString()
//
//
//    Retrieving driver information:
//    ------------------------------
//    The following functions are provided for retrieving information
//    from the driver:  
//
//                      GetVersion()
//                      GetBufferCount()
//                      GetPacketSize()
//                      GetSampleSize()
//                      GetPlayChannelCount()
//                      GetRecordChannelCount()
//                      GetRecordPlayLatency()
//
//    Determining cards configured in the system:
//    -------------------------------------------
//    When the driver is opened, the API object automatically retrieves
//    information regarding all cards reported by the driver.  This
//    information is stored in an array that is indexed by a card
//    reference number (k1212CardRef).  By calling GetNumCardsInSystem(),
//    the number of cards in this array is obtained.  
//
//    Referencing cards:
//    ------------------
//    Indexing is from zero to the number of cards minus one.  The card 
//    reference number serves as the index to the array, and must be used 
//    in all calls that are specific to a card.
//
//    Card Selection:
//    ---------------
//    Generally, after opening the driver, you will want to call
//    GetNumCardsInSystem() and store the result to be used as an upper
//    limit to the card reference numbers.  You can then use the 
//    GetCardInfo() call for each card in the system to obtain information
//    about each card.  Based on the information obtained, which includes
//    bus and device numbers, the card(s) to be opened can be selected.
//    Referencing cards with this API is always done using the card reference
//    number. 
//    
//    Wave Device Selection:
//    ----------------------
//
//    Enabling/Disabling Wave Devices:
//    --------------------------------
//
//    Opening/Closing Wave Devices:
//    -----------------------------
//
//
//    Playback on a Wave Device:
//    --------------------------
//
//    Recording from a Wave Device:
//    -----------------------------
//
//    Retrieving card information:
//    ----------------------------
//    Information can be retrieved at any time.
//    The following API calls are defined for providing various types
//    of information about the card:
//
//                      GetCardInfo()
//                      GetCardState()
//                      GetClockSourceRate()
//                      GetInputSensitivity() 
//                      GetChannelVolume()
//                      GetChannelRoute()
//
//    Each of the above functions is described in the API function 
//    descriptions.
//
//    Setting card information:
//    -------------------------
//    There are several parameters that can be set on the card. Setting
//    parameters can only be done during certain times.  See the API function
//    descriptions for details on when each of the following can be called:
//
//                      SetClockSourceRate()
//                      SetInputSensitivity()
//                      SetChannelVolume()
//                      SetChannelRoute()
//
//
//    ---------------------------------------------------------------------
//    API FUNCTION DESCRIPTIONS:
//       (For each call, the "Allowable Call States" line describes the
//        card states in which the call will succeed.  The "State Change"
//        line describes the new state of the card if the call succeeds.)
//    ---------------------------------------------------------------------
//    ---------------------------------------------------------------------
//      static CKorg1212Api* OpenDriver(void);
//
//       This call creates an API object for interfacing to the
//       Korg 1212 I/O card and returns a pointer to the object.  The
//       pointer should be stored, as it is required for all calls
//       provided by the API, and also for properly releasing allocated
//       resources with the CloseDriver() call.
//       
//       example:
//          CKorg1212Api* pK1212Api = CKorg1212Api::OpenDriver();
//          
//       In the above example, subsequent API calls can be made using
//       pK1212Api.
//
//       Allowable Call States:  N/A
//       State Change:           N/A
//
//    ---------------------------------------------------------------------
//      static void  CloseDriver(void);
//
//       This call deletes the Korg 1212 I/O API object.  As a
//       result, the card will automatically be closed, if it is still
//       open.
//
//       Allowable Call States:  any
//       State Change:           K1212_STATE_READY 
//                               Note: if the card was in an inoperative 
//                                     state, it will remain in that state.
//
//    ---------------------------------------------------------------------
//      K1212CmdRet GetVersion(WORD& version);
//
//       This call fills in the 'version' parameter with the 
//       current version of the VxD.  The MSB is the major version and
//       the LSB is the minor version.
//
//       Allowable Call States:  any
//       State Change:           N/A
//
//    ---------------------------------------------------------------------
//      DWORD GetNumCardsInSystem(void);
//
//       This call returns the number of 1212 I/O cards that the VxD reports 
//       are present.  Interface information for these cards is stored in an 
//       array that is indexed from zero to the number of cards minus one.
//       Use the number of cards obtained from this call as an upper limit 
//       to the card indeces (card reference numbers).
//
//       Allowable Call States:  any state
//       State Change:           N/A
//
//    ---------------------------------------------------------------------
//      K1212CmdRet GetRecordPlayLatency(DWORD& numFrames);
//
//       This call fills numFrames with the number of samples of 
//       latency between the record and playback audio streams.  This
//       value can be used to properly time align newly recorded
//       channels with existing channels.
//
//       Allowable Call States:  any state >= K1212_STATE_OPEN
//       State Change:           N/A
//
//    ---------------------------------------------------------------------
//      K1212CmdRet GetCardInfo(k1212CardRef   cardRefNum,
//                              k1212CardInfo& cardInfo);
//
//       This call fills cardInfo with information regarding the card.  
//       See the definition of the k1212CardInfo struct in 1212api.h
//       for details on the information provided.
//
//       Allowable Call States:  any state
//       State Change:           N/A
//
//    ---------------------------------------------------------------------
//      K1212CmdRet GetCardState(k1212CardRef cardRefNum,
//                               CardState&   cardState);
//
//       This call fills cardState with the current state of the card.
//
//       Allowable Call States:  any state
//       State Change:           N/A
//
//    ---------------------------------------------------------------------
//      K1212CmdRet GetDeviceName(k1212CardRef      cardRefNum,
//                                WORD              devNumber,
//                                char __far*       devName);
//
//       This call fills devName with the name of the device.  devName must
//       point to a character buffer of at least 30 bytes.
//
//       Allowable Call States:  any state
//       State Change:           N/A
//
//    ---------------------------------------------------------------------
//      K1212CmdRet GetClockSourceRate(k1212CardRef      cardRefNum,
//                                     ClockSourceIndex& curClkSrcVal);
//
//       This call fills in curClkSrcVal with a value indicating the
//       current selection for the clock source and rate on the card.  
//
//       Allowable Call States:  any state >= K1212_STATE_OPEN
//       State Change:           N/A
//
//    ---------------------------------------------------------------------
//      K1212CmdRet SetClockSourceRate(k1212CardRef     cardRefNum,
//                                     ClockSourceIndex clkSrcIdx);
//
//       This call sets the card's clock source and rate to the
//       value specified by clkSrcIdx.  See the definition of 
//       ClockSourceIndex in 1212api.h for details on the meaning 
//       of each value.
//
//       The application cannot count on any default values here.  If
//       you care what the clock is set to, make sure you explicitly
//       set it with this call before setting up and triggering play.
//
//       Allowable Call States:  K1212_STATE_OPEN
//       State Change:           N/A
//
//    ---------------------------------------------------------------------
//      K1212CmdRet GetInputSensitivity(k1212CardRef cardRefNum,
//                                      WORD&        curRightSens, 
//                                      WORD&        curLeftSens);
//
//       This call fills in curRightSens and curLeftSens with the
//       current sensitivity values for the right and left ADC channels,
//       respectively.
//
//       Allowable Call States:  any state >= K1212_STATE_OPEN
//       State Change:           N/A
//
//    ---------------------------------------------------------------------
//      K1212CmdRet SetInputSensitivity(k1212CardRef cardRefNum,
//                                      WORD         rightSens, 
//                                      WORD         leftSens);
//
//       This call sets the left and right input sensitivity for the
//       ADC inputs.  Valid values are 0-127.  0 provides the highest
//       volume, while 127 provides the lowest volume (highest
//       attenuation).
//
//       Allowable Call States:  any state >= K1212_STATE_OPEN
//       State Change:           N/A
//
//    ---------------------------------------------------------------------
//      static const char* GetStateString(CardState state);
//
//       This function returns a pointer to a descriptive string
//       corresponding to the given state value.  This call can be
//       made at any time.
//
//       Allowable Call States:  any
//       State Change:           N/A
//
//    ---------------------------------------------------------------------
//      static const char* GetCmdRetString(K1212CmdRet cmdRetVal);
//
//       This function returns a pointer to a descriptive string
//       corresponding to the given command return value.  This call 
//       can be made at any time.
//
//       Allowable Call States:  any
//       State Change:           N/A
//
//    ---------------------------------------------------------------------
//      static const char* GetClockSelString(ClockSourceIndex clkSrcIdx);
//
//       This function returns a pointer to a descriptive string
//       corresponding to the given clock source/rate value.  This call
//       can be made at any time.
//
//       Allowable Call States:  any
//       State Change:           N/A
//
//    ---------------------------------------------------------------------
//      static const char* GetStopModeString(StopModeSelector stopMode);
//
//       This function returns a pointer to a descriptive string
//       corresponding to the given stop mode value.  This call can be
//       made at any time.
//
//       Allowable Call States:  any
//       State Change:           N/A
//
//    ---------------------------------------------------------------------
//      K1212CmdRet AddMixerClient          (k1212CardRef cardRefNum,
//                                           WORD         deviceId,
//                                           PMXDCLIENT   pMixerClient
//                  );
//       This function adds the specified mixer client to the linked
//       list of mixer clients for the specified device.
//
//       Allowable Call States:  any
//       State Change:           N/A
//
//    ---------------------------------------------------------------------
//      K1212CmdRet RemoveMixerClient       (k1212CardRef cardRefNum,
//                                           WORD         deviceId,
//                                           PMXDCLIENT   pMixerClient
//                  );
//       This function removes the specified mixer client from the linked
//       list of mixer clients for the specified device.
//
//       Allowable Call States:  any
//       State Change:           N/A
//
//    ---------------------------------------------------------------------
//      PMXDCLIENT  GetMixerClient          (k1212CardRef cardRefNum,
//                                           WORD         deviceId,
//                                           PMXDCLIENT   pMixerClient
//                  );
//       This function returns a pointer to a mixer client information structure.
//       It is set up to be used as an iterative loop.  Setting pMixerClient to
//       zero will return the first client in the list.  Subsequent calls can then
//       be made specifiying the previously returned client pointer in pMixerClient
//       to step through the linked list until zero is returned, indicating the end
//       of the list.  This call can be made at any time.
//
//       Allowable Call States:  any
//       State Change:           N/A
//
// ------------------------------------------------------------------------------


class CKorg1212ApiImpl;    // forward declaration of private implementation class

class CKorg1212Api {
   public:
      static CKorg1212Api* OpenDriver(void);
      static void          CloseDriver(void)   {delete _instance; _instance=0;}

      ~CKorg1212Api();     // destructor for the Korg 1212 driver API object.  This
                           //    function releases resources and closes the connection to 
                           //    the driver
                           
      // -----------------------------------------------------------------------------
      // API calls to the driver.  All calls return a K1212CmdRet value, which should
      // be K1212_CMDRET_Success if all went well.  Parameters passed by reference 
      // are used for returning function specific data.
      // -----------------------------------------------------------------------------
      // driver general information
      K1212CmdRet GetVersion              (WORD __far & version);
      DWORD       GetNumCardsInSystem     (void);
      K1212CmdRet GetRecordPlayLatency    (WORD __far &  numFrames);
      K1212CmdRet VerifyWaveInFormat      (k1212CardRef        cardRefNum,
                                           WORD                deviceId,
                                           k1212WaveFormatData formatData
                  );                                           
      K1212CmdRet VerifyWaveOutFormat     (k1212CardRef        cardRefNum,
                                           WORD                deviceId,
                                           k1212WaveFormatData formatData
                  );

      // card specific information
      K1212CmdRet GetCardInfo             (k1212CardRef                cardRefNum,
                                           k1212PublicCardInfo __far & cardInfo
                  );
      K1212CmdRet GetCardState            (k1212CardRef      cardRefNum,
                                           CardState __far & cardState
                  );
      K1212CmdRet GetDeviceName           (k1212CardRef      cardRefNum,
                                           WORD              devNumber,
                                           char __far*       devName
                  );
      K1212CmdRet GetDeviceInState        (k1212CardRef      cardRefNum,
                                           WORD              devNumber,
                                           DWORD __far &     devState
                  );
      K1212CmdRet GetDeviceOutState       (k1212CardRef      cardRefNum,
                                           WORD              devNumber,
                                           DWORD __far &     devState
                  );

      // card access
      K1212CmdRet EnableWaveInDevs          (k1212CardRef cardRefNum);
      K1212CmdRet EnableWaveOutDevs         (k1212CardRef cardRefNum);
      K1212CmdRet DisableWaveInDevs         (k1212CardRef cardRefNum);
      K1212CmdRet DisableWaveOutDevs        (k1212CardRef cardRefNum);
      K1212CmdRet OpenWaveInDevice          (k1212CardRef                cardRefNum,
                                             WORD                        deviceId,
                                             k1212DeviceOpenData __far & devOpenData
                  );
      K1212CmdRet OpenWaveOutDevice         (k1212CardRef                cardRefNum,
                                             WORD                        deviceId,
                                             k1212DeviceOpenData __far & devOpenData
                  );
      K1212CmdRet CloseWaveInDevice         (k1212CardRef  cardRefNum,
                                             WORD          deviceId
                  );
      K1212CmdRet CloseWaveOutDevice        (k1212CardRef  cardRefNum,
                                             WORD          deviceId
                  );

      // play mode functions
      K1212CmdRet AddRecordBuffer         (k1212CardRef                cardRefNum, 
                                           WORD                        deviceId,
                                           k1212WaveBufferInfo __far & waveBufInfo
                  );
      K1212CmdRet AddPlayBuffer           (k1212CardRef                cardRefNum,
                                           WORD                        deviceId,
                                           k1212WaveBufferInfo __far & waveBufInfo
                  );
      K1212CmdRet StartWaveInDevice       (k1212CardRef cardRefNum,
                                           WORD         deviceId
                  );
      K1212CmdRet StopWaveInDevice        (k1212CardRef cardRefNum,
                                           WORD         deviceId
                  );
      K1212CmdRet PauseWaveOutDevice      (k1212CardRef cardRefNum,
                                           WORD         deviceId
                  );
      K1212CmdRet ResumeWaveOutDevice     (k1212CardRef cardRefNum,
                                           WORD         deviceId
                  );
      K1212CmdRet ResetWaveDevice         (k1212CardRef        cardRefNum,
                                           WORD                deviceId,
                                           K1212WaveDeviceType deviceType
                  );
      K1212CmdRet BreakLoop               (k1212CardRef        cardRefNum,
                                           WORD                deviceId
                  );

      // clock, input sensitivity, volume, and routing
      K1212CmdRet GetWavePosition         (k1212CardRef        cardRefNum,
                                           WORD                deviceId,
                                           K1212WaveDeviceType deviceType,
                                           MMTIME __far &      mmtime
                  );
      K1212CmdRet GetWaveMasterVolume     (k1212CardRef  cardRefNum,
                                           WORD  __far & volume
                  );
      K1212CmdRet SetWaveMasterVolume     (k1212CardRef cardRefNum,
                                           WORD         volume
                  );
      K1212CmdRet GetWaveDeviceVolume     (k1212CardRef  cardRefNum,
                                           WORD          deviceId,
                                           WORD  __far & leftVolume,   
                                           WORD  __far & rightVolume
                  );
      K1212CmdRet SetWaveDeviceVolume     (k1212CardRef cardRefNum,
                                           WORD         deviceId,
                                           WORD         leftVolume,   
                                           WORD         rightVolume
                  );
      K1212CmdRet QueryWaveDeviceMute     (k1212CardRef cardRefNum,
                                           WORD         deviceId,
                                           WORD __far & muteStatus
                  );
      K1212CmdRet MuteWaveDevice          (k1212CardRef cardRefNum,
                                           WORD         deviceId
                  );
      K1212CmdRet UnmuteWaveDevice        (k1212CardRef cardRefNum,
                                           WORD         deviceId
                  );
      K1212CmdRet GetClockSourceRate      (k1212CardRef             cardRefNum,
                                           ClockSourceIndex __far & curClkSrcVal
                  );
      K1212CmdRet SetClockSourceRate      (k1212CardRef     cardRefNum,
                                           ClockSourceIndex clkSrcIdx
                  );
      K1212CmdRet GetInputSensitivity     (k1212CardRef cardRefNum,
                                           WORD __far & curRightSens, 
                                           WORD __far & curLeftSens
                  );
      K1212CmdRet SetInputSensitivity     (k1212CardRef cardRefNum,
                                           WORD         rightSens, 
                                           WORD         leftSens
                  );


      // mixer functions
      K1212CmdRet GetDestinationShortName (k1212CardRef cardRefNum,
                                           char __far*  destName
                  );
      K1212CmdRet GetDestinationLongName  (k1212CardRef cardRefNum,
                                           char __far*  destName
                  );
      K1212CmdRet GetSourceName           (k1212CardRef cardRefNum,
                                           WORD         deviceId,
                                           char __far*  sourceName
                  );
      K1212CmdRet GetMixerName            (char __far*  mixerName);
      DWORD       GetDestType             (k1212CardRef cardRefNum);
      DWORD       GetSourceType           (k1212CardRef cardRefNum,
                                           WORD         deviceId
                  );
      WORD        GetNumDestMixControls   (k1212CardRef cardRefNum);
      WORD        GetNumSourceMixControls (k1212CardRef cardRefNum,
                                           WORD         deviceId
                  );
      const CKorg1212MixControl __far*        
                  GetMasterVolumeControl  (k1212CardRef cardRefNum);

      const CKorg1212MixControl __far*        
                  GetMixControl           (k1212CardRef cardRefNum,
                                           WORD         deviceId,
                                           WORD         controlId
                  );
      K1212CmdRet AddMixerClient          (k1212CardRef cardRefNum,
                                           WORD         deviceId,
                                           LPMXDCLIENT  lpMixerClient
                  );
      K1212CmdRet RemoveMixerClient       (k1212CardRef cardRefNum,
                                           WORD         deviceId,
                                           LPMXDCLIENT  lpMixerClient
                  );
      LPMXDCLIENT GetMixerClient          (k1212CardRef cardRefNum,
                                           WORD         deviceId,
                                           LPMXDCLIENT  lpMixerClient
                  );
                  
      // utility functions
      K1212CmdRet DevNodeToCardRef        (DWORD                devNode,
                                           k1212CardRef __far & cardRefNum
                  );
      
      K1212CmdRet PostMsgToFile           (k1212CardRef         cardRefNum,
                                           char __far *         buffer,          // v1.2
                                           DWORD                numBytesToWrite
                  );

      static const char __far* GetStateString     (CardState        state);
      static const char __far* GetCmdRetString    (K1212CmdRet      cmdRetVal);
      static const char __far* GetClockSelString  (ClockSourceIndex clkSrcIdx);
      static const char __far* GetStopModeString  (StopModeSelector stopMode);

   protected:
      CKorg1212Api();      // constructor for the Korg 1212 driver API object. This
                           //    function will allocate resources and open a connection
                           //    to the VxD using the CreateFile/DeviceIoControl (32 bit) 
                           //    or protected mode (16 bit) interface
      
   private:
      // ---------------------
      // private data members
      // ---------------------
      static CKorg1212Api* _instance;     // tracks whether the API object has been instantiated
      CKorg1212ApiImpl*    _impl;         // pointer to private implementation data and functions
};

#endif // K12PMAPIC_H
