/*
 * Decompiled with CFR 0.152.
 */
package org.apache.cassandra.utils.memory;

import com.google.common.annotations.VisibleForTesting;
import io.netty.util.concurrent.FastThreadLocal;
import java.lang.ref.PhantomReference;
import java.lang.ref.Reference;
import java.lang.ref.ReferenceQueue;
import java.nio.ByteBuffer;
import java.util.Arrays;
import java.util.Queue;
import java.util.concurrent.ConcurrentLinkedQueue;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;
import java.util.concurrent.atomic.AtomicLong;
import java.util.concurrent.atomic.AtomicLongFieldUpdater;
import org.apache.cassandra.concurrent.InfiniteLoopExecutor;
import org.apache.cassandra.config.DatabaseDescriptor;
import org.apache.cassandra.io.compress.BufferType;
import org.apache.cassandra.io.util.FileUtils;
import org.apache.cassandra.metrics.BufferPoolMetrics;
import org.apache.cassandra.utils.ExecutorUtils;
import org.apache.cassandra.utils.FBUtilities;
import org.apache.cassandra.utils.NoSpamLogger;
import org.apache.cassandra.utils.concurrent.Ref;
import org.apache.cassandra.utils.memory.MemoryUtil;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class BufferPool {
    public static final int CHUNK_SIZE = 65536;
    @VisibleForTesting
    public static long MEMORY_USAGE_THRESHOLD = (long)DatabaseDescriptor.getFileCacheSizeInMB() * 1024L * 1024L;
    @VisibleForTesting
    public static boolean ALLOCATE_ON_HEAP_WHEN_EXAHUSTED = DatabaseDescriptor.getBufferPoolUseHeapIfExhausted();
    @VisibleForTesting
    public static boolean DISABLED = Boolean.parseBoolean(System.getProperty("cassandra.test.disable_buffer_pool", "false"));
    @VisibleForTesting
    public static boolean DEBUG = false;
    private static final Logger logger = LoggerFactory.getLogger(BufferPool.class);
    private static final NoSpamLogger noSpamLogger = NoSpamLogger.getLogger(logger, 15L, TimeUnit.MINUTES);
    private static final ByteBuffer EMPTY_BUFFER = ByteBuffer.allocateDirect(0);
    private static final GlobalPool globalPool = new GlobalPool();
    private static final FastThreadLocal<LocalPool> localPool = new FastThreadLocal<LocalPool>(){

        protected LocalPool initialValue() {
            return new LocalPool();
        }
    };
    private static final ConcurrentLinkedQueue<LocalPoolRef> localPoolReferences = new ConcurrentLinkedQueue();
    private static final ReferenceQueue<Object> localPoolRefQueue = new ReferenceQueue();
    private static final InfiniteLoopExecutor EXEC = new InfiniteLoopExecutor("LocalPool-Cleaner", BufferPool::cleanupOneReference).start();

    public static ByteBuffer get(int size) {
        if (DISABLED) {
            return BufferPool.allocate(size, ALLOCATE_ON_HEAP_WHEN_EXAHUSTED);
        }
        return BufferPool.takeFromPool(size, ALLOCATE_ON_HEAP_WHEN_EXAHUSTED);
    }

    public static ByteBuffer get(int size, BufferType bufferType) {
        boolean direct;
        boolean bl = direct = bufferType == BufferType.OFF_HEAP;
        if (DISABLED || !direct) {
            return BufferPool.allocate(size, !direct);
        }
        return BufferPool.takeFromPool(size, !direct);
    }

    public static ByteBuffer tryGet(int size) {
        if (DISABLED) {
            return BufferPool.allocate(size, ALLOCATE_ON_HEAP_WHEN_EXAHUSTED);
        }
        return BufferPool.maybeTakeFromPool(size, ALLOCATE_ON_HEAP_WHEN_EXAHUSTED);
    }

    private static ByteBuffer allocate(int size, boolean onHeap) {
        return onHeap ? ByteBuffer.allocate(size) : ByteBuffer.allocateDirect(size);
    }

    private static ByteBuffer takeFromPool(int size, boolean allocateOnHeapWhenExhausted) {
        ByteBuffer ret = BufferPool.maybeTakeFromPool(size, allocateOnHeapWhenExhausted);
        if (ret != null) {
            return ret;
        }
        if (logger.isTraceEnabled()) {
            logger.trace("Requested buffer size {} has been allocated directly due to lack of capacity", (Object)FBUtilities.prettyPrintMemory(size));
        }
        return ((LocalPool)BufferPool.localPool.get()).allocate(size, allocateOnHeapWhenExhausted);
    }

    private static ByteBuffer maybeTakeFromPool(int size, boolean allocateOnHeapWhenExhausted) {
        if (size < 0) {
            throw new IllegalArgumentException("Size must be positive (" + size + ")");
        }
        if (size == 0) {
            return EMPTY_BUFFER;
        }
        if (size > 65536) {
            if (logger.isTraceEnabled()) {
                logger.trace("Requested buffer size {} is bigger than {}, allocating directly", (Object)FBUtilities.prettyPrintMemory(size), (Object)FBUtilities.prettyPrintMemory(65536L));
            }
            return ((LocalPool)BufferPool.localPool.get()).allocate(size, allocateOnHeapWhenExhausted);
        }
        return ((LocalPool)localPool.get()).get(size);
    }

    public static void put(ByteBuffer buffer) {
        if (!DISABLED && !buffer.hasArray()) {
            ((LocalPool)localPool.get()).put(buffer);
        }
    }

    @VisibleForTesting
    static void reset() {
        ((LocalPool)localPool.get()).reset();
        globalPool.reset();
    }

    @VisibleForTesting
    static Chunk currentChunk() {
        return ((LocalPool)localPool.get()).chunks[0];
    }

    @VisibleForTesting
    static int numChunks() {
        int ret = 0;
        for (Chunk chunk : ((LocalPool)localPool.get()).chunks) {
            if (chunk == null) continue;
            ++ret;
        }
        return ret;
    }

    @VisibleForTesting
    static void assertAllRecycled() {
        globalPool.debug.check();
    }

    @VisibleForTesting
    static void releaseLocal() {
        ((LocalPool)localPool.get()).release();
    }

    public static long sizeInBytes() {
        return globalPool.sizeInBytes();
    }

    private static void cleanupOneReference() throws InterruptedException {
        Reference<Object> obj = localPoolRefQueue.remove(100L);
        if (obj instanceof LocalPoolRef) {
            ((LocalPoolRef)obj).release();
            localPoolReferences.remove(obj);
        }
    }

    private static ByteBuffer allocateDirectAligned(int capacity) {
        int align = MemoryUtil.pageSize();
        if (Integer.bitCount(align) != 1) {
            throw new IllegalArgumentException("Alignment must be a power of 2");
        }
        ByteBuffer buffer = ByteBuffer.allocateDirect(capacity + align);
        long address = MemoryUtil.getAddress(buffer);
        long offset = address & (long)(align - 1);
        if (offset == 0L) {
            buffer.limit(capacity);
        } else {
            int pos = (int)((long)align - offset);
            buffer.position(pos);
            buffer.limit(pos + capacity);
        }
        return buffer.slice();
    }

    @VisibleForTesting
    public static int roundUpNormal(int size) {
        return BufferPool.roundUp(size, 1024);
    }

    private static int roundUp(int size, int unit) {
        int mask = unit - 1;
        return size + mask & ~mask;
    }

    @VisibleForTesting
    public static void shutdownLocalCleaner(long timeout, TimeUnit unit) throws InterruptedException, TimeoutException {
        ExecutorUtils.shutdownNow(Arrays.asList(EXEC));
        ExecutorUtils.awaitTermination(timeout, unit, Arrays.asList(EXEC));
    }

    static final class Chunk {
        private final ByteBuffer slab;
        private final long baseAddress;
        private final int shift;
        private volatile long freeSlots;
        private static final AtomicLongFieldUpdater<Chunk> freeSlotsUpdater = AtomicLongFieldUpdater.newUpdater(Chunk.class, "freeSlots");
        private volatile LocalPool owner;
        private volatile long lastAcquired;
        private volatile long lastRecycled;
        private final Chunk original;

        Chunk(Chunk recycle) {
            assert (recycle.freeSlots == 0L);
            this.slab = recycle.slab;
            this.baseAddress = recycle.baseAddress;
            this.shift = recycle.shift;
            this.freeSlots = -1L;
            this.original = recycle.original;
            if (DEBUG) {
                globalPool.debug.recycle(this.original);
            }
        }

        Chunk(ByteBuffer slab) {
            assert (!slab.hasArray());
            this.slab = slab;
            this.baseAddress = MemoryUtil.getAddress(slab);
            this.shift = 0x1F & Integer.numberOfTrailingZeros(slab.capacity() / 64);
            this.freeSlots = slab.capacity() == 0 ? 0L : -1L;
            this.original = DEBUG ? this : null;
        }

        void acquire(LocalPool owner) {
            assert (this.owner == null);
            this.owner = owner;
        }

        void release() {
            this.owner = null;
            this.tryRecycle();
        }

        void tryRecycle() {
            assert (this.owner == null);
            if (this.isFree() && freeSlotsUpdater.compareAndSet(this, -1L, 0L)) {
                this.recycle();
            }
        }

        void recycle() {
            assert (this.freeSlots == 0L);
            globalPool.recycle(new Chunk(this));
        }

        static Chunk getParentChunk(ByteBuffer buffer) {
            Object attachment = MemoryUtil.getAttachment(buffer);
            if (attachment instanceof Chunk) {
                return (Chunk)attachment;
            }
            if (attachment instanceof Ref) {
                return (Chunk)((Ref)attachment).get();
            }
            return null;
        }

        ByteBuffer setAttachment(ByteBuffer buffer) {
            if (Ref.DEBUG_ENABLED) {
                MemoryUtil.setAttachment(buffer, new Ref<Chunk>(this, null));
            } else {
                MemoryUtil.setAttachment(buffer, this);
            }
            return buffer;
        }

        boolean releaseAttachment(ByteBuffer buffer) {
            Object attachment = MemoryUtil.getAttachment(buffer);
            if (attachment == null) {
                return false;
            }
            if (attachment instanceof Ref) {
                ((Ref)attachment).release();
            }
            return true;
        }

        @VisibleForTesting
        void reset() {
            Chunk parent = Chunk.getParentChunk(this.slab);
            if (parent != null) {
                parent.free(this.slab, false);
            } else {
                FileUtils.clean(this.slab);
            }
        }

        @VisibleForTesting
        long setFreeSlots(long val) {
            long ret = this.freeSlots;
            this.freeSlots = val;
            return ret;
        }

        int capacity() {
            return 64 << this.shift;
        }

        final int unit() {
            return 1 << this.shift;
        }

        final boolean isFree() {
            return this.freeSlots == -1L;
        }

        int free() {
            return Long.bitCount(this.freeSlots) * this.unit();
        }

        ByteBuffer get(int size) {
            long cur;
            int index;
            long candidate;
            int slotCount = size - 1 + this.unit() >>> this.shift;
            if (slotCount > 64) {
                return null;
            }
            long slotBits = -1L >>> 64 - slotCount;
            long searchMask = 0x1111111111111111L;
            searchMask *= 15L >>> (slotCount - 1 & 3);
            searchMask &= -1L >>> slotCount - 1;
            do {
                if ((index = Long.numberOfTrailingZeros((cur = this.freeSlots) & searchMask)) == 64) {
                    return null;
                }
                searchMask ^= 1L << index;
            } while (((candidate = slotBits << index) & cur) != candidate);
            while (!freeSlotsUpdater.compareAndSet(this, cur, cur & (candidate ^ 0xFFFFFFFFFFFFFFFFL))) {
                cur = this.freeSlots;
                assert ((candidate & cur) == candidate);
            }
            return this.get(index << this.shift, size);
        }

        private ByteBuffer get(int offset, int size) {
            this.slab.limit(offset + size);
            this.slab.position(offset);
            return this.setAttachment(this.slab.slice());
        }

        int roundUp(int v) {
            return BufferPool.roundUp(v, this.unit());
        }

        long free(ByteBuffer buffer, boolean tryRelease) {
            long next;
            long cur;
            if (!this.releaseAttachment(buffer)) {
                return 1L;
            }
            long address = MemoryUtil.getAddress(buffer);
            assert (address >= this.baseAddress & address <= this.baseAddress + (long)this.capacity());
            int position = (int)(address - this.baseAddress);
            int size = this.roundUp(buffer.capacity());
            int slotCount = size >> this.shift;
            long slotBits = (1L << slotCount) - 1L;
            long shiftedSlotBits = slotBits << (position >>= this.shift);
            if (slotCount == 64) {
                assert (size == this.capacity());
                assert (position == 0);
                shiftedSlotBits = -1L;
            }
            do {
                cur = this.freeSlots;
                next = cur | shiftedSlotBits;
                assert (next == (cur ^ shiftedSlotBits));
                if (!tryRelease || next != -1L) continue;
                next = 0L;
            } while (!freeSlotsUpdater.compareAndSet(this, cur, next));
            return next;
        }

        public String toString() {
            return String.format("[slab %s, slots bitmap %s, capacity %d, free %d]", this.slab, Long.toBinaryString(this.freeSlots), this.capacity(), this.free());
        }
    }

    private static final class LocalPoolRef
    extends PhantomReference<LocalPool> {
        private final Chunk[] chunks;

        public LocalPoolRef(LocalPool localPool, ReferenceQueue<? super LocalPool> q) {
            super(localPool, q);
            this.chunks = localPool.chunks;
        }

        public void release() {
            for (int i = 0; i < this.chunks.length; ++i) {
                if (this.chunks[i] == null) continue;
                this.chunks[i].release();
                this.chunks[i] = null;
            }
        }
    }

    static final class LocalPool {
        private static final BufferPoolMetrics metrics = new BufferPoolMetrics();
        private final Chunk[] chunks = new Chunk[3];
        private byte chunkCount = 0;

        public LocalPool() {
            localPoolReferences.add(new LocalPoolRef(this, localPoolRefQueue));
        }

        private Chunk addChunkFromGlobalPool() {
            Chunk chunk = globalPool.get();
            if (chunk == null) {
                return null;
            }
            this.addChunk(chunk);
            return chunk;
        }

        private void addChunk(Chunk chunk) {
            chunk.acquire(this);
            if (this.chunkCount < 3) {
                byte by = this.chunkCount;
                this.chunkCount = (byte)(by + 1);
                this.chunks[by] = chunk;
                return;
            }
            int smallestChunkIdx = 0;
            if (this.chunks[1].free() < this.chunks[0].free()) {
                smallestChunkIdx = 1;
            }
            if (this.chunks[2].free() < this.chunks[smallestChunkIdx].free()) {
                smallestChunkIdx = 2;
            }
            this.chunks[smallestChunkIdx].release();
            if (smallestChunkIdx != 2) {
                this.chunks[smallestChunkIdx] = this.chunks[2];
            }
            this.chunks[2] = chunk;
        }

        public ByteBuffer get(int size) {
            for (Chunk chunk : this.chunks) {
                if (chunk == null) break;
                ByteBuffer buffer = chunk.get(size);
                if (buffer == null) continue;
                return buffer;
            }
            Chunk chunk = this.addChunkFromGlobalPool();
            if (chunk != null) {
                return chunk.get(size);
            }
            return null;
        }

        private ByteBuffer allocate(int size, boolean onHeap) {
            LocalPool.metrics.misses.mark();
            return BufferPool.allocate(size, onHeap);
        }

        public void put(ByteBuffer buffer) {
            Chunk chunk = Chunk.getParentChunk(buffer);
            if (chunk == null) {
                FileUtils.clean(buffer);
                return;
            }
            LocalPool owner = chunk.owner;
            long free = chunk.free(buffer, owner == null | owner == this);
            if (free == 0L) {
                chunk.recycle();
                if (owner == this) {
                    this.removeFromLocalQueue(chunk);
                }
            } else if (free == -1L && owner != this && chunk.owner == null) {
                chunk.tryRecycle();
            }
        }

        private void removeFromLocalQueue(Chunk chunk) {
            if (this.chunks[0] == chunk) {
                this.chunks[0] = this.chunks[1];
                this.chunks[1] = this.chunks[2];
            } else if (this.chunks[1] == chunk) {
                this.chunks[1] = this.chunks[2];
            } else assert (this.chunks[2] == chunk);
            this.chunks[2] = null;
            this.chunkCount = (byte)(this.chunkCount - 1);
        }

        @VisibleForTesting
        void reset() {
            this.chunkCount = 0;
            for (int i = 0; i < this.chunks.length; ++i) {
                if (this.chunks[i] == null) continue;
                this.chunks[i].owner = null;
                this.chunks[i].freeSlots = 0L;
                this.chunks[i].recycle();
                this.chunks[i] = null;
            }
        }

        @VisibleForTesting
        void release() {
            this.chunkCount = 0;
            for (int i = 0; i < this.chunks.length; ++i) {
                if (this.chunks[i] == null) continue;
                this.chunks[i].release();
                this.chunks[i] = null;
            }
        }
    }

    static final class GlobalPool {
        static final int MACRO_CHUNK_SIZE = 0x100000;
        private final Debug debug = new Debug();
        private final Queue<Chunk> macroChunks = new ConcurrentLinkedQueue<Chunk>();
        private final Queue<Chunk> chunks = new ConcurrentLinkedQueue<Chunk>();
        private final AtomicLong memoryUsage = new AtomicLong();
        static final /* synthetic */ boolean $assertionsDisabled;

        GlobalPool() {
        }

        public Chunk get() {
            Chunk chunk = this.getInternal();
            if (DEBUG && chunk != null) {
                this.debug.acquire(chunk);
            }
            return chunk;
        }

        private Chunk getInternal() {
            do {
                Chunk chunk;
                if ((chunk = this.chunks.poll()) == null) continue;
                return chunk;
            } while (this.allocateMoreChunks());
            return this.chunks.poll();
        }

        private boolean allocateMoreChunks() {
            Chunk chunk;
            long cur;
            do {
                if ((cur = this.memoryUsage.get()) + 0x100000L <= MEMORY_USAGE_THRESHOLD) continue;
                noSpamLogger.info("Maximum memory usage reached ({}), cannot allocate chunk of {}", MEMORY_USAGE_THRESHOLD, 0x100000);
                return false;
            } while (!this.memoryUsage.compareAndSet(cur, cur + 0x100000L));
            try {
                chunk = new Chunk(BufferPool.allocateDirectAligned(0x100000));
            }
            catch (OutOfMemoryError oom) {
                noSpamLogger.error("Buffer pool failed to allocate chunk of {}, current size {} ({}). Attempting to continue; buffers will be allocated in on-heap memory which can degrade performance. Make sure direct memory size (-XX:MaxDirectMemorySize) is large enough to accommodate off-heap memtables and caches.", 0x100000, this.sizeInBytes(), oom.toString());
                return false;
            }
            chunk.acquire(null);
            this.macroChunks.add(chunk);
            for (int i = 0; i < 0x100000; i += 65536) {
                Chunk add2 = new Chunk(chunk.get(65536));
                this.chunks.add(add2);
                if (!DEBUG) continue;
                this.debug.register(add2);
            }
            return true;
        }

        public void recycle(Chunk chunk) {
            this.chunks.add(chunk);
        }

        public long sizeInBytes() {
            return this.memoryUsage.get();
        }

        @VisibleForTesting
        void reset() {
            while (!this.chunks.isEmpty()) {
                this.chunks.poll().reset();
            }
            while (!this.macroChunks.isEmpty()) {
                this.macroChunks.poll().reset();
            }
            this.memoryUsage.set(0L);
        }

        static {
            boolean bl = $assertionsDisabled = !BufferPool.class.desiredAssertionStatus();
            if (!$assertionsDisabled && Integer.bitCount(65536) != 1) {
                throw new AssertionError();
            }
            if (!$assertionsDisabled && Integer.bitCount(0x100000) != 1) {
                throw new AssertionError();
            }
            if (DISABLED) {
                logger.info("Global buffer pool is disabled, allocating {}", (Object)(ALLOCATE_ON_HEAP_WHEN_EXAHUSTED ? "on heap" : "off heap"));
            } else {
                logger.info("Global buffer pool is enabled, when pool is exhausted (max is {}) it will allocate {}", (Object)FBUtilities.prettyPrintMemory(MEMORY_USAGE_THRESHOLD), (Object)(ALLOCATE_ON_HEAP_WHEN_EXAHUSTED ? "on heap" : "off heap"));
            }
        }
    }

    static final class Debug {
        volatile long recycleRound = 0L;
        final Queue<Chunk> allChunks = new ConcurrentLinkedQueue<Chunk>();

        Debug() {
        }

        void register(Chunk chunk) {
            this.allChunks.add(chunk);
        }

        void acquire(Chunk chunk) {
            chunk.lastAcquired = this.recycleRound;
        }

        void recycle(Chunk chunk) {
            chunk.lastRecycled = this.recycleRound;
        }

        void check() {
            for (Chunk chunk : this.allChunks) {
                assert (chunk.lastRecycled >= chunk.lastAcquired);
            }
            ++this.recycleRound;
        }
    }
}

