/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */

package org.apache.druid.msq.kernel;

import com.fasterxml.jackson.databind.ObjectMapper;
import org.apache.druid.msq.exec.DataServerQueryHandlerFactory;
import org.apache.druid.msq.exec.ProcessingBuffers;
import org.apache.druid.msq.exec.WorkerImpl;
import org.apache.druid.msq.exec.WorkerMemoryParameters;
import org.apache.druid.msq.exec.WorkerStorageParameters;
import org.apache.druid.msq.querykit.DataSegmentProvider;
import org.apache.druid.query.groupby.GroupingEngine;
import org.apache.druid.segment.IndexIO;
import org.apache.druid.segment.IndexMergerV9;
import org.apache.druid.segment.SegmentWrangler;
import org.apache.druid.segment.incremental.RowIngestionMeters;
import org.apache.druid.segment.loading.DataSegmentPusher;

import java.io.Closeable;
import java.io.File;

/**
 * Provides services and objects for the functioning of the frame processors. Scoped to a specific stage of a
 * specific query, i.e., one {@link WorkOrder}.
 *
 * Generated by {@link org.apache.druid.msq.exec.WorkerContext#frameContext(WorkOrder)}.
 */
public interface FrameContext extends Closeable
{
  SegmentWrangler segmentWrangler();

  GroupingEngine groupingEngine();

  RowIngestionMeters rowIngestionMeters();

  DataSegmentProvider dataSegmentProvider();

  DataServerQueryHandlerFactory dataServerQueryHandlerFactory();

  /**
   * Temporary directory, fully owned by this particular stage.
   */
  File tempDir();

  ObjectMapper jsonMapper();

  IndexIO indexIO();

  File persistDir();

  DataSegmentPusher segmentPusher();

  IndexMergerV9 indexMerger();

  ProcessingBuffers processingBuffers();

  WorkerMemoryParameters memoryParameters();

  WorkerStorageParameters storageParameters();

  default File tempDir(String name)
  {
    return new File(tempDir(), name);
  }

  /**
   * Releases resources used in processing. This is called when processing has completed, but before results are
   * cleaned up. Specifically, it is called by {@link WorkerImpl.KernelHolder#processorCloser}.
   */
  @Override
  void close();
}
