/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hertzbeat.alert.service.impl;

import com.fasterxml.jackson.databind.JsonNode;
import java.io.InputStream;
import java.net.URL;
import java.nio.charset.StandardCharsets;
import java.security.Signature;
import java.security.cert.CertificateFactory;
import java.security.cert.X509Certificate;
import java.text.MessageFormat;
import java.time.Instant;
import java.util.Base64;
import java.util.HashMap;
import java.util.Map;
import java.util.Optional;
import org.apache.commons.lang3.StringUtils;
import org.apache.hertzbeat.alert.dto.HuaweiCloudExternAlert;
import org.apache.hertzbeat.alert.reduce.AlarmCommonReduce;
import org.apache.hertzbeat.alert.service.ExternAlertService;
import org.apache.hertzbeat.alert.util.DateUtil;
import org.apache.hertzbeat.common.entity.alerter.SingleAlert;
import org.apache.hertzbeat.common.support.exception.IgnoreException;
import org.apache.hertzbeat.common.util.JsonUtil;
import org.apache.http.HttpEntity;
import org.apache.http.client.methods.CloseableHttpResponse;
import org.apache.http.client.methods.HttpGet;
import org.apache.http.client.methods.HttpUriRequest;
import org.apache.http.impl.client.CloseableHttpClient;
import org.apache.http.impl.client.HttpClients;
import org.apache.http.util.EntityUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Service;

@Service
public class HuaweiCloudExternAlertService
implements ExternAlertService {
    private static final Logger log = LoggerFactory.getLogger(HuaweiCloudExternAlertService.class);
    private static final String CERTIFICATE_TYPE = "X.509";
    private static final String CHARSET_UTF8 = StandardCharsets.UTF_8.name();
    private final AlarmCommonReduce alarmCommonReduce;

    public HuaweiCloudExternAlertService(AlarmCommonReduce alarmCommonReduce) {
        this.alarmCommonReduce = alarmCommonReduce;
    }

    @Override
    public void addExternAlert(String content) {
        HuaweiCloudExternAlert externAlert = (HuaweiCloudExternAlert)JsonUtil.fromJson((String)content, HuaweiCloudExternAlert.class);
        if (externAlert == null || StringUtils.isBlank((CharSequence)externAlert.getMessage())) {
            log.warn("Failure to parse external alert content. content: {}", (Object)content);
            return;
        }
        if (!this.isMessageValid(externAlert)) {
            log.warn("Huawei cloud alert verify failed. content: {}", (Object)content);
            return;
        }
        this.process(externAlert);
    }

    private void process(HuaweiCloudExternAlert externAlert) {
        if (HuaweiCloudExternAlert.AlertType.NOTIFICATION.getType().equals(externAlert.getType())) {
            Optional.ofNullable(this.buildSendAlert(externAlert)).ifPresent(this.alarmCommonReduce::reduceAndSendAlarm);
        } else if (HuaweiCloudExternAlert.AlertType.SUBSCRIPTION.getType().equals(externAlert.getType())) {
            this.autoSubscribeForUrl(externAlert.getSubscribeUrl());
        } else if (HuaweiCloudExternAlert.AlertType.UNSUBSCRIBE.getType().equals(externAlert.getType())) {
            log.warn("Huawei cloud notifies the recipient of the notification to cancel the subscription.");
        }
    }

    private SingleAlert buildSendAlert(HuaweiCloudExternAlert externAlert) {
        HuaweiCloudExternAlert.AlertMessage message = (HuaweiCloudExternAlert.AlertMessage)JsonUtil.fromJson((String)externAlert.getMessage(), HuaweiCloudExternAlert.AlertMessage.class);
        if (null == message || null == message.getData()) {
            log.warn("Failure to parse external alert message. message: {}", (Object)externAlert.getMessage());
            return null;
        }
        boolean isAlarm = null != message.getData().getAlarm() && message.getData().getAlarm() != false;
        Long alarmTime = DateUtil.getZonedTimeStampFromFormat(message.getData().getAlarmTime(), "yyyy/MM/dd HH:mm:ss 'GMT'XXX");
        return SingleAlert.builder().triggerTimes(Integer.valueOf(1)).status(isAlarm ? "firing" : "resolved").startAt(alarmTime).activeAt(Long.valueOf(Instant.now().toEpochMilli())).endAt(isAlarm ? null : alarmTime).labels(this.buildLabels(message.getData())).annotations(this.buildAnnotations(message.getData())).content(this.formatContent(externAlert.getSubject(), message.getData())).build();
    }

    private Map<String, String> buildAnnotations(HuaweiCloudExternAlert.AlertData alertData) {
        HashMap<String, String> annotations = new HashMap<String, String>(8);
        if (null != alertData) {
            this.putIfNotBlank(annotations, "region", alertData.getRegion());
            this.putIfNotBlank(annotations, "dimensionName", alertData.getDimensionName());
            this.putIfNotBlank(annotations, "resourceName", alertData.getResourceName());
            this.putIfNotBlank(annotations, "alarmRecordId", alertData.getAlarmRecordId());
        }
        return annotations;
    }

    private Map<String, String> buildLabels(HuaweiCloudExternAlert.AlertData alertData) {
        HashMap<String, String> labels = new HashMap<String, String>(8);
        labels.put("__source__", "huaweicloud-ces");
        if (null != alertData) {
            this.putIfNotBlank(labels, "namespace", alertData.getNamespace());
            this.putIfNotBlank(labels, "metricName", alertData.getMetricName());
            this.putIfNotBlank(labels, "resourceId", alertData.getResourceId());
            this.putIfNotBlank(labels, "level", alertData.getAlarmLevel());
        }
        return labels;
    }

    private String formatContent(String subject, HuaweiCloudExternAlert.AlertData alertData) {
        if (null == alertData) {
            return subject;
        }
        return MessageFormat.format("{0} threshold:{1}{2}, current\uff1a{3}", subject, alertData.getComparisonOperator(), alertData.getValue(), alertData.getCurrentData());
    }

    public void autoSubscribeForUrl(String subscribeUrl) {
        if (StringUtils.isBlank((CharSequence)subscribeUrl)) {
            return;
        }
        try (CloseableHttpClient httpClient = HttpClients.createDefault();){
            HttpGet httpGet = new HttpGet(subscribeUrl);
            try (CloseableHttpResponse response = httpClient.execute((HttpUriRequest)httpGet);){
                int statusCode = response.getStatusLine().getStatusCode();
                String responseBody = EntityUtils.toString((HttpEntity)response.getEntity());
                if (statusCode != 200) {
                    log.error("Subscribe url request failed with status code: " + statusCode + ", response: " + responseBody);
                    return;
                }
                JsonNode jsonResponse = JsonUtil.fromJson((String)responseBody);
                if (jsonResponse == null) {
                    throw new IgnoreException("Subscribe url failed with status code: " + statusCode + ", response: " + responseBody);
                }
                JsonNode surnNode = jsonResponse.get("subscription_urn");
                if (surnNode == null || StringUtils.isBlank((CharSequence)surnNode.asText())) {
                    throw new IgnoreException("Subscribe url failed with status code: " + statusCode + ", response: " + responseBody);
                }
                log.info("Successfully subscribed to Huawei Cloud(SMN) url.");
            }
        }
        catch (Exception e) {
            log.error("Failed to subscribe url request: {}", (Object)e.getMessage());
        }
    }

    private boolean isMessageValid(HuaweiCloudExternAlert externAlert) {
        try {
            String signMessage = this.buildSignMessage(externAlert);
            if (StringUtils.isBlank((CharSequence)signMessage)) {
                throw new SecurityException("Verify sign message is null");
            }
            X509Certificate cert = this.getCertificate(externAlert.getSigningCertUrl());
            return this.verifySignature(signMessage, cert, externAlert.getSignature());
        }
        catch (Exception e) {
            log.error("Failed to verify message signature: ", (Throwable)e);
            return false;
        }
    }

    private String buildSignMessage(HuaweiCloudExternAlert externAlert) {
        if (HuaweiCloudExternAlert.AlertType.NOTIFICATION.getType().equals(externAlert.getType())) {
            return this.buildNotificationMessage(externAlert);
        }
        if (HuaweiCloudExternAlert.AlertType.SUBSCRIPTION.getType().equals(externAlert.getType()) || HuaweiCloudExternAlert.AlertType.UNSUBSCRIBE.getType().equals(externAlert.getType())) {
            return this.buildSubscriptionMessage(externAlert);
        }
        return null;
    }

    private String buildNotificationMessage(HuaweiCloudExternAlert externAlert) {
        StringBuilder message = new StringBuilder();
        this.appendField(message, "message", externAlert.getMessage());
        this.appendField(message, "message_id", externAlert.getMessageId());
        if (StringUtils.isNotBlank((CharSequence)externAlert.getSubject())) {
            this.appendField(message, "subject", externAlert.getSubject());
        }
        this.appendField(message, "timestamp", externAlert.getTimestamp());
        this.appendField(message, "topic_urn", externAlert.getTopicUrn());
        this.appendField(message, "type", externAlert.getType());
        return message.toString();
    }

    private String buildSubscriptionMessage(HuaweiCloudExternAlert externAlert) {
        StringBuilder message = new StringBuilder();
        this.appendField(message, "message", externAlert.getMessage());
        this.appendField(message, "message_id", externAlert.getMessageId());
        this.appendField(message, "subscribe_url", externAlert.getSubscribeUrl());
        this.appendField(message, "timestamp", externAlert.getTimestamp());
        this.appendField(message, "topic_urn", externAlert.getTopicUrn());
        this.appendField(message, "type", externAlert.getType());
        return message.toString();
    }

    private X509Certificate getCertificate(String signCertUrl) throws Exception {
        URL url = new URL(signCertUrl);
        try (InputStream in = url.openStream();){
            CertificateFactory cf = CertificateFactory.getInstance(CERTIFICATE_TYPE);
            X509Certificate x509Certificate = (X509Certificate)cf.generateCertificate(in);
            return x509Certificate;
        }
    }

    private boolean verifySignature(String message, X509Certificate cert, String signature) throws Exception {
        Signature sig = Signature.getInstance(cert.getSigAlgName());
        sig.initVerify(cert.getPublicKey());
        sig.update(message.getBytes(CHARSET_UTF8));
        return sig.verify(Base64.getDecoder().decode(signature));
    }

    private void putIfNotBlank(Map<String, String> map, String key, String value) {
        if (StringUtils.isNotBlank((CharSequence)value)) {
            map.put(key, value);
        }
    }

    private void appendField(StringBuilder builder, String fieldName, String value) {
        builder.append(fieldName).append("\n").append(value).append("\n");
    }

    @Override
    public String supportSource() {
        return "huaweicloud-ces";
    }
}

