/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * license agreements; and to You under the Apache License, version 2.0:
 *
 *   https://www.apache.org/licenses/LICENSE-2.0
 *
 * This file is part of the Apache Pekko project, which was derived from Akka.
 */

/*
 * Copyright (C) 2009-2022 Lightbend Inc. <https://www.lightbend.com>
 */

package org.apache.pekko.http.javadsl.model;

import org.apache.pekko.annotation.DoNotInherit;
import org.apache.pekko.http.scaladsl.model.IllegalHeaderException;

/**
 * The base type representing Http headers. All actual header values will be instances of one of the
 * subtypes defined in the `headers` packages. Unknown headers will be subtypes of {@link
 * org.apache.pekko.http.javadsl.model.headers.RawHeader}. Not for user extension.
 */
@DoNotInherit
public abstract class HttpHeader {
  /** Returns the name of the header. */
  public abstract String name();

  /** Returns the String representation of the value of the header. */
  public abstract String value();

  /** Returns the lower-cased name of the header. */
  public abstract String lowercaseName();

  /** Returns true if and only if nameInLowerCase.equals(lowercaseName()). */
  public abstract boolean is(String nameInLowerCase);

  /** Returns !is(nameInLowerCase). */
  public abstract boolean isNot(String nameInLowerCase);

  /** Returns true if and only if the header is to be rendered in requests. */
  public abstract boolean renderInRequests();

  /** Returns true if and only if the header is to be rendered in responses. */
  public abstract boolean renderInResponses();

  /**
   * Attempts to parse the given header name and value string into a header model instance.
   *
   * @throws IllegalArgumentException if parsing is unsuccessful.
   */
  public static HttpHeader parse(String name, String value) {
    final org.apache.pekko.http.scaladsl.model.HttpHeader.ParsingResult result =
        org.apache.pekko.http.scaladsl.model.HttpHeader.parse(
            name,
            value,
            org.apache.pekko.http.impl.model.parser.HeaderParser$.MODULE$.DefaultSettings());

    if (result instanceof org.apache.pekko.http.scaladsl.model.HttpHeader$ParsingResult$Ok) {
      return ((org.apache.pekko.http.scaladsl.model.HttpHeader$ParsingResult$Ok) result).header();
    } else {
      throw new IllegalHeaderException(
          ((org.apache.pekko.http.scaladsl.model.HttpHeader$ParsingResult$Error) result).error());
    }
  }
}
