/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

grammar BaseRule;

import Symbol, Keyword, OracleKeyword, Literals;

parameterMarker
    : QUESTION_
    ;

literals
    : stringLiterals
    | numberLiterals
    | dateTimeLiterals
    | hexadecimalLiterals
    | bitValueLiterals
    | booleanLiterals
    | nullValueLiterals
    | intervalLiterals
    ;

intervalLiterals
    : INTERVAL simpleExpr intervalUnit (intervalPrecision)? (TO intervalUnit (intervalPrecision)?)?
    ;

intervalPrecision
    : LP_ INTEGER_ RP_
    ;

intervalUnit
    : SECOND | MINUTE | HOUR | DAY | MONTH | YEAR
    ;

stringLiterals
    : STRING_
    | NCHAR_TEXT
    | UCHAR_TEXT
    ;

numberLiterals
   : (PLUS_ | MINUS_)? (INTEGER_ | NUMBER_)
   ;

dateTimeLiterals
    : (DATE | TIME | TIMESTAMP) stringLiterals
    | LBE_ identifier stringLiterals RBE_
    ;

hexadecimalLiterals
    : HEX_DIGIT_
    ;

bitValueLiterals
    : BIT_NUM_
    ;

booleanLiterals
    : TRUE | FALSE
    ;

nullValueLiterals
    : NULL
    ;

identifier
    : IDENTIFIER_ | unreservedWord | DOUBLE_QUOTED_TEXT
    ;

unreservedWord
    : SELECT | INSERT | UPDATE | DELETE | CREATE | ALTER | DROP | TRUNCATE | SCHEMA | GRANT | REVOKE | ADD | SET | TABLE | COLUMN | INDEX | CONSTRAINT | PRIMARY | UNIQUE | FOREIGN | KEY | POSITION | PRECISION | FUNCTION | CONSTRUCTOR | RESULT | TRIGGER | MAP | PROCEDURE | SPECIFICATION | VIEW | INTO | VALUES | WITH | UNION | DISTINCT | CASE | WHEN | CAST | TRIM | SUBSTRING | FROM | NATURAL | JOIN | FULL | INNER | OUTER | LEFT | RIGHT | CROSS | USING | WHERE | AS | ON | IF | ELSE | ELSIF | THEN | FOR | TO | AND | OR | IS | NOT | NULL | TRUE | FALSE | EXISTS | BETWEEN | IN | ALL | ANY | LIKE | ORDER | GROUP | BY | ASC | DESC | HAVING | LIMIT | OFFSET | BEGIN | COMMIT | ROLLBACK | SAVEPOINT | BOOLEAN | BODY | DOUBLE | BYTE | CHAR | CHARACTER | ARRAY | INTERVAL | DATE | TIME | TIMEOUT | TIMESTAMP | LOCALTIME | LOCALTIMESTAMP | LOOP | EXIT | YEAR | QUARTER | MONTH | WEEK | DAY | HOUR | MINUTE | SECOND | MICROSECOND | TIMEZONE_HOUR | TIMEZONE_MINUTE | TIMEZONE_REGION | TIMEZONE_ABBR | MAX | MIN | SUM | COUNT | AVG | DEFAULT | DEFAULTS | CURRENT | ENABLE | DISABLE | CALL | INSTANCE | PRESERVE | DO | DEFINER | CURRENT_USER | SQL | CASCADED | LOCAL | CLOSE | OPEN | NEXT | NAME | COLLATION | NAMES | INTEGER | REAL | DECIMAL | TYPE | INT | SMALLINT | NUMERIC | FLOAT | TRIGGERS | GLOBAL_NAME | ROWTYPE | PERCENTILE_CONT | PERCENTILE_DISC | CUME_DIST | PARTITION | RANK | ROWID | REGR_SLOPE | REGR_INTERCEPT | REGR_COUNT | REGR_R2 | REGR_AVGX | REGR_AVGY | REGR_SXX | REGR_SYY | REGR_SXY | LPAD | ZONE | AT | SESSIONTIMEZONE | SYSTIMESTAMP | DBTIMEZONE | TO_CHAR | XMLELEMENT | XMLAGG | XMLCAST | COLUMN_VALUE | XMLCOLATTVAL | EVALNAME | XMLEXISTS | XMLFOREST | XMLPARSE | DOCUMENT | CONTENT | WELLFORMED | XMLPI | XMLQUERY | XMLROOT | STANDALONE | XMLSERIALIZE | ENCODING | INDENT | HIDE | SHOW | XMLTABLE | ISSCHEMAVALID | XMLNAMESPACES | ORDINALITY | PATH | EXTRACT | ANYDATA | ANYTYPE | ANYDATASET | AUTONOMOUS_TRANSACTION | WM_CONCAT | DECLARE | BULK | COLLECT | INSERTING | UPDATING | DELETING | ROWCOUNT | BULK_ROWCOUNT | RAISE | WHILE | GOTO | MUTABLE | IMMUTABLE | INDICES | PAIRS | PIPE | FORALL | HTTP | DIGEST | CONTAINER_DATA | CONDITION | EVALUATE | TERMINATED | WHITESPACE | OPTIONALLY | ENCLOSED | LRTRIM | GETLENGTH | APPI | APPINFO | ARRAYSIZE | AUTO | AUTOCOMMIT | AUTOP | AUTOPN | AUTORECOVERY | AUTOT | AUTOTRACE | BLO | BLOCKTERMINATOR | CMDS | CMDSEP | COLINVI | COLINVISIBLE | COLSEP | CON | CONCAT | COPYC | COPYCOMMIT | COPYTYPECHECK | DEF | DEFINE | DESCRIBE | ECHO | EDITF | EDITFILE | EMB | EMBEDDED | ERRORDETAILS | ERRORL | ERRORLOGGING | ESC | ESCAPE | ESCCHAR | EXITCOMMIT | OMMIT | FEEDBACK | FLAGGER | FLU | FLUSH | HEADING | HEA | HISTORY | HIST | JSONPRINT | LINESIZE | LIN | LOBOFFSET | LOBOF | LOGSOURCE | LONG | LONGCHUNKSIZE | MARK | MARKUP | NEWPAGE | NEWP | NUMFORMAT | NUMF | NUMWIDTH | NUM | PAGESIZE | PAGES | PAUSE | PAU | RECSEP | RECSEPCHAR | ROWLIMIT | ROWPREFETCH | SECUREDCOL | SERVEROUTPUT | SERVEROUT | SHIFTINOUT | SHIFT | SHOWMODE | SQLBLANKLINES | SQLBL | SQLCASE | SQLC | SQLCONTINUE | SQLCO | SQLNUMBER | SQLN | SQLPLUSCOMPATIBILITY | SQLPLUSCOMPAT | SQLPREFIX | SQLPRE | SQLPROMPT | SQLP | SQLTERMINATOR | SQLT | STATEMENTCACHE | STATEMENTC | SUFFIX | SUF | TAB | TERMOUT | TERM | TI | TIMING | TIMI | TRIMOUT | TRIMSPOOL | TRIMS | UNDERLINE | UND | VERIFY | VER | WRAP | WRA | XMLOPTIMIZATIONCHECK | XMLOPT | XQUERY
    | BEQUEATH | BINARY | ESCAPE | MOD | XOR | ROW | ROWS | UNKNOWN | ALWAYS | CASCADE | CHECK | CONVERSION | GENERATED | PRIVILEGES | READ | WRITE | REFERENCES | START | TRANSACTION | USER | ROLE | VISIBLE | INVISIBLE | EXECUTE | USE | DEBUG | UNDER | FLASHBACK | ARCHIVE | REFRESH | QUERY | REWRITE | KEEP | SEQUENCE | INHERIT | TRANSLATE | MERGE | AT | BITMAP | CACHE | NOCACHE | CHECKPOINT | CONNECT | CONSTRAINTS | CYCLE | NOCYCLE | DBTIMEZONE | ENCRYPT | DECRYPT | DEFERRABLE | DEFERRED | DIRECTORY | DIRECT_PATH | CREDENTIALS | EDITION | ELEMENT | END | EXCEPT | EXCEPTIONS | FORCE | NOFORCE | GLOBAL | IDENTIFIED | IDENTITY | IMMEDIATE | INCREMENT | INITIALLY | INVALIDATE | JAVA | LEVELS | MAXVALUE | MINVALUE | NOMAXVALUE | NOMINVALUE | NOSORT | MINING | MODEL | MODIFY | NATIONAL | NEW | NOORDER | NORELY | OF | ONLY | PRIOR | PROFILE | REF | REKEY | RELY | RENAME | REPLACE | RESOURCE | REVERSE | ROWID | SALT | SCOPE | SORT | SOURCE | SUBSTITUTABLE | TABLESPACE | TEMPORARY | TRANSLATION | TREAT | NO | UNUSED | VALIDATE | NOVALIDATE | VALUE | VARYING | VIRTUAL | ZONE | PUBLIC | SESSION | COMMENT | LOCK | ADVISOR | ADMINISTER | TUNING | MANAGE | MANAGEMENT | OBJECT | CLUSTER | CONTEXT | EXEMPT | REDACTION | POLICY | DATABASE | SYSTEM | AUDIT | LINK | ANALYZE | DICTIONARY | DIMENSION | INDEXTYPE | INDEXTYPES | EXTERNAL | JOB | CLASS | PROGRAM | SCHEDULER | LIBRARY | LOGMINING | MATERIALIZED | CUBE | MEASURE | FOLDER | BUILD | PROCESS | OPERATOR | OUTLINE | PLUGGABLE | CONTAINER | SEGMENT | RESTRICT | RESTRICTED | COST | SYNONYM | BACKUP | UNLIMITED | BECOME | CHANGE | NOTIFICATION | ACCESS | PRIVILEGE | PURGE | RESUMABLE | SYSGUID | SYSBACKUP | SYSDBA | SYSDG | SYSKM | SYSOPER | RECYCLEBIN | DBA_RECYCLEBIN | FIRST | NCHAR | RAW | VARCHAR | VARCHAR2 | STRING | NVARCHAR2 | LONG | BLOB | CLOB | NCLOB | BINARY_FLOAT | BINARY_DOUBLE | PLS_INTEGER | BINARY_INTEGER | NUMBER | NATURALN | POSITIVE | POSITIVEN | SIGNTYPE | SIMPLE_INTEGER | BFILE | MLSLABEL | UROWID | JSON | DEC | SHARING | PRIVATE | SHARDED | SHARD | DUPLICATED | METADATA | DATA | EXTENDED | NONE | MEMOPTIMIZE | PARENT | IDENTIFIER | WORK | CONTAINER_MAP | CONTAINERS_DEFAULT | WAIT | NOWAIT | BATCH | BLOCK | REBUILD | INVALIDATION | COMPILE | USABLE | UNUSABLE | ONLINE | MONITORING | NOMONITORING | USAGE | COALESCE | CLEANUP | PARALLEL | NOPARALLEL | LOG | REUSE | SETTINGS | STORAGE | MATCHED | ERRORS | REJECT | RETENTION | CHUNK | PCTVERSION | FREEPOOLS | AUTO | DEDUPLICATE | KEEP_DUPLICATES | COMPRESS | HIGH | MEDIUM | LOW | NOCOMPRESS | READS | CREATION | PCTFREE | PCTUSED | INITRANS | LOGGING | NOLOGGING | FILESYSTEM_LIKE_LOGGING | INITIAL | MINEXTENTS | MAXEXTENTS | BASIC | ADVANCED | PCTINCREASE | FREELISTS | DML | DDL | CAPACITY | FREELIST | GROUPS | OPTIMAL | BUFFER_POOL | RECYCLE | FLASH_CACHE | CELL_FLASH_CACHE | MAXSIZE | MAX_AUDIT_SIZE | MAX_DIAG_SIZE | STORE | LEVEL | LOCKING | INMEMORY | MEMCOMPRESS | PRIORITY | CRITICAL | DISTRIBUTE | RANGE | PARTITION | SUBPARTITION | SERVICE | DUPLICATE | ILM | DELETE_ALL | ENABLE_ALL | DISABLE_ALL | AFTER | MODIFICATION | DAYS | MONTHS | YEARS | TIER | ORGANIZATION | HEAP | PCTTHRESHOLD | PARAMETERS | LOCATION | MAPPING | NOMAPPING | INCLUDING | OVERFLOW | ATTRIBUTE | ATTRIBUTES | RESULT_CACHE | ROWDEPENDENCIES | NOROWDEPENDENCIES | ARCHIVAL | EXCHANGE | INDEXING | OFF | LESS | INTERNAL | VARRAY | NESTED | COLUMN_VALUE | RETURN | LOCATOR | MODE | LOB | SECUREFILE | BASICFILE | THAN | LIST | AUTOMATIC | HASH | PARTITIONS | SUBPARTITIONS | TEMPLATE | PARTITIONSET | REFERENCE | CONSISTENT | CLUSTERING | LINEAR | INTERLEAVED | YES | LOAD | MOVEMENT | ZONEMAP | WITHOUT | XMLTYPE | RELATIONAL | XML | VARRAYS | LOBS | TABLES | ALLOW | DISALLOW | NONSCHEMA | ANYSCHEMA | XMLSCHEMA | COLUMNS | OIDINDEX | EDITIONABLE | NONEDITIONABLE | DEPENDENT | INDEXES | SHRINK | SPACE | COMPACT | SUPPLEMENTAL | ADVISE | NOTHING | GUARD | SYNC | VISIBILITY | ACTIVE | DEFAULT_COLLATION | MOUNT | STANDBY | CLONE | RESETLOGS | NORESETLOGS | UPGRADE | DOWNGRADE | RECOVER | LOGFILE | TEST | CORRUPTION | CONTINUE | CANCEL | UNTIL | CONTROLFILE | SNAPSHOT | DATAFILE | MANAGED | ARCHIVED | DISCONNECT | NODELAY | INSTANCES | FINISH | LOGICAL | FILE | SIZE | AUTOEXTEND | BLOCKSIZE | OFFLINE | RESIZE | TEMPFILE | DATAFILES | ARCHIVELOG | MANUAL | NOARCHIVELOG | AVAILABILITY | PERFORMANCE | CLEAR | UNARCHIVED | UNRECOVERABLE | THREAD | MEMBER | PHYSICAL | FAR | TRACE | DISTRIBUTED | RECOVERY | FLUSH | NOREPLY | SWITCH | LOGFILES | PROCEDURAL | REPLICATION | SUBSET | ACTIVATE | APPLY | APPROX_RANK | MAXIMIZE | PROTECTION | SUSPEND | RESUME | QUIESCE | UNQUIESCE | SHUTDOWN | REGISTER | PREPARE | SWITCHOVER | FAILED | SKIP_SYMBOL | STOP | ABORT | VERIFY | CONVERT | FAILOVER | BIGFILE | SMALLFILE | TRACKING | CACHING | CONTAINERS | TARGET | UNDO | MOVE | MIRROR | COPY | UNPROTECTED | REDUNDANCY | REMOVE | LOST | LEAD_CDB | LEAD_CDB_URI | PROPERTY | DEFAULT_CREDENTIAL | TIME_ZONE | RESET | RELOCATE | CLIENT | PASSWORDFILE_METADATA_CACHE | NOSWITCH | POST_TRANSACTION | KILL | ROLLING | MIGRATION | PATCH | ENCRYPTION | WALLET | AFFINITY | MEMORY | SPFILE | PFILE | BOTH | SID | SHARED_POOL | BUFFER_CACHE | REDO | CONFIRM | MIGRATE | USE_STORED_OUTLINES | GLOBAL_TOPIC_ENABLED | INTERSECT | MINUS | LOCKED | FETCH | PERCENT | TIES | SIBLINGS | NULLS | LAST | ISOLATION | SERIALIZABLE | COMMITTED | FILTER | FACT | DETERMINISTIC | PIPELINED | PARALLEL_ENABLE | OUT | NOCOPY | ACCESSIBLE | PACKAGE | PACKAGES | USING_NLS_COMP | AUTHID | SEARCH | DEPTH | BREADTH | ANALYTIC | HIERARCHIES | MEASURES | OVER | LAG | LAG_DIFF | LAG_DIF_PERCENT | LEAD | LEAD_DIFF | LEAD_DIFF_PERCENT | HIERARCHY | WITHIN | ACROSS | ANCESTOR | BEGINNING | UNBOUNDED | PRECEDING | FOLLOWING | RANK | DENSE_RANK | AVERAGE_RANK | ROW_NUMBER | SHARE_OF | HIER_ANCESTOR | HIER_PARENT | HIER_LEAD | HIER_LAG | QUALIFY | HIER_CAPTION | HIER_DEPTH | HIER_DESCRIPTION | HIER_LEVEL | HIER_MEMBER_NAME | HIER_MEMBER_UNIQUE_NAME | CHAINED | STATISTICS | DANGLING | STRUCTURE | FAST | COMPLETE | ASSOCIATE | DISASSOCIATE | FUNCTIONS | TYPES | SELECTIVITY | RETURNING | VERSIONS | SCN | PERIOD | LATERAL | BADFILE | DISCARDFILE | PIVOT | UNPIVOT | INCLUDE | EXCLUDE | SAMPLE | SEED | OPTION | SHARDS | MATCH_RECOGNIZE | PATTERN | DEFINE | ONE | PER | MATCH | PAST | PERMUTE | CLASSIFIER | MATCH_NUMBER | RUNNING | FINAL | PREV | NOAUDIT | WHENEVER | SUCCESSFUL | USERS | GRANTED | ROLES | NAMESPACE | ROLLUP | GROUPING | SETS | DECODE | RESTORE | POINT | BEFORE | IGNORE | NAV | SINGLE | UPDATED | MAIN | RULES | UPSERT | SEQUENTIAL | ITERATE | DECREMENT | SOME | NAN | INFINITE | PRESENT | EMPTY | SUBMULTISET | LIKEC | LIKE2 | LIKE4 | REGEXP_LIKE | EQUALS_PATH | UNDER_PATH | FORMAT | STRICT | LAX | KEYS | JSON_EQUAL | JSON_EXISTS | PASSING | ERROR | JSON_TEXTCONTAINS | HAS | STARTS | LIKE_REGEX | EQ_REGEX | SYS | MAXDATAFILES | MAXINSTANCES | AL32UTF8 | AL16UTF16 | UTF8 | USER_DATA | MAXLOGFILES | MAXLOGMEMBERS | MAXLOGHISTORY | EXTENT | SYSAUX | LEAF | AUTOALLOCATE | UNIFORM | FILE_NAME_CONVERT | ALLOCATE | DEALLOCATE | SHARED | AUTHENTICATED | CHILD | DETERMINES | RELIES_ON | AGGREGATE | POLYMORPHIC | SQL_MARCO | LANGUAGE | AGENT | SELF | TDO | INDICATOR | STRUCT | LENGTH | DURATION | MAXLEN | CHARSETID | CHARSETFORM | SINGLE_C | SYSTIMESTAMP | CATEGORY | ORDER | NOKEEP | SCALE | NOSCALE | EXTEND | NOEXTEND | NOSHARD | INITIALIZED | EXTERNALLY | GLOBALLY | ACCESSED | RESTART | OPTIMIZE | QUOTA | DISKGROUP | NORMAL | FLEX | SITE | QUORUM | REGULAR | FAILGROUP | DISK | EXCLUDING | CONTENTS | LOCKDOWN | CLEAN | GUARANTEE | PRUNING | DEMAND | RESOLVE | RESOLVER | SHARE | EXCLUSIVE | ANCILLARY | BINDING | SCAN | COMPUTE | UNDROP | DISKS | COARSE | FINE | ALIAS | SCRUB | DISMOUNT | REBALANCE | COMPUTATION | CONSIDER | FRESH | MASTER | ENFORCED | TRUSTED | TRUST | ID | SYNCHRONOUS | ASYNCHRONOUS | REPEAT | FEATURE | STATEMENT | CLAUSE | UNPLUG | HOST | PORT | EVERY | MINUTES | HOURS | NORELOCATE | SAVE | DISCARD | STATE | APPLICATION | INSTALL | MINIMUM | VERSION | UNINSTALL | COMPATIBILITY | MATERIALIZE | SUBTYPE | RECORD | CONSTANT | CURSOR | OTHERS | EXCEPTION | CPU_PER_SESSION | CONNECT_TIME | AZURE_ROLE | AZURE_USER | IAM_GROUP_NAME | IAM_PRINCIPAL_NAME | LOGICAL_READS_PER_SESSION | PRIVATE_SGA | PERCENT_RANK | LISTAGG | ABS | ACCOUNT | ACOS | ACTIVE_COMPONENT | ACTIVE_FUNCTION | ACTIVE_TAG | ADD | ADD_COLUMN | ADD_GROUP | ADD_MONTHS | ADJ_DATE | ADMIN | ADMINISTRATOR | ALL | ALL_ROWS | ALTER | AND | AND_EQUAL | ANTIJOIN | APPEND | APPENDCHILDXML | APPEND_VALUES | AS | ASC | ASCII | ASCIISTR | ASIN | ASSEMBLY | ASYNC | ATAN | ATAN2 | AUTHENTICATION | AUTHORIZATION | BEGIN_OUTLINE_DATA | BEHALF | BETWEEN | BFILENAME | BINARY_DOUBLE_INFINITY | BINARY_DOUBLE_NAN | BINARY_FLOAT_INFINITY | BINARY_FLOAT_NAN | BIND_AWARE | BIN_TO_NUM | BITAND | BITMAPS | BITMAP_TREE | BITS | BLOCKS | BLOCK_RANGE | BOUND | BRANCH | BROADCAST | BUFFER | BULK | BY | BYPASS_RECURSIVE_CHECK | BYPASS_UJVC | BYTE | CACHE_CB | CACHE_INSTANCES | CACHE_TEMP_TABLE | CARDINALITY | CEIL | CERTIFICATE | CFILE | CHANGE_DUPKEY_ERROR_INDEX | CHAR | CHARTOROWID | CHAR_CS | CHECK_ACL_REWRITE | CHOOSE | CHR | CLOSE_CACHED_OPEN_CURSORS | CLUSTERING_FACTOR | CLUSTER_ID | CLUSTER_PROBABILITY | CLUSTER_SET | COALESCE_SQ | COLD | COLLECT | COLUMN | COLUMNAR | COLUMN_AUTH_INDICATOR | COLUMN_STATS | COMPLIANCE | COMPOSE | COMPOSITE | COMPOSITE_LIMIT | COMPOUND | CONCAT | CONFORMING | CONNECT_BY_CB_WHR_ONLY | CONNECT_BY_COMBINE_SW | CONNECT_BY_COST_BASED | CONNECT_BY_ELIM_DUPS | CONNECT_BY_FILTERING | CONNECT_BY_ISCYCLE | CONNECT_BY_ISLEAF | CONNECT_BY_ROOT | CONST | CORR | CORRUPT_XID | CORRUPT_XID_ALL | CORR_K | CORR_S | COS | COSH | COST_XML_QUERY_REWRITE | COVAR_POP | COVAR_SAMP | CO_AUTH_IND | CPU_COSTING | CPU_PER_CALL | CRASH | CREATE | CREATE_STORED_OUTLINES | CROSSEDITION | CSCONVERT | CUBE_GB | CUME_DIST | CUME_DISTM | CURRENT | CURRENTV | CURRENT_DATE | CURRENT_SCHEMA | CURRENT_TIME | CURRENT_TIMESTAMP | CURSOR_SHARING_EXACT | CURSOR_SPECIFIC_SEGMENT | CV | DATABASE_DEFAULT | DATAOBJNO | DATAOBJ_TO_PARTITION | DATE | DATE_MODE | DBA | DBMS_STATS | DB_ROLE_CHANGE | DB_VERSION | DEBUGGER | DECIMAL | DECLARE | DECOMPOSE | DECR | DEFAULT | DEFAULTS | DEFINED | DEGREE | DELAY | DELETE | DELETEXML | DENSE_RANKM | DEQUEUE | DEREF | DEREF_NO_REWRITE | DESC | DETACHED | DIRECT_LOAD | DISABLE_PRESET | DISABLE_RPKE | DISTINCT | DISTINGUISHED | DML_UPDATE | DOCFIDELITY | DOCUMENT | DOMAIN_INDEX_FILTER | DOMAIN_INDEX_NO_SORT | DOMAIN_INDEX_SORT | DRIVING_SITE | DROP | DROP_COLUMN | DROP_GROUP | DST_UPGRADE_INSERT_CONV | DUMP | DYNAMIC | DYNAMIC_SAMPLING | DYNAMIC_SAMPLING_EST_CDN | EACH | EDITIONING | EDITIONS | ELIMINATE_JOIN | ELIMINATE_OBY | ELIMINATE_OUTER_JOIN | ELSE | EMPTY_BLOB | EMPTY_CLOB | ENABLE_PRESET | ENCODING | END_OUTLINE_DATA | ENFORCE | ENQUEUE | ENTERPRISE | ENTITYESCAPING | ENTRY | ERROR_ARGUMENT | ERROR_ON_OVERLAP_TIME | ESTIMATE | EVALUATION | EVENTS | EXISTS | EXISTSNODE | EXP | EXPAND_GSET_TO_UNION | EXPAND_TABLE | EXPIRE | EXPLAIN | EXPLOSION | EXPORT | EXPR_CORR_CHECK | EXTENDS | EXTENTS | EXTRA | EXTRACT | EXTRACTVALUE | FACILITY | FACTORIZE_JOIN | FAILED_LOGIN_ATTEMPTS | FBTSCAN | FEATURE_ID | FEATURE_SET | FEATURE_VALUE | FIRSTM | FIRST_ROWS | FIRST_VALUE | FLAGGER | FLOAT | FLOB | FLOOR | FOLLOWS | FOR | FORCE_XML_QUERY_REWRITE | FOREVER | FORWARD | FROM | FROM_TZ | GATHER_PLAN_STATISTICS | GBY_CONC_ROLLUP | GBY_PUSHDOWN | GRANT | GREATEST | GROUP | GROUPING_ID | GROUP_BY | GROUP_ID | GUARANTEED | HASHKEYS | HASH_AJ | HASH_SJ | HAVING | HEADER | HELP | HEXTORAW | HEXTOREF | HIDE | HINTSET_BEGIN | HINTSET_END | HOT | HWM_BROKERED | HYBRID | IDGENERATORS | IDLE_TIME | IGNORE_OPTIM_EMBEDDED_HINTS | IGNORE_ROW_ON_DUPKEY_INDEX | IGNORE_WHERE_CLAUSE | IMPACT | IMPORT | IN | INCLUDE_VERSION | INCR | INCREMENTAL | INDENT | INDEX | INDEXED | INDEX_ASC | INDEX_COMBINE | INDEX_DESC | INDEX_FFS | INDEX_FILTER | INDEX_JOIN | INDEX_ROWS | INDEX_RRS | INDEX_RS | INDEX_RS_ASC | INDEX_RS_DESC | INDEX_SCAN | INDEX_SKIP_SCAN | INDEX_SS | INDEX_SS_ASC | INDEX_SS_DESC | INDEX_STATS | INFORMATIONAL | INITCAP | INLINE | INLINE_XMLTYPE_NT | INSERT | INSERTCHILDXML | INSERTCHILDXMLAFTER | INSERTCHILDXMLBEFORE | INSERTXMLAFTER | INSERTXMLBEFORE | INSTANTIABLE | INSTANTLY | INSTEAD | INSTR | INSTR2 | INSTR4 | INSTRB | INSTRC | INTEGER | INTERMEDIATE | INTERNAL_CONVERT | INTERNAL_USE | INTERPRETED | INTO | IN_MEMORY_METADATA | IN_XQUERY | IS | ISOLATION_LEVEL | ITERATION_NUMBER | INACTIVE_ACCOUNT_TIME | KERBEROS | KEYSIZE | KEY_LENGTH | LAST_DAY | LAST_VALUE | LAYER | LDAP_REGISTRATION | LDAP_REGISTRATION_ENABLED | LDAP_REG_SYNC_INTERVAL | LEADING | LEAST | LENGTH2 | LENGTH4 | LENGTHB | LENGTHC | LIFE | LIFETIME | LIKE | LIKE_EXPAND | LN | LNNVL | LOBNVL | LOCAL_INDEXES | LOGICAL_READS_PER_CALL | LOGOFF | LOGON | LOWER | LTRIM | MAKE_REF | MAXARCHLOGS | MAXTRANS | MEDIAN | MERGE_AJ | MERGE_CONST_ON | MERGE_SJ | METHOD | MINIMIZE | MINUS_NULL | MIRRORCOLD | MIRRORHOT | MODEL_COMPILE_SUBQUERY | MODEL_DONTVERIFY_UNIQUENESS | MODEL_DYNAMIC_SUBQUERY | MODEL_MIN_ANALYSIS | MODEL_NO_ANALYSIS | MODEL_PBY | MODEL_PUSH_REF | MONITOR | MONTHS_BETWEEN | MOUNTPATH | MULTISET | MV_MERGE | NAMED | NANVL | NATIVE | NATIVE_FULL_OUTER_JOIN | NCHAR_CS | NCHR | NEEDED | NESTED_TABLE_FAST_INSERT | NESTED_TABLE_GET_REFS | NESTED_TABLE_ID | NESTED_TABLE_SET_REFS | NESTED_TABLE_SET_SETID | NETWORK | NEVER | NEW_TIME | NEXT_DAY | NLJ_BATCHING | NLJ_INDEX_FILTER | NLJ_INDEX_SCAN | NLJ_PREFETCH | NLSSORT | NLS_CALENDAR | NLS_CHARACTERSET | NLS_CHARSET_DECL_LEN | NLS_CHARSET_ID | NLS_CHARSET_NAME | NLS_COMP | NLS_CURRENCY | NLS_DATE_FORMAT | NLS_DATE_LANGUAGE | NLS_INITCAP | NLS_ISO_CURRENCY | NLS_LANG | NLS_LANGUAGE | NLS_LENGTH_SEMANTICS | NLS_LOWER | NLS_NCHAR_CONV_EXCP | NLS_NUMERIC_CHARACTERS | NLS_SORT | NLS_SPECIAL_CHARS | NLS_TERRITORY | NLS_UPPER | NL_AJ | NL_SJ | NOAPPEND | NOCPU_COSTING | NOENTITYESCAPING | NOGUARANTEE | NOLOCAL | NOMINIMIZE | NOOVERRIDE | NOPARALLEL_INDEX | NOREPAIR | NOREVERSE | NOREWRITE | NOSCHEMACHECK | NOSEGMENT | NOSTRICT | NOT | NO_ACCESS | NO_BASETABLE_MULTIMV_REWRITE | NO_BIND_AWARE | NO_BUFFER | NO_CARTESIAN | NO_CHECK_ACL_REWRITE | NO_COALESCE_SQ | NO_CONNECT_BY_CB_WHR_ONLY | NO_CONNECT_BY_COMBINE_SW | NO_CONNECT_BY_COST_BASED | NO_CONNECT_BY_ELIM_DUPS | NO_CONNECT_BY_FILTERING | NO_COST_XML_QUERY_REWRITE | NO_CPU_COSTING | NO_DOMAIN_INDEX_FILTER | NO_DST_UPGRADE_INSERT_CONV | NO_ELIMINATE_JOIN | NO_ELIMINATE_OBY | NO_ELIMINATE_OUTER_JOIN | NO_EXPAND | NO_EXPAND_GSET_TO_UNION | NO_EXPAND_TABLE | NO_FACT | NO_FACTORIZE_JOIN | NO_FILTERING | NO_GBY_PUSHDOWN | NO_INDEX | NO_INDEX_FFS | NO_INDEX_SS | NO_LOAD | NO_MERGE | NO_MODEL_PUSH_REF | NO_MONITOR | NO_MONITORING | NO_MULTIMV_REWRITE | NO_NATIVE_FULL_OUTER_JOIN | NO_NLJ_BATCHING | NO_NLJ_PREFETCH | NO_ORDER_ROLLUPS | NO_OUTER_JOIN_TO_INNER | NO_PARALLEL | NO_PARALLEL_INDEX | NO_PARTIAL_COMMIT | NO_PLACE_DISTINCT | NO_PLACE_GROUP_BY | NO_PQ_MAP | NO_PRUNE_GSETS | NO_PULL_PRED | NO_PUSH_PRED | NO_PUSH_SUBQ | NO_PX_JOIN_FILTER | NO_QKN_BUFF | NO_QUERY_TRANSFORMATION | NO_REF_CASCADE | NO_RESULT_CACHE | NO_REWRITE | NO_SEMIJOIN | NO_SET_TO_JOIN | NO_SQL_TUNE | NO_STAR_TRANSFORMATION | NO_STATEMENT_QUEUING | NO_STATS_GSETS | NO_SUBQUERY_PRUNING | NO_SUBSTRB_PAD | NO_SWAP_JOIN_INPUTS | NO_TEMP_TABLE | NO_TRANSFORM_DISTINCT_AGG | NO_UNNEST | NO_USE_HASH | NO_USE_HASH_AGGREGATION | NO_USE_INVISIBLE_INDEXES | NO_USE_MERGE | NO_USE_NL | NO_XMLINDEX_REWRITE | NO_XMLINDEX_REWRITE_IN_SELECT | NO_XML_DML_REWRITE | NO_XML_QUERY_REWRITE | NTH_VALUE | NTILE | NULL | NULLIF | NUMTODSINTERVAL | NUMTOYMINTERVAL | NUM_INDEX_KEYS | NVL | NVL2 | OBJECTTOXML | OBJNO | OBJNO_REUSE | OCCURENCES | OID | OLAP | OLD | OLD_PUSH_PRED | OLTP | ON | OPAQUE | OPAQUE_TRANSFORM | OPAQUE_XCANONICAL | OPCODE | OPERATIONS | OPTIMIZER_FEATURES_ENABLE | OPTIMIZER_GOAL | OPT_ESTIMATE | OPT_PARAM | OR | ORADEBUG | ORA_BRANCH | ORA_CHECKACL | ORA_DST_AFFECTED | ORA_DST_CONVERT | ORA_DST_ERROR | ORA_GET_ACLIDS | ORA_GET_PRIVILEGES | ORA_HASH | ORA_ROWSCN | ORA_ROWSCN_RAW | ORA_ROWVERSION | ORA_TABVERSION | ORDERED | ORDERED_PREDICATES | ORDINALITY | OR_EXPAND | OR_PREDICATES | OTHER | OUTER_JOIN_TO_INNER | OUTLINE_LEAF | OUT_OF_LINE | OVERFLOW_NOMOVE | OVERLAPS | OWN | OWNER | OWNERSHIP | PARALLEL_INDEX | PARAM | PARITY | PARTIAL | PARTIALLY | PARTITION_HASH | PARTITION_LIST | PARTITION_RANGE | PASSWORD | PASSWORD_GRACE_TIME | PASSWORD_LIFE_TIME | PASSWORD_LOCK_TIME | PASSWORD_REUSE_MAX | PASSWORD_REUSE_TIME | PASSWORD_VERIFY_FUNCTION | PASSWORD_ROLLOVER_TIME | PATH | PATHS | PBL_HS_BEGIN | PBL_HS_END | PENDING | PERCENTILE_CONT | PERCENTILE_DISC | PERCENT_RANKM | PERMANENT | PERMISSION | PIKEY | PIV_GB | PIV_SSF | PLACE_DISTINCT | PLACE_GROUP_BY | PLAN | PLSCOPE_SETTINGS | PLSQL_CCFLAGS | PLSQL_CODE_TYPE | PLSQL_DEBUG | PLSQL_OPTIMIZE_LEVEL | PLSQL_WARNINGS | POWER | POWERMULTISET | POWERMULTISET_BY_CARDINALITY | PQ_DISTRIBUTE | PQ_MAP | PQ_NOMAP | PREBUILT | PRECEDES | PRECOMPUTE_SUBQUERY | PREDICATE_REORDERS | PREDICTION | PREDICTION_BOUNDS | PREDICTION_COST | PREDICTION_DETAILS | PREDICTION_PROBABILITY | PREDICTION_SET | PRESENTNNV | PRESENTV | PRESERVE_OID | PREVIOUS | PROJECT | PROPAGATE | PROTECTED | PULL_PRED | PUSH_PRED | PUSH_SUBQ | PX_GRANULE | PX_JOIN_FILTER | QB_NAME | QUERY_BLOCK | QUEUE | QUEUE_CURR | QUEUE_ROWP | RANDOM | RANDOM_LOCAL | RANKM | RAPIDLY | RATIO_TO_REPORT | RAWTOHEX | RAWTONHEX | RBA | RBO_OUTLINE | RDBA | RECORDS_PER_BLOCK | RECOVERABLE | REDUCED | REFERENCED | REFERENCING | REFTOHEX | REF_CASCADE_CURSOR | REGEXP_COUNT | REGEXP_INSTR | REGEXP_REPLACE | REGEXP_SUBSTR | REGR_AVGX | REGR_AVGY | REGR_COUNT | REGR_INTERCEPT | REGR_R2 | REGR_SLOPE | REGR_SXX | REGR_SXY | REGR_SYY | REMAINDER | REMOTE_MAPPED | REPAIR | REQUIRED | RESPECT | RESTORE_AS_INTERVALS | RESTRICT_ALL_REF_CONS | RETRY_ON_ROW_CHANGE | REVOKE | REWRITE_OR_ERROR | ROUND | ROWIDTOCHAR | ROWIDTONCHAR | ROWNUM | ROW_LENGTH | RPAD | RTRIM | RULE | SAVE_AS_INTERVALS | SB4 | SCALE_ROWS | SCAN_INSTANCES | SCHEMACHECK | SCN_ASCENDING | SD_ALL | SD_INHIBIT | SD_SHOW | SECUREFILE_DBA | SECURITY | SEG_BLOCK | SEG_FILE | SELECT | SEMIJOIN | SEMIJOIN_DRIVER | SEQUENCED | SERVERERROR | SESSIONS_PER_USER | SESSIONTZNAME | SESSION_CACHED_CURSORS | SET | SET_TO_JOIN | SEVERE | SHOW | SIGN | SIGNAL_COMPONENT | SIGNAL_FUNCTION | SIMPLE | SIN | SINGLETASK | SINH | SKIP_EXT_OPTIMIZER | SKIP_UNQ_UNUSABLE_IDX | SKIP_UNUSABLE_INDEXES | SMALLINT | SOUNDEX | SPLIT | SPREADSHEET | SQLLDR | SQL_TRACE | SQRT | STALE | STANDALONE | STANDBY_MAX_DATA_DELAY | STAR | STARTUP | STAR_TRANSFORMATION | STATEMENTS | STATEMENT_ID | STATEMENT_QUEUING | PRAGMA | RESTRICT_REFERENCES | RNDS | WNDS | RNPS | WNPS | OVERRIDING | STATIC | STATS_BINOMIAL_TEST | STATS_CROSSTAB | STATS_F_TEST | STATS_KS_TEST | STATS_MODE | STATS_MW_TEST | STATS_ONE_WAY_ANOVA | STATS_T_TEST_INDEP | STATS_T_TEST_INDEPU | STATS_T_TEST_ONE | STATS_T_TEST_PAIRED | STATS_WSR_TEST | STDDEV | STDDEV_POP | STDDEV_SAMP | STREAMS | STRIP | STRIPE_COLUMNS | STRIPE_WIDTH | SUBPARTITION_REL | SUBQUERIES | SUBQUERY_PRUNING | SUBSTR | SUBSTR2 | SUBSTR4 | SUBSTRB | SUBSTRC | SUMMARY | SWAP_JOIN_INPUTS | SYSASM | SYSDATE | SYSTEM_DEFINED | SYS_AUDIT | SYS_CHECKACL | SYS_CONNECT_BY_PATH | SYS_CONTEXT | SYS_DBURIGEN | SYS_DL_CURSOR | SYS_DM_RXFORM_CHR | SYS_DM_RXFORM_NUM | SYS_DOM_COMPARE | SYS_DST_PRIM2SEC | SYS_DST_SEC2PRIM | SYS_ET_BFILE_TO_RAW | SYS_ET_BLOB_TO_IMAGE | SYS_ET_IMAGE_TO_BLOB | SYS_ET_RAW_TO_BFILE | SYS_EXTPDTXT | SYS_EXTRACT_UTC | SYS_FBT_INSDEL | SYS_FILTER_ACLS | SYS_GETTOKENID | SYS_GET_ACLIDS | SYS_GET_PRIVILEGES | SYS_GUID | SYS_MAKEXML | SYS_MAKE_XMLNODEID | SYS_MKXMLATTR | SYS_OPTLOBPRBSC | SYS_OPTXICMP | SYS_OPTXQCASTASNQ | SYS_OP_ADT2BIN | SYS_OP_ADTCONS | SYS_OP_ALSCRVAL | SYS_OP_ATG | SYS_OP_BIN2ADT | SYS_OP_BITVEC | SYS_OP_BL2R | SYS_OP_BLOOM_FILTER | SYS_OP_BLOOM_FILTER_LIST | SYS_OP_C2C | SYS_OP_CAST | SYS_OP_CEG | SYS_OP_CL2C | SYS_OP_COMBINED_HASH | SYS_OP_COMP | SYS_OP_CONVERT | SYS_OP_COUNTCHG | SYS_OP_CSCONV | SYS_OP_CSCONVTEST | SYS_OP_CSR | SYS_OP_CSX_PATCH | SYS_OP_DECOMP | SYS_OP_DESCEND | SYS_OP_DISTINCT | SYS_OP_DRA | SYS_OP_DUMP | SYS_OP_EXTRACT | SYS_OP_GROUPING | SYS_OP_GUID | SYS_OP_IIX | SYS_OP_ITR | SYS_OP_LBID | SYS_OP_LOBLOC2BLOB | SYS_OP_LOBLOC2CLOB | SYS_OP_LOBLOC2ID | SYS_OP_LOBLOC2NCLOB | SYS_OP_LOBLOC2TYP | SYS_OP_LSVI | SYS_OP_LVL | SYS_OP_MAKEOID | SYS_OP_MAP_NONNULL | SYS_OP_MSR | SYS_OP_NICOMBINE | SYS_OP_NIEXTRACT | SYS_OP_NII | SYS_OP_NIX | SYS_OP_NOEXPAND | SYS_OP_NUMTORAW | SYS_OP_OIDVALUE | SYS_OP_OPNSIZE | SYS_OP_PAR | SYS_OP_PARGID | SYS_OP_PARGID_1 | SYS_OP_PAR_1 | SYS_OP_PIVOT | SYS_OP_R2O | SYS_OP_RAWTONUM | SYS_OP_RDTM | SYS_OP_REF | SYS_OP_RMTD | SYS_OP_ROWIDTOOBJ | SYS_OP_RPB | SYS_OP_TOSETID | SYS_OP_TPR | SYS_OP_TRTB | SYS_OP_UNDESCEND | SYS_OP_VECAND | SYS_OP_VECBIT | SYS_OP_VECOR | SYS_OP_VECXOR | SYS_OP_VERSION | SYS_OP_VREF | SYS_OP_VVD | SYS_OP_XPTHATG | SYS_OP_XPTHIDX | SYS_OP_XPTHOP | SYS_OP_XTXT2SQLT | SYS_ORDERKEY_DEPTH | SYS_ORDERKEY_MAXCHILD | SYS_ORDERKEY_PARENT | SYS_PARALLEL_TXN | SYS_PATHID_IS_ATTR | SYS_PATHID_IS_NMSPC | SYS_PATHID_LASTNAME | SYS_PATHID_LASTNMSPC | SYS_PATH_REVERSE | SYS_PXQEXTRACT | SYS_RID_ORDER | SYS_ROW_DELTA | SYS_SC_2_XMLT | SYS_SYNRCIREDO | SYS_TYPEID | SYS_UMAKEXML | SYS_XMLANALYZE | SYS_XMLCONTAINS | SYS_XMLCONV | SYS_XMLEXNSURI | SYS_XMLGEN | SYS_XMLAGG | SYS_XMLI_LOC_ISNODE | SYS_XMLI_LOC_ISTEXT | SYS_XMLLOCATOR_GETSVAL | SYS_XMLNODEID | SYS_XMLNODEID_GETCID | SYS_XMLNODEID_GETLOCATOR | SYS_XMLNODEID_GETOKEY | SYS_XMLNODEID_GETPATHID | SYS_XMLNODEID_GETPTRID | SYS_XMLNODEID_GETRID | SYS_XMLNODEID_GETSVAL | SYS_XMLNODEID_GETTID | SYS_XMLTRANSLATE | SYS_XMLTYPE2SQL | SYS_XMLT_2_SC | SYS_XQBASEURI | SYS_XQCASTABLEERRH | SYS_XQCODEP2STR | SYS_XQCODEPEQ | SYS_XQCON2SEQ | SYS_XQCONCAT | SYS_XQDELETE | SYS_XQDFLTCOLATION | SYS_XQDOC | SYS_XQDOCURI | SYS_XQED4URI | SYS_XQENDSWITH | SYS_XQERR | SYS_XQERRH | SYS_XQESHTMLURI | SYS_XQEXLOBVAL | SYS_XQEXSTWRP | SYS_XQEXTRACT | SYS_XQEXTRREF | SYS_XQEXVAL | SYS_XQFB2STR | SYS_XQFNBOOL | SYS_XQFNCMP | SYS_XQFNDATIM | SYS_XQFNLNAME | SYS_XQFNNM | SYS_XQFNNSURI | SYS_XQFNPREDTRUTH | SYS_XQFNQNM | SYS_XQFNROOT | SYS_XQFORMATNUM | SYS_XQFTCONTAIN | SYS_XQFUNCR | SYS_XQGETCONTENT | SYS_XQINDXOF | SYS_XQINSERT | SYS_XQINSPFX | SYS_XQIRI2URI | SYS_XQLANG | SYS_XQLLNMFRMQNM | SYS_XQMKNODEREF | SYS_XQNILLED | SYS_XQNODENAME | SYS_XQNORMSPACE | SYS_XQNORMUCODE | SYS_XQNSP4PFX | SYS_XQNSPFRMQNM | SYS_XQPFXFRMQNM | SYS_XQPOLYABS | SYS_XQPOLYADD | SYS_XQPOLYCEL | SYS_XQPOLYCST | SYS_XQPOLYCSTBL | SYS_XQPOLYDIV | SYS_XQPOLYFLR | SYS_XQPOLYMOD | SYS_XQPOLYMUL | SYS_XQPOLYRND | SYS_XQPOLYSQRT | SYS_XQPOLYSUB | SYS_XQPOLYUMUS | SYS_XQPOLYUPLS | SYS_XQPOLYVEQ | SYS_XQPOLYVGE | SYS_XQPOLYVGT | SYS_XQPOLYVLE | SYS_XQPOLYVLT | SYS_XQPOLYVNE | SYS_XQREF2VAL | SYS_XQRENAME | SYS_XQREPLACE | SYS_XQRESVURI | SYS_XQRNDHALF2EVN | SYS_XQRSLVQNM | SYS_XQRYENVPGET | SYS_XQRYVARGET | SYS_XQRYWRP | SYS_XQSEQ2CON | SYS_XQSEQ2CON4XC | SYS_XQSEQDEEPEQ | SYS_XQSEQINSB | SYS_XQSEQRM | SYS_XQSEQRVS | SYS_XQSEQSUB | SYS_XQSEQTYPMATCH | SYS_XQSTARTSWITH | SYS_XQSTATBURI | SYS_XQSTR2CODEP | SYS_XQSTRJOIN | SYS_XQSUBSTRAFT | SYS_XQSUBSTRBEF | SYS_XQTOKENIZE | SYS_XQTREATAS | SYS_XQXFORM | SYS_XQ_ASQLCNV | SYS_XQ_ATOMCNVCHK | SYS_XQ_NRNG | SYS_XQ_PKSQL2XML | SYS_XQ_UPKXML2SQL | TABLE | TABLESPACE_NO | TABLE_STATS | TABNO | TAN | TANH | TEMP_TABLE | THE | THEN | THROUGH | TIMES | TIMEZONE_ABBR | TIMEZONE_HOUR | TIMEZONE_MINUTE | TIMEZONE_OFFSET | TIMEZONE_REGION | TIV_GB | TIV_SSF | TO | TOPLEVEL | TO_BINARY_DOUBLE | TO_BINARY_FLOAT | TO_BLOB | TO_CLOB | TO_DATE | TO_DSINTERVAL | TO_LOB | TO_MULTI_BYTE | TO_NCHAR | TO_NCLOB | TO_NUMBER | TO_SINGLE_BYTE | TO_TIME | TO_TIMESTAMP | TO_TIMESTAMP_TZ | TO_TIME_TZ | TO_YMINTERVAL | TRACING | TRAILING | TRANSFORM_DISTINCT_AGG | TRANSITION | TRANSITIONAL | TRIGGER | TRUNC | TX | TZ_OFFSET | UB2 | UBA | UID | UNBOUND | UNION | UNIQUE | UNISTR | UNLOCK | UNNEST | UNPACKED | UNRESTRICTED | UPDATABLE | UPDATE | UPDATEXML | UPD_INDEXES | UPD_JOININDEX | UPPER | USERENV | USERGROUP | USER_DEFINED | USER_RECYCLEBIN | USE_ANTI | USE_CONCAT | USE_HASH | USE_HASH_AGGREGATION | USE_INVISIBLE_INDEXES | USE_MERGE | USE_MERGE_CARTESIAN | USE_NL | USE_NL_WITH_INDEX | USE_PRIVATE_OUTLINES | USE_SEMI | USE_TTT_FOR_GSETS | USE_WEAK_NAME_RESL | VALIDATION | VALUES | VARIANCE | VAR_POP | VAR_SAMP | VECTOR_READ | VECTOR_READ_TRACE | VERSIONING | VERSIONS_ENDSCN | VERSIONS_ENDTIME | VERSIONS_OPERATION | VERSIONS_STARTSCN | VERSIONS_STARTTIME | VERSIONS_XID | VIEW | VOLUME | VSIZE | WELLFORMED | WHERE | WHITESPACE | WIDTH_BUCKET | WITH | WRAPPED | XID | XMLATTRIBUTES | XMLCAST | XMLCDATA | XMLCOLATTVAL | XMLCOMMENT | XMLCONCAT | XMLDIFF | XMLEXISTS | XMLEXISTS2 | XMLFOREST | XMLINDEX_REWRITE | XMLINDEX_REWRITE_IN_SELECT | XMLINDEX_SEL_IDX_TBL | XMLISNODE | XMLISVALID | XMLNAMESPACES | XMLPARSE | XMLPATCH | XMLPI | XMLQUERY | XMLROOT | XMLSERIALIZE | XMLTABLE | XMLTOOBJECT | XMLTRANSFORM | XMLTRANSFORMBLOB | XML_DML_RWT_STMT | XPATHTABLE | XMLSEQUENCE | XS_SYS_CONTEXT | X_DYN_PRUNE | FEATURE_COMPARE | FEATURE_DETAILS | CLUSTER_DETAILS | CLUSTER_DISTANCE | PERSISTABLE | DB_RECOVERY_FILE_DEST_SIZE | MANDATORY | BACKUPS | NOPROMPT | IMMUTABLE | BLOCKCHAIN | IDLE | HASHING | DEFINITION | COLLATE | XDB | XMLINDEX | INDEX_ALL_PATHS | NONBLOCKING | MODIFY_COLUMN_TYPE | CERTIFICATE_DN | KERBEROS_PRINCIPAL_NAME | DELEGATE | TIME_UNIT | DEVICE | PARAMETER | SHO | ERR | CON_ID | CON_NAME | BTI | BTITLE | HISTORY | LNO | LOBPREFETCH | PDBS | PNO | RECYC | REL | RELEASE | REPF | REPFOOTER | REPH | REPHEADER | ROWPREF | ROWPREFETCH | SGA | SPOO | SPOOL | SPPARAMETER | SPPARAMETERS | SQLCODE | STATEMENTC | STATEMENTCACHE | TTI | TLE | XQUERY | SPO | CRE | REP | APP | EXCEPTION_INIT
    | SINGLE_K | SINGLE_M | SINGLE_G | SINGLE_T | SINGLE_P | SINGLE_E | SINGLE_H | V1
    ;

schemaName
    : identifier
    ;

tableName
    : (owner DOT_)? name
    ;

viewName
    : (owner DOT_)? name
    ;

triggerName
    : (owner DOT_)? name
    ;

materializedViewName
    : (owner DOT_)? name
    ;

columnName
    : (owner? DOT_)? name (DOT_ nestedItem)*
    ;

objectName
    : (owner DOT_)? name
    ;

clusterName
    : (owner DOT_)? name
    ;

indexName
    : (owner DOT_)? name
    ;

statisticsTypeName
    : (owner DOT_)? name
    ;

function
    : (owner DOT_)? name
    ;

packageName
    : (owner DOT_)? name
    ;

profileName
    : identifier
    ;

rollbackSegmentName
    : identifier
    ;

typeName
    : (owner DOT_)? name
    ;

indexTypeName
    : (owner DOT_)? name
    ;

modelName
    : (owner DOT_)? name
    ;

operatorName
    : (owner DOT_)? name
    ;

dimensionName
    : (owner DOT_)? name
    ;

directoryName
    : (owner DOT_)? name
    ;

constraintName
    : identifier
    ;

contextName
    : identifier
    ;

savepointName
    : identifier
    ;

synonymName
    : identifier
    ;

owner
    : identifier
    ;

name
    : identifier
    ;

tablespaceName
    : identifier
    ;

newTablespaceName
    : identifier
    ;

subprogramName
    : identifier
    ;

methodName
    : identifier
    ;

tablespaceSetName
    : identifier
    ;

serviceName
    : identifier
    ;

ilmPolicyName
    : identifier
    ;

policyName
    : identifier
    ;

connectionQualifier
    : identifier
    ;

functionName
    : identifier
    ;

cursorName
    : identifier
    ;

featureId
    : numberLiterals
    ;

dbLink
    : identifier (DOT_ identifier)*
    ;

parameterValue
    : literals | identifier
    ;

dispatcherName
    : stringLiterals
    ;

clientId
    : stringLiterals
    ;

opaqueFormatSpec
    : identifier
    ;

accessDriverType
    : identifier
    ;

varrayItem
    : identifier
    ;

nestedItem
    : identifier
    ;

storageTable
    : identifier
    ;

lobSegname
    : identifier
    ;

locationSpecifier
    : identifier
    ;

xmlSchemaURLName
    : identifier
    ;

elementName
    : identifier
    ;

subpartitionName
    : identifier
    ;

parameterName
    : identifier
    ;

editionName
    : identifier
    ;

outlineName
    : identifier
    ;

containerName
    : identifier
    ;

newName
    : identifier
    ;

partitionName
    : identifier
    ;

partitionSetName
    : identifier
    ;

partitionKeyValue
    : INTEGER_ | dateTimeLiterals | toDateFunction
    ;

subpartitionKeyValue
    : INTEGER_ | dateTimeLiterals
    ;

encryptAlgorithmName
    : STRING_
    ;

integrityAlgorithm
    : STRING_
    ;

zonemapName
    : identifier
    ;

flashbackArchiveName
    : identifier
    ;

roleName
    : identifier
    ;

username
    : identifier | STRING_
    ;

password
    : identifier | STRING_
    ;

logGroupName
    : identifier
    ;

columnNames
    : LP_? columnName (COMMA_ columnName)* RP_?
    ;

tableNames
    : LP_? tableName (COMMA_ tableName)* RP_?
    ;

oracleId
    : identifier | (STRING_ DOT_)* STRING_
    ;

collationName
    : STRING_ | IDENTIFIER_
    ;

columnCollationName
    : identifier
    ;

alias
    : identifier | STRING_
    ;

dataTypeLength
    : LP_ (length=INTEGER_ (COMMA_ (MINUS_)? INTEGER_)? type=(CHAR | BYTE)?)? RP_
    ;

primaryKey
    : PRIMARY KEY
    ;

exprs
    : expr (COMMA_ expr)*
    ;

exprList
    : LP_ exprs RP_
    ;

// TODO comb expr
expr
    : expr andOperator expr
    | expr orOperator expr
    | notOperator expr
    | LP_ expr RP_
    | booleanPrimary
    | expr datetimeExpr
    | multisetExpr
    | builtinFunctionsExpr
    | dataManipulationLanguageExpr
    ;

dataManipulationLanguageExpr
    : UPDATING | INSERTING | DELETING
    ;

andOperator
    : AND | AND_
    ;

orOperator
    : OR | OR_
    ;

notOperator
    : NOT | NOT_
    ;

booleanPrimary
    : booleanPrimary IS NOT? (TRUE | FALSE | UNKNOWN | NULL | NAN)
    | (PRIOR | DISTINCT) predicate
    | CONNECT_BY_ROOT predicate
    | booleanPrimary SAFE_EQ_ predicate
    | booleanPrimary comparisonOperator (ALL | ANY) subquery
    | booleanPrimary comparisonOperator predicate
    | predicate
    ;

comparisonOperator
    : EQ_ | GTE_ | GT_ | LTE_ | LT_ | NEQ_ | (GT_ | LT_) EQ_ | LT_ GT_ | (NOT_ | CARET_) EQ_
    ;

predicate
    : bitExpr NOT? IN subquery
    | PRIOR predicate
    | bitExpr NOT? IN LP_ expr (COMMA_ expr)* RP_
    | bitExpr NOT? IN LP_ expr (COMMA_ expr)* RP_ AND predicate
    | bitExpr NOT? IN stringLiterals
    | bitExpr NOT? BETWEEN bitExpr AND predicate
    | bitExpr NOT? LIKE simpleExpr (ESCAPE simpleExpr)?
    | bitExpr
    ;

bitExpr
    : bitExpr VERTICAL_BAR_ bitExpr
    | bitExpr AMPERSAND_ bitExpr
    | bitExpr SIGNED_LEFT_SHIFT_ bitExpr
    | bitExpr SIGNED_RIGHT_SHIFT_ bitExpr
    | bitExpr PLUS_ bitExpr
    | simpleExpr
    | bitExpr MINUS_ bitExpr
    | bitExpr ASTERISK_ bitExpr
    | bitExpr SLASH_ bitExpr
    | bitExpr MOD_ bitExpr
    | bitExpr CARET_ bitExpr
    | bitExpr DOT_ bitExpr
    | bitExpr ARROW_ bitExpr
    ;

simpleExpr
    : functionCall
    | pseudorecord
    | parameterMarker
    | literals
    | simpleExpr OR_ simpleExpr
    | (PLUS_ | MINUS_ | TILDE_ | NOT_ | BINARY) simpleExpr
    | ROW? LP_ expr (COMMA_ expr)* RP_
    | EXISTS? subquery
    | LBE_ identifier expr RBE_
    | caseExpression
    | columnName joinOperator?
    | privateExprOfDb
    | PRIOR identifier
    ;

functionCall
    : aggregationFunction | analyticFunction | specialFunction | regularFunction | xmlFunction
    ;

pseudorecord
    : COLON_ (NEW | OLD | PARENT) DOT_ identifier
    ;

aggregationFunction
    : aggregationFunctionName LP_ (((DISTINCT | ALL)? expr (COMMA_ expr)*) | ASTERISK_) (COMMA_ stringLiterals)? listaggOverflowClause? orderByClause? RP_ (WITHIN GROUP LP_ orderByClause RP_)? keepClause? overClause? overClause?
    ;

keepClause
    : KEEP LP_ DENSE_RANK (FIRST | LAST) orderByClause RP_ overClause?
    ;

aggregationFunctionName
    : MAX | MIN | SUM | COUNT | AVG | GROUPING | LISTAGG | PERCENT_RANK | PERCENTILE_CONT | PERCENTILE_DISC | CUME_DIST | RANK
    | REGR_SLOPE | REGR_INTERCEPT | REGR_COUNT | REGR_R2 | REGR_AVGX | REGR_AVGY | REGR_SXX | REGR_SYY | REGR_SXY
    | COLLECT | CORR | CORR_S | CORR_K | COVAR_POP | COVAR_SAMP | DENSE_RANK | FIRST
    | GROUP_ID | GROUPING_ID | LAST | MEDIAN | STATS_BINOMIAL_TEST | STATS_CROSSTAB | STATS_F_TEST | STATS_KS_TEST
    | STATS_MODE | STATS_MW_TEST | STATS_ONE_WAY_ANOVA | STATS_T_TEST_ONE | STATS_T_TEST_PAIRED | STATS_T_TEST_INDEP
    | STATS_T_TEST_INDEPU | STATS_WSR_TEST | STDDEV | STDDEV_POP | STDDEV_SAMP | VAR_POP | VAR_SAMP | VARIANCE
    ;

listaggOverflowClause
    : ON OVERFLOW (ERROR | (TRUNCATE stringLiterals? ((WITH | WITHOUT) COUNT)?))
    ;

analyticClause
    : queryPartitionClause? (orderByClause windowingClause?)?
    ;

queryPartitionClause
    : PARTITION BY (exprs | exprList)
    ;

windowingClause
    : (ROWS | RANGE) ((BETWEEN (UNBOUNDED PRECEDING | CURRENT ROW | expr (PRECEDING | FOLLOWING)) AND (UNBOUNDED FOLLOWING | CURRENT ROW | expr (PRECEDING | FOLLOWING)))
    | (UNBOUNDED PRECEDING | CURRENT ROW | expr PRECEDING))
    ;

analyticFunction
    : specifiedAnalyticFunctionName = (LEAD | LAG) ((LP_ expr leadLagInfo? RP_ respectOrIgnoreNulls?) | (LP_ expr respectOrIgnoreNulls? leadLagInfo? RP_)) overClause
    | specifiedAnalyticFunctionName = (NTILE | MEDIAN | RATIO_TO_REPORT) LP_ expr RP_ overClause?
    | specifiedAnalyticFunctionName = NTH_VALUE LP_ expr COMMA_ expr RP_ fromFirstOrLast? respectOrIgnoreNulls? overClause
    | specifiedAnalyticFunctionName = (PERCENTILE_CONT | PERCENTILE_DISC | LISTAGG) LP_ expr (COMMA_ expr)* RP_ WITHIN GROUP LP_ orderByClause RP_ overClause?
    | specifiedAnalyticFunctionName = (CORR | COVAR_POP | COVAR_SAMP) LP_ expr COMMA_ expr RP_ overClause?
    | specifiedAnalyticFunctionName = (PERCENT_RANK | RANK | ROW_NUMBER) LP_ RP_ overClause
    | analyticFunctionName LP_ dataType* RP_ overClause
    ;

fromFirstOrLast
    : FROM FIRST | FROM LAST
    ;

leadLagInfo
    : COMMA_ expr (COMMA_ expr)?
    ;

specialFunction
    : castFunction | charFunction | extractFunction | formatFunction | firstOrLastValueFunction | trimFunction | featureFunction
    | setFunction | translateFunction | cursorFunction | toDateFunction | approxRank | wmConcatFunction | predictionCostFunction | jsonObjectFunction
    ;

jsonObjectFunction
    : JSON_OBJECT LP_ jsonObjectContent RP_
    ;

jsonObjectContent
    : jsonObjectKeyValue (COMMA_ jsonObjectKeyValue)* jsonOnNullClause? jsonReturnClause?
    ;

jsonOnNullClause
    : (NULL | ABSENT) ON NULL
    ;

jsonReturnClause
    : RETURNING VARCHAR2 (LP_ INTEGER_ ( BYTE | CHAR)? RP_)?
    ;

jsonObjectKeyValue
    : (KEY? expr VALUE expr) (FORMAT JSON)?
    ;

predictionCostFunction
    : PREDICTION_COST LP_ (schemaName DOT_)? modelName (COMMA_ classExpr=expr)? costMatrixClause? miningAttributeClause? RP_
    ;

wmConcatFunction
    : (owner DOT_)? WM_CONCAT LP_ expr RP_ overClause?
    ;

approxRank
    : APPROX_RANK LP_ queryPartitionClause? orderByClause RP_
    ;

toDateFunction
    : TO_DATE LP_ char=STRING_ (DEFAULT returnValue=STRING_ ON CONVERSION ERROR)? (COMMA_ fmt=STRING_ (COMMA_ STRING_)?)? RP_
    ;

cursorFunction
    : CURSOR subquery
    ;

translateFunction
    : TRANSLATE LP_ expr USING (CHAR_CS | NCHAR_CS) RP_
    ;

setFunction
    : SET LP_ expr RP_
    ;

featureFunction
    : featureFunctionName LP_ (schemaName DOT_)? modelName (COMMA_ featureId)? (COMMA_ numberLiterals (COMMA_ numberLiterals)?)?
    (DESC | ASC | ABS)? costMatrixClause? miningAttributeClause (AND miningAttributeClause)? RP_
    ;

featureFunctionName
    : FEATURE_COMPARE | FEATURE_DETAILS | FEATURE_SET | FEATURE_ID | FEATURE_VALUE | CLUSTER_DETAILS | CLUSTER_DISTANCE | CLUSTER_ID | CLUSTER_PROBABILITY | CLUSTER_SET
    | PREDICTION_PROBABILITY | PREDICTION_SET | PREDICTION_BOUNDS | PREDICTION | PREDICTION_DETAILS
    ;

costMatrixClause
    : COST (MODEL (AUTO)?)? | LP_ literals RP_ (COMMA_ LP_ literals RP_)* VALUES LP_ LP_ literals (COMMA_ literals)* RP_ (COMMA_ LP_ literals (COMMA_ literals)* RP_) RP_
    ;

miningAttributeClause
    : USING (ASTERISK_ | ((schemaName DOT_)? tableName DOT_ ASTERISK_ | expr (AS? alias)?) (COMMA_ ((schemaName DOT_)? tableName DOT_ ASTERISK_ | expr (AS? alias)?))*)
    ;

trimFunction
    : TRIM LP_ trimOperands? expr RP_
    ;

trimOperands
    : (trimType expr? FROM) | (expr FROM)
    ;

trimType
    : LEADING
    | TRAILING
    | BOTH
    ;

firstOrLastValueFunction
    : (FIRST_VALUE | LAST_VALUE)  (LP_ expr respectOrIgnoreNulls? RP_ | LP_ expr RP_ respectOrIgnoreNulls?) overClause
    ;

respectOrIgnoreNulls
    : (RESPECT | IGNORE) NULLS
    ;

overClause
    : OVER LP_ analyticClause RP_
    ;

formatFunction
    : (owner DOT_)* name DOT_ FORMAT LP_ expr (COMMA_ expr)* RP_
    ;

castFunction
    : CAST LP_ ((MULTISET subquery) | expr) AS dataType RP_
    | XMLCAST LP_ expr AS dataType RP_
    ;

charFunction
    : (CHR | CHAR) LP_ expr (COMMA_ expr)* (USING charSet)? RP_
    ;

charSet
    : NCHAR_CS
    | ignoredIdentifier
    ;

extractFunction
    : EXTRACT LP_ (YEAR | MONTH | DAY | HOUR | MINUTE | SECOND | TIMEZONE_HOUR | TIMEZONE_MINUTE | TIMEZONE_REGION | TIMEZONE_ABBR) FROM expr RP_
    ;

regularFunction
    : (owner DOT_)? regularFunctionName LP_ (expr (COMMA_ expr)* | ASTERISK_)? RP_
    ;

regularFunctionName
    : identifier | IF | LOCALTIME | LOCALTIMESTAMP | INTERVAL | DECODE | TRANSLATE
    ;

joinOperator
    : LP_ PLUS_ RP_
    ;

caseExpression
    : CASE simpleExpr? caseWhen+ caseElse? END
    ;

caseWhen
    : WHEN expr THEN expr
    ;

caseElse
    : ELSE expr
    ;

subquery
    : matchNone
    ;

orderByClause
    : ORDER SIBLINGS? BY orderByItem (COMMA_ orderByItem)*
    ;

orderByItem
    : (columnName | numberLiterals | expr) (ASC | DESC)? (NULLS FIRST | NULLS LAST)?
    ;

attribute
    : (owner DOT_)? identifier
    ;

attributeName
    : oracleId
    ;

simpleExprs
    : simpleExpr (COMMA_ simpleExpr)*
    ;

lobItem
    : attributeName | columnName
    ;

lobItems
    : lobItem (COMMA_ lobItem)*
    ;

lobItemList
    : LP_ lobItems RP_
    ;

dataType
    : dataTypeName dataTypeLength? | specialDatatype | dataTypeName dataTypeLength? datetimeTypeSuffix | customDataType | typeAttribute
    ;

customDataType
    : dataTypeName DOT_ identifier
    ;

specialDatatype
    : dataTypeName (LP_ NUMBER_ CHAR RP_) | NATIONAL? dataTypeName VARYING? LP_ (INTEGER_ | NUMBER_) RP_ | (SYS DOT_)? dataTypeName
    ;

dataTypeName
    : CHARACTER | CHAR | NCHAR | RAW | VARCHAR | VARCHAR2 | NVARCHAR2 | LONG | LONG RAW | BLOB | CLOB | NCLOB | BINARY_FLOAT | BINARY_DOUBLE
    | BOOLEAN | PLS_INTEGER | BINARY_INTEGER | INTEGER | NUMBER | NATURAL | NATURALN | POSITIVE | POSITIVEN | SIGNTYPE
    | SIMPLE_INTEGER | BFILE | MLSLABEL | UROWID | DATE | TIMESTAMP | TIMESTAMP WITH TIME ZONE | TIMESTAMP WITH LOCAL TIME ZONE
    | INTERVAL DAY TO SECOND | INTERVAL YEAR TO MONTH | JSON | FLOAT | REAL | DOUBLE PRECISION | INT | SMALLINT
    | DECIMAL | NUMERIC | DEC | IDENTIFIER_ | XMLTYPE | ROWID | ANYDATA | ANYTYPE | ANYDATASET
    ;

datetimeTypeSuffix
    : (WITH LOCAL? TIME ZONE)? | TO MONTH | TO SECOND (LP_ NUMBER_ RP_)?
    ;

typeAttribute
    : (variableName | objectName) MOD_ TYPE
    ;

treatFunction
    : TREAT LP_ expr AS REF? dataTypeName RP_
    ;

privateExprOfDb
    : treatFunction | caseExpr | intervalExpression | objectAccessExpression | constructorExpr
    ;

caseExpr
    : CASE (simpleCaseExpr | searchedCaseExpr) elseClause? END
    ;

simpleCaseExpr
    : expr searchedCaseExpr+
    ;

searchedCaseExpr
    : WHEN expr THEN simpleExpr
    ;

elseClause
    : ELSE expr
    ;

intervalExpression
    : LP_ expr MINUS_ expr RP_ (intervalDayToSecondExpression | intervalYearToMonthExpression)
    ;

intervalDayToSecondExpression
    : DAY (LP_ leadingFieldPrecision RP_)? TO SECOND (LP_ fractionalSecondPrecision RP_)?
    ;

intervalYearToMonthExpression
    : YEAR (LP_ leadingFieldPrecision RP_)? TO MONTH
    ;

leadingFieldPrecision
    : INTEGER_
    ;

fractionalSecondPrecision
    : INTEGER_
    ;

objectAccessExpression
    : (LP_ simpleExpr RP_ | treatFunction) DOT_ (attributeName (DOT_ attributeName)* (DOT_ functionCall)? | functionCall)
    ;

constructorExpr
    : NEW dataTypeName exprList
    ;

ignoredIdentifier
    : IDENTIFIER_
    ;

ignoredIdentifiers
    : ignoredIdentifier (COMMA_ ignoredIdentifier)*
    ;

matchNone
    : 'Default does not match anything'
    ;

odciParameters
    : STRING_
    ;

databaseName
    : identifier
    ;

locationName
    : STRING_
    ;

fileName
    : identifier | STRING_
    ;

asmFileName
    : fullyQualifiedFileName
    | numericFileName
    | incompleteFileName
    | aliasFileName
    ;

fullyQualifiedFileName
    : PLUS_ diskgroupName SLASH_ dbName SLASH_ fileType SLASH_ fileTypeTag DOT_ fileNumber DOT_ incarnationNumber
    ;

dbName
    : identifier
    ;

fileType
    : CONTROLFILE
    | DATAFILE
    | ONLINELOG
    | ARCHIVELOG
    | TEMPFILE
    | BACKUPSET
    | PARAMETERFILE
    | DATAGUARDCONFIG
    | FLASHBACK
    | CHANGETRACKING
    | DUMPSET
    | XTRANSPORT
    | AUTOBACKUP
    ;

fileTypeTag
    : currentBackup
    | tablespaceName
    | groupGroup POUND_
    | threadThread POUND_ UL_ seqSequence POUND_
    | hasspfileTimestamp
    | serverParameterFile
    | dbName
    | logLog POUND_
    | changeTrackingFile
    | userObj POUND_ UL_ fileName POUND_
    ;

currentBackup
    : identifier
    ;

groupGroup
    : identifier
    ;

threadThread
    : identifier
    ;

seqSequence
    : identifier
    ;

hasspfileTimestamp
    : timestampValue
    ;

serverParameterFile
    : identifier
    ;

logLog
    : identifier
    ;

changeTrackingFile
    : identifier
    ;

userObj
    : identifier
    ;

numericFileName
    : PLUS_ diskgroupName DOT_ fileNumber DOT_ incarnationNumber
    ;

incompleteFileName
    : PLUS_ diskgroupName (LP_ templateName RP_)?
    ;

aliasFileName
    : PLUS_ diskgroupName (LP_ templateName RP_)? SLASH_ aliasName
    ;

fileNumber
    : INTEGER_
    ;

incarnationNumber
    : INTEGER_
    ;

instanceName
    : STRING_
    ;

logminerSessionName
    : identifier
    ;

tablespaceGroupName
    : identifier | STRING_
    ;

copyName
    : identifier
    ;

mirrorName
    : identifier
    ;

uriString
    : identifier
    ;

qualifiedCredentialName
    : identifier
    ;

pdbName
    : identifier
    ;

diskgroupName
    : identifier
    ;

templateName
    : identifier
    ;

aliasName
    : pathString
    ;

domain
    : identifier
    ;

dateValue
    : dateTimeLiterals | stringLiterals | numberLiterals | expr
    ;

sessionId
    : STRING_
    ;

serialNumber
    : numberLiterals
    ;

instanceId
    : NUMBER_
    ;

sqlId
    : identifier
    ;

logFileName
    : stringLiterals
    ;

logFileGroupsArchivedLocationName
    : stringLiterals
    ;

asmVersion
    : stringLiterals
    ;

walletPassword
    : identifier
    ;

hsmAuthString
    : identifier
    ;

targetDbName
    : identifier
    ;

certificateId
    : identifier
    ;

categoryName
    : identifier
    ;

offset
    : numberLiterals | expr | nullValueLiterals
    ;

rowcount
    : numberLiterals | expr | nullValueLiterals
    ;

percent
    : numberLiterals | expr | nullValueLiterals
    ;

rollbackSegment
    : identifier
    ;

queryName
    : alias
    ;

cycleValue
    : STRING_
    ;

noCycleValue
    : STRING_
    ;

orderingColumn
    : columnName
    ;

subavName
    : (owner DOT_)? name
    ;

baseAvName
    : (owner DOT_)? name
    ;

measName
    : identifier
    ;

levelRef
    : identifier
    ;

offsetExpr
    : expr | numberLiterals
    ;

memberKeyExpr
    : identifier
    ;

depthExpression
    : identifier
    ;

unitName
    : (owner DOT_)? name
    ;

procedureName
    : identifier
    ;

cpuCost
    : INTEGER_
    ;

ioCost
    : INTEGER_
    ;

networkCost
    : INTEGER_
    ;

defaultSelectivity
    : INTEGER_
    ;

dataItem
    : variableName
    ;

variableName
    : identifier | stringLiterals
    ;

validTimeColumn
    : columnName
    ;

attrDim
    : identifier
    ;

hierarchyName
    : (owner DOT_)? name
    ;

analyticViewName
    : (owner DOT_)? name
    ;

namespace
    : identifier
    ;

restorePoint
    : identifier
    | stringLiterals
    ;

scnValue
    : literals
    ;

timestampValue
    : LP_? expr+ RP_?
    ;

scnTimestampExpr
    : scnValue | timestampValue
    ;

referenceModelName
    : identifier
    ;

mainModelName
    : identifier
    ;

measureColumn
    : columnName
    ;

dimensionColumn
    : columnName
    ;

pattern
    : stringLiterals
    ;

analyticFunctionName
    : identifier
    ;

condition
    : comparisonCondition
    | floatingPointCondition
    | condition (AND | OR) condition | NOT condition
    | modelCondition
    | multisetCondition
    | patternMatchingCondition
    | rangeCondition
    | nullCondition
    | xmlCondition
    | jsonCondition
    | LP_ condition RP_ | NOT condition | condition (AND | OR) condition
    | existsCondition
    | inCondition
    | isOfTypeCondition
    ;

comparisonCondition
    : simpleComparisonCondition | groupComparisonCondition
    ;

simpleComparisonCondition
    : (expr (EQ_ | NEQ_ | GT_ | LT_ | GTE_ | LTE_) expr)
    | (exprList (EQ_ | NEQ_) LP_ (expressionList | subquery) RP_)
    ;

expressionList
    : exprs | LP_ expr? (COMMA_ expr?)* RP_
    ;

groupComparisonCondition
    : (expr (EQ_ | NEQ_ | GT_ | LT_ | GTE_ | LTE_) (ANY | SOME | ALL) LP_ (expressionList | subquery) RP_)
    | (exprList (EQ_ | NEQ_) (ANY | SOME | ALL) LP_ ((expressionList (SQ_ expressionList)*) | subquery) RP_)
    ;

floatingPointCondition
    : expr IS NOT? (NAN | INFINITE)
    ;

logicalCondition
    : (condition (AND | OR) condition) | NOT condition
    ;

modelCondition
    : isAnyCondition | isPresentCondition
    ;

isAnyCondition
    : (dimensionColumn IS)? ANY
    ;

isPresentCondition
    : cellReference IS PRESENT
    ;

cellReference
    : identifier
    ;

multisetCondition
    : isASetCondition
    | isEmptyCondition
    | memberCondition
    | submultisetCondition
    ;

isASetCondition
    : tableName IS NOT? A SET
    ;

isEmptyCondition
    : tableName IS NOT? EMPTY
    ;

memberCondition
    : expr NOT? MEMBER OF? tableName
    ;

submultisetCondition
    : tableName NOT? SUBMULTISET OF? tableName
    ;

patternMatchingCondition
    : likeCondition | regexpLikeCondition
    ;

likeCondition
    : searchValue NOT? (LIKE | LIKEC | LIKE2 | LIKE4) pattern (ESCAPE escapeChar)?
    ;

searchValue
    : identifier | stringLiterals
    ;

escapeChar
    : stringLiterals
    ;

regexpLikeCondition
    : REGEXP_LIKE LP_ searchValue COMMA_ pattern (COMMA_ matchParam)? RP_
    ;

matchParam
    : stringLiterals
    ;

rangeCondition
    : expr NOT? BETWEEN expr AND expr
    ;

nullCondition
    : expr IS NOT? NULL
    ;

xmlCondition
    : equalsPathCondition | underPathCondition
    ;

equalsPathCondition
    : EQUALS_PATH LP_ columnName COMMA_ pathString (COMMA_ correlationInteger)? RP_
    ;

pathString
    : stringLiterals
    ;

correlationInteger
    : INTEGER_
    ;

underPathCondition
    : UNDER_PATH LP_ columnName (COMMA_ levels)? COMMA_ pathString (COMMA_ correlationInteger)? RP_
    ;

level
    : identifier
    ;

levels
    : INTEGER_
    ;

jsonCondition
    : isJsonCondition | jsonExistsCondition | jsonTextcontainsCondition
    ;

isJsonCondition
    : expr IS NOT? JSON (FORMAT JSON)? (STRICT | LAX)? ((WITH | WITHOUT) UNIQUE KEYS)?
    ;

jsonEqualCondition
    : JSON_EQUAL LP_ expr COMMA_ expr RP_
    ;

jsonExistsCondition
    : JSON_EXISTS LP_ expr (FORMAT JSON)? COMMA_ jsonBasicPathExpr
    jsonPassingClause? jsonExistsOnErrorClause? jsonExistsOnEmptyClause? RP_
    ;

jsonPassingClause
    : PASSING expr AS identifier (COMMA_ expr AS identifier)*
    ;

jsonExistsOnErrorClause
    : (ERROR | TRUE | FALSE) ON ERROR
    ;

jsonExistsOnEmptyClause
    : (ERROR | TRUE | FALSE) ON EMPTY
    ;

jsonTextcontainsCondition
    : JSON_TEXTCONTAINS LP_ columnName COMMA_ jsonBasicPathExpr COMMA_ stringLiterals RP_
    ;

jsonBasicPathExpr
    : jsonAbsolutePathExpr | jsonRelativePathExpr
    ;

jsonAbsolutePathExpr
    : DOLLAR_ jsonNonfunctionSteps? jsonFunctionStep?
    ;

jsonNonfunctionSteps
    : ((jsonObjectStep | jsonArrayStep | jsonDescendentStep) jsonFilterExpr?)+
    ;

jsonObjectStep
    : DOT_ASTERISK_ | DOT_ jsonFieldName
    ;

jsonFieldName
    : jsonString | (letter (letter | digit)*)
    ;

letter
    : identifier
    ;

digit
    : numberLiterals
    ;

jsonArrayStep
    : LBT_ (ASTERISK_ | INTEGER_ (TO INTEGER_)? (COMMA_ INTEGER_ (TO INTEGER_)?)*) RBT_
    ;

jsonDescendentStep
    : DOT_ DOT_ jsonFieldName
    ;

jsonFunctionStep
    : DOT_ jsonItemMethod LP_ RP_
    ;

jsonItemMethod
    : identifier
    ;

jsonFilterExpr
    : QUESTION_ LP_ jsonCond RP_
    ;

jsonCond
    : jsonCond OR_ jsonCond | jsonCond AND_ jsonCond | jsonNegation
    | LP_ jsonCond RP_ | jsonComparison | jsonExistsCond
    | jsonInCond | jsonLikeCond | jsonLikeRegexCond
    | jsonEqRegexCond | jsonHasSubstringCond | jsonStartsWithCond
    ;

jsonDisjunction
    : jsonCond OR_ jsonCond
    ;

jsonConjunction
    : jsonCond AND_ jsonCond
    ;

jsonNegation
    : NOT_ LP_ jsonCond RP_
    ;

jsonExistsCond
    : EXISTS LP_ jsonRelativePathExpr RP_
    ;

jsonHasSubstringCond
    : jsonRelativePathExpr HAS SUBSTRING (jsonString | jsonVar)
    ;

jsonStartsWithCond
    : jsonRelativePathExpr STARTS WITH (jsonString | jsonVar)
    ;

jsonLikeCond
    : jsonRelativePathExpr LIKE (jsonString | jsonVar)
    ;

jsonLikeRegexCond
    : jsonRelativePathExpr LIKE_REGEX (jsonString | jsonVar)
    ;

jsonEqRegexCond
    : jsonRelativePathExpr EQ_REGEX (jsonString | jsonVar)
    ;

jsonInCond
    : jsonRelativePathExpr IN valueList
    ;

valueList
    : LP_ (jsonScalar | jsonVar) (COMMA_ (jsonScalar | jsonVar))* RP_
    ;

jsonComparison
    : (jsonRelativePathExpr jsonComparePred (jsonVar | jsonScalar))
    | ((jsonVar | jsonScalar) jsonComparePred jsonRelativePathExpr)
    | (jsonScalar jsonComparePred jsonScalar)
    ;

jsonRelativePathExpr
    : AT_ jsonNonfunctionSteps? jsonFunctionStep?
    ;

jsonComparePred
    : DEQ_ | NEQ_ | LT_ | LTE_ | GTE_ | GT_
    ;

jsonVar
    : DOLLAR_ identifier
    ;

jsonScalar
    : jsonNumber | TRUE | FALSE | NULL | jsonString
    ;

jsonNumber
    : numberLiterals
    ;

jsonString
    : stringLiterals | identifier
    ;

compoundCondition
    : LP_ condition RP_
    | NOT condition
    | condition (AND | OR) condition
    ;

existsCondition
    : EXISTS LP_ subquery RP_
    ;

inCondition
    : (expr NOT? IN LP_ (expressionList | subquery) RP_)
    | (exprList NOT? IN LP_ ((expressionList (COMMA_ expressionList)*) | subquery) RP_)
    ;

isOfTypeCondition
    : expr IS NOT? OF TYPE? LP_ ONLY? typeName (COMMA_ ONLY? typeName)* RP_
    ;

databaseCharset
    : AL32UTF8
    ;

nationalCharset
    : AL16UTF16 | UTF8
    ;

filenamePattern
    : STRING_
    ;

replacementFilenamePattern
    : STRING_
    ;

connectString
    : STRING_
    ;

argument
    : identifier
    ;

dataSource
    : identifier
    ;

implementationType
    : (owner DOT_)? name
    ;

implementationPackage
    : (owner DOT_)? name
    ;

label
    : identifier
    ;

libName
    : identifier
    ;

externalDatatype
    : dataType
    ;

capacityUnit
    : SINGLE_K | SINGLE_M | SINGLE_G | SINGLE_T | SINGLE_P | SINGLE_E
    ;

timeUnit
    : SINGLE_M | SINGLE_H
    ;

attributeDimensionName
    : identifier
    ;

sequenceName
    : identifier
    ;

spfileName
    : STRING_
    ;

pfileName
    : STRING_
    ;

characterSetName
    : identifier
    ;

quotaUnit
    : SINGLE_M | SINGLE_G | SINGLE_T | SINGLE_P | SINGLE_E
    ;

siteName
    : identifier
    ;

diskName
    : identifier
    ;

searchString
    : STRING_
    ;

attributeValue
    : STRING_
    ;

joinGroupName
    : identifier
    ;

restorePointName
    : identifier
    ;

libraryName
    : identifier
    ;

matchString
    : identifier | ASTERISK_
    ;

parameterType
    : (owner DOT_)? identifier
    ;

returnType
    : identifier
    ;

failgroupName
    : identifier
    ;

asmVolumeName
    : identifier
    ;

mountpathName
    : identifier
    ;

usageName
    : identifier
    ;

usergroupName
    : STRING_
    ;

varrayType
    : (owner DOT_)? name
    ;

stagingLogName
    : identifier
    ;

featureName
   : STRING_
   ;

optionName
    : STRING_
    ;

clauseOption
    : STRING_
    ;

clauseOptionPattern
    : STRING_
    ;

optionValue
    : STRING_
    ;

clause
    : STRING_
    ;

sqlStatement
    : STRING_
    ;


transportSecret
    : STRING_
    ;

hostName
    : STRING_
    ;

mapObject
    : STRING_
    ;

refreshInterval
    : INTEGER_
    ;

sourcePdbName
    : STRING_
    ;

appName
    : STRING_
    ;

commentValue
    : STRING_
    ;

appVersion
    : NUMBER_
    ;

startAppVersion
    : NUMBER_
    ;

endAppVersion
    : NUMBER_
    ;

patchNumber
    : INTEGER_
    ;

snapshotInterval
    : INTEGER_
    ;

snapshotName
    : STRING_
    ;

maxPdbSnapshots
    : INTEGER_
    ;

maxNumberOfSnapshots
    : INTEGER_
    ;

datetimeExpr
    : AT (LOCAL | TIME ZONE expr)
    ;

xmlFunction
    : xmlElementFunction
    | xmlCdataFunction
    | xmlAggFunction
    | xmlColattvalFunction
    | xmlExistsFunction
    | xmlForestFunction
    | xmlParseFunction
    | xmlPiFunction
    | xmlQueryFunction
    | xmlRootFunction
    | xmlSerializeFunction
    | xmlTableFunction
    | xmlIsSchemaValidFunction
    | specifiedFunctionName = (SYS_XMLGEN | SYS_XMLAGG | APPENDCHILDXML | DELETEXML | EXISTSNODE | EXTRACT | EXTRACTVALUE
        | INSERTCHILDXML | INSERTCHILDXMLAFTER | INSERTCHILDXMLBEFORE | INSERTXMLAFTER | INSERTXMLBEFORE
        | SYS_DBURIGEN | UPDATEXML | XMLCONCAT | XMLDIFF | XMLEXISTS | XMLISVALID | XMLPATCH | XMLSEQUENCE | XMLTRANSFORM) exprList
    | specifiedFunctionName = (DEPTH | PATH) LP_ correlationInteger RP_
    | specifiedFunctionName = XMLCOMMENT LP_ stringLiterals RP_
    ;

xmlElementFunction
    : XMLELEMENT LP_ identifier (COMMA_ xmlAttributes)? (COMMA_ exprWithAlias)* RP_
    ;

exprWithAlias
    : expr (AS alias)?
    ;

xmlAttributes
    : XMLATTRIBUTES LP_ exprWithAlias (COMMA_ exprWithAlias)* RP_
    ;

xmlCdataFunction
    : XMLCDATA LP_ stringLiterals RP_
    ;

xmlAggFunction
    : XMLAGG LP_ expr orderByClause? RP_
    ;

xmlColattvalFunction
    : XMLCOLATTVAL LP_ expr (xmlAsAliasOrEvalnameExpr)? (COMMA_ expr (xmlAsAliasOrEvalnameExpr)?)* RP_
    ;

xmlAsAliasOrEvalnameExpr
    :AS (alias | EVALNAME expr)
    ;

xmlExistsFunction
    : XMLEXISTS LP_ STRING_ xmlPassingClause? RP_
    ;

xmlForestFunction
   : XMLFOREST LP_ expr (xmlAsAliasOrEvalnameExpr)? (COMMA_ expr (xmlAsAliasOrEvalnameExpr)?)* RP_
   ;

xmlParseFunction
    : XMLPARSE LP_ (DOCUMENT | CONTENT) expr (WELLFORMED)? RP_
    ;

xmlPiFunction
    : XMLPI LP_ (EVALNAME expr | (NAME)? identifier) (COMMA_ expr)? RP_
    ;

xmlQueryFunction
    : XMLQUERY LP_ STRING_ xmlPassingClause? RETURNING CONTENT (NULL ON EMPTY)? RP_
    ;

xmlPassingClause
    : PASSING (BY VALUE)? expr (AS alias)? (COMMA_ expr (AS alias)?)*
    ;

xmlRootFunction
    : XMLROOT LP_ expr COMMA_ VERSION (expr | NO VALUE) (COMMA_ STANDALONE (YES | NO | NO VALUE))? RP_
    ;

xmlSerializeFunction
    : XMLSERIALIZE LP_ (DOCUMENT | CONTENT) expr (AS dataType)? (ENCODING STRING_)? (VERSION stringLiterals)? (NO INDENT | INDENT (SIZE EQ_ INTEGER_)?)? ((HIDE | SHOW) DEFAULTS)? RP_
    ;

xmlTableFunction
    : XMLTABLE LP_ (xmlNamespacesClause COMMA_)? STRING_ xmlTableOptions RP_
    ;

xmlIsSchemaValidFunction
    : (owner DOT_)* name DOT_ ISSCHEMAVALID LP_ expr (COMMA_ expr)* RP_
    ;

xmlNamespacesClause
    : XMLNAMESPACES LP_ (defaultString COMMA_)? (xmlNamespaceStringAsIdentifier | defaultString) (COMMA_ (xmlNamespaceStringAsIdentifier | defaultString))* RP_
    ;

xmlNamespaceStringAsIdentifier
    : STRING_ AS identifier
    ;

defaultString
    : DEFAULT STRING_
    ;

xmlTableOptions
    : xmlPassingClause? (RETURNING SEQUENCE BY REF)? (COLUMNS xmlTableColumn (COMMA_ xmlTableColumn)*)?
    ;

xmlTableColumn
    : columnName (FOR ORDINALITY | (dataType | XMLTYPE (LP_ SEQUENCE RP_ BY REF)?) (PATH STRING_)? (DEFAULT expr)?)
    ;

multisetExpr
    : columnName MULTISET multisetOperator (ALL | DISTINCT)? columnName
    ;

multisetOperator
    : EXCEPT
    | INTERSECT
    | UNION
    ;

builtinFunctionsExpr
    : (packageIdentifier DOT_ builtinFunction) AT_ dbLink LP_ expr RP_
    ;

packageIdentifier
    : identifier
    ;

builtinFunction
   : GETLENGTH
   ;

superview
    : identifier
    ;

primaryName
    : identifier
    ;

directoryObjectName
    : identifier
    ;

serverFileName
    : identifier
    ;

keyForBlob
    : identifier
    ;

sourceText
    : .*? RBE_
    ;

fullPathName
    : STRING_
    ;

directoryObject
    : identifier
    ;

credentialName
    : identifier
    ;

agentDblink
    : STRING_
    ;

xPathsList
    : STRING_
    ;

sizeClause
    : INTEGER_ capacityUnit?
    ;

maxsizeClause
    : MAXSIZE (UNLIMITED | sizeClause)
    ;

editionType
    : VIEW | SYNONYM | PROCEDURE | FUNCTION | PACKAGE | PACKAGE BODY | TRIGGER | TYPE | TYPE BODY | LIBRARY
    ;

containerDataClause
    : (SET CONTAINER_DATA EQ_ (ALL | DEFAULT | containerName (COMMA_ containerName)*)
    | (ADD | REMOVE) CONTAINER_DATA containerName (COMMA_ containerName)*)
    (FOR (schemaName DOT_)? name)?
    ;

proxyClause
    : GRANT CONNECT THROUGH (ENTERPRISE USERS | dbUserProxy dbUserProxyClauses)
    | REVOKE CONNECT THROUGH (ENTERPRISE USERS | username)
    ;

dbUserProxy
    : identifier
    ;

dbUserProxyClauses
    : (WITH ((ROLE (ALL EXCEPT)? roleName (COMMA_ roleName)*) | NO ROLES))? (AUTHENTICATION REQUIRED | AUTHENTICATED USING PASSWORD)?
    ;

delimSpec
     : terminatedBySpec? optionallyEnclosedBySpec? lrtrimSpec?
     ;

terminatedBySpec
     : TERMINATED BY (STRING_ |WHITESPACE)
     ;

optionallyEnclosedBySpec
     : OPTIONALLY? ENCLOSED BY STRING_ AND STRING_
     ;

lrtrimSpec
     : LRTRIM
     ;
