/*
 * Decompiled with CFR 0.152.
 */
package org.apache.tinkerpop.gremlin.driver;

import java.net.ConnectException;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Random;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.CompletionException;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;
import java.util.concurrent.atomic.AtomicReference;
import java.util.function.Consumer;
import java.util.stream.Collectors;
import javax.net.ssl.SSLException;
import org.apache.commons.lang3.exception.ExceptionUtils;
import org.apache.tinkerpop.gremlin.driver.Cluster;
import org.apache.tinkerpop.gremlin.driver.Connection;
import org.apache.tinkerpop.gremlin.driver.ConnectionPool;
import org.apache.tinkerpop.gremlin.driver.Host;
import org.apache.tinkerpop.gremlin.driver.RequestOptions;
import org.apache.tinkerpop.gremlin.driver.ResultSet;
import org.apache.tinkerpop.gremlin.driver.exception.ConnectionException;
import org.apache.tinkerpop.gremlin.driver.exception.NoHostAvailableException;
import org.apache.tinkerpop.gremlin.util.message.RequestMessage;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public abstract class Client {
    private static final Logger logger = LoggerFactory.getLogger(Client.class);
    public static final String TOO_MANY_IN_FLIGHT_REQUESTS = "Number of active requests (%s) exceeds pool size (%s). Consider increasing the value for maxConnectionPoolSize.";
    protected final Cluster cluster;
    protected volatile boolean initialized;
    private static final Random random = new Random();

    Client(Cluster cluster) {
        this.cluster = cluster;
    }

    public RequestMessage.Builder buildMessage(RequestMessage.Builder builder) {
        return builder;
    }

    protected abstract void initializeImplementation();

    protected abstract Connection chooseConnection(RequestMessage var1) throws TimeoutException, ConnectionException;

    public abstract CompletableFuture<Void> closeAsync();

    public Client alias(String graphOrTraversalSource) {
        return new AliasClusteredClient(this, graphOrTraversalSource);
    }

    public synchronized Client init() {
        if (this.initialized) {
            return this;
        }
        logger.debug("Initializing client on cluster [{}]", (Object)this.cluster);
        this.cluster.init();
        this.initializeImplementation();
        this.initialized = true;
        return this;
    }

    public ResultSet submit(String gremlin) {
        return this.submit(gremlin, RequestOptions.EMPTY);
    }

    public ResultSet submit(String gremlin, Map<String, Object> parameters) {
        try {
            return this.submitAsync(gremlin, parameters).get();
        }
        catch (RuntimeException re) {
            throw re;
        }
        catch (Exception ex) {
            throw new RuntimeException(ex);
        }
    }

    public ResultSet submit(String gremlin, RequestOptions options) {
        try {
            return this.submitAsync(gremlin, options).get();
        }
        catch (RuntimeException re) {
            throw re;
        }
        catch (Exception ex) {
            throw new RuntimeException(ex);
        }
    }

    public CompletableFuture<ResultSet> submitAsync(String gremlin) {
        return this.submitAsync(gremlin, RequestOptions.build().create());
    }

    public CompletableFuture<ResultSet> submitAsync(String gremlin, Map<String, Object> parameters) {
        RequestOptions.Builder options = RequestOptions.build();
        if (parameters != null && !parameters.isEmpty()) {
            parameters.forEach(options::addParameter);
        }
        return this.submitAsync(gremlin, options.create());
    }

    @Deprecated
    public CompletableFuture<ResultSet> submitAsync(String gremlin, String graphOrTraversalSource, Map<String, Object> parameters) {
        RequestOptions.Builder options = RequestOptions.build();
        options.addG(graphOrTraversalSource);
        if (parameters != null && !parameters.isEmpty()) {
            parameters.forEach(options::addParameter);
        }
        options.batchSize(this.cluster.connectionPoolSettings().resultIterationBatchSize);
        return this.submitAsync(gremlin, options.create());
    }

    public CompletableFuture<ResultSet> submitAsync(String gremlin, RequestOptions options) {
        int batchSize = options.getBatchSize().orElse(this.cluster.connectionPoolSettings().resultIterationBatchSize);
        RequestMessage.Builder request = this.buildMessage(RequestMessage.build((String)gremlin)).addChunkSize(batchSize);
        options.getTimeout().ifPresent(timeout -> request.addTimeoutMillis(timeout.longValue()));
        options.getParameters().ifPresent(params -> request.addBindings(params));
        options.getG().ifPresent(g -> request.addG(g));
        options.getLanguage().ifPresent(lang -> request.addLanguage(lang));
        options.getMaterializeProperties().ifPresent(mp -> request.addMaterializeProperties(mp));
        options.getBulkResults().ifPresent(bulked -> request.addBulkResults(Boolean.parseBoolean(bulked)));
        return this.submitAsync(request.create());
    }

    /*
     * Loose catch block
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    public CompletableFuture<ResultSet> submitAsync(RequestMessage msg) {
        CompletableFuture<ResultSet> completableFuture;
        if (this.isClosing()) {
            throw new IllegalStateException("Client is closed");
        }
        if (!this.initialized) {
            this.init();
        }
        CompletableFuture<ResultSet> future = new CompletableFuture<ResultSet>();
        Connection connection = null;
        try {
            connection = this.chooseConnection(msg);
            connection.write(msg, future);
            completableFuture = future;
        }
        catch (RuntimeException re) {
            try {
                throw re;
                catch (Exception ex) {
                    throw new RuntimeException(ex);
                }
            }
            catch (Throwable throwable) {
                if (logger.isDebugEnabled()) {
                    logger.debug("Submitted {} to - {}", (Object)msg, (Object)(null == connection ? "connection not initialized" : connection.toString()));
                }
                throw throwable;
            }
        }
        if (logger.isDebugEnabled()) {
            logger.debug("Submitted {} to - {}", (Object)msg, (Object)(null == connection ? "connection not initialized" : connection.toString()));
        }
        return completableFuture;
    }

    public abstract boolean isClosing();

    public void close() {
        this.closeAsync().join();
    }

    public Cluster getCluster() {
        return this.cluster;
    }

    public static class AliasClusteredClient
    extends Client {
        private final Client client;
        private final String graphOrTraversalSource;
        final CompletableFuture<Void> close = new CompletableFuture();

        AliasClusteredClient(Client client, String graphOrTraversalSource) {
            super(client.cluster);
            this.client = client;
            this.graphOrTraversalSource = graphOrTraversalSource;
        }

        @Override
        public CompletableFuture<ResultSet> submitAsync(RequestMessage msg) {
            RequestMessage.Builder builder = RequestMessage.from((RequestMessage)msg);
            builder.addG(this.graphOrTraversalSource);
            return super.submitAsync(builder.create());
        }

        @Override
        public synchronized Client init() {
            if (this.close.isDone()) {
                throw new IllegalStateException("Client is closed");
            }
            this.client.init();
            return this;
        }

        @Override
        public RequestMessage.Builder buildMessage(RequestMessage.Builder builder) {
            if (this.close.isDone()) {
                throw new IllegalStateException("Client is closed");
            }
            builder.addG(this.graphOrTraversalSource);
            return this.client.buildMessage(builder);
        }

        @Override
        protected void initializeImplementation() {
            if (this.close.isDone()) {
                throw new IllegalStateException("Client is closed");
            }
            if (this.cluster.availableHosts().isEmpty()) {
                throw new NoHostAvailableException();
            }
        }

        @Override
        protected Connection chooseConnection(RequestMessage msg) throws TimeoutException, ConnectionException {
            if (this.close.isDone()) {
                throw new IllegalStateException("Client is closed");
            }
            return this.client.chooseConnection(msg);
        }

        @Override
        public void close() {
            this.client.close();
        }

        @Override
        public synchronized CompletableFuture<Void> closeAsync() {
            return this.client.closeAsync();
        }

        @Override
        public boolean isClosing() {
            return this.client.isClosing();
        }

        @Override
        public Client alias(String graphOrTraversalSource) {
            if (this.close.isDone()) {
                throw new IllegalStateException("Client is closed");
            }
            return new AliasClusteredClient(this.client, graphOrTraversalSource);
        }
    }

    public static final class ClusteredClient
    extends Client {
        ConcurrentMap<Host, ConnectionPool> hostConnectionPools = new ConcurrentHashMap<Host, ConnectionPool>();
        private final AtomicReference<CompletableFuture<Void>> closing = new AtomicReference<Object>(null);
        private Throwable initializationFailure = null;
        private final Consumer<Host> initializeConnectionSetupForHost = host -> {
            try {
                this.hostConnectionPools.put((Host)host, new ConnectionPool((Host)host, this));
                host.makeAvailable();
                this.cluster.loadBalancingStrategy().onNew((Host)host);
            }
            catch (RuntimeException ex) {
                String errMsg = "Could not initialize client for " + host;
                logger.error(errMsg);
                throw ex;
            }
        };

        ClusteredClient(Cluster cluster) {
            super(cluster);
        }

        @Override
        public boolean isClosing() {
            return this.closing.get() != null;
        }

        public ResultSet submit(String gremlin, String graphOrTraversalSource) {
            return this.submit(gremlin, graphOrTraversalSource, null);
        }

        public ResultSet submit(String gremlin, String graphOrTraversalSource, Map<String, Object> parameters) {
            try {
                return this.submitAsync(gremlin, graphOrTraversalSource, parameters).get();
            }
            catch (RuntimeException re) {
                throw re;
            }
            catch (Exception ex) {
                throw new RuntimeException(ex);
            }
        }

        @Override
        public Client alias(String graphOrTraversalSource) {
            return new AliasClusteredClient(this, graphOrTraversalSource);
        }

        @Override
        protected Connection chooseConnection(RequestMessage msg) throws TimeoutException, ConnectionException {
            Iterator<Host> possibleHosts = this.cluster.loadBalancingStrategy().select(msg);
            Host bestHost = possibleHosts.hasNext() ? possibleHosts.next() : this.chooseRandomHost();
            ConnectionPool pool = (ConnectionPool)this.hostConnectionPools.get(bestHost);
            return pool.borrowConnection(this.cluster.connectionPoolSettings().maxWaitForConnection, TimeUnit.MILLISECONDS);
        }

        private Host chooseRandomHost() {
            ArrayList<Host> hosts = new ArrayList<Host>(this.cluster.allHosts());
            int ix = random.nextInt(hosts.size());
            return (Host)hosts.get(ix);
        }

        @Override
        protected void initializeImplementation() {
            List<Host> unavailableHosts;
            try {
                CompletableFuture.allOf((CompletableFuture[])this.cluster.allHosts().stream().map(host -> CompletableFuture.runAsync(() -> this.initializeConnectionSetupForHost.accept((Host)host), this.cluster.hostScheduler())).toArray(CompletableFuture[]::new)).join();
            }
            catch (CompletionException ex) {
                logger.error("Initialization failed", (Throwable)ex);
                this.initializationFailure = ex;
            }
            if (this.cluster.availableHosts().isEmpty()) {
                this.throwNoHostAvailableException();
            }
            if (!(unavailableHosts = this.cluster.allHosts().stream().filter(host -> !host.isAvailable()).collect(Collectors.toList())).isEmpty()) {
                this.handleUnavailableHosts(unavailableHosts);
            }
        }

        private void throwNoHostAvailableException() {
            Throwable rootCause = ExceptionUtils.getRootCause((Throwable)this.initializationFailure);
            if (rootCause instanceof SSLException || rootCause instanceof ConnectException) {
                throw new NoHostAvailableException(this.initializationFailure);
            }
            throw new NoHostAvailableException();
        }

        @Override
        public synchronized CompletableFuture<Void> closeAsync() {
            if (this.closing.get() != null) {
                return this.closing.get();
            }
            CompletableFuture<Void> allPoolsClosedFuture = CompletableFuture.allOf((CompletableFuture[])this.hostConnectionPools.values().stream().map(ConnectionPool::closeAsync).toArray(CompletableFuture[]::new));
            this.closing.set(allPoolsClosedFuture);
            return this.closing.get();
        }

        private void handleUnavailableHosts(List<Host> unavailableHosts) {
            for (Host host : unavailableHosts) {
                CompletableFuture<Void> f = CompletableFuture.runAsync(() -> host.makeUnavailable(this::tryReInitializeHost), this.cluster.hostScheduler());
                f.exceptionally(t -> {
                    logger.error("", t.getCause() == null ? t : t.getCause());
                    return null;
                });
            }
        }

        public boolean tryReInitializeHost(Host host) {
            logger.debug("Trying to re-initiate host connection pool on {}", (Object)host);
            try {
                this.initializeConnectionSetupForHost.accept(host);
                return true;
            }
            catch (Exception ex) {
                logger.debug("Failed re-initialization attempt on {}", (Object)host, (Object)ex);
                return false;
            }
        }
    }
}

