/* ------- file strlcpy.c ------- */
/* Implementation of strlcpy and strlcat
   See http://www.courtesan.com/todd/papers/strlcpy.html

   These routines are explicitly designed to move no
   data and simply return the projected size of the
   final strings when called with sz == 0.

   In addition they treat src == NULL as an empty string

   strlcat expects that the sz parameter is greater than
   the size of the dst string.  If not, the treatment is
   exactly the same as if a sz value of 0 was used.
*/

#include "libsupp.h"

/* NOTE: these routines are deliberately designed to
   not require any assistance from the standard
   libraries.  This makes them more useful in any
   embedded systems that must minimize the load size.

   Public domain, by C.B. Falconer
   bug reports to mailto:cbfalconer@worldnet.att.net
*/

/* NOTE: Modified to fit libsupp requirements by Juan M. Guerrero,  2008-09-17.
   <juan.guerrero@gmx.de>
*/


/* ---------------------- */

size_t libsupp_strlcpy(char *dst, const char *src, size_t sz)
{
   const char *start = src;

   if (src && sz--) {
      while ((*dst++ = *src))
         if (sz--) src++;
         else {
            *(--dst) = '\0';
            break;
         }
   }
   if (src) {
      while (*src++) continue;
      return src - start - 1;
   }
   else if (sz) *dst = '\0';
   return 0;
} /* strlcpy */

/* ---------------------- */

size_t libsupp_strlcat(char *dst, const char *src, size_t sz)
{
   char  *start = dst;

   while (*dst++)    /* assumes sz >= strlen(dst) */
      if (sz) sz--;    /* i.e. well formed string */
   dst--;
   return dst - start + libsupp_strlcpy(dst, src, sz);
} /* strlcat */

/* ---------------------- */

#ifdef TESTING
#include <stdio.h>

typedef size_t (*op)(char *, const char *, size_t);
#define FLD 16

/* ---------------------- */

static void putleftinfld(const char *str, int field, int quote)
{
   int                used;
   static const char *nullstr = "(NULL)";

   if (!str) str = nullstr;
   if (quote) used = printf("\"%s\"", str);
   else       used = printf("%s", str);
   while (used++ < field) putchar(' ');
} /* putleftinfld */

/* ---------------------- */

static void dotest(op fn, char *s1, char *s2, size_t sz)
{
   unsigned long lgh;

   putleftinfld(s1, FLD, 1); putleftinfld(s2, FLD, 1);
   if      (fn == libsupp_strlcat) printf(" cat ");
   else if (fn == libsupp_strlcpy) printf(" cpy ");
   else                        printf(" ??? ");
   lgh = fn(s1, s2, sz);
   printf("%3lu %3lu \"%s\"\n", (unsigned long)sz, lgh, s1);
} /* dotest */

/* ---------------------- */

int main(void)
{
   char         *str1     = "string1";
   char          str2[10] = "";
   char          str3[5]  = "";
   char          str4[]   = "";

   puts("Testing lgh = stringop(dest, source, sz)\n");
   putleftinfld(" dest", FLD, 0);
   putleftinfld(" source", FLD, 0);
   puts(" opn  sz lgh  result");
   putleftinfld(" ====", FLD, 0);
   putleftinfld(" ======", FLD, 0);
   puts(" ===  == ===  ======");

   dotest(libsupp_strlcpy, str2, str1, sizeof str2);
   dotest(libsupp_strlcpy, str3, str1, sizeof str3);
   dotest(libsupp_strlcpy, str4, str1, sizeof str4);
   dotest(libsupp_strlcat, str2, str1, sizeof str2);
   dotest(libsupp_strlcpy, str2, "x ", sizeof str2);
   dotest(libsupp_strlcat, str2, str1, sizeof str2);
   dotest(libsupp_strlcpy, str2, "x ", sizeof str2);
   dotest(libsupp_strlcat, str2, str1, 0);
   dotest(libsupp_strlcpy, str2, str1, 0);
   dotest(libsupp_strlcat, str2, "longer string", 0);
   dotest(libsupp_strlcpy, str2, NULL, sizeof str2);
   dotest(libsupp_strlcpy, str2, "x ", sizeof str2);
   dotest(libsupp_strlcat, str2, NULL, sizeof str2);

   return 0;
} /* main */
#endif
/* ------- End file cbf_strlcpy.c ------- */
