/* Function tanh vectorized with AVX2.
   Copyright (C) 2021-2025 Free Software Foundation, Inc.
   This file is part of the GNU C Library.

   The GNU C Library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Lesser General Public
   License as published by the Free Software Foundation; either
   version 2.1 of the License, or (at your option) any later version.

   The GNU C Library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Lesser General Public License for more details.

   You should have received a copy of the GNU Lesser General Public
   License along with the GNU C Library; if not, see
   https://www.gnu.org/licenses/.  */

/*
 * ALGORITHM DESCRIPTION:
 *
 *   NOTE: Since the hyperbolic tangent function is odd
 *         (tanh(x) = -tanh(-x)), below algorithm deals with the absolute
 *         value of the argument |x|: tanh(x) = sign(x) * tanh(|x|)
 *
 *   We use a table lookup method to compute tanh(|x|).
 *   The basic idea is to split the input range into a number of subintervals
 *   and to approximate tanh(.) with a polynomial on each of them.
 *
 *   IEEE SPECIAL CONDITIONS:
 *   x = [+, -]0, r = [+, -]0
 *   x = +Inf,   r = +1
 *   x = -Inf,   r = -1
 *   x = QNaN,   r = QNaN
 *   x = SNaN,   r = QNaN
 *
 *
 *   ALGORITHM DETAILS
 *   We handle special values in a callout function, aside from main path
 *   computations. "Special" for this algorithm are:
 *   INF, NAN, |x| > HUGE_THRESHOLD
 *
 *
 *   Main path computations are organized as follows:
 *   Actually we split the interval [0, SATURATION_THRESHOLD)
 *   into a number of subintervals.  On each subinterval we approximate tanh(.)
 *   with a minimax polynomial of pre-defined degree. Polynomial coefficients
 *   are computed beforehand and stored in table. We also use
 *
 *       y := |x| + B,
 *
 *   here B depends on subinterval and is used to make argument
 *   closer to zero.
 *   We also add large fake interval [SATURATION_THRESHOLD, HUGE_THRESHOLD],
 *   where 1.0 + 0.0*y + 0.0*y^2 ... coefficients are stored - just to
 *   preserve main path computation logic but return 1.0 for all arguments.
 *
 *   Hence reconstruction looks as follows:
 *   we extract proper polynomial and range reduction coefficients
 *        (Pj and B), corresponding to subinterval, to which |x| belongs,
 *        and return
 *
 *       r := sign(x) * (P0 + P1 * y + ... + Pn * y^n)
 *
 *   NOTE: we use multiprecision technique to multiply and sum the first
 *         K terms of the polynomial. So Pj, j = 0..K are stored in
 *         table each as a pair of target precision numbers (Pj and PLj) to
 *         achieve wider than target precision.
 *
 *
 */

/* Offsets for data table __svml_dtanh_data_internal
 */
#define _dbP				0
#define _dbSignMask			7680
#define _dbAbsMask			7712
#define _iExpMantMask			7744
#define _iExpMask			7776
#define _iMinIdxOfsMask			7808
#define _iMaxIdxMask			7840

#include <sysdep.h>

	.section .text.avx2, "ax", @progbits
ENTRY(_ZGVdN4v_tanh_avx2)
	pushq	%rbp
	cfi_def_cfa_offset(16)
	movq	%rsp, %rbp
	cfi_def_cfa(6, 16)
	cfi_offset(6, -16)
	andq	$-32, %rsp
	subq	$96, %rsp
	lea	_dbP+96+__svml_dtanh_data_internal(%rip), %r8
	vmovupd	%ymm0, (%rsp)

	/* if VMIN, VMAX is defined for I type */
	vpxor	%xmm11, %xmm11, %xmm11

	/*  Constant loading  */
	vmovups	_iMaxIdxMask+__svml_dtanh_data_internal(%rip), %xmm8
	vandpd	_dbAbsMask+__svml_dtanh_data_internal(%rip), %ymm0, %ymm1
	vandpd	_dbSignMask+__svml_dtanh_data_internal(%rip), %ymm0, %ymm2
	vextractf128 $1, %ymm0, %xmm15
	vshufps	$221, %xmm15, %xmm0, %xmm14

	/* Here huge arguments, INF and NaNs are filtered out to callout. */
	vpand	_iExpMantMask+__svml_dtanh_data_internal(%rip), %xmm14, %xmm12
	vpsubd	_iMinIdxOfsMask+__svml_dtanh_data_internal(%rip), %xmm12, %xmm9
	vpcmpgtd %xmm11, %xmm9, %xmm10
	vpcmpgtd %xmm8, %xmm9, %xmm0
	vpand	%xmm10, %xmm9, %xmm7
	vblendvps %xmm0, %xmm8, %xmm7, %xmm7

	/*
	 * VSHRIMM( I, iIndex, = iIndex, (17 - 4) );
	 * VGATHER_MATRIX( L2D, p, TAB._dbP, iIndex, 0, T_ITEM_SIZE, T_ITEM_GRAN, 13, 0, 0 );
	 */
	vpsrld	$10, %xmm7, %xmm6
	vmovd	%xmm6, %edx
	vpcmpgtd _iExpMask+__svml_dtanh_data_internal(%rip), %xmm12, %xmm13
	vmovmskps %xmm13, %eax
	vpextrd	$1, %xmm6, %ecx
	movslq	%edx, %rdx
	movslq	%ecx, %rcx
	vpextrd	$2, %xmm6, %esi
	vpextrd	$3, %xmm6, %edi
	movslq	%esi, %rsi
	movslq	%edi, %rdi
	vmovupd	-96(%rdx, %r8), %xmm3
	vmovupd	-96(%rcx, %r8), %xmm4
	vmovupd	-80(%rcx, %r8), %xmm13
	vmovupd	-64(%rcx, %r8), %xmm9
	vmovupd	-80(%rdx, %r8), %xmm14
	vmovupd	-64(%rdx, %r8), %xmm10
	vmovupd	-48(%rdx, %r8), %xmm6
	vinsertf128 $1, -96(%rsi, %r8), %ymm3, %ymm0
	vinsertf128 $1, -96(%rdi, %r8), %ymm4, %ymm15
	vmovupd	-48(%rcx, %r8), %xmm3
	vunpckhpd %ymm15, %ymm0, %ymm0
	vinsertf128 $1, -80(%rsi, %r8), %ymm14, %ymm12
	vinsertf128 $1, -64(%rsi, %r8), %ymm10, %ymm8
	vinsertf128 $1, -80(%rdi, %r8), %ymm13, %ymm11
	vinsertf128 $1, -64(%rdi, %r8), %ymm9, %ymm7
	vunpcklpd %ymm11, %ymm12, %ymm15
	vunpckhpd %ymm11, %ymm12, %ymm14
	vunpcklpd %ymm7, %ymm8, %ymm13
	vunpckhpd %ymm7, %ymm8, %ymm12
	vmovupd	-32(%rdx, %r8), %xmm9
	vmovupd	-32(%rcx, %r8), %xmm8
	vinsertf128 $1, -48(%rsi, %r8), %ymm6, %ymm4
	vinsertf128 $1, -48(%rdi, %r8), %ymm3, %ymm5
	vunpcklpd %ymm5, %ymm4, %ymm11
	vunpckhpd %ymm5, %ymm4, %ymm10
	vmovupd	-16(%rdx, %r8), %xmm3
	vmovupd	-16(%rcx, %r8), %xmm4
	vinsertf128 $1, -32(%rsi, %r8), %ymm9, %ymm7
	vinsertf128 $1, -32(%rdi, %r8), %ymm8, %ymm6
	vunpcklpd %ymm6, %ymm7, %ymm9
	vunpckhpd %ymm6, %ymm7, %ymm8
	vinsertf128 $1, -16(%rsi, %r8), %ymm3, %ymm5
	vinsertf128 $1, -16(%rdi, %r8), %ymm4, %ymm6
	vunpcklpd %ymm6, %ymm5, %ymm7
	vunpckhpd %ymm6, %ymm5, %ymm6
	vmovupd	(%rdx, %r8), %xmm3
	vmovupd	(%rcx, %r8), %xmm5
	vinsertf128 $1, (%rsi, %r8), %ymm3, %ymm4
	vinsertf128 $1, (%rdi, %r8), %ymm5, %ymm5
	vunpcklpd %ymm5, %ymm4, %ymm3
	vaddpd	%ymm3, %ymm1, %ymm1
	vfmadd213pd %ymm7, %ymm1, %ymm6
	vfmadd213pd %ymm8, %ymm1, %ymm6
	vfmadd213pd %ymm9, %ymm1, %ymm6
	vfmadd213pd %ymm10, %ymm1, %ymm6
	vfmadd213pd %ymm11, %ymm1, %ymm6
	vfmadd213pd %ymm12, %ymm1, %ymm6
	vfmadd213pd %ymm13, %ymm1, %ymm6
	vfmadd213pd %ymm14, %ymm1, %ymm6
	vfmadd213pd %ymm15, %ymm1, %ymm6
	vfmadd213pd %ymm0, %ymm1, %ymm6
	vorpd	%ymm2, %ymm6, %ymm0
	testl	%eax, %eax

	/* Go to special inputs processing branch */
	jne	L(SPECIAL_VALUES_BRANCH)
	# LOE rbx r12 r13 r14 r15 eax ymm0

	/* Restore registers
	 * and exit the function
	 */

L(EXIT):
	movq	%rbp, %rsp
	popq	%rbp
	cfi_def_cfa(7, 8)
	cfi_restore(6)
	ret
	cfi_def_cfa(6, 16)
	cfi_offset(6, -16)

	/* Branch to process
	 * special inputs
	 */

L(SPECIAL_VALUES_BRANCH):
	vmovupd	(%rsp), %ymm1
	vmovupd	%ymm0, 64(%rsp)
	vmovupd	%ymm1, 32(%rsp)
	# LOE rbx r12 r13 r14 r15 eax ymm0

	xorl	%edx, %edx
	# LOE rbx r12 r13 r14 r15 eax edx

	vzeroupper
	movq	%r12, 16(%rsp)
	/*  DW_CFA_expression: r12 (r12) (DW_OP_lit8; DW_OP_minus; DW_OP_const4s: -32; DW_OP_and; DW_OP_const4s: -80; DW_OP_plus)  */
	.cfi_escape 0x10, 0x0c, 0x0e, 0x38, 0x1c, 0x0d, 0xe0, 0xff, 0xff, 0xff, 0x1a, 0x0d, 0xb0, 0xff, 0xff, 0xff, 0x22
	movl	%edx, %r12d
	movq	%r13, 8(%rsp)
	/*  DW_CFA_expression: r13 (r13) (DW_OP_lit8; DW_OP_minus; DW_OP_const4s: -32; DW_OP_and; DW_OP_const4s: -88; DW_OP_plus)  */
	.cfi_escape 0x10, 0x0d, 0x0e, 0x38, 0x1c, 0x0d, 0xe0, 0xff, 0xff, 0xff, 0x1a, 0x0d, 0xa8, 0xff, 0xff, 0xff, 0x22
	movl	%eax, %r13d
	movq	%r14, (%rsp)
	/*  DW_CFA_expression: r14 (r14) (DW_OP_lit8; DW_OP_minus; DW_OP_const4s: -32; DW_OP_and; DW_OP_const4s: -96; DW_OP_plus)  */
	.cfi_escape 0x10, 0x0e, 0x0e, 0x38, 0x1c, 0x0d, 0xe0, 0xff, 0xff, 0xff, 0x1a, 0x0d, 0xa0, 0xff, 0xff, 0xff, 0x22
	# LOE rbx r15 r12d r13d

	/* Range mask
	 * bits check
	 */

L(RANGEMASK_CHECK):
	btl	%r12d, %r13d

	/* Call scalar math function */
	jc	L(SCALAR_MATH_CALL)
	# LOE rbx r15 r12d r13d

	/* Special inputs
	 * processing loop
	 */

L(SPECIAL_VALUES_LOOP):
	incl	%r12d
	cmpl	$4, %r12d

	/* Check bits in range mask */
	jl	L(RANGEMASK_CHECK)
	# LOE rbx r15 r12d r13d

	movq	16(%rsp), %r12
	cfi_restore(12)
	movq	8(%rsp), %r13
	cfi_restore(13)
	movq	(%rsp), %r14
	cfi_restore(14)
	vmovupd	64(%rsp), %ymm0

	/* Go to exit */
	jmp	L(EXIT)
	/*  DW_CFA_expression: r12 (r12) (DW_OP_lit8; DW_OP_minus; DW_OP_const4s: -32; DW_OP_and; DW_OP_const4s: -80; DW_OP_plus)  */
	.cfi_escape 0x10, 0x0c, 0x0e, 0x38, 0x1c, 0x0d, 0xe0, 0xff, 0xff, 0xff, 0x1a, 0x0d, 0xb0, 0xff, 0xff, 0xff, 0x22
	/*  DW_CFA_expression: r13 (r13) (DW_OP_lit8; DW_OP_minus; DW_OP_const4s: -32; DW_OP_and; DW_OP_const4s: -88; DW_OP_plus)  */
	.cfi_escape 0x10, 0x0d, 0x0e, 0x38, 0x1c, 0x0d, 0xe0, 0xff, 0xff, 0xff, 0x1a, 0x0d, 0xa8, 0xff, 0xff, 0xff, 0x22
	/*  DW_CFA_expression: r14 (r14) (DW_OP_lit8; DW_OP_minus; DW_OP_const4s: -32; DW_OP_and; DW_OP_const4s: -96; DW_OP_plus)  */
	.cfi_escape 0x10, 0x0e, 0x0e, 0x38, 0x1c, 0x0d, 0xe0, 0xff, 0xff, 0xff, 0x1a, 0x0d, 0xa0, 0xff, 0xff, 0xff, 0x22
	# LOE rbx r12 r13 r14 r15 ymm0

	/* Scalar math function call
	 * to process special input
	 */

L(SCALAR_MATH_CALL):
	movl	%r12d, %r14d
	vmovsd	32(%rsp, %r14, 8), %xmm0
	call	tanh@PLT
	# LOE rbx r14 r15 r12d r13d xmm0

	vmovsd	%xmm0, 64(%rsp, %r14, 8)

	/* Process special inputs in loop */
	jmp	L(SPECIAL_VALUES_LOOP)
	# LOE rbx r15 r12d r13d
END(_ZGVdN4v_tanh_avx2)

	.section .rodata, "a"
	.align	32

#ifdef __svml_dtanh_data_internal_typedef
typedef unsigned int VUINT32;
typedef struct {
	__declspec(align(32)) VUINT32 _dbP[60*16][2];
	__declspec(align(32)) VUINT32 _dbSignMask[4][2];
	__declspec(align(32)) VUINT32 _dbAbsMask[4][2];
	__declspec(align(32)) VUINT32 _iExpMantMask[8][1];
	__declspec(align(32)) VUINT32 _iExpMask[8][1];
	__declspec(align(32)) VUINT32 _iMinIdxOfsMask[8][1];
	__declspec(align(32)) VUINT32 _iMaxIdxMask[8][1];
} __svml_dtanh_data_internal;
#endif
__svml_dtanh_data_internal:
	/* Polynomial coefficients */
	.quad	0x0000000000000000 /* PL0 = +0.000000000000000000000e-01 */
	.quad	0x0000000000000000 /* PH0 = +0.000000000000000000000e-01 */
	.quad	0x3FF0000000000000 /* P1  = +1.000000000000000014103e+00 */
	.quad	0xBD197DEAD79668D3 /* P2  = -2.264132406596103056796e-14 */
	.quad	0xBFD555555553AF3C /* P3  = -3.333333333273349741024e-01 */
	.quad	0xBE052F7CCA134846 /* P4  = -6.165791385711493738399e-10 */
	.quad	0x3FC11111563849D6 /* P5  = +1.333333655353061107201e-01 */
	.quad	0xBEB038623673FFB2 /* P6  = -9.668021563879858950855e-07 */
	.quad	0xBFAB9F685E64022E /* P7  = -5.395055916051593179252e-02 */
	.quad	0xBF2A54E2B28F2207 /* P8  = -2.008940439550829012647e-04 */
	.quad	0x3F97CFB9328A230E /* P9  = +2.325333949059698582189e-02 */
	.quad	0xBF75CA6D61723E02 /* P10 = -5.320002811586290441790e-03 */
	.quad	0x0000000000000000 /* B = +0 */
	.quad	0x3FF0000000000000 /* A = +1.0 */
	.quad	0x0000000000000000 /* Align value = +0 */
	.quad	0x0000000000000000 /* Align value = +0 */
	.quad	0x3C3708A564FAD29A /* PL0 = +1.248663375337163807466e-18 */
	.quad	0x3FC0E6973998DA48 /* PH0 = +1.320370703922029154143e-01 */
	.quad	0x3FEF712EB25C0888 /* P1  = +9.825662120422444519229e-01 */
	.quad	0xBFC09B296F7C1EA9 /* P2  = -1.297351641044220078331e-01 */
	.quad	0xBFD3DD77541EDDA7 /* P3  = -3.103922196855485849143e-01 */
	.quad	0x3FB58FFCF4309615 /* P4  = +8.422833406128689275566e-02 */
	.quad	0x3FBD3ABE845DCF49 /* P5  = +1.141776154670967208833e-01 */
	.quad	0xBFA791DF538C37FA /* P6  = -4.603479285115947936529e-02 */
	.quad	0xBFA4F872F69CD6E8 /* P7  = -4.095801601799370195284e-02 */
	.quad	0x3F9772E49EF6412B /* P8  = +2.289921970583567527179e-02 */
	.quad	0x3F8CBC0807393909 /* P9  = +1.403051635784581776625e-02 */
	.quad	0xBF85F06A30F93319 /* P10 = -1.071246110873285040939e-02 */
	.quad	0xBFC1000000000000 /* B = -.132813 */
	.quad	0x3FF0000000000000 /* A = +1 */
	.quad	0x0000000000000000 /* Align value = +0 */
	.quad	0x0000000000000000 /* Align value = +0 */
	.quad	0x3C6004EE5739DEAC /* PL0 = +6.947247374112211856530e-18 */
	.quad	0x3FC2DC968E6E0D62 /* PH0 = +1.473568149050193398786e-01 */
	.quad	0x3FEF4E1E606D96DF /* P1  = +9.782859691010478680677e-01 */
	.quad	0xBFC273BD70994AB9 /* P2  = -1.441571044730005866646e-01 */
	.quad	0xBFD382B548270D2C /* P3  = -3.048527912726111386771e-01 */
	.quad	0x3FB7CD2D582A6B29 /* P4  = +9.297450449450351894400e-02 */
	.quad	0x3FBC1278CCCBF0DB /* P5  = +1.096568584434324642303e-01 */
	.quad	0xBFA9C7F5115B86A1 /* P6  = -5.035367810138536095866e-02 */
	.quad	0xBFA371C21BAF618E /* P7  = -3.797728145554222910481e-02 */
	.quad	0x3F9958943F68417E /* P8  = +2.475196492201935923783e-02 */
	.quad	0x3F8930D5CFFD4152 /* P9  = +1.230017701132682667572e-02 */
	.quad	0xBF875CF7ADD31B76 /* P10 = -1.140779017658897660092e-02 */
	.quad	0xBFC3000000000000 /* B = -.148438 */
	.quad	0x3FF0000000000000 /* A = +1 */
	.quad	0x0000000000000000 /* Align value = +0 */
	.quad	0x0000000000000000 /* Align value = +0 */
	.quad	0x3C7EABE24E052A1F /* PL0 = +2.660321779421749543501e-17 */
	.quad	0x3FC4D04783618C71 /* PH0 = +1.626061812886266111366e-01 */
	.quad	0x3FEF2765AF97A4B3 /* P1  = +9.735592298067302883212e-01 */
	.quad	0xBFC443654205FEA5 /* P2  = -1.583067486171689074207e-01 */
	.quad	0xBFD31F2E208A5B97 /* P3  = -2.987780874040536844467e-01 */
	.quad	0x3FB9F235BD339878 /* P4  = +1.013520800512156573576e-01 */
	.quad	0x3FBAD0B0DFCCA141 /* P5  = +1.047468706498238100104e-01 */
	.quad	0xBFABD1B9600E608E /* P6  = -5.433444306908184548967e-02 */
	.quad	0xBFA1CEBEAF07DB58 /* P7  = -3.478046309094534453598e-02 */
	.quad	0x3F9AFC9FB1D8EFD2 /* P8  = +2.635430834764902126383e-02 */
	.quad	0x3F8573444F1AB502 /* P9  = +1.047376028449287564018e-02 */
	.quad	0xBF8874FBC8F24406 /* P10 = -1.194187838544459322219e-02 */
	.quad	0xBFC5000000000000 /* B = -.164063 */
	.quad	0x3FF0000000000000 /* A = +1 */
	.quad	0x0000000000000000 /* Align value = +0 */
	.quad	0x0000000000000000 /* Align value = +0 */
	.quad	0x3C7FB199D361A790 /* PL0 = +2.748994907060158996213e-17 */
	.quad	0x3FC6C170259E21F7 /* PH0 = +1.777782615356639783766e-01 */
	.quad	0x3FEEFD17479F7C65 /* P1  = +9.683948897253570478266e-01 */
	.quad	0xBFC609530FE4DF8D /* P2  = -1.721595599753950294577e-01 */
	.quad	0xBFD2B3465D71B4DE /* P3  = -2.921920692959484052676e-01 */
	.quad	0x3FBBFD2D34AC509B /* P4  = +1.093319181057403192166e-01 */
	.quad	0x3FB9778C3C16A0FE /* P5  = +9.948040453912551395183e-02 */
	.quad	0xBFADAC4D9E63C665 /* P6  = -5.795519407719210697372e-02 */
	.quad	0xBFA0139CCAD02D60 /* P7  = -3.139963126894929339124e-02 */
	.quad	0x3F9C5BF43BA6F19D /* P8  = +2.769452680671379432854e-02 */
	.quad	0x3F8190B703350341 /* P9  = +8.576803002712575184772e-03 */
	.quad	0xBF8936606782858A /* P10 = -1.231074634444230850234e-02 */
	.quad	0xBFC7000000000000 /* B = -.179688 */
	.quad	0x3FF0000000000000 /* A = +1 */
	.quad	0x0000000000000000 /* Align value = +0 */
	.quad	0x0000000000000000 /* Align value = +0 */
	.quad	0x3C6A917CA3624D50 /* PL0 = +1.152216693509785660691e-17 */
	.quad	0x3FC8AFD7B974FABB /* PH0 = +1.928662925292508878439e-01 */
	.quad	0x3FEECF47624A5D03 /* P1  = +9.628025932060214187231e-01 */
	.quad	0xBFC7C4C2CB4FDE4D /* P2  = -1.856921665891938814679e-01 */
	.quad	0xBFD23F69CB2C1F9D /* P3  = -2.851204380135586155453e-01 */
	.quad	0x3FBDEC5703A03814 /* P4  = +1.168875106670557712458e-01 */
	.quad	0x3FB8095003D0CF15 /* P5  = +9.389209836154706616487e-02 */
	.quad	0xBFAF554B47B10CBB /* P6  = -6.119761705533607365968e-02 */
	.quad	0xBF9C89743FE7BC1B /* P7  = -2.786809577986213853937e-02 */
	.quad	0x3F9D74725B746E7C /* P8  = +2.876452143855921824991e-02 */
	.quad	0x3F7B2D8AFB70B88C /* P9  = +6.635229968237631511880e-03 */
	.quad	0xBF89A0A2883EF6CB /* P10 = -1.251341799058582545252e-02 */
	.quad	0xBFC9000000000000 /* B = -.195313 */
	.quad	0x3FF0000000000000 /* A = +1 */
	.quad	0x0000000000000000 /* Align value = +0 */
	.quad	0x0000000000000000 /* Align value = +0 */
	.quad	0x3C7608279E8609CB /* PL0 = +1.910958764623660748269e-17 */
	.quad	0x3FCA9B46D2DDC5E3 /* PH0 = +2.078636674519166172015e-01 */
	.quad	0x3FEE9E0BB72A01A1 /* P1  = +9.567926957534390123919e-01 */
	.quad	0xBFC974FAD10C5330 /* P2  = -1.988824387305156976885e-01 */
	.quad	0xBFD1C40ACCBA4044 /* P3  = -2.775904654781735703430e-01 */
	.quad	0x3FBFBE24E2987853 /* P4  = +1.239951184474830487522e-01 */
	.quad	0x3FB6885B4345E47F /* P5  = +8.801813499839460539687e-02 */
	.quad	0xBFB06563D5670584 /* P6  = -6.404708824176991770896e-02 */
	.quad	0xBF98CD1D620DF6E2 /* P7  = -2.421995078065365147772e-02 */
	.quad	0x3F9E44EF3E844D21 /* P8  = +2.955983943054463683119e-02 */
	.quad	0x3F7325FA0148CAAE /* P9  = +4.674889165971292322643e-03 */
	.quad	0xBF89B4C8556C2D92 /* P10 = -1.255184660614964011319e-02 */
	.quad	0xBFCB000000000000 /* B = -.210938 */
	.quad	0x3FF0000000000000 /* A = +1 */
	.quad	0x0000000000000000 /* Align value = +0 */
	.quad	0x0000000000000000 /* Align value = +0 */
	.quad	0x3C6F19DAA20F51D5 /* PL0 = +1.348790537832000351176e-17 */
	.quad	0x3FCC83876CA98E15 /* PH0 = +2.227639465883021474557e-01 */
	.quad	0x3FEE697B662D07CD /* P1  = +9.503762241004040620296e-01 */
	.quad	0xBFCB194C7ED76ACF /* P2  = -2.117095584242946953999e-01 */
	.quad	0xBFD141A19E419762 /* P3  = -2.696308179350720680191e-01 */
	.quad	0x3FC0B89C64BC7B98 /* P4  = +1.306338779331468503007e-01 */
	.quad	0x3FB4F721150BBFC5 /* P5  = +8.189589275184434216748e-02 */
	.quad	0xBFB105AAFAB87898 /* P6  = -6.649273511036069461061e-02 */
	.quad	0xBF94FB3B31248C01 /* P7  = -2.048962104266749732921e-02 */
	.quad	0x3F9ECD31E588709C /* P8  = +3.007963145692880855964e-02 */
	.quad	0x3F664A91A335C105 /* P9  = +2.721104095762541127495e-03 */
	.quad	0xBF89754E32E1E26E /* P10 = -1.243077366619723806134e-02 */
	.quad	0xBFCD000000000000 /* B = -.226563 */
	.quad	0x3FF0000000000000 /* A = +1 */
	.quad	0x0000000000000000 /* Align value = +0 */
	.quad	0x0000000000000000 /* Align value = +0 */
	.quad	0x3C6AC6C889D8111D /* PL0 = +1.161245469312620769170e-17 */
	.quad	0x3FCE6864FE55A3D0 /* PH0 = +2.375608674877001114112e-01 */
	.quad	0x3FEE31AEE116B82B /* P1  = +9.435648342384913826391e-01 */
	.quad	0xBFCCB114B69E808B /* P2  = -2.241540805525839833707e-01 */
	.quad	0xBFD0B8AB913BA99D /* P3  = -2.612713735858507980441e-01 */
	.quad	0x3FC1823322BED48A /* P4  = +1.367858810096190233514e-01 */
	.quad	0x3FB35822B7929893 /* P5  = +7.556359273675842651653e-02 */
	.quad	0xBFB18B03CC78D2DA /* P6  = -6.852744810096158580830e-02 */
	.quad	0xBF911CCC3C8D5E5D /* P7  = -1.671141738492420009734e-02 */
	.quad	0x3F9F0DEC2D99B12F /* P8  = +3.032654789278515819797e-02 */
	.quad	0x3F4A28398B4EBD98 /* P9  = +7.982521989244205404918e-04 */
	.quad	0xBF88E60CB2FAB9A4 /* P10 = -1.215753480150000985458e-02 */
	.quad	0xBFCF000000000000 /* B = -.242188 */
	.quad	0x3FF0000000000000 /* A = +1 */
	.quad	0x0000000000000000 /* Align value = +0 */
	.quad	0x0000000000000000 /* Align value = +0 */
	.quad	0x3C89D2B6774FB61D /* PL0 = +4.479593208720169247958e-17 */
	.quad	0x3FD09C744F539BE4 /* PH0 = +2.595492148088267558848e-01 */
	.quad	0x3FEDD823B0400D42 /* P1  = +9.326342050921214825882e-01 */
	.quad	0xBFCEFBF7FF305FCC /* P2  = -2.420644756355144687086e-01 */
	.quad	0xBFCFC01DC4F24A41 /* P3  = -2.480504237797323303990e-01 */
	.quad	0x3FC291A2C26D5548 /* P4  = +1.450694512701977626753e-01 */
	.quad	0x3FB0D562E672D188 /* P5  = +6.575601698097532991976e-02 */
	.quad	0xBFB2201ECC119E06 /* P6  = -7.080261690281738261872e-02 */
	.quad	0xBF8695D50F778D31 /* P7  = -1.102796987010509974642e-02 */
	.quad	0x3F9EEC8CFBC031A0 /* P8  = +3.019924437107734972427e-02 */
	.quad	0xBF6030F0A4D3660A /* P9  = -1.976461417694923328722e-03 */
	.quad	0xBF87845288A4AEF5 /* P10 = -1.148285369398347838494e-02 */
	.quad	0xBFD1000000000000 /* B = -.265625 */
	.quad	0x3FF0000000000000 /* A = +1 */
	.quad	0x0000000000000000 /* Align value = +0 */
	.quad	0x0000000000000000 /* Align value = +0 */
	.quad	0x3C8B6AAB614D1C8D /* PL0 = +4.756035418366735312727e-17 */
	.quad	0x3FD275F7E1CF7F63 /* PH0 = +2.884502129727392616410e-01 */
	.quad	0x3FED56658F74C9CC /* P1  = +9.167964746359813351341e-01 */
	.quad	0xBFD0ECC045EBD596 /* P2  = -2.644501383614054083635e-01 */
	.quad	0xBFCD5A4BDE179180 /* P3  = -2.293181261476426808811e-01 */
	.quad	0x3FC3C00047D34767 /* P4  = +1.542969084462655120552e-01 */
	.quad	0x3FAAC7CE84FD609F /* P5  = +5.230565427217581251974e-02 */
	.quad	0xBFB288948D2E8B43 /* P6  = -7.239654967137902384931e-02 */
	.quad	0xBF6D6605AAD5A1C0 /* P7  = -3.588687008847041164896e-03 */
	.quad	0x3F9DDB0790848E97 /* P8  = +2.915584392134337382866e-02 */
	.quad	0xBF75FDE291BAD5B4 /* P9  = -5.369076763306269573660e-03 */
	.quad	0xBF84CEA5C52E0A78 /* P10 = -1.015977390284671071888e-02 */
	.quad	0xBFD3000000000000 /* B = -.296875 */
	.quad	0x3FF0000000000000 /* A = +1 */
	.quad	0x0000000000000000 /* Align value = +0 */
	.quad	0x0000000000000000 /* Align value = +0 */
	.quad	0x3C7139A81C8A6ECF /* PL0 = +1.494049799478574591322e-17 */
	.quad	0x3FD4470650036407 /* PH0 = +3.168350011233659890841e-01 */
	.quad	0x3FECC9A69DFDDD48 /* P1  = +8.996155820631566629678e-01 */
	.quad	0xBFD23DED3A37A09F /* P2  = -2.850297039535778028925e-01 */
	.quad	0xBFCAD302395D51C1 /* P3  = -2.095644741153943890185e-01 */
	.quad	0x3FC4A8FE3F309C22 /* P4  = +1.614072617096278705115e-01 */
	.quad	0x3FA3D161188AA436 /* P5  = +3.870681213931741151586e-02 */
	.quad	0xBFB288CFE5494E98 /* P6  = -7.240008685885823969403e-02 */
	.quad	0x3F6C7903EED8D334 /* P7  = +3.475673371918475361081e-03 */
	.quad	0x3F9BE023CDFB02F6 /* P8  = +2.722221321778569498033e-02 */
	.quad	0xBF80F8296F2C3A95 /* P9  = -8.285831170295390358336e-03 */
	.quad	0xBF8152DF4790049B /* P10 = -8.458847400108650973189e-03 */
	.quad	0xBFD5000000000000 /* B = -.328125 */
	.quad	0x3FF0000000000000 /* A = +1 */
	.quad	0x0000000000000000 /* Align value = +0 */
	.quad	0x0000000000000000 /* Align value = +0 */
	.quad	0x3C7751FE0FEE8335 /* PL0 = +2.022712113430213599928e-17 */
	.quad	0x3FD60EF7120502A9 /* PH0 = +3.446633983585721261456e-01 */
	.quad	0x3FEC32D951E56E6F /* P1  = +8.812071418319202070776e-01 */
	.quad	0xBFD370255FC004F8 /* P2  = -3.037198481616338996824e-01 */
	.quad	0xBFC832F0EBC6BB41 /* P3  = -1.890545989276351359107e-01 */
	.quad	0x3FC54C99A0FF432F /* P4  = +1.664001499289269127540e-01 */
	.quad	0x3F99DAC0CC283C18 /* P5  = +2.524853941036661688369e-02 */
	.quad	0xBFB227B3896A026D /* P6  = -7.091829399906553280461e-02 */
	.quad	0x3F84663364E1FB19 /* P7  = +9.960557476231411602383e-03 */
	.quad	0x3F9922D70DE07C57 /* P8  = +2.454696676442965935283e-02 */
	.quad	0xBF85C4A4EB6F86BC /* P9  = -1.062897532932837635222e-02 */
	.quad	0xBF7AAB61214FFE17 /* P10 = -6.511096396024671890972e-03 */
	.quad	0xBFD7000000000000 /* B = -.359375 */
	.quad	0x3FF0000000000000 /* A = +1 */
	.quad	0x0000000000000000 /* Align value = +0 */
	.quad	0x0000000000000000 /* Align value = +0 */
	.quad	0x3BFE67F266843B2C /* PL0 = +1.030196791298162288777e-19 */
	.quad	0x3FD7CD3115FC0F16 /* PH0 = +3.718989100163850869407e-01 */
	.quad	0x3FEB92F96CCC2C5B /* P1  = +8.616912007286247079761e-01 */
	.quad	0xBFD4827320135092 /* P2  = -3.204620183216856200247e-01 */
	.quad	0xBFC582B15550168A /* P3  = -1.680509249273891977521e-01 */
	.quad	0x3FC5AC3B9A2E4C31 /* P4  = +1.693186285816366254244e-01 */
	.quad	0x3F88FA599FCADAFB /* P5  = +1.219625491044728129762e-02 */
	.quad	0xBFB16EC8F5CA169E /* P6  = -6.809669495313605642174e-02 */
	.quad	0x3F90140EFC748BBE /* P7  = +1.570151725639922719844e-02 */
	.quad	0x3F95CFC49C1A28DC /* P8  = +2.130038454792147768770e-02 */
	.quad	0xBF8946ED8B1BF454 /* P9  = -1.234231549050882816697e-02 */
	.quad	0xBF7239E55C1DD50F /* P10 = -4.449745117985472755606e-03 */
	.quad	0xBFD9000000000000 /* B = -.390625 */
	.quad	0x3FF0000000000000 /* A = +1 */
	.quad	0x0000000000000000 /* Align value = +0 */
	.quad	0x0000000000000000 /* Align value = +0 */
	.quad	0x3C6412330191189C /* PL0 = +8.704448096175471149661e-18 */
	.quad	0x3FD9812B3B03F0A5 /* PH0 = +3.985088421175169703936e-01 */
	.quad	0x3FEAEB08C3C0E84D /* P1  = +8.411907027541559254748e-01 */
	.quad	0xBFD57446B1BC46CF /* P2  = -3.352219329545790787820e-01 */
	.quad	0xBFC2CA9ABC0444AD /* P3  = -1.468079965639267634401e-01 */
	.quad	0x3FC5CA95F9460D18 /* P4  = +1.702449290424759093710e-01 */
	.quad	0xBF2C2DAA35DD05C3 /* P5  = -2.149839664813813012186e-04 */
	.quad	0xBFB069A516EEB75D /* P6  = -6.411201295733578195472e-02 */
	.quad	0x3F9512716416FDC7 /* P7  = +2.057816670798986720058e-02 */
	.quad	0x3F921630CB1319A3 /* P8  = +1.766277541607908852593e-02 */
	.quad	0xBF8B76DA2EC99526 /* P9  = -1.341028647693549562145e-02 */
	.quad	0xBF63A97474A161E4 /* P10 = -2.400138332671485493040e-03 */
	.quad	0xBFDB000000000000 /* B = -.421875 */
	.quad	0x3FF0000000000000 /* A = +1 */
	.quad	0x0000000000000000 /* Align value = +0 */
	.quad	0x0000000000000000 /* Align value = +0 */
	.quad	0x3C89B79F5783381C /* PL0 = +4.461236087774530799537e-17 */
	.quad	0x3FDB2A6C993B829D /* PH0 = +4.244643684778937609003e-01 */
	.quad	0x3FEA3C0C1FBA328C /* P1  = +8.198299998926627915155e-01 */
	.quad	0xBFD6457212F78DE0 /* P2  = -3.479886231636708581604e-01 */
	.quad	0xBFC0129BDA380A66 /* P3  = -1.255678954622282824818e-01 */
	.quad	0x3FC5AB77F388FBDE /* P4  = +1.692953051696965507089e-01 */
	.quad	0xBF8822F3A6CADB7C /* P5  = -1.178541519889874597783e-02 */
	.quad	0xBFAE4A876370A4BD /* P6  = -5.916236008517603590739e-02 */
	.quad	0x3F991A89BC3B7710 /* P7  = +2.451529704455085335710e-02 */
	.quad	0x3F8C4A4328204D4B /* P8  = +1.381351915555364098800e-02 */
	.quad	0xBF8C5F921D01EC0B /* P9  = -1.385416174911393178490e-02 */
	.quad	0xBF3EE844C5B79FB8 /* P10 = -4.716079617694784908234e-04 */
	.quad	0xBFDD000000000000 /* B = -.453125 */
	.quad	0x3FF0000000000000 /* A = +1 */
	.quad	0x0000000000000000 /* Align value = +0 */
	.quad	0x0000000000000000 /* Align value = +0 */
	.quad	0x3C73FA437AD7AD87 /* PL0 = +1.732779905745858845932e-17 */
	.quad	0x3FDCC88C9902CF45 /* PH0 = +4.497405523536495697279e-01 */
	.quad	0x3FE9870845162D1D /* P1  = +7.977334355686341748810e-01 */
	.quad	0xBFD6F62358F73DA8 /* P2  = -3.587730759436120677668e-01 */
	.quad	0xBFBAC4345D675FE1 /* P3  = -1.045563438450467661101e-01 */
	.quad	0x3FC5539DA8287019 /* P4  = +1.666142531474868131862e-01 */
	.quad	0xBF96E3E0DC04A09F /* P5  = -2.235366194614185212822e-02 */
	.quad	0xBFAB5EC7147C207D /* P6  = -5.345747113284546871398e-02 */
	.quad	0x3F9C24166FFA7A58 /* P7  = +2.748141344511120915667e-02 */
	.quad	0x3F8451B907819844 /* P8  = +9.921498815128277696693e-03 */
	.quad	0xBF8C1C6D19191FCB /* P9  = -1.372609360545586670239e-02 */
	.quad	0x3F547372DF72E35A /* P10 = +1.248228245272117756098e-03 */
	.quad	0xBFDF000000000000 /* B = -.484375 */
	.quad	0x3FF0000000000000 /* A = +1 */
	.quad	0x0000000000000000 /* Align value = +0 */
	.quad	0x0000000000000000 /* Align value = +0 */
	.quad	0x3C848FE06EE49950 /* PL0 = +3.566941590788961528958e-17 */
	.quad	0x3FDF20211A36475D /* PH0 = +4.863360172249622803697e-01 */
	.quad	0x3FE86E67E6B80AC2 /* P1  = +7.634772783497611574659e-01 */
	.quad	0xBFD7C37C55474D9B /* P2  = -3.713064987943767913461e-01 */
	.quad	0xBFB2EBF15F3CB036 /* P3  = -7.391270232318521952684e-02 */
	.quad	0x3FC4718C8EF6E3AA /* P4  = +1.597152422016539530950e-01 */
	.quad	0xBFA277F8394E9B07 /* P5  = -3.607154559658991932071e-02 */
	.quad	0xBFA680312AB207E3 /* P6  = -4.394677778419955009224e-02 */
	.quad	0x3F9EDC9A8B57E286 /* P7  = +3.013841128810892143223e-02 */
	.quad	0x3F71B8C5E648EAF6 /* P8  = +4.326603932492947851719e-03 */
	.quad	0xBF89DB218356730C /* P9  = -1.262499029217558458029e-02 */
	.quad	0x3F6B05728E6EBC8E /* P10 = +3.298496001171330815865e-03 */
	.quad	0xBFE1000000000000 /* B = -.53125 */
	.quad	0x3FF0000000000000 /* A = +1 */
	.quad	0x0000000000000000 /* Align value = +0 */
	.quad	0x0000000000000000 /* Align value = +0 */
	.quad	0x3C8429831EDD94DE /* PL0 = +3.497576705878673192147e-17 */
	.quad	0x3FE10AF47E0BF610 /* PH0 = +5.325872861719194162333e-01 */
	.quad	0x3FE6EC5879F87EEE /* P1  = +7.163507826080299761242e-01 */
	.quad	0xBFD86AD001BFE200 /* P2  = -3.815193192563413204129e-01 */
	.quad	0xBFA239045B661385 /* P3  = -3.559125533778398983564e-02 */
	.quad	0x3FC2B4572D9CC147 /* P4  = +1.461285565105845078038e-01 */
	.quad	0xBFA99F4F01740705 /* P5  = -5.004355328311586406115e-02 */
	.quad	0xBF9F449C484F4879 /* P6  = -3.053516570418721511214e-02 */
	.quad	0x3F9F5F42169D7DDE /* P7  = +3.063681853325116830798e-02 */
	.quad	0xBF6111B1BA632A97 /* P8  = -2.083632588527460989469e-03 */
	.quad	0xBF84725FBE5B6E61 /* P9  = -9.983776089419639342530e-03 */
	.quad	0x3F7438A2986CFA9C /* P10 = +4.936823976832951342488e-03 */
	.quad	0xBFE3000000000000 /* B = -.59375 */
	.quad	0x3FF0000000000000 /* A = +1 */
	.quad	0x0000000000000000 /* Align value = +0 */
	.quad	0x0000000000000000 /* Align value = +0 */
	.quad	0x3C6BE9160BFB3505 /* PL0 = +1.210424670976053242391e-17 */
	.quad	0x3FE26D76F73233C7 /* PH0 = +5.758623912857893101247e-01 */
	.quad	0x3FE56363B5B93937 /* P1  = +6.683825063026124740752e-01 */
	.quad	0xBFD8A2244B27297E /* P2  = -3.848963483730115724200e-01 */
	.quad	0xBF52CA2F101EEF63 /* P3  = -1.146837196286797844817e-03 */
	.quad	0x3FC081BC342243AD /* P4  = +1.289592032012739958675e-01 */
	.quad	0xBFAE38DB4A932344 /* P5  = -5.902753148399722719732e-02 */
	.quad	0xBF91F814D4AE90C6 /* P6  = -1.754791782481459457885e-02 */
	.quad	0x3F9D056AE193C4F3 /* P7  = +2.834097863973723355792e-02 */
	.quad	0xBF7BD0B502D8F3A0 /* P8  = -6.790835451792626336974e-03 */
	.quad	0xBF7B763F7BB8AE2F /* P9  = -6.704566938008179114124e-03 */
	.quad	0x3F76036F42D9AB69 /* P10 = +5.374369252971835729099e-03 */
	.quad	0xBFE5000000000000 /* B = -.65625 */
	.quad	0x3FF0000000000000 /* A = +1 */
	.quad	0x0000000000000000 /* Align value = +0 */
	.quad	0x0000000000000000 /* Align value = +0 */
	.quad	0x3C8B64AF0450486E /* PL0 = +4.751979286662385162741e-17 */
	.quad	0x3FE3B75F8BCB742D /* PH0 = +6.161344271055263499548e-01 */
	.quad	0x3FE3DA23BC12369F /* P1  = +6.203783677353447780947e-01 */
	.quad	0xBFD8768FF4B46416 /* P2  = -3.822364701932782367281e-01 */
	.quad	0x3F9D67CB8AD9CB1A /* P3  = +2.871625933625941117406e-02 */
	.quad	0x3FBC168CB7827DF4 /* P4  = +1.097190807363331305006e-01 */
	.quad	0xBFB03A2B83C9272E /* P5  = -6.338760344911228324430e-02 */
	.quad	0xBF789FEB595297DC /* P6  = -6.011885959344067548074e-03 */
	.quad	0x3F98BD01B4C335E7 /* P7  = +2.415850320612902513532e-02 */
	.quad	0xBF83BADC303D6535 /* P8  = -9.633751127398152979976e-03 */
	.quad	0xBF6C54E7A1C1E3F3 /* P9  = -3.458454519258407989501e-03 */
	.quad	0x3F7408394B7EF3E7 /* P10 = +4.890655334688332484537e-03 */
	.quad	0xBFE7000000000000 /* B = -.71875 */
	.quad	0x3FF0000000000000 /* A = +1 */
	.quad	0x0000000000000000 /* Align value = +0 */
	.quad	0x0000000000000000 /* Align value = +0 */
	.quad	0x3C6A48557F6E0D3E /* PL0 = +1.139824111505584215867e-17 */
	.quad	0x3FE4E8D895B010DC /* PH0 = +6.534235881413468227663e-01 */
	.quad	0x3FE25652FAAF8A73 /* P1  = +5.730376144604875448991e-01 */
	.quad	0xBFD7F6C3A57C444B /* P2  = -3.744362941807295084434e-01 */
	.quad	0x3FAB7866E3F99EBE /* P3  = +5.365296872042567001598e-02 */
	.quad	0x3FB6FA1DF47CCD40 /* P4  = +8.975398272450707099784e-02 */
	.quad	0xBFB05508D3741B8E /* P5  = -6.379752314033580026840e-02 */
	.quad	0x3F6C3EFDF7BB279C /* P6  = +3.448005705512137236209e-03 */
	.quad	0x3F9372BADD6D3E27 /* P7  = +1.899234749299530050806e-02 */
	.quad	0xBF860FD5AE65F3DA /* P8  = -1.077238977881649471165e-02 */
	.quad	0xBF47266FFB07E628 /* P9  = -7.064863949032872448118e-04 */
	.quad	0x3F6F9763992C2A05 /* P10 = +3.856367614735181120799e-03 */
	.quad	0xBFE9000000000000 /* B = -.78125 */
	.quad	0x3FF0000000000000 /* A = +1 */
	.quad	0x0000000000000000 /* Align value = +0 */
	.quad	0x0000000000000000 /* Align value = +0 */
	.quad	0x3C6BB6A2B194E3AB /* PL0 = +1.201878007209462528697e-17 */
	.quad	0x3FE602609AAE7C22 /* PH0 = +6.877902051090851731630e-01 */
	.quad	0x3FE0DCBAFE191C7F /* P1  = +5.269446337560025312137e-01 */
	.quad	0xBFD732028428A9FB /* P2  = -3.624273577321727538225e-01 */
	.quad	0x3FB2D92389BE065B /* P3  = +7.362577545975439796588e-02 */
	.quad	0x3FB1F6A9C8C49993 /* P4  = +7.017003203927733370937e-02 */
	.quad	0xBFAF47C0B50B56EE /* P5  = -6.109430513394707378526e-02 */
	.quad	0x3F85A8EDD1356223 /* P6  = +1.057611269668352068104e-02 */
	.quad	0x3F8BE05C5CD1B4FA /* P7  = +1.361152799855823798207e-02 */
	.quad	0xBF85A0EFE4552F76 /* P8  = -1.056086936537046752272e-02 */
	.quad	0x3F559F2A6A356194 /* P9  = +1.319686337259627831943e-03 */
	.quad	0x3F6576F5E989208D /* P10 = +2.620201394425042596201e-03 */
	.quad	0xBFEB000000000000 /* B = -.84375 */
	.quad	0x3FF0000000000000 /* A = +1 */
	.quad	0x0000000000000000 /* Align value = +0 */
	.quad	0x0000000000000000 /* Align value = +0 */
	.quad	0x3C80328BD86C8B74 /* PL0 = +2.809809047161267929701e-17 */
	.quad	0x3FE704BB1B7FCB81 /* PH0 = +7.193275010198335595035e-01 */
	.quad	0x3FDEE264AAD6C40C /* P1  = +4.825679462765613089739e-01 */
	.quad	0xBFD637493CE659F1 /* P2  = -3.471243948673921548357e-01 */
	.quad	0x3FB6BE3A3DEE6F4A /* P3  = +8.884014141079635303208e-02 */
	.quad	0x3FAA85EB6470AC0F /* P4  = +5.180297471118688523488e-02 */
	.quad	0xBFACC0146EA4858D /* P5  = -5.615295267694895314457e-02 */
	.quad	0x3F8F8FB683CDDAC5 /* P6  = +1.541082944616557159055e-02 */
	.quad	0x3F819515DEE2CB91 /* P7  = +8.585139145315585602547e-03 */
	.quad	0xBF834E45E6AF9EA1 /* P8  = -9.426637747267209169415e-03 */
	.quad	0x3F65250F197CA56D /* P9  = +2.581147662472352252568e-03 */
	.quad	0x3F57A766026D036C /* P10 = +1.443719500187702367690e-03 */
	.quad	0xBFED000000000000 /* B = -.90625 */
	.quad	0x3FF0000000000000 /* A = +1 */
	.quad	0x0000000000000000 /* Align value = +0 */
	.quad	0x0000000000000000 /* Align value = +0 */
	.quad	0x3C716F7EEF7B61AD /* PL0 = +1.512291215142578135651e-17 */
	.quad	0x3FE7F0E1A4CD846E /* PH0 = +7.481544703297353660076e-01 */
	.quad	0x3FDC2D4CC872DC09 /* P1  = +4.402648885256331012598e-01 */
	.quad	0xBFD514A99F92ED53 /* P2  = -3.293861444796750250530e-01 */
	.quad	0x3FB9846A6CF2F337 /* P3  = +9.967675361526749494844e-02 */
	.quad	0x3FA20896939AB161 /* P4  = +3.522177268800664413493e-02 */
	.quad	0xBFA97E801F31EE0D /* P5  = -4.979324703978358553405e-02 */
	.quad	0x3F92A11F47B82085 /* P6  = +1.819275737037219740638e-02 */
	.quad	0x3F717D70FE289C34 /* P7  = +4.270020845559097605514e-03 */
	.quad	0xBF7FDCF1D3F6CE2D /* P8  = -7.779068604054678540132e-03 */
	.quad	0x3F69F607E81AF6B6 /* P9  = +3.169074480722534625181e-03 */
	.quad	0x3F3F925C80D0F889 /* P10 = +4.817462766516585511824e-04 */
	.quad	0xBFEF000000000000 /* B = -.96875 */
	.quad	0x3FF0000000000000 /* A = +1 */
	.quad	0x0000000000000000 /* Align value = +0 */
	.quad	0x0000000000000000 /* Align value = +0 */
	.quad	0x3C931A11D7E8606E /* PL0 = +6.627280241435322692188e-17 */
	.quad	0x3FE92BFB370D9B71 /* PH0 = +7.866188121086975515439e-01 */
	.quad	0x3FD866160E454111 /* P1  = +3.812308444367014680480e-01 */
	.quad	0xBFD33149F3801DBA /* P2  = -2.998833539899937679796e-01 */
	.quad	0x3FBBDB6D4C949899 /* P3  = +1.088169395412442909023e-01 */
	.quad	0x3F8D6AB2A74B9343 /* P4  = +1.436366627735597372494e-02 */
	.quad	0xBFA404D1047C5D72 /* P5  = -3.909924678571997970917e-02 */
	.quad	0x3F93C47D9ACCD919 /* P6  = +1.930423981976856424661e-02 */
	.quad	0xBF41B755642CFF1B /* P7  = -5.406538915408738478158e-04 */
	.quad	0xBF74B5301AA1E788 /* P8  = -5.055606752756853900641e-03 */
	.quad	0x3F69A84C5B2A3E68 /* P9  = +3.132008679422249529120e-03 */
	.quad	0xBF3CF47830328C11 /* P10 = -4.418176105877589308931e-04 */
	.quad	0xBFF1000000000000 /* B = -1.0625 */
	.quad	0x3FF0000000000000 /* A = +1 */
	.quad	0x0000000000000000 /* Align value = +0 */
	.quad	0x0000000000000000 /* Align value = +0 */
	.quad	0x3C884D471B8FD396 /* PL0 = +4.215701792312937090514e-17 */
	.quad	0x3FEA8DBCBC31897A /* PH0 = +8.298019099859594849278e-01 */
	.quad	0x3FD3EE730537C8EA /* P1  = +3.114287901836535219818e-01 */
	.quad	0xBFD08A05AD27CE32 /* P2  = -2.584242049190123217982e-01 */
	.quad	0x3FBC5255406F84B6 /* P3  = +1.106313021005175045399e-01 */
	.quad	0xBF772FA2F633AA5E /* P4  = -5.660664147607434209241e-03 */
	.quad	0xBF99DD8E4C473FC4 /* P5  = -2.525923100057504533247e-02 */
	.quad	0x3F9183C935B6495D /* P6  = +1.710428610165003372069e-02 */
	.quad	0xBF70471A3A591480 /* P7  = -3.974058583087303228038e-03 */
	.quad	0xBF603DDD4DEBB9A4 /* P8  = -1.982624278176818987264e-03 */
	.quad	0x3F62591E44D3C17F /* P9  = +2.239760512218135956425e-03 */
	.quad	0xBF4C195D3A9B1AB4 /* P10 = -8.575158328419569430544e-04 */
	.quad	0xBFF3000000000000 /* B = -1.1875 */
	.quad	0x3FF0000000000000 /* A = +1 */
	.quad	0x0000000000000000 /* Align value = +0 */
	.quad	0x0000000000000000 /* Align value = +0 */
	.quad	0x3C90DD1C9BFF7F64 /* PL0 = +5.850777430004479798187e-17 */
	.quad	0x3FEBAD50A4A68BC1 /* PH0 = +8.649066177207417327466e-01 */
	.quad	0x3FD01FBA72CEE1A5 /* P1  = +2.519365426228666233893e-01 */
	.quad	0xBFCBE432F647C4D6 /* P2  = -2.179015829602010702633e-01 */
	.quad	0x3FBABF92B6E5AC73 /* P3  = +1.044856735731387955105e-01 */
	.quad	0xBF922983AA24E217 /* P4  = -1.773648954369563555378e-02 */
	.quad	0xBF8C72214C14E23A /* P5  = -1.388956082756564056328e-02 */
	.quad	0x3F8ACB4D1F388E8B /* P6  = +1.308307887581540972153e-02 */
	.quad	0xBF740EF8B4A2EE3B /* P7  = -4.897090441029978580995e-03 */
	.quad	0xBF0EA9F30C8DC900 /* P8  = -5.848668076326342477133e-05 */
	.quad	0x3F53CC40D18713AE /* P9  = +1.208365725788622757410e-03 */
	.quad	0xBF4848B86029CBA1 /* P10 = -7.410908004444779592485e-04 */
	.quad	0xBFF5000000000000 /* B = -1.3125 */
	.quad	0x3FF0000000000000 /* A = +1 */
	.quad	0x0000000000000000 /* Align value = +0 */
	.quad	0x0000000000000000 /* Align value = +0 */
	.quad	0x3C8FB61781D22681 /* PL0 = +5.501032995458057064843e-17 */
	.quad	0x3FEC950A3340C8BF /* PH0 = +8.931933404003514764824e-01 */
	.quad	0x3FC9E1DFFD385423 /* P1  = +2.022056566644617586005e-01 */
	.quad	0xBFC71E2FF88EBA23 /* P2  = -1.806087459239772032583e-01 */
	.quad	0x3FB80AEBD07AB5BA /* P3  = +9.391664352252506838449e-02 */
	.quad	0xBF98404E27EAE6ED /* P4  = -2.368280523908243895884e-02 */
	.quad	0xBF772DA520B5006E /* P5  = -5.658764868087568802107e-03 */
	.quad	0x3F824C9268AF9423 /* P6  = +8.935111827620250551925e-03 */
	.quad	0xBF722AE76D206AE3 /* P7  = -4.435447701349490160113e-03 */
	.quad	0x3F4B807F56298D5E /* P8  = +8.392926941493230644497e-04 */
	.quad	0x3F3D71027DF95D2A /* P9  = +4.492407879061627603159e-04 */
	.quad	0xBF3EBD17676755FB /* P10 = -4.690343988874298905483e-04 */
	.quad	0xBFF7000000000000 /* B = -1.4375 */
	.quad	0x3FF0000000000000 /* A = +1 */
	.quad	0x0000000000000000 /* Align value = +0 */
	.quad	0x0000000000000000 /* Align value = +0 */
	.quad	0x3C95393C63CE8224 /* PL0 = +7.363407705201031038415e-17 */
	.quad	0x3FED4E6F464286B0 /* PH0 = +9.158245441687622445670e-01 */
	.quad	0x3FC4A45842B7DE1E /* P1  = +1.612654042980787191461e-01 */
	.quad	0xBFC2E7885AFDD3D0 /* P2  = -1.476908153814791087327e-01 */
	.quad	0x3FB4DD6DD51D3FEB /* P3  = +8.150373890862254580204e-02 */
	.quad	0xBF9A05D3ADAB489C /* P4  = -2.541285274021075503042e-02 */
	.quad	0xBF3459B643B4995C /* P5  = -3.105230313899165257622e-04 */
	.quad	0x3F766B30745F2E3A /* P6  = +5.473317409222350365811e-03 */
	.quad	0xBF6C2C891E555BDF /* P7  = -3.439204988051155730940e-03 */
	.quad	0x3F5194F30D6C576D /* P8  = +1.073109966176012791522e-03 */
	.quad	0x3EF4DBB43C3132A2 /* P9  = +1.989194766975849961365e-05 */
	.quad	0xBF2E45EBAB3C15A0 /* P10 = -2.309656316514087783666e-04 */
	.quad	0xBFF9000000000000 /* B = -1.5625 */
	.quad	0x3FF0000000000000 /* A = +1 */
	.quad	0x0000000000000000 /* Align value = +0 */
	.quad	0x0000000000000000 /* Align value = +0 */
	.quad	0x3C75111669651DAA /* PL0 = +1.827249135453834384396e-17 */
	.quad	0x3FEDE1EB5937518F /* PH0 = +9.338280432225917193634e-01 */
	.quad	0x3FC06129C7C8EBB1 /* P1  = +1.279651856910653382507e-01 */
	.quad	0xBFBE9763041064E1 /* P2  = -1.194974789545031421774e-01 */
	.quad	0x3FB1A5B9F9113928 /* P3  = +6.893503504509068635308e-02 */
	.quad	0xBF992145039F9AFE /* P4  = -2.454097590080105816526e-02 */
	.quad	0x3F66CB116EA49C89 /* P5  = +2.782377288116648315142e-03 */
	.quad	0x3F67F972FDF30001 /* P6  = +2.926563829163342740100e-03 */
	.quad	0xBF63A7B5975F02F3 /* P7  = -2.399305983061922438601e-03 */
	.quad	0x3F4FDE7B8777F4C8 /* P8  = +9.725669069095216373599e-04 */
	.quad	0xBF25918876626BA4 /* P9  = -1.645545082212515656240e-04 */
	.quad	0xBF1495123C991F00 /* P10 = -7.851527984669912693674e-05 */
	.quad	0xBFFB000000000000 /* B = -1.6875 */
	.quad	0x3FF0000000000000 /* A = +1 */
	.quad	0x0000000000000000 /* Align value = +0 */
	.quad	0x0000000000000000 /* Align value = +0 */
	.quad	0x3C9F29A5B7426D27 /* PL0 = +1.081172820484012446345e-16 */
	.quad	0x3FEE56B6F3EFABFC /* PH0 = +9.480852856044061915952e-01 */
	.quad	0x3FB9E3EFD94BB9FC /* P1  = +1.011342912204113371518e-01 */
	.quad	0xBFB88BD9760FECA7 /* P2  = -9.588393337610288420285e-02 */
	.quad	0x3FAD48A0350B3ACF /* P3  = +5.719471595295077387313e-02 */
	.quad	0xBF96CC6A5110F129 /* P4  = -2.226415748394675367257e-02 */
	.quad	0x3F71934687170384 /* P5  = +4.290843485649345772606e-03 */
	.quad	0x3F5407BAF73B3DF9 /* P6  = +1.222546180475235334287e-03 */
	.quad	0xBF591B626C0646DD /* P7  = -1.532407870488964407324e-03 */
	.quad	0x3F48B0E1DD283558 /* P8  = +7.535078860329375669277e-04 */
	.quad	0xBF2B322292840D2B /* P9  = -2.074877932117605962646e-04 */
	.quad	0xBE99E4061120C741 /* P10 = -3.858017559892704559672e-07 */
	.quad	0xBFFD000000000000 /* B = -1.8125 */
	.quad	0x3FF0000000000000 /* A = +1 */
	.quad	0x0000000000000000 /* Align value = +0 */
	.quad	0x0000000000000000 /* Align value = +0 */
	.quad	0x3C6AF8C2041C67CD /* PL0 = +1.169711482626385762338e-17 */
	.quad	0x3FEEB2DFEDD5EC93 /* PH0 = +9.593352933146824801369e-01 */
	.quad	0x3FB465A205CFB638 /* P1  = +7.967579500083210999681e-02 */
	.quad	0xBFB3914BF68D39FF /* P2  = -7.643580216720378576778e-02 */
	.quad	0x3FA7F21A08C5C734 /* P3  = +4.676896435820623621673e-02 */
	.quad	0xBF93DA9560EA9960 /* P4  = -1.938851741820124550772e-02 */
	.quad	0x3F73953FEC62820E /* P5  = +4.781007481284861359820e-03 */
	.quad	0x3F2749D5E1273E3C /* P6  = +1.776765426044646108071e-04 */
	.quad	0xBF4D46B0B498CE5A /* P7  = -8.934367007839658352859e-04 */
	.quad	0x3F4153D680E1F4C4 /* P8  = +5.287930851093571206574e-04 */
	.quad	0xBF28477014ECA6A2 /* P9  = -1.852344816708944640949e-04 */
	.quad	0x3EFFAC54E07CEB4B /* P10 = +3.020588886147182143902e-05 */
	.quad	0xBFFF000000000000 /* B = -1.9375 */
	.quad	0x3FF0000000000000 /* A = +1 */
	.quad	0x0000000000000000 /* Align value = +0 */
	.quad	0x0000000000000000 /* Align value = +0 */
	.quad	0x3C7A8AF2BB2231F2 /* PL0 = +2.302217989249372577466e-17 */
	.quad	0x3FEF1994DF724FC8 /* PH0 = +9.718727459135090285258e-01 */
	.quad	0x3FAC65B1BC0C9D58 /* P1  = +5.546336575053583942603e-02 */
	.quad	0xBFAB9937BDA747C8 /* P2  = -5.390333356957871365599e-02 */
	.quad	0x3FA15B42D9EF931C /* P3  = +3.389939222669210777241e-02 */
	.quad	0xBF8EACD8E8507A3C /* P4  = -1.497811755149058215502e-02 */
	.quad	0x3F7263A15721C682 /* P5  = +4.489546046998806349050e-03 */
	.quad	0xBF42A032ACDC3B32 /* P6  = -5.684134900735048121829e-04 */
	.quad	0xBF3431E79B5AD185 /* P7  = -3.081503340170088810438e-04 */
	.quad	0x3F31B51667C7DF5E /* P8  = +2.701930714290502424828e-04 */
	.quad	0xBF1F8709579250AD /* P9  = -1.202678157759563704341e-04 */
	.quad	0x3F01ED8ED1BF9595 /* P10 = +3.419487094883790833778e-05 */
	.quad	0xC001000000000000 /* B = -2.125 */
	.quad	0x3FF0000000000000 /* A = +1 */
	.quad	0x0000000000000000 /* Align value = +0 */
	.quad	0x0000000000000000 /* Align value = +0 */
	.quad	0x3C86F3F7C3DAFC55 /* PL0 = +3.981710680748877459333e-17 */
	.quad	0x3FEF73776B2AA2DB /* PH0 = +9.828450291725759901951e-01 */
	.quad	0x3FA16A7FC4D7B900 /* P1  = +3.401564863075812007064e-02 */
	.quad	0xBFA11E03803AD621 /* P2  = -3.343211117082156940532e-02 */
	.quad	0x3F9609591597297F /* P3  = +2.152003473546803654658e-02 */
	.quad	0xBF847E74ED9BBB0C /* P4  = -1.000682211039596246436e-02 */
	.quad	0x3F6BFF771725CD65 /* P5  = +3.417713736035987187864e-03 */
	.quad	0xBF491D1FF73C18FA /* P6  = -7.664114077392807421000e-04 */
	.quad	0x3EF53EE467B51DC5 /* P7  = +2.026145237479599375099e-05 */
	.quad	0x3F160135BE0D94A0 /* P8  = +8.394136922403255700685e-05 */
	.quad	0xBF0B32CB1D276A40 /* P9  = -5.187685350778849443841e-05 */
	.quad	0x3EF4DAF70C12D555 /* P10 = +1.988919462255396826584e-05 */
	.quad	0xC003000000000000 /* B = -2.375 */
	.quad	0x3FF0000000000000 /* A = +1 */
	.quad	0x0000000000000000 /* Align value = +0 */
	.quad	0x0000000000000000 /* Align value = +0 */
	.quad	0x3C19DBF4E2E5B7DC /* PL0 = +3.504575836708380670219e-19 */
	.quad	0x3FEFAA7934B75EBD /* PH0 = +9.895597486128832054320e-01 */
	.quad	0x3F9545200830A42C /* P1  = +2.077150392520736492125e-02 */
	.quad	0xBF950C46D285F6BC /* P2  = -2.055464420253970271376e-02 */
	.quad	0x3F8B79F5BFC6513F /* P3  = +1.341621390819425058164e-02 */
	.quad	0xBF7A50ADAD777898 /* P4  = -6.424597194806612772505e-03 */
	.quad	0x3F633A19BE8255E3 /* P5  = +2.347040444940816227383e-03 */
	.quad	0xBF44E609BC2557B7 /* P6  = -6.377742322836087134324e-04 */
	.quad	0x3F1AFCBAD60EAACD /* P7  = +1.029480968230231421206e-04 */
	.quad	0x3EE80476AC34A8EF /* P8  = +1.145240583485084317660e-05 */
	.quad	0xBEF278E23DE463E9 /* P9  = -1.761646478213091821804e-05 */
	.quad	0x3EE209FAF377264D /* P10 = +8.601658563106529694651e-06 */
	.quad	0xC005000000000000 /* B = -2.625 */
	.quad	0x3FF0000000000000 /* A = +1 */
	.quad	0x0000000000000000 /* Align value = +0 */
	.quad	0x0000000000000000 /* Align value = +0 */
	.quad	0x3C979D62702C631C /* PL0 = +8.193023793215066385979e-17 */
	.quad	0x3FEFCC04CDBCDC4B /* PH0 = +9.936546343150295390600e-01 */
	.quad	0x3F89E87D088D269A /* P1  = +1.265046770426474576547e-02 */
	.quad	0xBF89BE6721012B80 /* P2  = -1.257019586059526836624e-02 */
	.quad	0x3F80F1C13E8D39D3 /* P3  = +8.273610803056031004326e-03 */
	.quad	0xBF7082DBC9602757 /* P4  = -4.031046430108839563004e-03 */
	.quad	0x3F590BE9BD4E0A11 /* P5  = +1.528719197467002507978e-03 */
	.quad	0xBF3DCC2BEF6D0283 /* P6  = -4.546744598208711809986e-04 */
	.quad	0x3F1A08065C4A8E85 /* P7  = +9.930170842636406837764e-05 */
	.quad	0xBEE528117D0410F3 /* P8  = -1.008821337267942266431e-05 */
	.quad	0xBED0BE73A44FF565 /* P9  = -3.992069257383521775961e-06 */
	.quad	0x3EC9B0C11E342E38 /* P10 = +3.062539904901699218737e-06 */
	.quad	0xC007000000000000 /* B = -2.875 */
	.quad	0x3FF0000000000000 /* A = +1 */
	.quad	0x0000000000000000 /* Align value = +0 */
	.quad	0x0000000000000000 /* Align value = +0 */
	.quad	0x3C804B931AD7A3CC /* PL0 = +2.826768921701616830245e-17 */
	.quad	0x3FEFE06EB0688212 /* PH0 = +9.961465306733450209009e-01 */
	.quad	0x3F7F81BD8876224D /* P1  = +7.692089427458426472642e-03 */
	.quad	0xBF7F62A8C699A963 /* P2  = -7.662448196791823756776e-03 */
	.quad	0x3F74C31E2B2A6A28 /* P3  = +5.068891378551522166321e-03 */
	.quad	0xBF6470D537F16227 /* P4  = -2.495209162173734080001e-03 */
	.quad	0x3F4FAEEF61C89673 /* P5  = +9.668988091717359455754e-04 */
	.quad	0xBF33C5E80B349783 /* P6  = -3.017131341088651514023e-04 */
	.quad	0x3F138F3D31037A6B /* P7  = +7.461367590931028650557e-05 */
	.quad	0xBEEB3C780996FFE3 /* P8  = -1.298723536791163711556e-05 */
	.quad	0x3E9D0C75BC8BFEFC /* P9  = +4.328589367358221917138e-07 */
	.quad	0x3EAC3865227764D4 /* P10 = +8.410302755848104487452e-07 */
	.quad	0xC009000000000000 /* B = -3.125 */
	.quad	0x3FF0000000000000 /* A = +1 */
	.quad	0x0000000000000000 /* Align value = +0 */
	.quad	0x0000000000000000 /* Align value = +0 */
	.quad	0x3C5B978B202749F9 /* PL0 = +5.983054034451594408315e-18 */
	.quad	0x3FEFECD6B7EA3128 /* PH0 = +9.976609794698889643882e-01 */
	.quad	0x3F73238B786137FE /* P1  = +4.672570043181776968058e-03 */
	.quad	0xBF731815ACEA072E /* P2  = -4.661640805922390930706e-03 */
	.quad	0x3F6956F0816D5AEE /* P3  = +3.093213784647877798933e-03 */
	.quad	0xBF591A16286C4885 /* P4  = -1.532098425461232453877e-03 */
	.quad	0x3F43B3E3A00C6096 /* P5  = +6.012784434430592468442e-04 */
	.quad	0xBF29441B2A56DEC7 /* P6  = -1.927645836710038499293e-04 */
	.quad	0x3F0A99C3A2E857B6 /* P7  = +5.073669705184196724674e-05 */
	.quad	0xBEE61CB034DDC151 /* P8  = -1.054385361573597042258e-05 */
	.quad	0x3EB792BBC76D6107 /* P9  = +1.405070887824641788698e-06 */
	.quad	0x3E761472362A16F0 /* P10 = +8.225391704739515383837e-08 */
	.quad	0xC00B000000000000 /* B = -3.375 */
	.quad	0x3FF0000000000000 /* A = +1 */
	.quad	0x0000000000000000 /* Align value = +0 */
	.quad	0x0000000000000000 /* Align value = +0 */
	.quad	0x3C9C290AFCBDE00D /* PL0 = +9.770074992945060684926e-17 */
	.quad	0x3FEFF45F6D36133A /* PH0 = +9.985806592017987259879e-01 */
	.quad	0x3F673CEC093032DE /* P1  = +2.836667068100913999228e-03 */
	.quad	0xBF67347A7CD844D5 /* P2  = -2.832640870800243808078e-03 */
	.quad	0x3F5EDA25530355DB /* P3  = +1.883064698679040793627e-03 */
	.quad	0xBF4EAD3BBABC1BA9 /* P4  = -9.361783645268534848806e-04 */
	.quad	0x3F3842E61CD35432 /* P5  = +3.701984213198588740338e-04 */
	.quad	0xBF1F9AB7FD1A3DDD /* P6  = -1.205611036090218544867e-04 */
	.quad	0x3F0136C154EA3DED /* P7  = +3.283288480304320224929e-05 */
	.quad	0xBEDF12807F721E66 /* P8  = -7.408207230892235753013e-06 */
	.quad	0x3EB5B53687AD5112 /* P9  = +1.293889481520047941659e-06 */
	.quad	0xBE801E90FBFED147 /* P10 = -1.200988872775447204019e-07 */
	.quad	0xC00D000000000000 /* B = -3.625 */
	.quad	0x3FF0000000000000 /* A = +1 */
	.quad	0x0000000000000000 /* Align value = +0 */
	.quad	0x0000000000000000 /* Align value = +0 */
	.quad	0x3C9E323294294877 /* PL0 = +1.047637125334028950603e-16 */
	.quad	0x3FEFF8F21CDAAA62 /* PH0 = +9.991388858373506653976e-01 */
	.quad	0x3F5C3470628813F2 /* P1  = +1.721486807697344658108e-03 */
	.quad	0xBF5C2E38AC6FF8D2 /* P2  = -1.720004411026422324849e-03 */
	.quad	0x3F52C13234626F43 /* P3  = +1.144694354969070234454e-03 */
	.quad	0xBF42B0A47DF47BB4 /* P4  = -5.703738387728891173354e-04 */
	.quad	0x3F2DB2889E32FBFD /* P5  = +2.265731592156760387344e-04 */
	.quad	0xBF1385FBD54C5A55 /* P6  = -7.447576110695385196414e-05 */
	.quad	0x3EF5AFA812C6984E /* P7  = +2.068153223579892541184e-05 */
	.quad	0xBED47097C188A03C /* P8  = -4.873231795467276043290e-06 */
	.quad	0x3EAFF2B982F7EE8C /* P9  = +9.521288628073486288914e-07 */
	.quad	0xBE828EC5B57D424D /* P10 = -1.382656715739529384702e-07 */
	.quad	0xC00F000000000000 /* B = -3.875 */
	.quad	0x3FF0000000000000 /* A = +1 */
	.quad	0x0000000000000000 /* Align value = +0 */
	.quad	0x0000000000000000 /* Align value = +0 */
	.quad	0x3C9BA40DA6983BEC /* PL0 = +9.589840482158163453169e-17 */
	.quad	0x3FEFFCAAC3F20E65 /* PH0 = +9.995931460438894911036e-01 */
	.quad	0x3F4AA87CF664754C /* P1  = +8.135423820793490331956e-04 */
	.quad	0xBF4AA5B62919E224 /* P2  = -8.132113891426467676310e-04 */
	.quad	0x3F41C01B53B0B312 /* P3  = +5.416997368051531710388e-04 */
	.quad	0xBF31B8B54D091751 /* P4  = -2.704088811110632606347e-04 */
	.quad	0x3F1C431305954ECC /* P5  = +1.078110084525254933728e-04 */
	.quad	0xBF02B7DEAD0D44E6 /* P6  = -3.570221236393906131126e-05 */
	.quad	0x3EE51C6EFF109EA9 /* P7  = +1.006654199116272154479e-05 */
	.quad	0xBEC48CFB08072D17 /* P8  = -2.449834994621594976610e-06 */
	.quad	0x3EA1585EC59CAE34 /* P9  = +5.169271261920604503617e-07 */
	.quad	0xBE78832BAF950BA9 /* P10 = -9.131575131209528255629e-08 */
	.quad	0xC011000000000000 /* B = -4.25 */
	.quad	0x3FF0000000000000 /* A = +1 */
	.quad	0x0000000000000000 /* Align value = +0 */
	.quad	0x0000000000000000 /* Align value = +0 */
	.quad	0x3C8FBF237F4AFE10 /* PL0 = +5.507163370275307643966e-17 */
	.quad	0x3FEFFEC61279A3A4 /* PH0 = +9.998503075449787225182e-01 */
	.quad	0x3F339E78281A00EA /* P1  = +2.993625022114214863645e-04 */
	.quad	0xBF339DB7B072AD62 /* P2  = -2.993176899035080028902e-04 */
	.quad	0x3F2A259E658EF4E4 /* P3  = +1.994853835451177669594e-04 */
	.quad	0xBF1A219C312B10BA /* P4  = -9.968295880030927192162e-05 */
	.quad	0x3F04E146B4F5F4B7 /* P5  = +3.982541113154699160876e-05 */
	.quad	0xBEEBC5F137088210 /* P6  = -1.324329943580649487333e-05 */
	.quad	0x3ECF96736E300B00 /* P7  = +3.765547135882256916132e-06 */
	.quad	0xBEAF4874840B91EB /* P8  = -9.323068824421825762292e-07 */
	.quad	0x3E8B6AB2B5C8FD3F /* P9  = +2.042709991312793245971e-07 */
	.quad	0xBE650BCCE62FD2B7 /* P10 = -3.920140725219944650830e-08 */
	.quad	0xC013000000000000 /* B = -4.75 */
	.quad	0x3FF0000000000000 /* A = +1 */
	.quad	0x0000000000000000 /* Align value = +0 */
	.quad	0x0000000000000000 /* Align value = +0 */
	.quad	0x3C9C869C85471703 /* PL0 = +9.896883942603146946483e-17 */
	.quad	0x3FEFFF8C81C6DC33 /* PH0 = +9.999449286177707341139e-01 */
	.quad	0x3F1CDF5A2E4D7C69 /* P1  = +1.101397316012206760643e-04 */
	.quad	0xBF1CDEF1F9BE63BE /* P2  = -1.101336660539594564027e-04 */
	.quad	0x3F133EC10C83AAA0 /* P3  = +7.341435696487731017506e-05 */
	.quad	0xBF033DAB325FAACB /* P4  = -3.669909192168459445238e-05 */
	.quad	0x3EEEC598FA98BAD8 /* P5  = +1.467316890843338172161e-05 */
	.quad	0xBED47F1A15BA368E /* P6  = -4.886744445221253126882e-06 */
	.quad	0x3EB761FBE7D201C1 /* P7  = +1.393720509029845064726e-06 */
	.quad	0xBE974CD75A43BF6B /* P8  = -3.471994551992448536007e-07 */
	.quad	0x3E74B02965BBF8DC /* P9  = +7.706929621914905669946e-08 */
	.quad	0xBE504EF4E3892A66 /* P10 = -1.518840362012570189110e-08 */
	.quad	0xC015000000000000 /* B = -5.25 */
	.quad	0x3FF0000000000000 /* A = +1 */
	.quad	0x0000000000000000 /* Align value = +0 */
	.quad	0x0000000000000000 /* Align value = +0 */
	.quad	0x3C643810400471B0 /* PL0 = +8.768592603904887599187e-18 */
	.quad	0x3FEFFFD583014825 /* PH0 = +9.999797400180382433987e-01 */
	.quad	0x3F053E71416C43CA /* P1  = +4.051955345663706869871e-05 */
	.quad	0xBF053E550C7C8CC9 /* P2  = -4.051873253121394012080e-05 */
	.quad	0x3EFC52D0D90D4843 /* P3  = +2.701139380018752534477e-05 */
	.quad	0xBEEC523A6ADBE142 /* P4  = -1.350460237457883558350e-05 */
	.quad	0x3ED6A73E22D844B3 /* P5  = +5.400965660055565196396e-06 */
	.quad	0xBEBE31D10F23ACD0 /* P6  = -1.799738182979224868919e-06 */
	.quad	0x3EA13E14264DEAB2 /* P7  = +5.138663935333241981438e-07 */
	.quad	0xBE81385ABB98EDCC /* P8  = -1.282999997786486835638e-07 */
	.quad	0x3E5EB9164593E0B6 /* P9  = +2.861301981891537161158e-08 */
	.quad	0xBE387218CFE7772E /* P10 = -5.691705994073124478195e-09 */
	.quad	0xC017000000000000 /* B = -5.75 */
	.quad	0x3FF0000000000000 /* A = +1 */
	.quad	0x0000000000000000 /* Align value = +0 */
	.quad	0x0000000000000000 /* Align value = +0 */
	.quad	0x3C92530433F4C703 /* PL0 = +6.357512739163799046861e-17 */
	.quad	0x3FEFFFF05E8D3191 /* PH0 = +9.999925467214315633058e-01 */
	.quad	0x3EEF42DDFA52B575 /* P1  = +1.490650158538873335176e-05 */
	.quad	0xBEEF42CEB54212AA /* P2  = -1.490639048307961378200e-05 */
	.quad	0x3EE4D7201CBCB853 /* P3  = +9.937445518550804010127e-06 */
	.quad	0xBED4D6F764B66C37 /* P4  = -4.968574624976280456686e-06 */
	.quad	0x3EC0ABB806EBDE71 /* P5  = +1.987311456171617620608e-06 */
	.quad	0xBEA6399CF854F876 /* P6  = -6.623581475862682369330e-07 */
	.quad	0x3E8964B91728D7C9 /* P7  = +1.891959403186505598965e-07 */
	.quad	0xBE6961A0528444D6 /* P8  = -4.727645325404986954168e-08 */
	.quad	0x3E46AE3B0814EE00 /* P9  = +1.056147192151514779549e-08 */
	.quad	0xBE221B8194DACD16 /* P10 = -2.107984154277957626641e-09 */
	.quad	0xC019000000000000 /* B = -6.25 */
	.quad	0x3FF0000000000000 /* A = +1 */
	.quad	0x0000000000000000 /* Align value = +0 */
	.quad	0x0000000000000000 /* Align value = +0 */
	.quad	0x3C7BB5622CE1A79E /* PL0 = +2.403331811901679167526e-17 */
	.quad	0x3FEFFFFA3FF22708 /* PH0 = +9.999972580855862602789e-01 */
	.quad	0x3ED7003552D53503 /* P1  = +5.483821309338170039906e-06 */
	.quad	0xBED7003130C1AB92 /* P2  = -5.483806273169366545037e-06 */
	.quad	0x3ECEAAE13B699C45 /* P3  = +3.655850800133043324271e-06 */
	.quad	0xBEBEAACB305F3D07 /* P4  = -1.827905351959291114416e-06 */
	.quad	0x3EA8887F5F9C87EF /* P5  = +7.311461438267648556646e-07 */
	.quad	0xBE905AD08DF8454F /* P6  = -2.437046884027860662692e-07 */
	.quad	0x3E72B068300B703F /* P7  = +6.962228483613086736676e-08 */
	.quad	0xBE52AF921A71C058 /* P8  = -1.740252888706390465423e-08 */
	.quad	0x3E30B53EAA35300D /* P9  = +3.890131469838137725119e-09 */
	.quad	0xBE0AB60CDAD7E22E /* P10 = -7.773963050435300060566e-10 */
	.quad	0xC01B000000000000 /* B = -6.75 */
	.quad	0x3FF0000000000000 /* A = +1 */
	.quad	0x0000000000000000 /* Align value = +0 */
	.quad	0x0000000000000000 /* Align value = +0 */
	.quad	0x3C8BD1ACF80D7256 /* PL0 = +4.825835138930451121169e-17 */
	.quad	0x3FEFFFFDE2760A41 /* PH0 = +9.999989913051835488389e-01 */
	.quad	0x3EC0EC4F1EC27E55 /* P1  = +2.017388615341105998718e-06 */
	.quad	0xBEC0EC4E005E6EAC /* P2  = -2.017386580411626200507e-06 */
	.quad	0x3EB6906504BC4610 /* P3  = +1.344921673533307001969e-06 */
	.quad	0xBEA6905F0D52C8B5 /* P4  = -6.724581235377781360384e-07 */
	.quad	0x3E920D0F5CCE152B /* P5  = +2.689810941136721216499e-07 */
	.quad	0xBE7811505B10E753 /* P6  = -8.965891741619763761543e-08 */
	.quad	0x3E5B811EE4F9B8EE /* P7  = +2.561544781706659619288e-08 */
	.quad	0xBE3B80ABC067E840 /* P8  = -6.403452884688571158579e-09 */
	.quad	0x3E1898E394E09335 /* P9  = +1.431746793613569087489e-09 */
	.quad	0xBDF3ABB5BA711DB7 /* P10 = -2.862469657501951918569e-10 */
	.quad	0xC01D000000000000 /* B = -7.25 */
	.quad	0x3FF0000000000000 /* A = +1 */
	.quad	0x0000000000000000 /* Align value = +0 */
	.quad	0x0000000000000000 /* Align value = +0 */
	.quad	0x3C8AE01DB39A3791 /* PL0 = +4.662147961093911873193e-17 */
	.quad	0x3FEFFFFF38C76668 /* PH0 = +9.999996289217962797125e-01 */
	.quad	0x3EA8E712E56E1188 /* P1  = +7.421562696484951529573e-07 */
	.quad	0xBEA8E7124A650791 /* P2  = -7.421559942504648535596e-07 */
	.quad	0x3EA09A0B62D8EF94 /* P3  = +4.947702955735978541097e-07 */
	.quad	0xBE909A09C56C2107 /* P4  = -2.473847805916120382218e-07 */
	.quad	0x3E7A900A90A54A6E /* P5  = +9.895362410487317236618e-08 */
	.quad	0xBE61B5557BB449B6 /* P6  = -3.298434544432568302770e-08 */
	.quad	0x3E443CC74732CDCA /* P7  = +9.423781066565733462466e-09 */
	.quad	0xBE243CA8AA8D6E54 /* P8  = -2.355890888986360997159e-09 */
	.quad	0x3E0219C341E0D1B4 /* P9  = +5.267978308406275552691e-10 */
	.quad	0xBDDCF49A10950F13 /* P10 = -1.053394074620716018815e-10 */
	.quad	0xC01F000000000000 /* B = -7.75 */
	.quad	0x3FF0000000000000 /* A = +1 */
	.quad	0x0000000000000000 /* Align value = +0 */
	.quad	0x0000000000000000 /* Align value = +0 */
	.quad	0x3C75CB18F3775414 /* PL0 = +1.890271747518592444083e-17 */
	.quad	0x3FEFFFFFD38C39F0 /* PH0 = +9.999999172012490333827e-01 */
	.quad	0x3E8639E2F89493BB /* P1  = +1.655974950855472979393e-07 */
	.quad	0xBE8639E2D9B29562 /* P2  = -1.655974813708346974914e-07 */
	.quad	0x3E7DA2836A1F706E /* P3  = +1.103982989742589616541e-07 */
	.quad	0xBE6DA282C6733DAE /* P4  = -5.519913131581509871840e-08 */
	.quad	0x3E57B53A278851FD /* P5  = +2.207971980430773309147e-08 */
	.quad	0xBE3F9C4A72536E22 /* P6  = -7.359895614149337484810e-09 */
	.quad	0x3E220E81FBE19CDD /* P7  = +2.102073153607135257714e-09 */
	.quad	0xBE020E8875ADA8D8 /* P8  = -5.255211642212584097407e-10 */
	.quad	0x3DE07634328384FC /* P9  = +1.197748786062966341989e-10 */
	.quad	0xBDBA54078E3C351F /* P10 = -2.394539505021488953905e-11 */
	.quad	0xC021000000000000 /* B = -8.5 */
	.quad	0x3FF0000000000000 /* A = +1 */
	.quad	0x0000000000000000 /* Align value = +0 */
	.quad	0x0000000000000000 /* Align value = +0 */
	.quad	0x3C98B78738B0EDEF /* PL0 = +8.575399788039081964921e-17 */
	.quad	0x3FEFFFFFF9FBEA40 /* PH0 = +9.999999887944071019774e-01 */
	.quad	0x3E581056FAC28C46 /* P1  = +2.241118550516412682327e-08 */
	.quad	0xBE581056F63A4351 /* P2  = -2.241118525356742542550e-08 */
	.quad	0x3E500AE49533790A /* P3  = +1.494078933911655875521e-08 */
	.quad	0xBE400AE489ACBA90 /* P4  = -7.470394349637968945652e-09 */
	.quad	0x3E29AB0D59A1967B /* P5  = +2.988168557255271725494e-09 */
	.quad	0xBE111CB32D6EEF2B /* P6  = -9.960558400070350772418e-10 */
	.quad	0x3DF38CBADF396908 /* P7  = +2.844859618921805216353e-10 */
	.quad	0xBDD38CC7B92CECD3 /* P8  = -7.112220386749926320915e-11 */
	.quad	0x3DB1D2BBE2705032 /* P9  = +1.621008722427575444686e-11 */
	.quad	0xBD8C8199294E6380 /* P10 = -3.240784656869469020111e-12 */
	.quad	0xC023000000000000 /* B = -9.5 */
	.quad	0x3FF0000000000000 /* A = +1 */
	.quad	0x0000000000000000 /* Align value = +0 */
	.quad	0x0000000000000000 /* Align value = +0 */
	.quad	0x3C8EEEC16618B984 /* PL0 = +5.365957423487855307906e-17 */
	.quad	0x3FEFFFFFFF2F9279 /* PH0 = +9.999999984834878619111e-01 */
	.quad	0x3E2A0DB0D052B148 /* P1  = +3.033024167396880687734e-09 */
	.quad	0xBE2A0DB0CFA6AB71 /* P2  = -3.033024162734192808028e-09 */
	.quad	0x3E215E75D53A3105 /* P3  = +2.022016035353114070618e-09 */
	.quad	0xBE115E75D40AA47F /* P4  = -1.011008013562702155050e-09 */
	.quad	0x3DFBCA5CDC12ED1C /* P5  = +4.044047007631481841556e-10 */
	.quad	0xBDE286E85704FC22 /* P6  = -1.348015410318274576187e-10 */
	.quad	0x3DC52A8925354517 /* P7  = +3.850101197145027796396e-11 */
	.quad	0xBDA52A97EA3F5F4A /* P8  = -9.625355478142550638468e-12 */
	.quad	0x3D834C011A2AC0F7 /* P9  = +2.193802608697321032841e-12 */
	.quad	0xBD5EDD05BDCB3A62 /* P10 = -4.385948508419928563300e-13 */
	.quad	0xC025000000000000 /* B = -10.5 */
	.quad	0x3FF0000000000000 /* A = +1 */
	.quad	0x0000000000000000 /* Align value = +0 */
	.quad	0x0000000000000000 /* Align value = +0 */
	.quad	0x3C6BD8B474BBF792 /* PL0 = +1.207649585364892639612e-17 */
	.quad	0x3FEFFFFFFFE3CAD8 /* PH0 = +9.999999997947623953110e-01 */
	.quad	0x3DFC3527E43C565F /* P1  = +4.104751852963940338559e-10 */
	.quad	0xBDFC3527E420F415 /* P2  = -4.104751852036136216697e-10 */
	.quad	0x3DF2CE1A8D806DAD /* P3  = +2.736501142887952919489e-10 */
	.quad	0xBDE2CE1A8DDF690A /* P4  = -1.368250573053032426141e-10 */
	.quad	0x3DCE169832D8BD68 /* P5  = +5.473022586854025789680e-11 */
	.quad	0xBDB40F0FE853DA5B /* P6  = -1.824340550195944358477e-11 */
	.quad	0x3D96EA8D930D31A1 /* P7  = +5.210545794901128943676e-12 */
	.quad	0xBD76EA9DB0D09839 /* P8  = -1.302650427355019556441e-12 */
	.quad	0x3D54E474FD4303A1 /* P9  = +2.968990047962355000258e-13 */
	.quad	0xBD30B526CA2B228A /* P10 = -5.935740124899435401321e-14 */
	.quad	0xC027000000000000 /* B = -11.5 */
	.quad	0x3FF0000000000000 /* A = +1 */
	.quad	0x0000000000000000 /* Align value = +0 */
	.quad	0x0000000000000000 /* Align value = +0 */
	.quad	0x3C56E8953D525FD5 /* PL0 = +4.967494994909661698725e-18 */
	.quad	0x3FEFFFFFFFFC2EB9 /* PH0 = +9.999999999722241073030e-01 */
	.quad	0x3DCE8A37A48016C2 /* P1  = +5.555177547354687971427e-11 */
	.quad	0xBDCE8A37A479B7D4 /* P2  = -5.555177547084873157964e-11 */
	.quad	0x3DC45C250CFA9C16 /* P3  = +3.703451575129414499553e-11 */
	.quad	0xBDB45C250D9F8467 /* P4  = -1.851725791056759260154e-11 */
	.quad	0x3DA049BB33CBD4E9 /* P5  = +7.406930640558963265190e-12 */
	.quad	0xBD85B7A407C422C1 /* P6  = -2.468976464832073512208e-12 */
	.quad	0x3D68CF9CED2B3FD5 /* P7  = +7.051706989348171774536e-13 */
	.quad	0xBD48CFAE64C352B3 /* P8  = -1.762945685274427023683e-13 */
	.quad	0x3D269EAE08690D52 /* P9  = +4.018091287355461204663e-14 */
	.quad	0xBD0216CBEAFFF5AA /* P10 = -8.033151495672990022322e-15 */
	.quad	0xC029000000000000 /* B = -12.5 */
	.quad	0x3FF0000000000000 /* A = +1 */
	.quad	0x0000000000000000 /* Align value = +0 */
	.quad	0x0000000000000000 /* Align value = +0 */
	.quad	0x3C8ACF1392B106D3 /* PL0 = +4.650601502940921454330e-17 */
	.quad	0x3FEFFFFFFFFF7BBD /* PH0 = +9.999999999962408958609e-01 */
	.quad	0x3DA088529889B316 /* P1  = +7.518115268189742464885e-12 */
	.quad	0xBDA088529887F4C4 /* P2  = -7.518115268005149164680e-12 */
	.quad	0x3D960B18BF1DF711 /* P3  = +5.012076679213679703380e-12 */
	.quad	0xBD860B18BFD99A48 /* P4  = -2.506038344573564868987e-12 */
	.quad	0x3D71A27E7CA64143 /* P5  = +1.002419056539285288454e-12 */
	.quad	0xBD5783530EA76D91 /* P6  = -3.341396294294381580191e-13 */
	.quad	0x3D3ADCC75CBD2A03 /* P7  = +9.543447641637910477850e-14 */
	.quad	0xBD1ADCDA46BE5F17 /* P8  = -2.385887543769010971872e-14 */
	.quad	0x3CF87D77650BE5B8 /* P9  = +5.437895260471143131391e-15 */
	.quad	0xBCD395AE6E74C6D2 /* P10 = -1.087168847335561258239e-15 */
	.quad	0xC02B000000000000 /* B = -13.5 */
	.quad	0x3FF0000000000000 /* A = +1 */
	.quad	0x0000000000000000 /* Align value = +0 */
	.quad	0x0000000000000000 /* Align value = +0 */
	.quad	0x3C97A8A295292858 /* PL0 = +8.208271151146829171896e-17 */
	.quad	0x3FEFFFFFFFFFEE19 /* PH0 = +9.999999999994911847878e-01 */
	.quad	0x3D71E642BB008F95 /* P1  = +1.017466259229268282255e-12 */
	.quad	0xBD71E642BAFEEC54 /* P2  = -1.017466259207593392022e-12 */
	.quad	0x3D67DDAE41647741 /* P3  = +6.783108169938233581038e-13 */
	.quad	0xBD57DDAE4230F34B /* P4  = -3.391554091734942426856e-13 */
	.quad	0x3D4317C33FAE2536 /* P5  = +1.356626669455791324801e-13 */
	.quad	0xBD2975040D3E26B9 /* P6  = -4.522088139411435138867e-14 */
	.quad	0x3D0D155DCD0F0AFB /* P7  = +1.291565189902030307333e-14 */
	.quad	0xBCED157247832B20 /* P8  = -3.228947666403019234175e-15 */
	.quad	0x3CCA83D70F607C28 /* P9  = +7.359390959466796619024e-16 */
	.quad	0xBCA5343952C1E19E /* P10 = -1.471323041436694087188e-16 */
	.quad	0xC02D000000000000 /* B = -14.5 */
	.quad	0x3FF0000000000000 /* A = +1 */
	.quad	0x0000000000000000 /* Align value = +0 */
	.quad	0x0000000000000000 /* Align value = +0 */
	.quad	0x3C9B7876CBC5306E /* PL0 = +9.530765996816607711732e-17 */
	.quad	0x3FEFFFFFFFFFFD93 /* PH0 = +9.999999999999310551502e-01 */
	.quad	0x3D436121E2640D76 /* P1  = +1.376990843765503869546e-13 */
	.quad	0xBD436121E26250EA /* P2  = -1.376990843736775811281e-13 */
	.quad	0x3D39D6D7CA259186 /* P3  = +9.179938654047876451320e-14 */
	.quad	0xBD29D6D7CB0327CE /* P4  = -4.589969336188563660531e-14 */
	.quad	0x3D14ABE4DC31244A /* P5  = +1.835994545584345768382e-14 */
	.quad	0xBCFB8FDB82AB6BB7 /* P6  = -6.119980791767901275443e-15 */
	.quad	0x3CDF7CF757491B60 /* P7  = +1.747943407988343076526e-15 */
	.quad	0xBCBF7D0D833640FB /* P8  = -4.369905470133249448357e-16 */
	.quad	0x3C9CB512F6BDC754 /* P9  = +9.959852600692493655511e-17 */
	.quad	0xBC76F50AB1B0E9BA /* P10 = -1.991219205936492089091e-17 */
	.quad	0xC02F000000000000 /* B = -15.5 */
	.quad	0x3FF0000000000000 /* A = +1 */
	.quad	0x0000000000000000 /* Align value = +0 */
	.quad	0x0000000000000000 /* Align value = +0 */
	.quad	0x3C6FFE15D5F78543 /* PL0 = +1.387454417328248962819e-17 */
	.quad	0x3FEFFFFFFFFFFFE1 /* PH0 = +9.999999999999965583086e-01 */
	.quad	0x3CFEE00288B99C26 /* P1  = +6.855635762864742358597e-15 */
	.quad	0xBCFEE0027D060EE2 /* P2  = -6.855635607998342735403e-15 */
	.quad	0x3CF4954AA23148A2 /* P3  = +4.570381865813341696777e-15 */
	.quad	0xBCE4954B5DAD3010 /* P4  = -2.285192173571711474199e-15 */
	.quad	0x3CD07883DD8793BD /* P5  = +9.143109661358222028007e-16 */
	.quad	0xBCB5F5F4BB87ADCF /* P6  = -3.047668447080103869032e-16 */
	.quad	0x3C98F1A905097685 /* P7  = +8.654183371862458774513e-17 */
	.quad	0xBC78F2D585007222 /* P8  = -2.163943551222030413627e-17 */
	.quad	0x3C58A37CC5082B5F /* P9  = +5.342649626494471588064e-18 */
	.quad	0xBC33AE7917F94D17 /* P10 = -1.066938163384541013918e-18 */
	.quad	0xC031000000000000 /* B = -17 */
	.quad	0x3FF0000000000000 /* A = +1 */
	.quad	0x0000000000000000 /* Align value = +0 */
	.quad	0x0000000000000000 /* Align value = +0 */
	.quad	0x3C91BF1D80474F0F /* PL0 = +6.157069264461989135096e-17 */
	.quad	0x3FEFFFFFFFFFFFFE /* PH0 = +9.999999999999997779554e-01 */
	.quad	0x3CB72071400E6275 /* P1  = +3.209478247225075961360e-16 */
	.quad	0xBCB72071400A9F37 /* P2  = -3.209478247103497434502e-16 */
	.quad	0x3CAED5EC39A77629 /* P3  = +2.139652050028423711308e-16 */
	.quad	0xBC9ED5EC3B530600 /* P4  = -1.069826028468029104719e-16 */
	.quad	0x3C88AB2BFED159DE /* P5  = +4.279326904335078988705e-17 */
	.quad	0xBC70721D1220B3FC /* P6  = -1.426441958074916244382e-17 */
	.quad	0x3C52C96049721FB8 /* P7  = +4.073700029965821523731e-18 */
	.quad	0xBC32C971215735DC /* P8  = -1.018438939975201710113e-18 */
	.quad	0x3C112EF658AB41A9 /* P9  = +2.328791246104218830028e-19 */
	.quad	0xBBEB7B598C6AD3DE /* P10 = -4.655603964908654142787e-20 */
	.quad	0xC03287E0C98F84E5 /* B = -18.530774 */
	.quad	0x3FF0000000000000 /* A = +1 */
	.quad	0x0000000000000000 /* Align value = +0 */
	.quad	0x0000000000000000 /* Align value = +0 */
	.quad	0x0000000000000000 /* PL0 = +0.000000000000000000000e-01 */
	.quad	0x3FF0000000000000 /* PH0 = +1.000000000000000000000e+00 */
	.quad	0x0000000000000000 /* P1  = +0.000000000000000000000e-01 */
	.quad	0x0000000000000000 /* P2  = +0.000000000000000000000e-01 */
	.quad	0x0000000000000000 /* P3  = +0.000000000000000000000e-01 */
	.quad	0x0000000000000000 /* P4  = +0.000000000000000000000e-01 */
	.quad	0x0000000000000000 /* P5  = +0.000000000000000000000e-01 */
	.quad	0x0000000000000000 /* P6  = +0.000000000000000000000e-01 */
	.quad	0x0000000000000000 /* P7  = +0.000000000000000000000e-01 */
	.quad	0x0000000000000000 /* P8  = +0.000000000000000000000e-01 */
	.quad	0x0000000000000000 /* P9  = +0.000000000000000000000e-01 */
	.quad	0x0000000000000000 /* P10 = +0.000000000000000000000e-01 */
	.quad	0x0000000000000000 /* B = +0 */
	.quad	0x0000000000000000 /* A = +0 */
	.quad	0x0000000000000000 /* Align value = +0 */
	.quad	0x0000000000000000 /* Align value = +0 */
	.align	32
	.quad	0x8000000000000000, 0x8000000000000000, 0x8000000000000000, 0x8000000000000000 /* _dbSignMask */
	.align	32
	.quad	0x7fffffffffffffff, 0x7fffffffffffffff, 0x7fffffffffffffff, 0x7fffffffffffffff /* _dbAbsMask */
	.align	32
	.long	0x7ffe0000, 0x7ffe0000, 0x7ffe0000, 0x7ffe0000, 0x7ffe0000, 0x7ffe0000, 0x7ffe0000, 0x7ffe0000 /* _iExpMantMask */
	.align	32
	.long	0x7fe00000, 0x7fe00000, 0x7fe00000, 0x7fe00000, 0x7fe00000, 0x7fe00000, 0x7fe00000, 0x7fe00000 /* _iExpMask */
	.align	32
	.long	0x3fbe0000, 0x3fbe0000, 0x3fbe0000, 0x3fbe0000, 0x3fbe0000, 0x3fbe0000, 0x3fbe0000, 0x3fbe0000 /* _iMinIdxOfsMask */
	.align	32
	.long	0x00760000, 0x00760000, 0x00760000, 0x00760000, 0x00760000, 0x00760000, 0x00760000, 0x00760000 /* _iMaxIdxMask */
	.align	32
	.type	__svml_dtanh_data_internal, @object
	.size	__svml_dtanh_data_internal, .-__svml_dtanh_data_internal
