require File.dirname(__FILE__) + '/../test_helper'

class UserTest < ActiveSupport::TestCase
  fixtures :all

  should_have_many :posts
  should_have_many :dogs
  should_have_many :cats

  should_have_many :friendships
  should_have_many :friends

  should_have_one :address
  should_have_one :address, :dependent => :destroy

  should_have_db_indices :email, :name
  should_have_index :age
  should_have_db_index [:email, :name], :unique => true
  should_have_db_index :age, :unique => false

  should_fail do
    should_have_db_index :phone
    should_have_db_index :email, :unique => false
    should_have_db_index :age, :unique => true
  end

  should_have_named_scope :old,       :conditions => "age > 50"
  should_have_named_scope :eighteen,  :conditions => { :age => 18 }

  should_have_named_scope 'recent(5)',            :limit => 5
  should_have_named_scope 'recent(1)',            :limit => 1
  should_have_named_scope 'recent_via_method(7)', :limit => 7

  context "when given an instance variable" do
    setup { @count = 2 }
    should_have_named_scope 'recent(@count)', :limit => 2
  end

  should_not_allow_values_for :email, "blah", "b lah"
  should_allow_values_for :email, "a@b.com", "asdf@asdf.com"
  should_allow_values_for :age, 1, 10, 99
  should_not_allow_values_for :age, "a", "-"
  should_not_allow_values_for :ssn, "a", 1234567890
  should_ensure_length_in_range :email, 1..100
  should_ensure_value_in_range :age, 1..100, :low_message  => /greater/,
                                             :high_message => /less/
  should_fail do
    should_ensure_value_in_range :age, 1..100, :low_message  => /more/,
                                               :high_message => /less/
  end
  should_fail do
    should_ensure_value_in_range :age, 1..100, :low_message  => /greater/,
                                               :high_message => /fewer/
  end
  should_not_allow_mass_assignment_of :password
  should_have_class_methods :find, :destroy
  should_have_instance_methods :email, :age, :email=, :valid?
  should_have_db_columns :name, :email, :age
  should_have_db_column :id,    :type => "integer"
  should_have_db_column :email, :type => "string", :default => nil, :precision => nil, :limit    => 255,
                                :null => true,     :scale   => nil
  should_validate_acceptance_of :eula
  should_validate_uniqueness_of :email, :scoped_to => :name, :case_sensitive => false

  should_ensure_length_is :ssn, 9, :message => "Social Security Number is not the right length"
  should_validate_numericality_of :ssn

  should_have_readonly_attributes :name

  should_fail do
    should_not_allow_mass_assignment_of :name, :age
  end

  should_have_one :profile, :through => :registration

  should_fail do
    should_have_one :profile, :through => :interview
    should_have_one :address, :through => :registration
  end
end
