# pgConnection wrapper object to libtclpq's PGconn

# Construction opens a new database connection 
# The passed argument string can be empty to use all default parameters,
# or it can contain one or more parameter settings separated by
# whitespace.
# Each parameter setting is in the form keyword = value.
# (To write a null value or a value containing spaces, surround it
# with single quotes, eg, keyword = 'a value'. Single quotes within
# the value must be written as \'. Spaces around the equal sign 
# are optional.)
#
# The currently recognized parameter keywords are: 
#
# host -- host to connect to. If a non-0-length string is specified,
#   TCP/IP communication is used.
#   Without a host name, libpq will connect using a local Unix domain
#   socket.
#
# port -- port number to connect to at the server host,
#   or socket filename extension for Unix-domain connections.
# dbname -- database name.
# user -- user name for authentication.
# password -- password used if the backend demands password
#   authentication.
#
# connectdb uses environment variables or built-in default values for
# unspecified options.
package require Itcl
#package require tclpq

class pgConnection {

    constructor {conninfo} {}	;# open database connection    
    destructor {}		;# close database connection

    # query interface
    public method exec {query}		;# Submit a query, return the result.
    public method transaction {args}	;# evaluate commands in a transaction
    public method reset {}		;# Reset comms with the backend. 
    public method apply {query var args};# apply functional args to query tuples

    # asynchronous query interface
    public method rexec {query var}	;# Asynchronous query, result(s) to var
    public method mrexec {query var}	;# Asynchronous query, result(s) to var
    public method aexec {query callback args}	;# Asynch query with callback
    public method cancel {}		;# Request cancellation of query

    # pgConnection accessors 
    public method db {}		;# the database name of the connection. 
    public method user {}	;# the user name of the connection. 
    public method password {}	;# the password of the connection. 
    public method host {}	;# the server host name of the connection. 
    public method port {}	;# the port of the connection. 
    public method tty {}	;# the debug tty of the connection. 
    public method options {}	;# the backend options used in the connection. 
    public method status {}	;# status - the status of the connection.
    public method errorMessage {};# the error message most recently generated

    # methods accessing low level structures and functions
    protected method notifies {}	;# process pending notifies
    protected method connection {}	;# postgres connection for this object
    protected method backendPID {}	;# process ID connection server
    protected method sendQuery {query}	;# Submit a query asynchronously
    protected method getResult {}	;# Wait for result from PQsendQuery
    protected method consumeInput {}	;# consume any available input 
    protected method isBusy {}		;# TRUE if a query is busy
    protected method socket {}		;# tcl fd for connection socket.

    # private methods implementing asynchronous queries
    private variable temp	;# temporary result for rir
    private variable execr	;# temporary result for rir
    private proc rir {that temp var}		;# interrupt routine
    private proc mrir {that var}		;# interrupt routine
    private proc ir {that callback args}	;# interrupt routine

    private variable connection	;# postgres connection for this object
}

# open a new database connection using parameters taken from a string.
body pgConnection::constructor {conninfo} {
    set connection [::tclpq::PQconnectdb $conninfo]
    if {[status]} {
        error [errorMessage]
    }
}
    
# Close the connection to the backend.
body pgConnection::destructor {} {
    ::tclpq::PQfinish $connection
}

# exec - Submit a query to Postgres and wait for the result.
# Returns a PGresult object or possibly a NULL pointer.
# A non-NULL pointer will generally be returned except in out-of-memory
# conditions or serious errors such as inability to send the query to
# the backend.
# If a NULL is returned, it should be treated like a PGRES_FATAL_ERROR
# result. Use errorMessage to get more information about the error.
body pgConnection::exec {query} {
    #puts stderr "$this pgConnection::exec $query"
    set socket [socket]
    if {[fileevent $socket readable] != ""} {
        error "$this exec: connection busy. [errorMessage]"
    }

    catch {unset temp}
    catch {unset ${this}execr}
    fileevent $socket readable \
        [list [code rir] $this [scope temp] ${this}execr]

    if {![sendQuery $query]} {
        error "$this exec: [errorMessage]"
    }

    vwait ${this}execr

    if {[llength [set ${this}execr]]} {
        set result [lindex [set ${this}execr] 0]
        unset ${this}execr
        #puts "transformed result: $result"
        return [uplevel namespace origin $result]
    } else {
        unset execr
        error "$this: [errorMessage]"
    }

    if {[fileevent [socket] readable] != ""} {
        error "$this rexec: connection busy. [errorMessage]"
    }
    
    set result [::tclpq::PQexec $connection $query]
    #puts stderr "$this pgConnection::exec result $result"
    $this notifies	;# process notifies
    if {$result != "NULL"} {
        set result [uplevel pgResult \#auto $result]
        #puts "transformed result: $result"
        return [uplevel namespace origin $result]
    } else {
        error "$this: [errorMessage]"
    }
}

# getResult Wait for the next result from a prior PQsendQuery,
# and return it. NULL is returned when the query is complete and
# there will be no more results. 
#
# getResult must be called repeatedly until it returns NULL,
# indicating that the query is done. (If called when no query is active,
# getResult will just return NULL at once.)
# Note that PQgetResult will block only if a query is active and the 
# necessary response data has not yet been read by consumeInput.
body pgConnection::getResult {} {
    set result [::tclpq::PQgetResult $connection]
    if {$result != "NULL"} {
        return [namespace origin [uplevel [pgResult \#auto $result]]]
    } else {
        return ""
    }
}

# consumeInput If input is available from the backend, consume it. 
#
# consumeInput normally returns 1 indicating "no error",
# but returns 0 if there was some kind of trouble
# (in which case errorMessage is set).
#
# Note that the result does not say whether any input data was
# actually collected. After calling consumeInput, the application may 
# check isBusy and/or notifies to see if their state has changed.
# consumeInput may be called even if the application is not prepared
# to deal with a result or notification just yet.
# The routine will read available data and save it in a buffer,
# thereby causing a select(2) read-ready indication to go away.
# The application can thus use consumeInput to clear the select
# condition immediately, and then examine the results at leisure.
body pgConnection::consumeInput {} {
    set result [::tclpq::PQconsumeInput $connection]
    $this notifies	;# process notifies
    return $result
}

# interrupt routine
# called when connection is readable to collect input
# lappends results to the passed variable as they become available
#
# that: dbconnection
# temp: a fully qualified variable into which to accumulate results
# var: a fully qualified variable to store collected results
body pgConnection::rir {that temp var} {
    if {[$that consumeInput]} {
        # collect all the results we can
        while {![$that isBusy]} {
            set result [::tclpq::PQgetResult [$that connection]]
            if {$result != "NULL"} {
                # make a result object in the callback's scope
                lappend $temp \
                    [pgResult [namespace qualifiers $var]::\#auto $result]
            } else {
                # error consuming input or query's complete
                fileevent [$that socket] readable ""
                if {[catch {
                    set $var [set $temp]
                    unset $temp
                }]} {
                    lappend $var {}
                }
                return
            }
        }
    } else {
        # error consuming input or query's complete
        fileevent [$that socket] readable ""
        catch {
            set $var $tmp
            unset tmp
        }
        lappend $var {}
    }
}

# interrupt routine for multiple results delivered asynchronously
# called when connection is readable to collect input
# lappends results to the passed variable as they become available
# End of results is signalled by a {} appended to the variable
# that: dbconnection
# callback: a process to call with collected results
# args: arguments to the callback proc
body pgConnection::mrir {that var} {
    upvar \#0 $var resvar
    if {[$that consumeInput]} {
        # collect all the results we can
        while {![$that isBusy]} {
            set result [::tclpq::PQgetResult [$that connection]]
            if {$result != "NULL"} {
                # make a result object in the callback's scope
                lappend resvar \
                    [pgResult [namespace qualifiers $var]::\#auto $result]
            } else {
                # error consuming input or query's complete
                fileevent [$that socket] readable ""
                lappend resvar {}
                return
            }
        }
    } else {
        # error consuming input or query's complete
        fileevent [$that socket] readable ""
        lappend resvar {}
    }
}

# Asynchronous query processing which sets a result var
# perform an asynchronous query, setting var with the result(s)
# var has results lappended each time the query returns a result
# (for queries with multiple results)
#
# The final result is signalled by the appending of {} to the result
# Useful for pipelining processing of results
#
# query: the query to process
# var: a global var which will have results lappended to it
body pgConnection::mrexec {query var} {
    set socket [socket]
    if {[fileevent $socket readable] != ""} {
        error "$this rexec: connection busy. [errorMessage]"
    }

    fileevent $socket readable \
        [eval list [code mrir] $this $var]

    if {![sendQuery $query]} {
        error "$this rexec: [errorMessage]"
    }
}

# Asynchronous query processing which sets a result var
# perform an asynchronous query, setting var with a list of the result(s)
# var is only set once even if the query returns multiple results
# 
# query: the query to process
# var: a global var which will have results lappended to it
body pgConnection::rexec {query var} {
    puts stderr "rexec $var"
    set socket [socket]
    if {[fileevent $socket readable] != ""} {
        error "$this rexec: connection busy. [errorMessage]"
    }

    catch {unset temp}
    fileevent $socket readable \
        [list [code rir] $this [scope temp] [namespace which -variable $var]]

    if {![sendQuery $query]} {
        error "$this rexec: [errorMessage]"
    }
}

# Asynchronous query processing
# perform an asynchronous query, calling callback with the result
# 
# query: the query to process
# callback: a process to call with collected results
# args: arguments to the callback proc
body pgConnection::aexec {query callback args} {
    set socket [socket]
    if {[fileevent $socket readable] != ""} {
        error "$this aexec: connection busy. [errorMessage]"
    }
    fileevent $socket readable \
        [eval list [code ir] $this $callback $args]
    if {![sendQuery $query]} {
        error "$this aexec: [errorMessage]"
    }
}

# interrupt routine called when connection is readable to collect input
# dispatches to a callback
#
# that: dbconnection
# callback: a process to call with collected results
# args: arguments to the callback proc
#
# The callback is called once for each completed result
body pgConnection::ir {that callback args} {
    if {[$that consumeInput]} {
        # collect all the results we can
        while {![$that isBusy]} {
            set result [::tclpq::PQgetResult [$that connection]]
            if {$result != "NULL"} {
                # make a result object in the callback's scope
                set result \
                    [pgResult \
                         [namespace qualifiers \
                              [namespace origin $callback]]::\#auto \
                         $result]
                # deliver the result to the callback
                catch {eval [list $callback] [list $that] [list $result] $args}
            } else {
                # error consuming input or query's complete
                fileevent [$that socket] readable ""
                catch {eval [list $callback] [list $that] [list {}] $args}
                return
            }
        }
    } else {
        # error consuming input or query's complete
        fileevent [$that socket] readable ""
        catch {eval [list $callback] [list $that] [list {}] $args}
    }
}

# postgres connection for this object
body pgConnection::connection {} {
    return $connection
}

# Reset the communication port with the backend. 
body pgConnection::reset {} {
    return [::tclpq::PQreset $connection]
}

# db - the database name of the connection. 
body pgConnection::db {} {
    return [::tclpq::PQdb $connection]
}

# user - the user name of the connection. 
body pgConnection::user {} {
    return [::tclpq::PQuser $connection]
}

# password - the password of the connection. 
body pgConnection::password {} {
    return [::tclpq::PQpass $connection]
}

# host - the server host name of the connection. 
body pgConnection::host {} {
    return [::tclpq::PQhost $connection]
}

# port - the port of the connection. 
body pgConnection::port {} {
    return [::tclpq::PQport $connection]
}

# tty - the debug tty of the connection. 
body pgConnection::tty {} {
    return [::tclpq::PQtty $connection]
}

# options - the backend options used in the connection. 
body pgConnection::options {} {
    return [::tclpq::PQoptions $connection]
}

# status - the status of the connection.
# The status can be CONNECTION_OK or CONNECTION_BAD. 
# A failed connection attempt is signaled by status CONNECTION_BAD.
# Ordinarily, an OK status will remain so until finish,
# but a communications failure might result in the status changing to
# CONNECTION_BAD prematurely.
# In that case the application could try to recover by calling reset.
body pgConnection::status {} {
    return [::tclpq::PQstatus $connection]
}

# errorMessage - the error message most recently generated
# by an operation on the connection. 
# Nearly all libpq functions will set PQerrorMessage if they fail.
# Note that by libpq convention, a non-empty PQerrorMessage will 
# include a trailing newline.
body pgConnection::errorMessage {} {
    return [::tclpq::PQerrorMessage $connection]
}

# backendPID - the process ID of the backend server handling
# this connection. 
# The backend PID is useful for debugging purposes and for comparison
# to NOTIFY messages (which include the PID of the notifying backend).
# Note that the PID belongs to a process executing on the database server
# host, not the local host!
body pgConnection::backendPID {} {
    return [::tclpq::PQbackendPID $connection]
}

# notifies - process pending notifies
# the default implementation does nothing with notifications
# to actually process them, subclass this and define a notifies
# proc that does what you want
body pgConnection::notifies {} {
    set notification [::tclpq::PQnotifies $connection]
}

# sendQuery Submit a query to Postgres without waiting for the result(s).
# TRUE is returned if the query was successfully dispatched,
# FALSE if not (use errorMessage to get information about the failure). 
#
# After successfully calling sendQuery, call getResult one or more
# times to obtain the query results.
# sendQuery may not be called again (on the same connection) until
# getResult has returned NULL, indicating that the query is done.
body pgConnection::sendQuery {query} {
    return [::tclpq::PQsendQuery $connection $query]
}

# isBusy Returns TRUE if a query is busy, that is, getResult
# would block waiting for input. A FALSE return indicates that
# getResult can be called with assurance of not blocking. 
#
# isBusy will not itself attempt to read data from the backend;
# therefore consumeInput must be invoked first, or the busy state
# will never end.
body pgConnection::isBusy {} {
    return [::tclpq::PQisBusy $connection]
}

# socket Obtain the tcl file descriptor for the backend connection socket.
#
# socket should be used to obtain the backend socket in preparation
# for executing fileevent readable.
# This allows an application to wait for either backend responses
# or other conditions.
# On fileevent readable, consumeInput should be called to read the data;
# after which, isBusy, getResult, and/or notifies can be
# used to process the response.
body pgConnection::socket {} {
    return "sock[::tclpq::PQsocket $connection]"
}

# cancel Request that Postgres abandon processing of the current query. 
#
# TRUE if the cancel request was successfully dispatched, FALSE if not.
# (If not, errorMessage tells why not.)
#
# Successful dispatch is no guarantee that the request has any effect.
# Regardless of the return value of requestCancel, the application must
# continue with the normal result-reading sequence using getResult.
# If the cancellation is effective, the current query will terminate early
# and return an error result. If the cancellation fails (say because the
# backend was already done processing the query), then there will be no
# visible result at all.
#
# Note that if the current query is part of a transaction,
# cancellation will abort the whole transaction. 
body pgConnection::cancel {} {
    return [::tclpq::PQrequestCancel $connection]
}

# transaction - evaluate commands inside a transaction
# abort if evaluation errors
body pgConnection::transaction {args} {
    exec begin
    if [catch {eval uplevel $args} info] {
        exec rollback		;# transaction aborted
        error "Transaction abort: $info"	;# propagate the error
    } else {
        exec commit
        return $info	;# transaction's value
    }
}

# apply functional args to query tuples
body pgConnection::apply {query var args} {
    upvar $var tuple
    set result [exec $query]
    set nr [$result ntuples]
    for {set i 0} {$i < $nr} {incr i} {
        catch {unset tuple}
        $result tupleArray $i tuple	;# get tuple as $var array 
        eval uplevel $args		;# evaluate the code
    }
}
