# pgResult is a wrapper for the PGresult type returned by libpq
package require Itcl
#package require tclpq

class pgResult {
    constructor {pgresult} {}	;# construct from a pgresult
    destructor {}		;# clear the pgresult

    # tuple accessors
    public method apply {var args};# apply functional args to query tuples
    public method tupleArray {tuplenum arrayname}	;# tuple as an array
    public method tuple {tuplenum}			;# tuple as a plist
    public method getTuple {tuplenum}			;# tuple as value list
    public method getTuples {tuplenum}			;# list of value lists
    public method columnNum {fieldnum}	;# vertical slice of tuples

    # cardinality of result
    public method ntuples {}	;# number of tuples
    public method nfields {}	;# number of fields (attributes)
    public method attributes {}	;# list of attributes names

    # status of this result
    public method status {}	;# the result status of a query.
    public method errorMessage {};# error associated with the query, or ""
    public method binaryTuples {};# predicate: result contains binary data

    # field accessors
    public method getvalue {tup_num field_num}	;# field value of one tuple
    public method getisnull {tup_num field_num}	;# predicate: is field NULL?
    public method getlength {tup_num field_num}	;# length of a field in bytes.

    # field structural accessors
    public method fname {field_index}	;# name of indexed attribute
    public method ftype {field_num}	;# field type of given field index.
    public method fmod {field_index}	;# type-specific modification data
    public method fsize {field_index}	;# size in bytes of the field
    public method fnumber {field_name}	;# field index associated with name

    # status of generating command
    public method cmdStatus {}	;# command status of SQL command generating this
    public method cmdTuples {}	;# number of rows affected by the SQL command. 
    public method oidStatus {}	;# the object id of the tuple inserted,

    public method dump {}	;# dump object variables (debugging)

    private variable result
}

body pgResult::constructor {pgresult} {
    #puts stderr "constructing $this from $pgresult"
    set result $pgresult
}

body pgResult::destructor {} {
    ::tclpq::PQclear $result
}

# dump object variables (debugging)
body pgResult::dump {} {
    puts stderr "$this:"
    foreach fn {status errorMessage ntuples nfields
        cmdStatus cmdTuples oidStatus} {
        puts stderr "\t$fn: [$fn]"
    }
}

# resultStatus - the result status of the query.
# can return one of the following values:
# $PGRES_EMPTY_QUERY
# $PGRES_COMMAND_OK       the query was a command returning no data
# $PGRES_TUPLES_OK        the query successfully returned tuples
# $PGRES_COPY_OUT         Copy Out (from server) data transfer started
# $PGRES_COPY_IN          Copy In (to server) data transfer started
# $PGRES_BAD_RESPONSE     an unexpected response was received
# $PGRES_NONFATAL_ERROR
# $PGRES_FATAL_ERROR
# Note that a SELECT that happens to retrieve zero tuples still 
# shows $PGRES_TUPLES_OK.
# $PGRES_COMMAND_OK is for commands that can never return tuples.
body pgResult::status {} {
    #puts stderr "pgResult::status = [::tclpq::PQresultStatus $result]"
    return [::tclpq::PQresultStatus $result]
}

# errorMessage returns the error message associated with the query,
# or an empty string if there was no error. 
body pgResult::errorMessage {} {
    return [::tclpq::PQresultErrorMessage $result]
}

# ntuples - the number of tuples (instances) in the query result. 
body pgResult::ntuples {} {
    return [::tclpq::PQntuples $result]
}

body pgResult::tupleArray {tuplenum arrayname} {
    upvar $arrayname a
    set i 0
    foreach n [attributes] {
        set a($n) [getvalue $tuplenum $i]
        incr i
    }
}

body pgResult::tuple {tuplenum} {
    set i 0
    foreach n [attributes] {
        lappend a $n [getvalue $tuplenum $i]
        incr i
    }
    return $a
}

body pgResult::columnNum {fieldnum} {
    set nr [ntuples]
    for {set i 0} {$i < $nr} {incr i} {
        lappend a [getvalue $i $fieldnum]
    }
    return $a
}

body pgResult::getTuple {tuplenum} {
    set i 0
    foreach n [attributes] {
        lappend a [getvalue $tuplenum $i]
        incr i
    }
    return $a
}

body pgResult::getTuples {tuplenum} {
    set i 0
    set attr [attributes]
    set nr [ntuples]
    for {set row 0} {$row < $nr} {incr row} {
        set r {}
        foreach n $attr {
            lappend r [getvalue $tuplenum $i]
            incr i
        }
        lappend a $r
    }
    return $a
}

# nfields - the number of fields (attributes) in each tuple of the result. 
body pgResult::nfields {} {
    return [::tclpq::PQnfields $result]
}

# attributes - a list of attributes in each tuple of the result. 
body pgResult::attributes {} {
    set nf [::tclpq::PQnfields $result]
    for {set i 0} {$i < $nf} {incr i} {
        lappend attrs [fname $i]
    }
    return $attrs
}

# binaryTuples - 1 if the PGresult contains binary tuple data,
# 0 if it contains ASCII data. 
# Currently, binary tuple data can only be returned by a query
# that extracts data from a BINARY cursor.
body pgResult::binaryTuples {} {
    return [::tclpq::PQbinaryTuples $result]
}

# fname - the field (attribute) name associated with the given
# field index. Field indices start at 0. 
body pgResult::fname {field_index} {
    return [::tclpq::PQfname $result $field_index]
}

# fnumber - the field (attribute) index associated with the given
# field name. -1 is returned if the given name does not match any field.
body pgResult::fnumber {field_name} {
    return [::tclpq::PQfnumber $result $field_name]
}

# ftype - the field type associated with the given field index.
# The integer returned is an internal coding of the type.
# Field indices start at 0.
body pgResult::ftype {field_num} {
    return [::tclpq::PQftype $result $field_num]
}

# fsize - the size in bytes of the field associated with the given
# field index. Field indices start at 0. 
#
# fsize returns the space allocated for this field in a database tuple,
# in other words the size of the server's binary representation of the
# data type. -1 is returned if the field is variable size.
body pgResult::fsize {field_index} {
    return [::tclpq::PQfsize $result $field_index]
}

# fmod - the type-specific modification data of the field associated
# with the given field index.  Field indices start at 0. 
body pgResult::fmod {field_index} {
    return [::tclpq::PQfmod $result $field_index]
}

# getvalue - a single field (attribute) value of one tuple of a PGresult.
# Tuple and field indices start at 0. 
# For most queries, the value returned by PQgetvalue is a null-terminated
# ASCII string representation of the attribute value.
# But if PQbinaryTuples() is TRUE, the value returned by PQgetvalue is
# the binary representation of the type in the internal format of the
# backend server (but not including the size word, if the field is
# variable-length). It is then the programmer's responsibility to cast
# and convert the data to the correct C type.
# The pointer returned by PQgetvalue points to storage that is part of
# the PGresult structure.
# One should not modify it, and one must explicitly copy the value into
# other storage if it is to be used past the lifetime of the PGresult 
# structure itself.
body pgResult::getvalue {tup_num field_num} {
    return [::tclpq::PQgetvalue $result $tup_num $field_num]
}

# getlength - the length of a field (attribute) in bytes.
# Tuple and field indices start at 0. 
#
# This is the actual data length for the particular data value,
# that is the size of the object pointed to by PQgetvalue.
# Note that for ASCII-represented values, this size has little to do
# with the binary size reported by PQfsize.
body pgResult::getlength {tup_num field_num} {
    return [::tclpq::PQgetlength $result $tup_num $field_num]
}

# getisnull Tests a field for a NULL entry.
# Tuple and field indices start at 0. 
# returns 1 if the field contains a NULL, 0 if non-null value.
# (Note that getvalue will return an empty string, not a null pointer,
# for a NULL field.)
body pgResult::getisnull {tup_num field_num} {
    return [::tclpq::PQgetisnull $result $tup_num $field_num]
}

# cmdStatus - command status string from the SQL command that 
# generated the PGresult. 
body pgResult::cmdStatus {} {
    return [::tclpq::PQcmdStatus $result]
}

# cmdTuples - the number of rows affected by the SQL command. 
#
# If the SQL command that generated the PGresult was INSERT, UPDATE or 
# DELETE, this returns a string containing the number of rows affected.
# If the command was anything else, it returns the empty string.
body pgResult::cmdTuples {} {
    return [::tclpq::PQcmdTuples $result]
}

# oidStatus - a string with the object id of the tuple inserted,
# if the SQL command was an INSERT. Otherwise, returns an empty string. 
body pgResult::oidStatus {} {
    return [::tclpq::PQoidStatus $result]        
}

# apply functional args to query tuples
body pgResult::apply {var args} {
    upvar $var tuple
    set nr [ntuples]
    for {set i 0} {$i < $nr} {incr i} {
        catch {unset tuple}
        tupleArray $i tuple	;# get tuple as $var array 
        eval uplevel $args		;# evaluate the code
    }
}
