/* Swig Interface for Postgresql's libpq
 */
%module tclpq

%{
#include <libpq-fe.h>
#include <tclDecls.h>

/* an extra wrapper to make connected db also a channel */
static int _wrap_PQconnectdb(ClientData clientData, Tcl_Interp *interp, int objc, Tcl_Obj *CONST objv[])
{
    PGconn * _result;
    char * _arg0;
    Tcl_Obj * tcl_result;
    int templength;
    int sock;
    Tcl_Channel chan;

    tcl_result = Tcl_GetObjResult(interp);
    if ((objc < 2) || (objc > 2)) {
        Tcl_SetStringObj(tcl_result,
                         "Wrong # args. tclpq::PQconnectdb conninfo ",-1);
        return TCL_ERROR;
    }

    if ((_arg0 = Tcl_GetStringFromObj(objv[1], &templength)) == NULL)
        return TCL_ERROR;

    _result = (PGconn *)PQconnectdb(_arg0);

    sock = PQsocket(_result);
    chan = Tcl_MakeTcpClientChannel(sock);
    if (chan == (Tcl_Channel) NULL) {
        return TCL_ERROR;
    }
    if (Tcl_SetChannelOption(interp, chan,
                             "-translation", "binary binary") == TCL_ERROR) {
        Tcl_Close((Tcl_Interp *) NULL, chan);
        return TCL_ERROR;
    }
    if (Tcl_SetChannelOption(interp, chan,
                             "-buffering", "none") == TCL_ERROR) {
        Tcl_Close((Tcl_Interp *) NULL, chan);
        return TCL_ERROR;
    }
    Tcl_RegisterChannel(interp, chan);
    /*fprintf(stderr, "socket: %s\n", Tcl_GetChannelName(chan));*/

    tcl_result = Tcl_GetObjResult(interp);
    SWIG_SetPointerObj(tcl_result,(void *) _result,"_PGconn_p");
    return TCL_OK;
}
%}

%init %{
    Tcl_PkgProvide(interp,"tclpq","1.0");
    Tcl_CreateObjCommand(interp, SWIG_prefix "PQconnectdb",
                         _wrap_PQconnectdb,
                         (ClientData) NULL,
                         (Tcl_CmdDeleteProc *) NULL);
%}

struct PGconn;
struct PGresult;

typedef enum
{
    CONNECTION_OK = 0,
    CONNECTION_BAD
} ConnStatusType;

typedef enum
{
    PGRES_EMPTY_QUERY = 0,
    PGRES_COMMAND_OK,		/* a query command that doesn't return anything
                                 * was executed properly by the backend */
    PGRES_TUPLES_OK,		/* a query command that returns tuples
                                 * was executed properly by the backend,
                                 * PGresult contains the result tuples */
    PGRES_COPY_OUT,		/* Copy Out data transfer in progress */
    PGRES_COPY_IN,		/* Copy In data transfer in progress */
    PGRES_BAD_RESPONSE,		/* an unexpected response was recv'd from
                                 * the backend */
    PGRES_NONFATAL_ERROR,
    PGRES_FATAL_ERROR
} ExecStatusType;

struct PQconninfoOption;

%inline %{
    PQconninfoOption get_PQconninfo(PQconninfoOption *a, int index) {
        return a[index];
    }
%}

%typemap(tcl8,in) PQconninfoOption * {
    interp->result = "Can't create PQconninfoOptions";
    return TCL_ERROR;
}

// Describe how we want to return a conninfoOption record
%typemap(tcl8,out) PQconninfoOption * {
    char temp[20];
    Tcl_DString result;
    PQconninfoOption *option;

    if ($source) {
        Tcl_DStringInit(&result);
        if ($source->keyword) {
            for (option = $source; option->keyword != NULL; option++) {

                Tcl_DStringStartSublist(&result);

                Tcl_DStringAppendElement(&result,option->keyword);
                Tcl_DStringAppendElement(&result,option->envvar);
                Tcl_DStringAppendElement(&result,option->compiled);
                Tcl_DStringAppendElement(&result,option->val);
                Tcl_DStringAppendElement(&result,option->label);
                Tcl_DStringAppendElement(&result,option->dispchar);
                sprintf(temp,"%d",option->dispsize);
                Tcl_DStringAppendElement(&result,temp);
                Tcl_DStringEndSublist(&result);
            }
            Tcl_DStringResult(interp, &result);
        } else {
            Tcl_ResetResult(interp);
        }
    }
}

%typemap(tcl8,in) PGnotify * {
    interp->result = "Can't create PGnotify objects";
    return TCL_ERROR;
}

// Describe how we want to return a PGnotify record
%typemap(tcl8,out) PGnotify * {
    char temp[20];
    Tcl_DString result;
    PGnotify *notify;

    if ($source) {
        Tcl_DStringInit(&result);
        if ($source) {
            Tcl_DStringStartSublist(&result);
            
            Tcl_DStringAppendElement(&result,notify->relname);
            sprintf(temp,"%d",notify->be_pid);
            Tcl_DStringAppendElement(&result,temp);
            Tcl_DStringEndSublist(&result);
        }
        Tcl_DStringResult(interp, &result);
        free(notify);
    } else {
        Tcl_ResetResult(interp);
    }
}

%apply int { Oid };

/*PGconn *PQconnectdb(const char *conninfo);*/
PGconn *PQsetdbLogin(const char *pghost,
                     const char *pgport, 
                     const char *pgoptions,
                     const char *pgtty,
                     const char *dbName,
                     const char *login,
                     const char *pwd);

/* get info about connection options known to PQconnectdb */
PQconninfoOption *PQconndefaults(void);

/* close the current connection and free the PGconn data structure */
void PQfinish(PGconn *conn);

/*
 * close the current connection and restablish a new one with the same
 * parameters
 */
void PQreset(PGconn *conn);

/* issue a cancel request */
int PQrequestCancel(PGconn *conn);

/* Accessor functions for PGconn objects */
char *PQdb(PGconn *conn);
char *PQuser(PGconn *conn);
char *PQpass(PGconn *conn);
char *PQhost(PGconn *conn);
char *PQport(PGconn *conn);
char *PQtty(PGconn *conn);
char *PQoptions(PGconn *conn);

ConnStatusType PQstatus(PGconn *conn);
char *PQerrorMessage(PGconn *conn);
int	PQsocket(PGconn *conn);
int	PQbackendPID(PGconn *conn);

/* Enable/disable tracing */
void PQtrace(PGconn *conn, FILE *debug_port);
void PQuntrace(PGconn *conn);

/* Simple synchronous query */
PGresult *PQexec(PGconn *conn, const char *query);
PGnotify *PQnotifies(PGconn *conn);

/* Interface for multiple-result or asynchronous queries */
int	PQsendQuery(PGconn *conn, const char *query);
PGresult *PQgetResult(PGconn *conn);

/* Routines for managing an asychronous query */
int	PQisBusy(PGconn *conn);
int	PQconsumeInput(PGconn *conn);

/* Routines for copy in/out */
int	PQgetline(PGconn *conn, char *string, int length);
int	PQputline(PGconn *conn, const char *string);
int	PQgetlineAsync(PGconn *conn, char *buffer, int bufsize);
int	PQputnbytes(PGconn *conn, const char *buffer, int nbytes);
int	PQendcopy(PGconn *conn);

/* Accessor functions for PGresult objects */
ExecStatusType PQresultStatus(PGresult *res);
const char *PQresultErrorMessage(PGresult *res);
int	PQntuples(PGresult *res);
int	PQnfields(PGresult *res);
int	PQbinaryTuples(PGresult *res);
char *PQfname(PGresult *res, int field_num);
int	PQfnumber(PGresult *res, const char *field_name);
Oid	PQftype(PGresult *res, int field_num);
int	PQfsize(PGresult *res, int field_num);
int	PQfmod(PGresult *res, int field_num);
char *PQcmdStatus(PGresult *res);
const char *PQoidStatus(PGresult *res);
const char *PQcmdTuples(PGresult *res);
char *PQgetvalue(PGresult *res, int tup_num, int field_num);
int	PQgetlength(PGresult *res, int tup_num, int field_num);
int	PQgetisnull(PGresult *res, int tup_num, int field_num);

/* Delete a PGresult */
void PQclear(PGresult *res);

/* Make an empty PGresult with given status (some apps find this useful).
 * If conn is not NULL and status indicates an error, the conn's
 * errorMessage is copied.
 */
PGresult * PQmakeEmptyPGresult(PGconn *conn, ExecStatusType status);

/* Large-object access routines */
int	lo_open(PGconn *conn, Oid lobjId, int mode);
int	lo_close(PGconn *conn, int fd);
int	lo_read(PGconn *conn, int fd, char *buf, int len);
int	lo_write(PGconn *conn, int fd, char *buf, int len);
int	lo_lseek(PGconn *conn, int fd, int offset, int whence);
Oid	lo_creat(PGconn *conn, int mode);
int	lo_tell(PGconn *conn, int fd);
int	lo_unlink(PGconn *conn, Oid lobjId);
Oid	lo_import(PGconn *conn, char *filename);
int	lo_export(PGconn *conn, Oid lobjId, char *filename);
