/* Subcommands for tcl "browse" command */
#include <stdio.h>
#include <tcl.h>
#include "tcl_browse.h"

int browseRescan();
int browseRedraw();
int browseCmds();
int browseChdir();
int browseQuit();
int browseDelete();
int browseRename();
int browseMessage();
int browsePrint();
int browseMove();
int browseBell();
int browseShell();
int browseTag();
int browseAdd();
int browseMode();

struct subcmd browsecmds[] = {
	{ browseCmds, "commands", 0, 0, "" },
	{ browseChdir, "chdir", 1, 1, "directory" },
	{ browseQuit, "exit", 0, 1, "[code]" },
	{ browseDelete, "delete", 1, -1, "file..." },
	{ browseRename, "rename", 2, 2, "from to" },
	{ browseMove, "move", 1, 1, "entry" },
	{ browseMessage, "message", 0, -1, "[text]..." },
	{ browsePrint, "print", 0, -1, "[text]..." },
	{ browseTag, "tag", 2, -1, "{+,-,/}[PT] file..." },
	{ browseBell, "bell", 0, 0, "" },
	{ browseRedraw, "redraw", 0, 0, "" },
	{ browseRescan, "rescan", 0, 0, "" },
	{ browseShell, "shell", 1, 1, "command" },
	{ browseMode, "mode", 1, 1, "[line|page|wide|narrow]" },
	{ browseAdd, "add", 1, -1, "file..." },
};
int browsecnt = sizeof browsecmds / sizeof *browsecmds;

int
cmdBrowse(clientData, interp, argc, argv)
ClientData clientData;
Tcl_Interp *interp;
int argc;
char **argv;
{
	return Handle(browsecnt, browsecmds, interp, argc, argv);
}

browseDelete(interp, argc, argv)
Tcl_Interp *interp;
int argc;
char **argv;
{
	int i;
	int result;
	char *file_name();

	while(*argv) {
		i = get_index(*argv);
		result = 0;

		if(i==-1) {
			if(unlink(*argv) == 0)
				result = 1;
			else
				save_errno(*argv);
		} else {
			if(unlink(file_name(i)) == 0) {
				result = 1;
				delete_entry(i);
			}
			else
				save_errno(*argv);
		}
		if(result==0)
			break;
		++argv;
	}
	sprintf(interp->result, "%d", result);
	return TCL_OK;
}

browseRename(interp, argc, argv)
Tcl_Interp *interp;
int argc;
char **argv;
{
	int i;
	int result;

	i = get_index(*argv);
	result = 0;

	if(i == -1) {
		if(link(argv[0], argv[1]) == 0) {
			if(unlink(argv[0]) == 0)
				result = 1;
			else
				save_errno(*argv);
		}
		else
			save_errno(*argv);
	} else {
		if(domove(i, argv[1]))
			result = 1;
	}
	sprintf(interp->result, "%d", result);
	return TCL_OK;
}

browseChdir(interp, argc, argv)
Tcl_Interp *interp;
int argc;
char **argv;
{
	strcpy(interp->result, enter(*argv)?"1":"0");
	return TCL_OK;
}

browseQuit(interp, argc, argv)
Tcl_Interp *interp;
int argc;
char **argv;
{
	int i;

	if(argc)
		i = atoi(*argv);
	else
		i = 0;

	tcl_end();
	tend();

	exit(i);
}

browseMessage(interp, argc, argv)
Tcl_Interp *interp;
int argc;
char **argv;
{
	int result;
	extern int display_up;

	cmdline();
	result = browsePrint(interp, argc, argv);
	if(atoi(interp->result) >= 80)
		end_screenmode();
	else
		if(!display_up) putchar('\n');
	return result;
}

browseCmds(interp, argc, argv)
Tcl_Interp *interp;
int argc;
char **argv;
{
	return Format(browsecnt, browsecmds, interp);
}

browsePrint(interp, argc, argv)
Tcl_Interp *interp;
int argc;
char **argv;
{
	extern int display_up;
	int count = 0;

	while(*argv) {
		count += strlen(*argv)+1;
		if(display_up)
			ctlouts(*argv);
		else
			outs(*argv);
		++argv;
		if(*argv)
			putchar(' ');
	}

	sprintf(interp->result, "%d", count);

	return TCL_OK;
}

browseMove(interp, argc, argv)
Tcl_Interp *interp;
int argc;
char **argv;
{
	extern int curr, nentries;
	int new;

	switch(**argv) {
		case '*': new = nentries-1; break;
		case '-': new = curr - atoi(*argv+1); break;
		case '+': new = curr + atoi(*argv+1); break;
		default: new = atoi(*argv); break;
	}
	if(new >= 0 && new < nentries) {
		curr = new;
		strcpy(interp->result, "1");
	} else {
		save_errmsg("entry out of range");
		strcpy(interp->result, "0");
	}
	return TCL_OK;
}

browseBell(interp, argc, argv)
Tcl_Interp *interp;
int argc;
char **argv;
{
	putchar(7);
	return TCL_OK;
}

browseShell(interp, argc, argv)
Tcl_Interp *interp;
int argc;
char **argv;
{
	sprintf(interp->result, "%d", system(*argv, 1));
	return TCL_OK;
}

browseRedraw(interp, argc, argv)
Tcl_Interp *interp;
int argc;
char **argv;
{
	redraw();
	return TCL_OK;
}

browseRescan(interp, argc, argv)
Tcl_Interp *interp;
int argc;
char **argv;
{
	sprintf(interp->result, "%d", newdir());
	return TCL_OK;
}

browseTag(interp, argc, argv)
Tcl_Interp *interp;
int argc;
char **argv;
{
	int i;
	int result;
	int mode;
	int tagbit;

	switch(**argv) {
		case '+': mode = 1; break;
		case '-': mode = -1; break;
		case '/': mode = 0; break;
		default: goto tcl_error;
	}

	++*argv;

	switch(**argv) {
		case 'p': case 'P': tagbit = 'P'; break;
		case 't': case 'T': tagbit = 'T'; break;
		case 0: tagbit = 'T'; break; /* default */
		default: goto tcl_error;
	}

	result = 1;
	while(*++argv) {
		i = get_index(*argv);
		if(i == -1) {
			result = 0;
			break;
		}
		tag(i, mode, tagbit);
	}
	sprintf(interp->result, "%d", result);
	return TCL_OK;

tcl_error:
	Tcl_Return(interp,
		"invalid tag option:  should be \"tag {+,-,/}[PT] file...\"",
		TCL_STATIC);
	return TCL_ERROR;
}

browseMode(interp, argc, argv)
Tcl_Interp *interp;
int argc;
char **argv;
{
	if(strcmp(*argv, "line") == 0) {
		end_screenmode();
	} else if(strcmp(*argv, "page") == 0) {
		end_linemode();
	} else if(strcmp(*argv, "narrow") == 0) {
		set_quickmode(1);
	} else if(strcmp(*argv, "wide") == 0) {
		set_quickmode(0);
	} else {
		Tcl_Return(interp,
			"invalid mode name:  should be \"mode [page|line|wide|narrow]\"",
			TCL_STATIC);
		return TCL_ERROR;
	}
	return TCL_OK;
}

browseAdd(interp, argc, argv)
Tcl_Interp *interp;
int argc;
char **argv;
{
	int result;

	result = 1;
	while(*argv) {
		if(!addfile(*argv)) {
			result = 0;
			break;
		}
		argv++;
	}
	sprintf(interp->result, "%d", result);
	return TCL_OK;
}
