#
# NeoWebScript - Server Side Programming based on Safe Tcl
#
# Copyright (C) NeoSoft, All Rights Reserved.  See NeoWebScript LICENSE
# files with this distribution for conditions on use and redistribution.
#
# $Id: common.tcl,v 1.28 1997/12/30 22:47:55 tsunami Exp $
#

#
# PROPERTY LIST STORAGE AND RETRIEVAL
#
set parallelBase "$server(SERVER_ROOT)/neoscript-data/users"
set parallelSystemBase "$server(SERVER_ROOT)/neoscript-data/system"

proc setup_data_access {} {
    global parallelDir webpageOwnerName webenv

    if [info exists parallelDir] return

    global webenv parallelBase

    if [info exists NeoWebDirConf(WebPageOwnerName)] {
    set webpageOwnerName $NeoWebDirConf(WebPageOwnerName)
    } else {
    set webpageOwnerName [id convert userid $webenv(NEO_DOCUMENT_UID)]
    }
    set parallelDir $parallelBase/[cindex $webpageOwnerName 0]/$webpageOwnerName/
}

proc create_user_dir {plusSubDir} {
    global parallelDir errorCode

    setup_data_access

    if {[catch {mkdir -path $parallelDir$plusSubDir} result] == 1} {
	if {[lrange $errorCode 0 1] != "POSIX EEXIST"} {
	    error "$errorCode"
	}
    }
}

proc ul_name_check {dbname dbtype fileNameVar} {
    if ![regexp {^[-a-zA-Z0-9_=+.]*$} $dbname] {
	error "illegal database name: $dbname, lowercase, uppercase, 0-9, _, -, +, =, and .  only"
    }
    upvar $fileNameVar dbFileName
    global parallelDir

    set dbFileName ${parallelDir}$dbtype/$dbname.$dbtype
}

proc db_name_check {dbname dbtype fileNameVar {projectName {}}} {
    if ![regexp {^[a-zA-Z0-9_]*$} $dbname] {
	return -code error "illegal database name: $dbname, upper/lowercase and 0-9 only"
    }
    upvar $fileNameVar dbFileName
    global parallelDir


    set dbFileName ${parallelDir}$dbtype
    if {$projectName != {}} {
	    if ![regexp {^[a-zA-Z0-9_]*$} $projectName] {
	    return -code error "illegal project name: $projectName, upper/lowercase and 0-9 only"
	    }
	    append dbFileName "/$projectName"
	    if {[catch {mkdir -path $dbFileName} result] == 1} {
		    if {[lrange $errorCode 0 1] != "POSIX EEXIST"} {
			    return -code error "$errorCode"
		    }
	    }
    }
    append dbFileName "/$dbname.$dbtype"
}

proc dbfetch {database id arrayName args} {
    global errorCode

    setup_data_access

    set ssearch [lsearch -exact $args "-singleVar"]
    if {$ssearch > -1} {	
	set singleVar "-singleVar"
	set args [lreplace $args $ssearch $ssearch]
    } else {
	set singleVar {}
    }

    if {[cindex $args 0] == "-"} {import_keyvalue_pairs data $args}
    if ![info exists data(project)] {set data(project) {}}
    db_name_check $database db dbFileName $data(project)

    if {[catch {db open $dbFileName hash rwl 0664} db] == 1} {
	if {[lrange $errorCode 0 1] == "POSIX ENOENT"} {
		return 0
	} else {
	   return -code error "$dbFileName: $errorCode"
	}
    }

    db get $db $id list
    db close $db

    set haveData [expr ![lempty $list]]

    if {$singleVar == "-singleVar"} {
	    set $arrayName $list
    } else {
	    upvar $arrayName array
	    array set array $list
    }

    return $haveData
}

proc dbstore {database id arrayName args} {
    global errorCode

    setup_data_access

    set ssearch [lsearch -exact $args "-singleVar"]
    if {$ssearch > -1} {	
	set list $arrayName
	set args [lreplace $args $ssearch $ssearch]
    } else {
	upvar $arrayName array
	set list [array get array]
    }

    if {[cindex $args 0] == "-"} {import_keyvalue_pairs data $args}
    if ![info exists data(project)] {set data(project) {}}
    db_name_check $database db dbFileName $data(project)

    if {[catch {set db [db open $dbFileName hash rwL 0664]}] == 1} {
	if {[lrange $errorCode 0 1] != "POSIX ENOENT"} {
	    return -code error "$dbFileName: $errorCode"
	} else {
        create_user_dir db
	    set db [db open $dbFileName hash ctL 0664]
	}
    }
    db put $db $id $list
    db close $db
    return
}

SAFE_proc_and_alias dbdelkey {safeInterp database id args} {
    global errorCode

    setup_data_access

    if {[cindex $args 0] == "-"} {import_keyvalue_pairs data $args}
    if ![info exists data(project)] {set data(project) {}}
    db_name_check $database db dbFileName $data(project)

    if {[catch {set db [db open $dbFileName hash rw 0664]}] == 1} {
	if {[lrange $errorCode 0 1] != "POSIX ENOENT"} {
	    return -code error "$dbFileName: $errorCode"
	} else {
	    return 0
	}
    }
    set result [db del $db $id]
    db close $db
    return $result
}

SAFE_proc_and_alias dbexists {safeInterp database args} {
    setup_data_access
    if {[cindex $args 0] == "-"} {import_keyvalue_pairs data $args}
    if ![info exists data(project)] {set data(project) {}}
    db_name_check $database db dbFileName $data(project)

    return [file readable $dbFileName]
}

SAFE_proc_and_alias filerm {safeInterp type database args} {
    setup_data_access
    if {[cindex $args 0] == "-"} {import_keyvalue_pairs data $args}
    if ![info exists data(project)] {set data(project) {}}
    db_name_check $database $type dbFileName $data(project)

    unlink -nocomplain $dbFileName
}

SAFE_proc_and_alias filemv {safeInterp type old_database database args} {
    setup_data_access
    if {[cindex $args 0] == "-"} {import_keyvalue_pairs data $args}
    if ![info exists data(project)] {set data(project) {}}
    db_name_check $old_database $type old_dbFileName $data(project)
    db_name_check $database $type dbFileName $data(project)

    if ![file exists $old_dbFileName] {
	return -code error "attempt to move non-existent $type file $old_database"
    }
    if [file exists $dbFileName] {
	unlink $dbFileName
    }
    link $old_dbFileName $dbFileName
    unlink $old_dbFileName
}

SAFE_proc_and_alias filecp {safeInterp type old_database database args} {
    setup_data_access
    if {[cindex $args 0] == "-"} {import_keyvalue_pairs data $args}
    if ![info exists data(project)] {set data(project) {}}
    db_name_check $old_database $type old_dbFileName $data(project)
    db_name_check $database $type dbFileName $data(project)

    if ![file exists $old_dbFileName] {
	error "attempt to copy non-existent $type file $old_database"
    }
    if [file exists $dbFileName] {
	unlink $dbFileName
    }
    link $old_dbFileName $dbFileName
}

SAFE_proc_and_alias dbkeys {safeInterp database args} {
    global errorCode

    setup_data_access

    if {[cindex $args 0] == "-"} {
	    import_keyvalue_pairs data $args
    } else {
	    set data(pattern) [lindex $args 0]
    }

    if ![info exists data(project)] {set data(project) {}}
    db_name_check $database db dbFileName $data(project)

    if {![info exists data(pattern)] || $data(pattern) == {}} {
	    set data(pattern) *
    }

    if {[catch {set db [db open $dbFileName hash rw 0664]}] == 1} {
	if {[lrange $errorCode 0 1] != "POSIX ENOENT"} {
	    return -code error "$dbFileName: $errorCode"
	} else {
	    return ""
	}
    }
    set keys ""
    db searchall $db key -glob $data(pattern) {
	lappend keys $key
    }
    db close $db

    return $keys
}

SAFE_proc_and_alias dbfiles {safeInterp args} {
    global parallelDir errorCode
    setup_data_access

    if {[cindex $args 0] == "-"} {
	    import_keyvalue_pairs data $args
    } else {
	    set data(pattern) [lindex $args 0]
    }

    if {![info exists data(pattern)] || $data(pattern) == {}} {
	    set data(pattern) *
    }

    set db "${parallelDir}db"
    if [info exists data(project)] {
	    append db "/$data(project)"
    }
    append db "/$data(pattern).db"

    set result {}
    foreach file [glob -nocomplain $db] {
	lappend result [file root [file tail $file]]
    }
    return $result
}

SAFE_proc_and_alias projectrm {safeInterp type projectName} {
    setup_data_access
    global parallelDir errorCode
    set path $parallelDir$type/$projectName
    foreach file [glob -nocomplain $path/*.$type] {
	    unlink -nocomplain $file
    }
    if {[catch {rmdir -path $path} result] == 1} {
	    return -code error "$errorCode"
    }
}

# proc to import key value pairs of the form "-color blue"
# into an array in the caller's context.
#
# and dialog box thingie that uses it
#
# i am really missing incr tcl
#
#
#neo_dialog -text "This is an alert mesesage... or something."
#
#neo_dialog -text "This is another alert message."
#

proc import_keyvalue_pairs {arrayName string} {
    upvar $arrayName array

    set len [llength $string]
    if {$len % 2 != 0} {
        error "unmatched key-value pair"
    }

    for {set i 0} {$i < $len} {incr i 2} {
        set key [lindex $string $i]
        if {[string index $key 0] != "-"} {
            error "key $key of key-value pairs doesn't start with a dash"
        }
        set array([string range $key 1 end]) [lindex $string [expr $i + 1]]
    }
}

SAFE_proc_and_alias delete_data_file {safeInterp datafile args} {
    global parallelDir errorCode
    setup_data_access

    if {[cindex $args 0] == "-"} {
	    import_keyvalue_pairs data $args
    }

    if ![info exists data(project)] {set data(project) {}}
    db_name_check $datafile data dataFileName $data(project)

    if {[catch {unlink $dataFileName} result] == 1} {
	if {[lrange $errorCode 0 1] != "POSIX ENOENT"} {
	    error "$result"
	}
    }
    return
}

#
# List the data files in the server-maintained data directory
# for the user.
#
#
SAFE_proc_and_alias list_data_files {safeInterp args} {
    global parallelDir errorCode
    setup_data_access

    if {[cindex $args 0] == "-"} {
	    import_keyvalue_pairs data $args
    } else {
	    set data(pattern) [lindex $args 0]
    }

    if {![info exists data(pattern)] || $data(pattern) == {}} {
	    set data(pattern) *
    }

    set db "${parallelDir}data"
    if [info exists data(project)] {
	    append db "/$data(project)"
    }
    append db "/$data(pattern).data"

    set result {}
    foreach file [glob -nocomplain $db] {
	lappend result [file root [file tail $file]]
    }
    return $result
}

#
#Return 1 if a datafile exists in the server-maintained data directory.
#
#
SAFE_proc_and_alias data_file_exists {safeInterp datafile args} {
    global parallelDir errorCode
    setup_data_access

    if {[cindex $args 0] == "-"} { import_keyvalue_pairs data $args }

    if ![info exists data(project)] {set data(project) {}}
    db_name_check $datafile data dataFileName $data(project)

    return [file exists $dataFileName]
}

#
# check_password tests sent_pw against real_pw and returns a value
# suitable to be returned to the C auth code in apache.  real_pw
# is already encrypted, but sent_pw is not.  the first two bytes of
# real_pw is used as the salt.
#
proc check_password {sent_pw real_pw} {
    return [cequal [neo_crypt $sent_pw $real_pw] $real_pw]
}

# Authorization routine for DB Auth.
proc tcl_db_auth {dbOwner dbName username sent_pw} {
    global parallelBase
    upvar _${dbOwner}_${dbName}_auth_cache authCache
    if [info exists authCache($username)] {
        lassign $authCache($username) password expire
        if {$expire > [clock seconds] && [check_password $sent_pw $password]} {
            return
        }
    }
    set password *
    set dbFileName $parallelBase/[cindex $dbOwner 0]/$dbOwner/db/$dbName.db
    if {[catch {set db [db open $dbFileName hash r]}] == 1} {
        return "user $username: unable to access $dbFileName"
    }
    set found [db get $db $username password]
    db close $db
    if $found {
        set authCache($username) [list $password [expr [clock seconds] + 60]]
        if [check_password $sent_pw $password] return
    }
    return "user $username: invalid password"
}

proc tcl_db_access {user type} {
        if {$type == "valid-user"} { return OK }
        return AUTH_REQUIRED
}

# Authorization routine for /etc/passwd access.  Requires the 'getpass'
# program to be built.

proc tcl_passwd_auth {username sent_pw} {
        global server
        if [catch {exec $server(SERVER_ROOT)/bin/getpass $username} password] {
                return "user $username not found"
        }
        if [check_password $sent_pw $password] return
        return "$pwError (/etc/passwd)"
}

proc tcl_passwd_access {user type} {
    if {$type == "valid-user"} { return OK }
    return AUTH_REQUIRED
}

# Routines for the test authentication based on Tcl
proc test_tcl_auth {args} {
    return [neo_crypt test ab]
}

proc test_tcl_access {args} {
    return OK
}
