#
# NeoWebScript - Server Side Programming based on Safe Tcl
#
# Copyright (C) NeoSoft, All Rights Reserved.  See NeoWebScript LICENSE
# files with this distribution for conditions on use and redistribution.
#
# $Id: init.tcl,v 1.50 1998/01/05 21:11:08 tsunami Exp $
#

#
# If we're debugging, we load devel.tcl every time we need to
# create a safe interpreter.
#
# If we're not debugging, we load devel.tcl at startup.
#
# devel_setup is executed every time a safe interpreter is
# created, in either case.
#

set debugging 1
rename copyfile ""
rename unsupported0 copyfile

# Path to the program which unpacks mime uploads for us
set webunpackPath $server(SERVER_ROOT)/bin/webunpack

# safe_and_trusted_proc: for procs that have identical args and code
# in either interp.  Autoloaded.
proc safe_and_trusted_proc {name arglist body} {
    global safe_proc_cache
    set safe_proc_cache($name) [list proc $name $arglist $body]
    proc $name $arglist $body
}

# Declare as safe_proc to make available only for safe interp.  Autoloaded.
proc safe_proc {name arglist body} {
    global safe_proc_cache
    set safe_proc_cache($name) [list proc $name $arglist $body]
}

# Safe_alias declare an alias the safe interp.  Autoloaded.
proc safe_alias {name args} {
    global safe_alias_cache
    set safe_alias_cache($name) $args
}

# Declare as SAFE_proc_and_alias for situation where SAFE_$procname is
# the alias and it takes the interpreter as its first argument.
# Arglist must allow for safeInterp name to be passed as first arg.
proc SAFE_proc_and_alias {name arglist body} {
    global safe_proc_cache safe_alias_cache
    proc SAFE_$name $arglist $body
    set safe_alias_cache($name) "SAFE_$name \$safeInterp"
}

# Declare as alias_proc if the routine will be called exactly the
# same in either trusted or safe interp.  Command in trusted interp
# is the same as the alias name, and safeInterp is not passed.
proc safe_and_trusted_alias {name arglist body} {
    global safe_proc_cache safe_alias_cache
    proc $name $arglist $body
    set safe_alias_cache($name) $name
}

# load in shared base functions
source $server(SERVER_ROOT)/conf/common.tcl

# Load in Postgres if the configuration says we have it
if ![info exists NeoWebServerConf(HavePostgres)] {
    set NeoWebServerConf(HavePostgres) 0
}
if $NeoWebServerConf(HavePostgres) {
	source $server(SERVER_ROOT)/conf/postgres.tcl
}

set localprocs $server(SERVER_ROOT)/conf/nwslocal/local.tcl
proc local_setup {safeInterp} { }
if [catch { source $localprocs }] {
    puts stderr $errorInfo
}

if !$debugging {
    source $server(SERVER_ROOT)/conf/devel.tcl
}

safe_and_trusted_alias dump_environment {} {
    global webenv

    html "<pre>"
    foreach var [lsort [array names webenv]] {
        html "$var = $webenv($var)\n"
    }
    html "</pre>"
}

safe_and_trusted_proc parray {arrayName} {
    upvar $arrayName array
    html "<pre><b>$arrayName</b>\n"
    foreach element [lsort [array names array]] {
	html "$element = $array($element)\n"
    }
    html "</pre>"
}

safe_and_trusted_alias backlink {{linktext ""} {nolinktext ""}} {
    global webenv

    if ![info exists webenv(HTTP_REFERER)] {
	return $nolinktext
    }

    if {$linktext == ""} {
	set linktext "Back"
    }
    return "<a href=$webenv(HTTP_REFERER)>$linktext</a>"
}

#
# PROPERTY LIST STORAGE AND RETRIEVAL
#

SAFE_proc_and_alias dbfetch {safeInterp database id arrayName args} {
    global errorCode

    setup_data_access

    set ssearch [lsearch -exact $args "-singleVar"]
    if {$ssearch > -1} {	
	set singleVar "-singleVar"
	set args [lreplace $args $ssearch $ssearch]
    } else {
	set singleVar {}
    }

    if {[cindex $args 0] == "-"} {import_keyvalue_pairs data $args}
    if ![info exists data(project)] {set data(project) {}}
    db_name_check $database db dbFileName $data(project)

    if {[catch {db open $dbFileName hash rwl 0664} db] == 1} {
	if {[lrange $errorCode 0 1] == "POSIX ENOENT"} {
	    return 0
	} else {
	    error "$dbFileName: $errorCode" $errorInfo
	}
    }

    db get $db $id list
    db close $db

    set haveData [expr ![lempty $list]]

    if {$singleVar == "-singleVar"} {
	$safeInterp eval set $arrayName [list $list]
    } else {
	$safeInterp eval array set $arrayName [list $list]
    }

    return $haveData
}

SAFE_proc_and_alias dbstore {safeInterp database id arrayName args} {
    global errorCode

    setup_data_access

    set ssearch [lsearch -exact $args "-singleVar"]
    if {$ssearch > -1} {	
	set list $arrayName
	set args [lreplace $args $ssearch $ssearch]
    } else {
	set list [$safeInterp eval array get $arrayName]
    }

    if {[cindex $args 0] == "-"} {import_keyvalue_pairs data $args}
    if ![info exists data(project)] {set data(project) {}}
    db_name_check $database db dbFileName $data(project)

    if {[catch {set db [db open $dbFileName hash rwL 0664]}] == 1} {
	if {[lrange $errorCode 0 1] != "POSIX ENOENT"} {
	    return -code error "$dbFileName: $errorCode"
	} else {
	    create_user_dir db
	    set db [db open $dbFileName hash ctL 0664]
	}
    }
    db put $db $id $list
    db close $db
    return
}

SAFE_proc_and_alias load_cookies {safeInterp {cookieVar "cookies"}} {
    global webenv

    if ![info exists webenv(HTTP_COOKIE)] return

    foreach pair [split $webenv(HTTP_COOKIE) ";"] {
	set pair [split [string trim $pair] "="]
	set key [lindex $pair 0]
	set value [lindex $pair 1]
	$safeInterp eval set ${cookieVar}($key) [list $value]
    }
}

#
# Crack the response (we're running as a pseudo-CGI in this case)
# into a global array of key-value pairs in the safe interpreter.
#
# Note info exists may return true when there is no data.
# We could check the action here and see if it's a POST
# and then we'd know which variable to look at.  The whole
# thing is somewhat kludgey.
#
# Now that we are supporting MIME types to handle file uploads, we
# need to find out if the response is mime, crack it out, and then
# handle it correctly.  Seek.  Locate.  Exterminate.
#
SAFE_proc_and_alias load_response {safeInterp {responseVar "response"} {multiple ""}} {
    global webenv

    set responseString ""

    if {[info exists webenv(QUERY_STRING)]} {
	set responseString $webenv(QUERY_STRING)
    }

    if {[info exists webenv(NEO_POST_DATA)]} {
	set responseString $webenv(NEO_POST_DATA)
    }

    foreach key $multiple {
	set response(__$key) {}
    }

    foreach pair [split $responseString "&"] {
	set pair [split $pair "="]
	set key [www_unescape_string [lindex $pair 0]]
	set value [www_unescape_string [lindex $pair 1]]
	if [info exists response(__$key)] {
	    lappend response($key) $value
	} else {
	    if [info exists response($key)] {
		set response(__$key) {}
		set response($key) [list $response($key) $value]
	    } else {
		set response($key) $value
	    }
	}
    }
    $safeInterp eval array set $responseVar [list [array get response]]
}

#
# Increment a page counter based on the DOCUMENT_URI and return the result.
# Start it at zero if it doesn't exist (i.e. it returns 1 on the first
# try.)
#
SAFE_proc_and_alias incr_page_counter {safeInterp {start 0}} {
    global webenv

    if {![dbfetch pagecounters $webenv(DOCUMENT_URI) accessInfo] \
      || ![info exists accessInfo(counter)] \
      || $start > $accessInfo(counter)} {
	set accessInfo(counter) $start
    }
    set counter [incr accessInfo(counter)]
    dbstore pagecounters $webenv(DOCUMENT_URI) accessInfo
    return $counter
}

#
# Pick an element in a list at random and return it
# Now uses keyvalue pairs and will allow a text file as its list
#
SAFE_proc_and_alias random_pick_html {safeInterp args} {
    if {[cindex $args 0] == "-"} {
	import_keyvalue_pairs data $args
    }

    if [info exists data(file)] {
	set nFilename [SAFE_filename $data(file)]
	if [lempty $nFilename] {
	    return -code error "$data(file): Invalid pathname requested"
	}

        if [file exists $nFilename] {
            set fp [open $nFilename r]
            set list ""
            while {[gets $fp line] != -1} {
                lappend list $line
            }
            close $fp
        } else {
            return -code error "$data(file): file does not exist"
        }
    } else {
	set list [lindex $args 0]
    }

    return [lindex $list [expr [clock seconds] % [llength $list]]]
}

#
# Log a message to the log directory we keep for the users.
#
SAFE_proc_and_alias log_message {safeInterp logfile string} {
    global parallelDir errorCode errorInfo webenv

    setup_data_access

    db_name_check $logfile log logFileName

    if {[catch {set fp [open $logFileName "CREAT WRONLY APPEND" 0660]} result] == 1} {
	if {[lrange $errorCode 0 1] != "POSIX ENOENT"} {
	    error "$logFileName: $errorCode"
	}
	create_user_dir log
	set fp [open $logFileName "CREAT WRONLY APPEND" 0660]
    }
    puts $fp [list [clock seconds] $webenv(REMOTE_HOST) $string]
    close $fp
    return
}

#
# Access a data file with all the file stuff present in safe interpreters.
#
SAFE_proc_and_alias access_data_file {safeInterp datafile args} {
    global parallelDir errorCode
    setup_data_access

    if {[cindex $args 0] == "-"} {import_keyvalue_pairs data $args}
    if ![info exists data(project)] {set data(project) {}}
    db_name_check $datafile data dataFileName $data(project)

    if {[catch {set fp [open $dataFileName "CREAT RDWR" 0660]}] == 1} {
	if {[lrange $errorCode 0 1] != "POSIX ENOENT"} {
	    error "$logFileName: $errorCode"
	}
	create_user_dir data
	set fp [open $dataFileName "CREAT RDWR" 0660]
    }

    interp transfer "" $fp $safeInterp
    return $fp
}

SAFE_proc_and_alias delete_data_file {safeInterp datafile} { 
    global parallelDir errorCode
    setup_data_access 
    
    db_name_check $datafile data dataFileName
    
    if {[catch {unlink $dataFileName} result] == 1} {
        if {[lrange $errorCode 0 1] != "POSIX ENOENT"} {
            error "$result"
        }
    }
    return
}

#
# Send mail configuration: if no NeoWebServerConf SendMail
#
if {![info exists $NeoWebServerConf(SendMail)]} {
    if {[file executable /usr/sbin/sendmail]} {
        set NeoWebServerConf(SendMail) /usr/sbin/sendmail
    } elseif {[file executable /usr/lib/sendmail]} {
        set NeoWebServerConf(SendMail) /usr/lib/sendmail
    }
}

SAFE_proc_and_alias open_outbound_mail {safeInterp args} {
    
    global webpageOwnerName webenv NeoWebServerConf
 
    if {[cindex $args 0] != "-"} {
	lassign $args data(subject) data(to) data(from)
    } else {
	import_keyvalue_pairs data $args
    }

    if ![info exists data(subject)] {
        error "-subject not specified"
    }
    
    setup_data_access

    regsub {^www\.} $webenv(SERVER_NAME) {} server_name
    
    if {![info exists data(from)] || [lempty $data(from)]} {
        set data(from) "$webpageOwnerName@$server_name"
    }
    if {![info exists data(to)]} {  
        set data(to) "$webpageOwnerName@$server_name"
    }
    regsub -all " " $data(from) "" data(from)


    set fp [open "|$NeoWebServerConf(SendMail) -t -f $data(from)" w]
    puts $fp "From: $data(from)"
    puts $fp "To: $data(to)"
    puts $fp "Subject: $data(subject)"
    foreach header [lmatch [array names data] {[A-Z]*}] {
	if [info exists data($header)] {
	    puts $fp "$header: $data($header)"
	}
    }
    puts $fp "X-Mailer: $webenv(SERVER_SOFTWARE)"
    puts $fp "X-Webserver-Host: $webenv(SERVER_NAME)"
    puts $fp "X-Sender-URI: $webenv(DOCUMENT_URI)"
    puts $fp "X-Visitor-Host: $webenv(REMOTE_HOST) at [clock format [clock seconds]]"
    puts $fp ""

    interp transfer "" $fp $safeInterp
    return $fp
}

#
# Post news
#

#Newsgroups: comp.lang.tcl
#Subject: Foo
#Summary:
#Followup-To:
#Distribution:
#Organization: NeoSoft, Inc. +1 713 968 5800
#Keywords:
#Cc:

SAFE_proc_and_alias open_post_news {safeInterp args} {
    global webpageOwnerName webenv
    
    import_keyvalue_pairs data $args 

    if ![info exists data(newsgroups)] {
        error "-newsgroups not specified"
    }

    if ![info exists data(subject)] {
        error "-subject not specified"
    }

    if ![info exists data(distribution)] {  
        set data(distribution) ""
    }

    setup_data_access

    if {[info exists data(from)]} {
        set from $data(from)
    } else {
	set from "$webpageOwnerName@[string trimleft $webenv(SERVER_NAME) www.]"
    }

    set fp [socket news 119]
    fconfigure $fp -translation lf
    gets $fp line
    if {[string index $line 0] != 2} {
        error "news server not ready: $line"
    }
    puts $fp "post"
    flush $fp
    gets $fp line
    if {[string index $line 0] != "3"} {
        error "news server post command failed: $line"
    }

    puts $fp "From: $from"
    puts $fp "Newsgroups: $data(newsgroups)"
    puts $fp "Subject: $data(subject)"
    puts $fp "Distribution: $data(distribution)"
    if [info exists data(approved)] {
	puts $fp "Approved: $data(approved)"
    }
    puts $fp "X-News-Posting-Program: $webenv(SERVER_SOFTWARE)"
    puts $fp "X-Webserver-Host: $webenv(SERVER_NAME)"
    puts $fp "X-Sender-URI: $webenv(DOCUMENT_URI)"
    puts $fp "X-Visitor-Host: $webenv(REMOTE_HOST) at [clock format [clock seconds]]"

    puts $fp ""
    interp transfer "" $fp $safeInterp
    return $fp
}


#
# Disk usage configuration: if no NeoWebServerConf Du
#
if {![info exists $NeoWebServerConf(Du)]} {
    if {[file executable /usr/bin/du]} {
        set NeoWebServerConf(Du) /usr/bin/du
    }
}

SAFE_proc_and_alias directory_listing {safeInterp {pattern *}} {
    global NeoWebServerConf

    set duFP [open "|$NeoWebServerConf(Du) -sk" r]
    gets $duFP line
    close $duFP

    set total [lindex $line 0]
    html "Total space used from this level down is $total Kbytes."

    html "<table border>"
    html "<tr><th>name</th><th>type</th><th>size</th><th>last mod</th></tr>"
    html "<tr><td><a href=..>..</a></td><td>directory</td><td>.</td><td>.</td>"
    foreach file [lsort [glob -nocomplain $pattern]] {
        if {[catch {file stat $file x} result] == 1} {
	    html "<tr><td>$file</td><td>$result</td></tr>"
	} else {
	    set date [clock format $x(mtime)]
	    html [format "<tr><td><a href=%s>%s</a></td><td>%s</td><td align=right>%s</td><td>%s</td></tr>\n" $file $file $x(type) $x(size) $date]
	}
    }
    html "</table>"
}

#
# guess at the number of hits we're getting
# per hour by calculating the rate of growth
# of the log file.
#
# assumes logs are in SERVER_ROOT/logs/access_log
#
safe_and_trusted_alias estimate_hits_per_hour {} {
    global server

    set hitsInInterval 1000
    set bytesPerLogEntry 131

    set fp [open "$server(SERVER_ROOT)/logs/access_log"]
    set size [fstat $fp size]
    set offset [expr $size - $hitsInInterval * $bytesPerLogEntry]
    if {$offset > 0} {
	seek $fp $offset
	gets $fp
    } else {
	set hitsInInterval [expr $size / $bytesPerLogEntry]
    }

    set result [gets $fp line]
    close $fp
    if {$result < 0} {return 0}

    set ET [expr [clock seconds] - [lindex $line 0]]
    return [expr ($hitsInInterval * 3600) / $ET]
}

#
# Return the hostname of the machine that is fetching the current
# page.  If we don't have the name because they're running the
# server with minimal DNS, perform a DNS lookup for the hostname
# and return that.
#
# If we can't figure out the hostname, we return the ip address.
#
safe_and_trusted_alias remote_hostname {{ip 0}} {
    global webenv

    if {$ip != 0} {
	if {[catch {host_info official_name $ip} result] == 1} {
	   return $ip
	}
    } else {
	if {$webenv(REMOTE_HOST) != $webenv(REMOTE_ADDR)} {
	   return $webenv(REMOTE_HOST)
	}
	if {[catch {host_info official_name $webenv(REMOTE_ADDR)} result] == 1} {
	   return $webenv(REMOTE_ADDR)
	}
    }
    return $result
}

#
# destroy the safe interpreter if it already exists, then
# create a new one and install our services (exported
# procs) into it
#
proc setup_safe_interpreter {} {
    global debugging parallelDir errorInfo NeoWebServerConf NeoWebDirConf

    if [info exists parallelDir] {
        unset parallelDir
        set errorInfo ""
    }

    if {[info exists NeoWebDirConf(Supervisor)] && $NeoWebDirConf(Supervisor)} {
	set safeInterp [interp create]
	load {} Neo $safeInterp
    } else {
	set safeInterp [interp create -safe]
    }

    load {} Tclx $safeInterp

    $safeInterp alias log_message SAFE_log_message $safeInterp

    $safeInterp alias access_data_file SAFE_access_data_file $safeInterp
    $safeInterp alias delete_data_file SAFE_delete_data_file $safeInterp

    $safeInterp alias load_cookies SAFE_load_cookies $safeInterp
    $safeInterp alias load_response SAFE_load_response $safeInterp

    $safeInterp alias incr_page_counter SAFE_incr_page_counter $safeInterp

    $safeInterp alias abort_page abort_page
    $safeInterp alias flush_page flush_page

    $safeInterp alias html html

    $safeInterp alias include_file SAFE_include_file $safeInterp
    $safeInterp alias include_virtual SAFE_include_virtual $safeInterp
    $safeInterp alias load_file SAFE_load_file $safeInterp
    $safeInterp alias load_virtual SAFE_load_virtual $safeInterp

    $safeInterp alias unquote_string www_unescape_string
    $safeInterp alias quote_string www_escape_string
    $safeInterp alias escape_sgml_chars www_escape_sgml_chars

    $safeInterp alias estimate_hits_per_hour estimate_hits_per_hour
    $safeInterp alias remote_hostname remote_hostname
    $safeInterp alias gm_timestr_822 gm_timestr_822

    extend_slave $safeInterp

    if $debugging {
	global server
	uplevel #0 source $server(SERVER_ROOT)/conf/devel.tcl
    }

    if $NeoWebServerConf(HavePostgres) {
	postgres_setup $safeInterp
    }

    if [catch {local_setup $safeInterp} errorMsg] {
	html {local_setup error}
	global errorInfo
	html $errorInfo code
    }

    if [catch {devel_setup $safeInterp} errorMsg] {
	puts stderr {devel_setup error}
	global errorInfo
	puts stderr $errorInfo
    }

    setup_slave_interp_unknown $safeInterp

    return $safeInterp
}

proc handle_subst_request {safeInterp handle} {
    set text [read $handle]
    $safeInterp eval {
	rename include_file ""
	rename include_virtual ""
	rename load_file ""
	rename load_virtual ""
	rename html ""
    }
    set html [$safeInterp eval [list subst $text]]
    html $html
    close $handle
}

#
# The code called from the interface from apache
#
# safeInterp is the interpreter we created for
# our mod_neoinclude C module when it asked us to
# (setup_safe_interpreter, above)
#
# tag and tag_val are the key-value pairs in the server
# side include
#
proc handle_neoscript_request {safeInterp tag tag_val} {
    global debugging errorInfo

    switch $tag {
	"return" {
	    handle_server_side_return $safeInterp $tag_val
	}
	"code" {
	    handle_server_side_eval $safeInterp $tag_val
	}
	"eval" {
	    handle_server_side_eval $safeInterp $tag_val
	}
	"var" {
	    handle_server_side_variable $safeInterp $tag_val
	}
	"expr" {
	    handle_server_side_expr $safeInterp $tag_val
	}
	default {
	    html "\[Error -- unrecognized tag '$tag' in neoscript directive\]"
	}
    }

    return
}

#
# The code called from handle_neoscript_request to do an eval=
#
proc handle_server_side_eval {safeInterp ssiCode} {
    global debugging errorInfo

    if {[catch {$safeInterp eval $ssiCode} result] == 1} {
	html "\{NEOWEBSCRIPT ERROR: $result\}"
	if {$debugging} {
	    html "\n<pre>$errorInfo</pre>"
	}
    }
    return
}

#
# The code called from handle_neoscript_request to do a return=
#
proc handle_server_side_return {safeInterp ssiCode} {
    global debugging errorInfo

    if {[catch {$safeInterp eval $ssiCode} result] == 1} {
	html "\{NEOWEBSCRIPT ERROR: $result\}"
	if {$debugging} {
	    html "\n<pre>$errorInfo</pre>"
	}
    } else {
	if {$result != ""} {
	    html $result
	}
    }
    return
}

#
# The code called from handle_neoscript_request to do a var=
#
proc handle_server_side_variable {safeInterp ssiVar} {
    global debugging

    if {[catch {$safeInterp eval set $ssiVar} result] == 1} {
	html "\{NEOWEBSCRIPT ERROR: $result\}"
	if {$debugging} {
	    html "\n<pre>$errorInfo</pre>"
	}
    } else {
	if {$result != ""} {
	    html $result
	}
    }
    return
}

#
# The code called from handle_neoscript_request to do a var=
#
proc handle_server_side_expr {safeInterp ssiExpr} {
    global debugging

    if {[catch {$safeInterp eval expr $ssiExpr} result] == 1} {
	html "\{NEOWEBSCRIPT ERROR: $result\}"
	if {$debugging} {
	    html "\n<pre>$errorInfo</pre>"
	}
    } else {
	if {$result != ""} {
	    html $result
	}
    }
    return
}

#
# The code called when generating an image is requested
#
proc handle_image_request {safeInterp} {
    package require Image
    return [send_image_request $safeInterp]
}

package ifneeded Image 1.0 {
    source $webenv(SERVER_ROOT)/conf/image.tcl
}

proc setup_slave_interp_unknown {safeInterp} {
    # Experimental implementation of unknown
    global webenv SafePaths auto_path NeoWebDirConf UnknownTcl
    set local_lib_path [file dirname [www_simplify_pathname $webenv(SCRIPT_FILENAME)]]
    set SafePaths [concat $auto_path $local_lib_path]
    $safeInterp eval "set auto_path [list $SafePaths]"
    $safeInterp eval $UnknownTcl
    if [interp issafe $safeInterp] {
	$safeInterp alias source SAFE_source $safeInterp
	$safeInterp alias open SAFE_open $safeInterp
	$safeInterp alias glob SAFE_glob $safeInterp
	$safeInterp alias read_file SAFE_read_file $safeInterp
	$safeInterp alias file file
    }
    $safeInterp alias SAFE_autoload SAFE_autoload $safeInterp
    $safeInterp alias SAFE_pkgUnknown SAFE_pkgUnknown $safeInterp
    $safeInterp eval {
	unknown_handler sunk slave_unknown
	proc slave_unknown {args} {
	    set cmd [lindex $args 0]
	    if [SAFE_autoload $cmd] {
		return [uplevel $args]
	    }
	    return -code continue
	}
	if [interp issafe] {
	    package unknown neoweb_pkgUnknown
	    proc neoweb_pkgUnknown {args} {
		eval SAFE_pkgUnknown $args
	    }
	}
    }
}

proc SAFE_autoload {safeInterp cmd} {
    global safe_proc_cache safe_alias_cache
    if [info exists safe_proc_cache($cmd)] {
	$safeInterp eval $safe_proc_cache($cmd)
	return 1
    }
    if [info exists safe_alias_cache($cmd)] {
	eval $safeInterp alias $cmd [subst $safe_alias_cache($cmd)]
	return 1
    }
    return 0
}

proc SAFE_pkgUnknown {safeInterp name {version ""}} {
    global SafeIfNeeded
    if [info exists SafeIfNeeded($name)] {
	global webenv
	eval $SafeIfNeeded($name)
	return
    }
    global auto_path SafePaths
    set auto_length [llength $auto_path]
    if [catch {eval package require $name $version}] return
    set SafePaths [concat $SafePaths [lrange $auto_path $auto_length end]]
    foreach pkginfo [info loaded] {
	lassign $pkginfo lpath lname
	if {$lname == $name} {
	    load $lpath $lname $safeInterp
	    return
	}
    }
    return
}

proc safe_pkgIfNeeded {name body} {
    global SafeIfNeeded
    set SafeIfNeeded($name) $body
}

proc read_file {filename} {
    set fp [open $filename]
    set data [read $fp]
    close $fp
    return $data
}

if {[info commands import] == ""} {
    set UnknownTcl [read_file $server(SERVER_ROOT)/conf/unknown.tcl]
} else {
    set UnknownTcl [read_file $server(SERVER_ROOT)/conf/unknown.itcl]
}
lappend auto_path $server(SERVER_ROOT)/neoscript-tcl

proc SAFE_filename {filename} {
    global webenv SafePaths
    if {[cindex $filename 0] != "/"} {
	set filename [file join [file dirname $webenv(SCRIPT_FILENAME)] $filename]
    }
    set filename [www_simplify_pathname $filename]
    if {[lsearch -exact $SafePaths [file dirname $filename]] < 0} {
        return 
    }
    #if ![isfile_safe $filename] return

    return $filename
}

proc SAFE_open {safeInterp filename args} {
    set nFilename [SAFE_filename $filename]

    if [lempty $nFilename] {
	return -code error "$filename: Invalid pathname requested"
    }
    set channel [open $nFilename]
    interp transfer {} $channel $safeInterp
    return $channel
}

proc SAFE_source {safeInterp filename} {
    set nFilename [SAFE_filename $filename]

    if [lempty $nFilename] {
	return -code error "$filename: Invalid pathname requested"
    }
    $safeInterp eval [read_file $filename]
}

proc isfile_safe {fileName} {
    global SafePaths
    if {[cindex $fileName 0] == "/" || [string match *..* $fileName] || [string match */.* $fileName] || [string match *~* $fileName]} {
	return 0
    }
    return 1
}

proc SAFE_glob {safeInterp args} {
    foreach pattern $args {
	if ![isfile_safe $pattern] {
            return -code error "Illegal filename: $pattern"
        }
    }
    return [eval glob $args]
}

proc SAFE_read_file {safeInterp filename} {
	set nFilename [SAFE_filename $filename]

	if [lempty $nFilename] {
        return -code error "Illegal filename: $filename"
    }
    return [read_file $filename]
}   

#
# handle_neoscript_mime_upload
#
# Called from C to setup for MIME uploads.  Is expected to create
# a unique file to receive the data in, open it for write, write
# out the content type and content length, and return the name
# of the file it used.
#
# The C code then takes this filename, opens it for append, and copies
# in all of the incoming data.  Once it's in, the C code calls
# finish_neoscript_mime_upload
#
proc handle_neoscript_mime_upload {} {
    global webenv mimeFilename mimeResponse mimeUploadedFiles

    set mimeFilename "/tmp/neoscript.mime.[clock seconds].[id process]"
    set fp [open $mimeFilename w]
    puts $fp "Content-Type: $webenv(CONTENT_TYPE)"
    puts $fp "Content-Length: $webenv(CONTENT_LENGTH)"
    puts $fp ""
    close $fp
    return $mimeFilename
}

#
# finish_neoscript_mime_upload
#
# We've received a MIME upload, we now use our webunpack tool, which
# is a customized version of the CMU mpack code, to crack it out into
# a directory.  Set up the directory, dig the names of the fields out
# of the "andTheRest" part.  Put the results in the mime response and
# mime uploaded files arrays.
#
proc finish_neoscript_mime_upload {} {
    global webenv mimeFilename mimeResponse mimeUploadedFiles mimeUnpackDir
    global webunpackPath

    set mimeUnpackDir $mimeFilename.dir
    mkdir $mimeUnpackDir

    set unpackInfo [exec $webunpackPath $mimeUnpackDir <$mimeFilename]

    foreach line [split $unpackInfo "\n"] {
	set andTheRest [join [lassign $line tempName contentType fieldType requestedFileName]]

	if ![regexp {[Nn][Aa][Mm][Ee]="([^"]*)"} $andTheRest dummy name] {
	    error "mime upload failed to find name field in '$andTheRest'"
	}

	if [string match *\* $requestedFileName] {
	    set requestedFileName [lindex [split $requestedFileName \\] end]
	}

	set fieldTempFileName $mimeUnpackDir/$tempName
	if {$fieldType == "field"} {
	    set mimeResponse($name) [read_file $fieldTempFileName]
	    unlink $fieldTempFileName
	} elseif {$fieldType == "file"} {
	    set mimeUploadedFiles($name) [list $tempName $requestedFileName [file size $fieldTempFileName] $contentType]
	} else {
	    error "unrecognized field type '$fieldType' processing mime file $mimeFilename"
	}
    }
}

proc cleanup_neoscript_mime_upload {} {
    global mimeFilename mimeUnpackDir

    exec rm -rf $mimeFilename $mimeUnpackDir
}

# 
# load_mime_response
# 
# Load the mime response variables into the safe interpreter and returns a
# list of input field names (also stored in the global array "filesUploaded"
# as each array element).
# 
SAFE_proc_and_alias load_mime_response {safeInterp {responseVar "response"} {fileInfoVar "filesUploaded"}} {
    global webenv mimeResponse mimeUploadedFiles

    $safeInterp eval array set $responseVar [list [array get mimeResponse]]
    $safeInterp eval array set $fileInfoVar [list [array get mimeUploadedFiles]]

    return [array names mimeUploadedFiles]
}

#
# save_mime_file
# 
# (no longer needed, see open_mime_file)
# 
SAFE_proc_and_alias save_mime_file {safeInterp fieldName dataFileName} {
    global mimeUploadedFiles mimeUnpackDir

    if {![info exists mimeUploadedFiles($fieldName)]} {
	error "mime upload file for field name '$fieldName' failed -- no such field"
    }

    setup_data_access
    set tempName [lindex $mimeUploadedFiles($fieldName) 0]
    ul_name_check $dataFileName data mimeSaveFileName
    create_user_dir data
    exec cp $mimeUnpackDir/$tempName $mimeSaveFileName
}

#
# open_mime_file
# 
# Takes an input field name (see load_mime_response), and returns a channel
# identifier (i.e. a file handle) to the MIME-uploaded file.
#
SAFE_proc_and_alias open_mime_file {safeInterp fieldName} {
    global mimeUploadedFiles mimeUnpackDir parallelDir errorCode

    if ![info exists mimeUploadedFiles($fieldName)] {
        error "mime upload file for field name '$fieldName' failed -- no such file"
    }

    setup_data_access
    set tempName [lindex $mimeUploadedFiles($fieldName) 0]
    set mimeSaveFileName $mimeUnpackDir/$tempName

    if {[catch {set fp [open $mimeSaveFileName "CREAT RDWR" 0660]}] == 1} {
        error "$logFileName: $errorCode"
    }
    interp transfer "" $fp $safeInterp
    return $fp
}
