// MAM - The Modeling and Animation Machine 
// VRS - The Virtual Rendering System 
// Copyright (c) 1994-96 Juergen Doellner
// Institut fuer Informatik, Westfaelische Wilhelms Universitaet Muenster 
// See the LICENSE file in this distribution.

#ifndef MAM_SHAREDOBJ_H
#define MAM_SHAREDOBJ_H

#include<stdlib.h>
#include<iostream.h>
#include<assert.h>
#include<string.h>

//- SharedObj
class SharedObj {
//. Shareable Object Base Class
public:
    //- SharedObj
    SharedObj();
    SharedObj(const SharedObj&);
    virtual ~SharedObj();

    //- ClassName, className, isA
    static const char* ClassName(); 
    virtual const char* className() const; 
    virtual int isA(const char*) const;
	//. Shared objects provide runtime type information. Their class
	//. name and base class relationships can be inquired.

    //- ref, unref, unrefDeferred
    virtual void ref() const;
    virtual void unref() const;
    virtual void unrefDeferred() const;
    virtual unsigned int references() const;
    static void flush();
	//. Shareable objects maintain a reference counter. Objects which
	//. hold a reference or pointer to a shareable object increment
	//. its reference counter. If the object is not refered any more,
	//. the reference counter must be decremented. If the reference
	//. counter gets zero, the object is deleted. The deletion can be
	//. deferred. Deferred dereferenced objects are deleted by
	//. 'SharedObj::flush'.

    //- ref, unref, unrefDeferred
    static void ref(const SharedObj*);
    static void unref(const SharedObj*);
    static void unrefDeferred(const SharedObj*);
        //. reference management for null objects
        //. The static reference methods are the same as the non-static, except
        //. that they are ignored if a null object pointer is given. 
private:
    SharedObj& operator=(const SharedObj&);

private:
    unsigned int references_;

    static SharedObj** queue_;
    static unsigned int size_;
    static unsigned int current_;
};


//- SO
template<class T>
class SO {
//. Smart Pointer for Shareable Objects
public:
    //- SO 
    SO() { obj_ = NULL; }
    SO(const SO<T>& so) { 
	obj_ = so.obj_;
	SharedObj::ref((const SharedObj*) obj_);  // Tobias: typecast
    }
    SO(T* o) { obj_ = o; SharedObj::ref((const SharedObj*) obj_); } // Tobias: typecast
    ~SO() { SharedObj::unref((const SharedObj*)obj_); } // Tobias: typecast
        //. SO<T> represents a pointer to an object of type T.
	//. The object is referenced if the pointer is constructed.
	//. The object is dereferenced if the pointer is destructed.
        //. The default constructor sets the pointer to null. If a 
	//. shareable object is given, it is stored and referenced.

    //- operator= 
    SO<T>& operator=(T* t) {
	SharedObj::ref((const SharedObj*)t); SharedObj::unref((const SharedObj*)obj_); obj_ = t; // Tobias: typecasts
	return *this;	
    }
    SO<T>& operator=(const SO<T>& vrs) {
        SharedObj::ref((const SharedObj*)vrs.obj_); SharedObj::unref((const SharedObj*)obj_); obj_ = vrs.obj_; // Tobias: typecast
        return *this;
    }
	//. If a shared object is assigned, the current object is dereferenced,
	//. and the assign object is referenced.

    //- operator==, operator!=
    int operator==(const T* t) const { return obj_==t; }
    int operator!=(const T* t) const { return obj_!=t; }
        //. Comparison.

    //- operator-> 
    T* operator->() { return obj_; }
    const T* operator->() const { return obj_; }
    operator T* () const { return obj_; }
	//. SO<T> objects can convert to constant references and to non-constant
	//. references of the shared object they store. Implicit conversion
	//. to the type of the shared object is provided. 
    T& operator*() { return *obj_; }
private:
    T* obj_;
};


//
// RTTI RUNTIME TYPE INFORMATION MACROS
//
#define RTTI\
    virtual int isA(const char*) const; \
    virtual const char* className() const; \
    static const char* ClassName();

#define CLASS_NAME(T) \
extern const char* MAM_NAME(T*); 


//
// RTTI Implementation
//
#define RTTI_IMPL(A,B) \
const char* A::ClassName() { \
    static const char* ClassName_ = #A; \
    return ClassName_; \
} \
const char* A::className() const { \
    return A::ClassName(); \
} \
\
int A::isA(const char* name) const { \
    return (strcmp(name, #A)==0) || B::isA(name); \
} \
const char* MAM_NAME(A*) { return A::ClassName(); } \
\
SharedObj* A##Ct() { return new A(); } \
\
static int A##classInit = Factory::registerClass(A::ClassName(), A##Ct); \
 
#define CLASSRELATION(A,B) \
const char* A::ClassName() { \
    static const char* ClassName_ = #A; \
    return ClassName_; \
} \
const char* A::className() const { \
    return A::ClassName(); \
} \
\
int A::isA(const char* name) const { \
    return (strcmp(name, #A)==0) || B::isA(name); \
} \
const char* MAM_NAME(A*) { return A::ClassName(); }


#define TEMPLATERELATION(A,B) \
template<class T> \
const char* A<T>::className() const { \
    return A<T>::ClassName(); \
} \
template<class T> \
const char* A<T>::ClassName() { \
    return NULL; \
    static char* theName_ = 0; \
    if(theName_==0) {  \
        char tempname[255]; \
        sprintf(tempname, #A "<%s>", MAM_NAME((T*)0) ); \
        theName_ = new char[strlen(tempname)+1]; \
        strcpy(theName_, tempname); \
    } \
    return theName_; \
} \
\
template<class T> \
int A<T>::isA(const char* name) const { \
    return (strcmp(name, A<T>::className())==0) || B::isA(name); \
}


extern const char* MAM_NAME(SharedObj*);
extern const char* MAM_NAME(int*);
extern const char* MAM_NAME(float*);
extern const char* MAM_NAME(double*);
extern const char* MAM_NAME(char*);
extern const char* MAM_NAME(char**);

#include "base/massert.h"


#endif
