/*
 * radiusd.h	Structures, prototypes and global variables
 *		for the Cistron Radius server.
 *
 * Version:	@(#)radiusd.h  2.00  19-Jul-1999  miquels@cistron.nl
 *
 */

#include "radius.h"
#include "conf.h"
#include "token.h"

typedef	unsigned int UINT4;

/* Server data structures */

typedef struct dict_attr {
	char			name[32];
	int			value;
	int			type;
	int			vendor;
	struct dict_attr	*next;
} DICT_ATTR;

typedef struct dict_value {
	char			attrname[32];
	char			name[32];
	int			value;
	struct dict_value	*next;
} DICT_VALUE;

typedef struct dict_vendor {
	char			vendorname[32];
	int			vendorpec;
	int			vendorcode;
	struct dict_vendor	*next;
} DICT_VENDOR;

typedef struct value_pair {
	char			name[32];
	int			attribute;
	int			type;
	int			length; /* of strvalue */
	UINT4			lvalue;
	int			operator;
	int			addport;
	char			strvalue[AUTH_STRING_LEN];
	struct value_pair	*next;
} VALUE_PAIR;

typedef struct radius_packet {
	char			username[AUTH_STRING_LEN];
	UINT4			src_ipaddr;
	UINT4			dst_ipaddr;
	u_short			src_port;
	u_short			dst_port;
	int			id;
	int			code;
	char			secret[32];
	char			vector[16];
	char			digest[16];
	time_t			timestamp;
	int			issigned;
	u_char			*data;		/* Raw received/sent data */
	int			data_len;
	VALUE_PAIR		*vps;
} RADIUS_PACKET;

typedef struct proxy_struct {
	/* Proxy support fields */
	u_char			realm[32];
	int			validated;	/* Already md5 checked */
	UINT4			server_ipaddr;
	UINT4			server_id;
	VALUE_PAIR		*server_reply;	/* Reply from other server */
	int			server_code;	/* Reply code from other srv */
} PROXY_DATA;

typedef struct auth_req {
	RADIUS_PACKET		*packet;
	PROXY_DATA		*proxy;
	int			child_pid;	/* Process ID of child */
	time_t			timestamp;
	struct auth_req		*next;		/* Next active request */
} REQUEST;

typedef struct client {
	UINT4			ipaddr;
	char			longname[256];
	u_char			secret[32];
	char			shortname[32];
	struct client		*next;
} CLIENT;

typedef struct nas {
	UINT4			ipaddr;
	char			longname[256];
	char			shortname[32];
	char			nastype[32];
	struct nas		*next;
} NAS;

typedef struct realm {
	char			realm[64];
	char			server[64];
	UINT4			ipaddr;
	int			auth_port;
	int			acct_port;
	int			striprealm;
	struct realm		*next;
} REALM;

enum {
  PW_OPERATOR_EQUAL = 0,	/* = */
  PW_OPERATOR_NOT_EQUAL,	/* != */
  PW_OPERATOR_LESS_THAN,	/* < */
  PW_OPERATOR_GREATER_THAN,	/* > */
  PW_OPERATOR_LESS_EQUAL,	/* <= */
  PW_OPERATOR_GREATER_EQUAL,	/* >= */
};

#define DEBUG	if(debug_flag)log_debug
#define DEBUG2  if (debug_flag > 1)log_debug

#define SECONDS_PER_DAY		86400
#define MAX_REQUEST_TIME	30
#define CLEANUP_DELAY		5
#define MAX_REQUESTS		255

#define L_DBG			1
#define L_AUTH			2
#define L_INFO			3
#define L_ERR			4
#define L_PROXY			5
#define L_CONS			128

#define VENDOR(x) (x >> 16)

/*
 *	Global variables.
 */
extern char		*recv_buffer;
extern char		*send_buffer;
extern char		*progname;
extern int		debug_flag;
extern char		*radacct_dir;
extern char		*radius_dir;
extern UINT4		expiration_seconds;
extern UINT4		warning_seconds;
extern int		radius_pid;
extern int		use_dbm;
extern int		log_stripped_names;
extern int		cache_passwd;
extern UINT4		myip;
extern UINT4		warning_seconds;
extern int		log_auth_detail;
extern int		log_auth;
extern int		log_auth_pass;
extern int		auth_port;
extern int		acct_port;

/*
 *	Function prototypes.
 */

/* acct.c */
int		rad_accounting(REQUEST *, int);
int		rad_accounting_orig(REQUEST *, int, char *);
int		radzap(UINT4 nas, int port, char *user, time_t t);
char		*uue(void *);
int		rad_check_multi(char *name, VALUE_PAIR *request, int maxsimul);

/* attrprint.c */
void		fprint_attr_list(FILE *, VALUE_PAIR *);
void		fprint_attr_val(FILE *, VALUE_PAIR *);

/* dict.c */
int		dict_init(char *);
DICT_ATTR	*dict_attrget(int);
DICT_ATTR	*dict_attrfind(char *);
DICT_VALUE	*dict_valfind(char *);
DICT_VALUE	*dict_valget(UINT4 value, char *);
int		dict_vendorcode(int);
int		dict_vendorpec(int);

/* md5.c */

void		md5_calc(u_char *, u_char *, u_int);

/* radiusd.c */
int		radius_exec_program(char *, VALUE_PAIR *, VALUE_PAIR **, int, char **user_msg);
void		debug_pair(FILE *, VALUE_PAIR *);
int		log_err (char *);
void		sig_cleanup(int);

/* util.c */
char *		ip_hostname (UINT4);
UINT4		ip_getaddr (char *);
char *		ip_ntoa(char *, UINT4);
UINT4		ip_addr(char *);
struct passwd	*rad_getpwnam(char *);
#if defined (sun) && defined(__svr4__)
void		(*sun_signal(int signo, void (*func)(int)))(int);
#define signal sun_signal
#endif

/* radius.c */
int		rad_send_reply(int, REQUEST *, VALUE_PAIR *, char *, int);
REQUEST	*radrecv (UINT4, u_short, u_char *, int);
int		calc_digest (u_char *, REQUEST *);
int		calc_acctdigest(u_char *digest, REQUEST *request);
void		request_free(REQUEST *request);

/* valuepair.c */
VALUE_PAIR	*paircreate(int attr, int type);
void		pairfree(VALUE_PAIR *);
VALUE_PAIR	*pairfind(VALUE_PAIR *, int);
void		pairdelete(VALUE_PAIR **, int);
void		pairadd(VALUE_PAIR **, VALUE_PAIR *);
VALUE_PAIR	*paircopy(VALUE_PAIR *vp);
void		pairmove(VALUE_PAIR **to, VALUE_PAIR **from);
void		pairmove2(VALUE_PAIR **to, VALUE_PAIR **from, int attr);
VALUE_PAIR	*pairread(char **ptr, int *eol);
int		userparse(char *buffer, VALUE_PAIR **first_pair);

/* files.c */
int		user_find(char *name, VALUE_PAIR *,
				VALUE_PAIR **, VALUE_PAIR **);
void		presuf_setup(VALUE_PAIR *request_pairs);
int		hints_setup(VALUE_PAIR *request_pairs);
int		huntgroup_access(VALUE_PAIR *request_pairs);
CLIENT		*client_find(UINT4 ipno);
char		*client_name(UINT4 ipno);
int		read_clients_file(char *);
REALM		*realm_find(char *);
NAS		*nas_find(UINT4 ipno);
char		*nas_name(UINT4 ipno);
char		*nas_name2(REQUEST *r);
char		*auth_name(REQUEST *request, int do_cid);
int		read_naslist_file(char *);
int		read_config_files(void);
int		presufcmp(VALUE_PAIR *check, char *name, char *rest);

/* version.c */
void		version();

/* log.c */
int		log(int, char *, ...);
int		log_debug(char *, ...);

/* pam.c */
#ifdef WITH_PAM
int		pam_pass(char *name, char *passwd, const char *pamauth);
#define PAM_DEFAULT_TYPE    "radius"
#endif

/* proxy.c */
int proxy_send(REQUEST *request, int activefd);
int proxy_receive(REQUEST *request, int activefd);

/* auth.c */
int		rad_auth_init(REQUEST *request, int activefd);
int		rad_authenticate (REQUEST *, int);

/* exec.c */
char		*radius_xlate(char *, VALUE_PAIR *req, VALUE_PAIR *reply);

/* timestr.c */
int		timestr_match(char *, time_t);

