/*
 *
 *	RADIUS Accounting
 *	Remote Authentication Dial In User Service
 *
 *
 *	Livingston Enterprises, Inc.
 *	6920 Koll Center Parkway
 *	Pleasanton, CA   94566
 *
 *	Copyright 1992 - 1994 Livingston Enterprises, Inc.
 *
 *	Permission to use, copy, modify, and distribute this software for any
 *	purpose and without fee is hereby granted, provided that this
 *	copyright and permission notice appear on all copies and supporting
 *	documentation, the name of Livingston Enterprises, Inc. not be used
 *	in advertising or publicity pertaining to distribution of the
 *	program without specific prior permission, and notice be given
 *	in supporting documentation that copying and distribution is by
 *	permission of Livingston Enterprises, Inc.   
 *
 *	Livingston Enterprises, Inc. makes no representations about
 *	the suitability of this software for any purpose.  It is
 *	provided "as is" without express or implied warranty.
 *
 */

static char sccsid[] =
"@(#)acct.c	1.6  Copyright 1994 Livingston Enterprises Inc";


#include	<sys/types.h>
#include	<sys/socket.h>
#include	<sys/time.h>
#include	<sys/file.h>
#include	<netinet/in.h>

#include	<stdio.h>
#include	<netdb.h>
#include	<pwd.h>
#include	<time.h>
#include	<ctype.h>
#include	<unistd.h>
#include	<signal.h>
#include	<errno.h>
#include	<sys/wait.h>

#include	"radiusd.h"


#if defined(RADUTMP_VER) && RADUTMP_VER == 2

/*
 *	We haven't finished version 2 accounting yet.
 */

#include "radutmp.h"

/*
 *	Show framed users in the RADIUS utmp file.
 */
int rad_accounting(authreq, activefd)
AUTH_REQ	*authreq;
int		activefd;
{
	struct radutmp utu;
	VALUE_PAIR *vp;
	int status = -1;
	char *s;
	char buf[1024];
	FILE *fp;
	time_t t;
	int fd;
	int vj = 0;

	time(&t);
	memset(&ut, 0, sizeof(ut));
	ut.ut_time = t;

	/*
	 *	First, find the interesting attributes.
	 */
	for (vp = authreq->request; vp; vp = vp->next) {
		switch (vp->attribute) {
			case PW_USER_NAME:
				/*
				 *	XXX - Hack hack :)
				 */
				s = vp->strvalue;
				while(*s == 'S' || *s == 'P' ||
					*s == 'C' || *s == 'T') s++;
				strncpy(ut.login, s, sizeof(ut.login));
				break;
			case PW_ACCT_STATUS_TYPE:
				status = vp->lvalue;
				break;
			case PW_FRAMED_IP_ADDRESS:
			case PW_LOGIN_IP_HOST:
				ut.nas_ip = vp->lvalue;
				break;
			case PW_LOGIN_SERVICE:
				ut.proto = lproto(vp->lvalue);
				break;
			case PW_FRAMED_PROTOCOL:
				ut.proto = fproto(vp->lvalue);
				break;
			case PW_NAS_IP_ADDRESS:
				ut.server_ip = vp->lvalue;
				break;
			case PW_NAS_PORT_ID:
				ut.port = vp->lvalue;
				break;
			case PW_FRAMED_COMPRESSION:
				vj++;
				break;
			case PW_ACCT_DELAY_TIME:
				ut.ut_time -= ntohl(vp->lvalue);
				break;
			case PW_ACCT_SESSION_TIME:
				ut.length = ntohl(vp->lvalue);
				break;
		}
	}

	if (vj && ut.proto == P_SLIP) ut.proto = P_CSLIP;

	/*
	 *	Fill out the UTMP struct for the radwtmp file.
	 *	(this one must be "last" - compatible).
	 */
#ifdef __linux__
	/*
	 *	Linux has a field for the client address.
	 */
	wt.ut_addr = nas_address;
#endif
	/*
	 *	We use the tty field to store the terminal servers' port
	 *	and address so that the tty field is unique.
	 */
	if (framed_address)
		sprintf(buf, "%02d:%s", nas_port, ip_hostname(framed_address));
	else
		sprintf(buf, "%02d", nas_port);
	strncpy(wt.ut_line, buf, UT_LINESIZE);

	/*
	 *	In the wtmp file, we use the hostname
	 *	field for the protocol.
	 */
	if (protocol == PW_PPP)
		strcpy(wt.ut_host, "PPP");
	else if (protocol == PW_SLIP)
		strcpy(wt.ut_host, "SLIP");
	else
		strcpy(wt.ut_host, "TERM");
	wt.ut_time = t;
#ifdef __linux__
	/*
	 *	And we can use the ID field to store
	 *	the portnumber, in Linux at least.
	 */
	sprintf(buf, "%d", nas_port);
	strncpy(wt.ut_id, buf, 2);
	wt.ut_type = status == PW_STATUS_START ? USER_PROCESS : DEAD_PROCESS;
#endif
	if (status == PW_STATUS_STOP)
		wt.ut_name[0] = 0;

	/*
	 *	Enter into the radutmp file.
	 */
	if (ut.ut_name[0] && (fd = open(RADUTMP, O_RDWR)) >= 0) {

	    if (status == PW_STATUS_STOP || status == PW_STATUS_START) {

		/*
		 *	Delete all entries having our server/port
		 */
		while(read(fd, &u, sizeof(u)) == sizeof(u)) {
			if (strncmp(ut.ut_line, u.ut_line, RUT_LINESIZE) == 0 &&
			    strncmp(ut.ut_serv, u.ut_serv, RUT_SERVSIZE) == 0) {
				memset(&u, 0, sizeof(u));
				lseek(fd, -sizeof(u), SEEK_CUR);
				write(fd, &u, sizeof(u));
			}
		}
	    }
	    if (status == PW_STATUS_START) {
		/*
		 *	Find a free slot.
		 */
		lseek(fd, 0, SEEK_SET);
		while(read(fd, &u, sizeof(u)) == sizeof(u)) {
			if (u.ut_name[0] == 0 && u.ut_line[0] == 0) {
				lseek(fd, -sizeof(u), SEEK_CUR);
				break;
			}
		}
		/*
		 *	Write it.
		 */
		write(fd, &ut, sizeof(ut));
	    }
	    close(fd);
	}

	/*
	 *	Now write a RADIUS wtmp log file.
	 */
	if ((fp = fopen(RADWTMP, "a")) != NULL) {
		fwrite(&wt, sizeof(wt), 1, fp);
		fclose(fp);
	}

	/*
	 *	Now send back an ACK to the NAS.
	 */
	send_acct_reply(authreq, (VALUE_PAIR *)NULL, 
			(char *)NULL,activefd);
	pairfree(authreq->request);
	memset(authreq, 0, sizeof(AUTH_REQ));
	free(authreq);

	return 0;
}
#endif /* RADUTMP_VER == 2 */

#if defined(RADUTMP_VER) && RADUTMP_VER == 1

#include "radutmp.h"

static char trans[64] =
   "ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789+/";
#define ENC(c) trans[c]

/*
 *	UUencode 4 bits. We use this to turn a 4 byte field
 (	(an IP adres) into 6 bytes of ASCII.
 */
static char *uue(void *in)
{
	int i;
	static unsigned char res[7];
	unsigned char *data = (char *)in;

	res[0] = ENC(data[0] >> 2);
	res[1] = ENC((data[0] << 4) & 060) + ((data[1] >> 4) & 017);
	res[2] = ENC((data[1] << 2) & 074) + ((data[2] >> 6) & 03);
	res[3] = ENC(data[2] & 077);

	res[4] = ENC(data[3] >> 2);
	res[5] = ENC((data[3] << 4) & 060);
	res[6] = 0;

	for(i = 0; i < 6; i++) {
		if (res[i] == ' ') res[i] = '`';
		if (res[i] < 32 || res[i] > 127)
			printf("uue: protocol error ?!\n");
	}
	return res;
}


static char *shortname(UINT4 ip)
{
	FILE	*clientfd;
	u_char	buffer[128];
	char	hostnm[256];
	u_char	secret[64];
	static	char	shortnm[64];
	UINT4	ipaddr;
	UINT4	get_ipaddr();
	char	*ip_hostname();

	/* Find the client in the database */
	sprintf(buffer, "%s/%s", radius_dir, RADIUS_CLIENTS);
	if((clientfd = fopen(buffer, "r")) == (FILE *)NULL) {
		fprintf(stderr, "%s: couldn't open %s to find clients\n",
				progname, buffer);
		return(NULL);
	}
	ipaddr = (UINT4)0;
	while(fgets(buffer, sizeof(buffer), clientfd) != (char *)NULL) {
		if(*buffer == '#') {
			continue;
		}
		shortnm[0] = 0;
		if(sscanf(buffer, "%s%s%s", hostnm, secret, shortnm) != 3) {
			continue;
		}
		ipaddr = get_ipaddr(hostnm);
		if(ipaddr == ip) {
			break;
		}
	}
	fclose(clientfd);

	return shortnm[0] ? shortnm : NULL;
}

/*
 *	Show framed users in the RADIUS utmp file.
 */
int rad_accounting(authreq, activefd)
AUTH_REQ	*authreq;
int		activefd;
{
	struct radutmp ut, u;
	struct utmp wt;
	VALUE_PAIR *vp;
	int status = -1;
	UINT4 framed_address = 0, nas_address = 0;
	int nas_port = 32;
	unsigned int ip;
	char *s;
	int protocol = -1;
	char buf[1024];
	char *proto;
	FILE *fp;
	time_t t;
	int fd;

	/*
	 *	Add any specific attributes for this username.
	 */
	hints_setup(authreq->request);

	time(&t);
	memset(&ut, 0, sizeof(ut));
	memset(&wt, 0, sizeof(wt));

	/*
	 *	First, find the interesting attributes.
	 */
	for (vp = authreq->request; vp; vp = vp->next) {
		switch (vp->attribute) {
			case PW_USER_NAME:
				strncpy(ut.ut_name, vp->strvalue, RUT_NAMESIZE);
				strncpy(wt.ut_name, vp->strvalue, UT_NAMESIZE);
				break;
			case PW_ACCT_STATUS_TYPE:
				status = vp->lvalue;
				break;
			case PW_FRAMED_IP_ADDRESS:
				framed_address = vp->lvalue;
				break;
			case PW_FRAMED_PROTOCOL:
				protocol = vp->lvalue;
				break;
			case PW_NAS_IP_ADDRESS:
				nas_address = vp->lvalue;
				break;
			case PW_NAS_PORT_ID:
				nas_port = vp->lvalue;
				break;
		}
	}

	/*
	 *	Fill out the rest of the RADUTMP struct.
	 */
	if (nas_address) {
		sprintf(buf, "%s", ip_hostname(nas_address));
		strncpy(ut.ut_serv, buf, RUT_SERVSIZE);
	}
	if (framed_address) {
		sprintf(buf, "%s", ip_hostname(framed_address));
		strncpy(ut.ut_host, buf, RUT_HOSTSIZE);
	}
	if (protocol == PW_PPP)
		ut.ut_id[0] = 'P';
	else if (protocol == PW_SLIP)
		ut.ut_id[0] = 'S';
	else
		ut.ut_id[0] = 'T';
	ut.ut_time = t;
	sprintf(ut.ut_line, "S%d", nas_port);

	/*
	 *	Fill out the UTMP struct for the radwtmp file.
	 *	(this one must be "last" - compatible).
	 */
#ifdef __linux__
	/*
	 *	Linux has a field for the client address.
	 */
	wt.ut_addr = htonl(nas_address);
#endif
	/*
	 *	We use the tty field to store the terminal servers' port
	 *	and address so that the tty field is unique.
	 */
	ip = htonl(nas_address);
	s  = shortname(nas_address);
	if (s == NULL) s = uue(&ip);
#if UT_LINESIZE > 8
	sprintf(buf, "%03d:%s", nas_port, s);
#else
	sprintf(buf, "%02d%s", nas_port, s);
#endif
	strncpy(wt.ut_line, buf, UT_LINESIZE);

	/*
	 *	We store the dynamic IP address in the hostname field.
	 */
	if (framed_address)
		strncpy(wt.ut_host, ip_hostname(framed_address), UT_HOSTSIZE);
#ifdef __svr4__
	wt.ut_xtime = t;
#else
	wt.ut_time = t;
#endif
#ifdef USER_PROCESS
	/*
	 *	And we can use the ID field to store
	 *	the protocol.
	 */
	if (protocol == PW_PPP)
		strcpy(wt.ut_id, "P");
	else if (protocol == PW_SLIP)
		strcpy(wt.ut_id, "S");
	else
		strcpy(wt.ut_id, "T");
	wt.ut_type = status == PW_STATUS_START ? USER_PROCESS : DEAD_PROCESS;
#endif
	if (status == PW_STATUS_STOP)
		wt.ut_name[0] = 0;

	/*
	 *	Enter into the radutmp file.
	 */
	if (ut.ut_name[0] && (fd = open(RADUTMP, O_RDWR)) >= 0) {

	    if (status == PW_STATUS_STOP || status == PW_STATUS_START) {

		/*
		 *	Delete all entries having our server/port
		 */
		while(read(fd, &u, sizeof(u)) == sizeof(u)) {
			if (strncmp(ut.ut_line, u.ut_line, RUT_LINESIZE) == 0 &&
			    strncmp(ut.ut_serv, u.ut_serv, RUT_SERVSIZE) == 0) {
				memset(&u, 0, sizeof(u));
				lseek(fd, -sizeof(u), SEEK_CUR);
				write(fd, &u, sizeof(u));
			}
		}
	    }
	    if (status == PW_STATUS_START) {
		/*
		 *	Find a free slot.
		 */
		lseek(fd, 0, SEEK_SET);
		while(read(fd, &u, sizeof(u)) == sizeof(u)) {
			if (u.ut_name[0] == 0 && u.ut_line[0] == 0) {
				lseek(fd, -sizeof(u), SEEK_CUR);
				break;
			}
		}
		/*
		 *	Write it.
		 */
		write(fd, &ut, sizeof(ut));
	    }
	    close(fd);
	}

	/*
	 *	Write a RADIUS wtmp log file.
	 */
	if ((fp = fopen(RADWTMP, "a")) != NULL) {
		fwrite(&wt, sizeof(wt), 1, fp);
		fclose(fp);
	}

	/*
	 *	Now send back an ACK to the NAS.
	 */
	send_acct_reply(authreq, (VALUE_PAIR *)NULL, 
			(char *)NULL,activefd);
	pairfree(authreq->request);
	memset(authreq, 0, sizeof(AUTH_REQ));
	free(authreq);

	return 0;
}
#endif /* RADUTMP_VER == 1 */

#if !defined(RADUTMP_VER) || RADUTMP_VER == 0

rad_accounting(authreq, activefd)
AUTH_REQ	*authreq;
int		activefd;
{
	FILE		*outfd;
	char		*ip_hostname();
	char		clientname[128];
	char		buffer[512];
	VALUE_PAIR	*pair;
	long		curtime;

	strcpy(clientname, ip_hostname(authreq->ipaddr));

	/*
	 * Create a directory for this client.
	 */
	sprintf(buffer, "%s/%s", radacct_dir, clientname);
	mkdir(buffer, 0755);

	/*
	 * Write Detail file.
	 */
	sprintf(buffer, "%s/%s/detail", radacct_dir, clientname);
	if((outfd = fopen(buffer, "a")) == (FILE *)NULL) {
		sprintf(buffer,
			"Acct: Couldn't open file %s/%s/detail\n",
			radacct_dir, clientname);
		log_err(buffer);
		/* don't respond if we can't save record */
	} else {

		/* Post a timestamp */
		curtime = time(0);
		fputs(ctime(&curtime), outfd);

		/* Write each attribute/value to the log file */
		pair = authreq->request;
		while(pair != (VALUE_PAIR *)NULL) {
			fputs("\t", outfd);
			fprint_attr_val(outfd, pair);
			fputs("\n", outfd);
		pair = pair->next;
		}
		fputs("\n", outfd);
		fclose(outfd);
		/* let NAS know it is OK to delete from buffer */
		send_acct_reply(authreq, (VALUE_PAIR *)NULL, 
				(char *)NULL,activefd);
	}

	pairfree(authreq->request);
	memset(authreq, 0, sizeof(AUTH_REQ));
	free(authreq);
	return;
}
#endif /* RADUTMP_VER == 0 */

/*************************************************************************
 *
 *	Function: send_acct_reply
 *
 *	Purpose: Reply to the request with an ACKNOWLEDGE.  Also attach
 *		 reply attribute value pairs and any user message provided.
 *
 *************************************************************************/

int send_acct_reply(authreq, reply, msg, activefd)
AUTH_REQ	*authreq;
VALUE_PAIR	*reply;
char		*msg;
int		activefd;
{
	AUTH_HDR		*auth;
	u_short			total_length;
	struct	sockaddr_in	saremote;
	struct	sockaddr_in	*sin;
	u_char			*ptr;
	int			len;
	UINT4			lvalue;
	u_char			digest[16];
	int			secretlen;

	auth = (AUTH_HDR *)send_buffer;

	/* Build standard header */
	auth->code = PW_ACCOUNTING_RESPONSE;
	auth->id = authreq->id;
	memcpy(auth->vector, authreq->vector, AUTH_VECTOR_LEN);

	DEBUG("Sending Accounting Ack of id %d to %lx (%s)\n",
		authreq->id, authreq->ipaddr, ip_hostname(authreq->ipaddr));

	total_length = AUTH_HDR_LEN;

	/* Load up the configuration values for the user */
	ptr = auth->data;
	while(reply != (VALUE_PAIR *)NULL) {
		debug_pair(stdout, reply);
		*ptr++ = reply->attribute;

		switch(reply->type) {

		case PW_TYPE_STRING:
			len = strlen(reply->strvalue);
			*ptr++ = len + 2;
			strcpy(ptr, reply->strvalue);
			ptr += len;
			total_length += len + 2;
			break;
			
		case PW_TYPE_INTEGER:
		case PW_TYPE_IPADDR:
			*ptr++ = sizeof(UINT4) + 2;
			lvalue = htonl(reply->lvalue);
			memcpy(ptr, &lvalue, sizeof(UINT4));
			ptr += sizeof(UINT4);
			total_length += sizeof(UINT4) + 2;
			break;

		default:
			break;
		}

		reply = reply->next;
	}

	/* Append the user message */
	if(msg != (char *)NULL) {
		len = strlen(msg);
		if(len > 0 && len < AUTH_STRING_LEN) {
			*ptr++ = PW_REPLY_MESSAGE;
			*ptr++ = len + 2;
			memcpy(ptr, msg, len);
			ptr += len;
			total_length += len + 2;
		}
	}

	auth->length = htons(total_length);

	/* Calculate the response digest */
	secretlen = strlen(authreq->secret);
	memcpy(send_buffer + total_length, authreq->secret, secretlen);
	md5_calc(digest, (char *)auth, total_length + secretlen);
	memcpy(auth->vector, digest, AUTH_VECTOR_LEN);
	memset(send_buffer + total_length, 0, secretlen);

	sin = (struct sockaddr_in *) &saremote;
        memset ((char *) sin, '\0', sizeof (saremote));
	sin->sin_family = AF_INET;
	sin->sin_addr.s_addr = htonl(authreq->ipaddr);
	sin->sin_port = htons(authreq->udp_port);

	/* Send it to the user */
	sendto(activefd, (char *)auth, (int)total_length, (int)0,
			(struct sockaddr *) sin, sizeof(struct sockaddr_in));

	return 0;
}
