/*
 *
 *	RADIUS
 *	Remote Authentication Dial In User Service
 *
 *
 *	Livingston Enterprises, Inc.
 *	6920 Koll Center Parkway
 *	Pleasanton, CA   94566
 *
 *	Copyright 1992 Livingston Enterprises, Inc.
 *
 *	Permission to use, copy, modify, and distribute this software for any
 *	purpose and without fee is hereby granted, provided that this
 *	copyright and permission notice appear on all copies and supporting
 *	documentation, the name of Livingston Enterprises, Inc. not be used
 *	in advertising or publicity pertaining to distribution of the
 *	program without specific prior permission, and notice be given
 *	in supporting documentation that copying and distribution is by
 *	permission of Livingston Enterprises, Inc.   
 *
 *	Livingston Enterprises, Inc. makes no representations about
 *	the suitability of this software for any purpose.  It is
 *	provided "as is" without express or implied warranty.
 *
 */

static char sccsid[] =
"@(#)users.c	1.12 Copyright 1992 Livingston Enterprises Inc";

#include	<sys/types.h>
#include	<sys/socket.h>
#include	<sys/time.h>
#include	<netinet/in.h>

#include	<stdio.h>
#include	<netdb.h>
#include	<pwd.h>
#include	<grp.h>
#include	<time.h>
#include	<ctype.h>

#ifdef DBM

#include	<dbm.h>

#endif /* DBM */

#include	"radiusd.h"

static	void fieldcpy();
static	int  userparse();

#define FIND_MODE_NAME	0
#define FIND_MODE_REPLY	1
#define FIND_MODE_SKIP	2
#define FIND_MODE_FLUSH	3
#define FIND_MODE_FOUND	4

/*************************************************************************
 *
 *	Function: pairmove
 *
 *	Purpose: Moves attributes from one list to the other
 *		 if not already present.
 *
 *************************************************************************/
static void
pairmove(to, from)
VALUE_PAIR **to;
VALUE_PAIR **from;
{
	VALUE_PAIR *tail, *i, *next;
	VALUE_PAIR *last = NULL;

	if (*to == NULL) {
		*to = *from;
		*from = NULL;
		return;
	}

	for(tail = *to; tail->next; tail = tail->next)
		;

	for(i = *from; i; i = next) {
		next = i->next;
		if (pairfind(*to, i->attribute)) {
			last = i;
			continue;
		}
		if (last)
			last->next = next;
		else
			*from = next;
		tail->next = i;
		i->next = NULL;
		tail = i;
	}
}


/*************************************************************************
 *
 *	Function: portcmp
 *
 *	Purpose: Compare a portno with a range.
 *
 *************************************************************************/
static int portcmp(check, request)
VALUE_PAIR *check;
VALUE_PAIR *request;
{
	char buf[256];
	char *s, *p;
	int lo, hi;
	int port = request->lvalue;

	strcpy(buf, check->strvalue);
	s = strtok(buf, ",");
	while(s) {
		if ((p = strchr(s, '-')) != NULL)
			p++;
		else
			p = s;
		lo = atoi(s);
		hi = atoi(p);
		if (lo <= port && port <= hi) {
			return 0;
		}
		s = strtok(NULL, ",");
	}

	return -1;
}


/*************************************************************************
 *
 *	Function: groupcmp
 *
 *	Purpose: See if user is member of a group.
 *		 FIXME: We don't handle additional groups yet.
 *
 *************************************************************************/
static int groupcmp(check, request)
VALUE_PAIR *check;
VALUE_PAIR *request;
{
	struct passwd *pwd;
	struct group *grp;

	if ((pwd = getpwnam(request->strvalue)) == NULL)
		return -1;

	if ((grp = getgrnam(check->strvalue)) == NULL)
		return -1;

	return (pwd->pw_gid == grp->gr_gid) ? 0 : -1;
}

/*************************************************************************
 *
 *	Function: paircmp
 *
 *	Purpose: Compares two pair lists except for the password information.
 *		 Return 0 on match.
 *
 *************************************************************************/
static int paircmp(request, check)
VALUE_PAIR *request;
VALUE_PAIR *check;
{
	VALUE_PAIR *check_item = check;
	VALUE_PAIR *auth_item;
	int result = 0;

	while (result == 0 && check_item != (VALUE_PAIR *)NULL) {
		switch (check_item->attribute) {
			/*
			 *	Attributes we skip during comparison.
			 */
			case PW_EXPIRATION:
			case PW_PASSWORD:
				check_item = check_item->next;
				continue;
		}
		/*
		 *	See if this item is present in the request.
		 */
		auth_item = request;
		while(auth_item != (VALUE_PAIR *)NULL) {
			if (check_item->attribute == auth_item->attribute ||
			    check_item->attribute == PW_HUNTGROUP_NAME ||
			    (check_item->attribute == PW_GROUP_NAME &&
			     auth_item->attribute  == PW_USER_NAME))
				break;
			auth_item = auth_item->next;
		}
		if (auth_item == (VALUE_PAIR *)NULL) {
			result = -1;
			continue;
		}

		/*
		 *	OK it is present now compare them.
		 */
		
		switch(check_item->type) {

			case PW_TYPE_STRING:
				if (check_item->attribute == PW_NAS_PORT_ID) {
					if (portcmp(check_item, auth_item) != 0)
						result = -1;
				}
			 	else
				if (check_item->attribute == PW_GROUP_NAME) {
					if (groupcmp(check_item, auth_item)!=0)
						result = -1;
				}
				else
				if (check_item->attribute == PW_HUNTGROUP_NAME){
					if (!huntgroup_match(request,
						check_item->strvalue))
							result = -1;
				}
				else
				if (strcmp(check_item->strvalue,
						auth_item->strvalue) != 0) {
					result = -1;
				}
				break;

			case PW_TYPE_INTEGER:
			case PW_TYPE_IPADDR:
				if(check_item->lvalue != auth_item->lvalue) {
					result = -1;
				}
				break;

			default:
				result = -1;
				break;
		}
		check_item = check_item->next;
	}

	return result;

}

/*************************************************************************
 *
 *	Function: hunt_paircmp
 *
 *	Purpose: Compares two pair lists. At least one of the check pairs
 *		 has to be present in the request.
 *
 *************************************************************************/
static int hunt_paircmp(request, check)
VALUE_PAIR *request;
VALUE_PAIR *check;
{
	VALUE_PAIR *check_item = check;
	VALUE_PAIR *auth_item;
	int result = -1;

	if (check == NULL) return 0;

	while (result != 0 && check_item != (VALUE_PAIR *)NULL) {
		switch (check_item->attribute) {
			/*
			 *	Attributes we skip during comparison.
			 */
			case PW_EXPIRATION:
			case PW_PASSWORD:
				check_item = check_item->next;
				continue;
		}
		/*
		 *	See if this item is present in the request.
		 */
		auth_item = request;
		while(auth_item != (VALUE_PAIR *)NULL) {
			if (check_item->attribute == auth_item->attribute ||
			    (check_item->attribute == PW_GROUP_NAME &&
			     auth_item->attribute  == PW_USER_NAME))
				break;
			auth_item = auth_item->next;
		}
		if (auth_item == (VALUE_PAIR *)NULL)
			continue;

		/*
		 *	OK it is present now compare them.
		 */
		
		switch(check_item->type) {

			case PW_TYPE_STRING:
				if (check_item->attribute == PW_NAS_PORT_ID) {
					if (portcmp(check_item, auth_item) == 0)
						result = 0;
				}
			 	else
				if (check_item->attribute == PW_GROUP_NAME) {
					if (groupcmp(check_item, auth_item)==0)
						result = 0;
				}
				else
				if (check_item->attribute == PW_HUNTGROUP_NAME){
					if (huntgroup_match(request,
						check_item->strvalue))
							result = 0;
				}
				else
				if (strcmp(check_item->strvalue,
						auth_item->strvalue) == 0) {
					result = 0;
				}
				break;

			case PW_TYPE_INTEGER:
			case PW_TYPE_IPADDR:
				if(check_item->lvalue == auth_item->lvalue) {
					result = 0;
				}
				break;

			default:
				break;
		}
		check_item = check_item->next;
	}

	return result;

}


/*************************************************************************
 *
 *	Function: user_find
 *
 *	Purpose: Find the named user in the database.  Create the
 *		 set of attribute-value pairs to check and reply with
 *		 for this user from the database. The main code only
 *		 needs to check the password, the rest is done here.
 *
 *************************************************************************/

user_find(name, request_pairs, check_pairs, reply_pairs)
char	*name;
VALUE_PAIR	*request_pairs;
VALUE_PAIR	**check_pairs;
VALUE_PAIR	**reply_pairs;
{
	FILE		*userfd;
	char		buffer[256];
	char		msg[128];
	char		entry[128];
	char		*ptr;
	int		namelen;
	int		mode;
	int		nas_port = 0;
	int		fallthrough;
	VALUE_PAIR	*check_tmp;
	VALUE_PAIR	*reply_tmp;
	VALUE_PAIR	*tmp;
#ifdef DBM
	datum		named;
	datum		contentd;
#endif /* DBM */
	int match;

	/* 
	 * Check for valid input, zero length names not permitted 
	 */

	mode = FIND_MODE_NAME;

	ptr=name;
	while (*ptr != '\0') {
		if (*ptr == ' ' || *ptr == '\t') {
			*ptr = '\0';
		} else {
			ptr++;
		}
	}

	namelen=strlen(name);

	if (namelen < 1) {
		fprintf(stderr, "%s: zero length username not permitted\n",progname);
		return(-1);
	}

	/*
	 *	Find the NAS port ID.
	 */
	if ((tmp = pairfind(request_pairs, PW_NAS_PORT_ID)) != NULL)
		nas_port = tmp->lvalue;
		
	/*
	 *	Open the user table
	 */
	sprintf(buffer, "%s/%s", radius_dir, RADIUS_USERS);
#ifdef DBM
	if(dbminit(buffer) != 0) {
#else /* DBM */
	if((userfd = fopen(buffer, "r")) == (FILE *)NULL) {
#endif /* DBM */
		fprintf(stderr, "%s:Couldn't open %s for reading\n",
				progname, buffer);
		return(-1);
	}

	*check_pairs = (VALUE_PAIR *)NULL;
	*reply_pairs = (VALUE_PAIR *)NULL;


#ifdef DBM
	named.dptr = name;
	named.dsize = strlen(name);
	contentd = fetch(named);

	if(contentd.dsize == 0) {
		named.dptr = "DEFAULT";
		named.dsize = strlen("DEFAULT");
		contentd = fetch(named);
		if(contentd.dsize == 0) {
			dbmclose();
			return(-1);
		}
	}

	/*
	 * Parse the check values
	 */
	ptr = contentd.dptr;
	contentd.dptr[contentd.dsize] = '\0';

	if (userparse(ptr, &check_first, 0) != 0) {
		sprintf(msg, "%s: Parse error for user %s\n",
				progname, name);
		fprintf(stderr, msg);
		log_err(msg);
		pairfree(check_first);
		dbmclose();
		return(-1);
	}
	while(*ptr != '\n' && *ptr != '\0') {
		ptr++;
	}
	if(*ptr != '\n') {
		pairfree(check_first);
		dbmclose();
		return(-1);
	}
	ptr++;
	/*
	 * Parse the reply values
	 */
	if(userparse(ptr, &reply_first, 0) != 0) {
		fprintf(stderr, "%s: Parse error for user %s\n",
			progname, name);
		pairfree(check_first);
		pairfree(reply_first);
		dbmclose();
		return(-1);
	}
	dbmclose();

#else /* DBM */

	while(fgets(buffer, sizeof(buffer), userfd) != (char *)NULL) {
		if (buffer[0] == '#' || buffer[0] == '\n') continue;
re_parse:
		if(mode == FIND_MODE_NAME) {
			/*
			 * Find the entry starting with the users name
			 */
			if (isspace(buffer[0])) continue;
			if (namelen > strlen(buffer))  continue;

			match = 0;
			if ((isspace(buffer[namelen]) &&
			     strncmp(buffer, name, namelen) == 0) ||
			     strncmp(buffer, "DEFAULT", 7) == 0)
				match = 1;
			if (!match) continue;

			if(strncmp(buffer, "DEFAULT", 7) == 0) {
				ptr = &buffer[7];
			}
			else {
				ptr = &buffer[namelen];
			}
			strncpy(entry, buffer, ptr - buffer);
			entry[ptr - buffer] = 0;

			/*
			 * Parse the check values
			 */
			check_tmp = NULL;
			reply_tmp = NULL;
			if(userparse(ptr, &check_tmp, 0) != 0) {
				sprintf(msg,"%s: Parse error for user %s\n",
					progname, entry);
				fprintf(stderr,msg);
				log_err(msg);
				pairfree(*check_pairs);
				pairfree(*reply_pairs);
				*check_pairs = NULL;
				*reply_pairs = NULL;
				fclose(userfd);
				return(-1);
			}
			/*
			 *	See if this entry matches.
			 */
			if (paircmp(request_pairs, check_tmp) == 0) {
				pairmove(check_pairs, &check_tmp);
				mode = FIND_MODE_REPLY;
				reply_tmp = NULL;
			}
			pairfree(check_tmp);
			check_tmp = NULL;
		}
		else {
			if(*buffer == ' ' || *buffer == '\t') {
				/*
				 * Parse the reply values
				 */
				if(userparse(buffer, &reply_tmp, nas_port)!=0) {
					fprintf(stderr,
						"%s: Parse error for user %s\n",
						progname, entry);
					pairfree(*check_pairs);
					pairfree(*reply_pairs);
					*check_pairs = NULL;
					*reply_pairs = NULL;
					fclose(userfd);
					return(-1);
				}
			}
			else {
				/*
				 *	Done with this entry...
				 */
				fallthrough = 0;
				if ((tmp =
					pairfind(reply_tmp, PW_FALL_THROUGH))
					!= NULL && tmp->lvalue)
						fallthrough = 1;
				pairdelete(&reply_tmp, PW_FALL_THROUGH);

				pairmove(reply_pairs, &reply_tmp);
				pairfree(reply_tmp);
				reply_tmp = NULL;

				if (!fallthrough) {
					/* We are done */
					fclose(userfd);
					return(0);
				}
				mode = FIND_MODE_NAME;
				goto re_parse;
			}
		}
	}
	/*
	 *	Make sure that we also read the last line of the file!
	 */
	if (mode == FIND_MODE_REPLY) {
		buffer[0] = 0;
		goto re_parse;
	}
	fclose(userfd);
#endif /* DBM */

	/*
	 *	See if we succeeded.
	 */
	if (*reply_pairs == NULL)
		return -1;

	return 0;
}

#define PARSE_MODE_NAME		0
#define PARSE_MODE_EQUAL	1
#define PARSE_MODE_VALUE	2
#define PARSE_MODE_INVALID	3

/*************************************************************************
 *
 *	Function: userparse
 *
 *	Purpose: Parses the buffer to extract the attribute-value pairs.
 *
 *************************************************************************/

static int
userparse(buffer, first_pair, nas_port)
char		*buffer;
VALUE_PAIR	**first_pair;
int		nas_port;
{
	int		mode;
	int		x;
	char		attrstr[64];
	char		valstr[64];
	char		*s;
	DICT_ATTR	*attr;
	DICT_ATTR	*dict_attrfind();
	DICT_VALUE	*dval;
	DICT_VALUE	*dict_valfind();
	VALUE_PAIR	*pair;
	VALUE_PAIR	*link;
	UINT4		ipstr2long();
	UINT4		get_ipaddr();
	struct tm	*tm;
	time_t		timeval;

	mode = PARSE_MODE_NAME;
	while(*buffer != '\n' && *buffer != '\0') {

		if(*buffer == ' ' || *buffer == '\t' || *buffer == ',') {
			buffer++;
			continue;
		}

		switch(mode) {

		case PARSE_MODE_NAME:
			/* Attribute Name */
			fieldcpy(attrstr, &buffer);
			if((attr = dict_attrfind(attrstr)) ==
						(DICT_ATTR *)NULL) {
				return(-1);
			}
			mode = PARSE_MODE_EQUAL;
			break;

		case PARSE_MODE_EQUAL:
			/* Equal sign */
			if(*buffer == '=') {
				mode = PARSE_MODE_VALUE;
				buffer++;
			}
			else {
				return(-1);
			}
			break;

		case PARSE_MODE_VALUE:
			/* Value */
			fieldcpy(valstr, &buffer);

			if((pair = (VALUE_PAIR *)malloc(sizeof(VALUE_PAIR))) ==
						(VALUE_PAIR *)NULL) {
				fprintf(stderr, "%s: no memory\n",
						progname);
				exit(-1);
			}
			strcpy(pair->name, attr->name);
			pair->attribute = attr->value;
			pair->type = attr->type;

			switch(pair->type) {

			case PW_TYPE_STRING:
				strcpy(pair->strvalue, valstr);
				break;

			case PW_TYPE_INTEGER:
				/*
				 *	For PW_NAS_PORT_ID, allow a
				 *	port range instead of just a port.
				 */
				if (attr->value == PW_NAS_PORT_ID) {
					for(s = valstr; *s; s++)
						if (!isdigit(*s)) break;
					if (*s) {
						pair->type = PW_TYPE_STRING;
						strcpy(pair->strvalue, valstr);
						break;
					}
				}
				if (isdigit(*valstr)) {
					pair->lvalue = atoi(valstr);
				}
				else if((dval = dict_valfind(valstr)) ==
							(DICT_VALUE *)NULL) {
					free(pair);
					return(-1);
				}
				else {
					pair->lvalue = dval->value;
				}
				break;

			case PW_TYPE_IPADDR:
				/*
				 *	We allow a "+" at the end to
				 *	indicate that we should add the
				 *	portno. to the IP address.
				 */
				x = 0;
				if (valstr[0]) {
					for(s = valstr; s[1]; s++)
						;
					if (*s == '+') {
						*s = 0;
						x = nas_port;
					}
				}
				pair->lvalue = get_ipaddr(valstr);
				/*
				 *	We can actually add them together
				 *	because radiusd stores IP addresses
				 *	in host order.
				 */
				if (pair->lvalue && pair->lvalue != -1)
					pair->lvalue += x;
				break;

			case PW_TYPE_DATE:
				timeval = time(0);
				tm = localtime(&timeval);
				user_gettime(valstr, tm);
#ifdef TIMELOCAL
				pair->lvalue = (UINT4)timelocal(tm);
#else /* TIMELOCAL */
				pair->lvalue = (UINT4)mktime(tm);
#endif /* TIMELOCAL */
				break;

			default:
				free(pair);
				return(-1);
			}
			pair->next = (VALUE_PAIR *)NULL;
			if(*first_pair == (VALUE_PAIR *)NULL) {
				*first_pair = pair;
			}
			else {
				link = *first_pair;
				while(link->next != (VALUE_PAIR *)NULL) {
					link = link->next;
				}
				link->next = pair;
			}
			mode = PARSE_MODE_NAME;
			break;

		default:
			mode = PARSE_MODE_NAME;
			break;
		}
	}
	return(0);
}

/*************************************************************************
 *
 *	Function: fieldcpy
 *
 *	Purpose: Copy a data field from the buffer.  Advance the buffer
 *		 past the data field.
 *
 *************************************************************************/

static	void
fieldcpy(string, uptr)
char	*string;
char	**uptr;
{
	char	*ptr;

	ptr = *uptr;
	if(*ptr == '"') {
		ptr++;
		while(*ptr != '"' && *ptr != '\0' && *ptr != '\n') {
			*string++ = *ptr++;
		}
		*string = '\0';
		if(*ptr == '"') {
			ptr++;
		}
		*uptr = ptr;
		return;
	}

	while(*ptr != ' ' && *ptr != '\t' && *ptr != '\0' && *ptr != '\n' &&
						*ptr != '=' && *ptr != ',') {
			*string++ = *ptr++;
	}
	*string = '\0';
	*uptr = ptr;
	return;
}

/*************************************************************************
 *
 *	Function: user_update
 *
 *	Purpose: Updates a user in the database.  Replaces the original
 *		 entry with the name, the list of check items, and the
 *		 list of reply items which are supplied.
 *
 *************************************************************************/

user_update(name, user_check, user_reply)
char		*name;
VALUE_PAIR	*user_check;
VALUE_PAIR	*user_reply;
{
	FILE		*oldfd;
	FILE		*userfd;
	char		buffer[256];
	char		buffer1[256];
	int		namelen;
	int		mode;

	sprintf(buffer, "%s/%s", radius_dir, RADIUS_USERS);
	sprintf(buffer1, "%s/%s", radius_dir, RADIUS_HOLD);

	/* Move the user table to a temporary location */
	if(rename(buffer, buffer1) != 0) {
		fprintf(stderr, "%s: Couldn't rename %s\n",
				progname, buffer);
		return(-1);
	}

	/* Open the old user file (using the temporary name) */
	if((oldfd = fopen(buffer1, "r")) == (FILE *)NULL) {
		fprintf(stderr, "%s: Couldn't open %s for reading\n",
				progname, buffer1);
		exit(-1);
	}

	/* Open the new user file */
	if((userfd = fopen(buffer, "w")) == (FILE *)NULL) {
		fprintf(stderr, "%s: Couldn't open %s for writing\n",
				progname, buffer);
		exit(-1);
	}

	mode = FIND_MODE_NAME;
	namelen = strlen(name);

	/* Copy the old to the new, only recreating the changed user */
	while(fgets(buffer, sizeof(buffer), oldfd) != (char *)NULL) {
		if(mode == FIND_MODE_NAME) {
			if((strncmp(buffer, name, namelen) == 0 &&
		 	 (buffer[namelen] == ' ' || buffer[namelen] == '\t'))) {

				/* Write our new information */
				fprintf(userfd, "%s\t", name);
				while(user_check != (VALUE_PAIR *)NULL) {
					fprint_attr_val(userfd, user_check);
					if(user_check->next !=
							(VALUE_PAIR *)NULL) {
						fprintf(userfd, ", ");
					}
					user_check = user_check->next;
				}
				fprintf(userfd, "\n\t");
				while(user_reply != (VALUE_PAIR *)NULL) {
					fprint_attr_val(userfd, user_reply);
					if(user_reply->next !=
							(VALUE_PAIR *)NULL) {
						fprintf(userfd, ",\n\t");
					}
					user_reply = user_reply->next;
				}
				fprintf(userfd, "\n");
				mode = FIND_MODE_SKIP;
			}
			else {
				fputs(buffer, userfd);
			}
		}
		else if(mode == FIND_MODE_SKIP) {
			if(*buffer != ' ' && *buffer != '\t') {
				fputs(buffer, userfd);
				mode = FIND_MODE_FLUSH;
			}
		}
		else {
			fputs(buffer, userfd);
		}
	}
	fclose(oldfd);
	fclose(userfd);
	return(0);
}

/*************************************************************************
 *
 *	Function: user_prtime
 *
 *	Purpose: Turns printable string into correct tm struct entries
 *
 *************************************************************************/

static char *months[] = {
	"Jan", "Feb", "Mar", "Apr", "May", "Jun",
	"Jul", "Aug", "Sep", "Oct", "Nov", "Dec" };

user_gettime(valstr, tm)
char		*valstr;
struct tm	*tm;
{
	char	buffer[48];
	int	i;

	/* Get the month */
	for(i = 0;i < 12;i++) {
		if(strncmp(months[i], valstr, 3) == 0) {
			tm->tm_mon = i;
			i = 13;
		}
	}

	/* Get the Day */
	tm->tm_mday = atoi(&valstr[4]);

	/* Now the year */
	tm->tm_year = atoi(&valstr[7]) - 1900;
}

/*************************************************************************
 *
 *	Function: matches
 *
 *	Purpose: Matches a username with a wildcard expression.
 *		 Is very limited for now.
 *
 *************************************************************************/
static int matches(char *name, char *wild, char *matchpart)
{
	int len, wlen;
	int ret = 0;

	len = strlen(name);
	wlen = strlen(wild);

	if (len == 0 || wlen == 0) return 0;

	if (wild[0] == '*') {
		wild++;
		wlen--;
		if (wlen <= len && strcmp(name + (len - wlen), wild) == 0) {
			strcpy(matchpart, name);
			matchpart[wlen] = 0;
			ret = 1;
		}
	} else if (wild[wlen - 1] == '*') {
		wild[--wlen] = 0;
		if (wlen <= len && strncmp(name, wild, wlen) == 0) {
			strcpy(matchpart, name + wlen);
			ret = 1;
		}
	}

	return ret;
}




/*************************************************************************
 *
 *	Function: hints_setup
 *
 *	Purpose: Add hints to the info sent by the terminal server
 *		 based on the pattern of the username.
 *
 *************************************************************************/

int
hints_setup(request_pairs)
VALUE_PAIR	*request_pairs;
{
	FILE		*hintsfd;
	char		msg[128];
	char		buffer[256];
	char		newname[32];
	char		*name;
	char		*ptr;
	int		mode;
	VALUE_PAIR	*add_first;
	VALUE_PAIR	*last;
	VALUE_PAIR	*tmp;

	/* 
	 * Check for valid input, zero length names not permitted 
	 */

	mode = FIND_MODE_NAME;
	add_first = NULL;

	if ((tmp = pairfind(request_pairs, PW_USER_NAME)) == NULL) {
		sprintf(msg,"%s: no user name\n", progname);
		fprintf(stderr,msg);
		return -1;
	}
	ptr = name = tmp->strvalue;
	for(ptr = name; *ptr; ptr++)
		if (*ptr == ' ' || *ptr == '\t') {
			*ptr = 0;
			break;
		}

	if (name[0] == 0)
		/*
		 *	Will be complained about later.
		 */
		return 0;

	/*
	 *	Small check: if Framed-Protocol present but Service-Type
	 *	is missing, add Service-Type = Framed-User.
	 */
	if (pairfind(request_pairs, PW_FRAMED_PROTOCOL) != NULL &&
	    pairfind(request_pairs, PW_SERVICE_TYPE) == NULL) {
		if ((tmp = malloc(sizeof(VALUE_PAIR))) != NULL) {
			memset(tmp, 0, sizeof(VALUE_PAIR));
			strcpy(tmp->name, "Service-Type");
			tmp->attribute = PW_SERVICE_TYPE;
			tmp->type = PW_TYPE_INTEGER;
			tmp->lvalue = PW_FRAMED_USER;
			pairmove(&request_pairs, &tmp);
		}
	}

	/*
	 * Open the hints table
	 */
	sprintf(buffer, "%s/%s", radius_dir, RADIUS_HINTS);
	if((hintsfd = fopen(buffer, "r")) == (FILE *)NULL)
		/*
		 *	May actually not exist.
		 */
		return 0;

	add_first = NULL;

	while(mode != FIND_MODE_FOUND &&
	      fgets(buffer, sizeof(buffer), hintsfd) != (char *)NULL) {

		if (mode != FIND_MODE_REPLY && add_first) {
			pairfree(add_first);
			add_first = NULL;
		}

		if (*buffer == '#' || *buffer == '\n')
			continue;

		if(mode == FIND_MODE_NAME) {
			/*
			 * Start of a new entry.
			 */
			if (*buffer == ' ' || *buffer == '\t')
				continue;
			ptr = buffer;
			while(*ptr != ' ' && *ptr != '\t' &&
			      *ptr != '\n' && *ptr)
				ptr++;
			*ptr = 0;

			if (matches(name, buffer, newname))
				mode = FIND_MODE_REPLY;

		} else if (mode == FIND_MODE_REPLY) {
			if(*buffer == ' ' || *buffer == '\t') {
				/*
				 * Parse the reply values
				 */
				if(userparse(buffer, &add_first, 0) != 0) {
					fprintf(stderr,
					    "%s: Parse error for hint %s\n",
					    progname, name);
					pairfree(add_first);
					fclose(hintsfd);
					return(-1);
				}
				continue;
			}
			else {
				/* We are done */
				mode = FIND_MODE_FOUND;
			}
		}
	}
	fclose(hintsfd);

	if (mode != FIND_MODE_FOUND && mode != FIND_MODE_REPLY) return 0;

	/*
	 *	See if we need to adjust the name.
	 */
	if ((tmp = pairfind(add_first, PW_USER_NAME_IS_STAR)) != NULL
	    && tmp->lvalue) {
		tmp = pairfind(request_pairs, PW_USER_NAME);
		if (tmp)
			strcpy(tmp->strvalue, newname);
	}

	/*
	 *	Now add all attributes to the request list,
	 *	except the PW_USER_NAME_IS_STAR one.
	 */
	pairdelete(&add_first, PW_USER_NAME_IS_STAR);
	for(last = request_pairs; last && last->next; last = last->next)
		;
	if (last) last->next = add_first;

	return 0;
}


/*************************************************************************
 *
 *	Function: huntgroup_match
 *
 *	Purpose: See if the huntgroup matches.
 *
 *************************************************************************/
int
huntgroup_match(request_pairs, huntgroup)
VALUE_PAIR	*request_pairs;
char		*huntgroup;
{
	FILE		*fp;
	char		buffer[256];
	char		*ptr;
	int		mode;
	VALUE_PAIR	*tmp;

	/* 
	 * Check for valid input, zero length names not permitted 
	 */

	mode = FIND_MODE_NAME;
	tmp = NULL;

	/*
	 * Open the huntgroups table
	 */
	sprintf(buffer, "%s/%s", radius_dir, RADIUS_HUNTGROUPS);
	if((fp = fopen(buffer, "r")) == (FILE *)NULL)
		/*
		 *	May actually not exist.
		 */
		return 0;

	while(mode != FIND_MODE_FOUND &&
	      fgets(buffer, sizeof(buffer), fp) != (char *)NULL) {

		if (*buffer == '#' || *buffer == '\n' || isspace(*buffer))
			continue;

		/*
		 *	Start of a new entry.
		 */
		if (*buffer == ' ' || *buffer == '\t')
			continue;

		ptr = buffer;
		while(*ptr != ' ' && *ptr != '\t' &&
		      *ptr != '\n' && *ptr)
			ptr++;
		if (*ptr) *ptr++ = 0;

		if (strcmp(buffer, huntgroup) != 0) continue;

		if (userparse(ptr, &tmp, 0) != 0) {
			fprintf(stderr, "%s: Parse error for huntgroup %s\n",
				progname, buffer);
			pairfree(tmp);
			fclose(fp);
			return(-1);
		}

		if (paircmp(request_pairs, tmp) == 0)
			mode = FIND_MODE_FOUND;

		pairfree(tmp);
		tmp = NULL;
	}
	fclose(fp);

	return mode == FIND_MODE_FOUND;
}

/*************************************************************************
 *
 *	Function: huntgroup_access
 *
 *	Purpose: See if we have access to the huntgroup.
 *		 Returns 0 if we don't have access.
 *		         1 if we do have access.
 *		        -1 on error.
 *
 *************************************************************************/

int
huntgroup_access(request_pairs)
VALUE_PAIR	*request_pairs;
{
	FILE		*fp;
	char		buffer[256];
	char		msg[256];
	char		*ptr;
	int		mode;
	VALUE_PAIR	*check_tmp;
	int		result = 1;

	mode = FIND_MODE_NAME;
	check_tmp = NULL;

	/*
	 *	Open the huntgroups table
	 */
	sprintf(buffer, "%s/%s", radius_dir, RADIUS_HUNTGROUPS);
	if ((fp = fopen(buffer, "r")) == (FILE *)NULL) {
		/*
		 *	No error if it doesn't exist.
		 */
		return 1;
	}

	while(fgets(buffer, sizeof(buffer), fp) != (char *)NULL) {
		if (buffer[0] == '#' || buffer[0] == '\n') continue;
re_parse:
		if(mode == FIND_MODE_NAME) {
			/*
			 *	Find an entry starting with a name.
			 */
			if (isspace(buffer[0])) continue;

			ptr = buffer;
			while(*ptr != ' ' && *ptr != '\t' && *ptr)
				ptr++;
			if (*ptr) *ptr++ = 0;

			/*
			 *	Parse the check values
			 */
			check_tmp = NULL;
			if(userparse(ptr, &check_tmp, 0) != 0) {
				sprintf(msg,"%s: Parse error for huntgroup %s\n"
					, progname, buffer);
				fprintf(stderr,msg);
				log_err(msg);
				fclose(fp);
				return -1;
			}
			/*
			 *	See if this entry matches.
			 */
			if (paircmp(request_pairs, check_tmp) == 0) {
				result = 0;
				mode = FIND_MODE_REPLY;
			}

			pairfree(check_tmp);
			check_tmp = NULL;
		}
		else {
			if (*buffer == ' ' || *buffer == '\t') {
				/*
				 * Parse the reply values
				 */
				if(userparse(buffer, &check_tmp, 0) != 0) {
					fprintf(stderr,
						"%s: huntgroups parse error\n",
						progname);
					pairfree(check_tmp);
					fclose(fp);
					return -1;
				}
			}
			else {
				/*
				 *	Done with this entry...
				 */
				if (hunt_paircmp(request_pairs, check_tmp) == 0)
					result = 1;

				pairfree(check_tmp);
				check_tmp = NULL;
				mode = FIND_MODE_FOUND;

				break;
			}
		}
	}
	if (mode == FIND_MODE_REPLY) {
		buffer[0] = 0;
		goto re_parse;
	}
	fclose(fp);

	return result;
}

#ifdef CISTRON_COMPAT
char *cistron_compat(request, reply)
VALUE_PAIR *	request;
VALUE_PAIR *	reply;
{
	VALUE_PAIR	*framed;
	VALUE_PAIR	*nas;
	VALUE_PAIR	*proto;
	static char	buf[512];
	char		buf1[16];
	char		buf2[16];

	if ((framed = pairfind(reply, PW_FRAMED_IP_ADDRESS)) == NULL ||
	    (nas    = pairfind(request, PW_NAS_IP_ADDRESS)) == NULL ||
	    (proto  = pairfind(reply, PW_FRAMED_PROTOCOL)) == NULL)
		return 0;

	ipaddr2str(buf1, framed->lvalue);
	ipaddr2str(buf2, nas->lvalue);

	sprintf(buf, "\r\n"
		"Your IP address: %s\r\n"
		" Our IP address: %s\r\n"
		"\r\n"
		"Entering %s mode.\r\n",
		buf1, buf2, proto->lvalue == PW_PPP ? "PPP" : "SLIP");
		

	return buf;
}
#endif

