/*
 *
 *	RADIUS Accounting
 *	Remote Authentication Dial In User Service
 *
 *
 *	Livingston Enterprises, Inc.
 *	6920 Koll Center Parkway
 *	Pleasanton, CA   94566
 *
 *	Copyright 1992 - 1994 Livingston Enterprises, Inc.
 *	Copyright 1996 - 1997 Cistron Internet Services B.V.
 *
 *	Permission to use, copy, modify, and distribute this software for any
 *	purpose and without fee is hereby granted, provided that this
 *	copyright and permission notice appear on all copies and supporting
 *	documentation, the name of Livingston Enterprises, Inc. not be used
 *	in advertising or publicity pertaining to distribution of the
 *	program without specific prior permission, and notice be given
 *	in supporting documentation that copying and distribution is by
 *	permission of Livingston Enterprises, Inc.   
 *
 *	Livingston Enterprises, Inc. makes no representations about
 *	the suitability of this software for any purpose.  It is
 *	provided "as is" without express or implied warranty.
 *
 */

char acct_sccsid[] =
"@(#)acct.c	2.1  Copyright 1994 Livingston Enterprises Inc\n"
"		     Copyright 1997 Cistron Internet Services B.V.";


#include	<sys/types.h>
#include	<sys/socket.h>
#include	<sys/time.h>
#include	<sys/file.h>
#include	<sys/stat.h>
#include	<netinet/in.h>

#include	<stdio.h>
#include	<stdlib.h>
#include	<netdb.h>
#include	<pwd.h>
#include	<time.h>
#include	<ctype.h>
#include	<unistd.h>
#include	<signal.h>
#include	<errno.h>
#include	<sys/wait.h>

#include	"radiusd.h"
#include	"radutmp.h"

/*
 *	FIXME: this should be configurable.
 */
int	doradwtmp = 1;

static char trans[64] =
   "ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789+/";
#define ENC(c) trans[c]

/*
 *	UUencode 4 bits base64. We use this to turn a 4 byte field
 *	(an IP adres) into 6 bytes of ASCII. This is used for the
 *	wtmp file if we didn't find a short name in the clients file.
 */
char *uue(void *in)
{
	int i;
	static unsigned char res[7];
	unsigned char *data = (char *)in;

	res[0] = ENC(data[0] >> 2);
	res[1] = ENC((data[0] << 4) & 060) + ((data[1] >> 4) & 017);
	res[2] = ENC((data[1] << 2) & 074) + ((data[2] >> 6) & 03);
	res[3] = ENC(data[2] & 077);

	res[4] = ENC(data[3] >> 2);
	res[5] = ENC((data[3] << 4) & 060);
	res[6] = 0;

	for(i = 0; i < 6; i++) {
		if (res[i] == ' ') res[i] = '`';
		if (res[i] < 32 || res[i] > 127)
			printf("uue: protocol error ?!\n");
	}
	return res;
}


/*
 *	Convert a struct radutmp to a normal struct utmp
 *	as good as we possibly can.
 */
static void make_wtmp(struct radutmp *ut, struct utmp *wt, int status)
{
	UINT4		ip;
	char		buf[32];
	CLIENT		*cl;
	char		*s;

	/*
	 *	Fill out the UTMP struct for the radwtmp file.
	 *	(this one must be "last" - compatible).
	 */
	ip = ut->framed_address;
#ifdef __linux__
	/*
	 *	Linux has a field for the client address.
	 */
	wt->ut_addr = ip;
#endif
	/*
	 *	We use the tty field to store the terminal servers' port
	 *	and address so that the tty field is unique.
	 */
	if ((cl = client_find(ntohl(ut->nas_address))) != NULL)
		s = cl->shortname;
	if (s == NULL || s[0] == 0) s = uue(&ip);
#if UT_LINESIZE > 9
	sprintf(buf, "%03d:%s", ut->nas_port, s);
#else
	sprintf(buf, "%02d%s", ut->nas_port, s);
#endif
	strncpy(wt->ut_line, buf, UT_LINESIZE);

	/*
	 *	We store the dynamic IP address in the hostname field.
	 */
	if (ut->framed_address) {
		ipaddr2str(buf, ntohl(ut->framed_address));
		strncpy(wt->ut_host, buf, UT_HOSTSIZE);
	}
#ifdef __svr4__
	wt->ut_xtime = ut->time;
#else
	wt->ut_time = ut->time;
#endif
#ifdef USER_PROCESS
	/*
	 *	And we can use the ID field to store
	 *	the protocol.
	 */
	if (ut->proto == P_PPP)
		strcpy(wt->ut_id, "P");
	else if (ut->proto == P_SLIP)
		strcpy(wt->ut_id, "S");
	else
		strcpy(wt->ut_id, "T");
	wt->ut_type = status == PW_STATUS_START ? USER_PROCESS : DEAD_PROCESS;
#endif
	if (status == PW_STATUS_STOP)
		wt->ut_name[0] = 0;
}


/*
 *	Zap a user, or all users on a NAS, from the radutmp file.
 */
int radzap(UINT4 nasaddr, int port, char *user, time_t t)
{
	struct radutmp	u;
	struct utmp	wt;
	FILE		*fp;
	int		fd;
	UINT4		netaddr;

	if (t == 0) time(&t);
	fp = fopen(RADWTMP, "a");
	netaddr = htonl(nasaddr);

	if ((fd = open(RADUTMP, O_RDWR|O_CREAT, 0644)) >= 0) {
		int r;

		/*
		 *	Lock the utmp file, prefer lockf() over flock().
		 */
#ifdef F_WRLCK
		(void)lockf(fd, 0, F_WRLCK);
#else
		(void)flock(fd, LOCK_EX);
#endif
	 	/*
		 *	Find the entry for this NAS / portno combination.
		 */
		r = 0;
		while (read(fd, &u, sizeof(u)) == sizeof(u)) {
			if (((nasaddr != 0 && netaddr != u.nas_address) ||
			      (port >= 0   && port    != u.nas_port) ||
			      (user != NULL && strcmp(u.login, user) != 0) ||
			       u.type != P_LOGIN))
				continue;
			/*
			 *	Match. Zap it.
			 */
			lseek(fd, -sizeof(u), SEEK_CUR);
			u.type = P_IDLE;
			u.time = t;
			write(fd, &u, sizeof(u));

			/*
			 *	Add a logout entry to the wtmp file.
			 */
			if (fp != NULL)  {
				make_wtmp(&u, &wt, PW_STATUS_STOP);
				fwrite(&wt, sizeof(wt), 1, fp);
			}
		}
#ifdef F_WRLCK
		(void)lockf(fd, 0, F_UNLCK);
#else
		(void)flock(fd, LOCK_UN);
#endif
		close(fd);
	}
	if (fp) fclose(fp);

	return 0;
}


/*
 *	Show framed users in the RADIUS utmp file.
 */
int rad_accounting_new(authreq, activefd, dowtmp)
AUTH_REQ	*authreq;
int		activefd;
int		dowtmp;
{
	struct radutmp	ut, u;
	struct utmp	wt;
	VALUE_PAIR	*vp;
	int		rb_record = 0;
	int		status = -1;
	int		nas_address = 0;
	int		framed_address = 0;
	int		protocol = -1;
	FILE		*fp;
	time_t		t;
	int		fd;
	int		ret = 0;

	/*
	 *	Which type is this.
	 */
	if ((vp = pairfind(authreq->request, PW_ACCT_STATUS_TYPE)) == NULL) {
		log(L_ERR, "Accounting: no Accounting-Status-Type record.");
		return -1;
	}
	status = vp->lvalue;
	if (status == PW_STATUS_ACCOUNTING_ON ||
	    status == PW_STATUS_ACCOUNTING_OFF) rb_record = 1;

	if (!rb_record &&
	    (vp = pairfind(authreq->request, PW_USER_NAME)) == NULL) {
		/*
		 *	Looks like a reboot record (?). Buggy ComOS revisions
		 *	send the wrong type of records...
	 	 */
		if ((vp = pairfind(authreq->request, PW_ACCT_SESSION_TIME))
			!= NULL && vp->lvalue != 0) {
			log(L_ERR, "Accounting: no username in record");
			return -1;
		}
		log(L_INFO, "Accounting: converting reboot records.");
		if (status == PW_STATUS_STOP)
			status = PW_STATUS_ACCOUNTING_OFF;
		if (status == PW_STATUS_START)
			status = PW_STATUS_ACCOUNTING_ON;
		rb_record = 1;
	}


	/*
	 *	Add any specific attributes for this username.
	 */
	if (!rb_record) hints_setup(authreq->request);

	time(&t);
	memset(&ut, 0, sizeof(ut));
	memset(&wt, 0, sizeof(wt));

	/*
	 *	First, find the interesting attributes.
	 */
	for (vp = authreq->request; vp; vp = vp->next) {
		switch (vp->attribute) {
			case PW_USER_NAME:
				strncpy(ut.login, vp->strvalue, RUT_NAMESIZE);
				strncpy(wt.ut_name, vp->strvalue, UT_NAMESIZE);
				break;
			case PW_LOGIN_IP_HOST:
			case PW_FRAMED_IP_ADDRESS:
				framed_address = vp->lvalue;
				ut.framed_address = htonl(vp->lvalue);
				break;
			case PW_FRAMED_PROTOCOL:
				protocol = vp->lvalue;
				break;
			case PW_NAS_IP_ADDRESS:
				nas_address = vp->lvalue;
				ut.nas_address = htonl(vp->lvalue);
				break;
			case PW_NAS_PORT_ID:
				ut.nas_port = vp->lvalue;
				break;
			case PW_ACCT_DELAY_TIME:
				ut.delay = vp->lvalue;
				break;
			case PW_ACCT_SESSION_ID:
				memcpy(ut.session_id, vp->strvalue,
					sizeof(ut.session_id));
				break;
		}
	}

	/*
	 *	If we didn't find out the NAS address, use the
	 *	originator's IP address.
	 */
	if (nas_address == 0) {
		nas_address = authreq->ipaddr;
		ut.nas_address = htonl(nas_address);
	}

	if (protocol == PW_PPP)
		ut.proto = 'P';
	else if (protocol == PW_SLIP)
		ut.proto = 'S';
	else
		ut.proto = 'T';
	ut.time = t - ut.delay;
	make_wtmp(&ut, &wt, status);

	/*
	 *	See if this was a portmaster reboot.
	 */
	if (status == PW_STATUS_ACCOUNTING_ON && nas_address) {
		log(L_INFO, "NAS %s restarted (Accounting-On packet seen)",
			client_name(nas_address));
		radzap(nas_address, -1, NULL, ut.time);
		return 0;
	}
	if (status == PW_STATUS_ACCOUNTING_OFF && nas_address) {
		log(L_INFO, "NAS %s rebooted (Accounting-Off packet seen)",
			client_name(nas_address));
		radzap(nas_address, -1, NULL, ut.time);
		return 0;
	}

	/*
	 *	If we don't know this type of entry pretend we succeeded.
	 */
	if (status != PW_STATUS_START && status != PW_STATUS_STOP) {
		log(L_ERR, "NAS %s port %d unknown packet type %d)",
			client_name(nas_address), ut.nas_port, status);
		return 0;
	}

	/*
	 *	Enter into the radutmp file.
	 */
	if ((fd = open(RADUTMP, O_RDWR|O_CREAT, 0644)) >= 0) {
		int r;

		/*
		 *	Lock the utmp file, prefer lockf() over flock().
		 */
#ifdef F_WRLCK
		(void)lockf(fd, 0, F_WRLCK);
#else
		(void)flock(fd, LOCK_EX);
#endif
	 	/*
		 *	Find the entry for this NAS / portno combination.
		 */
		r = 0;
		while (read(fd, &u, sizeof(u)) == sizeof(u)) {
			if (u.nas_address != ut.nas_address ||
			    u.nas_port    != ut.nas_port)
				continue;
			if (status == PW_STATUS_STOP &&
			    strncmp(ut.session_id, u.session_id, 
			     sizeof(u.session_id)) != 0) {
				log(L_ERR,
		"Accounting: logout: entry for NAS %s port %d has wrong ID",
				client_name(nas_address), u.nas_port);
				r = -1;
				break;
			}
			if (status == PW_STATUS_START &&
			    strncmp(ut.session_id, u.session_id, 
			     sizeof(u.session_id)) == 0  &&
			    u.time >= ut.time) {
				log(L_ERR,
		"Accounting: login: entry for NAS %s port %d out of order",
				client_name(nas_address), u.nas_port);
				r = -1;
				break;
			}
			lseek(fd, -sizeof(u), SEEK_CUR);
			r = 1;
			break;
		}

		if (r >= 0 && status == PW_STATUS_START) {
			ut.type = P_LOGIN;
			write(fd, &ut, sizeof(u));
		}
		if (status == PW_STATUS_STOP) {
			if (r > 0) {
				u.type = P_IDLE;
				u.time = ut.time;
				u.delay = ut.delay;
				write(fd, &u, sizeof(u));
			} else if (r == 0) {
				log(L_ERR,
		"Accounting: logout: login entry for NAS %s port %d not found",
				client_name(nas_address), ut.nas_port);
				r = -1;
			}
		}
#ifdef F_WRLCK
		(void)lockf(fd, 0, F_UNLCK);
#else
		(void)flock(fd, LOCK_UN);
#endif
		close(fd);
	} else {
		log(L_ERR, "Accounting: %s: %s", RADUTMP, strerror(errno));
		ret = -1;
	}

	/*
	 *	Write a RADIUS wtmp log file.
	 */
	if (doradwtmp && (fp = fopen(RADWTMP, "a")) != NULL) {
		ret = 0;
		fwrite(&wt, sizeof(wt), 1, fp);
		fclose(fp);
	}


	return ret;
}



int rad_accounting_orig(authreq, activefd)
AUTH_REQ	*authreq;
int		activefd;
{
	FILE		*outfd;
	char		clientname[128];
	char		buffer[512];
	char		*s;
	VALUE_PAIR	*pair;
	UINT4		nas;
	CLIENT		*cl;
	long		curtime;
	int		ret = 0;
	struct stat	st;

	/*
	 *	See if we have an accounting directory. If not,
	 *	return.
	 */
	if (stat(radacct_dir, &st) < 0)
		return 0;

	/*
	 *	Find out the name of this terminal server. We try
	 *	to find the PW_NAS_IP_ADDRESS in the clients file.
	 *	If that fails, we look for the originating address.
	 *	Only if that fails we resort to a name lookup.
	 */
	cl = NULL;
	nas = authreq->ipaddr;
	if ((pair = pairfind(authreq->request, PW_NAS_IP_ADDRESS)) != NULL) {
		nas = pair->lvalue;
		if ((cl = client_find(nas)) != NULL) {
			if (cl->shortname[0])
				strcpy(clientname, cl->shortname);
			else
				strcpy(clientname, cl->longname);
		}
	}
	if (cl == NULL) {
		s = ip_hostname(nas);
		if (strlen(s) > 127 || strchr(s, '/'))
			return -1;
		strcpy(clientname, s);
	}

	/*
	 *	Create a directory for this client.
	 */
	sprintf(buffer, "%s/%s", radacct_dir, clientname);
	(void) mkdir(buffer, 0755);

	/*
	 *	Write Detail file.
	 */
	sprintf(buffer, "%s/%s/detail", radacct_dir, clientname);
	if((outfd = fopen(buffer, "a")) == (FILE *)NULL) {
		log(L_ERR,
			"Acct: Couldn't open file %s/%s/detail",
				radacct_dir, clientname);
		ret = -1;
	} else {

		/* Post a timestamp */
		curtime = time(0);
		fputs(ctime(&curtime), outfd);

		/* Write each attribute/value to the log file */
		pair = authreq->request;
		while(pair != (VALUE_PAIR *)NULL) {
			fputs("\t", outfd);
			fprint_attr_val(outfd, pair);
			fputs("\n", outfd);
		pair = pair->next;
		}
		fputs("\n", outfd);
		fclose(outfd);
	}

	return ret;
}

/*
 *	Reply to the request with an ACKNOWLEDGE.  Also attach
 *	reply attribute value pairs and any user message provided.
 */
static int send_acct_reply(AUTH_REQ *authreq, VALUE_PAIR *reply,
		char *msg, int activefd)
{
	AUTH_HDR		*auth;
	u_short			total_length;
	struct	sockaddr_in	saremote;
	struct	sockaddr_in	*sin;
	u_char			*ptr;
	int			len;
	UINT4			lvalue;
	u_char			digest[16];
	int			secretlen;

	auth = (AUTH_HDR *)send_buffer;

	/* Build standard header */
	auth->code = PW_ACCOUNTING_RESPONSE;
	auth->id = authreq->id;
	memcpy(auth->vector, authreq->vector, AUTH_VECTOR_LEN);

	DEBUG("Sending Accounting Ack of id %d to %lx (%s)\n",
	    authreq->id, (u_long)authreq->ipaddr, ip_hostname(authreq->ipaddr));

	total_length = AUTH_HDR_LEN;

	/* Load up the configuration values for the user */
	ptr = auth->data;
	while(reply != (VALUE_PAIR *)NULL) {
		debug_pair(stdout, reply);
		*ptr++ = reply->attribute;

		switch(reply->type) {

		case PW_TYPE_STRING:
			len = strlen(reply->strvalue);
			*ptr++ = len + 2;
			strcpy(ptr, reply->strvalue);
			ptr += len;
			total_length += len + 2;
			break;
			
		case PW_TYPE_INTEGER:
		case PW_TYPE_IPADDR:
			*ptr++ = sizeof(UINT4) + 2;
			lvalue = htonl(reply->lvalue);
			memcpy(ptr, &lvalue, sizeof(UINT4));
			ptr += sizeof(UINT4);
			total_length += sizeof(UINT4) + 2;
			break;

		default:
			break;
		}

		reply = reply->next;
	}

	/* Append the user message */
	if(msg != (char *)NULL) {
		len = strlen(msg);
		if(len > 0 && len < AUTH_STRING_LEN) {
			*ptr++ = PW_REPLY_MESSAGE;
			*ptr++ = len + 2;
			memcpy(ptr, msg, len);
			ptr += len;
			total_length += len + 2;
		}
	}

	auth->length = htons(total_length);

	/* Calculate the response digest */
	secretlen = strlen(authreq->secret);
	memcpy(send_buffer + total_length, authreq->secret, secretlen);
	md5_calc(digest, (char *)auth, total_length + secretlen);
	memcpy(auth->vector, digest, AUTH_VECTOR_LEN);
	memset(send_buffer + total_length, 0, secretlen);

	sin = (struct sockaddr_in *) &saremote;
        memset ((char *) sin, '\0', sizeof (saremote));
	sin->sin_family = AF_INET;
	sin->sin_addr.s_addr = htonl(authreq->ipaddr);
	sin->sin_port = htons(authreq->udp_port);

	/* Send it to the user */
	sendto(activefd, (char *)auth, (int)total_length, (int)0,
			(struct sockaddr *) sin, sizeof(struct sockaddr_in));

	return 0;
}



/*
 *	rad_accounting: call both the old and new style accounting functions.
 *	
 *	FIXME: we should verify the MD5 sum of the accounting request!
 */
int rad_accounting(AUTH_REQ *authreq, int activefd)
{
	int reply = 0;

	if (rad_accounting_orig(authreq, activefd) == 0)
		reply = 1;
	if (rad_accounting_new(authreq, activefd, 1) == 0)
		reply = 1;

	if (!reply) return -1;

	/*
	 *	Now send back an ACK to the NAS.
	 */
	send_acct_reply(authreq, (VALUE_PAIR *)NULL, 
			(char *)NULL,activefd);
	pairfree(authreq->request);
	memset(authreq, 0, sizeof(AUTH_REQ));
	free(authreq);

	return 0;
}


/*
 *	See if a user is already logged in.
 *
 *	FIXME:	check twice. If on the first pass the user exceeds his
 *		max. number of logins, do a second pass and validate all
 *		logins by querying the terminal server (using eg. telnet).
 *
 *	For now we implement another algorithm. If the total no. of logins
 *	exceeds the maximum, we re-read the utmp file. We keep on trying
 *	for 30 seconds. This does not solve the "lost accounting records"
 *	problem, but it does fix the problem with delayed logout records.
 */
int rad_check_multi(char *name, VALUE_PAIR *request, int maxsimul)
{
	FILE		*fp;
	int		count;
	struct radutmp	u;
	int		tries = 0;

	if ((fp = fopen(RADUTMP, "r")) < 0)
		return 0;

	while(tries < 15) {

		if (tries++ > 0) sleep(2);

		/*
		 *	FIXME: lockf() the file while reading?
		 */
		count = 0;
		rewind(fp);
		while(fread(&u, sizeof(u), 1, fp) == 1)
			if (strncmp(name, u.login, RUT_NAMESIZE) == 0
			    && u.type == P_LOGIN)
				count++;


		if (count < maxsimul) {
			fclose(fp);
			return 0;
		}
	}
	fclose(fp);

	return -1;
}

