/*
 *
 *	RADIUS
 *	Remote Authentication Dial In User Service
 *
 *
 *	Livingston Enterprises, Inc.
 *	6920 Koll Center Parkway
 *	Pleasanton, CA   94566
 *
 *	Copyright 1992 Livingston Enterprises, Inc.
 *	Copyright 1997 Cistron Internet Services B.V.
 *
 *	Permission to use, copy, modify, and distribute this software for any
 *	purpose and without fee is hereby granted, provided that this
 *	copyright and permission notice appear on all copies and supporting
 *	documentation, the name of Livingston Enterprises, Inc. not be used
 *	in advertising or publicity pertaining to distribution of the
 *	program without specific prior permission, and notice be given
 *	in supporting documentation that copying and distribution is by
 *	permission of Livingston Enterprises, Inc.   
 *
 *	Livingston Enterprises, Inc. makes no representations about
 *	the suitability of this software for any purpose.  It is
 *	provided "as is" without express or implied warranty.
 *
 */

/* don't look here for the version, run radiusd -v or look in version.c */
char radiusd_sccsid[] =
"@(#)radiusd.c	1.17 Copyright 1992 Livingston Enterprises Inc\n"
"		1.73 Copyright 1997 Cistron Internet Services B.V.";

#include	<sys/types.h>
#include	<sys/socket.h>
#include	<sys/time.h>
#include	<sys/file.h>
#include	<netinet/in.h>

#include	<stdio.h>
#include	<stdlib.h>
#include	<netdb.h>
#include	<fcntl.h>
#include	<pwd.h>
#include	<time.h>
#include	<ctype.h>
#include	<unistd.h>
#include	<signal.h>
#include	<errno.h>
#include	<sys/wait.h>
#include	<malloc.h>
#include	<getopt.h>

#if !defined(NOSHADOW)
#include	<shadow.h>
#endif /* !NOSHADOW */

#include	"radiusd.h"

char		recv_buffer[4096];
char		send_buffer[4096];
char		*progname;
int		use_dbm = 0;
int		sockfd;
int		acctfd;
int		debug_flag;
int		spawn_flag;
int		log_failures = 0;
int		log_success  = 0;
int		acct_pid;
int		radius_pid;
char		*radius_dir;
char		*radlog_dir;
char		*radacct_dir;
UINT4		expiration_seconds;
UINT4		warning_seconds;
static AUTH_REQ	*first_request;

#if !defined(__linux__) && !defined(__GNU_LIBRARY__)
extern int	errno;
#endif

static int	config_init(void);
static void	usage(void);

static void	sig_fatal (int);
static void	sig_hup (int);
static void	sig_cleanup (int);

static AUTH_REQ	*radrecv (UINT4, u_short, u_char *, int);
static int	radrespond (AUTH_REQ *, int);
static void 	rad_authenticate (AUTH_REQ *, int);
static void	rad_spawn_child (AUTH_REQ *, int);

static void	send_challenge (AUTH_REQ *, char *, char *, int);
static void	send_reject (AUTH_REQ *, char *, int);
static void	send_accept (AUTH_REQ *, VALUE_PAIR *, char *, int);
static int	calc_digest (u_char *, AUTH_REQ *);
static int	unix_pass (char *, char *);

static int	set_expiration (VALUE_PAIR *, UINT4);
static int	pw_expired (UINT4);


int main(int argc, char **argv)
{
	int			salen;
	int			result;
	struct	sockaddr	salocal;
	struct	sockaddr	saremote;
	struct	sockaddr_in	*sin;
	struct	servent		*svp;
        u_short                 lport;
	AUTH_REQ		*authreq;
	char			argval;
	int			t;
	int			pid;
	fd_set			readfds;
	int			status;
	int			dontfork = 0;
#ifdef RADIUS_PID
	FILE			*fp;
#endif
#ifdef RLIMIT_CORE
	struct rlimit rlim;
#endif

	if ((progname = strrchr(argv[0], '/')) == NULL)
		progname = argv[0];
	else
		progname++;

	debug_flag = 0;
	spawn_flag = 1;
	radacct_dir = RADACCT_DIR;
	radlog_dir = RADLOG_DIR;
	radius_dir = RADIUS_DIR;

	signal(SIGHUP, sig_hup);
	signal(SIGINT, sig_fatal);
	signal(SIGQUIT, sig_fatal);
	signal(SIGTRAP, sig_fatal);
	signal(SIGIOT, sig_fatal);
	signal(SIGTERM, sig_fatal);
	signal(SIGCHLD, sig_cleanup);
#if 0
	signal(SIGFPE, sig_fatal);
	signal(SIGSEGV, sig_fatal);
	signal(SIGILL, sig_fatal);
#endif

	/*
	 *	Close unused file descriptors.
	 */
	for (t = 32; t >= 3; t--)
			close(t);

	/*
	 *	Process the options.
	 */
	while((argval = getopt(argc, argv, "a:l:d:bfsvxyz")) != EOF) {

		switch(argval) {

		case 'a':
			radacct_dir = optarg;
			break;
		
		case 'l':
			radlog_dir = optarg;
			break;
		
		case 'f':
			dontfork = 1;
			break;

		case 'd':
			radius_dir = optarg;
			break;
		
#ifdef USE_DBM
		case 'b':
			use_dbm++;
			break;
#endif

		case 's':	/* Single process mode */
			spawn_flag = 0;
			break;

		case 'v':
			version();
			break;

		case 'x':
			debug_flag = 1;
			break;
		
		case 'y':
			log_failures = 1;
			break;

		case 'z':
			log_success = 1;
			break;

		default:
			usage();
			break;
		}
	}

	log(L_INFO, "Starting - reading configuration files ...");

	/* Initialize the dictionary */
	if(dict_init() != 0) {
		exit(1);
	}

	/* Initialize Configuration Values */
	if(config_init() != 0) {
		exit(1);
	}

	/* Read users file etc. */
	if (read_config_files() != 0) {
		exit(1);
	}

	/*
	 *	Open Authentication socket.
	 */
	svp = getservbyname ("radius", "udp");
	if (svp == (struct servent *) 0) {
		lport = htons(1645);
	} else
		lport = (u_short) svp->s_port;

	sockfd = socket (AF_INET, SOCK_DGRAM, 0);
	if (sockfd < 0) {
		perror("auth socket");
		exit(1);
	}

	sin = (struct sockaddr_in *) & salocal;
        memset ((char *) sin, '\0', sizeof (salocal));
	sin->sin_family = AF_INET;
	sin->sin_addr.s_addr = INADDR_ANY;
	sin->sin_port = lport;

	result = bind (sockfd, & salocal, sizeof (*sin));
	if (result < 0) {
		perror ("auth bind");
		exit(1);
	}

	/*
	 *	Open Accounting Socket.
	 */
	svp = getservbyname ("radacct", "udp");
	if (svp == (struct servent *) 0) {
		lport = ntohs(lport) + 1;
		lport = htons(lport);
	} else
		lport = (u_short) svp->s_port;
	
	acctfd = socket (AF_INET, SOCK_DGRAM, 0);
	if (acctfd < 0) {
		perror ("acct socket");
		exit(1);
	}

	sin = (struct sockaddr_in *) & salocal;
        memset ((char *) sin, '\0', sizeof (salocal));
	sin->sin_family = AF_INET;
	sin->sin_addr.s_addr = INADDR_ANY;
	sin->sin_port = lport;

	result = bind (acctfd, & salocal, sizeof (*sin));
	if (result < 0) {
		perror ("acct bind");
		exit(1);
	}

	/*
	 *	Disconnect from session
	 */
	if(debug_flag == 0 && dontfork == 0) {
		pid = fork();
		if(pid < 0) {
			log(L_ERR|L_CONS, "Couldn't fork");
			exit(1);
		}
		if(pid > 0) {
			exit(0);
		}
#if defined(__linux__) || defined(__svr4__)
		setsid();
#endif
	}
	radius_pid = getpid();
#ifdef RADIUS_PID
	if ((fp = fopen(RADIUS_PID, "w")) != NULL) {
		fprintf(fp, "%d\n", radius_pid);
		fclose(fp);
	}
#endif

#if !defined(M_UNIX) && !defined(__linux__)
	/*
	 *	Open system console as stderr
	 */
	t = open("/dev/console", O_WRONLY | O_NOCTTY);
	if (t != 2) {
		dup2(t, 2);
		close(t);
	}
#endif
	/*
	 *	If we are able to spawn processes, we will start a child
	 *	to listen for Accounting requests.  If not, we will 
	 *	listen for them ourself.
	 */
	if (spawn_flag) {
		acct_pid = fork();
		if(acct_pid < 0) {
			log(L_ERR|L_CONS, "Couldn't fork");
			exit(1);
		}
		if(acct_pid > 0) {
			close(acctfd);
			acctfd = -1;
			log(L_INFO, "Ready to process requests.");
		}
		else {
			close(sockfd);
			sockfd = -1;
		}
	} else
		log(L_INFO, "Ready to process requests.");


	/*
	 *	Receive user requests
	 */
	sin = (struct sockaddr_in *) & saremote;

	for(;;) {

		FD_ZERO(&readfds);
		if(sockfd >= 0) {
			FD_SET(sockfd, &readfds);
		}
		if(acctfd >= 0) {
			FD_SET(acctfd, &readfds);
		}

		status = select(32, &readfds, NULL, NULL, NULL);
		if(status == -1) {
			if (errno == EINTR)
				continue;
			sig_fatal(101);
		}
		if(sockfd >= 0 && FD_ISSET(sockfd, &readfds)) {
			salen = sizeof (saremote);
			result = recvfrom (sockfd, (char *) recv_buffer,
				(int) sizeof(recv_buffer),
				(int) 0, &saremote, &salen);

			if(result > 0) {
				authreq = radrecv(
					ntohl(sin->sin_addr.s_addr),
					ntohs(sin->sin_port),
					recv_buffer, result);
				radrespond(authreq, sockfd);
			}
			else if(result < 0 && errno == EINTR) {
				result = 0;
			}
		}
		if(acctfd >=0 && FD_ISSET(acctfd, &readfds)) {
			salen = sizeof (saremote);
			result = recvfrom (acctfd, (char *) recv_buffer,
				(int) sizeof(recv_buffer),
				(int) 0, &saremote, &salen);

			if(result > 0) {
				authreq = radrecv(
					ntohl(sin->sin_addr.s_addr),
					ntohs(sin->sin_port),
					recv_buffer, result);
				radrespond(authreq, acctfd);
			}
			else if(result < 0 && errno == EINTR) {
				result = 0;
			}
		}
	}
}


/*
 *	Receive UDP client requests, build an authorization request
 *	structure, and attach attribute-value pairs contained in
 *	the request to the new structure.
 */
static AUTH_REQ	*radrecv(UINT4 host, u_short udp_port,
			u_char *buffer, int length)
{
	u_char		*ptr;
	AUTH_HDR	*auth;
	int		totallen;
	int		attribute;
	int		attrlen;
	DICT_ATTR	*attr;
	UINT4		lvalue;
	VALUE_PAIR	*first_pair;
	VALUE_PAIR	*prev;
	VALUE_PAIR	*pair;
	AUTH_REQ	*authreq;

	/*
	 *	Pre-allocate the new request data structure
	 */

	if((authreq = (AUTH_REQ *)malloc(sizeof(AUTH_REQ))) ==
						(AUTH_REQ *)NULL) {
		log(L_ERR|L_CONS, "no memory");
		exit(1);
	}

	auth = (AUTH_HDR *)buffer;
	totallen = ntohs(auth->length);

	DEBUG("radrecv: Request from host %lx code=%d, id=%d, length=%d",
				(u_long)host, auth->code, auth->id, totallen);

	/*
	 *	Fill header fields
	 */
	authreq->ipaddr = host;
	authreq->udp_port = udp_port;
	authreq->id = auth->id;
	authreq->code = auth->code;
	memcpy(authreq->vector, auth->vector, AUTH_VECTOR_LEN);

	/*
	 *	Extract attribute-value pairs
	 */
	ptr = auth->data;
	length -= AUTH_HDR_LEN;
	first_pair = (VALUE_PAIR *)NULL;
	prev = (VALUE_PAIR *)NULL;

	while(length > 0) {

		attribute = *ptr++;
		attrlen = *ptr++;
		if(attrlen < 2) {
			length = 0;
			continue;
		}
		attrlen -= 2;
		if((attr = dict_attrget(attribute)) == (DICT_ATTR *)NULL) {
			DEBUG("Received unknown attribute %d", attribute);
		}
		else if ( attrlen >= AUTH_STRING_LEN ) {
			DEBUG("attribute %d too long, %d >= %d", attribute,
				attrlen, AUTH_STRING_LEN);
		}
		else {
			if((pair = (VALUE_PAIR *)malloc(sizeof(VALUE_PAIR))) ==
						(VALUE_PAIR *)NULL) {
				log(L_ERR|L_CONS, "no memory");
				exit(1);
			}
			strcpy(pair->name, attr->name);
			pair->attribute = attr->value;
			pair->type = attr->type;
			pair->next = (VALUE_PAIR *)NULL;

			switch(attr->type) {

			case PW_TYPE_STRING:
				memcpy(pair->strvalue, ptr, attrlen);
				pair->strvalue[attrlen] = '\0';
				debug_pair(stdout, pair);
				if(first_pair == (VALUE_PAIR *)NULL) {
					first_pair = pair;
				}
				else {
					prev->next = pair;
				}
				prev = pair;
				break;
			
			case PW_TYPE_INTEGER:
			case PW_TYPE_IPADDR:
				memcpy(&lvalue, ptr, sizeof(UINT4));
				pair->lvalue = ntohl(lvalue);
				debug_pair(stdout, pair);
				if(first_pair == (VALUE_PAIR *)NULL) {
					first_pair = pair;
				}
				else {
					prev->next = pair;
				}
				prev = pair;
				break;
			
			default:
				DEBUG("    %s (Unknown Type %d)", attr->name,attr->type);
				free(pair);
				break;
			}

		}
		ptr += attrlen;
		length -= attrlen + 2;
	}
	authreq->request = first_pair;
	return(authreq);
}

/*
 *	Respond to supported requests:
 *
 *		PW_AUTHENTICATION_REQUEST - Authentication request from
 *				a client network access server.
 *
 *		PW_ACCOUNTING_REQUEST - Accounting request from
 *				a client network access server.
 */
int radrespond(AUTH_REQ *authreq, int activefd)
{
	switch(authreq->code) {

	case PW_AUTHENTICATION_REQUEST:
		if(spawn_flag) {
			rad_spawn_child(authreq, activefd);
		}
		else {
			rad_authenticate(authreq, activefd);
		}
		break;
	
	case PW_ACCOUNTING_REQUEST:
		rad_accounting(authreq, activefd);
		break;
	
	case PW_PASSWORD_REQUEST:
		/*
		 *	FIXME: print an error message here.
		 *	We don't support this anymore.
		 */
		/* rad_passchange(authreq, activefd); */
		break;
	
	default:
		break;
	}

	return 0;
}



/*
 *	Spawns child processes to perform password authentication
 *	and respond to RADIUS clients.  This functions also
 *	cleans up complete child requests, and verifies that there
 *	is only one process responding to each request (duplicate
 *	requests are filtered out.
 */
static void rad_spawn_child(AUTH_REQ *authreq, int activefd)
{
	AUTH_REQ	*curreq;
	AUTH_REQ	*prevreq;
	UINT4		curtime;
	int		request_count;
	int		child_pid;

	curtime = (UINT4)time(0);
	request_count = 0;
	curreq = first_request;
	prevreq = (AUTH_REQ *)NULL;
	while(curreq != (AUTH_REQ *)NULL) {
		if(curreq->child_pid == -1 &&
				curreq->timestamp + CLEANUP_DELAY <= curtime) {
			/* Request completed, delete it */
			if(prevreq == (AUTH_REQ *)NULL) {
				first_request = curreq->next;
				pairfree(curreq->request);
				free(curreq);
				curreq = first_request;
			}
			else {
				prevreq->next = curreq->next;
				pairfree(curreq->request);
				free(curreq);
				curreq = prevreq->next;
			}
		}
		else if(curreq->ipaddr == authreq->ipaddr &&
					curreq->id == authreq->id) {
			/* This is a duplicate request - just drop it */
			log(L_ERR, "Dropping duplicate authentication packet from %s - ID: %d",
				client_name(authreq->ipaddr), authreq->id);
			pairfree(authreq->request);
			free(authreq);
			return;
		}
		else {
			if(curreq->timestamp + MAX_REQUEST_TIME <= curtime &&
						curreq->child_pid != -1) {
				/* This request seems to have hung - kill it */
				child_pid = curreq->child_pid;
				log(L_ERR,
					"Killing unresponsive child pid %d",
								child_pid);
				curreq->child_pid = -1;
				kill(child_pid, SIGTERM);
			}
			prevreq = curreq;
			curreq = curreq->next;
			request_count++;
		}
	}

	/* This is a new request */
	if(request_count > MAX_REQUESTS) {
		log(L_ERR, "Dropping request (too many): from %s - ID: %d",
				client_name(authreq->ipaddr), authreq->id);
		pairfree(authreq->request);
		free(authreq);
		return;
	}

	/* Add this request to the list */
	authreq->next = (AUTH_REQ *)NULL;
	authreq->child_pid = -1;
	authreq->timestamp = curtime;

	if(prevreq == (AUTH_REQ *)NULL) {
		first_request = authreq;
	}
	else {
		prevreq->next = authreq;
	}

	/* fork our child */
	child_pid = fork();
	if(child_pid < 0) {
		log(L_ERR, "Fork failed for request from %s - ID: %d",
				client_name(authreq->ipaddr), authreq->id);
	}
	if(child_pid == 0) {
		/* This is the child, it should go ahead and respond */
		rad_authenticate(authreq, activefd);
		exit(0);
	}

	/* Register the Child */
	authreq->child_pid = child_pid;
	return;
}


/*ARGSUSED*/
static void sig_cleanup(int sig)
{
	int		status;
        pid_t		pid;
	AUTH_REQ	*curreq;
 
        for (;;) {
		pid = waitpid((pid_t)-1, &status, WNOHANG);
		signal(SIGCHLD, sig_cleanup);
                if (pid <= 0)
                        return;

#if defined (aix)
		kill(pid, SIGKILL);
#endif

		if(pid == acct_pid) {
			sig_fatal(100);
		}
		curreq = first_request;
		while(curreq != (AUTH_REQ *)NULL) {
			if(curreq->child_pid == pid) {
				curreq->child_pid = -1;
				curreq->timestamp = (UINT4)time(0);
				break;
			}
			curreq = curreq->next;
		}
        }
}


/*
 *	Process and reply to an authentication request
 */
static void rad_authenticate(AUTH_REQ *authreq, int activefd)
{
	VALUE_PAIR	*namepair;
	VALUE_PAIR	*check_item;
	VALUE_PAIR	*auth_item;
	VALUE_PAIR	*user_check;
	VALUE_PAIR	*user_reply;
	VALUE_PAIR	*auth_type_pair;
	int		auth_type = -1;
	int		result;
	char		pw_digest[16];
	char		string[128];
	char		userpass[128];
	int		i;
	char		umsg[128];
	char		*user_msg;
	int		retval;
	char		*ptr;
	char		*exec_program;

	/*
	 *	Get the username from the request
	 */
	namepair = pairfind(authreq->request, PW_USER_NAME);

	if((namepair == (VALUE_PAIR *)NULL) || 
	   (strlen(namepair->strvalue) <= 0)) {
		log(L_ERR, "Authenticate: from %s - No User Name",
			client_name(authreq->ipaddr));
		pairfree(authreq->request);
		memset(authreq, 0, sizeof(AUTH_REQ));
		free(authreq);
		return;
	}

	/*
	 *	Verify the client and Calculate the MD5 Password Digest
	 */
	if(calc_digest(pw_digest, authreq) != 0) {
		/*
		 *	We dont respond when this fails
		 */
		log(L_ERR, "Authenticate: from %s - Security Breach: %s",
			client_name(authreq->ipaddr), namepair->strvalue);
		pairfree(authreq->request);
		memset(authreq, 0, sizeof(AUTH_REQ));
		free(authreq);
		return;
	}

	/*
	 *	Add any specific attributes for this username.
	 */
	hints_setup(authreq->request);

	/*
	 *	See if the user has access to this huntgroup.
	 */
	if (!huntgroup_access(authreq->request)) {
		log(L_AUTH, "huntgroup_access: %s - access denied",
			namepair->strvalue);
		send_reject(authreq, (char *)NULL, activefd);
		pairfree(authreq->request);
		memset(authreq, 0, sizeof(AUTH_REQ));
		free(authreq);
		return;
	}


	/*
	 *	User *must* have a password.
	 */
	if ((auth_item = pairfind(authreq->request, PW_PASSWORD)) == NULL) {
		log(L_ERR, "Authenticate: from %s - %s No Password",
			client_name(authreq->ipaddr), namepair->strvalue);
		send_reject(authreq, (char *)NULL, activefd);
		pairfree(authreq->request);
		memset(authreq, 0, sizeof(AUTH_REQ));
		free(authreq);
		return;
	}

	/*
	 *	Get the user from the database
	 */
	if (user_find(namepair->strvalue, authreq->request,
	   &user_check, &user_reply) != 0) {
		log(L_AUTH, "Authenticate: from %s - Invalid User: %s",
			client_name(authreq->ipaddr), namepair->strvalue);
		send_reject(authreq, (char *)NULL, activefd);
		pairfree(authreq->request);
		memset(authreq, 0, sizeof(AUTH_REQ));
		free(authreq);
		return;
	}

	if ((auth_type_pair = pairfind(user_check, PW_AUTH_TYPE)) != NULL)
		auth_type = auth_type_pair->lvalue;

	/*
	 *	Validate the user
	 */

	/*
	 *	Look for matching check items. We skip the whole lot
	 *	if the authentication type is PW_AUTH_NONE.
	 *
	 *	FIXME: This is a _mess_! Clean it up.
	 */
	userpass[0] = 0;
	result = auth_type == PW_AUTH_NONE ? 1 : 0;
	user_msg = (char *)NULL;
	check_item = user_check;
	while(result == 0 && check_item != (VALUE_PAIR *)NULL) {

		/*
		 *	Check expiration date if we are doing password aging.
		 */
		if (check_item->attribute == PW_EXPIRATION) {
			/* Has this user's password expired */
			retval = pw_expired(check_item->lvalue);
			if(retval < 0) {
				result = -1;
				user_msg = "Password Has Expired\r\n";
			}
			else {
				if(retval > 0) {
					sprintf(umsg,
					  "Password Will Expire in %d Days\r\n",
					  retval);
					user_msg = umsg;
				}
				check_item = check_item->next;
			}
			continue;
		}

		if (check_item->attribute == PW_AUTH_TYPE &&
		    auth_type == PW_AUTH_SYSTEM) {
			/*
			 *	Check the password against /etc/passwd.
			 */
			memcpy(string, auth_item->strvalue, AUTH_PASS_LEN);
			for(i = 0;i < AUTH_PASS_LEN;i++) {
				string[i] ^= pw_digest[i];
			}
			strcpy(userpass, string);
			string[AUTH_PASS_LEN] = '\0';
			if (unix_pass(namepair->strvalue, string) != 0) {
				result = -1;
				user_msg = (char *)NULL;
			}
			check_item = check_item->next;
			continue;
		}

		/*
		 *	Skip the check if it's not a password.
		 */
		if (check_item->attribute != PW_PASSWORD) {
			check_item = check_item->next;
			continue;
		}

		/*
		 *	Special handling for passwords which are encrypted,
		 *	and sometimes authenticated against the UNIX passwd
		 *	database.
		 *	Also they can come using the Three-Way CHAP.
		 *
		 */
		if (auth_item->attribute == PW_CHAP_PASSWORD) {
			/*
			 *	Use MD5 to verify
			 */
			ptr = string;
			*ptr++ = *auth_item->strvalue;
			strcpy(ptr, check_item->strvalue);
			ptr += strlen(check_item->strvalue);
			memcpy(ptr, authreq->vector, AUTH_VECTOR_LEN);
			md5_calc(pw_digest, string,
				1 + CHAP_VALUE_LENGTH +
				strlen(check_item->strvalue));
			/*
			 *	Compare them
			 */
			strcpy(userpass, pw_digest);
			if(memcmp(pw_digest, auth_item->strvalue + 1,
					CHAP_VALUE_LENGTH) != 0) {
				result = -1;
			}
		}
		else {
			/*
			 *	No CHAP. Decrypt the password
			 */
			memcpy(string, auth_item->strvalue, AUTH_PASS_LEN);
			for(i = 0;i < AUTH_PASS_LEN;i++) {
				string[i] ^= pw_digest[i];
			}
			strcpy(userpass, string);
			string[AUTH_PASS_LEN] = '\0';

			/*
			 *	Test Code for Challenge
			 */
			if(strcmp(string, "challenge") == 0) {
				send_challenge(authreq, 
			"You want me to challenge you??\r\nOkay I will",
					"1",activefd);
				pairfree(authreq->request);
				memset(authreq, 0, sizeof(AUTH_REQ));
				free(authreq);
				return;
			}
			/*
			 *	For backward compatibility, we check the
			 *	password to see if it is the magic value
			 *	UNIX if auth_type was not set.
			 */
			if ((auth_type < 0 &&
			     strcmp(check_item->strvalue, "UNIX") == 0)) {
				if (unix_pass(namepair->strvalue,
							string) != 0) {
					result = -1;
					user_msg = (char *)NULL;
				}
			}
			/*
			 *	Nope - then it's a plain text password.
			 */
			else if (auth_type != PW_AUTH_SYSTEM &&
				strcmp(check_item->strvalue, string) != 0) {
				result = -1;
				user_msg = (char *)NULL;
			}
		}
		check_item = check_item->next;
	}
	if (result < 0) {
		/*
		 *	Failed to verify the user.
		 */
		send_reject(authreq, user_msg, activefd);
		if (log_failures) {
			log(L_AUTH,
				"Login incorrect: [%s/%s] (from %s)",
				namepair->strvalue, userpass,
				client_name(authreq->ipaddr));
		}
	}

	if (result >= 0 &&
	   (check_item = pairfind(user_check, PW_SIMULTANEOUS_USE)) != NULL) { 
		/*
		 *	User authenticated O.K. Now we have to check
		 *	for the Simulteneous-Use parameter.
		 */
		if (rad_check_multi(namepair->strvalue, authreq->request,
		    check_item->lvalue) < 0) {

			if (check_item->lvalue > 1) {
				sprintf(umsg,
		"\r\nYou are already logged in %d times  - access denied\r\n\n",
					(int)check_item->lvalue);
				user_msg = umsg;
			} else {
				user_msg =
		"\r\nYou are already logged in - access denied\r\n\n";
			}
			send_reject(authreq, user_msg, activefd);
			log(L_ERR, "Multiple logins: [%s] (from %s) max. %d",
				namepair->strvalue,
				client_name(authreq->ipaddr),
				check_item->lvalue);
			result = -1;
		}
	}

	/*
	 *	See if we need to execute a program.
	 *	FIXME: somehow cache this info, and only execute the
	 *	program when we receive an Accounting-START packet.
	 *	Only at that time we know dynamic IP etc.
	 */
	exec_program = NULL;
	if ((auth_item = pairfind(user_reply, PW_EXEC_PROGRAM))!=NULL){
		exec_program = strdup(auth_item->strvalue);
		pairdelete(&user_reply, PW_EXEC_PROGRAM);
	}

	if (result >= 0) {
#ifdef CISTRON_COMPAT
		/*
		 *	FIXME: don't do this with Auto-PPP.
		 */
		if (user_msg == NULL)
			user_msg = cistron_compat(authreq->request, user_reply);
#endif
		send_accept(authreq, user_reply, user_msg, activefd);
		if (log_success) {
			log(L_AUTH,
				"Login OK: [%s/%s] (from %s)",
				namepair->strvalue, userpass,
				client_name(authreq->ipaddr));
		}
		if (exec_program)
			radius_exec_program(exec_program,
				authreq->request, user_reply);
	}

	if (exec_program) free(exec_program);
	pairfree(authreq->request);
	memset(authreq, 0, sizeof(AUTH_REQ));
	free(authreq);
	pairfree(user_check);
	pairfree(user_reply);

	return;
}


/*
 *	Reply to the request with a REJECT.  Also attach
 *	any user message provided.
 */
static void send_reject(AUTH_REQ *authreq, char *msg, int activefd)
{
	AUTH_HDR		*auth;
	struct	sockaddr	saremote;
	struct	sockaddr_in	*sin;
	char			digest[AUTH_VECTOR_LEN];
	int			secretlen;
	int			total_length;
	u_char			*ptr;
	int			len;

	auth = (AUTH_HDR *)send_buffer;

	/* Build standard response header */
	if(authreq->code == PW_PASSWORD_REQUEST) {
		auth->code = PW_PASSWORD_REJECT;
	}
	else {
		auth->code = PW_AUTHENTICATION_REJECT;
	}
	auth->id = authreq->id;
	memcpy(auth->vector, authreq->vector, AUTH_VECTOR_LEN);
	total_length = AUTH_HDR_LEN;

	/* Append the user message */
	if(msg != (char *)NULL) {
		len = strlen(msg);
		if(len > 0 && len < AUTH_STRING_LEN) {
			ptr = auth->data;
			*ptr++ = PW_REPLY_MESSAGE;
			*ptr++ = len + 2;
			memcpy(ptr, msg, len);
			ptr += len;
			total_length += len + 2;
		}
	}

	/* Set total length in the header */
	auth->length = htons(total_length);

	/* Calculate the response digest */
	secretlen = strlen(authreq->secret);
	memcpy(send_buffer + total_length, authreq->secret, secretlen);
	md5_calc(digest, (char *)auth, total_length + secretlen);
	memcpy(auth->vector, digest, AUTH_VECTOR_LEN);
	memset(send_buffer + total_length, 0, secretlen);

	sin = (struct sockaddr_in *) &saremote;
        memset ((char *) sin, '\0', sizeof (saremote));
	sin->sin_family = AF_INET;
	sin->sin_addr.s_addr = htonl(authreq->ipaddr);
	sin->sin_port = htons(authreq->udp_port);

	DEBUG("Sending Reject of id %d to %lx (%s)",
		authreq->id, (u_long)authreq->ipaddr,
		client_name(authreq->ipaddr));
	
	/* Send it to the user */
	sendto(activefd, (char *)auth, (int)total_length, (int)0,
			(struct sockaddr *) &saremote, sizeof(struct sockaddr_in));
}


/*
 *	Reply to the request with a CHALLENGE.  Also attach
 *	any user message provided and a state value.
 */
static void send_challenge(AUTH_REQ *authreq, char *msg,
				char *state, int activefd)
{
	AUTH_HDR		*auth;
	struct	sockaddr_in	saremote;
	struct	sockaddr_in	*sin;
	char			digest[AUTH_VECTOR_LEN];
	int			secretlen;
	int			total_length;
	u_char			*ptr;
	int			len;

	auth = (AUTH_HDR *)send_buffer;

	/* Build standard response header */
	auth->code = PW_ACCESS_CHALLENGE;
	auth->id = authreq->id;
	memcpy(auth->vector, authreq->vector, AUTH_VECTOR_LEN);
	total_length = AUTH_HDR_LEN;

	/* Append the user message */
	if(msg != (char *)NULL) {
		len = strlen(msg);
		if(len > 0 && len < AUTH_STRING_LEN) {
			ptr = auth->data;
			*ptr++ = PW_REPLY_MESSAGE;
			*ptr++ = len + 2;
			memcpy(ptr, msg, len);
			ptr += len;
			total_length += len + 2;
		}
	}

	/* Append the state info */
	if((state != (char *)NULL) && (strlen(state) > 0)) {
		len = strlen(state);
		*ptr++ = PW_STATE;
		*ptr++ = len + 2;
		memcpy(ptr, state, len);
		ptr += len;
		total_length += len + 2;
	}

	/* Set total length in the header */
	auth->length = htons(total_length);

	/* Calculate the response digest */
	secretlen = strlen(authreq->secret);
	memcpy(send_buffer + total_length, authreq->secret, secretlen);
	md5_calc(digest, (char *)auth, total_length + secretlen);
	memcpy(auth->vector, digest, AUTH_VECTOR_LEN);
	memset(send_buffer + total_length, 0, secretlen);

	sin = (struct sockaddr_in *) &saremote;
        memset ((char *) sin, '\0', sizeof (saremote));
	sin->sin_family = AF_INET;
	sin->sin_addr.s_addr = htonl(authreq->ipaddr);
	sin->sin_port = htons(authreq->udp_port);

	DEBUG("Sending Challenge of id %d to %lx (%s)",
		authreq->id, (u_long)authreq->ipaddr,
		client_name(authreq->ipaddr));
	
	/* Send it to the user */
	sendto(activefd, (char *)auth, (int)total_length, (int)0,
			(struct sockaddr *) &saremote, sizeof(struct sockaddr_in));
}


/*
 *	Reply to the request with an ACKNOWLEDGE.  Also attach
 *	reply attribute value pairs and any user message provided.
 */
static void send_accept(AUTH_REQ *authreq, VALUE_PAIR *reply,
			char *msg, int activefd)
{
	AUTH_HDR		*auth;
	u_short			total_length;
	struct	sockaddr	saremote;
	struct	sockaddr_in	*sin;
	u_char			*ptr;
	int			len;
	UINT4			lvalue;
	u_char			digest[16];
	int			secretlen;

	auth = (AUTH_HDR *)send_buffer;

	/* Build standard header */
	auth->code = PW_AUTHENTICATION_ACK;
	auth->id = authreq->id;
	memcpy(auth->vector, authreq->vector, AUTH_VECTOR_LEN);

	DEBUG("Sending Ack of id %d to %lx (%s)",
		authreq->id, (u_long)authreq->ipaddr,
		client_name(authreq->ipaddr));

	total_length = AUTH_HDR_LEN;

	/* Load up the configuration values for the user */
	ptr = auth->data;
	while(reply != (VALUE_PAIR *)NULL) {
		debug_pair(stdout, reply);
		*ptr++ = reply->attribute;

		switch(reply->type) {

		case PW_TYPE_STRING:
			len = strlen(reply->strvalue);
			if (len >= AUTH_STRING_LEN) {
				len = AUTH_STRING_LEN - 1;
			}
			*ptr++ = len + 2;
			memcpy(ptr, reply->strvalue,len);
			ptr += len;
			total_length += len + 2;
			break;
			
		case PW_TYPE_INTEGER:
		case PW_TYPE_IPADDR:
			*ptr++ = sizeof(UINT4) + 2;
			lvalue = htonl(reply->lvalue);
			memcpy(ptr, &lvalue, sizeof(UINT4));
			ptr += sizeof(UINT4);
			total_length += sizeof(UINT4) + 2;
			break;

		default:
			break;
		}

		reply = reply->next;
	}

	/* Append the user message */
	if(msg != (char *)NULL) {
		len = strlen(msg);
		if(len > 0 && len < AUTH_STRING_LEN) {
			*ptr++ = PW_REPLY_MESSAGE;
			*ptr++ = len + 2;
			memcpy(ptr, msg, len);
			ptr += len;
			total_length += len + 2;
		}
	}

	auth->length = htons(total_length);

	/* Append secret and calculate the response digest */
	secretlen = strlen(authreq->secret);
	memcpy(send_buffer + total_length, authreq->secret, secretlen);
	md5_calc(digest, (char *)auth, total_length + secretlen);
	memcpy(auth->vector, digest, AUTH_VECTOR_LEN);
	memset(send_buffer + total_length, 0, secretlen);

	sin = (struct sockaddr_in *) &saremote;
        memset ((char *) sin, '\0', sizeof (saremote));
	sin->sin_family = AF_INET;
	sin->sin_addr.s_addr = htonl(authreq->ipaddr);
	sin->sin_port = htons(authreq->udp_port);

	/* Send it to the user */
	sendto(activefd, (char *)auth, (int)total_length, (int)0,
			&saremote, sizeof(struct sockaddr_in));
}


/*
 *	Check the users password against the standard UNIX
 *	password table.
 */
static int unix_pass(char *name, char *passwd)
{
#ifndef __linux__
	char		*crypt();
#endif
	struct passwd	*pwd;
	char		*encpw;
	char		*encrypted_pass;
#if !defined(NOSHADOW)
#if defined(M_UNIX)
	struct passwd	*spwd;
#else
	struct spwd	*spwd;
#endif
#endif /* !NOSHADOW */
	
	/* Get encrypted password from password file */
	if((pwd = rad_getpwnam(name)) == NULL) {
		return(-1);
	}

	encrypted_pass = pwd->pw_passwd;

#if !defined(NOSHADOW)
	if(strcmp(pwd->pw_passwd, "x") == 0) {
		if((spwd = getspnam(name)) == NULL) {
			return(-1);
		}
#if defined(M_UNIX)
		encrypted_pass = spwd->pw_passwd;
#else
		encrypted_pass = spwd->sp_pwdp;
#endif	/* M_UNIX */
	}
#endif	/* !NOSHADOW */

	/* We might have a passwordless account. */
	if (encrypted_pass[0] == 0) return 0;

	/* Run encryption algorythm */
	encpw = crypt(passwd, encrypted_pass);

	/* Check it */
	if(strcmp(encpw, encrypted_pass)) {
		return(-1);
	}
	return(0);
}


/*
 *	Validates the requesting client NAS.  Calculates the
 *	digest to be used for decrypting the users password
 *	based on the clients private key.
 */
static int calc_digest(u_char *digest, AUTH_REQ *authreq)
{
	u_char	buffer[128];
	int	secretlen;
	CLIENT	*cl;

	/*
	 *	See if we know this client.
	 */
	if ((cl = client_find(authreq->ipaddr)) == NULL) {
		log(L_ERR, "request from unknown client: %s",
			client_name(authreq->ipaddr));
		return -1;
	}

	/* Use the secret to setup the decryption digest */
	secretlen = strlen(cl->secret);
	strcpy(buffer, cl->secret);
	memcpy(buffer + secretlen, authreq->vector, AUTH_VECTOR_LEN);
	md5_calc(digest, buffer, secretlen + AUTH_VECTOR_LEN);
	strcpy(authreq->secret, cl->secret);
	memset(buffer, 0, sizeof(buffer));
	return(0);
}


/*
 *	Print the Attribute-value pair to the desired File.
 *	FIXME: work with logging routines in log.c
 */
void debug_pair(FILE *fd, VALUE_PAIR *pair)
{
	if(debug_flag) {
		fputs("    ", fd);
		fprint_attr_val(fd, pair);
		fputs("\n", fd);
	}
}


/*
 *	Display the syntax for starting this program.
 */
static void usage(void)
{
	fprintf(stderr,
#ifdef USE_DBM
		"Usage: %s [-a acct_dir] [-d db_dir] [-l logdir] [-bsxyz]\n",
#else
		"Usage: %s [-a acct_dir] [-d db_dir] [-l logdir] [-sxyz]\n",
#endif
		progname);
	exit(1);
}


/*
 *	Intializes configuration values:
 *
 *		expiration_seconds - When updating a user password,
 *			the amount of time to add to the current time
 *			to set the time when the password will expire.
 *			This is stored as the VALUE Password-Expiration
 *			in the dictionary as number of days.
 *
 *		warning_seconds - When acknowledging a user authentication
 *			time remaining for valid password to notify user
 *			of password expiration.
 *
 *	These values are read from the SERVER_CONFIG part of the
 *	dictionary (of all places!)
 */
int config_init()
{
	DICT_VALUE	*dval;

	if((dval = dict_valfind("Password-Expiration")) == (DICT_VALUE *)NULL) {
		expiration_seconds = (UINT4)0;
	}
	else {
		expiration_seconds = dval->value * (UINT4)SECONDS_PER_DAY;
	}
	if((dval = dict_valfind("Password-Warning")) == (DICT_VALUE *)NULL) {
		warning_seconds = (UINT4)0;
	}
	else {
		warning_seconds = dval->value * (UINT4)SECONDS_PER_DAY;
	}
	return(0);
}


/*
 *	Set the new expiration time by updating or adding
 *	the Expiration attribute-value pair.
 *
 *	FIXME: This is not used. Did we delete too much code?
 *	Investigate what it was used for in 1.16.
 */
static int set_expiration(VALUE_PAIR *user_check, UINT4 expiration)
{
	VALUE_PAIR	*exppair;
	VALUE_PAIR	*prev;
	struct timeval	tp;
	struct timezone	tzp;

	if(user_check == (VALUE_PAIR *)NULL) {
		return(-1);
	}

	/* Look for an existing expiration entry */
	exppair = user_check;
	prev = (VALUE_PAIR *)NULL;
	while(exppair != (VALUE_PAIR *)NULL) {
		if(exppair->attribute == PW_EXPIRATION) {
			break;
		}
		prev = exppair;
		exppair = exppair->next;
	}
	if(exppair == (VALUE_PAIR *)NULL) {
		/* Add a new attr-value pair */
		if((exppair = (VALUE_PAIR *)malloc(sizeof(VALUE_PAIR))) ==
					(VALUE_PAIR *)NULL) {
			log(L_ERR|L_CONS, "no memory");
			exit(1);
		}
		/* Initialize it */
		strcpy(exppair->name, "Expiration");
		exppair->attribute = PW_EXPIRATION;
		exppair->type = PW_TYPE_DATE;
		*exppair->strvalue = '\0';
		exppair->lvalue = (UINT4)0;
		exppair->next = (VALUE_PAIR *)NULL;

		/* Attach it to the list. */
		prev->next = exppair;
	}

	/* calculate a new expiration */
	gettimeofday(&tp, &tzp);
	exppair->lvalue = tp.tv_sec + expiration;
	return(0);
}


/*
 *	Tests to see if the users password has expired.
 *
 *	Return: Number of days before expiration if a warning is required
 *		otherwise 0 for success and -1 for failure.
 */
static int pw_expired(UINT4 exptime)
{
	struct timeval	tp;
	struct timezone	tzp;
	UINT4		exp_remain;
	int		exp_remain_int;

	if(expiration_seconds == (UINT4)0) {
		return(0);
	}

	gettimeofday(&tp, &tzp);
	if(tp.tv_sec > exptime) {
		return(-1);
	}
	if(warning_seconds != (UINT4)0) {
		if(tp.tv_sec > exptime - warning_seconds) {
			exp_remain = exptime - tp.tv_sec;
			exp_remain /= (UINT4)SECONDS_PER_DAY;
			exp_remain_int = exp_remain;
			return(exp_remain_int);
		}
	}
	return(0);
}


/*
 *	We got a fatal signal. Clean up and exit.
 */
static void sig_fatal(int sig)
{
	if(acct_pid > 0) {
		kill(acct_pid, SIGKILL);
	}
	switch(sig) {
		case 100:
			log(L_ERR, "accounting process died - exit.");
			break;
		case 101:
			log(L_ERR, "failed in select() - exit.");
			break;
		case SIGTERM:
			log(L_INFO, "exit.", sig);
			break;
		default:
			log(L_ERR, "exit on signal (%d)", sig);
			break;
	}

	exit(1);
}


/*
 *	We got the hangup signal. Do nothing.
 *	FIXME: re-read the configuration files here.
 */
/*ARGSUSED*/
static void sig_hup(int sig)
{
	return;
}

/*
 *	Execute a program on successful authentication.
 *
 *	Macros that can be used on the command line.
 *	%p   Port number
 *	%n   NAS IP address
 *	%f   Framed IP address
 *	%u   User name
 */
int radius_exec_program(char *cmd, VALUE_PAIR *request, VALUE_PAIR *reply)
{
	int pid;
	char *p, buf[1024];
	int n, i = 0;
	char *argv[32];
	int argc = -1;
	VALUE_PAIR *tmp;

	if ((pid = fork()) != 0) {
		if (pid < 0) {
			log(L_ERR|L_CONS, "Couldn't fork");
			return -1;
		}
		return 0;
	}

	/*
	 *	Replace %<whatever> in the command string.
	 */
	for(p = cmd; *p; p++) {
		if (*p != '%') {
			buf[i++] = *p;
			continue;
		}
		if (*++p == 0) break;
		switch(*p) {
			case '%':
				buf[i++] = *p;
				break;
			case 'f': /* Framed IP address */
				n = 0;
				if ((tmp = pairfind(reply,
				     PW_FRAMED_IP_ADDRESS)) != NULL) {
					n = tmp->lvalue;
				}
				ipaddr2str(buf + i, n);
				i += strlen(buf + i);
				break;
			case 'n': /* NAS IP address */
				n = 0;
				if ((tmp = pairfind(request,
				     PW_NAS_IP_ADDRESS)) != NULL) {
					n = tmp->lvalue;
				}
				ipaddr2str(buf + i, n);
				i += strlen(buf + i);
				break;
			case 'p': /* Port number */
				n = 0;
				if ((tmp = pairfind(request,
				     PW_NAS_PORT_ID)) != NULL) {
					n = tmp->lvalue;
				}
				sprintf(buf + i, "%d", n);
				i += strlen(buf + i);
				break;
			case 'u': /* User name */
				if ((tmp = pairfind(request,
				     PW_USER_NAME)) != NULL)
					strcpy(buf + i, tmp->strvalue);
				else
					strcpy(buf + i, "unknown");
				i += strlen(buf + i);
				break;
			default:
				buf[i++] = '%';
				buf[i++] = *p;
				break;
		}
	}
	buf[i++] = 0;

	/*
	 *	XXX FIXME: This is debugging info.
	 */
	log(L_INFO, "Exec-Program: %s", buf);

	/*
	 *	Build vector list and execute.
	 */
	p = strtok(buf, " \t");
	if (p) do {
		argv[++argc] = p;
		p = strtok(NULL, " \t");
	} while(p != NULL);
	argv[++argc] = p;
	if (argc == 0) {
		log(L_ERR, "Exec-Program: empty command line.");
		exit(1);
	}
	for(n = 32; n >= 0; n--)
		close(n);
	execvp(argv[0], argv);
	log(L_ERR, "Exec-Program: %s: %m", argv[0]);
	exit(1);

	return 0;
}

